import csv
import sys
from pathlib import Path
import heapq
import statistics
from common import Validate, SanitizedConfig
from abc import ABC, abstractmethod
import os


class Aggregator(ABC):
    """
    Aggregator classes used to "aggregate" a pool of elements, and produce an
    "average" (precisely, some "measure of central tendency") from the elements.
    """

    @staticmethod
    @abstractmethod
    def get_type() -> str:
        """
        Return a string indicating the type of average this aggregator
        produces.
        """
        pass

    @abstractmethod
    def add(self, n: float):
        """
        Add/aggregate an element to the pool of elements used by this aggregator
        to produce an average calculation.
        """
        pass

    @abstractmethod
    def get_avg(self) -> float:
        """
        Produce an average from the pool of elements aggregated using add().
        """
        pass


class SimpleMedian(Aggregator):
    """
    Simple median calculation: if the number of samples being generated are low,
    this is the fastest median method.
    """

    def __init__(self):
        self.elements = []

    @staticmethod
    def get_type() -> str:
        return "median"

    def add(self, n: float):
        self.elements.append(n)

    def get_avg(self) -> float:
        return statistics.median(self.elements)


class StreamingMedian(Aggregator):
    """
    Calculate medians incrementally using heaps: Theoretically the fastest way
    to calculate a median from a stream of elements, but realistically is only
    faster when dealing with huge numbers of samples that would be generated by
    i.e. enabling this workflow in precommit and using longer periods of time.
    """

    def __init__(self):
        # Gist: we keep a minheap and a maxheap, and store the median as the top
        # of the minheap. When a new element comes it gets put into the heap
        # based on if the element is bigger than the current median. Then, the
        # heaps are heapified and the median is repopulated by heapify.
        self.minheap_larger = []
        self.maxheap_smaller = []

    @staticmethod
    def get_type() -> str:
        return "median"

    # Note: numbers on maxheap should be negative, as heapq
    # is minheap by default

    def add(self, n: float):
        if len(self.maxheap_smaller) == 0 or -self.maxheap_smaller[0] >= n:
            heapq.heappush(self.maxheap_smaller, -n)
        else:
            heapq.heappush(self.minheap_larger, n)

        # Ensure minheap has more elements than maxheap
        if len(self.maxheap_smaller) > len(self.minheap_larger) + 1:
            heapq.heappush(self.minheap_larger, -heapq.heappop(self.maxheap_smaller))
        elif len(self.maxheap_smaller) < len(self.minheap_larger):
            heapq.heappush(self.maxheap_smaller, -heapq.heappop(self.minheap_larger))

    def get_avg(self) -> float:
        if len(self.maxheap_smaller) == len(self.minheap_larger):
            # Equal number of elements smaller and larger than "median":
            # thus, there are two median values. The median would then become
            # the average of both median values.
            return (-self.maxheap_smaller[0] + self.minheap_larger[0]) / 2.0
        else:
            # Otherwise, median is always in minheap, as minheap is always
            # bigger
            return -self.maxheap_smaller[0]


class Aggregate:
    """
    Static class providing methods for aggregating data
    """

    @staticmethod
    def hist_avg(
        benchmark_name: str, res_dir: str, cutoff: str, aggregator=SimpleMedian
    ):
        if not os.path.isdir(res_dir):
            print(f"Not a directory: {res_dir}.", file=sys.stderr)
            exit(1)

        def get_csv_samples() -> list[str]:
            """Get all valid .csv samples from the results folder."""
            cache_dir = Path(f"{res_dir}")
            # Filter all benchmark .csv files in the result directory:
            return list(
                filter(
                    # Make sure the .csv "file" is a file:
                    lambda f: f.is_file()
                    # Make sure timestamp of .csv file is good format:
                    # [-19:-4] corresponds to the timestamp in the filename.
                    and Validate.timestamp(str(f)[-19:-4])
                    # Make sure timestamp is bigger than cutoff timestamp:
                    and str(f)[-19:-4] > cutoff,
                    cache_dir.glob(f"{benchmark_name}-*_*.csv"),
                )
            )

        # Calculate median of every desired metric:
        samples_aggregate = dict()
        filtered_samples = get_csv_samples()
        if len(filtered_samples) == 0:
            print(
                f"WARNING: No results for {benchmark_name} found from {cutoff} to now",
                file=sys.stderr,
            )
        for sample_path in filtered_samples:
            with open(sample_path, "r") as sample_file:
                for sample in csv.DictReader(sample_file):
                    test = sample["TestCase"]
                    # Construct entry in aggregator for test if it doesn't exist
                    # already:
                    if test not in samples_aggregate:
                        samples_aggregate[test] = {
                            metric: aggregator()
                            for metric in SanitizedConfig.METRICS_TOLERANCES
                        }

                    # For each metric of concern, add to aggregator:
                    for metric in SanitizedConfig.METRICS_TOLERANCES:
                        sample_value = Validate.sanitize_stat(sample[metric])
                        if not isinstance(sample_value, float):
                            print(
                                f"Malformatted statistic in {str(sample_path)}: "
                                + f"'{sample[metric]}' for {test}."
                            )
                            exit(1)
                        # Add metric from sample for current test to aggregate:
                        samples_aggregate[test][metric].add(sample_value)

        # Calculate + write new average (from samples_aggregate) in new .csv file:
        with open(
            f"{res_dir}/{benchmark_name}-{aggregator.get_type()}.csv", "w"
        ) as output_csv:
            writer = csv.DictWriter(
                output_csv,
                fieldnames=["TestCase", *SanitizedConfig.METRICS_TOLERANCES.keys()],
            )
            writer.writeheader()
            for test in samples_aggregate:
                writer.writerow(
                    {"TestCase": test}
                    | {
                        metric: samples_aggregate[test][metric].get_avg()
                        for metric in SanitizedConfig.METRICS_TOLERANCES
                    }
                )


if __name__ == "__main__":
    if len(sys.argv) != 5:
        print(
            f"Usage: {sys.argv[0]} <path to /devops> <benchmark name> <absolute path to benchmark results> <cutoff timestamp YYYYMMDD_HHMMSS>"
        )
        exit(1)
    if not Validate.timestamp(sys.argv[4]):
        print(f"Bad cutoff timestamp, please use YYYYMMDD_HHMMSS.", file=sys.stderr)
        exit(1)
    if not Validate.filepath(sys.argv[1]):
        print(f"Not a valid filepath: {sys.argv[1]}", file=sys.stderr)
        exit(1)
    # If the filepath provided passed filepath validation, then it is clean
    SanitizedConfig.load(sys.argv[1])

    Aggregate.hist_avg(sys.argv[2], sys.argv[3], sys.argv[4])
