/*
 * Copyright (C) 2024-2025 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#if ENABLE(GPU_PROCESS)

#include "RemoteDeviceProxy.h"
#include "RemoteGPUProxy.h"
#include "RemotePresentationContextProxy.h"
#include "WebGPUIdentifier.h"
#include <WebCore/WebGPUXREye.h>
#include <WebCore/WebGPUXRView.h>

namespace WebCore {
class WebXRFrame;
}

namespace WebCore::WebGPU {
class Device;
class XRProjectionLayer;
class XRView;
}

namespace WebKit::WebGPU {

class ConvertToBackingContext;

class RemoteXRViewProxy final : public WebCore::WebGPU::XRView {
    WTF_MAKE_TZONE_ALLOCATED(RemoteXRViewProxy);
public:
    static Ref<RemoteXRViewProxy> create(RemoteDeviceProxy& parent, ConvertToBackingContext& convertToBackingContext, WebGPUIdentifier identifier)
    {
        return adoptRef(*new RemoteXRViewProxy(parent, convertToBackingContext, identifier));
    }

    virtual ~RemoteXRViewProxy();

    RemoteDeviceProxy& parent() const { return m_parent; }
    RemoteGPUProxy& root() const { return m_parent->root(); }

private:
    friend class DowncastConvertToBackingContext;

    RemoteXRViewProxy(RemoteDeviceProxy&, ConvertToBackingContext&, WebGPUIdentifier);

    RemoteXRViewProxy(const RemoteXRViewProxy&) = delete;
    RemoteXRViewProxy(RemoteXRViewProxy&&) = delete;
    RemoteXRViewProxy& operator=(const RemoteXRViewProxy&) = delete;
    RemoteXRViewProxy& operator=(RemoteXRViewProxy&&) = delete;

    bool isRemoteXRViewProxy() const final { return true; }

    WebGPUIdentifier backing() const { return m_backing; }

    template<typename T>
    [[nodiscard]] IPC::Error send(T&& message)
    {
        return root().protectedStreamClientConnection()->send(WTF::move(message), backing());
    }
    template<typename T>
    [[nodiscard]] IPC::Connection::SendSyncResult<T> sendSync(T&& message)
    {
        return root().protectedStreamClientConnection()->sendSync(WTF::move(message), backing());
    }

    WebGPUIdentifier m_backing;
    const Ref<ConvertToBackingContext> m_convertToBackingContext;
    const Ref<RemoteDeviceProxy> m_parent;
};

} // namespace WebKit::WebGPU

SPECIALIZE_TYPE_TRAITS_BEGIN(WebKit::WebGPU::RemoteXRViewProxy)
    static bool isType(const WebCore::WebGPU::XRView& view) { return view.isRemoteXRViewProxy(); }
SPECIALIZE_TYPE_TRAITS_END()

#endif // ENABLE(GPU_PROCESS)
