/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Himanshi Mathur <himanshi18037@iiitd.ac.in>
 *   2020      Hidayat Khan <huk2209@gmail.com>
 */

#define SIMDE_TEST_X86_AVX512_INSN sub

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/set.h>
#include <simde/x86/avx512/sub.h>

static int
test_simde_mm_mask_sub_epi16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int16_t src[8];
    const simde__mmask8 k;
    const int16_t a[8];
    const int16_t b[8];
    const int16_t r[8];
  } test_vec[] = {
{ { -INT16_C( 19530), -INT16_C(  9691),  INT16_C(  7784),  INT16_C( 15633), -INT16_C( 15335), -INT16_C(   957), -INT16_C( 15585), -INT16_C(  4238) },
      UINT8_C( 81),
      {  INT16_C( 10517),  INT16_C( 12352),  INT16_C(  3674),  INT16_C( 15175),  INT16_C(  2236), -INT16_C( 32616),  INT16_C( 19552),  INT16_C(  4918) },
      {  INT16_C(  4465), -INT16_C( 28548), -INT16_C( 18142), -INT16_C(  6231), -INT16_C( 23043), -INT16_C( 16378), -INT16_C(  2792),  INT16_C( 11537) },
      {  INT16_C(  6052), -INT16_C(  9691),  INT16_C(  7784),  INT16_C( 15633),  INT16_C( 25279), -INT16_C(   957),  INT16_C( 22344), -INT16_C(  4238) } },
    { {  INT16_C( 20766),  INT16_C( 30813), -INT16_C( 23201),  INT16_C(  7091),  INT16_C( 19629),  INT16_C(  3483), -INT16_C( 11624),  INT16_C(  2593) },
      UINT8_C(227),
      { -INT16_C( 25955),  INT16_C( 22021), -INT16_C(  5053), -INT16_C(  5805),  INT16_C(  5106), -INT16_C(  6143),  INT16_C( 11812),  INT16_C( 29958) },
      {  INT16_C( 32651),  INT16_C( 12500), -INT16_C(  4046),  INT16_C( 32478), -INT16_C(  5237),  INT16_C( 23831),  INT16_C(  8460), -INT16_C( 22208) },
      {  INT16_C(  6930),  INT16_C(  9521), -INT16_C( 23201),  INT16_C(  7091),  INT16_C( 19629), -INT16_C( 29974),  INT16_C(  3352), -INT16_C( 13370) } },
    { {  INT16_C( 18107), -INT16_C(   512),  INT16_C( 21298),  INT16_C(  9703), -INT16_C(  6041), -INT16_C( 29939),  INT16_C(  4886), -INT16_C( 24063) },
      UINT8_C(146),
      { -INT16_C( 11563), -INT16_C( 14907),  INT16_C( 17328), -INT16_C( 25519), -INT16_C( 20902),  INT16_C( 31656),  INT16_C( 21231),  INT16_C( 13622) },
      {  INT16_C( 13650), -INT16_C( 23193), -INT16_C( 29668),  INT16_C(  1292), -INT16_C( 26471), -INT16_C( 21221), -INT16_C( 16999),  INT16_C( 28223) },
      {  INT16_C( 18107),  INT16_C(  8286),  INT16_C( 21298),  INT16_C(  9703),  INT16_C(  5569), -INT16_C( 29939),  INT16_C(  4886), -INT16_C( 14601) } },
    { {  INT16_C(  1168),  INT16_C( 16436), -INT16_C( 31416), -INT16_C( 23844), -INT16_C( 31437),  INT16_C(  8734),  INT16_C( 21719),  INT16_C( 10583) },
      UINT8_C(137),
      { -INT16_C( 12609),  INT16_C( 19366), -INT16_C( 21541),  INT16_C( 29669), -INT16_C( 27962), -INT16_C( 31732),  INT16_C( 31441), -INT16_C( 10732) },
      {  INT16_C( 21678),  INT16_C( 13086), -INT16_C( 16335), -INT16_C( 18841), -INT16_C( 30242),  INT16_C( 13197), -INT16_C( 18719), -INT16_C( 24388) },
      {  INT16_C( 31249),  INT16_C( 16436), -INT16_C( 31416), -INT16_C( 17026), -INT16_C( 31437),  INT16_C(  8734),  INT16_C( 21719),  INT16_C( 13656) } },
    { {  INT16_C( 25220),  INT16_C( 24555), -INT16_C( 12275), -INT16_C( 11054), -INT16_C(  8606),  INT16_C( 13400),  INT16_C( 27737),  INT16_C(  1802) },
      UINT8_C(192),
      {  INT16_C( 15144), -INT16_C(  5903), -INT16_C( 22622),  INT16_C( 11207), -INT16_C(  1484), -INT16_C(  5620), -INT16_C( 21322),  INT16_C(  6511) },
      { -INT16_C( 12648),  INT16_C( 26662), -INT16_C(  1375),  INT16_C( 32715), -INT16_C(   174), -INT16_C( 16680), -INT16_C(  8183),  INT16_C( 12671) },
      {  INT16_C( 25220),  INT16_C( 24555), -INT16_C( 12275), -INT16_C( 11054), -INT16_C(  8606),  INT16_C( 13400), -INT16_C( 13139), -INT16_C(  6160) } },
    { {  INT16_C( 28699), -INT16_C( 17127), -INT16_C(  8168),  INT16_C( 19688), -INT16_C(  2598), -INT16_C( 28361), -INT16_C( 22879),  INT16_C( 14762) },
      UINT8_C(116),
      { -INT16_C( 23856), -INT16_C( 13547), -INT16_C( 27283),  INT16_C( 27677), -INT16_C(  9107),  INT16_C( 19829), -INT16_C( 22949), -INT16_C( 13464) },
      {  INT16_C(  9663), -INT16_C( 24349),  INT16_C( 12302),  INT16_C(   890),  INT16_C(  2919),  INT16_C(  3492), -INT16_C(  8523), -INT16_C( 31103) },
      {  INT16_C( 28699), -INT16_C( 17127),  INT16_C( 25951),  INT16_C( 19688), -INT16_C( 12026),  INT16_C( 16337), -INT16_C( 14426),  INT16_C( 14762) } },
    { { -INT16_C( 26752), -INT16_C(  4783),  INT16_C( 28204), -INT16_C( 26279), -INT16_C( 12726), -INT16_C( 23065),  INT16_C( 20340),  INT16_C( 13169) },
      UINT8_C(117),
      { -INT16_C( 11436), -INT16_C( 31613), -INT16_C( 31154),  INT16_C( 23019), -INT16_C(  2006),  INT16_C(  2063), -INT16_C( 27270),  INT16_C(  4488) },
      {  INT16_C( 30182),  INT16_C( 21565), -INT16_C( 10546), -INT16_C( 25441),  INT16_C( 17597),  INT16_C(  3344),  INT16_C( 17589),  INT16_C(  2690) },
      {  INT16_C( 23918), -INT16_C(  4783), -INT16_C( 20608), -INT16_C( 26279), -INT16_C( 19603), -INT16_C(  1281),  INT16_C( 20677),  INT16_C( 13169) } },
    { {  INT16_C(  1303),  INT16_C( 25998),  INT16_C( 31371), -INT16_C( 19009), -INT16_C( 12686), -INT16_C(  4930),  INT16_C( 18019),  INT16_C( 18941) },
      UINT8_C(188),
      { -INT16_C( 25286),  INT16_C(  4490),  INT16_C( 10044), -INT16_C( 32306), -INT16_C(  9417),  INT16_C( 31542),  INT16_C( 16477),  INT16_C( 25235) },
      { -INT16_C(  1841),  INT16_C( 18925), -INT16_C( 23625), -INT16_C( 31301), -INT16_C( 22431), -INT16_C( 22552),  INT16_C( 12709), -INT16_C(  8093) },
      {  INT16_C(  1303),  INT16_C( 25998), -INT16_C( 31867), -INT16_C(  1005),  INT16_C( 13014), -INT16_C( 11442),  INT16_C( 18019), -INT16_C( 32208) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i src = simde_mm_loadu_epi16(test_vec[i].src);
    simde__m128i a = simde_mm_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_mm_loadu_epi16(test_vec[i].b);
    simde__m128i r = simde_mm_mask_sub_epi16(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x8(r, simde_mm_loadu_epi16(test_vec[i].r));
  }
  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m128i src = simde_test_x86_random_i16x8();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i16x8();
    simde__m128i b = simde_test_x86_random_i16x8();
    simde__m128i r = simde_mm_mask_sub_epi16(src, k, a, b);

    simde_test_x86_write_i16x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm_maskz_sub_epi16 (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int16_t a[8];
    const int16_t b[8];
    const int16_t r[8];
  } test_vec[] = {
    { UINT8_C(129),
      { -INT16_C( 12892), -INT16_C(  9446), -INT16_C(  4636),  INT16_C(  9642), -INT16_C(  9968),  INT16_C( 22217),  INT16_C( 21558),  INT16_C( 22438) },
      { -INT16_C( 25702), -INT16_C(  4078),  INT16_C( 15860),  INT16_C(  6091), -INT16_C(  4295),  INT16_C( 15654),  INT16_C( 19422), -INT16_C( 31810) },
      {  INT16_C( 12810),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0), -INT16_C( 11288) } },
    { UINT8_C( 24),
      {  INT16_C( 24280), -INT16_C( 14851),  INT16_C(  8712), -INT16_C(  7722),  INT16_C( 11499),  INT16_C( 16408),  INT16_C( 28626),  INT16_C( 28122) },
      { -INT16_C( 13695), -INT16_C( 16542),  INT16_C( 31125), -INT16_C( 31240),  INT16_C( 13983), -INT16_C(  5277), -INT16_C(  6412), -INT16_C( 13053) },
      {  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C( 23518), -INT16_C(  2484),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } },
    { UINT8_C( 69),
      { -INT16_C( 28160),  INT16_C(  9037),  INT16_C( 12136), -INT16_C( 27378),  INT16_C( 20039), -INT16_C( 18841), -INT16_C( 10968), -INT16_C(  3529) },
      { -INT16_C(  2505), -INT16_C( 20344),  INT16_C(  3567),  INT16_C(  9552),  INT16_C( 15216),  INT16_C( 22297), -INT16_C(  6594),  INT16_C( 16284) },
      { -INT16_C( 25655),  INT16_C(     0),  INT16_C(  8569),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0), -INT16_C(  4374),  INT16_C(     0) } },
    { UINT8_C(121),
      {  INT16_C( 25321),  INT16_C(  6369),  INT16_C( 30320), -INT16_C( 16545),  INT16_C(  5598), -INT16_C( 19481), -INT16_C(  9651),  INT16_C( 17386) },
      { -INT16_C( 26014),  INT16_C( 28466),  INT16_C( 22506),  INT16_C(  9695),  INT16_C( 13937),  INT16_C( 22372), -INT16_C( 23598), -INT16_C( 17200) },
      { -INT16_C( 14201),  INT16_C(     0),  INT16_C(     0), -INT16_C( 26240), -INT16_C(  8339),  INT16_C( 23683),  INT16_C( 13947),  INT16_C(     0) } },
    { UINT8_C(  5),
      { -INT16_C( 11086),  INT16_C( 10357),  INT16_C( 13364),  INT16_C( 18694), -INT16_C( 18148), -INT16_C(  2410), -INT16_C(  9565),  INT16_C( 15960) },
      { -INT16_C( 14580),  INT16_C( 25640),  INT16_C( 20134), -INT16_C(  8747),  INT16_C( 11442),  INT16_C( 21935),  INT16_C( 27645), -INT16_C( 20646) },
      {  INT16_C(  3494),  INT16_C(     0), -INT16_C(  6770),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } },
    { UINT8_C( 64),
      { -INT16_C( 10289),  INT16_C(  1140), -INT16_C( 16930), -INT16_C( 26848),  INT16_C(  5716),  INT16_C( 11835),  INT16_C( 31086),  INT16_C( 13626) },
      { -INT16_C( 24927), -INT16_C(  4133), -INT16_C( 18317), -INT16_C( 24415), -INT16_C(  2456), -INT16_C( 11363),  INT16_C( 19536),  INT16_C(  8211) },
      {  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C( 11550),  INT16_C(     0) } },
    { UINT8_C( 35),
      {  INT16_C(  9351),  INT16_C( 17665), -INT16_C( 26300),  INT16_C( 23193), -INT16_C( 14380),  INT16_C( 19912), -INT16_C(   767), -INT16_C( 24338) },
      { -INT16_C(  8488), -INT16_C( 28397), -INT16_C( 19585),  INT16_C( 30457), -INT16_C( 13232), -INT16_C( 25402), -INT16_C(  6432),  INT16_C( 26560) },
      {  INT16_C( 17839), -INT16_C( 19474),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0), -INT16_C( 20222),  INT16_C(     0),  INT16_C(     0) } },
    { UINT8_C( 10),
      { -INT16_C( 21311),  INT16_C( 23118), -INT16_C( 22459),  INT16_C(  3118),  INT16_C( 31600),  INT16_C( 27918), -INT16_C( 20886),  INT16_C( 18502) },
      { -INT16_C( 10303),  INT16_C( 30151),  INT16_C( 15824), -INT16_C( 25403),  INT16_C( 25092), -INT16_C(  5508), -INT16_C(  7134), -INT16_C(  7179) },
      {  INT16_C(     0), -INT16_C(  7033),  INT16_C(     0),  INT16_C( 28521),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0),  INT16_C(     0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m128i a = simde_x_mm_loadu_epi16(test_vec[i].a);
    simde__m128i b = simde_x_mm_loadu_epi16(test_vec[i].b);
    simde__m128i r = simde_mm_maskz_sub_epi16(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i16x8(r, simde_x_mm_loadu_epi16(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m128i a = simde_test_x86_random_i16x8();
    simde__m128i b = simde_test_x86_random_i16x8();
    simde__m128i r = simde_mm_maskz_sub_epi16(k, a, b);

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i16x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i16x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_sub_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi8(INT8_C(  82), INT8_C(  83), INT8_C( 117), INT8_C(  65),
                           INT8_C( -47), INT8_C(-122), INT8_C( 116), INT8_C(  14),
                           INT8_C(  76), INT8_C(   1), INT8_C( -50), INT8_C(   4),
                           INT8_C(  83), INT8_C( -77), INT8_C( 112), INT8_C(-102),
                           INT8_C(  99), INT8_C(-118), INT8_C( -47), INT8_C( -67),
                           INT8_C(  60), INT8_C( -34), INT8_C(  78), INT8_C(-110),
                           INT8_C( -58), INT8_C(  87), INT8_C( -61), INT8_C(  26),
                           INT8_C( -17), INT8_C( -46), INT8_C( 116), INT8_C( -20),
                           INT8_C(-120), INT8_C(  48), INT8_C(  24), INT8_C(  46),
                           INT8_C( 103), INT8_C( -34), INT8_C(  42), INT8_C(  51),
                           INT8_C( -65), INT8_C(  86), INT8_C(  61), INT8_C( -56),
                           INT8_C(  58), INT8_C( 119), INT8_C(  93), INT8_C(  -1),
                           INT8_C( -58), INT8_C(-121), INT8_C( -32), INT8_C(  98),
                           INT8_C( -66), INT8_C(  79), INT8_C(  99), INT8_C( -93),
                           INT8_C(  77), INT8_C( -56), INT8_C( -78), INT8_C(  39),
                           INT8_C( -50), INT8_C( -60), INT8_C( -68), INT8_C(  -4)),
      simde_mm512_set_epi8(INT8_C( 106), INT8_C(  39), INT8_C(-106), INT8_C(  14),
                           INT8_C( -28), INT8_C(  -2), INT8_C(  39), INT8_C( -28),
                           INT8_C(  54), INT8_C(  70), INT8_C( -19), INT8_C( -12),
                           INT8_C( -42), INT8_C(  28), INT8_C( -13), INT8_C(  -6),
                           INT8_C( 116), INT8_C(   2), INT8_C(  23), INT8_C( 121),
                           INT8_C( 112), INT8_C( -35), INT8_C(-124), INT8_C(  10),
                           INT8_C( -16), INT8_C(-117), INT8_C(  26), INT8_C(-125),
                           INT8_C(  36), INT8_C( 109), INT8_C(  29), INT8_C( -35),
                           INT8_C(  -9), INT8_C( -85), INT8_C( -38), INT8_C(  95),
                           INT8_C( -88), INT8_C(   3), INT8_C(   4), INT8_C( 100),
                           INT8_C(  85), INT8_C(  21), INT8_C(  66), INT8_C( -33),
                           INT8_C( -77), INT8_C(  -5), INT8_C(-126), INT8_C( 122),
                           INT8_C( -30), INT8_C( -83), INT8_C(  74), INT8_C(-108),
                           INT8_C(  83), INT8_C( -96), INT8_C( -57), INT8_C(-108),
                           INT8_C(  89), INT8_C(  59), INT8_C( 111), INT8_C( -47),
                           INT8_C( -10), INT8_C( -18), INT8_C(  20), INT8_C(-125)),
      simde_mm512_set_epi8(INT8_C( -24), INT8_C(  44), INT8_C( -33), INT8_C(  51),
                           INT8_C( -19), INT8_C(-120), INT8_C(  77), INT8_C(  42),
                           INT8_C(  22), INT8_C( -69), INT8_C( -31), INT8_C(  16),
                           INT8_C( 125), INT8_C(-105), INT8_C( 125), INT8_C( -96),
                           INT8_C( -17), INT8_C(-120), INT8_C( -70), INT8_C(  68),
                           INT8_C( -52), INT8_C(   1), INT8_C( -54), INT8_C(-120),
                           INT8_C( -42), INT8_C( -52), INT8_C( -87), INT8_C(-105),
                           INT8_C( -53), INT8_C( 101), INT8_C(  87), INT8_C(  15),
                           INT8_C(-111), INT8_C(-123), INT8_C(  62), INT8_C( -49),
                           INT8_C( -65), INT8_C( -37), INT8_C(  38), INT8_C( -49),
                           INT8_C( 106), INT8_C(  65), INT8_C(  -5), INT8_C( -23),
                           INT8_C(-121), INT8_C( 124), INT8_C( -37), INT8_C(-123),
                           INT8_C( -28), INT8_C( -38), INT8_C(-106), INT8_C( -50),
                           INT8_C( 107), INT8_C( -81), INT8_C(-100), INT8_C(  15),
                           INT8_C( -12), INT8_C(-115), INT8_C(  67), INT8_C(  86),
                           INT8_C( -40), INT8_C( -42), INT8_C( -88), INT8_C( 121)) },
    { simde_mm512_set_epi8(INT8_C(-108), INT8_C(-116), INT8_C(  21), INT8_C(-123),
                           INT8_C( -53), INT8_C(  42), INT8_C(  66), INT8_C(  13),
                           INT8_C(   9), INT8_C( 115), INT8_C(  86), INT8_C( 126),
                           INT8_C( -24), INT8_C(  35), INT8_C(  -5), INT8_C( 103),
                           INT8_C(  38), INT8_C( 111), INT8_C(  24), INT8_C( -71),
                           INT8_C(  -1), INT8_C(  17), INT8_C( -63), INT8_C( -13),
                           INT8_C(  14), INT8_C(  82), INT8_C(  78), INT8_C(-102),
                           INT8_C(  -7), INT8_C(  93), INT8_C(  25), INT8_C( 103),
                           INT8_C( 113), INT8_C( -15), INT8_C( -19), INT8_C( -73),
                           INT8_C( -11), INT8_C( 103), INT8_C( -97), INT8_C( 123),
                           INT8_C(  28), INT8_C(  53), INT8_C( -15), INT8_C( 122),
                           INT8_C(   3), INT8_C( -54), INT8_C( -61), INT8_C(  58),
                           INT8_C( -44), INT8_C(  -3), INT8_C( -43), INT8_C( -35),
                           INT8_C(-118), INT8_C( -18), INT8_C(  15), INT8_C(  54),
                           INT8_C(-102), INT8_C( -58), INT8_C( -74), INT8_C( -70),
                           INT8_C(  46), INT8_C(  48), INT8_C( -35), INT8_C(  92)),
      simde_mm512_set_epi8(INT8_C(   6), INT8_C(  68), INT8_C(  77), INT8_C( -94),
                           INT8_C( -48), INT8_C(-101), INT8_C(  -8), INT8_C(  82),
                           INT8_C(  50), INT8_C( -15), INT8_C(   6), INT8_C(  30),
                           INT8_C( -47), INT8_C( -15), INT8_C( -14), INT8_C( -97),
                           INT8_C(  28), INT8_C( -47), INT8_C( -92), INT8_C( -84),
                           INT8_C( -37), INT8_C( -33), INT8_C(-123), INT8_C( -19),
                           INT8_C(  58), INT8_C(  29), INT8_C(  93), INT8_C( -55),
                           INT8_C(-127), INT8_C( -60), INT8_C(  32), INT8_C( 116),
                           INT8_C( -46), INT8_C(  51), INT8_C( -40), INT8_C(  10),
                           INT8_C(   4), INT8_C(  50), INT8_C(  48), INT8_C(  53),
                           INT8_C(  78), INT8_C(  21), INT8_C(  64), INT8_C( 107),
                           INT8_C(  16), INT8_C(  48), INT8_C( -46), INT8_C(  62),
                           INT8_C(  75), INT8_C(  85), INT8_C(-115), INT8_C( -14),
                           INT8_C( -99), INT8_C(  86), INT8_C(-116), INT8_C( -74),
                           INT8_C(  38), INT8_C(  27), INT8_C(-115), INT8_C(  55),
                           INT8_C( -91), INT8_C( -71), INT8_C( -14), INT8_C( -84)),
      simde_mm512_set_epi8(INT8_C(-114), INT8_C(  72), INT8_C( -56), INT8_C( -29),
                           INT8_C(  -5), INT8_C(-113), INT8_C(  74), INT8_C( -69),
                           INT8_C( -41), INT8_C(-126), INT8_C(  80), INT8_C(  96),
                           INT8_C(  23), INT8_C(  50), INT8_C(   9), INT8_C( -56),
                           INT8_C(  10), INT8_C( -98), INT8_C( 116), INT8_C(  13),
                           INT8_C(  36), INT8_C(  50), INT8_C(  60), INT8_C(   6),
                           INT8_C( -44), INT8_C(  53), INT8_C( -15), INT8_C( -47),
                           INT8_C( 120), INT8_C(-103), INT8_C(  -7), INT8_C( -13),
                           INT8_C( -97), INT8_C( -66), INT8_C(  21), INT8_C( -83),
                           INT8_C( -15), INT8_C(  53), INT8_C( 111), INT8_C(  70),
                           INT8_C( -50), INT8_C(  32), INT8_C( -79), INT8_C(  15),
                           INT8_C( -13), INT8_C(-102), INT8_C( -15), INT8_C(  -4),
                           INT8_C(-119), INT8_C( -88), INT8_C(  72), INT8_C( -21),
                           INT8_C( -19), INT8_C(-104), INT8_C(-125), INT8_C(-128),
                           INT8_C( 116), INT8_C( -85), INT8_C(  41), INT8_C(-125),
                           INT8_C(-119), INT8_C( 119), INT8_C( -21), INT8_C( -80)) },
    { simde_mm512_set_epi8(INT8_C(   2), INT8_C( -77), INT8_C( -19), INT8_C(  41),
                           INT8_C( -13), INT8_C(  75), INT8_C(-123), INT8_C(  96),
                           INT8_C( -86), INT8_C( -24), INT8_C( -27), INT8_C( -84),
                           INT8_C(  35), INT8_C( -86), INT8_C( -72), INT8_C( -97),
                           INT8_C(  44), INT8_C(  11), INT8_C(-106), INT8_C(  44),
                           INT8_C(   0), INT8_C(  90), INT8_C( -79), INT8_C(  91),
                           INT8_C( 119), INT8_C(  59), INT8_C( 105), INT8_C(-128),
                           INT8_C( 110), INT8_C( -29), INT8_C(  67), INT8_C( 114),
                           INT8_C( -39), INT8_C( -49), INT8_C( 105), INT8_C( -40),
                           INT8_C( -33), INT8_C( 120), INT8_C( -27), INT8_C( 100),
                           INT8_C( -90), INT8_C(  86), INT8_C( -18), INT8_C( -57),
                           INT8_C(  84), INT8_C( -26), INT8_C( -77), INT8_C(  17),
                           INT8_C( -47), INT8_C(  51), INT8_C( -83), INT8_C(  53),
                           INT8_C(  71), INT8_C(  96), INT8_C( 110), INT8_C( -89),
                           INT8_C(  27), INT8_C( -45), INT8_C(-126), INT8_C(  40),
                           INT8_C(  95), INT8_C( -87), INT8_C( -62), INT8_C( -52)),
      simde_mm512_set_epi8(INT8_C( -84), INT8_C( 127), INT8_C(  61), INT8_C( -16),
                           INT8_C(  30), INT8_C(   6), INT8_C(-112), INT8_C( 104),
                           INT8_C( -60), INT8_C( -88), INT8_C( -39), INT8_C( -19),
                           INT8_C(  44), INT8_C(  36), INT8_C( 105), INT8_C( 120),
                           INT8_C( -26), INT8_C(  21), INT8_C(  14), INT8_C(  42),
                           INT8_C(  49), INT8_C( -84), INT8_C(-120), INT8_C(-107),
                           INT8_C( 123), INT8_C( -47), INT8_C(  21), INT8_C( -10),
                           INT8_C(  95), INT8_C( 124), INT8_C( -33), INT8_C( -34),
                           INT8_C( -33), INT8_C( -71), INT8_C(  11), INT8_C(  74),
                           INT8_C( 104), INT8_C( 108), INT8_C( -35), INT8_C( -59),
                           INT8_C( -55), INT8_C(-126), INT8_C( 107), INT8_C(  23),
                           INT8_C(  29), INT8_C( -27), INT8_C( 123), INT8_C(  23),
                           INT8_C( -83), INT8_C( -90), INT8_C(   9), INT8_C(  94),
                           INT8_C(  91), INT8_C(  69), INT8_C( -51), INT8_C(-103),
                           INT8_C( -72), INT8_C( -45), INT8_C(  16), INT8_C( 108),
                           INT8_C( -80), INT8_C(  27), INT8_C(  58), INT8_C( -83)),
      simde_mm512_set_epi8(INT8_C(  86), INT8_C(  52), INT8_C( -80), INT8_C(  57),
                           INT8_C( -43), INT8_C(  69), INT8_C( -11), INT8_C(  -8),
                           INT8_C( -26), INT8_C(  64), INT8_C(  12), INT8_C( -65),
                           INT8_C(  -9), INT8_C(-122), INT8_C(  79), INT8_C(  39),
                           INT8_C(  70), INT8_C( -10), INT8_C(-120), INT8_C(   2),
                           INT8_C( -49), INT8_C( -82), INT8_C(  41), INT8_C( -58),
                           INT8_C(  -4), INT8_C( 106), INT8_C(  84), INT8_C(-118),
                           INT8_C(  15), INT8_C( 103), INT8_C( 100), INT8_C(-108),
                           INT8_C(  -6), INT8_C(  22), INT8_C(  94), INT8_C(-114),
                           INT8_C( 119), INT8_C(  12), INT8_C(   8), INT8_C( -97),
                           INT8_C( -35), INT8_C( -44), INT8_C(-125), INT8_C( -80),
                           INT8_C(  55), INT8_C(   1), INT8_C(  56), INT8_C(  -6),
                           INT8_C(  36), INT8_C(-115), INT8_C( -92), INT8_C( -41),
                           INT8_C( -20), INT8_C(  27), INT8_C( -95), INT8_C(  14),
                           INT8_C(  99), INT8_C(   0), INT8_C( 114), INT8_C( -68),
                           INT8_C( -81), INT8_C(-114), INT8_C(-120), INT8_C(  31)) },
    { simde_mm512_set_epi8(INT8_C(  17), INT8_C(  99), INT8_C( -13), INT8_C( -49),
                           INT8_C(  45), INT8_C(-128), INT8_C(  55), INT8_C( 105),
                           INT8_C( -34), INT8_C( -51), INT8_C( -97), INT8_C(-103),
                           INT8_C(-124), INT8_C( 111), INT8_C(  74), INT8_C(  75),
                           INT8_C( 102), INT8_C(  98), INT8_C(-117), INT8_C(   9),
                           INT8_C( -74), INT8_C(  61), INT8_C(  99), INT8_C( 124),
                           INT8_C(  79), INT8_C(-114), INT8_C(  19), INT8_C(  97),
                           INT8_C(-100), INT8_C(-124), INT8_C( -17), INT8_C( -62),
                           INT8_C(  25), INT8_C(  -3), INT8_C(  -7), INT8_C(  72),
                           INT8_C(-117), INT8_C( -27), INT8_C( -56), INT8_C(  92),
                           INT8_C( -20), INT8_C( -53), INT8_C(   2), INT8_C( -38),
                           INT8_C( -81), INT8_C(  59), INT8_C(  66), INT8_C(  90),
                           INT8_C(  36), INT8_C( 100), INT8_C( 112), INT8_C( 123),
                           INT8_C( -72), INT8_C( -97), INT8_C(-115), INT8_C(  17),
                           INT8_C( -93), INT8_C(-122), INT8_C(  31), INT8_C(  27),
                           INT8_C( 109), INT8_C( 115), INT8_C(  53), INT8_C( -96)),
      simde_mm512_set_epi8(INT8_C( -43), INT8_C( -18), INT8_C( 114), INT8_C( -29),
                           INT8_C( 118), INT8_C(  -1), INT8_C( -20), INT8_C( -38),
                           INT8_C( -80), INT8_C(  88), INT8_C(-111), INT8_C( -91),
                           INT8_C(  44), INT8_C( -72), INT8_C( 106), INT8_C(  19),
                           INT8_C( -46), INT8_C( 107), INT8_C(  46), INT8_C(  44),
                           INT8_C( -65), INT8_C(-128), INT8_C(  41), INT8_C(  44),
                           INT8_C(  68), INT8_C(  69), INT8_C( -78), INT8_C( -47),
                           INT8_C( 109), INT8_C( 120), INT8_C( -57), INT8_C( -95),
                           INT8_C(  95), INT8_C(  80), INT8_C( -30), INT8_C(  97),
                           INT8_C( -48), INT8_C( -97), INT8_C( 111), INT8_C( -80),
                           INT8_C(-122), INT8_C( -81), INT8_C( -71), INT8_C(  85),
                           INT8_C(  77), INT8_C( -42), INT8_C(-115), INT8_C( -77),
                           INT8_C(  29), INT8_C(  77), INT8_C(  64), INT8_C( -20),
                           INT8_C(  27), INT8_C(  41), INT8_C(  13), INT8_C( 109),
                           INT8_C(  22), INT8_C( -98), INT8_C(  20), INT8_C( -28),
                           INT8_C(  66), INT8_C(  -7), INT8_C(-113), INT8_C(-119)),
      simde_mm512_set_epi8(INT8_C(  60), INT8_C( 117), INT8_C(-127), INT8_C( -20),
                           INT8_C( -73), INT8_C(-127), INT8_C(  75), INT8_C(-113),
                           INT8_C(  46), INT8_C( 117), INT8_C(  14), INT8_C( -12),
                           INT8_C(  88), INT8_C( -73), INT8_C( -32), INT8_C(  56),
                           INT8_C(-108), INT8_C(  -9), INT8_C(  93), INT8_C( -35),
                           INT8_C(  -9), INT8_C( -67), INT8_C(  58), INT8_C(  80),
                           INT8_C(  11), INT8_C(  73), INT8_C(  97), INT8_C(-112),
                           INT8_C(  47), INT8_C(  12), INT8_C(  40), INT8_C(  33),
                           INT8_C( -70), INT8_C( -83), INT8_C(  23), INT8_C( -25),
                           INT8_C( -69), INT8_C(  70), INT8_C(  89), INT8_C( -84),
                           INT8_C( 102), INT8_C(  28), INT8_C(  73), INT8_C(-123),
                           INT8_C(  98), INT8_C( 101), INT8_C( -75), INT8_C( -89),
                           INT8_C(   7), INT8_C(  23), INT8_C(  48), INT8_C(-113),
                           INT8_C( -99), INT8_C( 118), INT8_C(-128), INT8_C( -92),
                           INT8_C(-115), INT8_C( -24), INT8_C(  11), INT8_C(  55),
                           INT8_C(  43), INT8_C( 122), INT8_C( -90), INT8_C(  23)) },
    { simde_mm512_set_epi8(INT8_C(-124), INT8_C( -73), INT8_C(  74), INT8_C(   5),
                           INT8_C(  -9), INT8_C(  17), INT8_C( -81), INT8_C( -54),
                           INT8_C(  -5), INT8_C( -33), INT8_C( -12), INT8_C(  26),
                           INT8_C(  86), INT8_C( 122), INT8_C( -44), INT8_C( -23),
                           INT8_C(   0), INT8_C(  43), INT8_C( -25), INT8_C(-122),
                           INT8_C( -79), INT8_C(-122), INT8_C( -88), INT8_C(-121),
                           INT8_C(-102), INT8_C(  66), INT8_C( -93), INT8_C( 105),
                           INT8_C( 109), INT8_C( -68), INT8_C(  24), INT8_C( -54),
                           INT8_C(  40), INT8_C(  68), INT8_C(   2), INT8_C(  60),
                           INT8_C(   0), INT8_C(   5), INT8_C(  59), INT8_C( -54),
                           INT8_C( -76), INT8_C(  27), INT8_C( -23), INT8_C(  77),
                           INT8_C(-108), INT8_C( -28), INT8_C(-114), INT8_C(  56),
                           INT8_C( -54), INT8_C(-108), INT8_C( -15), INT8_C( -89),
                           INT8_C(-103), INT8_C( -45), INT8_C(  74), INT8_C(  -3),
                           INT8_C(-108), INT8_C(  55), INT8_C( -79), INT8_C( -62),
                           INT8_C(  14), INT8_C( 106), INT8_C( -16), INT8_C( -10)),
      simde_mm512_set_epi8(INT8_C( -47), INT8_C( 124), INT8_C(  57), INT8_C( -74),
                           INT8_C(  20), INT8_C( 124), INT8_C(  70), INT8_C( -69),
                           INT8_C( -65), INT8_C( -12), INT8_C( 124), INT8_C( -90),
                           INT8_C(-113), INT8_C(  63), INT8_C( -79), INT8_C( -70),
                           INT8_C( -76), INT8_C( -34), INT8_C( -60), INT8_C(  -4),
                           INT8_C( -41), INT8_C(  60), INT8_C(  77), INT8_C( -57),
                           INT8_C(  13), INT8_C(   2), INT8_C( 111), INT8_C( -39),
                           INT8_C(  41), INT8_C(  54), INT8_C( -37), INT8_C( 114),
                           INT8_C(  92), INT8_C(-111), INT8_C(  77), INT8_C(  14),
                           INT8_C(-104), INT8_C( -39), INT8_C( -74), INT8_C(  66),
                           INT8_C(  16), INT8_C( -26), INT8_C( -89), INT8_C(-114),
                           INT8_C( -68), INT8_C(   6), INT8_C(  62), INT8_C( -93),
                           INT8_C(  55), INT8_C(-113), INT8_C( -60), INT8_C( -56),
                           INT8_C( -37), INT8_C(   2), INT8_C( -15), INT8_C(  88),
                           INT8_C(  26), INT8_C(  54), INT8_C(  82), INT8_C( 124),
                           INT8_C( -38), INT8_C(-107), INT8_C(  40), INT8_C(  13)),
      simde_mm512_set_epi8(INT8_C( -77), INT8_C(  59), INT8_C(  17), INT8_C(  79),
                           INT8_C( -29), INT8_C(-107), INT8_C( 105), INT8_C(  15),
                           INT8_C(  60), INT8_C( -21), INT8_C( 120), INT8_C( 116),
                           INT8_C( -57), INT8_C(  59), INT8_C(  35), INT8_C(  47),
                           INT8_C(  76), INT8_C(  77), INT8_C(  35), INT8_C(-118),
                           INT8_C( -38), INT8_C(  74), INT8_C(  91), INT8_C( -64),
                           INT8_C(-115), INT8_C(  64), INT8_C(  52), INT8_C(-112),
                           INT8_C(  68), INT8_C(-122), INT8_C(  61), INT8_C(  88),
                           INT8_C( -52), INT8_C( -77), INT8_C( -75), INT8_C(  46),
                           INT8_C( 104), INT8_C(  44), INT8_C(-123), INT8_C(-120),
                           INT8_C( -92), INT8_C(  53), INT8_C(  66), INT8_C( -65),
                           INT8_C( -40), INT8_C( -34), INT8_C(  80), INT8_C(-107),
                           INT8_C(-109), INT8_C(   5), INT8_C(  45), INT8_C( -33),
                           INT8_C( -66), INT8_C( -47), INT8_C(  89), INT8_C( -91),
                           INT8_C( 122), INT8_C(   1), INT8_C(  95), INT8_C(  70),
                           INT8_C(  52), INT8_C( -43), INT8_C( -56), INT8_C( -23)) },
    { simde_mm512_set_epi8(INT8_C(   5), INT8_C( -68), INT8_C( -18), INT8_C( -37),
                           INT8_C(   5), INT8_C(  16), INT8_C(-109), INT8_C( -67),
                           INT8_C( -62), INT8_C(  -4), INT8_C(  14), INT8_C(-109),
                           INT8_C( -29), INT8_C(-121), INT8_C(-109), INT8_C( -55),
                           INT8_C(   1), INT8_C( -38), INT8_C( 107), INT8_C(  55),
                           INT8_C( -36), INT8_C( -76), INT8_C(  35), INT8_C( -40),
                           INT8_C(  10), INT8_C( -90), INT8_C( -48), INT8_C(-112),
                           INT8_C(  -9), INT8_C( -53), INT8_C( 105), INT8_C(  27),
                           INT8_C( -97), INT8_C(-124), INT8_C(   4), INT8_C( -36),
                           INT8_C( -16), INT8_C( -87), INT8_C( -89), INT8_C(-104),
                           INT8_C( -30), INT8_C(-101), INT8_C(  69), INT8_C(  79),
                           INT8_C(  59), INT8_C( -97), INT8_C( -15), INT8_C(  17),
                           INT8_C( 106), INT8_C( -85), INT8_C( 126), INT8_C(-121),
                           INT8_C( -91), INT8_C(  26), INT8_C(-115), INT8_C(-117),
                           INT8_C(  91), INT8_C(  73), INT8_C( -60), INT8_C(  69),
                           INT8_C( -23), INT8_C(  48), INT8_C(  70), INT8_C(  -8)),
      simde_mm512_set_epi8(INT8_C(  91), INT8_C(-103), INT8_C(  69), INT8_C(  61),
                           INT8_C( -82), INT8_C(  73), INT8_C( 122), INT8_C( -22),
                           INT8_C( 122), INT8_C(  76), INT8_C(  -9), INT8_C( 121),
                           INT8_C(-123), INT8_C(-119), INT8_C(-127), INT8_C( 126),
                           INT8_C( 105), INT8_C(  10), INT8_C(-120), INT8_C(-127),
                           INT8_C( -50), INT8_C(  15), INT8_C( -93), INT8_C( -86),
                           INT8_C(-125), INT8_C(  45), INT8_C( -39), INT8_C(-119),
                           INT8_C(  74), INT8_C( -92), INT8_C( -78), INT8_C(  53),
                           INT8_C(  17), INT8_C( -21), INT8_C( 105), INT8_C(-102),
                           INT8_C(  -1), INT8_C( -19), INT8_C( 110), INT8_C( -84),
                           INT8_C( -93), INT8_C(  19), INT8_C( -98), INT8_C(-128),
                           INT8_C( -23), INT8_C(  49), INT8_C( 100), INT8_C( 122),
                           INT8_C( -96), INT8_C(-103), INT8_C(  60), INT8_C( -24),
                           INT8_C(  23), INT8_C( -52), INT8_C( -37), INT8_C( -56),
                           INT8_C( -50), INT8_C(   4), INT8_C( -69), INT8_C(   1),
                           INT8_C( -25), INT8_C( -10), INT8_C(  93), INT8_C(  51)),
      simde_mm512_set_epi8(INT8_C( -86), INT8_C(  35), INT8_C( -87), INT8_C( -98),
                           INT8_C(  87), INT8_C( -57), INT8_C(  25), INT8_C( -45),
                           INT8_C(  72), INT8_C( -80), INT8_C(  23), INT8_C(  26),
                           INT8_C(  94), INT8_C(  -2), INT8_C(  18), INT8_C(  75),
                           INT8_C(-104), INT8_C( -48), INT8_C( -29), INT8_C( -74),
                           INT8_C(  14), INT8_C( -91), INT8_C(-128), INT8_C(  46),
                           INT8_C(-121), INT8_C( 121), INT8_C(  -9), INT8_C(   7),
                           INT8_C( -83), INT8_C(  39), INT8_C( -73), INT8_C( -26),
                           INT8_C(-114), INT8_C(-103), INT8_C(-101), INT8_C(  66),
                           INT8_C( -15), INT8_C( -68), INT8_C(  57), INT8_C( -20),
                           INT8_C(  63), INT8_C(-120), INT8_C( -89), INT8_C( -49),
                           INT8_C(  82), INT8_C( 110), INT8_C(-115), INT8_C(-105),
                           INT8_C( -54), INT8_C(  18), INT8_C(  66), INT8_C( -97),
                           INT8_C(-114), INT8_C(  78), INT8_C( -78), INT8_C( -61),
                           INT8_C(-115), INT8_C(  69), INT8_C(   9), INT8_C(  68),
                           INT8_C(   2), INT8_C(  58), INT8_C( -23), INT8_C( -59)) },
    { simde_mm512_set_epi8(INT8_C( -89), INT8_C(  43), INT8_C(  52), INT8_C(  82),
                           INT8_C( -37), INT8_C(  55), INT8_C( 112), INT8_C( -22),
                           INT8_C( -75), INT8_C( -36), INT8_C( -34), INT8_C( -15),
                           INT8_C(  35), INT8_C( -42), INT8_C(-101), INT8_C(  -5),
                           INT8_C(   2), INT8_C(  35), INT8_C(  14), INT8_C( -73),
                           INT8_C( -50), INT8_C( -33), INT8_C( -65), INT8_C(  94),
                           INT8_C(  -6), INT8_C( -21), INT8_C( -28), INT8_C(  21),
                           INT8_C( 102), INT8_C( -87), INT8_C( 114), INT8_C( 125),
                           INT8_C( 113), INT8_C( 124), INT8_C(-121), INT8_C(-122),
                           INT8_C(  23), INT8_C( 107), INT8_C(  24), INT8_C( 126),
                           INT8_C(  80), INT8_C(  59), INT8_C(  39), INT8_C( -61),
                           INT8_C(-105), INT8_C(  32), INT8_C(  55), INT8_C(  -9),
                           INT8_C(  60), INT8_C(-125), INT8_C(  72), INT8_C( -36),
                           INT8_C(  77), INT8_C( -65), INT8_C( 117), INT8_C( -85),
                           INT8_C(  98), INT8_C( -83), INT8_C( -69), INT8_C( -52),
                           INT8_C(  41), INT8_C( -10), INT8_C( -18), INT8_C(  56)),
      simde_mm512_set_epi8(INT8_C(  22), INT8_C( 122), INT8_C( -90), INT8_C(   2),
                           INT8_C( -65), INT8_C(  51), INT8_C( -94), INT8_C( -50),
                           INT8_C( -15), INT8_C(  19), INT8_C( -19), INT8_C(  66),
                           INT8_C( 119), INT8_C(-118), INT8_C(-112), INT8_C(-116),
                           INT8_C(  44), INT8_C( -12), INT8_C(  31), INT8_C(  43),
                           INT8_C( -16), INT8_C( -37), INT8_C( -24), INT8_C( -32),
                           INT8_C( -95), INT8_C( -86), INT8_C( -96), INT8_C(  80),
                           INT8_C(  68), INT8_C(  13), INT8_C(  -8), INT8_C(  67),
                           INT8_C( 107), INT8_C(-125), INT8_C( 104), INT8_C( -80),
                           INT8_C(  97), INT8_C( -78), INT8_C( 106), INT8_C( -53),
                           INT8_C( -36), INT8_C( -90), INT8_C(  74), INT8_C( -72),
                           INT8_C(  59), INT8_C( -81), INT8_C(  -8), INT8_C( -25),
                           INT8_C( -55), INT8_C( -99), INT8_C(  20), INT8_C(   9),
                           INT8_C( -89), INT8_C( -90), INT8_C( 108), INT8_C(  56),
                           INT8_C( -19), INT8_C(  81), INT8_C( 122), INT8_C(   6),
                           INT8_C(-119), INT8_C( 122), INT8_C( -35), INT8_C( 106)),
      simde_mm512_set_epi8(INT8_C(-111), INT8_C( -79), INT8_C(-114), INT8_C(  80),
                           INT8_C(  28), INT8_C(   4), INT8_C( -50), INT8_C(  28),
                           INT8_C( -60), INT8_C( -55), INT8_C( -15), INT8_C( -81),
                           INT8_C( -84), INT8_C(  76), INT8_C(  11), INT8_C( 111),
                           INT8_C( -42), INT8_C(  47), INT8_C( -17), INT8_C(-116),
                           INT8_C( -34), INT8_C(   4), INT8_C( -41), INT8_C( 126),
                           INT8_C(  89), INT8_C(  65), INT8_C(  68), INT8_C( -59),
                           INT8_C(  34), INT8_C(-100), INT8_C( 122), INT8_C(  58),
                           INT8_C(   6), INT8_C(  -7), INT8_C(  31), INT8_C( -42),
                           INT8_C( -74), INT8_C( -71), INT8_C( -82), INT8_C( -77),
                           INT8_C( 116), INT8_C(-107), INT8_C( -35), INT8_C(  11),
                           INT8_C(  92), INT8_C( 113), INT8_C(  63), INT8_C(  16),
                           INT8_C( 115), INT8_C( -26), INT8_C(  52), INT8_C( -45),
                           INT8_C( -90), INT8_C(  25), INT8_C(   9), INT8_C( 115),
                           INT8_C( 117), INT8_C(  92), INT8_C(  65), INT8_C( -58),
                           INT8_C( -96), INT8_C( 124), INT8_C(  17), INT8_C( -50)) },
    { simde_mm512_set_epi8(INT8_C( 105), INT8_C(-115), INT8_C( 121), INT8_C(-101),
                           INT8_C(   0), INT8_C(  63), INT8_C( -42), INT8_C( -34),
                           INT8_C(  -5), INT8_C( -47), INT8_C(-123), INT8_C( -52),
                           INT8_C( -86), INT8_C( -28), INT8_C( -63), INT8_C(  20),
                           INT8_C( -60), INT8_C( -63), INT8_C(  99), INT8_C(  78),
                           INT8_C(  56), INT8_C( -72), INT8_C( -55), INT8_C( -72),
                           INT8_C(  79), INT8_C( -81), INT8_C( 124), INT8_C( -85),
                           INT8_C( -65), INT8_C( 122), INT8_C( -25), INT8_C( -58),
                           INT8_C( -64), INT8_C(  52), INT8_C( -12), INT8_C(   1),
                           INT8_C( -62), INT8_C( -28), INT8_C( -28), INT8_C(-104),
                           INT8_C(  54), INT8_C(-103), INT8_C( -55), INT8_C( -22),
                           INT8_C( -91), INT8_C(   6), INT8_C(  -9), INT8_C( -31),
                           INT8_C(  18), INT8_C(-111), INT8_C(  58), INT8_C(  71),
                           INT8_C( -73), INT8_C( -96), INT8_C(  28), INT8_C(  -4),
                           INT8_C(  47), INT8_C(  66), INT8_C( 121), INT8_C(  38),
                           INT8_C(  69), INT8_C(-107), INT8_C( -57), INT8_C(-120)),
      simde_mm512_set_epi8(INT8_C( -49), INT8_C(  15), INT8_C( -15), INT8_C( -59),
                           INT8_C(-113), INT8_C( 102), INT8_C( -48), INT8_C( -78),
                           INT8_C(  31), INT8_C(  94), INT8_C(  79), INT8_C(  92),
                           INT8_C( 106), INT8_C( -68), INT8_C(  96), INT8_C( -97),
                           INT8_C( -27), INT8_C(-118), INT8_C( -11), INT8_C( 112),
                           INT8_C(-125), INT8_C(  70), INT8_C(  26), INT8_C( -38),
                           INT8_C( -16), INT8_C(-112), INT8_C(  10), INT8_C(  98),
                           INT8_C(  -4), INT8_C( 120), INT8_C( -33), INT8_C(-127),
                           INT8_C( -65), INT8_C( -40), INT8_C(  88), INT8_C(  -6),
                           INT8_C(  74), INT8_C(  41), INT8_C(  39), INT8_C(  79),
                           INT8_C(-125), INT8_C(  -7), INT8_C(  62), INT8_C(-112),
                           INT8_C(-119), INT8_C(  -9), INT8_C(  71), INT8_C( -68),
                           INT8_C( -79), INT8_C(  48), INT8_C( -20), INT8_C( -97),
                           INT8_C(-116), INT8_C( 120), INT8_C( -65), INT8_C(   6),
                           INT8_C( -32), INT8_C( -75), INT8_C(-106), INT8_C(  26),
                           INT8_C( -96), INT8_C(  50), INT8_C( -45), INT8_C(  16)),
      simde_mm512_set_epi8(INT8_C(-102), INT8_C( 126), INT8_C(-120), INT8_C( -42),
                           INT8_C( 113), INT8_C( -39), INT8_C(   6), INT8_C(  44),
                           INT8_C( -36), INT8_C( 115), INT8_C(  54), INT8_C( 112),
                           INT8_C(  64), INT8_C(  40), INT8_C(  97), INT8_C( 117),
                           INT8_C( -33), INT8_C(  55), INT8_C( 110), INT8_C( -34),
                           INT8_C( -75), INT8_C( 114), INT8_C( -81), INT8_C( -34),
                           INT8_C(  95), INT8_C(  31), INT8_C( 114), INT8_C(  73),
                           INT8_C( -61), INT8_C(   2), INT8_C(   8), INT8_C(  69),
                           INT8_C(   1), INT8_C(  92), INT8_C(-100), INT8_C(   7),
                           INT8_C( 120), INT8_C( -69), INT8_C( -67), INT8_C(  73),
                           INT8_C( -77), INT8_C( -96), INT8_C(-117), INT8_C(  90),
                           INT8_C(  28), INT8_C(  15), INT8_C( -80), INT8_C(  37),
                           INT8_C(  97), INT8_C(  97), INT8_C(  78), INT8_C( -88),
                           INT8_C(  43), INT8_C(  40), INT8_C(  93), INT8_C( -10),
                           INT8_C(  79), INT8_C(-115), INT8_C( -29), INT8_C(  12),
                           INT8_C( -91), INT8_C(  99), INT8_C( -12), INT8_C( 120)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_sub_epi8(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_sub_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask64 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi8(INT8_C( 121), INT8_C(  -8), INT8_C(-121), INT8_C( -19),
                           INT8_C(  19), INT8_C(  -3), INT8_C(  10), INT8_C( -37),
                           INT8_C(  96), INT8_C(  15), INT8_C( -45), INT8_C( -44),
                           INT8_C( -83), INT8_C( -37), INT8_C(   8), INT8_C( 111),
                           INT8_C( -30), INT8_C( -99), INT8_C( 116), INT8_C( 112),
                           INT8_C(  67), INT8_C(-123), INT8_C(  59), INT8_C( -62),
                           INT8_C(  33), INT8_C(  51), INT8_C( -16), INT8_C( -35),
                           INT8_C( -53), INT8_C( 108), INT8_C( -37), INT8_C( -15),
                           INT8_C(  26), INT8_C(  83), INT8_C( -47), INT8_C( -23),
                           INT8_C(   6), INT8_C(  52), INT8_C( -19), INT8_C( 108),
                           INT8_C( -33), INT8_C( 120), INT8_C(  55), INT8_C(-128),
                           INT8_C( -46), INT8_C( 117), INT8_C(  41), INT8_C( -54),
                           INT8_C( -99), INT8_C( -39), INT8_C( 117), INT8_C(  57),
                           INT8_C(  78), INT8_C(-110), INT8_C(  -8), INT8_C(-114),
                           INT8_C( -54), INT8_C(  20), INT8_C( 112), INT8_C(  39),
                           INT8_C( -60), INT8_C(  36), INT8_C( -53), INT8_C(  53)),
      UINT64_C(           567574400),
      simde_mm512_set_epi8(INT8_C( 115), INT8_C(  65), INT8_C(-123), INT8_C( 116),
                           INT8_C( -46), INT8_C( -68), INT8_C(  -8), INT8_C(  96),
                           INT8_C( -90), INT8_C( -14), INT8_C(  27), INT8_C(  33),
                           INT8_C(  21), INT8_C(  58), INT8_C( -12), INT8_C( -76),
                           INT8_C(  70), INT8_C( -70), INT8_C(  -4), INT8_C(  64),
                           INT8_C(  35), INT8_C( -32), INT8_C(-115), INT8_C( 109),
                           INT8_C( -69), INT8_C( -61), INT8_C(-126), INT8_C(  45),
                           INT8_C(  93), INT8_C( 100), INT8_C( -53), INT8_C( 105),
                           INT8_C( -51), INT8_C(  43), INT8_C( -96), INT8_C( -95),
                           INT8_C(  86), INT8_C( -66), INT8_C( -47), INT8_C( 123),
                           INT8_C(  63), INT8_C(  16), INT8_C( -66), INT8_C(  12),
                           INT8_C( 115), INT8_C( -39), INT8_C(  10), INT8_C( 108),
                           INT8_C(-120), INT8_C( 110), INT8_C( -25), INT8_C(   7),
                           INT8_C( 105), INT8_C( -93), INT8_C(  68), INT8_C(   3),
                           INT8_C( 113), INT8_C( -50), INT8_C( -34), INT8_C(  22),
                           INT8_C( -61), INT8_C(  75), INT8_C(  28), INT8_C( 116)),
      simde_mm512_set_epi8(INT8_C(  26), INT8_C(  -6), INT8_C( -92), INT8_C(   7),
                           INT8_C(-105), INT8_C( -92), INT8_C(  38), INT8_C( -63),
                           INT8_C(  77), INT8_C(  86), INT8_C( 113), INT8_C( -48),
                           INT8_C( 108), INT8_C( -92), INT8_C(  69), INT8_C(  74),
                           INT8_C(  67), INT8_C(  96), INT8_C( -34), INT8_C(  78),
                           INT8_C( 124), INT8_C(   9), INT8_C(  -1), INT8_C( -86),
                           INT8_C( -35), INT8_C(  26), INT8_C(  67), INT8_C(  46),
                           INT8_C(  75), INT8_C(-119), INT8_C(  68), INT8_C(  31),
                           INT8_C( -52), INT8_C(-102), INT8_C(  -4), INT8_C( 118),
                           INT8_C(   0), INT8_C( -44), INT8_C( 123), INT8_C( -73),
                           INT8_C(  84), INT8_C(  30), INT8_C(  -8), INT8_C(  64),
                           INT8_C( -20), INT8_C( -79), INT8_C( -85), INT8_C( -23),
                           INT8_C( -34), INT8_C(  -4), INT8_C( -85), INT8_C( 107),
                           INT8_C(  -6), INT8_C(  16), INT8_C( -66), INT8_C(-113),
                           INT8_C(  60), INT8_C( 127), INT8_C( -54), INT8_C( -36),
                           INT8_C(  73), INT8_C( -97), INT8_C( -65), INT8_C(  63)),
      simde_mm512_set_epi8(INT8_C( 121), INT8_C(  -8), INT8_C(-121), INT8_C( -19),
                           INT8_C(  19), INT8_C(  -3), INT8_C(  10), INT8_C( -37),
                           INT8_C(  96), INT8_C(  15), INT8_C( -45), INT8_C( -44),
                           INT8_C( -83), INT8_C( -37), INT8_C(   8), INT8_C( 111),
                           INT8_C( -30), INT8_C( -99), INT8_C( 116), INT8_C( 112),
                           INT8_C(  67), INT8_C(-123), INT8_C(  59), INT8_C( -62),
                           INT8_C(  33), INT8_C(  51), INT8_C( -16), INT8_C( -35),
                           INT8_C( -53), INT8_C( 108), INT8_C( -37), INT8_C( -15),
                           INT8_C(  26), INT8_C(  83), INT8_C( -92), INT8_C( -23),
                           INT8_C(   6), INT8_C(  52), INT8_C( -19), INT8_C( -60),
                           INT8_C( -21), INT8_C( -14), INT8_C(  55), INT8_C( -52),
                           INT8_C( -46), INT8_C(  40), INT8_C(  41), INT8_C( -54),
                           INT8_C( -99), INT8_C( 114), INT8_C(  60), INT8_C(-100),
                           INT8_C( 111), INT8_C(-109), INT8_C(-122), INT8_C( 116),
                           INT8_C(  53), INT8_C(  20), INT8_C( 112), INT8_C(  39),
                           INT8_C( -60), INT8_C(  36), INT8_C( -53), INT8_C(  53)) },
    { simde_mm512_set_epi8(INT8_C(  44), INT8_C(  68), INT8_C(  96), INT8_C(  88),
                           INT8_C(   1), INT8_C(  68), INT8_C(  46), INT8_C(  19),
                           INT8_C(  31), INT8_C(  85), INT8_C(  35), INT8_C(  68),
                           INT8_C( -79), INT8_C(  41), INT8_C(  28), INT8_C(  92),
                           INT8_C( -26), INT8_C(  20), INT8_C( -16), INT8_C(  -7),
                           INT8_C(  41), INT8_C(  71), INT8_C(  88), INT8_C(  39),
                           INT8_C( -42), INT8_C(  76), INT8_C(  40), INT8_C( 108),
                           INT8_C( -87), INT8_C(-126), INT8_C(  42), INT8_C(  58),
                           INT8_C(   2), INT8_C( -23), INT8_C(   5), INT8_C(-116),
                           INT8_C(  34), INT8_C(  66), INT8_C(  28), INT8_C(  86),
                           INT8_C(  50), INT8_C( -67), INT8_C(  20), INT8_C(  73),
                           INT8_C(  27), INT8_C( -29), INT8_C(  84), INT8_C( 112),
                           INT8_C( 104), INT8_C(  53), INT8_C( -89), INT8_C(-113),
                           INT8_C(  -4), INT8_C(  94), INT8_C(  75), INT8_C(  21),
                           INT8_C(-120), INT8_C( -25), INT8_C( 111), INT8_C(-128),
                           INT8_C(  -4), INT8_C(  58), INT8_C(-115), INT8_C(   4)),
      UINT64_C(          2779079274),
      simde_mm512_set_epi8(INT8_C( -70), INT8_C( -58), INT8_C(  69), INT8_C(  -7),
                           INT8_C( 115), INT8_C( -10), INT8_C( -39), INT8_C(  78),
                           INT8_C(  56), INT8_C( 116), INT8_C( 104), INT8_C( -92),
                           INT8_C(   5), INT8_C( -47), INT8_C(  27), INT8_C(  94),
                           INT8_C(-126), INT8_C(  88), INT8_C(  80), INT8_C( 112),
                           INT8_C(  21), INT8_C(  62), INT8_C(  86), INT8_C(-103),
                           INT8_C(  66), INT8_C(  -9), INT8_C( -26), INT8_C(  47),
                           INT8_C( -50), INT8_C(-118), INT8_C( 115), INT8_C(  49),
                           INT8_C(  42), INT8_C(   6), INT8_C(  92), INT8_C(   2),
                           INT8_C(  63), INT8_C(  -6), INT8_C( -32), INT8_C(  15),
                           INT8_C(  66), INT8_C(  82), INT8_C(  -9), INT8_C( -79),
                           INT8_C(-123), INT8_C(  52), INT8_C( -90), INT8_C(-111),
                           INT8_C(  62), INT8_C( -43), INT8_C( -50), INT8_C(  62),
                           INT8_C(   4), INT8_C( -92), INT8_C(  86), INT8_C( -32),
                           INT8_C(  69), INT8_C( -15), INT8_C(  55), INT8_C(-127),
                           INT8_C( -36), INT8_C(  56), INT8_C(  46), INT8_C(-119)),
      simde_mm512_set_epi8(INT8_C(  18), INT8_C(  68), INT8_C(  -9), INT8_C(  64),
                           INT8_C(-111), INT8_C(  37), INT8_C( -82), INT8_C( -95),
                           INT8_C(  54), INT8_C(  75), INT8_C( -77), INT8_C( -34),
                           INT8_C(  52), INT8_C( -80), INT8_C( -94), INT8_C(  90),
                           INT8_C(   0), INT8_C(  -8), INT8_C( 123), INT8_C(-111),
                           INT8_C(  16), INT8_C( 125), INT8_C( -51), INT8_C(  99),
                           INT8_C( -22), INT8_C( 121), INT8_C(  63), INT8_C( -55),
                           INT8_C( 117), INT8_C( 109), INT8_C(-126), INT8_C(-111),
                           INT8_C(  47), INT8_C(-127), INT8_C( 109), INT8_C(  -9),
                           INT8_C( -42), INT8_C(  36), INT8_C( -32), INT8_C( 115),
                           INT8_C( -89), INT8_C(   7), INT8_C(  90), INT8_C(  46),
                           INT8_C( -83), INT8_C( -35), INT8_C(  30), INT8_C( -19),
                           INT8_C( -99), INT8_C( -56), INT8_C( -70), INT8_C(  73),
                           INT8_C( -61), INT8_C(  27), INT8_C( 117), INT8_C(  47),
                           INT8_C(   0), INT8_C( -45), INT8_C(  59), INT8_C(  51),
                           INT8_C( -35), INT8_C(  70), INT8_C(  73), INT8_C(  33)),
      simde_mm512_set_epi8(INT8_C(  44), INT8_C(  68), INT8_C(  96), INT8_C(  88),
                           INT8_C(   1), INT8_C(  68), INT8_C(  46), INT8_C(  19),
                           INT8_C(  31), INT8_C(  85), INT8_C(  35), INT8_C(  68),
                           INT8_C( -79), INT8_C(  41), INT8_C(  28), INT8_C(  92),
                           INT8_C( -26), INT8_C(  20), INT8_C( -16), INT8_C(  -7),
                           INT8_C(  41), INT8_C(  71), INT8_C(  88), INT8_C(  39),
                           INT8_C( -42), INT8_C(  76), INT8_C(  40), INT8_C( 108),
                           INT8_C( -87), INT8_C(-126), INT8_C(  42), INT8_C(  58),
                           INT8_C(  -5), INT8_C( -23), INT8_C( -17), INT8_C(-116),
                           INT8_C(  34), INT8_C( -42), INT8_C(  28), INT8_C(-100),
                           INT8_C(-101), INT8_C( -67), INT8_C( -99), INT8_C(  73),
                           INT8_C(  27), INT8_C(  87), INT8_C(  84), INT8_C( -92),
                           INT8_C( 104), INT8_C(  13), INT8_C(  20), INT8_C(-113),
                           INT8_C(  -4), INT8_C(  94), INT8_C( -31), INT8_C(  21),
                           INT8_C(-120), INT8_C(  30), INT8_C(  -4), INT8_C(-128),
                           INT8_C(  -1), INT8_C(  58), INT8_C( -27), INT8_C(   4)) },
    { simde_mm512_set_epi8(INT8_C( -35), INT8_C(  32), INT8_C( -43), INT8_C( 108),
                           INT8_C(  83), INT8_C( -59), INT8_C(  -4), INT8_C( 125),
                           INT8_C( -31), INT8_C( 118), INT8_C( -25), INT8_C( -91),
                           INT8_C(  50), INT8_C( -74), INT8_C(  78), INT8_C(  95),
                           INT8_C( -84), INT8_C( -63), INT8_C(  87), INT8_C(-108),
                           INT8_C(  28), INT8_C( -70), INT8_C(  77), INT8_C(-113),
                           INT8_C( -20), INT8_C(  50), INT8_C(  95), INT8_C(-108),
                           INT8_C( 105), INT8_C( 114), INT8_C(-109), INT8_C(  19),
                           INT8_C( -79), INT8_C( 106), INT8_C(  61), INT8_C( -12),
                           INT8_C( 126), INT8_C(-117), INT8_C( 126), INT8_C(-125),
                           INT8_C( -93), INT8_C(  69), INT8_C( 104), INT8_C( 119),
                           INT8_C(  63), INT8_C(  95), INT8_C(-106), INT8_C( -66),
                           INT8_C( -47), INT8_C( -45), INT8_C( -60), INT8_C( -54),
                           INT8_C(-109), INT8_C( -45), INT8_C( -86), INT8_C( 121),
                           INT8_C(  23), INT8_C( -12), INT8_C(  67), INT8_C(  -6),
                           INT8_C( -37), INT8_C(  92), INT8_C( -35), INT8_C(  99)),
      UINT64_C(          1100920337),
      simde_mm512_set_epi8(INT8_C(  13), INT8_C( 104), INT8_C(  50), INT8_C(  43),
                           INT8_C(  82), INT8_C(  -5), INT8_C( -23), INT8_C( -47),
                           INT8_C(  99), INT8_C(-116), INT8_C( 118), INT8_C(  73),
                           INT8_C( -10), INT8_C( -88), INT8_C( -42), INT8_C( -58),
                           INT8_C( -49), INT8_C(  65), INT8_C( -18), INT8_C(  54),
                           INT8_C( -68), INT8_C(   1), INT8_C(  -7), INT8_C( -96),
                           INT8_C(   4), INT8_C( 115), INT8_C(  42), INT8_C(-106),
                           INT8_C(  31), INT8_C(  94), INT8_C( -71), INT8_C( -41),
                           INT8_C(  33), INT8_C(-106), INT8_C( -65), INT8_C(-107),
                           INT8_C(  71), INT8_C( -10), INT8_C( -21), INT8_C(-128),
                           INT8_C( -23), INT8_C(  20), INT8_C(   2), INT8_C(  96),
                           INT8_C(-128), INT8_C( -51), INT8_C( -38), INT8_C(  47),
                           INT8_C( -56), INT8_C( 123), INT8_C( -20), INT8_C( -50),
                           INT8_C(  -2), INT8_C(  40), INT8_C(  24), INT8_C( -98),
                           INT8_C(   4), INT8_C( -62), INT8_C( -44), INT8_C(  49),
                           INT8_C(  83), INT8_C( 115), INT8_C(   5), INT8_C(  57)),
      simde_mm512_set_epi8(INT8_C(  76), INT8_C( -52), INT8_C( -96), INT8_C(  -6),
                           INT8_C(-119), INT8_C( -87), INT8_C( 102), INT8_C(   5),
                           INT8_C(  24), INT8_C( -44), INT8_C( 110), INT8_C(-113),
                           INT8_C(-116), INT8_C(  -3), INT8_C(  62), INT8_C( -87),
                           INT8_C(   7), INT8_C( -54), INT8_C( -57), INT8_C( -66),
                           INT8_C(  42), INT8_C( -82), INT8_C(  46), INT8_C( -16),
                           INT8_C(  91), INT8_C( -73), INT8_C( -20), INT8_C( -77),
                           INT8_C( -11), INT8_C(  25), INT8_C(  12), INT8_C(  76),
                           INT8_C( -58), INT8_C(   3), INT8_C(-125), INT8_C( -36),
                           INT8_C(  18), INT8_C( -40), INT8_C( 111), INT8_C( 107),
                           INT8_C(  88), INT8_C(  48), INT8_C( 113), INT8_C( -90),
                           INT8_C(-117), INT8_C( 116), INT8_C(  46), INT8_C( -70),
                           INT8_C(  51), INT8_C( -55), INT8_C( 127), INT8_C(  82),
                           INT8_C( -88), INT8_C(  60), INT8_C( -59), INT8_C(  80),
                           INT8_C( -51), INT8_C(  11), INT8_C( -44), INT8_C(  33),
                           INT8_C(  29), INT8_C(   8), INT8_C(   5), INT8_C(  70)),
      simde_mm512_set_epi8(INT8_C( -35), INT8_C(  32), INT8_C( -43), INT8_C( 108),
                           INT8_C(  83), INT8_C( -59), INT8_C(  -4), INT8_C( 125),
                           INT8_C( -31), INT8_C( 118), INT8_C( -25), INT8_C( -91),
                           INT8_C(  50), INT8_C( -74), INT8_C(  78), INT8_C(  95),
                           INT8_C( -84), INT8_C( -63), INT8_C(  87), INT8_C(-108),
                           INT8_C(  28), INT8_C( -70), INT8_C(  77), INT8_C(-113),
                           INT8_C( -20), INT8_C(  50), INT8_C(  95), INT8_C(-108),
                           INT8_C( 105), INT8_C( 114), INT8_C(-109), INT8_C(  19),
                           INT8_C( -79), INT8_C(-109), INT8_C(  61), INT8_C( -12),
                           INT8_C( 126), INT8_C(-117), INT8_C( 126), INT8_C(  21),
                           INT8_C(-111), INT8_C(  69), INT8_C( 104), INT8_C( -70),
                           INT8_C( -11), INT8_C(  89), INT8_C( -84), INT8_C( -66),
                           INT8_C(-107), INT8_C( -45), INT8_C( 109), INT8_C( 124),
                           INT8_C(-109), INT8_C( -20), INT8_C(  83), INT8_C( 121),
                           INT8_C(  23), INT8_C( -12), INT8_C(  67), INT8_C(  16),
                           INT8_C( -37), INT8_C(  92), INT8_C( -35), INT8_C( -13)) },
    { simde_mm512_set_epi8(INT8_C(  27), INT8_C(  45), INT8_C(  71), INT8_C( -63),
                           INT8_C(  96), INT8_C(-106), INT8_C( -43), INT8_C(  10),
                           INT8_C( 104), INT8_C( -19), INT8_C(-110), INT8_C( 126),
                           INT8_C( -52), INT8_C( -56), INT8_C( -96), INT8_C( -27),
                           INT8_C(-125), INT8_C(-116), INT8_C(  25), INT8_C(  78),
                           INT8_C( -76), INT8_C( -85), INT8_C( -23), INT8_C( -19),
                           INT8_C(-106), INT8_C( 126), INT8_C(  19), INT8_C( -41),
                           INT8_C(  40), INT8_C(  78), INT8_C( -69), INT8_C(  57),
                           INT8_C(  73), INT8_C( -58), INT8_C(   3), INT8_C(  65),
                           INT8_C( -87), INT8_C( -37), INT8_C(   5), INT8_C(-126),
                           INT8_C(  14), INT8_C( -36), INT8_C( -37), INT8_C(  11),
                           INT8_C(  94), INT8_C(  24), INT8_C(   8), INT8_C( -31),
                           INT8_C( -38), INT8_C(  -1), INT8_C(  48), INT8_C(  32),
                           INT8_C(  88), INT8_C( -18), INT8_C( 123), INT8_C(  27),
                           INT8_C( 111), INT8_C(  27), INT8_C(  -3), INT8_C(  52),
                           INT8_C( -31), INT8_C(   2), INT8_C( -47), INT8_C(  64)),
      UINT64_C(           361367503),
      simde_mm512_set_epi8(INT8_C( -20), INT8_C(-104), INT8_C( -27), INT8_C(  38),
                           INT8_C(  31), INT8_C( -21), INT8_C(  79), INT8_C( -62),
                           INT8_C(  36), INT8_C(  95), INT8_C(  42), INT8_C(-102),
                           INT8_C( -80), INT8_C( -69), INT8_C( 107), INT8_C(-114),
                           INT8_C(  76), INT8_C( 123), INT8_C(-126), INT8_C( 108),
                           INT8_C( -55), INT8_C(  89), INT8_C( -46), INT8_C(  18),
                           INT8_C( 117), INT8_C(  25), INT8_C(-120), INT8_C(  27),
                           INT8_C(  34), INT8_C(  64), INT8_C(  71), INT8_C(  64),
                           INT8_C( -13), INT8_C( -73), INT8_C( 112), INT8_C(  25),
                           INT8_C( -18), INT8_C( -63), INT8_C( 109), INT8_C(   9),
                           INT8_C(  14), INT8_C(-125), INT8_C( -89), INT8_C(  70),
                           INT8_C(  10), INT8_C(  15), INT8_C( 120), INT8_C( -59),
                           INT8_C(  55), INT8_C( 108), INT8_C(  41), INT8_C(  -5),
                           INT8_C( -91), INT8_C(-120), INT8_C( -46), INT8_C( 122),
                           INT8_C( 116), INT8_C(-120), INT8_C( -67), INT8_C( -86),
                           INT8_C(  48), INT8_C(   2), INT8_C(  37), INT8_C( -26)),
      simde_mm512_set_epi8(INT8_C( -70), INT8_C(   3), INT8_C( 118), INT8_C(  37),
                           INT8_C( 104), INT8_C( 111), INT8_C( -17), INT8_C( 110),
                           INT8_C( -58), INT8_C(  58), INT8_C( 102), INT8_C(  64),
                           INT8_C( -67), INT8_C( -76), INT8_C( -30), INT8_C( 108),
                           INT8_C(  79), INT8_C(  46), INT8_C( -40), INT8_C( 101),
                           INT8_C( -13), INT8_C( -25), INT8_C(  60), INT8_C(  25),
                           INT8_C(  32), INT8_C( -21), INT8_C( 114), INT8_C( -21),
                           INT8_C(  71), INT8_C( -85), INT8_C(  34), INT8_C(  82),
                           INT8_C(-114), INT8_C( -30), INT8_C( -58), INT8_C( 116),
                           INT8_C(  58), INT8_C(-105), INT8_C( 117), INT8_C(  11),
                           INT8_C( -91), INT8_C( 118), INT8_C( -50), INT8_C(  -8),
                           INT8_C( -22), INT8_C(  59), INT8_C( -29), INT8_C( -88),
                           INT8_C( -82), INT8_C( -24), INT8_C(  18), INT8_C( 115),
                           INT8_C( -15), INT8_C(  55), INT8_C(  78), INT8_C(  60),
                           INT8_C(  -8), INT8_C( -91), INT8_C( 126), INT8_C(  15),
                           INT8_C(  23), INT8_C(   6), INT8_C( -21), INT8_C( 120)),
      simde_mm512_set_epi8(INT8_C(  27), INT8_C(  45), INT8_C(  71), INT8_C( -63),
                           INT8_C(  96), INT8_C(-106), INT8_C( -43), INT8_C(  10),
                           INT8_C( 104), INT8_C( -19), INT8_C(-110), INT8_C( 126),
                           INT8_C( -52), INT8_C( -56), INT8_C( -96), INT8_C( -27),
                           INT8_C(-125), INT8_C(-116), INT8_C(  25), INT8_C(  78),
                           INT8_C( -76), INT8_C( -85), INT8_C( -23), INT8_C( -19),
                           INT8_C(-106), INT8_C( 126), INT8_C(  19), INT8_C( -41),
                           INT8_C(  40), INT8_C(  78), INT8_C( -69), INT8_C(  57),
                           INT8_C(  73), INT8_C( -58), INT8_C(   3), INT8_C( -91),
                           INT8_C( -87), INT8_C(  42), INT8_C(   5), INT8_C(  -2),
                           INT8_C( 105), INT8_C( -36), INT8_C( -37), INT8_C(  11),
                           INT8_C(  32), INT8_C(  24), INT8_C(-107), INT8_C( -31),
                           INT8_C( -38), INT8_C(  -1), INT8_C(  48), INT8_C(  32),
                           INT8_C(  88), INT8_C(  81), INT8_C(-124), INT8_C(  62),
                           INT8_C( 124), INT8_C( -29), INT8_C(  -3), INT8_C(  52),
                           INT8_C(  25), INT8_C(  -4), INT8_C(  58), INT8_C( 110)) },
    { simde_mm512_set_epi8(INT8_C(   4), INT8_C(  97), INT8_C(  53), INT8_C( -46),
                           INT8_C(  92), INT8_C(-100), INT8_C(  47), INT8_C( 107),
                           INT8_C( -52), INT8_C(  68), INT8_C(  11), INT8_C( -16),
                           INT8_C( -66), INT8_C( -79), INT8_C( -14), INT8_C(  27),
                           INT8_C(  14), INT8_C( 125), INT8_C(  22), INT8_C( -82),
                           INT8_C(  44), INT8_C( -12), INT8_C(  94), INT8_C( -30),
                           INT8_C(  98), INT8_C( 125), INT8_C(-107), INT8_C(  37),
                           INT8_C( -66), INT8_C(  90), INT8_C(  68), INT8_C(  10),
                           INT8_C( -72), INT8_C( -10), INT8_C(-119), INT8_C(  -9),
                           INT8_C(  49), INT8_C(-107), INT8_C(  10), INT8_C(  47),
                           INT8_C(  58), INT8_C(-125), INT8_C(   4), INT8_C(  68),
                           INT8_C( -24), INT8_C( -12), INT8_C(  44), INT8_C(-128),
                           INT8_C( -52), INT8_C( -61), INT8_C( -14), INT8_C( -38),
                           INT8_C( -93), INT8_C( -34), INT8_C(  64), INT8_C( -67),
                           INT8_C(-123), INT8_C( 123), INT8_C( -93), INT8_C(  41),
                           INT8_C(  97), INT8_C(  -8), INT8_C( -86), INT8_C( -16)),
      UINT64_C(           944667126),
      simde_mm512_set_epi8(INT8_C( -24), INT8_C( -47), INT8_C(-119), INT8_C(   5),
                           INT8_C(  95), INT8_C(  82), INT8_C(  -3), INT8_C( -62),
                           INT8_C(-116), INT8_C( -98), INT8_C( -29), INT8_C(  77),
                           INT8_C( -38), INT8_C(-118), INT8_C( -85), INT8_C( 121),
                           INT8_C( -72), INT8_C(-111), INT8_C(  28), INT8_C( -18),
                           INT8_C(  64), INT8_C(-126), INT8_C( 122), INT8_C( -54),
                           INT8_C(  87), INT8_C( -22), INT8_C(  17), INT8_C(  50),
                           INT8_C( -83), INT8_C( -39), INT8_C(  77), INT8_C( -13),
                           INT8_C(  17), INT8_C( -66), INT8_C(-128), INT8_C(  77),
                           INT8_C( 107), INT8_C(  47), INT8_C( -68), INT8_C( -44),
                           INT8_C( -30), INT8_C( -22), INT8_C(  14), INT8_C(  26),
                           INT8_C(  59), INT8_C( 103), INT8_C( -54), INT8_C( -39),
                           INT8_C(  16), INT8_C(   5), INT8_C(  18), INT8_C(-104),
                           INT8_C(-119), INT8_C( -46), INT8_C( -92), INT8_C(  37),
                           INT8_C( -84), INT8_C(   2), INT8_C( -49), INT8_C(  99),
                           INT8_C( -79), INT8_C(  48), INT8_C(-103), INT8_C(   3)),
      simde_mm512_set_epi8(INT8_C( -56), INT8_C( -56), INT8_C(  57), INT8_C( -25),
                           INT8_C(  -3), INT8_C(  99), INT8_C(  -6), INT8_C(  31),
                           INT8_C( -96), INT8_C(  49), INT8_C( 110), INT8_C( -10),
                           INT8_C( -82), INT8_C(  32), INT8_C( -27), INT8_C( 112),
                           INT8_C(  84), INT8_C(  37), INT8_C( -62), INT8_C(  38),
                           INT8_C( -53), INT8_C( -97), INT8_C(  76), INT8_C(  13),
                           INT8_C(-124), INT8_C(-120), INT8_C( -86), INT8_C(  98),
                           INT8_C(  96), INT8_C(   4), INT8_C(   4), INT8_C(  94),
                           INT8_C( -41), INT8_C( -81), INT8_C( -40), INT8_C( -28),
                           INT8_C( -23), INT8_C( -59), INT8_C( -15), INT8_C( -40),
                           INT8_C( 113), INT8_C( 116), INT8_C(  41), INT8_C( -96),
                           INT8_C( -83), INT8_C(   4), INT8_C(  93), INT8_C(  28),
                           INT8_C( 114), INT8_C(  29), INT8_C( -56), INT8_C( -61),
                           INT8_C(-124), INT8_C(-107), INT8_C( -23), INT8_C( -89),
                           INT8_C(  38), INT8_C( -97), INT8_C( 109), INT8_C(  53),
                           INT8_C(-117), INT8_C(  76), INT8_C( -82), INT8_C( -65)),
      simde_mm512_set_epi8(INT8_C(   4), INT8_C(  97), INT8_C(  53), INT8_C( -46),
                           INT8_C(  92), INT8_C(-100), INT8_C(  47), INT8_C( 107),
                           INT8_C( -52), INT8_C(  68), INT8_C(  11), INT8_C( -16),
                           INT8_C( -66), INT8_C( -79), INT8_C( -14), INT8_C(  27),
                           INT8_C(  14), INT8_C( 125), INT8_C(  22), INT8_C( -82),
                           INT8_C(  44), INT8_C( -12), INT8_C(  94), INT8_C( -30),
                           INT8_C(  98), INT8_C( 125), INT8_C(-107), INT8_C(  37),
                           INT8_C( -66), INT8_C(  90), INT8_C(  68), INT8_C(  10),
                           INT8_C( -72), INT8_C( -10), INT8_C( -88), INT8_C( 105),
                           INT8_C(-126), INT8_C(-107), INT8_C(  10), INT8_C(  47),
                           INT8_C(  58), INT8_C( 118), INT8_C(   4), INT8_C(  68),
                           INT8_C(-114), INT8_C(  99), INT8_C( 109), INT8_C(-128),
                           INT8_C( -52), INT8_C( -24), INT8_C(  74), INT8_C( -43),
                           INT8_C(   5), INT8_C( -34), INT8_C(  64), INT8_C( 126),
                           INT8_C(-122), INT8_C(  99), INT8_C(  98), INT8_C(  46),
                           INT8_C(  97), INT8_C( -28), INT8_C( -21), INT8_C( -16)) },
    { simde_mm512_set_epi8(INT8_C( -50), INT8_C(   0), INT8_C(  80), INT8_C(-123),
                           INT8_C(  19), INT8_C( 112), INT8_C(  30), INT8_C(  95),
                           INT8_C(  58), INT8_C(  21), INT8_C(  13), INT8_C(  32),
                           INT8_C( 113), INT8_C( 126), INT8_C(  27), INT8_C( 113),
                           INT8_C( 121), INT8_C(  97), INT8_C(  51), INT8_C( -16),
                           INT8_C( -77), INT8_C(  84), INT8_C(  16), INT8_C(-112),
                           INT8_C( -40), INT8_C( -69), INT8_C(-116), INT8_C( -97),
                           INT8_C(-120), INT8_C( 102), INT8_C( -82), INT8_C( -42),
                           INT8_C(  43), INT8_C( -70), INT8_C(  46), INT8_C(  17),
                           INT8_C( 108), INT8_C( -47), INT8_C(  53), INT8_C( -84),
                           INT8_C(  19), INT8_C( -37), INT8_C( -32), INT8_C( -59),
                           INT8_C(  33), INT8_C( 110), INT8_C(  17), INT8_C(  67),
                           INT8_C(  51), INT8_C( -19), INT8_C(  91), INT8_C(  26),
                           INT8_C(  33), INT8_C( -43), INT8_C( -14), INT8_C( -56),
                           INT8_C( 112), INT8_C( -72), INT8_C(  96), INT8_C( -62),
                           INT8_C( -21), INT8_C(  96), INT8_C( -25), INT8_C( 104)),
      UINT64_C(          1662672283),
      simde_mm512_set_epi8(INT8_C(  55), INT8_C(  43), INT8_C(-128), INT8_C(  23),
                           INT8_C( -59), INT8_C( -21), INT8_C( -11), INT8_C( -65),
                           INT8_C(-101), INT8_C( -89), INT8_C( -88), INT8_C( -71),
                           INT8_C( -70), INT8_C(  37), INT8_C( 122), INT8_C(  74),
                           INT8_C( 109), INT8_C( -13), INT8_C( -13), INT8_C(  72),
                           INT8_C(  -1), INT8_C( -35), INT8_C(  80), INT8_C( -20),
                           INT8_C(  14), INT8_C(-104), INT8_C( -76), INT8_C(-122),
                           INT8_C( -35), INT8_C( -33), INT8_C(  63), INT8_C(  74),
                           INT8_C(  98), INT8_C(  54), INT8_C( -12), INT8_C(  -1),
                           INT8_C( -30), INT8_C(  96), INT8_C(  95), INT8_C(  58),
                           INT8_C( -63), INT8_C(  -6), INT8_C(-113), INT8_C(  55),
                           INT8_C(-128), INT8_C( -43), INT8_C( -90), INT8_C( -63),
                           INT8_C(   3), INT8_C(  -6), INT8_C( -45), INT8_C( -75),
                           INT8_C( -83), INT8_C(-118), INT8_C(  74), INT8_C(  35),
                           INT8_C(  38), INT8_C(   4), INT8_C(  35), INT8_C(  15),
                           INT8_C( -42), INT8_C(  71), INT8_C(  -1), INT8_C(  27)),
      simde_mm512_set_epi8(INT8_C(  32), INT8_C( 122), INT8_C(  89), INT8_C(  21),
                           INT8_C( -83), INT8_C( -46), INT8_C( -78), INT8_C(  71),
                           INT8_C( -35), INT8_C(  54), INT8_C( -65), INT8_C(-111),
                           INT8_C(  45), INT8_C(  -5), INT8_C( 102), INT8_C(  32),
                           INT8_C(-110), INT8_C( 116), INT8_C( -61), INT8_C(  36),
                           INT8_C( -25), INT8_C( 106), INT8_C( -63), INT8_C(  23),
                           INT8_C( -59), INT8_C(  25), INT8_C(-108), INT8_C( -84),
                           INT8_C( -23), INT8_C( 118), INT8_C( -35), INT8_C(  92),
                           INT8_C( -29), INT8_C(-121), INT8_C( -87), INT8_C(  93),
                           INT8_C(   6), INT8_C( -12), INT8_C(-123), INT8_C(  42),
                           INT8_C( 121), INT8_C(   3), INT8_C(  69), INT8_C(  75),
                           INT8_C(  68), INT8_C(  -1), INT8_C( -25), INT8_C(  83),
                           INT8_C(  -4), INT8_C( -73), INT8_C( -63), INT8_C(  12),
                           INT8_C( -93), INT8_C( -22), INT8_C(  40), INT8_C( -24),
                           INT8_C( -60), INT8_C(  99), INT8_C( 122), INT8_C(  49),
                           INT8_C( -46), INT8_C( 127), INT8_C(  18), INT8_C( 124)),
      simde_mm512_set_epi8(INT8_C( -50), INT8_C(   0), INT8_C(  80), INT8_C(-123),
                           INT8_C(  19), INT8_C( 112), INT8_C(  30), INT8_C(  95),
                           INT8_C(  58), INT8_C(  21), INT8_C(  13), INT8_C(  32),
                           INT8_C( 113), INT8_C( 126), INT8_C(  27), INT8_C( 113),
                           INT8_C( 121), INT8_C(  97), INT8_C(  51), INT8_C( -16),
                           INT8_C( -77), INT8_C(  84), INT8_C(  16), INT8_C(-112),
                           INT8_C( -40), INT8_C( -69), INT8_C(-116), INT8_C( -97),
                           INT8_C(-120), INT8_C( 102), INT8_C( -82), INT8_C( -42),
                           INT8_C(  43), INT8_C( -81), INT8_C(  75), INT8_C(  17),
                           INT8_C( 108), INT8_C( -47), INT8_C( -38), INT8_C(  16),
                           INT8_C(  19), INT8_C( -37), INT8_C( -32), INT8_C( -20),
                           INT8_C(  60), INT8_C( 110), INT8_C( -65), INT8_C(  67),
                           INT8_C(  51), INT8_C(  67), INT8_C(  91), INT8_C( -87),
                           INT8_C(  10), INT8_C( -96), INT8_C( -14), INT8_C(  59),
                           INT8_C(  98), INT8_C( -72), INT8_C(  96), INT8_C( -34),
                           INT8_C(   4), INT8_C(  96), INT8_C( -19), INT8_C( -97)) },
    { simde_mm512_set_epi8(INT8_C( -82), INT8_C(  17), INT8_C( 105), INT8_C(   8),
                           INT8_C( -41), INT8_C( 122), INT8_C( -11), INT8_C( -52),
                           INT8_C( -81), INT8_C( -30), INT8_C( 109), INT8_C( 119),
                           INT8_C( -78), INT8_C(-123), INT8_C(   5), INT8_C( -23),
                           INT8_C(  44), INT8_C( -23), INT8_C(-122), INT8_C(-101),
                           INT8_C( -30), INT8_C( 103), INT8_C(  30), INT8_C(  -6),
                           INT8_C( 113), INT8_C( -64), INT8_C(  -3), INT8_C(-100),
                           INT8_C(  72), INT8_C( -30), INT8_C(  59), INT8_C(  -7),
                           INT8_C(-101), INT8_C(  48), INT8_C( -62), INT8_C(   5),
                           INT8_C( -52), INT8_C(  72), INT8_C(  56), INT8_C(   6),
                           INT8_C(  86), INT8_C( -78), INT8_C( -43), INT8_C(  91),
                           INT8_C( -63), INT8_C( -91), INT8_C(-105), INT8_C( -98),
                           INT8_C(  39), INT8_C(   5), INT8_C(  77), INT8_C(  91),
                           INT8_C( -82), INT8_C(  20), INT8_C(  41), INT8_C(  62),
                           INT8_C(  27), INT8_C(  82), INT8_C( -39), INT8_C(  57),
                           INT8_C(-116), INT8_C( -85), INT8_C(-107), INT8_C(  31)),
      UINT64_C(           782232724),
      simde_mm512_set_epi8(INT8_C(  11), INT8_C(  49), INT8_C( -30), INT8_C(-117),
                           INT8_C(  85), INT8_C(  19), INT8_C(  44), INT8_C(-110),
                           INT8_C(  61), INT8_C( -27), INT8_C(  26), INT8_C( -12),
                           INT8_C( 110), INT8_C(  11), INT8_C(  45), INT8_C( -32),
                           INT8_C(  -1), INT8_C(  86), INT8_C( 125), INT8_C(  95),
                           INT8_C( -41), INT8_C( -73), INT8_C(  -6), INT8_C( 122),
                           INT8_C(  65), INT8_C( -38), INT8_C(-116), INT8_C(  84),
                           INT8_C(-121), INT8_C( -15), INT8_C(  41), INT8_C(-102),
                           INT8_C( -31), INT8_C( -83), INT8_C( -68), INT8_C(  89),
                           INT8_C(  27), INT8_C(-107), INT8_C( -85), INT8_C(  74),
                           INT8_C(  95), INT8_C( -86), INT8_C(  94), INT8_C( -13),
                           INT8_C( -84), INT8_C(  38), INT8_C( 116), INT8_C(-101),
                           INT8_C(  72), INT8_C(  32), INT8_C( -98), INT8_C(  48),
                           INT8_C( -94), INT8_C( -55), INT8_C( -17), INT8_C(  28),
                           INT8_C(  42), INT8_C(  70), INT8_C(  89), INT8_C(-115),
                           INT8_C( -86), INT8_C( 126), INT8_C( -92), INT8_C(  91)),
      simde_mm512_set_epi8(INT8_C( -46), INT8_C( -24), INT8_C( -24), INT8_C( -26),
                           INT8_C(  89), INT8_C( 108), INT8_C(  49), INT8_C( 123),
                           INT8_C( -86), INT8_C( -61), INT8_C( -22), INT8_C( -47),
                           INT8_C(  21), INT8_C(  76), INT8_C(   6), INT8_C( -21),
                           INT8_C( -19), INT8_C(  38), INT8_C(-116), INT8_C( -22),
                           INT8_C( -75), INT8_C(  54), INT8_C( -81), INT8_C(   9),
                           INT8_C(  94), INT8_C( -15), INT8_C(  26), INT8_C(-110),
                           INT8_C(  18), INT8_C( -49), INT8_C( -21), INT8_C(  70),
                           INT8_C(  50), INT8_C(  20), INT8_C( -59), INT8_C(  63),
                           INT8_C( -20), INT8_C( -92), INT8_C( -44), INT8_C(  37),
                           INT8_C(-125), INT8_C(   4), INT8_C(  53), INT8_C( -49),
                           INT8_C( -10), INT8_C(  11), INT8_C(  91), INT8_C( -86),
                           INT8_C( -34), INT8_C(-108), INT8_C( -80), INT8_C( 122),
                           INT8_C(  31), INT8_C(  31), INT8_C( -29), INT8_C(  70),
                           INT8_C(  28), INT8_C(  33), INT8_C( 109), INT8_C(  55),
                           INT8_C( -79), INT8_C(  95), INT8_C( 100), INT8_C( -33)),
      simde_mm512_set_epi8(INT8_C( -82), INT8_C(  17), INT8_C( 105), INT8_C(   8),
                           INT8_C( -41), INT8_C( 122), INT8_C( -11), INT8_C( -52),
                           INT8_C( -81), INT8_C( -30), INT8_C( 109), INT8_C( 119),
                           INT8_C( -78), INT8_C(-123), INT8_C(   5), INT8_C( -23),
                           INT8_C(  44), INT8_C( -23), INT8_C(-122), INT8_C(-101),
                           INT8_C( -30), INT8_C( 103), INT8_C(  30), INT8_C(  -6),
                           INT8_C( 113), INT8_C( -64), INT8_C(  -3), INT8_C(-100),
                           INT8_C(  72), INT8_C( -30), INT8_C(  59), INT8_C(  -7),
                           INT8_C(-101), INT8_C(  48), INT8_C(  -9), INT8_C(   5),
                           INT8_C(  47), INT8_C( -15), INT8_C( -41), INT8_C(   6),
                           INT8_C( -36), INT8_C( -78), INT8_C( -43), INT8_C(  36),
                           INT8_C( -74), INT8_C(  27), INT8_C(  25), INT8_C( -15),
                           INT8_C( 106), INT8_C(-116), INT8_C( -18), INT8_C(  91),
                           INT8_C(-125), INT8_C( -86), INT8_C(  41), INT8_C(  62),
                           INT8_C(  14), INT8_C(  82), INT8_C( -39), INT8_C(  86),
                           INT8_C(-116), INT8_C(  31), INT8_C(-107), INT8_C(  31)) },
    { simde_mm512_set_epi8(INT8_C(  82), INT8_C( -55), INT8_C(  13), INT8_C(-104),
                           INT8_C(  62), INT8_C(  20), INT8_C( -36), INT8_C(  92),
                           INT8_C( -73), INT8_C( -79), INT8_C(  -7), INT8_C( -22),
                           INT8_C( -50), INT8_C(-119), INT8_C( -83), INT8_C( -71),
                           INT8_C( 125), INT8_C(  29), INT8_C( -61), INT8_C(-111),
                           INT8_C(  -9), INT8_C(  67), INT8_C( -39), INT8_C( -17),
                           INT8_C(  23), INT8_C( -11), INT8_C(-122), INT8_C( -24),
                           INT8_C(  37), INT8_C(-122), INT8_C( -16), INT8_C( -40),
                           INT8_C( -34), INT8_C( -17), INT8_C( 100), INT8_C( 120),
                           INT8_C( -51), INT8_C(   8), INT8_C(  82), INT8_C(  19),
                           INT8_C( -50), INT8_C( -24), INT8_C( -20), INT8_C( -32),
                           INT8_C(  74), INT8_C( -84), INT8_C(   9), INT8_C(  14),
                           INT8_C(-102), INT8_C(  -2), INT8_C( 106), INT8_C(  41),
                           INT8_C(  98), INT8_C( -87), INT8_C(-124), INT8_C(  -3),
                           INT8_C(  80), INT8_C( 110), INT8_C( -32), INT8_C(  20),
                           INT8_C( -15), INT8_C(  65), INT8_C( -54), INT8_C( -49)),
      UINT64_C(          1883474426),
      simde_mm512_set_epi8(INT8_C( -36), INT8_C(-125), INT8_C( -71), INT8_C(-101),
                           INT8_C( -95), INT8_C(  -1), INT8_C(  65), INT8_C(  67),
                           INT8_C(   4), INT8_C( 126), INT8_C(  -9), INT8_C(  50),
                           INT8_C(  46), INT8_C(  17), INT8_C(  12), INT8_C(   7),
                           INT8_C(  31), INT8_C( -83), INT8_C(  63), INT8_C(  21),
                           INT8_C(-105), INT8_C(  56), INT8_C(   6), INT8_C(  88),
                           INT8_C(  -4), INT8_C( -51), INT8_C( -16), INT8_C( -27),
                           INT8_C( -26), INT8_C(  50), INT8_C(   3), INT8_C( -65),
                           INT8_C(  -3), INT8_C( -86), INT8_C(  39), INT8_C(  48),
                           INT8_C(  65), INT8_C(  36), INT8_C( -65), INT8_C( -82),
                           INT8_C(-107), INT8_C(  14), INT8_C( 110), INT8_C(  56),
                           INT8_C( 111), INT8_C( -32), INT8_C( 109), INT8_C( -95),
                           INT8_C(  69), INT8_C(-111), INT8_C(  -7), INT8_C(   9),
                           INT8_C( 116), INT8_C(  77), INT8_C( 122), INT8_C(  26),
                           INT8_C(   9), INT8_C( -79), INT8_C(-127), INT8_C(  -5),
                           INT8_C( 117), INT8_C(   5), INT8_C(  80), INT8_C( -85)),
      simde_mm512_set_epi8(INT8_C(  82), INT8_C(-125), INT8_C( -33), INT8_C(  83),
                           INT8_C( -98), INT8_C(  50), INT8_C( -27), INT8_C( -16),
                           INT8_C( -63), INT8_C(-111), INT8_C( -65), INT8_C(   3),
                           INT8_C( -15), INT8_C(  37), INT8_C(  46), INT8_C(  58),
                           INT8_C( -41), INT8_C(  72), INT8_C( 108), INT8_C(-124),
                           INT8_C(   9), INT8_C(  40), INT8_C( 115), INT8_C(  12),
                           INT8_C(   1), INT8_C(  41), INT8_C( -71), INT8_C(  87),
                           INT8_C( -55), INT8_C(  52), INT8_C( -97), INT8_C(  49),
                           INT8_C(  32), INT8_C(-115), INT8_C(  71), INT8_C(  64),
                           INT8_C( -61), INT8_C(  43), INT8_C( -42), INT8_C(  57),
                           INT8_C( -56), INT8_C( 113), INT8_C(  60), INT8_C(  75),
                           INT8_C(   7), INT8_C( -47), INT8_C(   4), INT8_C( 115),
                           INT8_C(  67), INT8_C(  44), INT8_C(  -1), INT8_C( -85),
                           INT8_C( -95), INT8_C( 108), INT8_C(  37), INT8_C( -99),
                           INT8_C( -88), INT8_C( -11), INT8_C(  47), INT8_C( -69),
                           INT8_C(-123), INT8_C(  17), INT8_C( -30), INT8_C(  36)),
      simde_mm512_set_epi8(INT8_C(  82), INT8_C( -55), INT8_C(  13), INT8_C(-104),
                           INT8_C(  62), INT8_C(  20), INT8_C( -36), INT8_C(  92),
                           INT8_C( -73), INT8_C( -79), INT8_C(  -7), INT8_C( -22),
                           INT8_C( -50), INT8_C(-119), INT8_C( -83), INT8_C( -71),
                           INT8_C( 125), INT8_C(  29), INT8_C( -61), INT8_C(-111),
                           INT8_C(  -9), INT8_C(  67), INT8_C( -39), INT8_C( -17),
                           INT8_C(  23), INT8_C( -11), INT8_C(-122), INT8_C( -24),
                           INT8_C(  37), INT8_C(-122), INT8_C( -16), INT8_C( -40),
                           INT8_C( -34), INT8_C(  29), INT8_C( -32), INT8_C( -16),
                           INT8_C( -51), INT8_C(   8), INT8_C(  82), INT8_C(  19),
                           INT8_C( -50), INT8_C( -99), INT8_C( -20), INT8_C( -32),
                           INT8_C(  74), INT8_C( -84), INT8_C( 105), INT8_C(  46),
                           INT8_C(   2), INT8_C(  -2), INT8_C( 106), INT8_C(  41),
                           INT8_C( -45), INT8_C( -87), INT8_C(-124), INT8_C( 125),
                           INT8_C(  97), INT8_C( -68), INT8_C(  82), INT8_C(  64),
                           INT8_C( -16), INT8_C(  65), INT8_C( 110), INT8_C( -49)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_sub_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_sub_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask64 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT64_C(          3290745653),
      simde_mm512_set_epi8(INT8_C(  68), INT8_C( -18), INT8_C(-120), INT8_C( -91),
                           INT8_C(  33), INT8_C( -44), INT8_C( 127), INT8_C(-128),
                           INT8_C( 121), INT8_C(  -8), INT8_C(-121), INT8_C( -19),
                           INT8_C(  19), INT8_C(  -3), INT8_C(  10), INT8_C( -37),
                           INT8_C(  96), INT8_C(  15), INT8_C( -45), INT8_C( -44),
                           INT8_C( -83), INT8_C( -37), INT8_C(   8), INT8_C( 111),
                           INT8_C( -30), INT8_C( -99), INT8_C( 116), INT8_C( 112),
                           INT8_C(  67), INT8_C(-123), INT8_C(  59), INT8_C( -62),
                           INT8_C(  33), INT8_C(  51), INT8_C( -16), INT8_C( -35),
                           INT8_C( -53), INT8_C( 108), INT8_C( -37), INT8_C( -15),
                           INT8_C(  26), INT8_C(  83), INT8_C( -47), INT8_C( -23),
                           INT8_C(   6), INT8_C(  52), INT8_C( -19), INT8_C( 108),
                           INT8_C( -33), INT8_C( 120), INT8_C(  55), INT8_C(-128),
                           INT8_C( -46), INT8_C( 117), INT8_C(  41), INT8_C( -54),
                           INT8_C( -99), INT8_C( -39), INT8_C( 117), INT8_C(  57),
                           INT8_C(  78), INT8_C(-110), INT8_C(  -8), INT8_C(-114)),
      simde_mm512_set_epi8(INT8_C( 115), INT8_C(  65), INT8_C(-123), INT8_C( 116),
                           INT8_C( -46), INT8_C( -68), INT8_C(  -8), INT8_C(  96),
                           INT8_C( -90), INT8_C( -14), INT8_C(  27), INT8_C(  33),
                           INT8_C(  21), INT8_C(  58), INT8_C( -12), INT8_C( -76),
                           INT8_C(  70), INT8_C( -70), INT8_C(  -4), INT8_C(  64),
                           INT8_C(  35), INT8_C( -32), INT8_C(-115), INT8_C( 109),
                           INT8_C( -69), INT8_C( -61), INT8_C(-126), INT8_C(  45),
                           INT8_C(  93), INT8_C( 100), INT8_C( -53), INT8_C( 105),
                           INT8_C( -51), INT8_C(  43), INT8_C( -96), INT8_C( -95),
                           INT8_C(  86), INT8_C( -66), INT8_C( -47), INT8_C( 123),
                           INT8_C(  63), INT8_C(  16), INT8_C( -66), INT8_C(  12),
                           INT8_C( 115), INT8_C( -39), INT8_C(  10), INT8_C( 108),
                           INT8_C(-120), INT8_C( 110), INT8_C( -25), INT8_C(   7),
                           INT8_C( 105), INT8_C( -93), INT8_C(  68), INT8_C(   3),
                           INT8_C( 113), INT8_C( -50), INT8_C( -34), INT8_C(  22),
                           INT8_C( -61), INT8_C(  75), INT8_C(  28), INT8_C( 116)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  84), INT8_C(   8), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C( -82), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  19), INT8_C(   0),
                           INT8_C(   0), INT8_C(  91), INT8_C(   0), INT8_C(   0),
                           INT8_C(  87), INT8_C(  10), INT8_C(   0), INT8_C(   0),
                           INT8_C( 105), INT8_C(   0), INT8_C( -27), INT8_C( -57),
                           INT8_C(   0), INT8_C(   0), INT8_C(-105), INT8_C(  35),
                           INT8_C(   0), INT8_C(  71), INT8_C(   0), INT8_C(  26)) },
    { UINT64_C(          1235205951),
      simde_mm512_set_epi8(INT8_C(-120), INT8_C( -25), INT8_C( 111), INT8_C(-128),
                           INT8_C(  -4), INT8_C(  58), INT8_C(-115), INT8_C(   4),
                           INT8_C(  26), INT8_C(  -6), INT8_C( -92), INT8_C(   7),
                           INT8_C(-105), INT8_C( -92), INT8_C(  38), INT8_C( -63),
                           INT8_C(  77), INT8_C(  86), INT8_C( 113), INT8_C( -48),
                           INT8_C( 108), INT8_C( -92), INT8_C(  69), INT8_C(  74),
                           INT8_C(  67), INT8_C(  96), INT8_C( -34), INT8_C(  78),
                           INT8_C( 124), INT8_C(   9), INT8_C(  -1), INT8_C( -86),
                           INT8_C( -35), INT8_C(  26), INT8_C(  67), INT8_C(  46),
                           INT8_C(  75), INT8_C(-119), INT8_C(  68), INT8_C(  31),
                           INT8_C( -52), INT8_C(-102), INT8_C(  -4), INT8_C( 118),
                           INT8_C(   0), INT8_C( -44), INT8_C( 123), INT8_C( -73),
                           INT8_C(  84), INT8_C(  30), INT8_C(  -8), INT8_C(  64),
                           INT8_C( -20), INT8_C( -79), INT8_C( -85), INT8_C( -23),
                           INT8_C( -34), INT8_C(  -4), INT8_C( -85), INT8_C( 107),
                           INT8_C(  -6), INT8_C(  16), INT8_C( -66), INT8_C(-113)),
      simde_mm512_set_epi8(INT8_C( -19), INT8_C( 117), INT8_C( 121), INT8_C(  67),
                           INT8_C( -91), INT8_C( -91), INT8_C(  98), INT8_C( 106),
                           INT8_C(  44), INT8_C(  68), INT8_C(  96), INT8_C(  88),
                           INT8_C(   1), INT8_C(  68), INT8_C(  46), INT8_C(  19),
                           INT8_C(  31), INT8_C(  85), INT8_C(  35), INT8_C(  68),
                           INT8_C( -79), INT8_C(  41), INT8_C(  28), INT8_C(  92),
                           INT8_C( -26), INT8_C(  20), INT8_C( -16), INT8_C(  -7),
                           INT8_C(  41), INT8_C(  71), INT8_C(  88), INT8_C(  39),
                           INT8_C( -42), INT8_C(  76), INT8_C(  40), INT8_C( 108),
                           INT8_C( -87), INT8_C(-126), INT8_C(  42), INT8_C(  58),
                           INT8_C(   2), INT8_C( -23), INT8_C(   5), INT8_C(-116),
                           INT8_C(  34), INT8_C(  66), INT8_C(  28), INT8_C(  86),
                           INT8_C(  50), INT8_C( -67), INT8_C(  20), INT8_C(  73),
                           INT8_C(  27), INT8_C( -29), INT8_C(  84), INT8_C( 112),
                           INT8_C( 104), INT8_C(  53), INT8_C( -89), INT8_C(-113),
                           INT8_C(  -4), INT8_C(  94), INT8_C(  75), INT8_C(  21)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C( -50), INT8_C(   0), INT8_C(   0),
                           INT8_C( -94), INT8_C(   0), INT8_C(   0), INT8_C( -27),
                           INT8_C( -54), INT8_C(   0), INT8_C(   0), INT8_C( -22),
                           INT8_C( -34), INT8_C(-110), INT8_C(  95), INT8_C(  97),
                           INT8_C(  34), INT8_C(   0), INT8_C( -28), INT8_C(  -9),
                           INT8_C( -47), INT8_C( -50), INT8_C(  87), INT8_C( 121),
                           INT8_C(   0), INT8_C(   0), INT8_C(   4), INT8_C( -36),
                           INT8_C(  -2), INT8_C( -78), INT8_C( 115), INT8_C( 122)) },
    { UINT64_C(          3694669449),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C( -45), INT8_C(  59), INT8_C(  51),
                           INT8_C( -35), INT8_C(  70), INT8_C(  73), INT8_C(  33),
                           INT8_C( -70), INT8_C( -58), INT8_C(  69), INT8_C(  -7),
                           INT8_C( 115), INT8_C( -10), INT8_C( -39), INT8_C(  78),
                           INT8_C(  56), INT8_C( 116), INT8_C( 104), INT8_C( -92),
                           INT8_C(   5), INT8_C( -47), INT8_C(  27), INT8_C(  94),
                           INT8_C(-126), INT8_C(  88), INT8_C(  80), INT8_C( 112),
                           INT8_C(  21), INT8_C(  62), INT8_C(  86), INT8_C(-103),
                           INT8_C(  66), INT8_C(  -9), INT8_C( -26), INT8_C(  47),
                           INT8_C( -50), INT8_C(-118), INT8_C( 115), INT8_C(  49),
                           INT8_C(  42), INT8_C(   6), INT8_C(  92), INT8_C(   2),
                           INT8_C(  63), INT8_C(  -6), INT8_C( -32), INT8_C(  15),
                           INT8_C(  66), INT8_C(  82), INT8_C(  -9), INT8_C( -79),
                           INT8_C(-123), INT8_C(  52), INT8_C( -90), INT8_C(-111),
                           INT8_C(  62), INT8_C( -43), INT8_C( -50), INT8_C(  62),
                           INT8_C(   4), INT8_C( -92), INT8_C(  86), INT8_C( -32)),
      simde_mm512_set_epi8(INT8_C(  23), INT8_C( -12), INT8_C(  67), INT8_C(  -6),
                           INT8_C( -37), INT8_C(  92), INT8_C( -35), INT8_C(  99),
                           INT8_C(  18), INT8_C(  68), INT8_C(  -9), INT8_C(  64),
                           INT8_C(-111), INT8_C(  37), INT8_C( -82), INT8_C( -95),
                           INT8_C(  54), INT8_C(  75), INT8_C( -77), INT8_C( -34),
                           INT8_C(  52), INT8_C( -80), INT8_C( -94), INT8_C(  90),
                           INT8_C(   0), INT8_C(  -8), INT8_C( 123), INT8_C(-111),
                           INT8_C(  16), INT8_C( 125), INT8_C( -51), INT8_C(  99),
                           INT8_C( -22), INT8_C( 121), INT8_C(  63), INT8_C( -55),
                           INT8_C( 117), INT8_C( 109), INT8_C(-126), INT8_C(-111),
                           INT8_C(  47), INT8_C(-127), INT8_C( 109), INT8_C(  -9),
                           INT8_C( -42), INT8_C(  36), INT8_C( -32), INT8_C( 115),
                           INT8_C( -89), INT8_C(   7), INT8_C(  90), INT8_C(  46),
                           INT8_C( -83), INT8_C( -35), INT8_C(  30), INT8_C( -19),
                           INT8_C( -99), INT8_C( -56), INT8_C( -70), INT8_C(  73),
                           INT8_C( -61), INT8_C(  27), INT8_C( 117), INT8_C(  47)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  88), INT8_C( 126), INT8_C(   0), INT8_C( 102),
                           INT8_C(  89), INT8_C(  29), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -17), INT8_C(  11),
                           INT8_C( 105), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -99), INT8_C(   0),
                           INT8_C( -40), INT8_C(  87), INT8_C(-120), INT8_C(   0),
                           INT8_C( -95), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  65), INT8_C(   0), INT8_C(   0), INT8_C( -79)) },
    { UINT64_C(          2480122489),
      simde_mm512_set_epi8(INT8_C(   4), INT8_C( -62), INT8_C( -44), INT8_C(  49),
                           INT8_C(  83), INT8_C( 115), INT8_C(   5), INT8_C(  57),
                           INT8_C( -76), INT8_C( -80), INT8_C(  40), INT8_C(  60),
                           INT8_C(  65), INT8_C( -98), INT8_C( -74), INT8_C(  17),
                           INT8_C( -35), INT8_C(  32), INT8_C( -43), INT8_C( 108),
                           INT8_C(  83), INT8_C( -59), INT8_C(  -4), INT8_C( 125),
                           INT8_C( -31), INT8_C( 118), INT8_C( -25), INT8_C( -91),
                           INT8_C(  50), INT8_C( -74), INT8_C(  78), INT8_C(  95),
                           INT8_C( -84), INT8_C( -63), INT8_C(  87), INT8_C(-108),
                           INT8_C(  28), INT8_C( -70), INT8_C(  77), INT8_C(-113),
                           INT8_C( -20), INT8_C(  50), INT8_C(  95), INT8_C(-108),
                           INT8_C( 105), INT8_C( 114), INT8_C(-109), INT8_C(  19),
                           INT8_C( -79), INT8_C( 106), INT8_C(  61), INT8_C( -12),
                           INT8_C( 126), INT8_C(-117), INT8_C( 126), INT8_C(-125),
                           INT8_C( -93), INT8_C(  69), INT8_C( 104), INT8_C( 119),
                           INT8_C(  63), INT8_C(  95), INT8_C(-106), INT8_C( -66)),
      simde_mm512_set_epi8(INT8_C( -51), INT8_C(  11), INT8_C( -44), INT8_C(  33),
                           INT8_C(  29), INT8_C(   8), INT8_C(   5), INT8_C(  70),
                           INT8_C(  13), INT8_C( 104), INT8_C(  50), INT8_C(  43),
                           INT8_C(  82), INT8_C(  -5), INT8_C( -23), INT8_C( -47),
                           INT8_C(  99), INT8_C(-116), INT8_C( 118), INT8_C(  73),
                           INT8_C( -10), INT8_C( -88), INT8_C( -42), INT8_C( -58),
                           INT8_C( -49), INT8_C(  65), INT8_C( -18), INT8_C(  54),
                           INT8_C( -68), INT8_C(   1), INT8_C(  -7), INT8_C( -96),
                           INT8_C(   4), INT8_C( 115), INT8_C(  42), INT8_C(-106),
                           INT8_C(  31), INT8_C(  94), INT8_C( -71), INT8_C( -41),
                           INT8_C(  33), INT8_C(-106), INT8_C( -65), INT8_C(-107),
                           INT8_C(  71), INT8_C( -10), INT8_C( -21), INT8_C(-128),
                           INT8_C( -23), INT8_C(  20), INT8_C(   2), INT8_C(  96),
                           INT8_C(-128), INT8_C( -51), INT8_C( -38), INT8_C(  47),
                           INT8_C( -56), INT8_C( 123), INT8_C( -20), INT8_C( -50),
                           INT8_C(  -2), INT8_C(  40), INT8_C(  24), INT8_C( -98)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( -88), INT8_C(   0), INT8_C(   0), INT8_C(  -2),
                           INT8_C(   0), INT8_C(   0), INT8_C(-108), INT8_C( -72),
                           INT8_C( -53), INT8_C(-100), INT8_C(   0), INT8_C(  -1),
                           INT8_C(   0), INT8_C(   0), INT8_C( -88), INT8_C(-109),
                           INT8_C( -56), INT8_C(   0), INT8_C(  59), INT8_C(   0),
                           INT8_C(  -2), INT8_C(   0), INT8_C( -92), INT8_C(   0),
                           INT8_C(   0), INT8_C( -54), INT8_C( 124), INT8_C( -87),
                           INT8_C(  65), INT8_C(   0), INT8_C(   0), INT8_C(  32)) },
    { UINT64_C(          2822554960),
      simde_mm512_set_epi8(INT8_C( -38), INT8_C(  -1), INT8_C(  48), INT8_C(  32),
                           INT8_C(  88), INT8_C( -18), INT8_C( 123), INT8_C(  27),
                           INT8_C( 111), INT8_C(  27), INT8_C(  -3), INT8_C(  52),
                           INT8_C( -31), INT8_C(   2), INT8_C( -47), INT8_C(  64),
                           INT8_C(  76), INT8_C( -52), INT8_C( -96), INT8_C(  -6),
                           INT8_C(-119), INT8_C( -87), INT8_C( 102), INT8_C(   5),
                           INT8_C(  24), INT8_C( -44), INT8_C( 110), INT8_C(-113),
                           INT8_C(-116), INT8_C(  -3), INT8_C(  62), INT8_C( -87),
                           INT8_C(   7), INT8_C( -54), INT8_C( -57), INT8_C( -66),
                           INT8_C(  42), INT8_C( -82), INT8_C(  46), INT8_C( -16),
                           INT8_C(  91), INT8_C( -73), INT8_C( -20), INT8_C( -77),
                           INT8_C( -11), INT8_C(  25), INT8_C(  12), INT8_C(  76),
                           INT8_C( -58), INT8_C(   3), INT8_C(-125), INT8_C( -36),
                           INT8_C(  18), INT8_C( -40), INT8_C( 111), INT8_C( 107),
                           INT8_C(  88), INT8_C(  48), INT8_C( 113), INT8_C( -90),
                           INT8_C(-117), INT8_C( 116), INT8_C(  46), INT8_C( -70)),
      simde_mm512_set_epi8(INT8_C( 116), INT8_C(-120), INT8_C( -67), INT8_C( -86),
                           INT8_C(  48), INT8_C(   2), INT8_C(  37), INT8_C( -26),
                           INT8_C( -55), INT8_C(  66), INT8_C(  80), INT8_C(  -7),
                           INT8_C(  21), INT8_C(-118), INT8_C(   7), INT8_C( -49),
                           INT8_C(  27), INT8_C(  45), INT8_C(  71), INT8_C( -63),
                           INT8_C(  96), INT8_C(-106), INT8_C( -43), INT8_C(  10),
                           INT8_C( 104), INT8_C( -19), INT8_C(-110), INT8_C( 126),
                           INT8_C( -52), INT8_C( -56), INT8_C( -96), INT8_C( -27),
                           INT8_C(-125), INT8_C(-116), INT8_C(  25), INT8_C(  78),
                           INT8_C( -76), INT8_C( -85), INT8_C( -23), INT8_C( -19),
                           INT8_C(-106), INT8_C( 126), INT8_C(  19), INT8_C( -41),
                           INT8_C(  40), INT8_C(  78), INT8_C( -69), INT8_C(  57),
                           INT8_C(  73), INT8_C( -58), INT8_C(   3), INT8_C(  65),
                           INT8_C( -87), INT8_C( -37), INT8_C(   5), INT8_C(-126),
                           INT8_C(  14), INT8_C( -36), INT8_C( -37), INT8_C(  11),
                           INT8_C(  94), INT8_C(  24), INT8_C(   8), INT8_C( -31)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(-124), INT8_C(   0), INT8_C( -82), INT8_C(   0),
                           INT8_C( 118), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C( -39), INT8_C( -36),
                           INT8_C( -51), INT8_C( -53), INT8_C(   0), INT8_C(   0),
                           INT8_C( 125), INT8_C(  61), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  -3), INT8_C(   0), INT8_C( -23),
                           INT8_C(   0), INT8_C(  84), INT8_C(   0), INT8_C(-101),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0)) },
    { UINT64_C(          2777207418),
      simde_mm512_set_epi8(INT8_C( -82), INT8_C( -24), INT8_C(  18), INT8_C( 115),
                           INT8_C( -15), INT8_C(  55), INT8_C(  78), INT8_C(  60),
                           INT8_C(  -8), INT8_C( -91), INT8_C( 126), INT8_C(  15),
                           INT8_C(  23), INT8_C(   6), INT8_C( -21), INT8_C( 120),
                           INT8_C( -20), INT8_C(-104), INT8_C( -27), INT8_C(  38),
                           INT8_C(  31), INT8_C( -21), INT8_C(  79), INT8_C( -62),
                           INT8_C(  36), INT8_C(  95), INT8_C(  42), INT8_C(-102),
                           INT8_C( -80), INT8_C( -69), INT8_C( 107), INT8_C(-114),
                           INT8_C(  76), INT8_C( 123), INT8_C(-126), INT8_C( 108),
                           INT8_C( -55), INT8_C(  89), INT8_C( -46), INT8_C(  18),
                           INT8_C( 117), INT8_C(  25), INT8_C(-120), INT8_C(  27),
                           INT8_C(  34), INT8_C(  64), INT8_C(  71), INT8_C(  64),
                           INT8_C( -13), INT8_C( -73), INT8_C( 112), INT8_C(  25),
                           INT8_C( -18), INT8_C( -63), INT8_C( 109), INT8_C(   9),
                           INT8_C(  14), INT8_C(-125), INT8_C( -89), INT8_C(  70),
                           INT8_C(  10), INT8_C(  15), INT8_C( 120), INT8_C( -59)),
      simde_mm512_set_epi8(INT8_C( -52), INT8_C( -61), INT8_C( -14), INT8_C( -38),
                           INT8_C( -93), INT8_C( -34), INT8_C(  64), INT8_C( -67),
                           INT8_C(-123), INT8_C( 123), INT8_C( -93), INT8_C(  41),
                           INT8_C(  97), INT8_C(  -8), INT8_C( -86), INT8_C( -16),
                           INT8_C( -70), INT8_C(   3), INT8_C( 118), INT8_C(  37),
                           INT8_C( 104), INT8_C( 111), INT8_C( -17), INT8_C( 110),
                           INT8_C( -58), INT8_C(  58), INT8_C( 102), INT8_C(  64),
                           INT8_C( -67), INT8_C( -76), INT8_C( -30), INT8_C( 108),
                           INT8_C(  79), INT8_C(  46), INT8_C( -40), INT8_C( 101),
                           INT8_C( -13), INT8_C( -25), INT8_C(  60), INT8_C(  25),
                           INT8_C(  32), INT8_C( -21), INT8_C( 114), INT8_C( -21),
                           INT8_C(  71), INT8_C( -85), INT8_C(  34), INT8_C(  82),
                           INT8_C(-114), INT8_C( -30), INT8_C( -58), INT8_C( 116),
                           INT8_C(  58), INT8_C(-105), INT8_C( 117), INT8_C(  11),
                           INT8_C( -91), INT8_C( 118), INT8_C( -50), INT8_C(  -8),
                           INT8_C( -22), INT8_C(  59), INT8_C( -29), INT8_C( -88)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  -3), INT8_C(   0), INT8_C( -86), INT8_C(   0),
                           INT8_C(   0), INT8_C( 114), INT8_C(   0), INT8_C(  -7),
                           INT8_C(  85), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( -37), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( 101), INT8_C( -43), INT8_C(   0), INT8_C( -91),
                           INT8_C(   0), INT8_C(   0), INT8_C(  -8), INT8_C(   0),
                           INT8_C(   0), INT8_C(  13), INT8_C( -39), INT8_C(  78),
                           INT8_C(  32), INT8_C(   0), INT8_C(-107), INT8_C(   0)) },
    { UINT64_C(          3908316288),
      simde_mm512_set_epi8(INT8_C(  16), INT8_C(   5), INT8_C(  18), INT8_C(-104),
                           INT8_C(-119), INT8_C( -46), INT8_C( -92), INT8_C(  37),
                           INT8_C( -84), INT8_C(   2), INT8_C( -49), INT8_C(  99),
                           INT8_C( -79), INT8_C(  48), INT8_C(-103), INT8_C(   3),
                           INT8_C(  54), INT8_C( 118), INT8_C( -53), INT8_C(  24),
                           INT8_C(  56), INT8_C(  78), INT8_C( 121), INT8_C( -10),
                           INT8_C(   4), INT8_C(  97), INT8_C(  53), INT8_C( -46),
                           INT8_C(  92), INT8_C(-100), INT8_C(  47), INT8_C( 107),
                           INT8_C( -52), INT8_C(  68), INT8_C(  11), INT8_C( -16),
                           INT8_C( -66), INT8_C( -79), INT8_C( -14), INT8_C(  27),
                           INT8_C(  14), INT8_C( 125), INT8_C(  22), INT8_C( -82),
                           INT8_C(  44), INT8_C( -12), INT8_C(  94), INT8_C( -30),
                           INT8_C(  98), INT8_C( 125), INT8_C(-107), INT8_C(  37),
                           INT8_C( -66), INT8_C(  90), INT8_C(  68), INT8_C(  10),
                           INT8_C( -72), INT8_C( -10), INT8_C(-119), INT8_C(  -9),
                           INT8_C(  49), INT8_C(-107), INT8_C(  10), INT8_C(  47)),
      simde_mm512_set_epi8(INT8_C( 114), INT8_C(  29), INT8_C( -56), INT8_C( -61),
                           INT8_C(-124), INT8_C(-107), INT8_C( -23), INT8_C( -89),
                           INT8_C(  38), INT8_C( -97), INT8_C( 109), INT8_C(  53),
                           INT8_C(-117), INT8_C(  76), INT8_C( -82), INT8_C( -65),
                           INT8_C( -24), INT8_C( -47), INT8_C(-119), INT8_C(   5),
                           INT8_C(  95), INT8_C(  82), INT8_C(  -3), INT8_C( -62),
                           INT8_C(-116), INT8_C( -98), INT8_C( -29), INT8_C(  77),
                           INT8_C( -38), INT8_C(-118), INT8_C( -85), INT8_C( 121),
                           INT8_C( -72), INT8_C(-111), INT8_C(  28), INT8_C( -18),
                           INT8_C(  64), INT8_C(-126), INT8_C( 122), INT8_C( -54),
                           INT8_C(  87), INT8_C( -22), INT8_C(  17), INT8_C(  50),
                           INT8_C( -83), INT8_C( -39), INT8_C(  77), INT8_C( -13),
                           INT8_C(  17), INT8_C( -66), INT8_C(-128), INT8_C(  77),
                           INT8_C( 107), INT8_C(  47), INT8_C( -68), INT8_C( -44),
                           INT8_C( -30), INT8_C( -22), INT8_C(  14), INT8_C(  26),
                           INT8_C(  59), INT8_C( 103), INT8_C( -54), INT8_C( -39)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(  20), INT8_C( -77), INT8_C( -17), INT8_C(   0),
                           INT8_C( 126), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( -73), INT8_C(-109), INT8_C(   5), INT8_C( 124),
                           INT8_C(   0), INT8_C(  27), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(  21), INT8_C(   0),
                           INT8_C(  83), INT8_C(  43), INT8_C(   0), INT8_C(   0),
                           INT8_C( -42), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0)) },
    { UINT64_C(          2902744348),
      simde_mm512_set_epi8(INT8_C(  19), INT8_C( -37), INT8_C( -32), INT8_C( -59),
                           INT8_C(  33), INT8_C( 110), INT8_C(  17), INT8_C(  67),
                           INT8_C(  51), INT8_C( -19), INT8_C(  91), INT8_C(  26),
                           INT8_C(  33), INT8_C( -43), INT8_C( -14), INT8_C( -56),
                           INT8_C( 112), INT8_C( -72), INT8_C(  96), INT8_C( -62),
                           INT8_C( -21), INT8_C(  96), INT8_C( -25), INT8_C( 104),
                           INT8_C( -56), INT8_C( -56), INT8_C(  57), INT8_C( -25),
                           INT8_C(  -3), INT8_C(  99), INT8_C(  -6), INT8_C(  31),
                           INT8_C( -96), INT8_C(  49), INT8_C( 110), INT8_C( -10),
                           INT8_C( -82), INT8_C(  32), INT8_C( -27), INT8_C( 112),
                           INT8_C(  84), INT8_C(  37), INT8_C( -62), INT8_C(  38),
                           INT8_C( -53), INT8_C( -97), INT8_C(  76), INT8_C(  13),
                           INT8_C(-124), INT8_C(-120), INT8_C( -86), INT8_C(  98),
                           INT8_C(  96), INT8_C(   4), INT8_C(   4), INT8_C(  94),
                           INT8_C( -41), INT8_C( -81), INT8_C( -40), INT8_C( -28),
                           INT8_C( -23), INT8_C( -59), INT8_C( -15), INT8_C( -40)),
      simde_mm512_set_epi8(INT8_C(   3), INT8_C(  -6), INT8_C( -45), INT8_C( -75),
                           INT8_C( -83), INT8_C(-118), INT8_C(  74), INT8_C(  35),
                           INT8_C(  38), INT8_C(   4), INT8_C(  35), INT8_C(  15),
                           INT8_C( -42), INT8_C(  71), INT8_C(  -1), INT8_C(  27),
                           INT8_C( -48), INT8_C( -77), INT8_C( 116), INT8_C(  52),
                           INT8_C(  99), INT8_C(  26), INT8_C(  93), INT8_C(-101),
                           INT8_C( -50), INT8_C(   0), INT8_C(  80), INT8_C(-123),
                           INT8_C(  19), INT8_C( 112), INT8_C(  30), INT8_C(  95),
                           INT8_C(  58), INT8_C(  21), INT8_C(  13), INT8_C(  32),
                           INT8_C( 113), INT8_C( 126), INT8_C(  27), INT8_C( 113),
                           INT8_C( 121), INT8_C(  97), INT8_C(  51), INT8_C( -16),
                           INT8_C( -77), INT8_C(  84), INT8_C(  16), INT8_C(-112),
                           INT8_C( -40), INT8_C( -69), INT8_C(-116), INT8_C( -97),
                           INT8_C(-120), INT8_C( 102), INT8_C( -82), INT8_C( -42),
                           INT8_C(  43), INT8_C( -70), INT8_C(  46), INT8_C(  17),
                           INT8_C( 108), INT8_C( -47), INT8_C(  53), INT8_C( -84)),
      simde_mm512_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C( 102), INT8_C(   0), INT8_C(  97), INT8_C(   0),
                           INT8_C(  61), INT8_C( -94), INT8_C(   0), INT8_C(  -1),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C(  75), INT8_C(   0), INT8_C(   0),
                           INT8_C(   0), INT8_C( -51), INT8_C(   0), INT8_C( -61),
                           INT8_C( -40), INT8_C( -98), INT8_C(   0), INT8_C(-120),
                           INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C( -45),
                           INT8_C( 125), INT8_C( -12), INT8_C(   0), INT8_C(   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_sub_epi8(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_sub_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi16(INT16_C( 21075), INT16_C( 30017), INT16_C(-11898), INT16_C( 29710),
                            INT16_C( 19457), INT16_C(-12796), INT16_C( 21427), INT16_C( 28826),
                            INT16_C( 25482), INT16_C(-11843), INT16_C( 15582), INT16_C( 20114),
                            INT16_C(-14761), INT16_C(-15590), INT16_C( -4142), INT16_C( 29932),
                            INT16_C(-30672), INT16_C(  6190), INT16_C( 26590), INT16_C( 10803),
                            INT16_C(-16554), INT16_C( 15816), INT16_C( 14967), INT16_C( 24063),
                            INT16_C(-14713), INT16_C( -8094), INT16_C(-16817), INT16_C( 25507),
                            INT16_C( 19912), INT16_C(-19929), INT16_C(-12604), INT16_C(-17156)),
      simde_mm512_set_epi16(INT16_C( 27175), INT16_C(-27122), INT16_C( -6914), INT16_C( 10212),
                            INT16_C( 13894), INT16_C( -4620), INT16_C(-10724), INT16_C( -3078),
                            INT16_C( 29698), INT16_C(  6009), INT16_C( 28893), INT16_C(-31734),
                            INT16_C( -3957), INT16_C(  6787), INT16_C(  9325), INT16_C(  7645),
                            INT16_C( -2133), INT16_C( -9633), INT16_C(-22525), INT16_C(  1124),
                            INT16_C( 21781), INT16_C( 17119), INT16_C(-19461), INT16_C(-32134),
                            INT16_C( -7507), INT16_C( 19092), INT16_C( 21408), INT16_C(-14444),
                            INT16_C( 22843), INT16_C( 28625), INT16_C( -2322), INT16_C(  5251)),
      simde_mm512_set_epi16(INT16_C( -6100), INT16_C( -8397), INT16_C( -4984), INT16_C( 19498),
                            INT16_C(  5563), INT16_C( -8176), INT16_C( 32151), INT16_C( 31904),
                            INT16_C( -4216), INT16_C(-17852), INT16_C(-13311), INT16_C(-13688),
                            INT16_C(-10804), INT16_C(-22377), INT16_C(-13467), INT16_C( 22287),
                            INT16_C(-28539), INT16_C( 15823), INT16_C(-16421), INT16_C(  9679),
                            INT16_C( 27201), INT16_C( -1303), INT16_C(-31108), INT16_C( -9339),
                            INT16_C( -7206), INT16_C(-27186), INT16_C( 27311), INT16_C(-25585),
                            INT16_C( -2931), INT16_C( 16982), INT16_C(-10282), INT16_C(-22407)) },
    { simde_mm512_set_epi16(INT16_C(-27508), INT16_C(  5509), INT16_C(-13526), INT16_C( 16909),
                            INT16_C(  2419), INT16_C( 22142), INT16_C( -6109), INT16_C( -1177),
                            INT16_C(  9839), INT16_C(  6329), INT16_C(  -239), INT16_C(-15885),
                            INT16_C(  3666), INT16_C( 20122), INT16_C( -1699), INT16_C(  6503),
                            INT16_C( 29169), INT16_C( -4681), INT16_C( -2713), INT16_C(-24709),
                            INT16_C(  7221), INT16_C( -3718), INT16_C(   970), INT16_C(-15558),
                            INT16_C(-11011), INT16_C(-10787), INT16_C(-29970), INT16_C(  3894),
                            INT16_C(-25914), INT16_C(-18758), INT16_C( 11824), INT16_C( -8868)),
      simde_mm512_set_epi16(INT16_C(  1604), INT16_C( 19874), INT16_C(-12133), INT16_C( -1966),
                            INT16_C( 13041), INT16_C(  1566), INT16_C(-11791), INT16_C( -3425),
                            INT16_C(  7377), INT16_C(-23380), INT16_C( -9249), INT16_C(-31251),
                            INT16_C( 14877), INT16_C( 24009), INT16_C(-32316), INT16_C(  8308),
                            INT16_C(-11725), INT16_C(-10230), INT16_C(  1074), INT16_C( 12341),
                            INT16_C( 19989), INT16_C( 16491), INT16_C(  4144), INT16_C(-11714),
                            INT16_C( 19285), INT16_C(-29198), INT16_C(-25258), INT16_C(-29514),
                            INT16_C(  9755), INT16_C(-29385), INT16_C(-23111), INT16_C( -3412)),
      simde_mm512_set_epi16(INT16_C(-29112), INT16_C(-14365), INT16_C( -1393), INT16_C( 18875),
                            INT16_C(-10622), INT16_C( 20576), INT16_C(  5682), INT16_C(  2248),
                            INT16_C(  2462), INT16_C( 29709), INT16_C(  9010), INT16_C( 15366),
                            INT16_C(-11211), INT16_C( -3887), INT16_C( 30617), INT16_C( -1805),
                            INT16_C(-24642), INT16_C(  5549), INT16_C( -3787), INT16_C( 28486),
                            INT16_C(-12768), INT16_C(-20209), INT16_C( -3174), INT16_C( -3844),
                            INT16_C(-30296), INT16_C( 18411), INT16_C( -4712), INT16_C(-32128),
                            INT16_C( 29867), INT16_C( 10627), INT16_C(-30601), INT16_C( -5456)) },
    { simde_mm512_set_epi16(INT16_C(   691), INT16_C( -4823), INT16_C( -3253), INT16_C(-31392),
                            INT16_C(-21784), INT16_C( -6740), INT16_C(  9130), INT16_C(-18273),
                            INT16_C( 11275), INT16_C(-27092), INT16_C(    90), INT16_C(-20133),
                            INT16_C( 30523), INT16_C( 27008), INT16_C( 28387), INT16_C( 17266),
                            INT16_C( -9777), INT16_C( 27096), INT16_C( -8328), INT16_C( -6812),
                            INT16_C(-22954), INT16_C( -4409), INT16_C( 21734), INT16_C(-19695),
                            INT16_C(-11981), INT16_C(-21195), INT16_C( 18272), INT16_C( 28327),
                            INT16_C(  7123), INT16_C(-32216), INT16_C( 24489), INT16_C(-15668)),
      simde_mm512_set_epi16(INT16_C(-21377), INT16_C( 15856), INT16_C(  7686), INT16_C(-28568),
                            INT16_C(-15192), INT16_C( -9747), INT16_C( 11300), INT16_C( 27000),
                            INT16_C( -6635), INT16_C(  3626), INT16_C( 12716), INT16_C(-30571),
                            INT16_C( 31697), INT16_C(  5622), INT16_C( 24444), INT16_C( -8226),
                            INT16_C( -8263), INT16_C(  2890), INT16_C( 26732), INT16_C( -8763),
                            INT16_C(-13950), INT16_C( 27415), INT16_C(  7653), INT16_C( 31511),
                            INT16_C(-21082), INT16_C(  2398), INT16_C( 23365), INT16_C(-12903),
                            INT16_C(-18221), INT16_C(  4204), INT16_C(-20453), INT16_C( 15021)),
      simde_mm512_set_epi16(INT16_C( 22068), INT16_C(-20679), INT16_C(-10939), INT16_C( -2824),
                            INT16_C( -6592), INT16_C(  3007), INT16_C( -2170), INT16_C( 20263),
                            INT16_C( 17910), INT16_C(-30718), INT16_C(-12626), INT16_C( 10438),
                            INT16_C( -1174), INT16_C( 21386), INT16_C(  3943), INT16_C( 25492),
                            INT16_C( -1514), INT16_C( 24206), INT16_C( 30476), INT16_C(  1951),
                            INT16_C( -9004), INT16_C(-31824), INT16_C( 14081), INT16_C( 14330),
                            INT16_C(  9101), INT16_C(-23593), INT16_C( -5093), INT16_C(-24306),
                            INT16_C( 25344), INT16_C( 29116), INT16_C(-20594), INT16_C(-30689)) },
    { simde_mm512_set_epi16(INT16_C(  4451), INT16_C( -3121), INT16_C( 11648), INT16_C( 14185),
                            INT16_C( -8499), INT16_C(-24679), INT16_C(-31633), INT16_C( 19019),
                            INT16_C( 26210), INT16_C(-29943), INT16_C(-18883), INT16_C( 25468),
                            INT16_C( 20366), INT16_C(  4961), INT16_C(-25468), INT16_C( -4158),
                            INT16_C(  6653), INT16_C( -1720), INT16_C(-29723), INT16_C(-14244),
                            INT16_C( -4917), INT16_C(   730), INT16_C(-20677), INT16_C( 16986),
                            INT16_C(  9316), INT16_C( 28795), INT16_C(-18273), INT16_C(-29423),
                            INT16_C(-23674), INT16_C(  7963), INT16_C( 28019), INT16_C( 13728)),
      simde_mm512_set_epi16(INT16_C(-10770), INT16_C( 29411), INT16_C( 30463), INT16_C( -4902),
                            INT16_C(-20392), INT16_C(-28251), INT16_C( 11448), INT16_C( 27155),
                            INT16_C(-11669), INT16_C( 11820), INT16_C(-16512), INT16_C( 10540),
                            INT16_C( 17477), INT16_C(-19759), INT16_C( 28024), INT16_C(-14431),
                            INT16_C( 24400), INT16_C( -7583), INT16_C(-12129), INT16_C( 28592),
                            INT16_C(-31057), INT16_C(-18091), INT16_C( 19926), INT16_C(-29261),
                            INT16_C(  7501), INT16_C( 16620), INT16_C(  6953), INT16_C(  3437),
                            INT16_C(  5790), INT16_C(  5348), INT16_C( 17145), INT16_C(-28791)),
      simde_mm512_set_epi16(INT16_C( 15221), INT16_C(-32532), INT16_C(-18815), INT16_C( 19087),
                            INT16_C( 11893), INT16_C(  3572), INT16_C( 22455), INT16_C( -8136),
                            INT16_C(-27657), INT16_C( 23773), INT16_C( -2371), INT16_C( 14928),
                            INT16_C(  2889), INT16_C( 24720), INT16_C( 12044), INT16_C( 10273),
                            INT16_C(-17747), INT16_C(  5863), INT16_C(-17594), INT16_C( 22700),
                            INT16_C( 26140), INT16_C( 18821), INT16_C( 24933), INT16_C(-19289),
                            INT16_C(  1815), INT16_C( 12175), INT16_C(-25226), INT16_C( 32676),
                            INT16_C(-29464), INT16_C(  2615), INT16_C( 10874), INT16_C(-23017)) },
    { simde_mm512_set_epi16(INT16_C(-31561), INT16_C( 18949), INT16_C( -2287), INT16_C(-20534),
                            INT16_C( -1057), INT16_C( -3046), INT16_C( 22138), INT16_C(-11031),
                            INT16_C(    43), INT16_C( -6266), INT16_C(-20090), INT16_C(-22393),
                            INT16_C(-26046), INT16_C(-23703), INT16_C( 28092), INT16_C(  6346),
                            INT16_C( 10308), INT16_C(   572), INT16_C(     5), INT16_C( 15306),
                            INT16_C(-19429), INT16_C( -5811), INT16_C(-27420), INT16_C(-29128),
                            INT16_C(-13676), INT16_C( -3673), INT16_C(-26157), INT16_C( 19197),
                            INT16_C(-27593), INT16_C(-20030), INT16_C(  3690), INT16_C( -3850)),
      simde_mm512_set_epi16(INT16_C(-11908), INT16_C( 14774), INT16_C(  5244), INT16_C( 18107),
                            INT16_C(-16396), INT16_C( 31910), INT16_C(-28865), INT16_C(-20038),
                            INT16_C(-19234), INT16_C(-15108), INT16_C(-10436), INT16_C( 19911),
                            INT16_C(  3330), INT16_C( 28633), INT16_C( 10550), INT16_C( -9358),
                            INT16_C( 23697), INT16_C( 19726), INT16_C(-26407), INT16_C(-18878),
                            INT16_C(  4326), INT16_C(-22642), INT16_C(-17402), INT16_C( 16035),
                            INT16_C( 14223), INT16_C(-15160), INT16_C( -9470), INT16_C( -3752),
                            INT16_C(  6710), INT16_C( 21116), INT16_C( -9579), INT16_C( 10253)),
      simde_mm512_set_epi16(INT16_C(-19653), INT16_C(  4175), INT16_C( -7531), INT16_C( 26895),
                            INT16_C( 15339), INT16_C( 30580), INT16_C(-14533), INT16_C(  9007),
                            INT16_C( 19277), INT16_C(  8842), INT16_C( -9654), INT16_C( 23232),
                            INT16_C(-29376), INT16_C( 13200), INT16_C( 17542), INT16_C( 15704),
                            INT16_C(-13389), INT16_C(-19154), INT16_C( 26412), INT16_C(-31352),
                            INT16_C(-23755), INT16_C( 16831), INT16_C(-10018), INT16_C( 20373),
                            INT16_C(-27899), INT16_C( 11487), INT16_C(-16687), INT16_C( 22949),
                            INT16_C( 31233), INT16_C( 24390), INT16_C( 13269), INT16_C(-14103)) },
    { simde_mm512_set_epi16(INT16_C(  1468), INT16_C( -4389), INT16_C(  1296), INT16_C(-27715),
                            INT16_C(-15620), INT16_C(  3731), INT16_C( -7289), INT16_C(-27703),
                            INT16_C(   474), INT16_C( 27447), INT16_C( -9036), INT16_C(  9176),
                            INT16_C(  2726), INT16_C(-12144), INT16_C( -2101), INT16_C( 26907),
                            INT16_C(-24700), INT16_C(  1244), INT16_C( -3927), INT16_C(-22632),
                            INT16_C( -7525), INT16_C( 17743), INT16_C( 15263), INT16_C( -3823),
                            INT16_C( 27307), INT16_C( 32391), INT16_C(-23270), INT16_C(-29301),
                            INT16_C( 23369), INT16_C(-15291), INT16_C( -5840), INT16_C( 18168)),
      simde_mm512_set_epi16(INT16_C( 23449), INT16_C( 17725), INT16_C(-20919), INT16_C( 31466),
                            INT16_C( 31308), INT16_C( -2183), INT16_C(-31351), INT16_C(-32386),
                            INT16_C( 26890), INT16_C(-30591), INT16_C(-12785), INT16_C(-23638),
                            INT16_C(-31955), INT16_C( -9847), INT16_C( 19108), INT16_C(-19915),
                            INT16_C(  4587), INT16_C( 27034), INT16_C(   -19), INT16_C( 28332),
                            INT16_C(-23789), INT16_C(-24960), INT16_C( -5839), INT16_C( 25722),
                            INT16_C(-24423), INT16_C( 15592), INT16_C(  6092), INT16_C( -9272),
                            INT16_C(-12796), INT16_C(-17663), INT16_C( -6154), INT16_C( 23859)),
      simde_mm512_set_epi16(INT16_C(-21981), INT16_C(-22114), INT16_C( 22215), INT16_C(  6355),
                            INT16_C( 18608), INT16_C(  5914), INT16_C( 24062), INT16_C(  4683),
                            INT16_C(-26416), INT16_C( -7498), INT16_C(  3749), INT16_C(-32722),
                            INT16_C(-30855), INT16_C( -2297), INT16_C(-21209), INT16_C(-18714),
                            INT16_C(-29287), INT16_C(-25790), INT16_C( -3908), INT16_C( 14572),
                            INT16_C( 16264), INT16_C(-22833), INT16_C( 21102), INT16_C(-29545),
                            INT16_C(-13806), INT16_C( 16799), INT16_C(-29362), INT16_C(-20029),
                            INT16_C(-29371), INT16_C(  2372), INT16_C(   314), INT16_C( -5691)) },
    { simde_mm512_set_epi16(INT16_C(-22741), INT16_C( 13394), INT16_C( -9417), INT16_C( 28906),
                            INT16_C(-18980), INT16_C( -8463), INT16_C(  9174), INT16_C(-25605),
                            INT16_C(   547), INT16_C(  3767), INT16_C(-12577), INT16_C(-16546),
                            INT16_C( -1301), INT16_C( -7147), INT16_C( 26281), INT16_C( 29309),
                            INT16_C( 29052), INT16_C(-30842), INT16_C(  5995), INT16_C(  6270),
                            INT16_C( 20539), INT16_C( 10179), INT16_C(-26848), INT16_C( 14327),
                            INT16_C( 15491), INT16_C( 18652), INT16_C( 19903), INT16_C( 30123),
                            INT16_C( 25261), INT16_C(-17460), INT16_C( 10742), INT16_C( -4552)),
      simde_mm512_set_epi16(INT16_C(  5754), INT16_C(-23038), INT16_C(-16589), INT16_C(-23858),
                            INT16_C( -3821), INT16_C( -4798), INT16_C( 30602), INT16_C(-28532),
                            INT16_C( 11508), INT16_C(  7979), INT16_C( -3877), INT16_C( -5920),
                            INT16_C(-24150), INT16_C(-24496), INT16_C( 17421), INT16_C( -1981),
                            INT16_C( 27523), INT16_C( 26800), INT16_C( 25010), INT16_C( 27339),
                            INT16_C( -9050), INT16_C( 19128), INT16_C( 15279), INT16_C( -1817),
                            INT16_C(-13923), INT16_C(  5129), INT16_C(-22618), INT16_C( 27704),
                            INT16_C( -4783), INT16_C( 31238), INT16_C(-30342), INT16_C( -8854)),
      simde_mm512_set_epi16(INT16_C(-28495), INT16_C(-29104), INT16_C(  7172), INT16_C(-12772),
                            INT16_C(-15159), INT16_C( -3665), INT16_C(-21428), INT16_C(  2927),
                            INT16_C(-10961), INT16_C( -4212), INT16_C( -8700), INT16_C(-10626),
                            INT16_C( 22849), INT16_C( 17349), INT16_C(  8860), INT16_C( 31290),
                            INT16_C(  1529), INT16_C(  7894), INT16_C(-19015), INT16_C(-21069),
                            INT16_C( 29589), INT16_C( -8949), INT16_C( 23409), INT16_C( 16144),
                            INT16_C( 29414), INT16_C( 13523), INT16_C(-23015), INT16_C(  2419),
                            INT16_C( 30044), INT16_C( 16838), INT16_C(-24452), INT16_C(  4302)) },
    { simde_mm512_set_epi16(INT16_C( 27021), INT16_C( 31131), INT16_C(    63), INT16_C(-10530),
                            INT16_C( -1071), INT16_C(-31284), INT16_C(-21788), INT16_C(-16108),
                            INT16_C(-15167), INT16_C( 25422), INT16_C( 14520), INT16_C(-13896),
                            INT16_C( 20399), INT16_C( 31915), INT16_C(-16518), INT16_C( -6202),
                            INT16_C(-16332), INT16_C( -3071), INT16_C(-15644), INT16_C( -7016),
                            INT16_C( 13977), INT16_C(-13846), INT16_C(-23290), INT16_C( -2079),
                            INT16_C(  4753), INT16_C( 14919), INT16_C(-18528), INT16_C(  7420),
                            INT16_C( 12098), INT16_C( 31014), INT16_C( 17813), INT16_C(-14456)),
      simde_mm512_set_epi16(INT16_C(-12529), INT16_C( -3643), INT16_C(-28826), INT16_C(-12110),
                            INT16_C(  8030), INT16_C( 20316), INT16_C( 27324), INT16_C( 24735),
                            INT16_C( -6774), INT16_C( -2704), INT16_C(-31930), INT16_C(  6874),
                            INT16_C( -3952), INT16_C(  2658), INT16_C(  -904), INT16_C( -8319),
                            INT16_C(-16424), INT16_C( 22778), INT16_C( 18985), INT16_C( 10063),
                            INT16_C(-31751), INT16_C( 16016), INT16_C(-30217), INT16_C( 18364),
                            INT16_C(-20176), INT16_C( -4961), INT16_C(-29576), INT16_C(-16634),
                            INT16_C( -8011), INT16_C(-27110), INT16_C(-24526), INT16_C(-11504)),
      simde_mm512_set_epi16(INT16_C(-25986), INT16_C(-30762), INT16_C( 28889), INT16_C(  1580),
                            INT16_C( -9101), INT16_C( 13936), INT16_C( 16424), INT16_C( 24693),
                            INT16_C( -8393), INT16_C( 28126), INT16_C(-19086), INT16_C(-20770),
                            INT16_C( 24351), INT16_C( 29257), INT16_C(-15614), INT16_C(  2117),
                            INT16_C(    92), INT16_C(-25849), INT16_C( 30907), INT16_C(-17079),
                            INT16_C(-19808), INT16_C(-29862), INT16_C(  6927), INT16_C(-20443),
                            INT16_C( 24929), INT16_C( 19880), INT16_C( 11048), INT16_C( 24054),
                            INT16_C( 20109), INT16_C( -7412), INT16_C(-23197), INT16_C( -2952)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_sub_epi16(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_sub_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(-2076524081), INT32_C( 1825078206), INT32_C(-1787857556), INT32_C(-1179707533),
                            INT32_C(  233802890), INT32_C( 1015107327), INT32_C(-1130135421), INT32_C(  769270921),
                            INT32_C(  970769619), INT32_C( -152032958), INT32_C(-1037455861), INT32_C( 1543352525),
                            INT32_C( 1997985923), INT32_C( 1878044503), INT32_C(   49641854), INT32_C(   78691943)),
      simde_mm512_set_epi32(INT32_C( 1273959589), INT32_C(  730948807), INT32_C(  152082522), INT32_C(  516109144),
                            INT32_C( -608654122), INT32_C(  326583665), INT32_C(-2143544685), INT32_C( 2015525957),
                            INT32_C(-1762782050), INT32_C( 1423018518), INT32_C(-1835751490), INT32_C(-1847524510),
                            INT32_C( 1152317453), INT32_C( -732966175), INT32_C(  916913335), INT32_C(-1961618071)),
      simde_mm512_set_epi32(INT32_C(  944483626), INT32_C( 1094129399), INT32_C(-1939940078), INT32_C(-1695816677),
                            INT32_C(  842457012), INT32_C(  688523662), INT32_C( 1013409264), INT32_C(-1246255036),
                            INT32_C(-1561415627), INT32_C(-1575051476), INT32_C(  798295629), INT32_C( -904090261),
                            INT32_C(  845668470), INT32_C(-1683956618), INT32_C( -867271481), INT32_C( 2040310014)) },
    { simde_mm512_set_epi32(INT32_C( 1516029066), INT32_C( 1696213023), INT32_C(  690963136), INT32_C( -395017807),
                            INT32_C(-1227102652), INT32_C( 1731549524), INT32_C( 1416885076), INT32_C( -891143280),
                            INT32_C(-1187279454), INT32_C(  699906112), INT32_C(  947982370), INT32_C(-1809113234),
                            INT32_C(  892884346), INT32_C(  173663466), INT32_C( -426903082), INT32_C(-1178201759)),
      simde_mm512_set_epi32(INT32_C(  565568434), INT32_C( 1477571639), INT32_C( -962268135), INT32_C(   93801511),
                            INT32_C( 1166085377), INT32_C(-1613873583), INT32_C( 1514751666), INT32_C(   -9777248),
                            INT32_C(  880861168), INT32_C(-1847118927), INT32_C( -454330268), INT32_C( -465889797),
                            INT32_C(  607148382), INT32_C( -892911578), INT32_C(-1830027716), INT32_C(  171620514)),
      simde_mm512_set_epi32(INT32_C(  950460632), INT32_C(  218641384), INT32_C( 1653231271), INT32_C( -488819318),
                            INT32_C( 1901779267), INT32_C( -949544189), INT32_C(  -97866590), INT32_C( -881366032),
                            INT32_C(-2068140622), INT32_C(-1747942257), INT32_C( 1402312638), INT32_C(-1343223437),
                            INT32_C(  285735964), INT32_C( 1066575044), INT32_C( 1403124634), INT32_C(-1349822273)) },
    { simde_mm512_set_epi32(INT32_C( -894737208), INT32_C( -894707310), INT32_C(-1734937643), INT32_C(-1821919338),
                            INT32_C(-1629473200), INT32_C( 1017176222), INT32_C(  555630880), INT32_C( 1893052174),
                            INT32_C( -395602197), INT32_C(  851153269), INT32_C( 1448617638), INT32_C( 1939202047),
                            INT32_C(-1165352739), INT32_C(  784136789), INT32_C(-1222569677), INT32_C(-1663359991)),
      simde_mm512_set_epi32(INT32_C(-2134962383), INT32_C(  711344265), INT32_C( -499544380), INT32_C(  658556967),
                            INT32_C(-1607446648), INT32_C(-2074003952), INT32_C(  449264495), INT32_C( -469125832),
                            INT32_C(-1465796532), INT32_C( -575249454), INT32_C( -236269065), INT32_C(  567769266),
                            INT32_C( -145854210), INT32_C(  502784491), INT32_C( -258238741), INT32_C( 1554234017)),
      simde_mm512_set_epi32(INT32_C( 1240225175), INT32_C(-1606051575), INT32_C(-1235393263), INT32_C( 1814490991),
                            INT32_C(  -22026552), INT32_C(-1203787122), INT32_C(  106366385), INT32_C(-1932789290),
                            INT32_C( 1070194335), INT32_C( 1426402723), INT32_C( 1684886703), INT32_C( 1371432781),
                            INT32_C(-1019498529), INT32_C(  281352298), INT32_C( -964330936), INT32_C( 1077373288)) },
    { simde_mm512_set_epi32(INT32_C( -658606825), INT32_C(-1465142546), INT32_C(-1613315081), INT32_C( 1981327993),
                            INT32_C( -540883338), INT32_C(  -52568431), INT32_C(  513288938), INT32_C(-1741957410),
                            INT32_C( -457290370), INT32_C(  949496535), INT32_C( -574503672), INT32_C( -516003313),
                            INT32_C( 1705152287), INT32_C(  268459282), INT32_C( -796672854), INT32_C(-2124069536)),
      simde_mm512_set_epi32(INT32_C(-1627464574), INT32_C(  688417349), INT32_C(-1204757032), INT32_C(-1541532775),
                            INT32_C( -489028243), INT32_C(  -14341503), INT32_C( 1546753292), INT32_C( -383774267),
                            INT32_C( 1479759913), INT32_C(-1792003336), INT32_C(  324281321), INT32_C(-1031805126),
                            INT32_C(-1668912025), INT32_C( -271675366), INT32_C(-1502890080), INT32_C( -582208760)),
      simde_mm512_set_epi32(INT32_C(  968857749), INT32_C( 2141407401), INT32_C( -408558049), INT32_C( -772106528),
                            INT32_C(  -51855095), INT32_C(  -38226928), INT32_C(-1033464354), INT32_C(-1358183143),
                            INT32_C(-1937050283), INT32_C(-1553467425), INT32_C( -898784993), INT32_C(  515801813),
                            INT32_C( -920902984), INT32_C(  540134648), INT32_C(  706217226), INT32_C(-1541860776)) },
    { simde_mm512_set_epi32(INT32_C( 1656401797), INT32_C(   50049750), INT32_C( -488722048), INT32_C( 1532620410),
                            INT32_C(  761833085), INT32_C(  -28253750), INT32_C( 1071891913), INT32_C( -578065038),
                            INT32_C( 2114869114), INT32_C( 1114386003), INT32_C( -192755303), INT32_C( -163390023),
                            INT32_C(-1012186074), INT32_C( -258665152), INT32_C(  548389384), INT32_C( -601025611)),
      simde_mm512_set_epi32(INT32_C( -962813354), INT32_C(-1563683363), INT32_C( 1476422960), INT32_C(-1996230234),
                            INT32_C(  594356694), INT32_C(  -37573818), INT32_C( 2109710080), INT32_C(-2049942476),
                            INT32_C(-1449482441), INT32_C(-1892730921), INT32_C( 1298337068), INT32_C(   30251788),
                            INT32_C( -250852108), INT32_C(-2130168940), INT32_C(  414197854), INT32_C( -971416192)),
      simde_mm512_set_epi32(INT32_C(-1675752145), INT32_C( 1613733113), INT32_C(-1965145008), INT32_C( -766116652),
                            INT32_C(  167476391), INT32_C(    9320068), INT32_C(-1037818167), INT32_C( 1471877438),
                            INT32_C( -730615741), INT32_C(-1287850372), INT32_C(-1491092371), INT32_C( -193641811),
                            INT32_C( -761333966), INT32_C( 1871503788), INT32_C(  134191530), INT32_C(  370390581)) },
    { simde_mm512_set_epi32(INT32_C(  841332080), INT32_C(  332746710), INT32_C( 1180202036), INT32_C(-1365461084),
                            INT32_C( -972107726), INT32_C( -919074620), INT32_C(  336794208), INT32_C(-2145769013),
                            INT32_C(-1090767268), INT32_C( 1447456701), INT32_C(-1878509449), INT32_C( 1479468832),
                            INT32_C(-2038652659), INT32_C( -428110707), INT32_C( -605535334), INT32_C( 1876977582)),
      simde_mm512_set_epi32(INT32_C(-1104919125), INT32_C(-1965384352), INT32_C( 1846340148), INT32_C( 1439724559),
                            INT32_C( 1174009148), INT32_C( -500908704), INT32_C( 2074430235), INT32_C(  746110301),
                            INT32_C( -229497465), INT32_C(  567264435), INT32_C(-1820479715), INT32_C( -409682629),
                            INT32_C(-1976550605), INT32_C(-1717329929), INT32_C(  392593328), INT32_C(  809330056)),
      simde_mm512_set_epi32(INT32_C( 1946251205), INT32_C(-1996836234), INT32_C( -666138112), INT32_C( 1489781653),
                            INT32_C(-2146116874), INT32_C( -418165916), INT32_C(-1737636027), INT32_C( 1403087982),
                            INT32_C( -861269803), INT32_C(  880192266), INT32_C(  -58029734), INT32_C( 1889151461),
                            INT32_C(  -62102054), INT32_C( 1289219222), INT32_C( -998128662), INT32_C( 1067647526)) },
    { simde_mm512_set_epi32(INT32_C(-1188475624), INT32_C(-1471681451), INT32_C( -219755555), INT32_C(-1657771963),
                            INT32_C( -257604504), INT32_C(  874981434), INT32_C(-1610485047), INT32_C(-1272947332),
                            INT32_C( 1561476022), INT32_C(  375243187), INT32_C( 1479356717), INT32_C( 1523794483),
                            INT32_C(-1698967593), INT32_C(  -80864233), INT32_C( 1644091986), INT32_C( -229623607)),
      simde_mm512_set_epi32(INT32_C(    9741774), INT32_C(  693305140), INT32_C(-1221395242), INT32_C(-1923328842),
                            INT32_C(   85084148), INT32_C( 1125599333), INT32_C( 2042080920), INT32_C( -456911551),
                            INT32_C( -399701639), INT32_C(-1860388051), INT32_C( -699039468), INT32_C(   84523143),
                            INT32_C(-1293034841), INT32_C(-1626054083), INT32_C(   96950550), INT32_C( 1663457642)),
      simde_mm512_set_epi32(INT32_C(-1198217398), INT32_C( 2129980705), INT32_C( 1001639687), INT32_C(  265556879),
                            INT32_C( -342688652), INT32_C( -250617899), INT32_C(  642401329), INT32_C( -816035781),
                            INT32_C( 1961177661), INT32_C(-2059336058), INT32_C(-2116571111), INT32_C( 1439271340),
                            INT32_C( -405932752), INT32_C( 1545189850), INT32_C( 1547141436), INT32_C(-1893081249)) },
    { simde_mm512_set_epi32(INT32_C(-1473946007), INT32_C(  121708864), INT32_C( 1020809582), INT32_C( 1669312470),
                            INT32_C( -682688365), INT32_C(  500732292), INT32_C( 1673154382), INT32_C(-1552445241),
                            INT32_C( 2068495467), INT32_C(-2039438173), INT32_C(  869593130), INT32_C( -471794528),
                            INT32_C(-1539319849), INT32_C( 1041904784), INT32_C( -120989465), INT32_C(-1180697219)),
      simde_mm512_set_epi32(INT32_C( 1388066655), INT32_C( 1341381019), INT32_C(-1738591736), INT32_C( -783428109),
                            INT32_C(-1884288937), INT32_C( 1340467391), INT32_C(-1349575878), INT32_C(-1411283384),
                            INT32_C( 1173507492), INT32_C( 1805408001), INT32_C( 1184512890), INT32_C(-1180223583),
                            INT32_C( -121255394), INT32_C(-2007254522), INT32_C(  970045213), INT32_C(-2132245994)),
      simde_mm512_set_epi32(INT32_C( 1432954634), INT32_C(-1219672155), INT32_C(-1535565978), INT32_C(-1842226717),
                            INT32_C( 1201600572), INT32_C( -839735099), INT32_C(-1272237036), INT32_C( -141161857),
                            INT32_C(  894987975), INT32_C(  450121122), INT32_C( -314919760), INT32_C(  708429055),
                            INT32_C(-1418064455), INT32_C(-1245807990), INT32_C(-1091034678), INT32_C(  951548775)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_sub_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_sub_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-8918603015426376770), INT64_C(-7678789729811228813),
                            INT64_C( 1004175767275392767), INT64_C(-4853894672476920695),
                            INT64_C( 4169423769698314562), INT64_C(-4455838992495169331),
                            INT64_C( 8581284199031418711), INT64_C(  213210139521498727)),
      simde_mm512_set_epi64(INT64_C( 5471614771911550151), INT64_C(  653189458799309656),
                            INT64_C(-2614149548239010447), INT64_C(-9206454317574095803),
                            INT64_C(-7571091253302818282), INT64_C(-7884492610685828254),
                            INT64_C( 4949165778807018209), INT64_C( 3938112789424641385)),
      simde_mm512_set_epi64(INT64_C( 4056526286371624695), INT64_C(-8331979188610538469),
                            INT64_C( 3618325315514403214), INT64_C( 4352559645097175108),
                            INT64_C(-6706229050708418772), INT64_C( 3428653618190658923),
                            INT64_C( 3632118420224400502), INT64_C(-3724902649903142658)) },
    { simde_mm512_set_epi64(INT64_C( 6511295259951638559), INT64_C( 2967664075761549745),
                            INT64_C(-5270365757443319468), INT64_C( 6085475067014298512),
                            INT64_C(-5099326425442830272), INT64_C( 4071553278820425582),
                            INT64_C( 3834909065354011882), INT64_C(-1833534772634840735)),
      simde_mm512_set_epi64(INT64_C( 2429097929157506103), INT64_C(-4132910169714111449),
                            INT64_C( 5008298561239924305), INT64_C( 6505808871316705184),
                            INT64_C( 3783269911324210097), INT64_C(-1951333638813837829),
                            INT64_C( 2607682447911370790), INT64_C(-7859909190821955422)),
      simde_mm512_set_epi64(INT64_C( 4082197330794132456), INT64_C( 7100574245475661194),
                            INT64_C( 8168079755026307843), INT64_C( -420333804302406672),
                            INT64_C(-8882596336767040369), INT64_C( 6022886917634263411),
                            INT64_C( 1227226617442641092), INT64_C( 6026374418187114687)) },
    { simde_mm512_set_epi64(INT64_C(-3842867043474089582), INT64_C(-7451500434811275370),
                            INT64_C(-6998534102691290978), INT64_C( 2386416460140752654),
                            INT64_C(-1699098497489596043), INT64_C( 6221765381557968895),
                            INT64_C(-5005151901524886955), INT64_C(-5250896777164676087)),
      simde_mm512_set_epi64(INT64_C(-9169593612463882103), INT64_C(-2145526774342039513),
                            INT64_C(-6903930781003860464), INT64_C( 1929576317104796984),
                            INT64_C(-6295548163810499630), INT64_C(-1014767906663728974),
                            INT64_C( -626439061431131669), INT64_C(-1109126945600980319)),
      simde_mm512_set_epi64(INT64_C( 5326726568989792521), INT64_C(-5305973660469235857),
                            INT64_C(  -94603321687430514), INT64_C(  456840143035955670),
                            INT64_C( 4596449666320903587), INT64_C( 7236533288221697869),
                            INT64_C(-4378712840093755286), INT64_C(-4141769831563695768)) },
    { simde_mm512_set_epi64(INT64_C(-2828694771467570450), INT64_C(-6929135509057262983),
                            INT64_C(-2323076243418915183), INT64_C( 2204559204661581534),
                            INT64_C(-1964047182976242985), INT64_C(-2467474478892946929),
                            INT64_C( 7323573307633065234), INT64_C(-3421683851370085024)),
      simde_mm512_set_epi64(INT64_C(-6989907120040154555), INT64_C(-5174392049312590951),
                            INT64_C(-2100360306224715135), INT64_C( 6643254808031531461),
                            INT64_C( 6355520434769769208), INT64_C( 1392777671661840186),
                            INT64_C(-7167922563252842470), INT64_C(-6454863739370065144)),
      simde_mm512_set_epi64(INT64_C( 4161212348572584105), INT64_C(-1754743459744672032),
                            INT64_C( -222715937194200048), INT64_C(-4438695603369949927),
                            INT64_C(-8319567617746012193), INT64_C(-3860252150554787115),
                            INT64_C(-3955248202823643912), INT64_C( 3033179887999980120)) },
    { simde_mm512_set_epi64(INT64_C( 7114191547200680662), INT64_C(-2099045211461521798),
                            INT64_C( 3272048189352501706), INT64_C( 4603740714898779506),
                            INT64_C( 9083293681064881747), INT64_C( -827877718383993415),
                            INT64_C(-4347306081260333760), INT64_C( 2355314473447527349)),
      simde_mm512_set_epi64(INT64_C(-4135251864850786851), INT64_C( 6341188330562253222),
                            INT64_C( 2552742567146072902), INT64_C( 9061135799886568500),
                            INT64_C(-6225479677819013161), INT64_C( 5576315246274779916),
                            INT64_C(-1077401597827861612), INT64_C( 1778966240326933888)),
      simde_mm512_set_epi64(INT64_C(-7197300661658084103), INT64_C(-8440233542023775020),
                            INT64_C(  719305622206428804), INT64_C(-4457395084987788994),
                            INT64_C(-3137970714825656708), INT64_C(-6404192964658773331),
                            INT64_C(-3269904483432472148), INT64_C(  576348233120593461)) },
    { simde_mm512_set_epi64(INT64_C( 3613493769008402390), INT64_C( 5068929150222120868),
                            INT64_C(-4175170887983036220), INT64_C( 1446520110991419851),
                            INT64_C(-4684809742159810627), INT64_C(-8068136647202511072),
                            INT64_C(-8755946494441583475), INT64_C(-2600754454225459282)),
      simde_mm512_set_epi64(INT64_C(-4745591504270353056), INT64_C( 7929970554391524367),
                            INT64_C( 5042330899658882400), INT64_C( 8909610017904704861),
                            INT64_C( -985684106122640205), INT64_C(-7818900835071115973),
                            INT64_C(-8489220204786376713), INT64_C( 1686175505197131144)),
      simde_mm512_set_epi64(INT64_C( 8359085273278755446), INT64_C(-2861041404169403499),
                            INT64_C(-9217501787641918620), INT64_C(-7463089906913285010),
                            INT64_C(-3699125636037170422), INT64_C( -249235812131395099),
                            INT64_C( -266726289655206762), INT64_C(-4286929959422590426)) },
    { simde_mm512_set_epi64(INT64_C(-5104463934349906859), INT64_C( -943842919202133947),
                            INT64_C(-1106402919107319750), INT64_C(-6916980604540002948),
                            INT64_C( 6706488448353419699), INT64_C( 6353788720156721715),
                            INT64_C(-7297010244684735465), INT64_C( 7061321315551033545)),
      simde_mm512_set_epi64(INT64_C(   41840601428328244), INT64_C(-5245852617508367178),
                            INT64_C(  365433634193623141), INT64_C( 8770670771023648065),
                            INT64_C(-1716705465228018899), INT64_C(-3002351653588715385),
                            INT64_C(-5553542352014646723), INT64_C(  416399443242670442)),
      simde_mm512_set_epi64(INT64_C(-5146304535778235103), INT64_C( 4302009698306233231),
                            INT64_C(-1471836553300942891), INT64_C( 2759092698145900603),
                            INT64_C( 8423193913581438598), INT64_C(-9090603699964114516),
                            INT64_C(-1743467892670088742), INT64_C( 6644921872308363103)) },
    { simde_mm512_set_epi64(INT64_C(-6330549896013078208), INT64_C( 4384343771802742742),
                            INT64_C(-2932124200533978748), INT64_C( 7186143354591613127),
                            INT64_C( 8884120384944776355), INT64_C( 3734874057999449248),
                            INT64_C(-6611328408496753520), INT64_C( -519645792221266563)),
      simde_mm512_set_epi64(INT64_C( 5961700889234495899), INT64_C(-7467194643704326669),
                            INT64_C(-8092959359289136961), INT64_C(-5796384256596801976),
                            INT64_C( 5040176301556389633), INT64_C( 5087444127355189153),
                            INT64_C( -520787949405881850), INT64_C( 4166312467639075350)),
      simde_mm512_set_epi64(INT64_C( 6154493288461977509), INT64_C(-6595205658202482205),
                            INT64_C( 5160835158755158213), INT64_C(-5464216462521136513),
                            INT64_C( 3843944083388386722), INT64_C(-1352570069355739905),
                            INT64_C(-6090540459090871670), INT64_C(-4685958259860341913)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_sub_epi64(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_sub_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -659.63), SIMDE_FLOAT32_C(  -759.67), SIMDE_FLOAT32_C(  -847.92), SIMDE_FLOAT32_C(   -61.45),
                         SIMDE_FLOAT32_C(  -337.36), SIMDE_FLOAT32_C(   139.68), SIMDE_FLOAT32_C(   658.69), SIMDE_FLOAT32_C(    86.55),
                         SIMDE_FLOAT32_C(  -150.13), SIMDE_FLOAT32_C(   450.66), SIMDE_FLOAT32_C(  -527.30), SIMDE_FLOAT32_C(  -641.78),
                         SIMDE_FLOAT32_C(   929.20), SIMDE_FLOAT32_C(  -281.32), SIMDE_FLOAT32_C(  -125.47), SIMDE_FLOAT32_C(  -963.36)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -406.77), SIMDE_FLOAT32_C(  -929.18), SIMDE_FLOAT32_C(   716.57), SIMDE_FLOAT32_C(     1.83),
                         SIMDE_FLOAT32_C(   179.14), SIMDE_FLOAT32_C(   145.16), SIMDE_FLOAT32_C(  -463.41), SIMDE_FLOAT32_C(  -573.03),
                         SIMDE_FLOAT32_C(    33.04), SIMDE_FLOAT32_C(   167.46), SIMDE_FLOAT32_C(  -891.13), SIMDE_FLOAT32_C(   473.74),
                         SIMDE_FLOAT32_C(  -547.95), SIMDE_FLOAT32_C(   516.90), SIMDE_FLOAT32_C(   -69.62), SIMDE_FLOAT32_C(  -976.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -252.86), SIMDE_FLOAT32_C(   169.51), SIMDE_FLOAT32_C( -1564.49), SIMDE_FLOAT32_C(   -63.28),
                         SIMDE_FLOAT32_C(  -516.50), SIMDE_FLOAT32_C(    -5.48), SIMDE_FLOAT32_C(  1122.10), SIMDE_FLOAT32_C(   659.58),
                         SIMDE_FLOAT32_C(  -183.17), SIMDE_FLOAT32_C(   283.20), SIMDE_FLOAT32_C(   363.83), SIMDE_FLOAT32_C( -1115.52),
                         SIMDE_FLOAT32_C(  1477.15), SIMDE_FLOAT32_C(  -798.22), SIMDE_FLOAT32_C(   -55.85), SIMDE_FLOAT32_C(    13.52)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -311.95), SIMDE_FLOAT32_C(  -956.32), SIMDE_FLOAT32_C(   248.48), SIMDE_FLOAT32_C(   995.45),
                         SIMDE_FLOAT32_C(   139.87), SIMDE_FLOAT32_C(   783.05), SIMDE_FLOAT32_C(   584.21), SIMDE_FLOAT32_C(  -920.08),
                         SIMDE_FLOAT32_C(  -210.14), SIMDE_FLOAT32_C(   816.06), SIMDE_FLOAT32_C(  -193.68), SIMDE_FLOAT32_C(   585.03),
                         SIMDE_FLOAT32_C(  -674.08), SIMDE_FLOAT32_C(   157.57), SIMDE_FLOAT32_C(  -919.13), SIMDE_FLOAT32_C(   451.36)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -736.64), SIMDE_FLOAT32_C(   551.91), SIMDE_FLOAT32_C(  -457.00), SIMDE_FLOAT32_C(  -294.64),
                         SIMDE_FLOAT32_C(  -589.82), SIMDE_FLOAT32_C(   788.44), SIMDE_FLOAT32_C(  -717.27), SIMDE_FLOAT32_C(   147.83),
                         SIMDE_FLOAT32_C(  -294.04), SIMDE_FLOAT32_C(  -678.25), SIMDE_FLOAT32_C(   428.59), SIMDE_FLOAT32_C(  -340.21),
                         SIMDE_FLOAT32_C(   447.13), SIMDE_FLOAT32_C(  -558.56), SIMDE_FLOAT32_C(  -584.22), SIMDE_FLOAT32_C(   801.21)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   424.69), SIMDE_FLOAT32_C( -1508.23), SIMDE_FLOAT32_C(   705.48), SIMDE_FLOAT32_C(  1290.09),
                         SIMDE_FLOAT32_C(   729.69), SIMDE_FLOAT32_C(    -5.39), SIMDE_FLOAT32_C(  1301.48), SIMDE_FLOAT32_C( -1067.91),
                         SIMDE_FLOAT32_C(    83.90), SIMDE_FLOAT32_C(  1494.31), SIMDE_FLOAT32_C(  -622.27), SIMDE_FLOAT32_C(   925.24),
                         SIMDE_FLOAT32_C( -1121.21), SIMDE_FLOAT32_C(   716.13), SIMDE_FLOAT32_C(  -334.91), SIMDE_FLOAT32_C(  -349.85)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -668.75), SIMDE_FLOAT32_C(  -693.34), SIMDE_FLOAT32_C(    34.22), SIMDE_FLOAT32_C(   781.55),
                         SIMDE_FLOAT32_C(   732.13), SIMDE_FLOAT32_C(  -735.61), SIMDE_FLOAT32_C(  -765.87), SIMDE_FLOAT32_C(  -276.25),
                         SIMDE_FLOAT32_C(   583.37), SIMDE_FLOAT32_C(   151.60), SIMDE_FLOAT32_C(  -526.34), SIMDE_FLOAT32_C(  -118.48),
                         SIMDE_FLOAT32_C(  -603.65), SIMDE_FLOAT32_C(   -96.99), SIMDE_FLOAT32_C(  -634.86), SIMDE_FLOAT32_C(   225.44)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     5.83), SIMDE_FLOAT32_C(   767.38), SIMDE_FLOAT32_C(   251.47), SIMDE_FLOAT32_C(  -790.79),
                         SIMDE_FLOAT32_C(   317.44), SIMDE_FLOAT32_C(   889.98), SIMDE_FLOAT32_C(   932.08), SIMDE_FLOAT32_C(   879.75),
                         SIMDE_FLOAT32_C(   583.36), SIMDE_FLOAT32_C(   192.11), SIMDE_FLOAT32_C(   241.22), SIMDE_FLOAT32_C(  -741.26),
                         SIMDE_FLOAT32_C(   815.78), SIMDE_FLOAT32_C(  -325.43), SIMDE_FLOAT32_C(   457.34), SIMDE_FLOAT32_C(   430.70)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -674.58), SIMDE_FLOAT32_C( -1460.72), SIMDE_FLOAT32_C(  -217.25), SIMDE_FLOAT32_C(  1572.34),
                         SIMDE_FLOAT32_C(   414.69), SIMDE_FLOAT32_C( -1625.59), SIMDE_FLOAT32_C( -1697.95), SIMDE_FLOAT32_C( -1156.00),
                         SIMDE_FLOAT32_C(     0.01), SIMDE_FLOAT32_C(   -40.51), SIMDE_FLOAT32_C(  -767.56), SIMDE_FLOAT32_C(   622.78),
                         SIMDE_FLOAT32_C( -1419.43), SIMDE_FLOAT32_C(   228.44), SIMDE_FLOAT32_C( -1092.20), SIMDE_FLOAT32_C(  -205.26)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -679.43), SIMDE_FLOAT32_C(   282.17), SIMDE_FLOAT32_C(   993.32), SIMDE_FLOAT32_C(   821.29),
                         SIMDE_FLOAT32_C(   165.53), SIMDE_FLOAT32_C(   519.53), SIMDE_FLOAT32_C(   873.49), SIMDE_FLOAT32_C(   728.89),
                         SIMDE_FLOAT32_C(   317.74), SIMDE_FLOAT32_C(   -77.37), SIMDE_FLOAT32_C(   975.52), SIMDE_FLOAT32_C(   188.84),
                         SIMDE_FLOAT32_C(  -557.86), SIMDE_FLOAT32_C(   759.72), SIMDE_FLOAT32_C(  -874.99), SIMDE_FLOAT32_C(    10.90)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   242.15), SIMDE_FLOAT32_C(   438.99), SIMDE_FLOAT32_C(   772.28), SIMDE_FLOAT32_C(  -279.74),
                         SIMDE_FLOAT32_C(  -310.93), SIMDE_FLOAT32_C(  -848.99), SIMDE_FLOAT32_C(   222.85), SIMDE_FLOAT32_C(   300.16),
                         SIMDE_FLOAT32_C(   693.31), SIMDE_FLOAT32_C(   248.74), SIMDE_FLOAT32_C(   748.13), SIMDE_FLOAT32_C(  -760.98),
                         SIMDE_FLOAT32_C(   787.06), SIMDE_FLOAT32_C(   732.48), SIMDE_FLOAT32_C(  -205.98), SIMDE_FLOAT32_C(   629.02)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -921.58), SIMDE_FLOAT32_C(  -156.82), SIMDE_FLOAT32_C(   221.04), SIMDE_FLOAT32_C(  1101.03),
                         SIMDE_FLOAT32_C(   476.46), SIMDE_FLOAT32_C(  1368.52), SIMDE_FLOAT32_C(   650.64), SIMDE_FLOAT32_C(   428.73),
                         SIMDE_FLOAT32_C(  -375.57), SIMDE_FLOAT32_C(  -326.11), SIMDE_FLOAT32_C(   227.39), SIMDE_FLOAT32_C(   949.82),
                         SIMDE_FLOAT32_C( -1344.92), SIMDE_FLOAT32_C(    27.24), SIMDE_FLOAT32_C(  -669.01), SIMDE_FLOAT32_C(  -618.12)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   271.85), SIMDE_FLOAT32_C(    70.43), SIMDE_FLOAT32_C(   982.50), SIMDE_FLOAT32_C(    45.42),
                         SIMDE_FLOAT32_C(   118.63), SIMDE_FLOAT32_C(  -985.91), SIMDE_FLOAT32_C(     8.06), SIMDE_FLOAT32_C(   547.65),
                         SIMDE_FLOAT32_C(  -976.69), SIMDE_FLOAT32_C(  -286.32), SIMDE_FLOAT32_C(   986.84), SIMDE_FLOAT32_C(   730.82),
                         SIMDE_FLOAT32_C(  -481.07), SIMDE_FLOAT32_C(   923.92), SIMDE_FLOAT32_C(   879.55), SIMDE_FLOAT32_C(   720.13)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   551.66), SIMDE_FLOAT32_C(  -312.49), SIMDE_FLOAT32_C(  -723.23), SIMDE_FLOAT32_C(   -17.59),
                         SIMDE_FLOAT32_C(   325.03), SIMDE_FLOAT32_C(  -395.41), SIMDE_FLOAT32_C(   883.19), SIMDE_FLOAT32_C(  -807.12),
                         SIMDE_FLOAT32_C(  -228.68), SIMDE_FLOAT32_C(   772.42), SIMDE_FLOAT32_C(  -645.24), SIMDE_FLOAT32_C(  -500.86),
                         SIMDE_FLOAT32_C(   -15.19), SIMDE_FLOAT32_C(   910.24), SIMDE_FLOAT32_C(   528.66), SIMDE_FLOAT32_C(  -744.64)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -279.81), SIMDE_FLOAT32_C(   382.92), SIMDE_FLOAT32_C(  1705.73), SIMDE_FLOAT32_C(    63.01),
                         SIMDE_FLOAT32_C(  -206.40), SIMDE_FLOAT32_C(  -590.50), SIMDE_FLOAT32_C(  -875.13), SIMDE_FLOAT32_C(  1354.77),
                         SIMDE_FLOAT32_C(  -748.01), SIMDE_FLOAT32_C( -1058.74), SIMDE_FLOAT32_C(  1632.08), SIMDE_FLOAT32_C(  1231.68),
                         SIMDE_FLOAT32_C(  -465.88), SIMDE_FLOAT32_C(    13.68), SIMDE_FLOAT32_C(   350.89), SIMDE_FLOAT32_C(  1464.77)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(    84.80), SIMDE_FLOAT32_C(  -329.58), SIMDE_FLOAT32_C(   766.75), SIMDE_FLOAT32_C(  -652.57),
                         SIMDE_FLOAT32_C(  -735.85), SIMDE_FLOAT32_C(   809.23), SIMDE_FLOAT32_C(   200.31), SIMDE_FLOAT32_C(  -623.13),
                         SIMDE_FLOAT32_C(  -845.05), SIMDE_FLOAT32_C(   364.16), SIMDE_FLOAT32_C(   572.02), SIMDE_FLOAT32_C(     0.80),
                         SIMDE_FLOAT32_C(  -325.98), SIMDE_FLOAT32_C(  -311.07), SIMDE_FLOAT32_C(   800.65), SIMDE_FLOAT32_C(  -125.96)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   485.48), SIMDE_FLOAT32_C(  -140.23), SIMDE_FLOAT32_C(  -453.31), SIMDE_FLOAT32_C(   -34.02),
                         SIMDE_FLOAT32_C(   893.13), SIMDE_FLOAT32_C(   152.27), SIMDE_FLOAT32_C(    79.60), SIMDE_FLOAT32_C(  -817.18),
                         SIMDE_FLOAT32_C(  -608.22), SIMDE_FLOAT32_C(  -450.43), SIMDE_FLOAT32_C(   547.33), SIMDE_FLOAT32_C(  -843.17),
                         SIMDE_FLOAT32_C(   492.07), SIMDE_FLOAT32_C(   125.25), SIMDE_FLOAT32_C(    50.68), SIMDE_FLOAT32_C(   718.03)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -400.68), SIMDE_FLOAT32_C(  -189.35), SIMDE_FLOAT32_C(  1220.06), SIMDE_FLOAT32_C(  -618.55),
                         SIMDE_FLOAT32_C( -1628.98), SIMDE_FLOAT32_C(   656.96), SIMDE_FLOAT32_C(   120.71), SIMDE_FLOAT32_C(   194.05),
                         SIMDE_FLOAT32_C(  -236.83), SIMDE_FLOAT32_C(   814.59), SIMDE_FLOAT32_C(    24.69), SIMDE_FLOAT32_C(   843.97),
                         SIMDE_FLOAT32_C(  -818.05), SIMDE_FLOAT32_C(  -436.32), SIMDE_FLOAT32_C(   749.97), SIMDE_FLOAT32_C(  -843.99)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -677.15), SIMDE_FLOAT32_C(   104.38), SIMDE_FLOAT32_C(  -475.85), SIMDE_FLOAT32_C(   787.23),
                         SIMDE_FLOAT32_C(   133.69), SIMDE_FLOAT32_C(  -960.64), SIMDE_FLOAT32_C(   242.81), SIMDE_FLOAT32_C(  -225.39),
                         SIMDE_FLOAT32_C(   314.69), SIMDE_FLOAT32_C(   228.04), SIMDE_FLOAT32_C(  -592.56), SIMDE_FLOAT32_C(   407.24),
                         SIMDE_FLOAT32_C(  -825.26), SIMDE_FLOAT32_C(  -290.43), SIMDE_FLOAT32_C(   962.34), SIMDE_FLOAT32_C(   893.07)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -995.46), SIMDE_FLOAT32_C(   431.24), SIMDE_FLOAT32_C(  -960.38), SIMDE_FLOAT32_C(   -49.08),
                         SIMDE_FLOAT32_C(   813.87), SIMDE_FLOAT32_C(   674.48), SIMDE_FLOAT32_C(   397.88), SIMDE_FLOAT32_C(  -954.85),
                         SIMDE_FLOAT32_C(   446.57), SIMDE_FLOAT32_C(   897.67), SIMDE_FLOAT32_C(   880.04), SIMDE_FLOAT32_C(   250.06),
                         SIMDE_FLOAT32_C(  -272.88), SIMDE_FLOAT32_C(  -311.12), SIMDE_FLOAT32_C(   208.86), SIMDE_FLOAT32_C(  -234.41)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   318.31), SIMDE_FLOAT32_C(  -326.86), SIMDE_FLOAT32_C(   484.53), SIMDE_FLOAT32_C(   836.31),
                         SIMDE_FLOAT32_C(  -680.18), SIMDE_FLOAT32_C( -1635.12), SIMDE_FLOAT32_C(  -155.07), SIMDE_FLOAT32_C(   729.46),
                         SIMDE_FLOAT32_C(  -131.88), SIMDE_FLOAT32_C(  -669.63), SIMDE_FLOAT32_C( -1472.60), SIMDE_FLOAT32_C(   157.18),
                         SIMDE_FLOAT32_C(  -552.38), SIMDE_FLOAT32_C(    20.69), SIMDE_FLOAT32_C(   753.48), SIMDE_FLOAT32_C(  1127.48)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -375.37), SIMDE_FLOAT32_C(   635.19), SIMDE_FLOAT32_C(  -375.80), SIMDE_FLOAT32_C(   342.82),
                         SIMDE_FLOAT32_C(  -159.29), SIMDE_FLOAT32_C(   450.42), SIMDE_FLOAT32_C(    65.30), SIMDE_FLOAT32_C(     7.10),
                         SIMDE_FLOAT32_C(  -943.32), SIMDE_FLOAT32_C(  -222.67), SIMDE_FLOAT32_C(  -766.83), SIMDE_FLOAT32_C(   277.09),
                         SIMDE_FLOAT32_C(    50.31), SIMDE_FLOAT32_C(   780.30), SIMDE_FLOAT32_C(  -514.83), SIMDE_FLOAT32_C(   450.20)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -353.63), SIMDE_FLOAT32_C(   190.41), SIMDE_FLOAT32_C(   122.56), SIMDE_FLOAT32_C(   371.55),
                         SIMDE_FLOAT32_C(  -453.54), SIMDE_FLOAT32_C(  -448.42), SIMDE_FLOAT32_C(   943.54), SIMDE_FLOAT32_C(  -548.29),
                         SIMDE_FLOAT32_C(   313.64), SIMDE_FLOAT32_C(  -524.65), SIMDE_FLOAT32_C(   682.10), SIMDE_FLOAT32_C(  -220.88),
                         SIMDE_FLOAT32_C(   -36.78), SIMDE_FLOAT32_C(  -595.06), SIMDE_FLOAT32_C(   283.20), SIMDE_FLOAT32_C(   943.66)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -21.74), SIMDE_FLOAT32_C(   444.78), SIMDE_FLOAT32_C(  -498.36), SIMDE_FLOAT32_C(   -28.73),
                         SIMDE_FLOAT32_C(   294.25), SIMDE_FLOAT32_C(   898.84), SIMDE_FLOAT32_C(  -878.24), SIMDE_FLOAT32_C(   555.39),
                         SIMDE_FLOAT32_C( -1256.96), SIMDE_FLOAT32_C(   301.98), SIMDE_FLOAT32_C( -1448.93), SIMDE_FLOAT32_C(   497.97),
                         SIMDE_FLOAT32_C(    87.09), SIMDE_FLOAT32_C(  1375.36), SIMDE_FLOAT32_C(  -798.03), SIMDE_FLOAT32_C(  -493.46)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_sub_ps(test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_sub_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -150.13), SIMDE_FLOAT64_C(  450.66),
                         SIMDE_FLOAT64_C( -527.30), SIMDE_FLOAT64_C( -641.78),
                         SIMDE_FLOAT64_C(  929.20), SIMDE_FLOAT64_C( -281.32),
                         SIMDE_FLOAT64_C( -125.47), SIMDE_FLOAT64_C( -963.36)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   33.04), SIMDE_FLOAT64_C(  167.46),
                         SIMDE_FLOAT64_C( -891.13), SIMDE_FLOAT64_C(  473.74),
                         SIMDE_FLOAT64_C( -547.95), SIMDE_FLOAT64_C(  516.90),
                         SIMDE_FLOAT64_C(  -69.62), SIMDE_FLOAT64_C( -976.88)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -183.17), SIMDE_FLOAT64_C(  283.20),
                         SIMDE_FLOAT64_C(  363.83), SIMDE_FLOAT64_C(-1115.52),
                         SIMDE_FLOAT64_C( 1477.15), SIMDE_FLOAT64_C( -798.22),
                         SIMDE_FLOAT64_C(  -55.85), SIMDE_FLOAT64_C(   13.52)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -659.63), SIMDE_FLOAT64_C( -759.67),
                         SIMDE_FLOAT64_C( -847.92), SIMDE_FLOAT64_C(  -61.45),
                         SIMDE_FLOAT64_C( -337.36), SIMDE_FLOAT64_C(  139.68),
                         SIMDE_FLOAT64_C(  658.69), SIMDE_FLOAT64_C(   86.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -406.77), SIMDE_FLOAT64_C( -929.18),
                         SIMDE_FLOAT64_C(  716.57), SIMDE_FLOAT64_C(    1.83),
                         SIMDE_FLOAT64_C(  179.14), SIMDE_FLOAT64_C(  145.16),
                         SIMDE_FLOAT64_C( -463.41), SIMDE_FLOAT64_C( -573.03)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -252.86), SIMDE_FLOAT64_C(  169.51),
                         SIMDE_FLOAT64_C(-1564.49), SIMDE_FLOAT64_C(  -63.28),
                         SIMDE_FLOAT64_C( -516.50), SIMDE_FLOAT64_C(   -5.48),
                         SIMDE_FLOAT64_C( 1122.10), SIMDE_FLOAT64_C(  659.58)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -210.14), SIMDE_FLOAT64_C(  816.06),
                         SIMDE_FLOAT64_C( -193.68), SIMDE_FLOAT64_C(  585.03),
                         SIMDE_FLOAT64_C( -674.08), SIMDE_FLOAT64_C(  157.57),
                         SIMDE_FLOAT64_C( -919.13), SIMDE_FLOAT64_C(  451.36)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -294.04), SIMDE_FLOAT64_C( -678.25),
                         SIMDE_FLOAT64_C(  428.59), SIMDE_FLOAT64_C( -340.21),
                         SIMDE_FLOAT64_C(  447.13), SIMDE_FLOAT64_C( -558.56),
                         SIMDE_FLOAT64_C( -584.22), SIMDE_FLOAT64_C(  801.21)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   83.90), SIMDE_FLOAT64_C( 1494.31),
                         SIMDE_FLOAT64_C( -622.27), SIMDE_FLOAT64_C(  925.24),
                         SIMDE_FLOAT64_C(-1121.21), SIMDE_FLOAT64_C(  716.13),
                         SIMDE_FLOAT64_C( -334.91), SIMDE_FLOAT64_C( -349.85)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -311.95), SIMDE_FLOAT64_C( -956.32),
                         SIMDE_FLOAT64_C(  248.48), SIMDE_FLOAT64_C(  995.45),
                         SIMDE_FLOAT64_C(  139.87), SIMDE_FLOAT64_C(  783.05),
                         SIMDE_FLOAT64_C(  584.21), SIMDE_FLOAT64_C( -920.08)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -736.64), SIMDE_FLOAT64_C(  551.91),
                         SIMDE_FLOAT64_C( -457.00), SIMDE_FLOAT64_C( -294.64),
                         SIMDE_FLOAT64_C( -589.82), SIMDE_FLOAT64_C(  788.44),
                         SIMDE_FLOAT64_C( -717.27), SIMDE_FLOAT64_C(  147.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  424.69), SIMDE_FLOAT64_C(-1508.23),
                         SIMDE_FLOAT64_C(  705.48), SIMDE_FLOAT64_C( 1290.09),
                         SIMDE_FLOAT64_C(  729.69), SIMDE_FLOAT64_C(   -5.39),
                         SIMDE_FLOAT64_C( 1301.48), SIMDE_FLOAT64_C(-1067.91)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  583.37), SIMDE_FLOAT64_C(  151.60),
                         SIMDE_FLOAT64_C( -526.34), SIMDE_FLOAT64_C( -118.48),
                         SIMDE_FLOAT64_C( -603.65), SIMDE_FLOAT64_C(  -96.99),
                         SIMDE_FLOAT64_C( -634.86), SIMDE_FLOAT64_C(  225.44)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  583.36), SIMDE_FLOAT64_C(  192.11),
                         SIMDE_FLOAT64_C(  241.22), SIMDE_FLOAT64_C( -741.26),
                         SIMDE_FLOAT64_C(  815.78), SIMDE_FLOAT64_C( -325.43),
                         SIMDE_FLOAT64_C(  457.34), SIMDE_FLOAT64_C(  430.70)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.01), SIMDE_FLOAT64_C(  -40.51),
                         SIMDE_FLOAT64_C( -767.56), SIMDE_FLOAT64_C(  622.78),
                         SIMDE_FLOAT64_C(-1419.43), SIMDE_FLOAT64_C(  228.44),
                         SIMDE_FLOAT64_C(-1092.20), SIMDE_FLOAT64_C( -205.26)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -668.75), SIMDE_FLOAT64_C( -693.34),
                         SIMDE_FLOAT64_C(   34.22), SIMDE_FLOAT64_C(  781.55),
                         SIMDE_FLOAT64_C(  732.13), SIMDE_FLOAT64_C( -735.61),
                         SIMDE_FLOAT64_C( -765.87), SIMDE_FLOAT64_C( -276.25)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    5.83), SIMDE_FLOAT64_C(  767.38),
                         SIMDE_FLOAT64_C(  251.47), SIMDE_FLOAT64_C( -790.79),
                         SIMDE_FLOAT64_C(  317.44), SIMDE_FLOAT64_C(  889.98),
                         SIMDE_FLOAT64_C(  932.08), SIMDE_FLOAT64_C(  879.75)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -674.58), SIMDE_FLOAT64_C(-1460.72),
                         SIMDE_FLOAT64_C( -217.25), SIMDE_FLOAT64_C( 1572.34),
                         SIMDE_FLOAT64_C(  414.69), SIMDE_FLOAT64_C(-1625.59),
                         SIMDE_FLOAT64_C(-1697.95), SIMDE_FLOAT64_C(-1156.00)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  317.74), SIMDE_FLOAT64_C(  -77.37),
                         SIMDE_FLOAT64_C(  975.52), SIMDE_FLOAT64_C(  188.84),
                         SIMDE_FLOAT64_C( -557.86), SIMDE_FLOAT64_C(  759.72),
                         SIMDE_FLOAT64_C( -874.99), SIMDE_FLOAT64_C(   10.90)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  693.31), SIMDE_FLOAT64_C(  248.74),
                         SIMDE_FLOAT64_C(  748.13), SIMDE_FLOAT64_C( -760.98),
                         SIMDE_FLOAT64_C(  787.06), SIMDE_FLOAT64_C(  732.48),
                         SIMDE_FLOAT64_C( -205.98), SIMDE_FLOAT64_C(  629.02)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -375.57), SIMDE_FLOAT64_C( -326.11),
                         SIMDE_FLOAT64_C(  227.39), SIMDE_FLOAT64_C(  949.82),
                         SIMDE_FLOAT64_C(-1344.92), SIMDE_FLOAT64_C(   27.24),
                         SIMDE_FLOAT64_C( -669.01), SIMDE_FLOAT64_C( -618.12)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -679.43), SIMDE_FLOAT64_C(  282.17),
                         SIMDE_FLOAT64_C(  993.32), SIMDE_FLOAT64_C(  821.29),
                         SIMDE_FLOAT64_C(  165.53), SIMDE_FLOAT64_C(  519.53),
                         SIMDE_FLOAT64_C(  873.49), SIMDE_FLOAT64_C(  728.89)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  242.15), SIMDE_FLOAT64_C(  438.99),
                         SIMDE_FLOAT64_C(  772.28), SIMDE_FLOAT64_C( -279.74),
                         SIMDE_FLOAT64_C( -310.93), SIMDE_FLOAT64_C( -848.99),
                         SIMDE_FLOAT64_C(  222.85), SIMDE_FLOAT64_C(  300.16)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -921.58), SIMDE_FLOAT64_C( -156.82),
                         SIMDE_FLOAT64_C(  221.04), SIMDE_FLOAT64_C( 1101.03),
                         SIMDE_FLOAT64_C(  476.46), SIMDE_FLOAT64_C( 1368.52),
                         SIMDE_FLOAT64_C(  650.64), SIMDE_FLOAT64_C(  428.73)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_sub_pd(test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_sub_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( -957186609), INT32_C(-1524765283), INT32_C( 1290068568), INT32_C( 1887468775),
                            INT32_C( -904096999), INT32_C(-1189693212), INT32_C(  221355870), INT32_C(-1952779315),
                            INT32_C( 1347985035), INT32_C(-2063939133), INT32_C(-1602582649), INT32_C(-2096850611),
                            INT32_C(-2084994527), INT32_C(  -75386963), INT32_C( 1835417512), INT32_C(-2072964471)),
      UINT16_C(35396),
      simde_mm512_set_epi32(INT32_C(  136551409), INT32_C( 1192962314), INT32_C( 2058621765), INT32_C(-2039270859),
                            INT32_C(  -26254502), INT32_C(  733381108), INT32_C( -187934344), INT32_C(  989979336),
                            INT32_C(-1964919382), INT32_C(  126554293), INT32_C(  254011928), INT32_C( 1490517506),
                            INT32_C(-1065486850), INT32_C(   45941921), INT32_C(-1082899768), INT32_C( -219628031)),
      simde_mm512_set_epi32(INT32_C( -680185335), INT32_C(  111102276), INT32_C( 1222454066), INT32_C( -422241261),
                            INT32_C(  -78061198), INT32_C(-2084414007), INT32_C( 1367041146), INT32_C(-1471398421),
                            INT32_C( -348147705), INT32_C(  673564238), INT32_C(-1457376577), INT32_C(  613875036),
                            INT32_C( -859069431), INT32_C( -733638834), INT32_C(-1673403701), INT32_C(  842474288)),
      simde_mm512_set_epi32(INT32_C(  816736744), INT32_C(-1524765283), INT32_C( 1290068568), INT32_C( 1887468775),
                            INT32_C(   51806696), INT32_C(-1189693212), INT32_C(-1554975490), INT32_C(-1952779315),
                            INT32_C( 1347985035), INT32_C( -547009945), INT32_C(-1602582649), INT32_C(-2096850611),
                            INT32_C(-2084994527), INT32_C(  779580755), INT32_C( 1835417512), INT32_C(-2072964471)) },
    { simde_mm512_set_epi32(INT32_C( 2077489237), INT32_C(-2066152618), INT32_C( -825076901), INT32_C(-1372574642),
                            INT32_C( -889460158), INT32_C(  498921453), INT32_C(  943332338), INT32_C(-1383811831),
                            INT32_C( 1827152592), INT32_C( 1728034912), INT32_C( 1350913629), INT32_C(  868578809),
                            INT32_C( 1368636899), INT32_C( -389235219), INT32_C(  602990700), INT32_C( -400551366)),
      UINT16_C(47779),
      simde_mm512_set_epi32(INT32_C( 1704357216), INT32_C( -538157327), INT32_C( 1370875608), INT32_C( 1508504457),
                            INT32_C(  -68294915), INT32_C(-1479685367), INT32_C( -615597542), INT32_C(-1638887359),
                            INT32_C(-1417912572), INT32_C( 1479002949), INT32_C( -647118153), INT32_C( 1670566025),
                            INT32_C(-1880268561), INT32_C(-1083232065), INT32_C( 2092339698), INT32_C(-1021873283)),
      simde_mm512_set_epi32(INT32_C( -839277498), INT32_C(  551588590), INT32_C( 1834572496), INT32_C( 1613035598),
                            INT32_C(-1678404828), INT32_C(-1769391216), INT32_C(-1638931514), INT32_C(  156804649),
                            INT32_C( 1764158657), INT32_C( -132604621), INT32_C(  446542816), INT32_C( 2037189710),
                            INT32_C(  109296986), INT32_C(  257019297), INT32_C(  473079611), INT32_C( 1127076998)),
      simde_mm512_set_epi32(INT32_C(-1751332582), INT32_C(-2066152618), INT32_C( -463696888), INT32_C( -104531141),
                            INT32_C( 1610109913), INT32_C(  498921453), INT32_C( 1023333972), INT32_C(-1383811831),
                            INT32_C( 1112896067), INT32_C( 1728034912), INT32_C(-1093660969), INT32_C(  868578809),
                            INT32_C( 1368636899), INT32_C( -389235219), INT32_C( 1619260087), INT32_C( 2146017015)) },
    { simde_mm512_set_epi32(INT32_C(  307630641), INT32_C(-1560148595), INT32_C(  376284729), INT32_C(  278591183),
                            INT32_C( -277186219), INT32_C( 1940926671), INT32_C(  662058232), INT32_C( 1091202812),
                            INT32_C( -701136301), INT32_C( -504607320), INT32_C( -251380880), INT32_C( 1860616049),
                            INT32_C(-1752161866), INT32_C(-1199997313), INT32_C(-1668691262), INT32_C( 1717921298)),
      UINT16_C( 2459),
      simde_mm512_set_epi32(INT32_C( 2079917891), INT32_C(-1199015072), INT32_C(  -98602729), INT32_C( -930567988),
                            INT32_C(-1256209763), INT32_C( 1068967165), INT32_C( 1289079409), INT32_C( 1251085533),
                            INT32_C( -727360546), INT32_C(-1724797341), INT32_C( 2093813635), INT32_C( 1051617285),
                            INT32_C( 1264716001), INT32_C(  940727836), INT32_C( 1722577424), INT32_C(-1275657732)),
      simde_mm512_set_epi32(INT32_C(  671797033), INT32_C(-1012795446), INT32_C( 2106088193), INT32_C( -458612579),
                            INT32_C( -261772865), INT32_C( -550994046), INT32_C( 2105186719), INT32_C( 1074097751),
                            INT32_C(-1251411324), INT32_C(   65867416), INT32_C(-1495248139), INT32_C(  315553116),
                            INT32_C(-1869712369), INT32_C(-1246794510), INT32_C( 1218370652), INT32_C( -240388126)),
      simde_mm512_set_epi32(INT32_C(  307630641), INT32_C(-1560148595), INT32_C(  376284729), INT32_C(  278591183),
                            INT32_C( -994436898), INT32_C( 1940926671), INT32_C(  662058232), INT32_C(  176987782),
                            INT32_C(  524050778), INT32_C( -504607320), INT32_C( -251380880), INT32_C(  736064169),
                            INT32_C(-1160538926), INT32_C(-1199997313), INT32_C(  504206772), INT32_C(-1035269606)) },
    { simde_mm512_set_epi32(INT32_C( -789716549), INT32_C(-1932674309), INT32_C(  548470804), INT32_C( -318652401),
                            INT32_C(-2041118423), INT32_C(-2107945718), INT32_C( -715661009), INT32_C( 1609073505),
                            INT32_C( 1214609500), INT32_C(  283085327), INT32_C(-1633515677), INT32_C( 1697029857),
                            INT32_C( 1976447422), INT32_C(  904412076), INT32_C( 1198927422), INT32_C(-1498026761)),
      UINT16_C(54315),
      simde_mm512_set_epi32(INT32_C( 1385182319), INT32_C(  795273310), INT32_C( 1955628796), INT32_C( -526907127),
                            INT32_C(-2141025282), INT32_C( -931446405), INT32_C(-1422139726), INT32_C(-1101084337),
                            INT32_C( -254080461), INT32_C( -595291883), INT32_C( 1292692652), INT32_C(-1849951866),
                            INT32_C( -815091127), INT32_C(  370112774), INT32_C( -520479179), INT32_C( 1681391452)),
      simde_mm512_set_epi32(INT32_C(-1825216267), INT32_C( 1555513845), INT32_C(-2081576252), INT32_C(-1972081268),
                            INT32_C( -563427058), INT32_C( 1922040193), INT32_C(-2102270715), INT32_C(-1257264155),
                            INT32_C( -894851768), INT32_C( 1793334666), INT32_C( 1049305530), INT32_C(-1935379009),
                            INT32_C(   -8279361), INT32_C(-1567490719), INT32_C(-2014130513), INT32_C(-1826154506)),
      simde_mm512_set_epi32(INT32_C(-1084568710), INT32_C( -760240535), INT32_C(  548470804), INT32_C( 1445174141),
                            INT32_C(-2041118423), INT32_C( 1441480698), INT32_C( -715661009), INT32_C( 1609073505),
                            INT32_C( 1214609500), INT32_C(  283085327), INT32_C(  243387122), INT32_C( 1697029857),
                            INT32_C( -806811766), INT32_C(  904412076), INT32_C( 1493651334), INT32_C( -787421338)) },
    { simde_mm512_set_epi32(INT32_C(  997407681), INT32_C(  -83308341), INT32_C( 1430458288), INT32_C( -655910274),
                            INT32_C(   17159218), INT32_C(  197891822), INT32_C(  -82165524), INT32_C(   98130061),
                            INT32_C( -696255503), INT32_C(  616388941), INT32_C( 1383637516), INT32_C(  255219509),
                            INT32_C(-1280964183), INT32_C(-1753221031), INT32_C(  480974923), INT32_C(-1444611560)),
      UINT16_C(47568),
      simde_mm512_set_epi32(INT32_C(-1796791424), INT32_C(  919413682), INT32_C(  907613991), INT32_C(-1471064632),
                            INT32_C(-2017464794), INT32_C(  -67778959), INT32_C(-1033884668), INT32_C( -839095279),
                            INT32_C( -881742684), INT32_C( 1193890045), INT32_C( -817450648), INT32_C( -450889209),
                            INT32_C(-1829442769), INT32_C( -254239276), INT32_C( 1531184539), INT32_C(  204100550)),
      simde_mm512_set_epi32(INT32_C(-1574624316), INT32_C( 1965632168), INT32_C( -507137262), INT32_C(  868285762),
                            INT32_C( -287712967), INT32_C(-1275855491), INT32_C(-1948986373), INT32_C(  378189270),
                            INT32_C( 2028975029), INT32_C( -983819985), INT32_C(-1530834794), INT32_C( -267906659),
                            INT32_C( 2013371063), INT32_C( -972550977), INT32_C(-1345658151), INT32_C(-2001069348)),
      simde_mm512_set_epi32(INT32_C( -222167108), INT32_C(  -83308341), INT32_C( 1414751253), INT32_C( 1955616902),
                            INT32_C(-1729751827), INT32_C(  197891822), INT32_C(  -82165524), INT32_C(-1217284549),
                            INT32_C( 1384249583), INT32_C(-2117257266), INT32_C( 1383637516), INT32_C( -182982550),
                            INT32_C(-1280964183), INT32_C(-1753221031), INT32_C(  480974923), INT32_C(-1444611560)) },
    { simde_mm512_set_epi32(INT32_C( 1875288432), INT32_C( 1158027251), INT32_C( -303056299), INT32_C( -939396673),
                            INT32_C( 1585003262), INT32_C( 1365783459), INT32_C(  111845672), INT32_C(-1286713478),
                            INT32_C(  674624782), INT32_C( 2020528740), INT32_C(  497192398), INT32_C( 1112540789),
                            INT32_C(-1764167278), INT32_C(-1540772359), INT32_C(  395629026), INT32_C(  984304916)),
      UINT16_C(16877),
      simde_mm512_set_epi32(INT32_C( -344292944), INT32_C( 1968428151), INT32_C( 2086978939), INT32_C( 1501910543),
                            INT32_C(-1262393002), INT32_C( 2081469023), INT32_C( 2016768793), INT32_C( 1922434397),
                            INT32_C( -253304624), INT32_C(  515280842), INT32_C(-1708348294), INT32_C( 2107558843),
                            INT32_C( 1919035054), INT32_C( 1742835915), INT32_C(  989439209), INT32_C( 2080310116)),
      simde_mm512_set_epi32(INT32_C( 1560352883), INT32_C( -937050525), INT32_C(   15000953), INT32_C(  298895006),
                            INT32_C( -255287325), INT32_C( -851082971), INT32_C( -981170631), INT32_C(   30364523),
                            INT32_C( -626854551), INT32_C( 1776719697), INT32_C(-1286673883), INT32_C( 2134458392),
                            INT32_C(-1884377437), INT32_C(-2042525337), INT32_C( 2143156805), INT32_C(-1045267304)),
      simde_mm512_set_epi32(INT32_C( 1875288432), INT32_C(-1389488620), INT32_C( -303056299), INT32_C( -939396673),
                            INT32_C( 1585003262), INT32_C( 1365783459), INT32_C(  111845672), INT32_C( 1892069874),
                            INT32_C(  373549927), INT32_C(-1261438855), INT32_C( -421674411), INT32_C( 1112540789),
                            INT32_C( -491554805), INT32_C( -509606044), INT32_C(  395629026), INT32_C(-1169389876)) },
    { simde_mm512_set_epi32(INT32_C(  726531409), INT32_C( -606374582), INT32_C(-1057918709), INT32_C( -811736744),
                            INT32_C(-1460245574), INT32_C( -627872087), INT32_C( 1799586442), INT32_C(-1105519928),
                            INT32_C(-1288829692), INT32_C(-2144392739), INT32_C( 1110910857), INT32_C( -282270116),
                            INT32_C(-1420141426), INT32_C( 1682561587), INT32_C( 1308021682), INT32_C(  712875579)),
      UINT16_C(17567),
      simde_mm512_set_epi32(INT32_C(-1065890522), INT32_C( 1362887862), INT32_C(-1905482051), INT32_C(  174767211),
                            INT32_C( 1968089357), INT32_C(-1207243832), INT32_C( -701927204), INT32_C(-1701909648),
                            INT32_C(-1822821880), INT32_C(-1418686446), INT32_C( 2002979046), INT32_C( -531029674),
                            INT32_C( -233545704), INT32_C( 1270923539), INT32_C( -515398077), INT32_C(  870828526)),
      simde_mm512_set_epi32(INT32_C(-1161246521), INT32_C(-1263382687), INT32_C( -761171059), INT32_C( 1052537110),
                            INT32_C(-1225204820), INT32_C( 1299827393), INT32_C(  477328169), INT32_C( 2043159101),
                            INT32_C(  984199920), INT32_C( 1963689737), INT32_C(-1149812166), INT32_C( -500241318),
                            INT32_C( -953270640), INT32_C( 1180984926), INT32_C( -645305643), INT32_C( 1026486800)),
      simde_mm512_set_epi32(INT32_C(  726531409), INT32_C(-1668696747), INT32_C(-1057918709), INT32_C( -811736744),
                            INT32_C(-1460245574), INT32_C( 1787896071), INT32_C( 1799586442), INT32_C(-1105519928),
                            INT32_C( 1487945496), INT32_C(-2144392739), INT32_C( 1110910857), INT32_C(  -30788356),
                            INT32_C(  719724936), INT32_C(   89938613), INT32_C(  129907566), INT32_C( -155658274)) },
    { simde_mm512_set_epi32(INT32_C( 1723004290), INT32_C(  721161302), INT32_C( 1077400739), INT32_C(  861837752),
                            INT32_C(-1943224858), INT32_C( 2112602876), INT32_C(-1445821889), INT32_C(-2100432693),
                            INT32_C(-1175934343), INT32_C(  805502143), INT32_C( 1163969458), INT32_C(  873642413),
                            INT32_C( 2052720739), INT32_C(-1010971457), INT32_C(  199344228), INT32_C(  251460647)),
      UINT16_C(59134),
      simde_mm512_set_epi32(INT32_C(-1391704351), INT32_C( -847303025), INT32_C(-1711491580), INT32_C( -147993971),
                            INT32_C(-1140349230), INT32_C(  172650828), INT32_C(-2090294261), INT32_C( -216506888),
                            INT32_C(-1813744120), INT32_C( 1589656338), INT32_C( 1010967585), INT32_C(-2076714127),
                            INT32_C( 1156626662), INT32_C( -264321123), INT32_C(-1099385436), INT32_C( -148901794)),
      simde_mm512_set_epi32(INT32_C( 1003282629), INT32_C( 1250297288), INT32_C(   26548422), INT32_C(-1100962758),
                            INT32_C( 1934048830), INT32_C( -886200980), INT32_C( -228926178), INT32_C(   21722717),
                            INT32_C(-1321187708), INT32_C(  904822803), INT32_C( -875700432), INT32_C(-1302414558),
                            INT32_C(  962131440), INT32_C( -729214075), INT32_C(-1094266114), INT32_C( 1122895720)),
      simde_mm512_set_epi32(INT32_C( 1899980316), INT32_C(-2097600313), INT32_C(-1738040002), INT32_C(  861837752),
                            INT32_C(-1943224858), INT32_C( 1058851808), INT32_C(-1861368083), INT32_C(-2100432693),
                            INT32_C( -492556412), INT32_C(  684833535), INT32_C( 1886668017), INT32_C( -774299569),
                            INT32_C(  194495222), INT32_C(  464892952), INT32_C(   -5119322), INT32_C(  251460647)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_sub_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_sub_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 8894478799917719473), INT64_C(-7614529333518044459),
                            INT64_C( 8458392650500739529), INT64_C( 7085639313865748967),
                            INT64_C(-7547504459018552290), INT64_C(-8310189466716392279),
                            INT64_C(-1750715323825344235), INT64_C(-2532781790488219528)),
      UINT8_C(106),
      simde_mm512_set_epi64(INT64_C(-7192427816606966254), INT64_C(-1619523557840103557),
                            INT64_C( 7616061596213068646), INT64_C( -560841280842371832),
                            INT64_C( -806373115982863580), INT64_C( -816793021936842074),
                            INT64_C( -317565234288882547), INT64_C(-7290553309909260368)),
      simde_mm512_set_epi64(INT64_C(-9084839040863053259), INT64_C(  332697972184433101),
                            INT64_C(-8959492887484217950), INT64_C( 7617292932467329680),
                            INT64_C(-2740045277871922718), INT64_C(-3634413508032825567),
                            INT64_C( -448440935066054877), INT64_C(-6805574594168851327)),
      simde_mm512_set_epi64(INT64_C( 8894478799917719473), INT64_C(-1952221530024536658),
                            INT64_C(-1871189590012265020), INT64_C( 7085639313865748967),
                            INT64_C( 1933672161889059138), INT64_C(-8310189466716392279),
                            INT64_C(  130875700777172330), INT64_C(-2532781790488219528)) },
    { simde_mm512_set_epi64(INT64_C(-3459089877760882917), INT64_C( 1753327656617706405),
                            INT64_C( 3932187030396497555), INT64_C(-4341921971190139713),
                            INT64_C(-7354864635860030437), INT64_C(-7512931671900842140),
                            INT64_C( 7677521206664265888), INT64_C(-8008068901606036732)),
      UINT8_C(  1),
      simde_mm512_set_epi64(INT64_C(-9084086707853197365), INT64_C( 5962789269656503800),
                            INT64_C( 6806616562165680967), INT64_C( 8724516399523474076),
                            INT64_C( -924171789017863248), INT64_C(-2255835938032964673),
                            INT64_C(-4560088794132063361), INT64_C(-5517329800302195238)),
      simde_mm512_set_epi64(INT64_C(-2849655299932577704), INT64_C( 2712991932590941674),
                            INT64_C( 2564329750539599066), INT64_C(-4536455326234991583),
                            INT64_C(-6477728239233614839), INT64_C(-5729565646249538826),
                            INT64_C( 3092410715614407585), INT64_C( 7984397770129184299)),
      simde_mm512_set_epi64(INT64_C(-3459089877760882917), INT64_C( 1753327656617706405),
                            INT64_C( 3932187030396497555), INT64_C(-4341921971190139713),
                            INT64_C(-7354864635860030437), INT64_C(-7512931671900842140),
                            INT64_C( 7677521206664265888), INT64_C( 4945016503278172079)) },
    { simde_mm512_set_epi64(INT64_C( -240340334077349403), INT64_C( 5647038489743797240),
                            INT64_C( 5171415873092064400), INT64_C(-1851380595205120917),
                            INT64_C( -836370148956202078), INT64_C( 8425549504970400810),
                            INT64_C( 2808549870315159479), INT64_C( 3545474415643732634)),
      UINT8_C(194),
      simde_mm512_set_epi64(INT64_C(-5877702108931305293), INT64_C(-5372639016544358566),
                            INT64_C(-4535660820549680684), INT64_C(-6747544612783901147),
                            INT64_C( 6705850594648382655), INT64_C(-1906321743942105225),
                            INT64_C( -281981608123407868), INT64_C(-5990711758326206044)),
      simde_mm512_set_epi64(INT64_C( 8110080903340414341), INT64_C(-3598578875674169061),
                            INT64_C( 4977285870543484474), INT64_C( 6776152673642620958),
                            INT64_C( 4245929756722282054), INT64_C( 3649495924615361625),
                            INT64_C( -638056186877872345), INT64_C(-8828385988165140326)),
      simde_mm512_set_epi64(INT64_C( 4458961061437831982), INT64_C(-1774060140870189505),
                            INT64_C( 5171415873092064400), INT64_C(-1851380595205120917),
                            INT64_C( -836370148956202078), INT64_C( 8425549504970400810),
                            INT64_C(  356074578754464477), INT64_C( 3545474415643732634)) },
    { simde_mm512_set_epi64(INT64_C(-6385979888474332285), INT64_C( 3716758445629922885),
                            INT64_C( 7861010731589253148), INT64_C(-6334773111204875550),
                            INT64_C(-5054960975820633825), INT64_C( 8639514840721539279),
                            INT64_C(-1027366943904624518), INT64_C(-4721195859159142702)),
      UINT8_C(222),
      simde_mm512_set_epi64(INT64_C(-7001132877809342173), INT64_C( 6512733899690414848),
                            INT64_C(  988878120815000883), INT64_C(-5994563704199492012),
                            INT64_C( 1587634372980811194), INT64_C( -914749563856678715),
                            INT64_C( 7495962388934953888), INT64_C(-7831181051188885332)),
      simde_mm512_set_epi64(INT64_C( 4229507402435677476), INT64_C( 2501842736425447642),
                            INT64_C( 8009397189160901283), INT64_C( 3833558633773719409),
                            INT64_C( 2852442819818074174), INT64_C(-8638015813272823849),
                            INT64_C( 8579593880416924807), INT64_C( 2713766728753976690)),
      simde_mm512_set_epi64(INT64_C( 7216103793464531967), INT64_C( 4010891163264967206),
                            INT64_C( 7861010731589253148), INT64_C( 8618621735736340195),
                            INT64_C(-1264808446837262980), INT64_C( 7723266249416145134),
                            INT64_C(-1083631491481970919), INT64_C(-4721195859159142702)) },
    { simde_mm512_set_epi64(INT64_C( 4051614369896270101), INT64_C( 6703896128856670897),
                            INT64_C(-5750389130785475983), INT64_C(-7878547924784098469),
                            INT64_C( 5491867996743881624), INT64_C(-2189602113514909499),
                            INT64_C( -887220462507309287), INT64_C(-5733898489940979010)),
      UINT8_C( 26),
      simde_mm512_set_epi64(INT64_C(  -99656633840764240), INT64_C(-3479731851565468885),
                            INT64_C(-7074577238264434881), INT64_C(-3836339826871533273),
                            INT64_C( 4198283975631841849), INT64_C(-3829622956767240841),
                            INT64_C( 5960966148924368684), INT64_C( -504125670847055963)),
      simde_mm512_set_epi64(INT64_C(-8344319212574510912), INT64_C(-3371415321000668561),
                            INT64_C(-8338525176508042897), INT64_C( 5173420397567361383),
                            INT64_C(-6751809518396836721), INT64_C(-8388491552134432960),
                            INT64_C(-9161028627110906680), INT64_C( 7472048750700349549)),
      simde_mm512_set_epi64(INT64_C( 4051614369896270101), INT64_C( 6703896128856670897),
                            INT64_C(-5750389130785475983), INT64_C(-9009760224438894656),
                            INT64_C(-7496650579680873046), INT64_C(-2189602113514909499),
                            INT64_C(-3324749297674276252), INT64_C(-5733898489940979010)) },
    { simde_mm512_set_epi64(INT64_C(-6378393891104748170), INT64_C(-8478287659785501826),
                            INT64_C(-2127236125072242134), INT64_C( 8702738982982040445),
                            INT64_C(  645844328650761785), INT64_C(-4561773442934600720),
                            INT64_C(-5793568656482259588), INT64_C( -379681413311801170)),
      UINT8_C(230),
      simde_mm512_set_epi64(INT64_C( -848706848545220792), INT64_C(-1124075123789220737),
                            INT64_C(-2005439629632543252), INT64_C( 8274388146286059619),
                            INT64_C( -261550962782015927), INT64_C(-8761037216848109215),
                            INT64_C(-3016365966836321630), INT64_C( 2543055264688040393)),
      simde_mm512_set_epi64(INT64_C( 1583638370136684317), INT64_C(-1184919915070849427),
                            INT64_C( 6948286910398693964), INT64_C( 2437457976149582578),
                            INT64_C( 3426542754873284897), INT64_C(-7983270512780038531),
                            INT64_C( 1779296328975282374), INT64_C(-5362999871220584978)),
      simde_mm512_set_epi64(INT64_C(-2432345218681905109), INT64_C(   60844791281628690),
                            INT64_C(-8953726540031237216), INT64_C( 8702738982982040445),
                            INT64_C(  645844328650761785), INT64_C( -777766704068070684),
                            INT64_C(-4795662295811604004), INT64_C( -379681413311801170)) },
    { simde_mm512_set_epi64(INT64_C(-2563692560784467599), INT64_C(-2764729313181954331),
                            INT64_C( 7449793955604076666), INT64_C(-6302011830015535814),
                            INT64_C(-5919077484698028869), INT64_C(-6127059769393124093),
                            INT64_C( 2958642729945465911), INT64_C( 2772140786646472311)),
      UINT8_C(198),
      simde_mm512_set_epi64(INT64_C(-3934991658845807023), INT64_C( 7561755153516237296),
                            INT64_C(-1521478373140770922), INT64_C( 6956443634033398294),
                            INT64_C(-5307063963483146371), INT64_C( 6556039892370535969),
                            INT64_C(-6645788521893978945), INT64_C(-6307512051127595595)),
      simde_mm512_set_epi64(INT64_C(-7270561721689602230), INT64_C( 8935792808270452615),
                            INT64_C( 1984489943341614372), INT64_C( 6860868624136070257),
                            INT64_C(-2243581398369652256), INT64_C(-6592818671779181804),
                            INT64_C( -308663241436655846), INT64_C(-8935526257161672911)),
      simde_mm512_set_epi64(INT64_C( 3335570062843795207), INT64_C(-1374037654754215319),
                            INT64_C( 7449793955604076666), INT64_C(-6302011830015535814),
                            INT64_C(-5919077484698028869), INT64_C(-5297885509559833843),
                            INT64_C(-6337125280457323099), INT64_C( 2772140786646472311)) },
    { simde_mm512_set_epi64(INT64_C(-7511866029206584895), INT64_C( 6685003933657692663),
                            INT64_C(  112057327023275278), INT64_C( 2785131907782223781),
                            INT64_C( -403719025987547254), INT64_C(-5974279397850363938),
                            INT64_C(-6601571580489345254), INT64_C( 1896379997419403836)),
      UINT8_C( 70),
      simde_mm512_set_epi64(INT64_C(-6334367433946281110), INT64_C(-5840485098030444461),
                            INT64_C(-6383956557021185117), INT64_C(-7600398675722821668),
                            INT64_C(-2279362749413199885), INT64_C(-8009539466982888201),
                            INT64_C(  340327559398526723), INT64_C(-2438629088141247826)),
      simde_mm512_set_epi64(INT64_C( 3758222621544461478), INT64_C( 8264387002851618510),
                            INT64_C( 5256515298231032169), INT64_C( 4555501816451377355),
                            INT64_C(-9184304616258229288), INT64_C( 5115688705834988612),
                            INT64_C(-3795492187184599084), INT64_C(-3221204559120447653)),
      simde_mm512_set_epi64(INT64_C(-7511866029206584895), INT64_C( 4341871972827488645),
                            INT64_C(  112057327023275278), INT64_C( 2785131907782223781),
                            INT64_C( -403719025987547254), INT64_C( 5321515900891674803),
                            INT64_C( 4135819746583125807), INT64_C( 1896379997419403836)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_sub_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_sub_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -417.79), SIMDE_FLOAT32_C(  -912.83), SIMDE_FLOAT32_C(   111.29), SIMDE_FLOAT32_C(  -470.87),
                         SIMDE_FLOAT32_C(   685.45), SIMDE_FLOAT32_C(   -92.85), SIMDE_FLOAT32_C(   704.55), SIMDE_FLOAT32_C(   450.79),
                         SIMDE_FLOAT32_C(  -761.01), SIMDE_FLOAT32_C(  -759.35), SIMDE_FLOAT32_C(   646.77), SIMDE_FLOAT32_C(   616.33),
                         SIMDE_FLOAT32_C(   922.76), SIMDE_FLOAT32_C(   721.94), SIMDE_FLOAT32_C(   721.78), SIMDE_FLOAT32_C(   651.66)),
      UINT16_C(55049),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   492.15), SIMDE_FLOAT32_C(   363.86), SIMDE_FLOAT32_C(  -906.93), SIMDE_FLOAT32_C(   -51.88),
                         SIMDE_FLOAT32_C(   976.36), SIMDE_FLOAT32_C(   844.84), SIMDE_FLOAT32_C(   525.57), SIMDE_FLOAT32_C(   575.43),
                         SIMDE_FLOAT32_C(  -719.61), SIMDE_FLOAT32_C(   570.91), SIMDE_FLOAT32_C(  -748.06), SIMDE_FLOAT32_C(   823.89),
                         SIMDE_FLOAT32_C(  -708.11), SIMDE_FLOAT32_C(  -805.87), SIMDE_FLOAT32_C(   626.28), SIMDE_FLOAT32_C(   344.43)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -814.48), SIMDE_FLOAT32_C(   843.19), SIMDE_FLOAT32_C(  -866.28), SIMDE_FLOAT32_C(  -230.51),
                         SIMDE_FLOAT32_C(  -264.51), SIMDE_FLOAT32_C(   935.39), SIMDE_FLOAT32_C(   479.68), SIMDE_FLOAT32_C(  -375.52),
                         SIMDE_FLOAT32_C(  -928.92), SIMDE_FLOAT32_C(  -243.75), SIMDE_FLOAT32_C(   771.60), SIMDE_FLOAT32_C(   150.31),
                         SIMDE_FLOAT32_C(  -627.83), SIMDE_FLOAT32_C(  -720.61), SIMDE_FLOAT32_C(   345.13), SIMDE_FLOAT32_C(   203.00)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  1306.63), SIMDE_FLOAT32_C(  -479.33), SIMDE_FLOAT32_C(   111.29), SIMDE_FLOAT32_C(   178.63),
                         SIMDE_FLOAT32_C(   685.45), SIMDE_FLOAT32_C(   -90.55), SIMDE_FLOAT32_C(    45.89), SIMDE_FLOAT32_C(   950.95),
                         SIMDE_FLOAT32_C(  -761.01), SIMDE_FLOAT32_C(  -759.35), SIMDE_FLOAT32_C(   646.77), SIMDE_FLOAT32_C(   616.33),
                         SIMDE_FLOAT32_C(   -80.28), SIMDE_FLOAT32_C(   721.94), SIMDE_FLOAT32_C(   721.78), SIMDE_FLOAT32_C(   141.43)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -594.79), SIMDE_FLOAT32_C(   -68.26), SIMDE_FLOAT32_C(   772.68), SIMDE_FLOAT32_C(  -615.12),
                         SIMDE_FLOAT32_C(   489.20), SIMDE_FLOAT32_C(  -609.74), SIMDE_FLOAT32_C(  -297.42), SIMDE_FLOAT32_C(  -701.58),
                         SIMDE_FLOAT32_C(    71.34), SIMDE_FLOAT32_C(  -811.20), SIMDE_FLOAT32_C(   -44.61), SIMDE_FLOAT32_C(   172.32),
                         SIMDE_FLOAT32_C(  -336.24), SIMDE_FLOAT32_C(  -959.77), SIMDE_FLOAT32_C(   896.40), SIMDE_FLOAT32_C(   321.28)),
      UINT16_C( 2266),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   136.73), SIMDE_FLOAT32_C(   408.70), SIMDE_FLOAT32_C(   907.04), SIMDE_FLOAT32_C(   175.32),
                         SIMDE_FLOAT32_C(   125.78), SIMDE_FLOAT32_C(  -176.42), SIMDE_FLOAT32_C(  -192.20), SIMDE_FLOAT32_C(   636.29),
                         SIMDE_FLOAT32_C(  -812.72), SIMDE_FLOAT32_C(  -295.02), SIMDE_FLOAT32_C(   426.00), SIMDE_FLOAT32_C(   348.29),
                         SIMDE_FLOAT32_C(   859.20), SIMDE_FLOAT32_C(   -28.95), SIMDE_FLOAT32_C(  -637.06), SIMDE_FLOAT32_C(  -450.15)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -929.87), SIMDE_FLOAT32_C(  -208.53), SIMDE_FLOAT32_C(   561.71), SIMDE_FLOAT32_C(   -74.05),
                         SIMDE_FLOAT32_C(   477.79), SIMDE_FLOAT32_C(   772.49), SIMDE_FLOAT32_C(   648.48), SIMDE_FLOAT32_C(   -58.61),
                         SIMDE_FLOAT32_C(   835.38), SIMDE_FLOAT32_C(  -689.00), SIMDE_FLOAT32_C(   607.03), SIMDE_FLOAT32_C(   421.78),
                         SIMDE_FLOAT32_C(  -574.15), SIMDE_FLOAT32_C(   302.76), SIMDE_FLOAT32_C(   178.11), SIMDE_FLOAT32_C(  -298.57)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -594.79), SIMDE_FLOAT32_C(   -68.26), SIMDE_FLOAT32_C(   772.68), SIMDE_FLOAT32_C(  -615.12),
                         SIMDE_FLOAT32_C(  -352.01), SIMDE_FLOAT32_C(  -609.74), SIMDE_FLOAT32_C(  -297.42), SIMDE_FLOAT32_C(  -701.58),
                         SIMDE_FLOAT32_C( -1648.10), SIMDE_FLOAT32_C(   393.98), SIMDE_FLOAT32_C(   -44.61), SIMDE_FLOAT32_C(   -73.49),
                         SIMDE_FLOAT32_C(  1433.35), SIMDE_FLOAT32_C(  -959.77), SIMDE_FLOAT32_C(  -815.17), SIMDE_FLOAT32_C(   321.28)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -914.76), SIMDE_FLOAT32_C(   285.68), SIMDE_FLOAT32_C(   695.03), SIMDE_FLOAT32_C(  -235.78),
                         SIMDE_FLOAT32_C(    90.17), SIMDE_FLOAT32_C(   891.02), SIMDE_FLOAT32_C(  -456.46), SIMDE_FLOAT32_C(   952.55),
                         SIMDE_FLOAT32_C(  -153.33), SIMDE_FLOAT32_C(  -533.35), SIMDE_FLOAT32_C(  -130.02), SIMDE_FLOAT32_C(  -580.21),
                         SIMDE_FLOAT32_C(  -857.73), SIMDE_FLOAT32_C(  -362.64), SIMDE_FLOAT32_C(   808.25), SIMDE_FLOAT32_C(   908.95)),
      UINT16_C(53407),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   415.38), SIMDE_FLOAT32_C(   622.33), SIMDE_FLOAT32_C(   849.49), SIMDE_FLOAT32_C(  -552.97),
                         SIMDE_FLOAT32_C(   837.01), SIMDE_FLOAT32_C(  -753.98), SIMDE_FLOAT32_C(   167.51), SIMDE_FLOAT32_C(   898.60),
                         SIMDE_FLOAT32_C(   -36.68), SIMDE_FLOAT32_C(  -931.19), SIMDE_FLOAT32_C(   230.22), SIMDE_FLOAT32_C(  -885.80),
                         SIMDE_FLOAT32_C(  -894.49), SIMDE_FLOAT32_C(  -402.23), SIMDE_FLOAT32_C(   -68.60), SIMDE_FLOAT32_C(  -153.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   247.18), SIMDE_FLOAT32_C(   507.40), SIMDE_FLOAT32_C(  -715.17), SIMDE_FLOAT32_C(   785.48),
                         SIMDE_FLOAT32_C(  -543.41), SIMDE_FLOAT32_C(   761.08), SIMDE_FLOAT32_C(   479.07), SIMDE_FLOAT32_C(  -938.93),
                         SIMDE_FLOAT32_C(  -655.56), SIMDE_FLOAT32_C(   618.55), SIMDE_FLOAT32_C(   224.83), SIMDE_FLOAT32_C(  -983.99),
                         SIMDE_FLOAT32_C(   -18.22), SIMDE_FLOAT32_C(  -142.62), SIMDE_FLOAT32_C(   120.01), SIMDE_FLOAT32_C(   186.92)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   168.20), SIMDE_FLOAT32_C(   114.93), SIMDE_FLOAT32_C(   695.03), SIMDE_FLOAT32_C( -1338.45),
                         SIMDE_FLOAT32_C(    90.17), SIMDE_FLOAT32_C(   891.02), SIMDE_FLOAT32_C(  -456.46), SIMDE_FLOAT32_C(   952.55),
                         SIMDE_FLOAT32_C(   618.88), SIMDE_FLOAT32_C(  -533.35), SIMDE_FLOAT32_C(  -130.02), SIMDE_FLOAT32_C(    98.19),
                         SIMDE_FLOAT32_C(  -876.27), SIMDE_FLOAT32_C(  -259.61), SIMDE_FLOAT32_C(  -188.61), SIMDE_FLOAT32_C(  -340.80)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -586.97), SIMDE_FLOAT32_C(  -706.71), SIMDE_FLOAT32_C(   862.31), SIMDE_FLOAT32_C(   901.76),
                         SIMDE_FLOAT32_C(  -777.23), SIMDE_FLOAT32_C(  -615.23), SIMDE_FLOAT32_C(   540.06), SIMDE_FLOAT32_C(  -837.05),
                         SIMDE_FLOAT32_C(   896.68), SIMDE_FLOAT32_C(  -818.79), SIMDE_FLOAT32_C(  -146.21), SIMDE_FLOAT32_C(  -751.20),
                         SIMDE_FLOAT32_C(  -724.86), SIMDE_FLOAT32_C(  -446.10), SIMDE_FLOAT32_C(   747.21), SIMDE_FLOAT32_C(  -830.22)),
      UINT16_C(24145),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   809.72), SIMDE_FLOAT32_C(  -191.45), SIMDE_FLOAT32_C(  -687.88), SIMDE_FLOAT32_C(  -561.69),
                         SIMDE_FLOAT32_C(   623.06), SIMDE_FLOAT32_C(  -685.16), SIMDE_FLOAT32_C(   155.59), SIMDE_FLOAT32_C(   -91.67),
                         SIMDE_FLOAT32_C(  -292.32), SIMDE_FLOAT32_C(   436.29), SIMDE_FLOAT32_C(   682.53), SIMDE_FLOAT32_C(  -427.71),
                         SIMDE_FLOAT32_C(  -252.26), SIMDE_FLOAT32_C(  -814.33), SIMDE_FLOAT32_C(  -116.78), SIMDE_FLOAT32_C(  -176.18)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -476.63), SIMDE_FLOAT32_C(  -403.49), SIMDE_FLOAT32_C(  -129.06), SIMDE_FLOAT32_C(  -540.32),
                         SIMDE_FLOAT32_C(  -296.84), SIMDE_FLOAT32_C(   354.93), SIMDE_FLOAT32_C(   301.70), SIMDE_FLOAT32_C(   818.26),
                         SIMDE_FLOAT32_C(   152.41), SIMDE_FLOAT32_C(    -7.33), SIMDE_FLOAT32_C(   901.12), SIMDE_FLOAT32_C(   276.49),
                         SIMDE_FLOAT32_C(  -421.45), SIMDE_FLOAT32_C(   -19.17), SIMDE_FLOAT32_C(   559.47), SIMDE_FLOAT32_C(   -62.60)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -586.97), SIMDE_FLOAT32_C(   212.04), SIMDE_FLOAT32_C(   862.31), SIMDE_FLOAT32_C(   -21.37),
                         SIMDE_FLOAT32_C(   919.90), SIMDE_FLOAT32_C( -1040.09), SIMDE_FLOAT32_C(  -146.11), SIMDE_FLOAT32_C(  -837.05),
                         SIMDE_FLOAT32_C(   896.68), SIMDE_FLOAT32_C(   443.62), SIMDE_FLOAT32_C(  -146.21), SIMDE_FLOAT32_C(  -704.20),
                         SIMDE_FLOAT32_C(  -724.86), SIMDE_FLOAT32_C(  -446.10), SIMDE_FLOAT32_C(   747.21), SIMDE_FLOAT32_C(  -113.58)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   853.44), SIMDE_FLOAT32_C(   804.93), SIMDE_FLOAT32_C(   753.54), SIMDE_FLOAT32_C(   129.42),
                         SIMDE_FLOAT32_C(  -911.24), SIMDE_FLOAT32_C(  -795.01), SIMDE_FLOAT32_C(  -264.21), SIMDE_FLOAT32_C(   110.23),
                         SIMDE_FLOAT32_C(   779.42), SIMDE_FLOAT32_C(   756.19), SIMDE_FLOAT32_C(   -61.94), SIMDE_FLOAT32_C(  -845.71),
                         SIMDE_FLOAT32_C(   522.75), SIMDE_FLOAT32_C(   703.06), SIMDE_FLOAT32_C(   989.80), SIMDE_FLOAT32_C(   594.14)),
      UINT16_C(58122),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   774.43), SIMDE_FLOAT32_C(   251.56), SIMDE_FLOAT32_C(  -915.66), SIMDE_FLOAT32_C(  -492.31),
                         SIMDE_FLOAT32_C(   722.32), SIMDE_FLOAT32_C(   853.19), SIMDE_FLOAT32_C(   466.28), SIMDE_FLOAT32_C(   573.97),
                         SIMDE_FLOAT32_C(  -516.73), SIMDE_FLOAT32_C(  -267.27), SIMDE_FLOAT32_C(   110.95), SIMDE_FLOAT32_C(   -68.16),
                         SIMDE_FLOAT32_C(  -400.30), SIMDE_FLOAT32_C(   327.53), SIMDE_FLOAT32_C(  -638.51), SIMDE_FLOAT32_C(   -96.92)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   101.96), SIMDE_FLOAT32_C(  -734.61), SIMDE_FLOAT32_C(   219.43), SIMDE_FLOAT32_C(  -507.66),
                         SIMDE_FLOAT32_C(  -747.54), SIMDE_FLOAT32_C(   794.68), SIMDE_FLOAT32_C(  -663.99), SIMDE_FLOAT32_C(  -123.94),
                         SIMDE_FLOAT32_C(  -793.12), SIMDE_FLOAT32_C(   673.57), SIMDE_FLOAT32_C(  -777.14), SIMDE_FLOAT32_C(   175.88),
                         SIMDE_FLOAT32_C(  -792.24), SIMDE_FLOAT32_C(  -246.51), SIMDE_FLOAT32_C(   848.21), SIMDE_FLOAT32_C(  -124.15)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   672.47), SIMDE_FLOAT32_C(   986.17), SIMDE_FLOAT32_C( -1135.09), SIMDE_FLOAT32_C(   129.42),
                         SIMDE_FLOAT32_C(  -911.24), SIMDE_FLOAT32_C(  -795.01), SIMDE_FLOAT32_C(  1130.27), SIMDE_FLOAT32_C(   697.91),
                         SIMDE_FLOAT32_C(   779.42), SIMDE_FLOAT32_C(   756.19), SIMDE_FLOAT32_C(   -61.94), SIMDE_FLOAT32_C(  -845.71),
                         SIMDE_FLOAT32_C(   391.94), SIMDE_FLOAT32_C(   703.06), SIMDE_FLOAT32_C( -1486.72), SIMDE_FLOAT32_C(   594.14)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -670.29), SIMDE_FLOAT32_C(   821.01), SIMDE_FLOAT32_C(  -293.06), SIMDE_FLOAT32_C(   -56.42),
                         SIMDE_FLOAT32_C(  -163.64), SIMDE_FLOAT32_C(  -919.47), SIMDE_FLOAT32_C(   636.75), SIMDE_FLOAT32_C(   555.64),
                         SIMDE_FLOAT32_C(   630.28), SIMDE_FLOAT32_C(   798.33), SIMDE_FLOAT32_C(  -536.88), SIMDE_FLOAT32_C(   256.29),
                         SIMDE_FLOAT32_C(   834.99), SIMDE_FLOAT32_C(  -678.50), SIMDE_FLOAT32_C(  -716.28), SIMDE_FLOAT32_C(  -235.17)),
      UINT16_C( 7968),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   575.18), SIMDE_FLOAT32_C(  -655.63), SIMDE_FLOAT32_C(   986.91), SIMDE_FLOAT32_C(   710.96),
                         SIMDE_FLOAT32_C(   921.30), SIMDE_FLOAT32_C(   -96.00), SIMDE_FLOAT32_C(   -68.75), SIMDE_FLOAT32_C(  -119.17),
                         SIMDE_FLOAT32_C(  -795.52), SIMDE_FLOAT32_C(  -851.06), SIMDE_FLOAT32_C(   982.58), SIMDE_FLOAT32_C(   432.45),
                         SIMDE_FLOAT32_C(   834.71), SIMDE_FLOAT32_C(  -931.48), SIMDE_FLOAT32_C(   421.86), SIMDE_FLOAT32_C(   549.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   699.42), SIMDE_FLOAT32_C(  -430.21), SIMDE_FLOAT32_C(  -842.83), SIMDE_FLOAT32_C(  -375.32),
                         SIMDE_FLOAT32_C(  -889.13), SIMDE_FLOAT32_C(    77.46), SIMDE_FLOAT32_C(  -426.32), SIMDE_FLOAT32_C(  -319.52),
                         SIMDE_FLOAT32_C(   633.46), SIMDE_FLOAT32_C(  -484.05), SIMDE_FLOAT32_C(   991.09), SIMDE_FLOAT32_C(   894.84),
                         SIMDE_FLOAT32_C(   148.17), SIMDE_FLOAT32_C(  -167.11), SIMDE_FLOAT32_C(  -811.87), SIMDE_FLOAT32_C(  -574.29)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -670.29), SIMDE_FLOAT32_C(   821.01), SIMDE_FLOAT32_C(  -293.06), SIMDE_FLOAT32_C(  1086.28),
                         SIMDE_FLOAT32_C(  1810.43), SIMDE_FLOAT32_C(  -173.46), SIMDE_FLOAT32_C(   357.57), SIMDE_FLOAT32_C(   200.35),
                         SIMDE_FLOAT32_C(   630.28), SIMDE_FLOAT32_C(   798.33), SIMDE_FLOAT32_C(    -8.51), SIMDE_FLOAT32_C(   256.29),
                         SIMDE_FLOAT32_C(   834.99), SIMDE_FLOAT32_C(  -678.50), SIMDE_FLOAT32_C(  -716.28), SIMDE_FLOAT32_C(  -235.17)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   640.00), SIMDE_FLOAT32_C(   440.55), SIMDE_FLOAT32_C(   793.44), SIMDE_FLOAT32_C(   554.05),
                         SIMDE_FLOAT32_C(   245.74), SIMDE_FLOAT32_C(  -388.16), SIMDE_FLOAT32_C(   -27.32), SIMDE_FLOAT32_C(  -923.44),
                         SIMDE_FLOAT32_C(   109.81), SIMDE_FLOAT32_C(   855.67), SIMDE_FLOAT32_C(  -513.53), SIMDE_FLOAT32_C(  -921.47),
                         SIMDE_FLOAT32_C(  -410.90), SIMDE_FLOAT32_C(  -404.15), SIMDE_FLOAT32_C(  -502.43), SIMDE_FLOAT32_C(  -674.13)),
      UINT16_C(34235),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   455.94), SIMDE_FLOAT32_C(   822.75), SIMDE_FLOAT32_C(   672.52), SIMDE_FLOAT32_C(   418.16),
                         SIMDE_FLOAT32_C(   993.17), SIMDE_FLOAT32_C(  -581.12), SIMDE_FLOAT32_C(   737.02), SIMDE_FLOAT32_C(   -48.12),
                         SIMDE_FLOAT32_C(   169.53), SIMDE_FLOAT32_C(   875.02), SIMDE_FLOAT32_C(   325.94), SIMDE_FLOAT32_C(  -197.05),
                         SIMDE_FLOAT32_C(   209.80), SIMDE_FLOAT32_C(   679.16), SIMDE_FLOAT32_C(  -743.34), SIMDE_FLOAT32_C(   192.93)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -400.32), SIMDE_FLOAT32_C(   747.89), SIMDE_FLOAT32_C(  -417.14), SIMDE_FLOAT32_C(  -149.76),
                         SIMDE_FLOAT32_C(  -769.13), SIMDE_FLOAT32_C(   952.70), SIMDE_FLOAT32_C(    55.59), SIMDE_FLOAT32_C(  -118.59),
                         SIMDE_FLOAT32_C(  -651.36), SIMDE_FLOAT32_C(   213.50), SIMDE_FLOAT32_C(   998.39), SIMDE_FLOAT32_C(   155.85),
                         SIMDE_FLOAT32_C(   985.22), SIMDE_FLOAT32_C(  -399.37), SIMDE_FLOAT32_C(  -660.54), SIMDE_FLOAT32_C(  -918.87)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   856.26), SIMDE_FLOAT32_C(   440.55), SIMDE_FLOAT32_C(   793.44), SIMDE_FLOAT32_C(   554.05),
                         SIMDE_FLOAT32_C(   245.74), SIMDE_FLOAT32_C( -1533.82), SIMDE_FLOAT32_C(   -27.32), SIMDE_FLOAT32_C(    70.47),
                         SIMDE_FLOAT32_C(   820.89), SIMDE_FLOAT32_C(   855.67), SIMDE_FLOAT32_C(  -672.45), SIMDE_FLOAT32_C(  -352.90),
                         SIMDE_FLOAT32_C(  -775.42), SIMDE_FLOAT32_C(  -404.15), SIMDE_FLOAT32_C(   -82.80), SIMDE_FLOAT32_C(  1111.80)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -717.43), SIMDE_FLOAT32_C(   307.65), SIMDE_FLOAT32_C(  -776.64), SIMDE_FLOAT32_C(   883.24),
                         SIMDE_FLOAT32_C(   462.38), SIMDE_FLOAT32_C(   941.52), SIMDE_FLOAT32_C(   465.21), SIMDE_FLOAT32_C(   772.92),
                         SIMDE_FLOAT32_C(  -448.96), SIMDE_FLOAT32_C(   167.95), SIMDE_FLOAT32_C(  -770.79), SIMDE_FLOAT32_C(   607.02),
                         SIMDE_FLOAT32_C(   588.25), SIMDE_FLOAT32_C(  -430.65), SIMDE_FLOAT32_C(  -379.22), SIMDE_FLOAT32_C(    62.66)),
      UINT16_C(21184),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   745.31), SIMDE_FLOAT32_C(   528.47), SIMDE_FLOAT32_C(   260.56), SIMDE_FLOAT32_C(   756.92),
                         SIMDE_FLOAT32_C(  -237.78), SIMDE_FLOAT32_C(   890.33), SIMDE_FLOAT32_C(  -276.66), SIMDE_FLOAT32_C(  -845.25),
                         SIMDE_FLOAT32_C(    73.01), SIMDE_FLOAT32_C(  -169.10), SIMDE_FLOAT32_C(  -390.26), SIMDE_FLOAT32_C(    55.87),
                         SIMDE_FLOAT32_C(   461.32), SIMDE_FLOAT32_C(  -911.03), SIMDE_FLOAT32_C(   362.01), SIMDE_FLOAT32_C(   998.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   177.96), SIMDE_FLOAT32_C(  -105.40), SIMDE_FLOAT32_C(  -516.55), SIMDE_FLOAT32_C(   -62.31),
                         SIMDE_FLOAT32_C(  -757.68), SIMDE_FLOAT32_C(   665.34), SIMDE_FLOAT32_C(   689.63), SIMDE_FLOAT32_C(   938.32),
                         SIMDE_FLOAT32_C(  -408.00), SIMDE_FLOAT32_C(   998.26), SIMDE_FLOAT32_C(  -263.70), SIMDE_FLOAT32_C(   807.54),
                         SIMDE_FLOAT32_C(   485.72), SIMDE_FLOAT32_C(   -74.68), SIMDE_FLOAT32_C(   725.36), SIMDE_FLOAT32_C(   301.00)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -717.43), SIMDE_FLOAT32_C(   633.87), SIMDE_FLOAT32_C(  -776.64), SIMDE_FLOAT32_C(   819.23),
                         SIMDE_FLOAT32_C(   462.38), SIMDE_FLOAT32_C(   941.52), SIMDE_FLOAT32_C(  -966.29), SIMDE_FLOAT32_C(   772.92),
                         SIMDE_FLOAT32_C(   481.01), SIMDE_FLOAT32_C( -1167.36), SIMDE_FLOAT32_C(  -770.79), SIMDE_FLOAT32_C(   607.02),
                         SIMDE_FLOAT32_C(   588.25), SIMDE_FLOAT32_C(  -430.65), SIMDE_FLOAT32_C(  -379.22), SIMDE_FLOAT32_C(    62.66)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_sub_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_sub_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -621.09), SIMDE_FLOAT64_C(  350.18),
                         SIMDE_FLOAT64_C(  873.40), SIMDE_FLOAT64_C( -136.67),
                         SIMDE_FLOAT64_C( -484.90), SIMDE_FLOAT64_C(  672.37),
                         SIMDE_FLOAT64_C( -983.97), SIMDE_FLOAT64_C( -747.18)),
      UINT8_C(213),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -615.22), SIMDE_FLOAT64_C(  861.93),
                         SIMDE_FLOAT64_C(  -99.63), SIMDE_FLOAT64_C( -760.72),
                         SIMDE_FLOAT64_C(  803.54), SIMDE_FLOAT64_C( -811.65),
                         SIMDE_FLOAT64_C( -888.48), SIMDE_FLOAT64_C(  353.19)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  451.54), SIMDE_FLOAT64_C(  490.96),
                         SIMDE_FLOAT64_C( -563.07), SIMDE_FLOAT64_C( -968.95),
                         SIMDE_FLOAT64_C( -964.80), SIMDE_FLOAT64_C( -259.48),
                         SIMDE_FLOAT64_C(  -97.31), SIMDE_FLOAT64_C(  696.26)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(-1066.76), SIMDE_FLOAT64_C(  370.97),
                         SIMDE_FLOAT64_C(  873.40), SIMDE_FLOAT64_C(  208.23),
                         SIMDE_FLOAT64_C( -484.90), SIMDE_FLOAT64_C( -552.17),
                         SIMDE_FLOAT64_C( -983.97), SIMDE_FLOAT64_C( -343.07)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  956.74), SIMDE_FLOAT64_C(  507.70),
                         SIMDE_FLOAT64_C(  525.25), SIMDE_FLOAT64_C( -653.24),
                         SIMDE_FLOAT64_C( -748.66), SIMDE_FLOAT64_C(  738.72),
                         SIMDE_FLOAT64_C(  584.29), SIMDE_FLOAT64_C( -344.89)),
      UINT8_C(200),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -70.99), SIMDE_FLOAT64_C( -712.48),
                         SIMDE_FLOAT64_C(  721.37), SIMDE_FLOAT64_C(  290.11),
                         SIMDE_FLOAT64_C(  739.65), SIMDE_FLOAT64_C(  378.13),
                         SIMDE_FLOAT64_C(  523.23), SIMDE_FLOAT64_C(  338.41)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -243.21), SIMDE_FLOAT64_C(   71.87),
                         SIMDE_FLOAT64_C(   81.06), SIMDE_FLOAT64_C(  409.05),
                         SIMDE_FLOAT64_C( -595.58), SIMDE_FLOAT64_C(  278.33),
                         SIMDE_FLOAT64_C( -484.02), SIMDE_FLOAT64_C( -861.59)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  172.22), SIMDE_FLOAT64_C( -784.35),
                         SIMDE_FLOAT64_C(  525.25), SIMDE_FLOAT64_C( -653.24),
                         SIMDE_FLOAT64_C( 1335.23), SIMDE_FLOAT64_C(  738.72),
                         SIMDE_FLOAT64_C(  584.29), SIMDE_FLOAT64_C( -344.89)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  475.39), SIMDE_FLOAT64_C(  345.93),
                         SIMDE_FLOAT64_C(  233.76), SIMDE_FLOAT64_C( -401.11),
                         SIMDE_FLOAT64_C( -964.57), SIMDE_FLOAT64_C(  939.13),
                         SIMDE_FLOAT64_C( -392.63), SIMDE_FLOAT64_C( -585.02)),
      UINT8_C( 75),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  496.11), SIMDE_FLOAT64_C( -235.94),
                         SIMDE_FLOAT64_C( -715.35), SIMDE_FLOAT64_C(  338.71),
                         SIMDE_FLOAT64_C( -776.11), SIMDE_FLOAT64_C(  941.96),
                         SIMDE_FLOAT64_C(   76.10), SIMDE_FLOAT64_C( -188.31)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  824.70), SIMDE_FLOAT64_C( -886.45),
                         SIMDE_FLOAT64_C(  497.17), SIMDE_FLOAT64_C( -965.13),
                         SIMDE_FLOAT64_C( -601.99), SIMDE_FLOAT64_C( -657.07),
                         SIMDE_FLOAT64_C(  201.36), SIMDE_FLOAT64_C( -807.98)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  475.39), SIMDE_FLOAT64_C(  650.51),
                         SIMDE_FLOAT64_C(  233.76), SIMDE_FLOAT64_C( -401.11),
                         SIMDE_FLOAT64_C( -174.12), SIMDE_FLOAT64_C(  939.13),
                         SIMDE_FLOAT64_C( -125.26), SIMDE_FLOAT64_C(  619.67)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -246.72), SIMDE_FLOAT64_C( -493.17),
                         SIMDE_FLOAT64_C( -501.93), SIMDE_FLOAT64_C(  -95.50),
                         SIMDE_FLOAT64_C(  754.55), SIMDE_FLOAT64_C( -990.48),
                         SIMDE_FLOAT64_C( -396.36), SIMDE_FLOAT64_C( -466.97)),
      UINT8_C( 69),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  601.28), SIMDE_FLOAT64_C( -873.85),
                         SIMDE_FLOAT64_C( -689.96), SIMDE_FLOAT64_C(   31.77),
                         SIMDE_FLOAT64_C(  -97.11), SIMDE_FLOAT64_C(  971.94),
                         SIMDE_FLOAT64_C(  389.02), SIMDE_FLOAT64_C( -650.79)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  136.61), SIMDE_FLOAT64_C(  436.94),
                         SIMDE_FLOAT64_C( -777.02), SIMDE_FLOAT64_C(  166.29),
                         SIMDE_FLOAT64_C( -377.75), SIMDE_FLOAT64_C(   71.16),
                         SIMDE_FLOAT64_C(  481.01), SIMDE_FLOAT64_C( -926.81)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -246.72), SIMDE_FLOAT64_C(-1310.79),
                         SIMDE_FLOAT64_C( -501.93), SIMDE_FLOAT64_C(  -95.50),
                         SIMDE_FLOAT64_C(  754.55), SIMDE_FLOAT64_C(  900.78),
                         SIMDE_FLOAT64_C( -396.36), SIMDE_FLOAT64_C(  276.02)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -389.46), SIMDE_FLOAT64_C(   -8.03),
                         SIMDE_FLOAT64_C( -523.51), SIMDE_FLOAT64_C(  466.89),
                         SIMDE_FLOAT64_C(  698.90), SIMDE_FLOAT64_C( -346.04),
                         SIMDE_FLOAT64_C( -734.67), SIMDE_FLOAT64_C(  404.34)),
      UINT8_C(100),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  989.13), SIMDE_FLOAT64_C(  228.14),
                         SIMDE_FLOAT64_C(  840.94), SIMDE_FLOAT64_C( -718.83),
                         SIMDE_FLOAT64_C(  274.95), SIMDE_FLOAT64_C(  -99.21),
                         SIMDE_FLOAT64_C(   84.76), SIMDE_FLOAT64_C( -295.84)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -956.22), SIMDE_FLOAT64_C(  564.94),
                         SIMDE_FLOAT64_C(  -97.16), SIMDE_FLOAT64_C( -407.99),
                         SIMDE_FLOAT64_C(  352.62), SIMDE_FLOAT64_C(  244.25),
                         SIMDE_FLOAT64_C(   43.92), SIMDE_FLOAT64_C(  624.69)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -389.46), SIMDE_FLOAT64_C( -336.80),
                         SIMDE_FLOAT64_C(  938.10), SIMDE_FLOAT64_C(  466.89),
                         SIMDE_FLOAT64_C(  698.90), SIMDE_FLOAT64_C( -343.46),
                         SIMDE_FLOAT64_C( -734.67), SIMDE_FLOAT64_C(  404.34)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -571.96), SIMDE_FLOAT64_C(   40.27),
                         SIMDE_FLOAT64_C(  676.69), SIMDE_FLOAT64_C( -150.37),
                         SIMDE_FLOAT64_C(  945.34), SIMDE_FLOAT64_C(   75.83),
                         SIMDE_FLOAT64_C(   64.75), SIMDE_FLOAT64_C(  239.06)),
      UINT8_C(209),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  792.47), SIMDE_FLOAT64_C( -265.19),
                         SIMDE_FLOAT64_C( -768.95), SIMDE_FLOAT64_C(  515.15),
                         SIMDE_FLOAT64_C(  350.59), SIMDE_FLOAT64_C(  422.68),
                         SIMDE_FLOAT64_C(  582.99), SIMDE_FLOAT64_C( -985.50)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   12.38), SIMDE_FLOAT64_C(  -71.80),
                         SIMDE_FLOAT64_C(  363.01), SIMDE_FLOAT64_C( -195.65),
                         SIMDE_FLOAT64_C(  967.47), SIMDE_FLOAT64_C(   -4.13),
                         SIMDE_FLOAT64_C( -478.81), SIMDE_FLOAT64_C(  909.10)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  780.09), SIMDE_FLOAT64_C( -193.39),
                         SIMDE_FLOAT64_C(  676.69), SIMDE_FLOAT64_C(  710.80),
                         SIMDE_FLOAT64_C(  945.34), SIMDE_FLOAT64_C(   75.83),
                         SIMDE_FLOAT64_C(   64.75), SIMDE_FLOAT64_C(-1894.60)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -879.88), SIMDE_FLOAT64_C( -687.95),
                         SIMDE_FLOAT64_C( -892.89), SIMDE_FLOAT64_C( -642.85),
                         SIMDE_FLOAT64_C(  533.08), SIMDE_FLOAT64_C(  898.29),
                         SIMDE_FLOAT64_C(  -29.99), SIMDE_FLOAT64_C(    5.58)),
      UINT8_C(186),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  649.80), SIMDE_FLOAT64_C( -257.91),
                         SIMDE_FLOAT64_C(  356.56), SIMDE_FLOAT64_C(  567.70),
                         SIMDE_FLOAT64_C(  -80.43), SIMDE_FLOAT64_C( -499.15),
                         SIMDE_FLOAT64_C( -866.12), SIMDE_FLOAT64_C(  639.40)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  702.45), SIMDE_FLOAT64_C(  464.79),
                         SIMDE_FLOAT64_C(  387.80), SIMDE_FLOAT64_C( -528.10),
                         SIMDE_FLOAT64_C( -409.82), SIMDE_FLOAT64_C( -696.40),
                         SIMDE_FLOAT64_C(  455.43), SIMDE_FLOAT64_C(  856.81)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -52.65), SIMDE_FLOAT64_C( -687.95),
                         SIMDE_FLOAT64_C(  -31.24), SIMDE_FLOAT64_C( 1095.80),
                         SIMDE_FLOAT64_C(  329.39), SIMDE_FLOAT64_C(  898.29),
                         SIMDE_FLOAT64_C(-1321.55), SIMDE_FLOAT64_C(    5.58)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -750.95), SIMDE_FLOAT64_C(  203.46),
                         SIMDE_FLOAT64_C(  194.87), SIMDE_FLOAT64_C(  667.81),
                         SIMDE_FLOAT64_C( -258.76), SIMDE_FLOAT64_C(  897.89),
                         SIMDE_FLOAT64_C(  571.10), SIMDE_FLOAT64_C( -320.96)),
      UINT8_C( 56),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -938.69), SIMDE_FLOAT64_C(   74.05),
                         SIMDE_FLOAT64_C( -981.48), SIMDE_FLOAT64_C( -656.78),
                         SIMDE_FLOAT64_C( -794.37), SIMDE_FLOAT64_C(  177.36),
                         SIMDE_FLOAT64_C(  380.50), SIMDE_FLOAT64_C(  812.91)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -10.37), SIMDE_FLOAT64_C( -894.99),
                         SIMDE_FLOAT64_C( -148.09), SIMDE_FLOAT64_C(  314.75),
                         SIMDE_FLOAT64_C( -740.28), SIMDE_FLOAT64_C( -372.00),
                         SIMDE_FLOAT64_C( -357.36), SIMDE_FLOAT64_C( -791.79)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -750.95), SIMDE_FLOAT64_C(  203.46),
                         SIMDE_FLOAT64_C( -833.39), SIMDE_FLOAT64_C( -971.53),
                         SIMDE_FLOAT64_C(  -54.09), SIMDE_FLOAT64_C(  897.89),
                         SIMDE_FLOAT64_C(  571.10), SIMDE_FLOAT64_C( -320.96)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_sub_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_sub_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(42308),
      simde_mm512_set_epi32(INT32_C( 1724059665), INT32_C(-1181331137), INT32_C( -956878955), INT32_C( 1254662027),
                            INT32_C( -334196329), INT32_C( -462422656), INT32_C(  391895544), INT32_C( 1081692585),
                            INT32_C(-1420053828), INT32_C(-1016697350), INT32_C( 1995028549), INT32_C(-2003231670),
                            INT32_C( 1672190791), INT32_C(  255109958), INT32_C(-2019884289), INT32_C(-1398510440)),
      simde_mm512_set_epi32(INT32_C( 1758500044), INT32_C(  727344602), INT32_C(-1303831643), INT32_C( 1021495274),
                            INT32_C(-2113209677), INT32_C( 1628670789), INT32_C(  684532718), INT32_C( 1920084108),
                            INT32_C( -516238646), INT32_C( 1525557846), INT32_C( 1058541430), INT32_C(  232836803),
                            INT32_C( 1824295576), INT32_C(-1334166784), INT32_C(-1267999587), INT32_C( 1992895333)),
      simde_mm512_set_epi32(INT32_C(  -34440379), INT32_C(          0), INT32_C(  346952688), INT32_C(          0),
                            INT32_C(          0), INT32_C(-2091093445), INT32_C(          0), INT32_C( -838391523),
                            INT32_C(          0), INT32_C( 1752712100), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1589276742), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(57112),
      simde_mm512_set_epi32(INT32_C( 1427084198), INT32_C(  800677318), INT32_C(-1624555826), INT32_C( -189169736),
                            INT32_C( -500462405), INT32_C(  393027187), INT32_C( -215642095), INT32_C( 1795082661),
                            INT32_C(-1120274966), INT32_C( 1416315501), INT32_C( 2071781830), INT32_C( 1981287236),
                            INT32_C( 1895228887), INT32_C( -102536112), INT32_C(-1592734830), INT32_C(-1858725491)),
      simde_mm512_set_epi32(INT32_C( -450919787), INT32_C( 1299130560), INT32_C( 1762509692), INT32_C(  310818231),
                            INT32_C( -225659966), INT32_C(-1193662266), INT32_C(  959080993), INT32_C(  -80526553),
                            INT32_C( -695376176), INT32_C(  -26080833), INT32_C(  542712435), INT32_C( 1266358760),
                            INT32_C(  181254235), INT32_C(-2068678559), INT32_C( 1863289430), INT32_C( -269529302)),
      simde_mm512_set_epi32(INT32_C( 1878003985), INT32_C( -498453242), INT32_C(          0), INT32_C( -499987967),
                            INT32_C( -274802439), INT32_C( 1586689453), INT32_C(-1174723088), INT32_C( 1875609214),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(  714928476),
                            INT32_C( 1713974652), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(45985),
      simde_mm512_set_epi32(INT32_C(-1997599226), INT32_C( 1542236612), INT32_C(  969579913), INT32_C(-1642088433),
                            INT32_C(  579114801), INT32_C(-1194258935), INT32_C(-1422143462), INT32_C( 1748279001),
                            INT32_C(-1953627340), INT32_C( 1674288033), INT32_C(  717963559), INT32_C(   34905906),
                            INT32_C( -149768860), INT32_C( 1400155142), INT32_C( 1757125654), INT32_C(-1787496119)),
      simde_mm512_set_epi32(INT32_C(   11674598), INT32_C( 1849959427), INT32_C(-1203439394), INT32_C( -261642074),
                            INT32_C(-2062167113), INT32_C( 1504166558), INT32_C( -111161554), INT32_C( -367200138),
                            INT32_C( 1040642836), INT32_C(  378025736), INT32_C( 1031970925), INT32_C(-1474878922),
                            INT32_C(-1560910320), INT32_C( 1296215099), INT32_C(-1595601438), INT32_C( -126839035)),
      simde_mm512_set_epi32(INT32_C(-2009273824), INT32_C(          0), INT32_C(-2121947989), INT32_C(-1380446359),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1310981908), INT32_C( 2115479139),
                            INT32_C( 1300697120), INT32_C(          0), INT32_C( -314007366), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(-1660657084)) },
    { UINT16_C(21153),
      simde_mm512_set_epi32(INT32_C( -788633826), INT32_C( 1642420282), INT32_C(  723895008), INT32_C(  207632598),
                            INT32_C(-2079938207), INT32_C( 1754477079), INT32_C( 1798135551), INT32_C(   23449555),
                            INT32_C( -151172429), INT32_C(  677778908), INT32_C(   90905464), INT32_C( 1354586615),
                            INT32_C(-1670436324), INT32_C( -505523122), INT32_C(-1519449460), INT32_C(-1685310582)),
      simde_mm512_set_epi32(INT32_C(  799456687), INT32_C(-1358763208), INT32_C(  737687311), INT32_C( 1515407453),
                            INT32_C(  439395016), INT32_C(  -78627541), INT32_C(-1674155016), INT32_C( 1063201251),
                            INT32_C( -686363587), INT32_C(  742525264), INT32_C(  701319512), INT32_C(   24989685),
                            INT32_C( -301118736), INT32_C( -785334161), INT32_C(-1489992316), INT32_C(  306022421)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(-1293783806), INT32_C(          0), INT32_C(-1307774855),
                            INT32_C(          0), INT32_C(          0), INT32_C( -822676729), INT32_C(          0),
                            INT32_C(  535191158), INT32_C(          0), INT32_C( -610414048), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(-1991333003)) },
    { UINT16_C(12143),
      simde_mm512_set_epi32(INT32_C( -246629264), INT32_C(  633039851), INT32_C( 1692158737), INT32_C( 1115946871),
                            INT32_C(  309808098), INT32_C( 1170830326), INT32_C( 1350105561), INT32_C(-1022199838),
                            INT32_C(  654046756), INT32_C( 1807741640), INT32_C(  224020334), INT32_C( 1191767429),
                            INT32_C( -990326759), INT32_C(   85294451), INT32_C( -252749112), INT32_C(-1788577569)),
      simde_mm512_set_epi32(INT32_C( 1174570840), INT32_C(  974062633), INT32_C(  983904988), INT32_C( 1803536893),
                            INT32_C( 1164598462), INT32_C( 1777437641), INT32_C(-1475760323), INT32_C( 1833217111),
                            INT32_C( 2013842885), INT32_C(  720911006), INT32_C(-1253744600), INT32_C( 1820529236),
                            INT32_C( -314819268), INT32_C(-1926268921), INT32_C( 2108913431), INT32_C( 1190393502)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(  708253749), INT32_C(          0),
                            INT32_C( -854790364), INT32_C( -606607315), INT32_C(-1469101412), INT32_C( 1439550347),
                            INT32_C(          0), INT32_C( 1086830634), INT32_C( 1477764934), INT32_C(          0),
                            INT32_C( -675507491), INT32_C( 2011563372), INT32_C( 1933304753), INT32_C( 1315996225)) },
    { UINT16_C(26005),
      simde_mm512_set_epi32(INT32_C( 1813548464), INT32_C( -757290941), INT32_C( 1295512986), INT32_C( 1291803276),
                            INT32_C( 2032260868), INT32_C(  316165049), INT32_C( 1037644878), INT32_C(-1728213057),
                            INT32_C(  231750243), INT32_C( 1220512969), INT32_C(-1711918828), INT32_C( 1618345779),
                            INT32_C( 1444876028), INT32_C( 1881924556), INT32_C(-1672732354), INT32_C(-1497726182)),
      simde_mm512_set_epi32(INT32_C(-2042300804), INT32_C( -199486597), INT32_C( -290224964), INT32_C(  -95049939),
                            INT32_C(  242789967), INT32_C(-2042388049), INT32_C(-1526333573), INT32_C( -943172088),
                            INT32_C(-1987449183), INT32_C( -802616226), INT32_C(  743071941), INT32_C(  -28537087),
                            INT32_C(-2054489846), INT32_C( 2118922267), INT32_C( 1876700525), INT32_C(  356823736)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C( -557804344), INT32_C( 1585737950), INT32_C(          0),
                            INT32_C(          0), INT32_C(-1936414198), INT32_C(          0), INT32_C( -785040969),
                            INT32_C(-2075767870), INT32_C(          0), INT32_C(          0), INT32_C( 1646882866),
                            INT32_C(          0), INT32_C( -236997711), INT32_C(          0), INT32_C(-1854549918)) },
    { UINT16_C(22214),
      simde_mm512_set_epi32(INT32_C( 1255503250), INT32_C(  603134448), INT32_C( 1664652192), INT32_C( -343768171),
                            INT32_C(-1798248429), INT32_C(-1446513257), INT32_C(  127732840), INT32_C(-1651163018),
                            INT32_C(  741467989), INT32_C(  859412594), INT32_C(  472043835), INT32_C( 1771260096),
                            INT32_C(-1144930983), INT32_C(  236371534), INT32_C( 1323254991), INT32_C( 1564105257)),
      simde_mm512_set_epi32(INT32_C(  438781482), INT32_C( 1278794690), INT32_C(-1026818029), INT32_C( 2082034838),
                            INT32_C(  -20030271), INT32_C( -682181759), INT32_C( 1547951192), INT32_C(  690567023),
                            INT32_C( -270117367), INT32_C( -771535010), INT32_C(  916148853), INT32_C( 1687091511),
                            INT32_C( -535908173), INT32_C( -185822843), INT32_C( -711684672), INT32_C( -424619293)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C( -675660242), INT32_C(          0), INT32_C( 1869164287),
                            INT32_C(          0), INT32_C( -764331498), INT32_C(-1420218352), INT32_C(          0),
                            INT32_C( 1011585356), INT32_C( 1630947604), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(  422194377), INT32_C( 2034939663), INT32_C(          0)) },
    { UINT16_C(35591),
      simde_mm512_set_epi32(INT32_C( 1513047065), INT32_C( -104652818), INT32_C( 1564491564), INT32_C(  -98950215),
                            INT32_C(  631827200), INT32_C( 1322294700), INT32_C(  436005702), INT32_C( 1825722103),
                            INT32_C( 2013933934), INT32_C( -532774987), INT32_C( 1616518393), INT32_C(  803856137),
                            INT32_C(-1663534883), INT32_C(-2021437227), INT32_C(-1476004613), INT32_C( -899510926)),
      simde_mm512_set_epi32(INT32_C( -910624932), INT32_C(  209536966), INT32_C(-1923748050), INT32_C(-1520303619),
                            INT32_C( -387141989), INT32_C(  959069600), INT32_C( 1208361371), INT32_C(-1838273096),
                            INT32_C(-1330134815), INT32_C(  126713528), INT32_C( -150313435), INT32_C(-1972942202),
                            INT32_C( 1666269875), INT32_C(-1750237431), INT32_C(  950405946), INT32_C( -725753907)),
      simde_mm512_set_epi32(INT32_C(-1871295299), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1018969189), INT32_C(          0), INT32_C( -772355669), INT32_C( -630972097),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( -271199796), INT32_C( 1868556737), INT32_C( -173757019)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_sub_epi32(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_sub_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C( 68),
      simde_mm512_set_epi64(INT64_C(-5073778595823407211), INT64_C( 5388732377458839959),
                            INT64_C(-1986090184057562632), INT64_C( 4645834279775613628),
                            INT64_C(-4366681866184837051), INT64_C(-8603814507289273529),
                            INT64_C( 1095688928769016575), INT64_C(-6006556600469720682)),
      simde_mm512_set_epi64(INT64_C( 3123921281503271845), INT64_C( 4387288797030316723),
                            INT64_C( 6995087775390049262), INT64_C( 8246698453208060618),
                            INT64_C( 6552221057784745846), INT64_C( 1000026456014490264),
                            INT64_C(-5730202701662528355), INT64_C( 8559420281310089233)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C( 1001443580428523236),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 8842903110405787823),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(204),
      simde_mm512_set_epi64(INT64_C(-6977414139090468936), INT64_C(-2149469661959479693),
                            INT64_C( -926175743870842459), INT64_C(-4811544340081196435),
                            INT64_C( 8898235206278318916), INT64_C( 8139946092291910736),
                            INT64_C(-6840744003613877875), INT64_C( 6930156028979502872)),
      simde_mm512_set_epi64(INT64_C( 7569921486333851063), INT64_C( -969202170885166906),
                            INT64_C( 4119221503364645671), INT64_C(-2986617930068653633),
                            INT64_C( 2330932160723884520), INT64_C(  778481013812787297),
                            INT64_C( 8002767168857919274), INT64_C( 6129279959849065926)),
      simde_mm512_set_epi64(INT64_C( 3899408448285231617), INT64_C(-1180267491074312787),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 6567303045554434396), INT64_C( 7361465078479123439),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(192),
      simde_mm512_set_epi64(INT64_C(-7052716116295772367), INT64_C(-5129303065907965926),
                            INT64_C( 7508801135919891252), INT64_C( 7191012346537132327),
                            INT64_C(  149919728852448612), INT64_C( 6013620545973361686),
                            INT64_C(-7677237369544501225), INT64_C(-4708878852454120811)),
      simde_mm512_set_epi64(INT64_C(-1123744148854811721), INT64_C( 6460346178530692910),
                            INT64_C(-1577112582756044012), INT64_C( 1623608174198300781),
                            INT64_C(-6334556732815677936), INT64_C( 5567201461485768162),
                            INT64_C( -544769504883831290), INT64_C( 6623855812203421065)),
      simde_mm512_set_epi64(INT64_C(-5928971967440960646), INT64_C( 6857094829270892780),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(222),
      simde_mm512_set_epi64(INT64_C(-8933266575011401193), INT64_C( 7722933385343389651),
                            INT64_C( -649280637934103076), INT64_C(  390435996262291959),
                            INT64_C(-7174469377841015730), INT64_C(-6525985736015203446),
                            INT64_C( 8306044009918255777), INT64_C(   50142018453906435)),
      simde_mm512_set_epi64(INT64_C( 1887187227961736491), INT64_C(-7190441041091155485),
                            INT64_C(-2947909158587725488), INT64_C( 3012144368111669237),
                            INT64_C(-1293295119823224721), INT64_C(-6399468268205275115),
                            INT64_C(-3387156489546934214), INT64_C( 3109105385305290966)),
      simde_mm512_set_epi64(INT64_C( 7626290270736413932), INT64_C(-3533369647275006480),
                            INT64_C(                   0), INT64_C(-2621708371849377278),
                            INT64_C(-5881174258017791009), INT64_C( -126517467809928331),
                            INT64_C(-6753543574244361625), INT64_C(                   0)) },
    { UINT8_C( 93),
      simde_mm512_set_epi64(INT64_C( 5028677960685124057), INT64_C(-4390314873532451292),
                            INT64_C( 7764191223641425774), INT64_C( 5118602135297642521),
                            INT64_C(  366336881617492680), INT64_C(-7681882161808553379),
                            INT64_C( 1609244596442152367), INT64_C(-5835843540630358257)),
      simde_mm512_set_epi64(INT64_C( 7634036541593595709), INT64_C( 7873607540226444741),
                            INT64_C( 3096289197137682472), INT64_C( 7819113534012013884),
                            INT64_C(-8273262016887294185), INT64_C( 5112701164509248624),
                            INT64_C( 2718885458801871633), INT64_C( 4792955315328338914)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C( 6182821659950655583),
                            INT64_C(                   0), INT64_C(-2700511398714371363),
                            INT64_C( 8639598898504786865), INT64_C( 5652160747391749613),
                            INT64_C(                   0), INT64_C( 7817945217750854445)) },
    { UINT8_C(190),
      simde_mm512_set_epi64(INT64_C( 4456650818438664127), INT64_C(  995359715745565897),
                            INT64_C(-7352635378048303309), INT64_C( 6205695288916304844),
                            INT64_C(-7184330752593853670), INT64_C(  997194843856987541),
                            INT64_C( 5044743345609311273), INT64_C( 4225839747634809341)),
      simde_mm512_set_epi64(INT64_C(-6555552775470033400), INT64_C(-8536029239954568098),
                            INT64_C( 3191469689436671745), INT64_C(-8823966696415154149),
                            INT64_C( 8060367379617854136), INT64_C( 7789131346128709699),
                            INT64_C( 5564185907705109132), INT64_C( 8728493965316737977)),
      simde_mm512_set_epi64(INT64_C(-7434540479800854089), INT64_C(                   0),
                            INT64_C( 7902639006224576562), INT64_C(-3417082088378092623),
                            INT64_C( 3202045941497843810), INT64_C(-6791936502271722158),
                            INT64_C( -519442562095797859), INT64_C(                   0)) },
    { UINT8_C(175),
      simde_mm512_set_epi64(INT64_C(-7091691161933191339), INT64_C( 3691148985472569659),
                            INT64_C( 7607504188179856729), INT64_C( 1015208009558607055),
                            INT64_C( 6717780929629073882), INT64_C( 2140431133564008060),
                            INT64_C( -856788406100589380), INT64_C( -408236379249004977)),
      simde_mm512_set_epi64(INT64_C( 2965962783505929737), INT64_C(-3313717634752884107),
                            INT64_C( 7246002868863283379), INT64_C( -798103029951459904),
                            INT64_C(-1823725975430138478), INT64_C( 2590442730915664800),
                            INT64_C(-1476473049354016749), INT64_C(-6212727131917710232)),
      simde_mm512_set_epi64(INT64_C( 8389090128270430540), INT64_C(                   0),
                            INT64_C(  361501319316573350), INT64_C(                   0),
                            INT64_C( 8541506905059212360), INT64_C( -450011597351656740),
                            INT64_C(  619684643253427369), INT64_C( 5804490752668705255)) },
    { UINT8_C( 88),
      simde_mm512_set_epi64(INT64_C( 8649780386596814773), INT64_C( 6942893632121331465),
                            INT64_C(-7144827915966656299), INT64_C(-6339391538184680078),
                            INT64_C( 7515152281876400903), INT64_C( 1884552116559207362),
                            INT64_C(-4410149851416144746), INT64_C(  -86029355262231679)),
      simde_mm512_set_epi64(INT64_C(-5712885529569296712), INT64_C( -645591285152396666),
                            INT64_C( 7156574621979737865), INT64_C( 4081962459563155405),
                            INT64_C( 6498487665674100718), INT64_C( 6719440106443908025),
                            INT64_C( 2713677162045545900), INT64_C( 1872630232785243895)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C( 7588484917273728131),
                            INT64_C(                   0), INT64_C( 8025390075961716133),
                            INT64_C( 1016664616202300185), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_sub_epi64(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_sub_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(26074),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -524.33), SIMDE_FLOAT32_C(  -241.59), SIMDE_FLOAT32_C(  -105.89), SIMDE_FLOAT32_C(  -289.61),
                         SIMDE_FLOAT32_C(  -891.58), SIMDE_FLOAT32_C(   378.73), SIMDE_FLOAT32_C(   -71.99), SIMDE_FLOAT32_C(   449.90),
                         SIMDE_FLOAT32_C(  -415.75), SIMDE_FLOAT32_C(   784.67), SIMDE_FLOAT32_C(  -496.30), SIMDE_FLOAT32_C(   526.56),
                         SIMDE_FLOAT32_C(    67.17), SIMDE_FLOAT32_C(  -881.21), SIMDE_FLOAT32_C(   348.77), SIMDE_FLOAT32_C(   537.04)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   392.86), SIMDE_FLOAT32_C(    15.96), SIMDE_FLOAT32_C(  -681.24), SIMDE_FLOAT32_C(   759.61),
                         SIMDE_FLOAT32_C(  -507.08), SIMDE_FLOAT32_C(  -150.50), SIMDE_FLOAT32_C(   409.54), SIMDE_FLOAT32_C(  -197.17),
                         SIMDE_FLOAT32_C(   554.42), SIMDE_FLOAT32_C(   844.38), SIMDE_FLOAT32_C(  -817.51), SIMDE_FLOAT32_C(   338.74),
                         SIMDE_FLOAT32_C(   -70.99), SIMDE_FLOAT32_C(  -221.33), SIMDE_FLOAT32_C(    59.42), SIMDE_FLOAT32_C(   138.47)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -257.55), SIMDE_FLOAT32_C(   575.35), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   529.23), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   647.07),
                         SIMDE_FLOAT32_C(  -970.17), SIMDE_FLOAT32_C(   -59.71), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   187.82),
                         SIMDE_FLOAT32_C(   138.16), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   289.35), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(10432),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -855.26), SIMDE_FLOAT32_C(   444.16), SIMDE_FLOAT32_C(   962.50), SIMDE_FLOAT32_C(   987.86),
                         SIMDE_FLOAT32_C(  -410.31), SIMDE_FLOAT32_C(    36.70), SIMDE_FLOAT32_C(   874.49), SIMDE_FLOAT32_C(  -627.16),
                         SIMDE_FLOAT32_C(   911.91), SIMDE_FLOAT32_C(  -816.98), SIMDE_FLOAT32_C(  -164.10), SIMDE_FLOAT32_C(  -340.48),
                         SIMDE_FLOAT32_C(   -77.39), SIMDE_FLOAT32_C(   952.25), SIMDE_FLOAT32_C(   134.46), SIMDE_FLOAT32_C(   698.09)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -179.27), SIMDE_FLOAT32_C(   894.92), SIMDE_FLOAT32_C(  -553.39), SIMDE_FLOAT32_C(   676.19),
                         SIMDE_FLOAT32_C(  -747.28), SIMDE_FLOAT32_C(  -915.60), SIMDE_FLOAT32_C(  -132.34), SIMDE_FLOAT32_C(  -335.46),
                         SIMDE_FLOAT32_C(   243.51), SIMDE_FLOAT32_C(   766.95), SIMDE_FLOAT32_C(   899.58), SIMDE_FLOAT32_C(   478.33),
                         SIMDE_FLOAT32_C(   -35.25), SIMDE_FLOAT32_C(  -117.47), SIMDE_FLOAT32_C(   258.33), SIMDE_FLOAT32_C(  -248.63)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  1515.89), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   336.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   668.40), SIMDE_FLOAT32_C( -1583.93), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C( 9219),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   878.16), SIMDE_FLOAT32_C(  -299.57), SIMDE_FLOAT32_C(   829.01), SIMDE_FLOAT32_C(  -823.97),
                         SIMDE_FLOAT32_C(   313.21), SIMDE_FLOAT32_C(  -396.40), SIMDE_FLOAT32_C(   940.94), SIMDE_FLOAT32_C(  -281.84),
                         SIMDE_FLOAT32_C(   235.34), SIMDE_FLOAT32_C(   443.88), SIMDE_FLOAT32_C(  -185.89), SIMDE_FLOAT32_C(  -220.35),
                         SIMDE_FLOAT32_C(  -983.75), SIMDE_FLOAT32_C(  -348.00), SIMDE_FLOAT32_C(   167.63), SIMDE_FLOAT32_C(   489.46)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   439.60), SIMDE_FLOAT32_C(    39.73), SIMDE_FLOAT32_C(   948.24), SIMDE_FLOAT32_C(  -515.41),
                         SIMDE_FLOAT32_C(  -519.45), SIMDE_FLOAT32_C(   273.14), SIMDE_FLOAT32_C(   256.99), SIMDE_FLOAT32_C(    69.80),
                         SIMDE_FLOAT32_C(  -548.50), SIMDE_FLOAT32_C(  -730.33), SIMDE_FLOAT32_C(   337.76), SIMDE_FLOAT32_C(    90.27),
                         SIMDE_FLOAT32_C(  -665.67), SIMDE_FLOAT32_C(   930.26), SIMDE_FLOAT32_C(  -181.77), SIMDE_FLOAT32_C(   530.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -119.23), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -669.54), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   349.40), SIMDE_FLOAT32_C(   -41.37)) },
    { UINT16_C(60216),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -294.33), SIMDE_FLOAT32_C(   963.39), SIMDE_FLOAT32_C(  -504.91), SIMDE_FLOAT32_C(  -654.23),
                         SIMDE_FLOAT32_C(  -988.36), SIMDE_FLOAT32_C(   634.30), SIMDE_FLOAT32_C(  -857.50), SIMDE_FLOAT32_C(  -235.19),
                         SIMDE_FLOAT32_C(  -903.31), SIMDE_FLOAT32_C(  -183.01), SIMDE_FLOAT32_C(  -989.08), SIMDE_FLOAT32_C(  -684.38),
                         SIMDE_FLOAT32_C(  -369.22), SIMDE_FLOAT32_C(   764.60), SIMDE_FLOAT32_C(   215.22), SIMDE_FLOAT32_C(  -906.73)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -656.49), SIMDE_FLOAT32_C(  -795.39), SIMDE_FLOAT32_C(   220.41), SIMDE_FLOAT32_C(   680.39),
                         SIMDE_FLOAT32_C(  -673.42), SIMDE_FLOAT32_C(   859.78), SIMDE_FLOAT32_C(   306.17), SIMDE_FLOAT32_C(   632.76),
                         SIMDE_FLOAT32_C(  -662.91), SIMDE_FLOAT32_C(    31.45), SIMDE_FLOAT32_C(  -162.68), SIMDE_FLOAT32_C(   929.60),
                         SIMDE_FLOAT32_C(  -957.67), SIMDE_FLOAT32_C(   222.14), SIMDE_FLOAT32_C(   292.45), SIMDE_FLOAT32_C(   -99.46)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   362.16), SIMDE_FLOAT32_C(  1758.78), SIMDE_FLOAT32_C(  -725.32), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -314.94), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -1163.67), SIMDE_FLOAT32_C(  -867.95),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -826.40), SIMDE_FLOAT32_C( -1613.98),
                         SIMDE_FLOAT32_C(   588.45), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C( 1065),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -160.16), SIMDE_FLOAT32_C(  -172.32), SIMDE_FLOAT32_C(  -146.34), SIMDE_FLOAT32_C(  -664.30),
                         SIMDE_FLOAT32_C(  -152.25), SIMDE_FLOAT32_C(   103.01), SIMDE_FLOAT32_C(  -445.68), SIMDE_FLOAT32_C(  -705.22),
                         SIMDE_FLOAT32_C(  -480.35), SIMDE_FLOAT32_C(  -454.79), SIMDE_FLOAT32_C(   524.00), SIMDE_FLOAT32_C(  -158.20),
                         SIMDE_FLOAT32_C(  -445.04), SIMDE_FLOAT32_C(  -960.28), SIMDE_FLOAT32_C(   167.13), SIMDE_FLOAT32_C(  -825.53)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -541.83), SIMDE_FLOAT32_C(  -457.69), SIMDE_FLOAT32_C(   312.80), SIMDE_FLOAT32_C(   -62.23),
                         SIMDE_FLOAT32_C(   416.18), SIMDE_FLOAT32_C(   853.40), SIMDE_FLOAT32_C(   -17.96), SIMDE_FLOAT32_C(   885.15),
                         SIMDE_FLOAT32_C(  -212.03), SIMDE_FLOAT32_C(  -855.73), SIMDE_FLOAT32_C(  -371.31), SIMDE_FLOAT32_C(  -695.44),
                         SIMDE_FLOAT32_C(  -895.68), SIMDE_FLOAT32_C(   538.84), SIMDE_FLOAT32_C(   882.30), SIMDE_FLOAT32_C(   585.87)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -750.39), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   895.31), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   450.64), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -1411.40)) },
    { UINT16_C( 4987),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   955.74), SIMDE_FLOAT32_C(    48.94), SIMDE_FLOAT32_C(   560.80), SIMDE_FLOAT32_C(   626.25),
                         SIMDE_FLOAT32_C(   986.71), SIMDE_FLOAT32_C(   -13.30), SIMDE_FLOAT32_C(  -833.84), SIMDE_FLOAT32_C(   647.36),
                         SIMDE_FLOAT32_C(  -398.46), SIMDE_FLOAT32_C(  -852.77), SIMDE_FLOAT32_C(   195.24), SIMDE_FLOAT32_C(  -431.65),
                         SIMDE_FLOAT32_C(  -246.40), SIMDE_FLOAT32_C(  -123.66), SIMDE_FLOAT32_C(   302.57), SIMDE_FLOAT32_C(  -312.92)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   864.85), SIMDE_FLOAT32_C(  -886.94), SIMDE_FLOAT32_C(   289.25), SIMDE_FLOAT32_C(    74.52),
                         SIMDE_FLOAT32_C(  -653.98), SIMDE_FLOAT32_C(    43.30), SIMDE_FLOAT32_C(  -126.09), SIMDE_FLOAT32_C(  -155.50),
                         SIMDE_FLOAT32_C(  -396.73), SIMDE_FLOAT32_C(   -53.65), SIMDE_FLOAT32_C(  -516.81), SIMDE_FLOAT32_C(  -892.08),
                         SIMDE_FLOAT32_C(   202.83), SIMDE_FLOAT32_C(  -327.18), SIMDE_FLOAT32_C(   221.07), SIMDE_FLOAT32_C(  -891.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   551.73),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -707.75), SIMDE_FLOAT32_C(   802.86),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -799.12), SIMDE_FLOAT32_C(   712.05), SIMDE_FLOAT32_C(   460.43),
                         SIMDE_FLOAT32_C(  -449.23), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    81.50), SIMDE_FLOAT32_C(   578.96)) },
    { UINT16_C(56258),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -30.48), SIMDE_FLOAT32_C(   682.33), SIMDE_FLOAT32_C(  -678.43), SIMDE_FLOAT32_C(   640.73),
                         SIMDE_FLOAT32_C(  -214.39), SIMDE_FLOAT32_C(   913.47), SIMDE_FLOAT32_C(   802.27), SIMDE_FLOAT32_C(  -719.14),
                         SIMDE_FLOAT32_C(   839.92), SIMDE_FLOAT32_C(   326.41), SIMDE_FLOAT32_C(   231.12), SIMDE_FLOAT32_C(  -599.80),
                         SIMDE_FLOAT32_C(  -175.19), SIMDE_FLOAT32_C(  -889.93), SIMDE_FLOAT32_C(  -271.66), SIMDE_FLOAT32_C(  -767.93)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   521.85), SIMDE_FLOAT32_C(   990.67), SIMDE_FLOAT32_C(  -279.18), SIMDE_FLOAT32_C(   874.22),
                         SIMDE_FLOAT32_C(  -573.38), SIMDE_FLOAT32_C(   750.45), SIMDE_FLOAT32_C(   668.60), SIMDE_FLOAT32_C(  -415.36),
                         SIMDE_FLOAT32_C(  -224.84), SIMDE_FLOAT32_C(   162.63), SIMDE_FLOAT32_C(  -940.52), SIMDE_FLOAT32_C(  -654.73),
                         SIMDE_FLOAT32_C(  -780.19), SIMDE_FLOAT32_C(   466.85), SIMDE_FLOAT32_C(  -383.81), SIMDE_FLOAT32_C(   542.46)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -552.33), SIMDE_FLOAT32_C(  -308.34), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -233.49),
                         SIMDE_FLOAT32_C(   358.99), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   133.67), SIMDE_FLOAT32_C(  -303.78),
                         SIMDE_FLOAT32_C(  1064.76), SIMDE_FLOAT32_C(   163.78), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   112.15), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(18374),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   292.05), SIMDE_FLOAT32_C(  -553.40), SIMDE_FLOAT32_C(   143.99), SIMDE_FLOAT32_C(  -940.99),
                         SIMDE_FLOAT32_C(    81.28), SIMDE_FLOAT32_C(   184.98), SIMDE_FLOAT32_C(   662.04), SIMDE_FLOAT32_C(   951.27),
                         SIMDE_FLOAT32_C(   953.92), SIMDE_FLOAT32_C(  -384.26), SIMDE_FLOAT32_C(  -149.83), SIMDE_FLOAT32_C(   751.91),
                         SIMDE_FLOAT32_C(  -625.68), SIMDE_FLOAT32_C(    58.69), SIMDE_FLOAT32_C(   581.13), SIMDE_FLOAT32_C(   892.26)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   104.19), SIMDE_FLOAT32_C(   819.72), SIMDE_FLOAT32_C(  -437.31), SIMDE_FLOAT32_C(   380.61),
                         SIMDE_FLOAT32_C(   930.00), SIMDE_FLOAT32_C(  -224.08), SIMDE_FLOAT32_C(  -557.43), SIMDE_FLOAT32_C(  -295.43),
                         SIMDE_FLOAT32_C(  -271.48), SIMDE_FLOAT32_C(  -705.78), SIMDE_FLOAT32_C(  -796.97), SIMDE_FLOAT32_C(   -62.19),
                         SIMDE_FLOAT32_C(  -247.25), SIMDE_FLOAT32_C(   225.36), SIMDE_FLOAT32_C(   312.68), SIMDE_FLOAT32_C(  -185.21)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -1373.12), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   409.06), SIMDE_FLOAT32_C(  1219.47), SIMDE_FLOAT32_C(  1246.70),
                         SIMDE_FLOAT32_C(  1225.40), SIMDE_FLOAT32_C(   321.52), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -166.67), SIMDE_FLOAT32_C(   268.45), SIMDE_FLOAT32_C(     0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_maskz_sub_ps(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_sub_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C( 63),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -415.75), SIMDE_FLOAT64_C(  784.67),
                         SIMDE_FLOAT64_C( -496.30), SIMDE_FLOAT64_C(  526.56),
                         SIMDE_FLOAT64_C(   67.17), SIMDE_FLOAT64_C( -881.21),
                         SIMDE_FLOAT64_C(  348.77), SIMDE_FLOAT64_C(  537.04)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  554.42), SIMDE_FLOAT64_C(  844.38),
                         SIMDE_FLOAT64_C( -817.51), SIMDE_FLOAT64_C(  338.74),
                         SIMDE_FLOAT64_C(  -70.99), SIMDE_FLOAT64_C( -221.33),
                         SIMDE_FLOAT64_C(   59.42), SIMDE_FLOAT64_C(  138.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  321.21), SIMDE_FLOAT64_C(  187.82),
                         SIMDE_FLOAT64_C(  138.16), SIMDE_FLOAT64_C( -659.88),
                         SIMDE_FLOAT64_C(  289.35), SIMDE_FLOAT64_C(  398.57)) },
    { UINT8_C(204),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  392.86), SIMDE_FLOAT64_C(   15.96),
                         SIMDE_FLOAT64_C( -681.24), SIMDE_FLOAT64_C(  759.61),
                         SIMDE_FLOAT64_C( -507.08), SIMDE_FLOAT64_C( -150.50),
                         SIMDE_FLOAT64_C(  409.54), SIMDE_FLOAT64_C( -197.17)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -661.30), SIMDE_FLOAT64_C( -524.33),
                         SIMDE_FLOAT64_C( -241.59), SIMDE_FLOAT64_C( -105.89),
                         SIMDE_FLOAT64_C( -289.61), SIMDE_FLOAT64_C( -891.58),
                         SIMDE_FLOAT64_C(  378.73), SIMDE_FLOAT64_C(  -71.99)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( 1054.16), SIMDE_FLOAT64_C(  540.29),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -217.47), SIMDE_FLOAT64_C(  741.08),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(198),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  911.91), SIMDE_FLOAT64_C( -816.98),
                         SIMDE_FLOAT64_C( -164.10), SIMDE_FLOAT64_C( -340.48),
                         SIMDE_FLOAT64_C(  -77.39), SIMDE_FLOAT64_C(  952.25),
                         SIMDE_FLOAT64_C(  134.46), SIMDE_FLOAT64_C(  698.09)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  243.51), SIMDE_FLOAT64_C(  766.95),
                         SIMDE_FLOAT64_C(  899.58), SIMDE_FLOAT64_C(  478.33),
                         SIMDE_FLOAT64_C(  -35.25), SIMDE_FLOAT64_C( -117.47),
                         SIMDE_FLOAT64_C(  258.33), SIMDE_FLOAT64_C( -248.63)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  668.40), SIMDE_FLOAT64_C(-1583.93),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( 1069.72),
                         SIMDE_FLOAT64_C( -123.87), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(149),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -179.27), SIMDE_FLOAT64_C(  894.92),
                         SIMDE_FLOAT64_C( -553.39), SIMDE_FLOAT64_C(  676.19),
                         SIMDE_FLOAT64_C( -747.28), SIMDE_FLOAT64_C( -915.60),
                         SIMDE_FLOAT64_C( -132.34), SIMDE_FLOAT64_C( -335.46)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -395.05), SIMDE_FLOAT64_C( -855.26),
                         SIMDE_FLOAT64_C(  444.16), SIMDE_FLOAT64_C(  962.50),
                         SIMDE_FLOAT64_C(  987.86), SIMDE_FLOAT64_C( -410.31),
                         SIMDE_FLOAT64_C(   36.70), SIMDE_FLOAT64_C(  874.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  215.78), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -286.31),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -505.29),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(-1209.95)) },
    { UINT8_C(196),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  235.34), SIMDE_FLOAT64_C(  443.88),
                         SIMDE_FLOAT64_C( -185.89), SIMDE_FLOAT64_C( -220.35),
                         SIMDE_FLOAT64_C( -983.75), SIMDE_FLOAT64_C( -348.00),
                         SIMDE_FLOAT64_C(  167.63), SIMDE_FLOAT64_C(  489.46)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -548.50), SIMDE_FLOAT64_C( -730.33),
                         SIMDE_FLOAT64_C(  337.76), SIMDE_FLOAT64_C(   90.27),
                         SIMDE_FLOAT64_C( -665.67), SIMDE_FLOAT64_C(  930.26),
                         SIMDE_FLOAT64_C( -181.77), SIMDE_FLOAT64_C(  530.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  783.84), SIMDE_FLOAT64_C( 1174.21),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(-1278.26),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(230),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  439.60), SIMDE_FLOAT64_C(   39.73),
                         SIMDE_FLOAT64_C(  948.24), SIMDE_FLOAT64_C( -515.41),
                         SIMDE_FLOAT64_C( -519.45), SIMDE_FLOAT64_C(  273.14),
                         SIMDE_FLOAT64_C(  256.99), SIMDE_FLOAT64_C(   69.80)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -138.55), SIMDE_FLOAT64_C(  878.16),
                         SIMDE_FLOAT64_C( -299.57), SIMDE_FLOAT64_C(  829.01),
                         SIMDE_FLOAT64_C( -823.97), SIMDE_FLOAT64_C(  313.21),
                         SIMDE_FLOAT64_C( -396.40), SIMDE_FLOAT64_C(  940.94)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  578.15), SIMDE_FLOAT64_C( -838.43),
                         SIMDE_FLOAT64_C( 1247.81), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  -40.07),
                         SIMDE_FLOAT64_C(  653.39), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C( 58),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -903.31), SIMDE_FLOAT64_C( -183.01),
                         SIMDE_FLOAT64_C( -989.08), SIMDE_FLOAT64_C( -684.38),
                         SIMDE_FLOAT64_C( -369.22), SIMDE_FLOAT64_C(  764.60),
                         SIMDE_FLOAT64_C(  215.22), SIMDE_FLOAT64_C( -906.73)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -662.91), SIMDE_FLOAT64_C(   31.45),
                         SIMDE_FLOAT64_C( -162.68), SIMDE_FLOAT64_C(  929.60),
                         SIMDE_FLOAT64_C( -957.67), SIMDE_FLOAT64_C(  222.14),
                         SIMDE_FLOAT64_C(  292.45), SIMDE_FLOAT64_C(  -99.46)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -826.40), SIMDE_FLOAT64_C(-1613.98),
                         SIMDE_FLOAT64_C(  588.45), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  -77.23), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(175),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -656.49), SIMDE_FLOAT64_C( -795.39),
                         SIMDE_FLOAT64_C(  220.41), SIMDE_FLOAT64_C(  680.39),
                         SIMDE_FLOAT64_C( -673.42), SIMDE_FLOAT64_C(  859.78),
                         SIMDE_FLOAT64_C(  306.17), SIMDE_FLOAT64_C(  632.76)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  367.28), SIMDE_FLOAT64_C( -294.33),
                         SIMDE_FLOAT64_C(  963.39), SIMDE_FLOAT64_C( -504.91),
                         SIMDE_FLOAT64_C( -654.23), SIMDE_FLOAT64_C( -988.36),
                         SIMDE_FLOAT64_C(  634.30), SIMDE_FLOAT64_C( -857.50)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(-1023.77), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -742.98), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  -19.19), SIMDE_FLOAT64_C( 1848.14),
                         SIMDE_FLOAT64_C( -328.13), SIMDE_FLOAT64_C( 1490.26)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_maskz_sub_pd(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_mask_sub_epi16)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm_maskz_sub_epi16)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_sub_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_sub_epi8)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_sub_epi8)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_sub_epi16)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_sub_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_sub_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_sub_epi32)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_sub_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_sub_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_sub_epi64)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_sub_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_sub_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_sub_ps)

  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_sub_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_sub_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_sub_pd)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
