//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file SAXSPhysicsListMessenger.cc
/// \brief Implementation of the SAXSPhysicsListMessenger class

#include "SAXSPhysicsListMessenger.hh"

#include "SAXSPhysicsList.hh"

#include "G4UIcmdWithABool.hh"
#include "G4UIcmdWithADoubleAndUnit.hh"
#include "G4UIcmdWithAString.hh"
#include "G4UIcmdWithAnInteger.hh"
#include "G4UIdirectory.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

SAXSPhysicsListMessenger::SAXSPhysicsListMessenger(SAXSPhysicsList* pPhys)
  : G4UImessenger(), fPhysicsList(pPhys)
{
  fPhysDir = new G4UIdirectory("/phys/");
  fPhysDir->SetGuidance("PhysicsList control");

  fCutsCmd = new G4UIcmdWithADoubleAndUnit("/phys/setCuts", this);
  fCutsCmd->SetGuidance("set cuts");
  fCutsCmd->SetParameterName("cuts", false);
  fCutsCmd->SetUnitCategory("Length");
  fCutsCmd->SetRange("cuts>0.");
  fCutsCmd->AvailableForStates(G4State_PreInit, G4State_Idle);

  fListCmd = new G4UIcmdWithAString("/phys/SelectPhysicsList", this);
  fListCmd->SetGuidance("Select the EM physics list");
  fListCmd->SetParameterName("PList", false);
  fListCmd->AvailableForStates(G4State_PreInit);

  fUseMIFlagCmd = new G4UIcmdWithABool("/phys/useMIFlag", this);
  fUseMIFlagCmd->SetGuidance("Enable (default) or disable MI for the PenelopeMI physics list");
  fUseMIFlagCmd->SetParameterName("useMIFlag", true);
  fUseMIFlagCmd->AvailableForStates(G4State_PreInit);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

SAXSPhysicsListMessenger::~SAXSPhysicsListMessenger()
{
  delete fPhysDir;
  delete fCutsCmd;
  delete fListCmd;
  delete fUseMIFlagCmd;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void SAXSPhysicsListMessenger::SetNewValue(G4UIcommand* command, G4String newValue)
{
  if (command == fCutsCmd) fPhysicsList->SetDefaultCutsValue(fCutsCmd->GetNewDoubleValue(newValue));
  if (command == fListCmd) fPhysicsList->SelectPhysicsList(newValue);
  if (command == fUseMIFlagCmd)
    fPhysicsList->SetUseMIFlag(fUseMIFlagCmd->GetNewBoolValue(newValue));
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
