/********************************************************************************
*                                                                               *
*               S t r i n g   t o   D o u b l e   C o n v e r s i o n           *
*                                                                               *
*********************************************************************************
* Copyright (C) 2005,2025 by Jeroen van der Zijp.   All Rights Reserved.        *
*********************************************************************************
* This library is free software; you can redistribute it and/or modify          *
* it under the terms of the GNU Lesser General Public License as published by   *
* the Free Software Foundation; either version 3 of the License, or             *
* (at your option) any later version.                                           *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                 *
* GNU Lesser General Public License for more details.                           *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public License      *
* along with this program.  If not, see <http://www.gnu.org/licenses/>          *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "fxmath.h"
#include "fxendian.h"
#include "FXElement.h"
#include "FXArray.h"
#include "FXMetaClass.h"
#include "fxascii.h"

/*
  Notes:
  - Thread-safe conversion of strings to double and float, with extra parameter
    for conversion success.
  - Checks both overflow and underflow. Note that characters are consumed in case of
    overflow or underflow, but OK flag is still false when the condition is raised.
  - Accuracy is in the order of 16 digits; this is the best one can do with doubles.
  - We only work with the first 20 significants digits; processing more will not
    increase accuracy and may cause trouble.
  - When compiled with FTZ/DAZ (-ffast-math) or equivalent, results from conversions
    will be less accurate!
*/


using namespace FX;

/*******************************************************************************/

namespace FX {

extern FXAPI FXdouble __strtod(const FXchar *beg,const FXchar** end=nullptr,FXbool* ok=nullptr);
extern FXAPI FXfloat __strtof(const FXchar *beg,const FXchar** end=nullptr,FXbool* ok=nullptr);

// Core conversion routines
extern FXAPI FXdouble floatFromDec64(FXulong value,FXint decex);
extern FXAPI FXdouble floatFromHex64(FXulong value,FXint binex);
extern FXAPI FXfloat floatFromDec32(FXulong value,FXint decex);
extern FXAPI FXfloat floatFromHex32(FXulong value,FXint binex);

/*******************************************************************************/

// Maximum decimal digits collected
const FXulong DECIMAX=FXULONG(1000000000000000000);

// Maximum hexadecimal digits collected
const FXulong HEXAMAX=FXULONG(0x1000000000000000);


// Values of double precision 10**x
static FXdouble tenTo64[]={
  1E+000,1E+001,1E+002,1E+003,1E+004,1E+005,1E+006,1E+007,
  1E+008,1E+009,1E+010,1E+011,1E+012,1E+013,1E+014,1E+015,
  1E+016,1E+017,1E+018,1E+019,1E+020,1E+021,1E+022,1E+023
  };


// Values of single precision 10**x
static FXfloat tenTo32[]={
  1E+000f,1E+001f,1E+002f,1E+003f,1E+004f,1E+005f,
  1E+006f,1E+007f,1E+008f,1E+009f,1E+010f,1E+011f
  };


// lower 64-bits
static const FXulong lofive[]={
  FXULONG(0x113faa2906a13b3f),FXULONG(0x4ac7ca59a424c507),FXULONG(0x5d79bcf00d2df649),FXULONG(0xf4d82c2c107973dc),
  FXULONG(0x79071b9b8a4be869),FXULONG(0x9748e2826cdee284),FXULONG(0xfd1b1b2308169b25),FXULONG(0xfe30f0f5e50e20f7),
  FXULONG(0xbdbd2d335e51a935),FXULONG(0xad2c788035e61382),FXULONG(0x4c3bcb5021afcc31),FXULONG(0xdf4abe242a1bbf3d),
  FXULONG(0xd71d6dad34a2af0d),FXULONG(0x8672648c40e5ad68),FXULONG(0x680efdaf511f18c2),FXULONG(0x0212bd1b2566def2),
  FXULONG(0x014bb630f7604b57),FXULONG(0x419ea3bd35385e2d),FXULONG(0x52064cac828675b9),FXULONG(0x7343efebd1940993),
  FXULONG(0x1014ebe6c5f90bf8),FXULONG(0xd41a26e077774ef6),FXULONG(0x8920b098955522b4),FXULONG(0x55b46e5f5d5535b0),
  FXULONG(0xeb2189f734aa831d),FXULONG(0xa5e9ec7501d523e4),FXULONG(0x47b233c92125366e),FXULONG(0x999ec0bb696e840a),
  FXULONG(0xc00670ea43ca250d),FXULONG(0x380406926a5e5728),FXULONG(0xc605083704f5ecf2),FXULONG(0xf7864a44c633682e),
  FXULONG(0x7ab3ee6afbe0211d),FXULONG(0x5960ea05bad82964),FXULONG(0x6fb92487298e33bd),FXULONG(0xa5d3b6d479f8e056),
  FXULONG(0x8f48a4899877186c),FXULONG(0x331acdabfe94de87),FXULONG(0x9ff0c08b7f1d0b14),FXULONG(0x07ecf0ae5ee44dd9),
  FXULONG(0xc9e82cd9f69d6150),FXULONG(0xbe311c083a225cd2),FXULONG(0x6dbd630a48aaf406),FXULONG(0x092cbbccdad5b108),
  FXULONG(0x25bbf56008c58ea5),FXULONG(0xaf2af2b80af6f24e),FXULONG(0x1af5af660db4aee1),FXULONG(0x50d98d9fc890ed4d),
  FXULONG(0xe50ff107bab528a0),FXULONG(0x1e53ed49a96272c8),FXULONG(0x25e8e89c13bb0f7a),FXULONG(0x77b191618c54e9ac),
  FXULONG(0xd59df5b9ef6a2417),FXULONG(0x4b0573286b44ad1d),FXULONG(0x4ee367f9430aec32),FXULONG(0x229c41f793cda73f),
  FXULONG(0x6b43527578c1110f),FXULONG(0x830a13896b78aaa9),FXULONG(0x23cc986bc656d553),FXULONG(0x2cbfbe86b7ec8aa8),
  FXULONG(0x7bf7d71432f3d6a9),FXULONG(0xdaf5ccd93fb0cc53),FXULONG(0xd1b3400f8f9cff68),FXULONG(0x23100809b9c21fa1),
  FXULONG(0xabd40a0c2832a78a),FXULONG(0x16c90c8f323f516c),FXULONG(0xae3da7d97f6792e3),FXULONG(0x99cd11cfdf41779c),
  FXULONG(0x40405643d711d583),FXULONG(0x482835ea666b2572),FXULONG(0xda3243650005eecf),FXULONG(0x90bed43e40076a82),
  FXULONG(0x5a7744a6e804a291),FXULONG(0x711515d0a205cb36),FXULONG(0x0d5a5b44ca873e03),FXULONG(0xe858790afe9486c2),
  FXULONG(0x626e974dbe39a872),FXULONG(0xfb0a3d212dc8128f),FXULONG(0x7ce66634bc9d0b99),FXULONG(0x1c1fffc1ebc44e80),
  FXULONG(0xa327ffb266b56220),FXULONG(0x4bf1ff9f0062baa8),FXULONG(0x6f773fc3603db4a9),FXULONG(0xcb550fb4384d21d3),
  FXULONG(0x7e2a53a146606a48),FXULONG(0x2eda7444cbfc426d),FXULONG(0xfa911155fefb5308),FXULONG(0x793555ab7eba27ca),
  FXULONG(0x4bc1558b2f3458de),FXULONG(0x9eb1aaedfb016f16),FXULONG(0x465e15a979c1cadc),FXULONG(0x0bfacd89ec191ec9),
  FXULONG(0xcef980ec671f667b),FXULONG(0x82b7e12780e7401a),FXULONG(0xd1b2ecb8b0908810),FXULONG(0x861fa7e6dcb4aa15),
  FXULONG(0x67a791e093e1d49a),FXULONG(0xe0c8bb2c5c6d24e0),FXULONG(0x58fae9f773886e18),FXULONG(0xaf39a475506a899e),
  FXULONG(0x6d8406c952429603),FXULONG(0xc8e5087ba6d33b83),FXULONG(0xfb1e4a9a90880a64),FXULONG(0x5cf2eea09a55067f),
  FXULONG(0xf42faa48c0ea481e),FXULONG(0xf13b94daf124da26),FXULONG(0x76c53d08d6b70858),FXULONG(0x54768c4b0c64ca6e),
  FXULONG(0xa9942f5dcf7dfd09),FXULONG(0xd3f93b35435d7c4c),FXULONG(0xc47bc5014a1a6daf),FXULONG(0x359ab6419ca1091b),
  FXULONG(0xc30163d203c94b62),FXULONG(0x79e0de63425dcf1d),FXULONG(0x985915fc12f542e4),FXULONG(0x3e6f5b7b17b2939d),
  FXULONG(0xa705992ceecf9c42),FXULONG(0x50c6ff782a838353),FXULONG(0xa4f8bf5635246428),FXULONG(0x871b7795e136be99),
  FXULONG(0x28e2557b59846e3f),FXULONG(0x331aeada2fe589cf),FXULONG(0x3ff0d2c85def7621),FXULONG(0x0fed077a756b53a9),
  FXULONG(0xd3e8495912c62894),FXULONG(0x64712dd7abbbd95c),FXULONG(0xbd8d794d96aacfb3),FXULONG(0xecf0d7a0fc5583a0),
  FXULONG(0xf41686c49db57244),FXULONG(0x311c2875c522ced5),FXULONG(0x7d633293366b828b),FXULONG(0xae5dff9c02033197),
  FXULONG(0xd9f57f830283fdfc),FXULONG(0xd072df63c324fd7b),FXULONG(0x4247cb9e59f71e6d),FXULONG(0x52d9be85f074e608),
  FXULONG(0x67902e276c921f8b),FXULONG(0x00ba1cd8a3db53b6),FXULONG(0x80e8a40eccd228a4),FXULONG(0x6122cd128006b2cd),
  FXULONG(0x796b805720085f81),FXULONG(0xcbe3303674053bb0),FXULONG(0xbedbfc4411068a9c),FXULONG(0xee92fb5515482d44),
  FXULONG(0x751bdd152d4d1c4a),FXULONG(0xd262d45a78a0635d),FXULONG(0x86fb897116c87c34),FXULONG(0xd45d35e6ae3d4da0),
  FXULONG(0x8974836059cca109),FXULONG(0x2bd1a438703fc94b),FXULONG(0x7b6306a34627ddcf),FXULONG(0x1a3bc84c17b1d542),
  FXULONG(0x20caba5f1d9e4a93),FXULONG(0x547eb47b7282ee9c),FXULONG(0xe99e619a4f23aa43),FXULONG(0x6405fa00e2ec94d4),
  FXULONG(0xde83bc408dd3dd04),FXULONG(0x9624ab50b148d445),FXULONG(0x3badd624dd9b0957),FXULONG(0xe54ca5d70a80e5d6),
  FXULONG(0x5e9fcf4ccd211f4c),FXULONG(0x7647c3200069671f),FXULONG(0x29ecd9f40041e073),FXULONG(0xf468107100525890),
  FXULONG(0x7182148d4066eeb4),FXULONG(0xc6f14cd848405530),FXULONG(0xb8ada00e5a506a7c),FXULONG(0xa6d90811f0e4851c),
  FXULONG(0x908f4a166d1da663),FXULONG(0x9a598e4e043287fe),FXULONG(0x40eff1e1853f29fd),FXULONG(0xd12bee59e68ef47c),
  FXULONG(0x82bb74f8301958ce),FXULONG(0xe36a52363c1faf01),FXULONG(0xdc44e6c3cb279ac1),FXULONG(0x29ab103a5ef8c0b9),
  FXULONG(0x7415d448f6b6f0e7),FXULONG(0x111b495b3464ad21),FXULONG(0xcab10dd900beec34),FXULONG(0x3d5d514f40eea742),
  FXULONG(0x0cb4a5a3112a5112),FXULONG(0x47f0e785eaba72ab),FXULONG(0x59ed216765690f56),FXULONG(0x306869c13ec3532c),
  FXULONG(0x1e414218c73a13fb),FXULONG(0xe5d1929ef90898fa),FXULONG(0xdf45f746b74abf39),FXULONG(0x6b8bba8c328eb783),
  FXULONG(0x066ea92f3f326564),FXULONG(0xc80a537b0efefebd),FXULONG(0xbd06742ce95f5f36),FXULONG(0x2c48113823b73704),
  FXULONG(0xf75a15862ca504c5),FXULONG(0x9a984d73dbe722fb),FXULONG(0xc13e60d0d2e0ebba),FXULONG(0x318df905079926a8),
  FXULONG(0xfdf17746497f7052),FXULONG(0xfeb6ea8bedefa633),FXULONG(0xfe64a52ee96b8fc0),FXULONG(0x3dfdce7aa3c673b0),
  FXULONG(0x06bea10ca65c084e),FXULONG(0x486e494fcff30a62),FXULONG(0x5a89dba3c3efccfa),FXULONG(0xf89629465a75e01c),
  FXULONG(0xf6bbb397f1135823),FXULONG(0x746aa07ded582e2c),FXULONG(0xa8c2a44eb4571cdc),FXULONG(0x92f34d62616ce413),
  FXULONG(0x77b020baf9c81d17),FXULONG(0x0ace1474dc1d122e),FXULONG(0x0d819992132456ba),FXULONG(0x10e1fff697ed6c69),
  FXULONG(0xca8d3ffa1ef463c1),FXULONG(0xbd308ff8a6b17cb2),FXULONG(0xac7cb3f6d05ddbde),FXULONG(0x6bcdf07a423aa96b),
  FXULONG(0x86c16c98d2c953c6),FXULONG(0xe871c7bf077ba8b7),FXULONG(0x11471cd764ad4972),FXULONG(0xd598e40d3dd89bcf),
  FXULONG(0x4aff1d108d4ec2c3),FXULONG(0xcedf722a585139ba),FXULONG(0xc2974eb4ee658828),FXULONG(0x733d226229feea32),
  FXULONG(0x0806357d5a3f525f),FXULONG(0xca07c2dcb0cf26f7),FXULONG(0xfc89b393dd02f0b5),FXULONG(0xbbac2078d443ace2),
  FXULONG(0xd54b944b84aa4c0d),FXULONG(0x0a9e795e65d4df11),FXULONG(0x4d4617b5ff4a16d5),FXULONG(0x504bced1bf8e4e45),
  FXULONG(0xe45ec2862f71e1d6),FXULONG(0x5d767327bb4e5a4c),FXULONG(0x3a6a07f8d510f86f),FXULONG(0x890489f70a55368b),
  FXULONG(0x2b45ac74ccea842e),FXULONG(0x3b0b8bc90012929d),FXULONG(0x09ce6ebb40173744),FXULONG(0xcc420a6a101d0515),
  FXULONG(0x9fa946824a12232d),FXULONG(0x47939822dc96abf9),FXULONG(0x59787e2b93bc56f7),FXULONG(0x57eb4edb3c55b65a),
  FXULONG(0xede622920b6b23f1),FXULONG(0xe95fab368e45eced),FXULONG(0x11dbcb0218ebb414),FXULONG(0xd652bdc29f26a119),
  FXULONG(0x4be76d3346f0495f),FXULONG(0x6f70a4400c562ddb),FXULONG(0xcb4ccd500f6bb952),FXULONG(0x7e2000a41346a7a7),
  FXULONG(0x8ed400668c0c28c8),FXULONG(0x728900802f0f32fa),FXULONG(0x4f2b40a03ad2ffb9),FXULONG(0xe2f610c84987bfa8),
  FXULONG(0x0dd9ca7d2df4d7c9),FXULONG(0x91503d1c79720dbb),FXULONG(0x75a44c6397ce912a),FXULONG(0xc986afbe3ee11aba),
  FXULONG(0xfbe85badce996168),FXULONG(0xfae27299423fb9c3),FXULONG(0xdccd879fc967d41a),FXULONG(0x5400e987bbc1c920),
  FXULONG(0x290123e9aab23b68),FXULONG(0xf9a0b6720aaf6521),FXULONG(0xf808e40e8d5b3e69),FXULONG(0xb60b1d1230b20e04),
  FXULONG(0xb1c6f22b5e6f48c2),FXULONG(0x1e38aeb6360b1af3),FXULONG(0x25c6da63c38de1b0),FXULONG(0x579c487e5a38ad0e),
  FXULONG(0x2d835a9df0c6d851),FXULONG(0xf8e431456cf88e65),FXULONG(0x1b8e9ecb641b58ff),FXULONG(0xe272467e3d222f3f),
  FXULONG(0x5b0ed81dcc6abb0f),FXULONG(0x98e947129fc2b4e9),FXULONG(0x3f2398d747b36224),FXULONG(0x8eec7f0d19a03aad),
  FXULONG(0x1953cf68300424ac),FXULONG(0x5fa8c3423c052dd7),FXULONG(0x3792f412cb06794d),FXULONG(0xe2bbd88bbee40bd0),
  FXULONG(0x5b6aceaeae9d0ec4),FXULONG(0xf245825a5a445275),FXULONG(0xeed6e2f0f0d56712),FXULONG(0x55464dd69685606b),
  FXULONG(0xaa97e14c3c26b886),FXULONG(0xd53dd99f4b3066a8),FXULONG(0xe546a8038efe4029),FXULONG(0xde98520472bdd033),
  FXULONG(0x963e66858f6d4440),FXULONG(0xdde7001379a44aa8),FXULONG(0x5560c018580d5d52),FXULONG(0xaab8f01e6e10b4a6),
  FXULONG(0xcab3961304ca70e8),FXULONG(0x3d607b97c5fd0d22),FXULONG(0x8cb89a7db77c506a),FXULONG(0x77f3608e92adb242),
  FXULONG(0x55f038b237591ed3),FXULONG(0x6b6c46dec52f6688),FXULONG(0x2323ac4b3b3da015),FXULONG(0xabec975e0a0d081a),
  FXULONG(0x96e7bd358c904a21),FXULONG(0x7e50d64177da2e54),FXULONG(0xdde50bd1d5d0b9e9),FXULONG(0x955e4ec64b44e864),
  FXULONG(0xbd5af13bef0b113e),FXULONG(0xecb1ad8aeacdd58e),FXULONG(0x67de18eda5814af2),FXULONG(0x80eacf948770ced7),
  FXULONG(0xa1258379a94d028d),FXULONG(0x096ee45813a04330),FXULONG(0x8bca9d6e188853fc),FXULONG(0x775ea264cf55347e),
  FXULONG(0x95364afe032a819e),FXULONG(0x3a83ddbd83f52205),FXULONG(0xc4926a9672793543),FXULONG(0x75b7053c0f178294),
  FXULONG(0x5324c68b12dd6339),FXULONG(0xd3f6fc16ebca5e04),FXULONG(0x88f4bb1ca6bcf585),FXULONG(0x2b31e9e3d06c32e6),
  FXULONG(0x3aff322e62439fd0),FXULONG(0x09befeb9fad487c3),FXULONG(0x4c2ebe687989a9b4),FXULONG(0x0f9d37014bf60a11),
  FXULONG(0x538484c19ef38c95),FXULONG(0x2865a5f206b06fba),FXULONG(0xf93f87b7442e45d4),FXULONG(0xf78f69a51539d749),
  FXULONG(0xb573440e5a884d1c),FXULONG(0x31680a88f8953031),FXULONG(0xfdc20d2b36ba7c3e),FXULONG(0x3d32907604691b4d),
  FXULONG(0xa63f9a49c2c1b110),FXULONG(0x0fcf80dc33721d54),FXULONG(0xd3c36113404ea4a9),FXULONG(0x645a1cac083126ea),
  FXULONG(0x3d70a3d70a3d70a4),FXULONG(0xcccccccccccccccd),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),
  FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),
  FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),
  FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),
  FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),
  FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),
  FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),
  FXULONG(0x0000000000000000),FXULONG(0x0000000000000000),FXULONG(0x4000000000000000),FXULONG(0x5000000000000000),
  FXULONG(0xa400000000000000),FXULONG(0x4d00000000000000),FXULONG(0xf020000000000000),FXULONG(0x6c28000000000000),
  FXULONG(0xc732000000000000),FXULONG(0x3c7f400000000000),FXULONG(0x4b9f100000000000),FXULONG(0x1e86d40000000000),
  FXULONG(0x1314448000000000),FXULONG(0x17d955a000000000),FXULONG(0x5dcfab0800000000),FXULONG(0x5aa1cae500000000),
  FXULONG(0xf14a3d9e40000000),FXULONG(0x6d9ccd05d0000000),FXULONG(0xe4820023a2000000),FXULONG(0xdda2802c8a800000),
  FXULONG(0xd50b2037ad200000),FXULONG(0x4526f422cc340000),FXULONG(0x9670b12b7f410000),FXULONG(0x3c0cdd765f114000),
  FXULONG(0xa5880a69fb6ac800),FXULONG(0x8eea0d047a457a00),FXULONG(0x72a4904598d6d880),FXULONG(0x47a6da2b7f864750),
  FXULONG(0x999090b65f67d924),FXULONG(0xfff4b4e3f741cf6d),FXULONG(0xbff8f10e7a8921a4),FXULONG(0xaff72d52192b6a0d),
  FXULONG(0x9bf4f8a69f764490),FXULONG(0x02f236d04753d5b4),FXULONG(0x01d762422c946590),FXULONG(0x424d3ad2b7b97ef5),
  FXULONG(0xd2e0898765a7deb2),FXULONG(0x63cc55f49f88eb2f),FXULONG(0x3cbf6b71c76b25fb),FXULONG(0x8bef464e3945ef7a),
  FXULONG(0x97758bf0e3cbb5ac),FXULONG(0x3d52eeed1cbea317),FXULONG(0x4ca7aaa863ee4bdd),FXULONG(0x8fe8caa93e74ef6a),
  FXULONG(0xb3e2fd538e122b44),FXULONG(0x60dbbca87196b616),FXULONG(0xbc8955e946fe31cd),FXULONG(0x6babab6398bdbe41),
  FXULONG(0xc696963c7eed2dd1),FXULONG(0xfc1e1de5cf543ca2),FXULONG(0x3b25a55f43294bcb),FXULONG(0x49ef0eb713f39ebe),
  FXULONG(0x6e3569326c784337),FXULONG(0x49c2c37f07965404),FXULONG(0xdc33745ec97be906),FXULONG(0x69a028bb3ded71a3),
  FXULONG(0xc40832ea0d68ce0c),FXULONG(0xf50a3fa490c30190),FXULONG(0x792667c6da79e0fa),FXULONG(0x577001b891185938),
  FXULONG(0xed4c0226b55e6f86),FXULONG(0x544f8158315b05b4),FXULONG(0x696361ae3db1c721),FXULONG(0x03bc3a19cd1e38e9),
  FXULONG(0x04ab48a04065c723),FXULONG(0x62eb0d64283f9c76),FXULONG(0x3ba5d0bd324f8394),FXULONG(0xca8f44ec7ee36479),
  FXULONG(0x7e998b13cf4e1ecb),FXULONG(0x9e3fedd8c321a67e),FXULONG(0xc5cfe94ef3ea101e),FXULONG(0xbba1f1d158724a12),
  FXULONG(0x2a8a6e45ae8edc97),FXULONG(0xf52d09d71a3293bd),FXULONG(0x593c2626705f9c56),FXULONG(0x6f8b2fb00c77836c),
  FXULONG(0x0b6dfb9c0f956447),FXULONG(0x4724bd4189bd5eac),FXULONG(0x58edec91ec2cb657),FXULONG(0x2f2967b66737e3ed),
  FXULONG(0xbd79e0d20082ee74),FXULONG(0xecd8590680a3aa11),FXULONG(0xe80e6f4820cc9495),FXULONG(0x3109058d147fdcdd),
  FXULONG(0xbd4b46f0599fd415),FXULONG(0x6c9e18ac7007c91a),FXULONG(0x03e2cf6bc604ddb0),FXULONG(0x84db8346b786151c),
  FXULONG(0xe612641865679a63),FXULONG(0x4fcb7e8f3f60c07e),FXULONG(0xe3be5e330f38f09d),FXULONG(0x5cadf5bfd3072cc5),
  FXULONG(0x73d9732fc7c8f7f6),FXULONG(0x2867e7fddcdd9afa),FXULONG(0xb281e1fd541501b8),FXULONG(0x1f225a7ca91a4226),
  FXULONG(0x3375788de9b06958),FXULONG(0x0052d6b1641c83ae),FXULONG(0xc0678c5dbd23a49a),FXULONG(0xf840b7ba963646e0),
  FXULONG(0xb650e5a93bc3d898),FXULONG(0xa3e51f138ab4cebe),FXULONG(0xc66f336c36b10137),FXULONG(0xb80b0047445d4184),
  FXULONG(0xa60dc059157491e5),FXULONG(0x87c89837ad68db2f),FXULONG(0x29babe4598c311fb),FXULONG(0xf4296dd6fef3d67a),
  FXULONG(0x1899e4a65f58660c),FXULONG(0x5ec05dcff72e7f8f),FXULONG(0x76707543f4fa1f73),FXULONG(0x6a06494a791c53a8),
  FXULONG(0x0487db9d17636892),FXULONG(0x45a9d2845d3c42b6),FXULONG(0x0b8a2392ba45a9b2),FXULONG(0x8e6cac7768d7141e),
  FXULONG(0x3207d795430cd926),FXULONG(0x7f44e6bd49e807b8),FXULONG(0x5f16206c9c6209a6),FXULONG(0x36dba887c37a8c0f),
  FXULONG(0xc2494954da2c9789),FXULONG(0xf2db9baa10b7bd6c),FXULONG(0x6f92829494e5acc7),FXULONG(0xcb772339ba1f17f9),
  FXULONG(0xff2a760414536efb),FXULONG(0xfef5138519684aba),FXULONG(0x7eb258665fc25d69),FXULONG(0xef2f773ffbd97a61),
  FXULONG(0xaafb550ffacfd8fa),FXULONG(0x95ba2a53f983cf38),FXULONG(0xdd945a747bf26183),FXULONG(0x94f971119aeef9e4),
  FXULONG(0x7a37cd5601aab85d),FXULONG(0xac62e055c10ab33a),FXULONG(0x577b986b314d6009),FXULONG(0xed5a7e85fda0b80b),
  FXULONG(0x14588f13be847307),FXULONG(0x596eb2d8ae258fc8),FXULONG(0x6fca5f8ed9aef3bb),FXULONG(0x25de7bb9480d5854),
  FXULONG(0xaf561aa79a10ae6a),FXULONG(0x1b2ba1518094da04),FXULONG(0x90fb44d2f05d0842),FXULONG(0x353a1607ac744a53),
  FXULONG(0x42889b8997915ce8),FXULONG(0x69956135febada11),FXULONG(0x43fab9837e699095),FXULONG(0x94f967e45e03f4bb),
  FXULONG(0x1d1be0eebac278f5),FXULONG(0x6462d92a69731732),FXULONG(0x7d7b8f7503cfdcfe),FXULONG(0x5cda735244c3d43e),
  FXULONG(0x3a0888136afa64a7),FXULONG(0x088aaa1845b8fdd0),FXULONG(0x8aad549e57273d45),FXULONG(0x36ac54e2f678864b),
  FXULONG(0x84576a1bb416a7dd),FXULONG(0x656d44a2a11c51d5),FXULONG(0x9f644ae5a4b1b325),FXULONG(0x873d5d9f0dde1fee),
  FXULONG(0xa90cb506d155a7ea),FXULONG(0x09a7f12442d588f2),FXULONG(0x0c11ed6d538aeb2f),FXULONG(0x8f1668c8a86da5fa),
  FXULONG(0xf96e017d694487bc),FXULONG(0x37c981dcc395a9ac),FXULONG(0x85bbe253f47b1417),FXULONG(0x93956d7478ccec8e),
  FXULONG(0x387ac8d1970027b2),FXULONG(0x06997b05fcc0319e),FXULONG(0x441fece3bdf81f03),FXULONG(0xd527e81cad7626c3),
  FXULONG(0x8a71e223d8d3b074),FXULONG(0xf6872d5667844e49),FXULONG(0xb428f8ac016561db),FXULONG(0xe13336d701beba52),
  FXULONG(0xecc0024661173473),FXULONG(0x27f002d7f95d0190),FXULONG(0x31ec038df7b441f4),FXULONG(0x7e67047175a15271),
  FXULONG(0x0f0062c6e984d386),FXULONG(0x52c07b78a3e60868),FXULONG(0xa7709a56ccdf8a82),FXULONG(0x88a66076400bb691),
  FXULONG(0x6acff893d00ea435),FXULONG(0x0583f6b8c4124d43),FXULONG(0xc3727a337a8b704a),FXULONG(0x744f18c0592e4c5c),
  FXULONG(0x1162def06f79df73),FXULONG(0x8addcb5645ac2ba8),FXULONG(0x6d953e2bd7173692),FXULONG(0xc8fa8db6ccdd0437),
  FXULONG(0x1d9c9892400a22a2),FXULONG(0x2503beb6d00cab4b),FXULONG(0x2e44ae64840fd61d),FXULONG(0x5ceaecfed289e5d2),
  FXULONG(0x7425a83e872c5f47),FXULONG(0xd12f124e28f77719),FXULONG(0x82bd6b70d99aaa6f),FXULONG(0x636cc64d1001550b),
  FXULONG(0x3c47f7e05401aa4e),FXULONG(0x65acfaec34810a71),FXULONG(0x7f1839a741a14d0d),FXULONG(0x1ede48111209a050),
  FXULONG(0x934aed0aab460432),FXULONG(0xf81da84d5617853f),FXULONG(0x36251260ab9d668e),FXULONG(0xc1d72b7c6b426019),
  FXULONG(0xb24cf65b8612f81f),FXULONG(0xdee033f26797b627),FXULONG(0x169840ef017da3b1),FXULONG(0x8e1f289560ee864e),
  FXULONG(0xf1a6f2bab92a27e2),FXULONG(0xae10af696774b1db),FXULONG(0xacca6da1e0a8ef29),FXULONG(0x17fd090a58d32af3),
  FXULONG(0xddfc4b4cef07f5b0),FXULONG(0x4abdaf101564f98e),FXULONG(0x9d6d1ad41abe37f1),FXULONG(0x84c86189216dc5ed),
  FXULONG(0x32fd3cf5b4e49bb4),FXULONG(0x3fbc8c33221dc2a1),FXULONG(0x0fabaf3feaa5334a),FXULONG(0x29cb4d87f2a7400e),
  FXULONG(0x743e20e9ef511012),FXULONG(0x914da9246b255416),FXULONG(0x1ad089b6c2f7548e),FXULONG(0xa184ac2473b529b1),
  FXULONG(0xc9e5d72d90a2741e),FXULONG(0x7e2fa67c7a658892),FXULONG(0xddbb901b98feeab7),FXULONG(0x552a74227f3ea565),
  FXULONG(0xd53a88958f87275f),FXULONG(0x8a892abaf368f137),FXULONG(0x2d2b7569b0432d85),FXULONG(0x9c3b29620e29fc73),
  FXULONG(0x8349f3ba91b47b8f),FXULONG(0x241c70a936219a73),FXULONG(0xed238cd383aa0110),FXULONG(0xf4363804324a40aa),
  FXULONG(0xb143c6053edcd0d5),FXULONG(0xdd94b7868e94050a),FXULONG(0xca7cf2b4191c8326),FXULONG(0xfd1c2f611f63a3f0),
  FXULONG(0xbc633b39673c8cec),FXULONG(0xd5be0503e085d813),FXULONG(0x4b2d8644d8a74e18),FXULONG(0xddf8e7d60ed1219e),
  FXULONG(0xcabb90e5c942b503),FXULONG(0x3d6a751f3b936243),FXULONG(0x0cc512670a783ad4),FXULONG(0x27fb2b80668b24c5),
  FXULONG(0xb1f9f660802dedf6),FXULONG(0x5e7873f8a0396973),FXULONG(0xdb0b487b6423e1e8),FXULONG(0x91ce1a9a3d2cda62),
  FXULONG(0x7641a140cc7810fb),FXULONG(0xa9e904c87fcb0a9d),FXULONG(0x546345fa9fbdcd44),FXULONG(0xa97c177947ad4095),
  FXULONG(0x49ed8eabcccc485d),FXULONG(0x5c68f256bfff5a74),FXULONG(0x73832eec6fff3111),FXULONG(0xc831fd53c5ff7eab),
  FXULONG(0xba3e7ca8b77f5e55),FXULONG(0x28ce1bd2e55f35eb),FXULONG(0x7980d163cf5b81b3),FXULONG(0xd7e105bcc332621f),
  FXULONG(0x8dd9472bf3fefaa7),FXULONG(0xb14f98f6f0feb951),FXULONG(0x6ed1bf9a569f33d3),FXULONG(0x0a862f80ec4700c8),
  FXULONG(0xcd27bb612758c0fa),FXULONG(0x8038d51cb897789c),FXULONG(0xe0470a63e6bd56c3),FXULONG(0x1858ccfce06cac74),
  FXULONG(0x0f37801e0c43ebc8),FXULONG(0xd30560258f54e6ba),FXULONG(0x47c6b82ef32a2069),FXULONG(0x4cdc331d57fa5441),
  FXULONG(0xe0133fe4adf8e952),FXULONG(0x58180fddd97723a6),FXULONG(0x570f09eaa7ea7648),
  };


// Powers of five, upper 64-bits
static const FXulong hifive[]={
  FXULONG(0xeef453d6923bd65a),FXULONG(0x9558b4661b6565f8),FXULONG(0xbaaee17fa23ebf76),FXULONG(0xe95a99df8ace6f53),
  FXULONG(0x91d8a02bb6c10594),FXULONG(0xb64ec836a47146f9),FXULONG(0xe3e27a444d8d98b7),FXULONG(0x8e6d8c6ab0787f72),
  FXULONG(0xb208ef855c969f4f),FXULONG(0xde8b2b66b3bc4723),FXULONG(0x8b16fb203055ac76),FXULONG(0xaddcb9e83c6b1793),
  FXULONG(0xd953e8624b85dd78),FXULONG(0x87d4713d6f33aa6b),FXULONG(0xa9c98d8ccb009506),FXULONG(0xd43bf0effdc0ba48),
  FXULONG(0x84a57695fe98746d),FXULONG(0xa5ced43b7e3e9188),FXULONG(0xcf42894a5dce35ea),FXULONG(0x818995ce7aa0e1b2),
  FXULONG(0xa1ebfb4219491a1f),FXULONG(0xca66fa129f9b60a6),FXULONG(0xfd00b897478238d0),FXULONG(0x9e20735e8cb16382),
  FXULONG(0xc5a890362fddbc62),FXULONG(0xf712b443bbd52b7b),FXULONG(0x9a6bb0aa55653b2d),FXULONG(0xc1069cd4eabe89f8),
  FXULONG(0xf148440a256e2c76),FXULONG(0x96cd2a865764dbca),FXULONG(0xbc807527ed3e12bc),FXULONG(0xeba09271e88d976b),
  FXULONG(0x93445b8731587ea3),FXULONG(0xb8157268fdae9e4c),FXULONG(0xe61acf033d1a45df),FXULONG(0x8fd0c16206306bab),
  FXULONG(0xb3c4f1ba87bc8696),FXULONG(0xe0b62e2929aba83c),FXULONG(0x8c71dcd9ba0b4925),FXULONG(0xaf8e5410288e1b6f),
  FXULONG(0xdb71e91432b1a24a),FXULONG(0x892731ac9faf056e),FXULONG(0xab70fe17c79ac6ca),FXULONG(0xd64d3d9db981787d),
  FXULONG(0x85f0468293f0eb4e),FXULONG(0xa76c582338ed2621),FXULONG(0xd1476e2c07286faa),FXULONG(0x82cca4db847945ca),
  FXULONG(0xa37fce126597973c),FXULONG(0xcc5fc196fefd7d0c),FXULONG(0xff77b1fcbebcdc4f),FXULONG(0x9faacf3df73609b1),
  FXULONG(0xc795830d75038c1d),FXULONG(0xf97ae3d0d2446f25),FXULONG(0x9becce62836ac577),FXULONG(0xc2e801fb244576d5),
  FXULONG(0xf3a20279ed56d48a),FXULONG(0x9845418c345644d6),FXULONG(0xbe5691ef416bd60c),FXULONG(0xedec366b11c6cb8f),
  FXULONG(0x94b3a202eb1c3f39),FXULONG(0xb9e08a83a5e34f07),FXULONG(0xe858ad248f5c22c9),FXULONG(0x91376c36d99995be),
  FXULONG(0xb58547448ffffb2d),FXULONG(0xe2e69915b3fff9f9),FXULONG(0x8dd01fad907ffc3b),FXULONG(0xb1442798f49ffb4a),
  FXULONG(0xdd95317f31c7fa1d),FXULONG(0x8a7d3eef7f1cfc52),FXULONG(0xad1c8eab5ee43b66),FXULONG(0xd863b256369d4a40),
  FXULONG(0x873e4f75e2224e68),FXULONG(0xa90de3535aaae202),FXULONG(0xd3515c2831559a83),FXULONG(0x8412d9991ed58091),
  FXULONG(0xa5178fff668ae0b6),FXULONG(0xce5d73ff402d98e3),FXULONG(0x80fa687f881c7f8e),FXULONG(0xa139029f6a239f72),
  FXULONG(0xc987434744ac874e),FXULONG(0xfbe9141915d7a922),FXULONG(0x9d71ac8fada6c9b5),FXULONG(0xc4ce17b399107c22),
  FXULONG(0xf6019da07f549b2b),FXULONG(0x99c102844f94e0fb),FXULONG(0xc0314325637a1939),FXULONG(0xf03d93eebc589f88),
  FXULONG(0x96267c7535b763b5),FXULONG(0xbbb01b9283253ca2),FXULONG(0xea9c227723ee8bcb),FXULONG(0x92a1958a7675175f),
  FXULONG(0xb749faed14125d36),FXULONG(0xe51c79a85916f484),FXULONG(0x8f31cc0937ae58d2),FXULONG(0xb2fe3f0b8599ef07),
  FXULONG(0xdfbdcece67006ac9),FXULONG(0x8bd6a141006042bd),FXULONG(0xaecc49914078536d),FXULONG(0xda7f5bf590966848),
  FXULONG(0x888f99797a5e012d),FXULONG(0xaab37fd7d8f58178),FXULONG(0xd5605fcdcf32e1d6),FXULONG(0x855c3be0a17fcd26),
  FXULONG(0xa6b34ad8c9dfc06f),FXULONG(0xd0601d8efc57b08b),FXULONG(0x823c12795db6ce57),FXULONG(0xa2cb1717b52481ed),
  FXULONG(0xcb7ddcdda26da268),FXULONG(0xfe5d54150b090b02),FXULONG(0x9efa548d26e5a6e1),FXULONG(0xc6b8e9b0709f109a),
  FXULONG(0xf867241c8cc6d4c0),FXULONG(0x9b407691d7fc44f8),FXULONG(0xc21094364dfb5636),FXULONG(0xf294b943e17a2bc4),
  FXULONG(0x979cf3ca6cec5b5a),FXULONG(0xbd8430bd08277231),FXULONG(0xece53cec4a314ebd),FXULONG(0x940f4613ae5ed136),
  FXULONG(0xb913179899f68584),FXULONG(0xe757dd7ec07426e5),FXULONG(0x9096ea6f3848984f),FXULONG(0xb4bca50b065abe63),
  FXULONG(0xe1ebce4dc7f16dfb),FXULONG(0x8d3360f09cf6e4bd),FXULONG(0xb080392cc4349dec),FXULONG(0xdca04777f541c567),
  FXULONG(0x89e42caaf9491b60),FXULONG(0xac5d37d5b79b6239),FXULONG(0xd77485cb25823ac7),FXULONG(0x86a8d39ef77164bc),
  FXULONG(0xa8530886b54dbdeb),FXULONG(0xd267caa862a12d66),FXULONG(0x8380dea93da4bc60),FXULONG(0xa46116538d0deb78),
  FXULONG(0xcd795be870516656),FXULONG(0x806bd9714632dff6),FXULONG(0xa086cfcd97bf97f3),FXULONG(0xc8a883c0fdaf7df0),
  FXULONG(0xfad2a4b13d1b5d6c),FXULONG(0x9cc3a6eec6311a63),FXULONG(0xc3f490aa77bd60fc),FXULONG(0xf4f1b4d515acb93b),
  FXULONG(0x991711052d8bf3c5),FXULONG(0xbf5cd54678eef0b6),FXULONG(0xef340a98172aace4),FXULONG(0x9580869f0e7aac0e),
  FXULONG(0xbae0a846d2195712),FXULONG(0xe998d258869facd7),FXULONG(0x91ff83775423cc06),FXULONG(0xb67f6455292cbf08),
  FXULONG(0xe41f3d6a7377eeca),FXULONG(0x8e938662882af53e),FXULONG(0xb23867fb2a35b28d),FXULONG(0xdec681f9f4c31f31),
  FXULONG(0x8b3c113c38f9f37e),FXULONG(0xae0b158b4738705e),FXULONG(0xd98ddaee19068c76),FXULONG(0x87f8a8d4cfa417c9),
  FXULONG(0xa9f6d30a038d1dbc),FXULONG(0xd47487cc8470652b),FXULONG(0x84c8d4dfd2c63f3b),FXULONG(0xa5fb0a17c777cf09),
  FXULONG(0xcf79cc9db955c2cc),FXULONG(0x81ac1fe293d599bf),FXULONG(0xa21727db38cb002f),FXULONG(0xca9cf1d206fdc03b),
  FXULONG(0xfd442e4688bd304a),FXULONG(0x9e4a9cec15763e2e),FXULONG(0xc5dd44271ad3cdba),FXULONG(0xf7549530e188c128),
  FXULONG(0x9a94dd3e8cf578b9),FXULONG(0xc13a148e3032d6e7),FXULONG(0xf18899b1bc3f8ca1),FXULONG(0x96f5600f15a7b7e5),
  FXULONG(0xbcb2b812db11a5de),FXULONG(0xebdf661791d60f56),FXULONG(0x936b9fcebb25c995),FXULONG(0xb84687c269ef3bfb),
  FXULONG(0xe65829b3046b0afa),FXULONG(0x8ff71a0fe2c2e6dc),FXULONG(0xb3f4e093db73a093),FXULONG(0xe0f218b8d25088b8),
  FXULONG(0x8c974f7383725573),FXULONG(0xafbd2350644eeacf),FXULONG(0xdbac6c247d62a583),FXULONG(0x894bc396ce5da772),
  FXULONG(0xab9eb47c81f5114f),FXULONG(0xd686619ba27255a2),FXULONG(0x8613fd0145877585),FXULONG(0xa798fc4196e952e7),
  FXULONG(0xd17f3b51fca3a7a0),FXULONG(0x82ef85133de648c4),FXULONG(0xa3ab66580d5fdaf5),FXULONG(0xcc963fee10b7d1b3),
  FXULONG(0xffbbcfe994e5c61f),FXULONG(0x9fd561f1fd0f9bd3),FXULONG(0xc7caba6e7c5382c8),FXULONG(0xf9bd690a1b68637b),
  FXULONG(0x9c1661a651213e2d),FXULONG(0xc31bfa0fe5698db8),FXULONG(0xf3e2f893dec3f126),FXULONG(0x986ddb5c6b3a76b7),
  FXULONG(0xbe89523386091465),FXULONG(0xee2ba6c0678b597f),FXULONG(0x94db483840b717ef),FXULONG(0xba121a4650e4ddeb),
  FXULONG(0xe896a0d7e51e1566),FXULONG(0x915e2486ef32cd60),FXULONG(0xb5b5ada8aaff80b8),FXULONG(0xe3231912d5bf60e6),
  FXULONG(0x8df5efabc5979c8f),FXULONG(0xb1736b96b6fd83b3),FXULONG(0xddd0467c64bce4a0),FXULONG(0x8aa22c0dbef60ee4),
  FXULONG(0xad4ab7112eb3929d),FXULONG(0xd89d64d57a607744),FXULONG(0x87625f056c7c4a8b),FXULONG(0xa93af6c6c79b5d2d),
  FXULONG(0xd389b47879823479),FXULONG(0x843610cb4bf160cb),FXULONG(0xa54394fe1eedb8fe),FXULONG(0xce947a3da6a9273e),
  FXULONG(0x811ccc668829b887),FXULONG(0xa163ff802a3426a8),FXULONG(0xc9bcff6034c13052),FXULONG(0xfc2c3f3841f17c67),
  FXULONG(0x9d9ba7832936edc0),FXULONG(0xc5029163f384a931),FXULONG(0xf64335bcf065d37d),FXULONG(0x99ea0196163fa42e),
  FXULONG(0xc06481fb9bcf8d39),FXULONG(0xf07da27a82c37088),FXULONG(0x964e858c91ba2655),FXULONG(0xbbe226efb628afea),
  FXULONG(0xeadab0aba3b2dbe5),FXULONG(0x92c8ae6b464fc96f),FXULONG(0xb77ada0617e3bbcb),FXULONG(0xe55990879ddcaabd),
  FXULONG(0x8f57fa54c2a9eab6),FXULONG(0xb32df8e9f3546564),FXULONG(0xdff9772470297ebd),FXULONG(0x8bfbea76c619ef36),
  FXULONG(0xaefae51477a06b03),FXULONG(0xdab99e59958885c4),FXULONG(0x88b402f7fd75539b),FXULONG(0xaae103b5fcd2a881),
  FXULONG(0xd59944a37c0752a2),FXULONG(0x857fcae62d8493a5),FXULONG(0xa6dfbd9fb8e5b88e),FXULONG(0xd097ad07a71f26b2),
  FXULONG(0x825ecc24c873782f),FXULONG(0xa2f67f2dfa90563b),FXULONG(0xcbb41ef979346bca),FXULONG(0xfea126b7d78186bc),
  FXULONG(0x9f24b832e6b0f436),FXULONG(0xc6ede63fa05d3143),FXULONG(0xf8a95fcf88747d94),FXULONG(0x9b69dbe1b548ce7c),
  FXULONG(0xc24452da229b021b),FXULONG(0xf2d56790ab41c2a2),FXULONG(0x97c560ba6b0919a5),FXULONG(0xbdb6b8e905cb600f),
  FXULONG(0xed246723473e3813),FXULONG(0x9436c0760c86e30b),FXULONG(0xb94470938fa89bce),FXULONG(0xe7958cb87392c2c2),
  FXULONG(0x90bd77f3483bb9b9),FXULONG(0xb4ecd5f01a4aa828),FXULONG(0xe2280b6c20dd5232),FXULONG(0x8d590723948a535f),
  FXULONG(0xb0af48ec79ace837),FXULONG(0xdcdb1b2798182244),FXULONG(0x8a08f0f8bf0f156b),FXULONG(0xac8b2d36eed2dac5),
  FXULONG(0xd7adf884aa879177),FXULONG(0x86ccbb52ea94baea),FXULONG(0xa87fea27a539e9a5),FXULONG(0xd29fe4b18e88640e),
  FXULONG(0x83a3eeeef9153e89),FXULONG(0xa48ceaaab75a8e2b),FXULONG(0xcdb02555653131b6),FXULONG(0x808e17555f3ebf11),
  FXULONG(0xa0b19d2ab70e6ed6),FXULONG(0xc8de047564d20a8b),FXULONG(0xfb158592be068d2e),FXULONG(0x9ced737bb6c4183d),
  FXULONG(0xc428d05aa4751e4c),FXULONG(0xf53304714d9265df),FXULONG(0x993fe2c6d07b7fab),FXULONG(0xbf8fdb78849a5f96),
  FXULONG(0xef73d256a5c0f77c),FXULONG(0x95a8637627989aad),FXULONG(0xbb127c53b17ec159),FXULONG(0xe9d71b689dde71af),
  FXULONG(0x9226712162ab070d),FXULONG(0xb6b00d69bb55c8d1),FXULONG(0xe45c10c42a2b3b05),FXULONG(0x8eb98a7a9a5b04e3),
  FXULONG(0xb267ed1940f1c61c),FXULONG(0xdf01e85f912e37a3),FXULONG(0x8b61313bbabce2c6),FXULONG(0xae397d8aa96c1b77),
  FXULONG(0xd9c7dced53c72255),FXULONG(0x881cea14545c7575),FXULONG(0xaa242499697392d2),FXULONG(0xd4ad2dbfc3d07787),
  FXULONG(0x84ec3c97da624ab4),FXULONG(0xa6274bbdd0fadd61),FXULONG(0xcfb11ead453994ba),FXULONG(0x81ceb32c4b43fcf4),
  FXULONG(0xa2425ff75e14fc31),FXULONG(0xcad2f7f5359a3b3e),FXULONG(0xfd87b5f28300ca0d),FXULONG(0x9e74d1b791e07e48),
  FXULONG(0xc612062576589dda),FXULONG(0xf79687aed3eec551),FXULONG(0x9abe14cd44753b52),FXULONG(0xc16d9a0095928a27),
  FXULONG(0xf1c90080baf72cb1),FXULONG(0x971da05074da7bee),FXULONG(0xbce5086492111aea),FXULONG(0xec1e4a7db69561a5),
  FXULONG(0x9392ee8e921d5d07),FXULONG(0xb877aa3236a4b449),FXULONG(0xe69594bec44de15b),FXULONG(0x901d7cf73ab0acd9),
  FXULONG(0xb424dc35095cd80f),FXULONG(0xe12e13424bb40e13),FXULONG(0x8cbccc096f5088cb),FXULONG(0xafebff0bcb24aafe),
  FXULONG(0xdbe6fecebdedd5be),FXULONG(0x89705f4136b4a597),FXULONG(0xabcc77118461cefc),FXULONG(0xd6bf94d5e57a42bc),
  FXULONG(0x8637bd05af6c69b5),FXULONG(0xa7c5ac471b478423),FXULONG(0xd1b71758e219652b),FXULONG(0x83126e978d4fdf3b),
  FXULONG(0xa3d70a3d70a3d70a),FXULONG(0xcccccccccccccccc),FXULONG(0x8000000000000000),FXULONG(0xa000000000000000),
  FXULONG(0xc800000000000000),FXULONG(0xfa00000000000000),FXULONG(0x9c40000000000000),FXULONG(0xc350000000000000),
  FXULONG(0xf424000000000000),FXULONG(0x9896800000000000),FXULONG(0xbebc200000000000),FXULONG(0xee6b280000000000),
  FXULONG(0x9502f90000000000),FXULONG(0xba43b74000000000),FXULONG(0xe8d4a51000000000),FXULONG(0x9184e72a00000000),
  FXULONG(0xb5e620f480000000),FXULONG(0xe35fa931a0000000),FXULONG(0x8e1bc9bf04000000),FXULONG(0xb1a2bc2ec5000000),
  FXULONG(0xde0b6b3a76400000),FXULONG(0x8ac7230489e80000),FXULONG(0xad78ebc5ac620000),FXULONG(0xd8d726b7177a8000),
  FXULONG(0x878678326eac9000),FXULONG(0xa968163f0a57b400),FXULONG(0xd3c21bcecceda100),FXULONG(0x84595161401484a0),
  FXULONG(0xa56fa5b99019a5c8),FXULONG(0xcecb8f27f4200f3a),FXULONG(0x813f3978f8940984),FXULONG(0xa18f07d736b90be5),
  FXULONG(0xc9f2c9cd04674ede),FXULONG(0xfc6f7c4045812296),FXULONG(0x9dc5ada82b70b59d),FXULONG(0xc5371912364ce305),
  FXULONG(0xf684df56c3e01bc6),FXULONG(0x9a130b963a6c115c),FXULONG(0xc097ce7bc90715b3),FXULONG(0xf0bdc21abb48db20),
  FXULONG(0x96769950b50d88f4),FXULONG(0xbc143fa4e250eb31),FXULONG(0xeb194f8e1ae525fd),FXULONG(0x92efd1b8d0cf37be),
  FXULONG(0xb7abc627050305ad),FXULONG(0xe596b7b0c643c719),FXULONG(0x8f7e32ce7bea5c6f),FXULONG(0xb35dbf821ae4f38b),
  FXULONG(0xe0352f62a19e306e),FXULONG(0x8c213d9da502de45),FXULONG(0xaf298d050e4395d6),FXULONG(0xdaf3f04651d47b4c),
  FXULONG(0x88d8762bf324cd0f),FXULONG(0xab0e93b6efee0053),FXULONG(0xd5d238a4abe98068),FXULONG(0x85a36366eb71f041),
  FXULONG(0xa70c3c40a64e6c51),FXULONG(0xd0cf4b50cfe20765),FXULONG(0x82818f1281ed449f),FXULONG(0xa321f2d7226895c7),
  FXULONG(0xcbea6f8ceb02bb39),FXULONG(0xfee50b7025c36a08),FXULONG(0x9f4f2726179a2245),FXULONG(0xc722f0ef9d80aad6),
  FXULONG(0xf8ebad2b84e0d58b),FXULONG(0x9b934c3b330c8577),FXULONG(0xc2781f49ffcfa6d5),FXULONG(0xf316271c7fc3908a),
  FXULONG(0x97edd871cfda3a56),FXULONG(0xbde94e8e43d0c8ec),FXULONG(0xed63a231d4c4fb27),FXULONG(0x945e455f24fb1cf8),
  FXULONG(0xb975d6b6ee39e436),FXULONG(0xe7d34c64a9c85d44),FXULONG(0x90e40fbeea1d3a4a),FXULONG(0xb51d13aea4a488dd),
  FXULONG(0xe264589a4dcdab14),FXULONG(0x8d7eb76070a08aec),FXULONG(0xb0de65388cc8ada8),FXULONG(0xdd15fe86affad912),
  FXULONG(0x8a2dbf142dfcc7ab),FXULONG(0xacb92ed9397bf996),FXULONG(0xd7e77a8f87daf7fb),FXULONG(0x86f0ac99b4e8dafd),
  FXULONG(0xa8acd7c0222311bc),FXULONG(0xd2d80db02aabd62b),FXULONG(0x83c7088e1aab65db),FXULONG(0xa4b8cab1a1563f52),
  FXULONG(0xcde6fd5e09abcf26),FXULONG(0x80b05e5ac60b6178),FXULONG(0xa0dc75f1778e39d6),FXULONG(0xc913936dd571c84c),
  FXULONG(0xfb5878494ace3a5f),FXULONG(0x9d174b2dcec0e47b),FXULONG(0xc45d1df942711d9a),FXULONG(0xf5746577930d6500),
  FXULONG(0x9968bf6abbe85f20),FXULONG(0xbfc2ef456ae276e8),FXULONG(0xefb3ab16c59b14a2),FXULONG(0x95d04aee3b80ece5),
  FXULONG(0xbb445da9ca61281f),FXULONG(0xea1575143cf97226),FXULONG(0x924d692ca61be758),FXULONG(0xb6e0c377cfa2e12e),
  FXULONG(0xe498f455c38b997a),FXULONG(0x8edf98b59a373fec),FXULONG(0xb2977ee300c50fe7),FXULONG(0xdf3d5e9bc0f653e1),
  FXULONG(0x8b865b215899f46c),FXULONG(0xae67f1e9aec07187),FXULONG(0xda01ee641a708de9),FXULONG(0x884134fe908658b2),
  FXULONG(0xaa51823e34a7eede),FXULONG(0xd4e5e2cdc1d1ea96),FXULONG(0x850fadc09923329e),FXULONG(0xa6539930bf6bff45),
  FXULONG(0xcfe87f7cef46ff16),FXULONG(0x81f14fae158c5f6e),FXULONG(0xa26da3999aef7749),FXULONG(0xcb090c8001ab551c),
  FXULONG(0xfdcb4fa002162a63),FXULONG(0x9e9f11c4014dda7e),FXULONG(0xc646d63501a1511d),FXULONG(0xf7d88bc24209a565),
  FXULONG(0x9ae757596946075f),FXULONG(0xc1a12d2fc3978937),FXULONG(0xf209787bb47d6b84),FXULONG(0x9745eb4d50ce6332),
  FXULONG(0xbd176620a501fbff),FXULONG(0xec5d3fa8ce427aff),FXULONG(0x93ba47c980e98cdf),FXULONG(0xb8a8d9bbe123f017),
  FXULONG(0xe6d3102ad96cec1d),FXULONG(0x9043ea1ac7e41392),FXULONG(0xb454e4a179dd1877),FXULONG(0xe16a1dc9d8545e94),
  FXULONG(0x8ce2529e2734bb1d),FXULONG(0xb01ae745b101e9e4),FXULONG(0xdc21a1171d42645d),FXULONG(0x899504ae72497eba),
  FXULONG(0xabfa45da0edbde69),FXULONG(0xd6f8d7509292d603),FXULONG(0x865b86925b9bc5c2),FXULONG(0xa7f26836f282b732),
  FXULONG(0xd1ef0244af2364ff),FXULONG(0x8335616aed761f1f),FXULONG(0xa402b9c5a8d3a6e7),FXULONG(0xcd036837130890a1),
  FXULONG(0x802221226be55a64),FXULONG(0xa02aa96b06deb0fd),FXULONG(0xc83553c5c8965d3d),FXULONG(0xfa42a8b73abbf48c),
  FXULONG(0x9c69a97284b578d7),FXULONG(0xc38413cf25e2d70d),FXULONG(0xf46518c2ef5b8cd1),FXULONG(0x98bf2f79d5993802),
  FXULONG(0xbeeefb584aff8603),FXULONG(0xeeaaba2e5dbf6784),FXULONG(0x952ab45cfa97a0b2),FXULONG(0xba756174393d88df),
  FXULONG(0xe912b9d1478ceb17),FXULONG(0x91abb422ccb812ee),FXULONG(0xb616a12b7fe617aa),FXULONG(0xe39c49765fdf9d94),
  FXULONG(0x8e41ade9fbebc27d),FXULONG(0xb1d219647ae6b31c),FXULONG(0xde469fbd99a05fe3),FXULONG(0x8aec23d680043bee),
  FXULONG(0xada72ccc20054ae9),FXULONG(0xd910f7ff28069da4),FXULONG(0x87aa9aff79042286),FXULONG(0xa99541bf57452b28),
  FXULONG(0xd3fa922f2d1675f2),FXULONG(0x847c9b5d7c2e09b7),FXULONG(0xa59bc234db398c25),FXULONG(0xcf02b2c21207ef2e),
  FXULONG(0x8161afb94b44f57d),FXULONG(0xa1ba1ba79e1632dc),FXULONG(0xca28a291859bbf93),FXULONG(0xfcb2cb35e702af78),
  FXULONG(0x9defbf01b061adab),FXULONG(0xc56baec21c7a1916),FXULONG(0xf6c69a72a3989f5b),FXULONG(0x9a3c2087a63f6399),
  FXULONG(0xc0cb28a98fcf3c7f),FXULONG(0xf0fdf2d3f3c30b9f),FXULONG(0x969eb7c47859e743),FXULONG(0xbc4665b596706114),
  FXULONG(0xeb57ff22fc0c7959),FXULONG(0x9316ff75dd87cbd8),FXULONG(0xb7dcbf5354e9bece),FXULONG(0xe5d3ef282a242e81),
  FXULONG(0x8fa475791a569d10),FXULONG(0xb38d92d760ec4455),FXULONG(0xe070f78d3927556a),FXULONG(0x8c469ab843b89562),
  FXULONG(0xaf58416654a6babb),FXULONG(0xdb2e51bfe9d0696a),FXULONG(0x88fcf317f22241e2),FXULONG(0xab3c2fddeeaad25a),
  FXULONG(0xd60b3bd56a5586f1),FXULONG(0x85c7056562757456),FXULONG(0xa738c6bebb12d16c),FXULONG(0xd106f86e69d785c7),
  FXULONG(0x82a45b450226b39c),FXULONG(0xa34d721642b06084),FXULONG(0xcc20ce9bd35c78a5),FXULONG(0xff290242c83396ce),
  FXULONG(0x9f79a169bd203e41),FXULONG(0xc75809c42c684dd1),FXULONG(0xf92e0c3537826145),FXULONG(0x9bbcc7a142b17ccb),
  FXULONG(0xc2abf989935ddbfe),FXULONG(0xf356f7ebf83552fe),FXULONG(0x98165af37b2153de),FXULONG(0xbe1bf1b059e9a8d6),
  FXULONG(0xeda2ee1c7064130c),FXULONG(0x9485d4d1c63e8be7),FXULONG(0xb9a74a0637ce2ee1),FXULONG(0xe8111c87c5c1ba99),
  FXULONG(0x910ab1d4db9914a0),FXULONG(0xb54d5e4a127f59c8),FXULONG(0xe2a0b5dc971f303a),FXULONG(0x8da471a9de737e24),
  FXULONG(0xb10d8e1456105dad),FXULONG(0xdd50f1996b947518),FXULONG(0x8a5296ffe33cc92f),FXULONG(0xace73cbfdc0bfb7b),
  FXULONG(0xd8210befd30efa5a),FXULONG(0x8714a775e3e95c78),FXULONG(0xa8d9d1535ce3b396),FXULONG(0xd31045a8341ca07c),
  FXULONG(0x83ea2b892091e44d),FXULONG(0xa4e4b66b68b65d60),FXULONG(0xce1de40642e3f4b9),FXULONG(0x80d2ae83e9ce78f3),
  FXULONG(0xa1075a24e4421730),FXULONG(0xc94930ae1d529cfc),FXULONG(0xfb9b7cd9a4a7443c),FXULONG(0x9d412e0806e88aa5),
  FXULONG(0xc491798a08a2ad4e),FXULONG(0xf5b5d7ec8acb58a2),FXULONG(0x9991a6f3d6bf1765),FXULONG(0xbff610b0cc6edd3f),
  FXULONG(0xeff394dcff8a948e),FXULONG(0x95f83d0a1fb69cd9),FXULONG(0xbb764c4ca7a4440f),FXULONG(0xea53df5fd18d5513),
  FXULONG(0x92746b9be2f8552c),FXULONG(0xb7118682dbb66a77),FXULONG(0xe4d5e82392a40515),FXULONG(0x8f05b1163ba6832d),
  FXULONG(0xb2c71d5bca9023f8),FXULONG(0xdf78e4b2bd342cf6),FXULONG(0x8bab8eefb6409c1a),FXULONG(0xae9672aba3d0c320),
  FXULONG(0xda3c0f568cc4f3e8),FXULONG(0x8865899617fb1871),FXULONG(0xaa7eebfb9df9de8d),FXULONG(0xd51ea6fa85785631),
  FXULONG(0x8533285c936b35de),FXULONG(0xa67ff273b8460356),FXULONG(0xd01fef10a657842c),FXULONG(0x8213f56a67f6b29b),
  FXULONG(0xa298f2c501f45f42),FXULONG(0xcb3f2f7642717713),FXULONG(0xfe0efb53d30dd4d7),FXULONG(0x9ec95d1463e8a506),
  FXULONG(0xc67bb4597ce2ce48),FXULONG(0xf81aa16fdc1b81da),FXULONG(0x9b10a4e5e9913128),FXULONG(0xc1d4ce1f63f57d72),
  FXULONG(0xf24a01a73cf2dccf),FXULONG(0x976e41088617ca01),FXULONG(0xbd49d14aa79dbc82),FXULONG(0xec9c459d51852ba2),
  FXULONG(0x93e1ab8252f33b45),FXULONG(0xb8da1662e7b00a17),FXULONG(0xe7109bfba19c0c9d),FXULONG(0x906a617d450187e2),
  FXULONG(0xb484f9dc9641e9da),FXULONG(0xe1a63853bbd26451),FXULONG(0x8d07e33455637eb2),FXULONG(0xb049dc016abc5e5f),
  FXULONG(0xdc5c5301c56b75f7),FXULONG(0x89b9b3e11b6329ba),FXULONG(0xac2820d9623bf429),FXULONG(0xd732290fbacaf133),
  FXULONG(0x867f59a9d4bed6c0),FXULONG(0xa81f301449ee8c70),FXULONG(0xd226fc195c6a2f8c),FXULONG(0x83585d8fd9c25db7),
  FXULONG(0xa42e74f3d032f525),FXULONG(0xcd3a1230c43fb26f),FXULONG(0x80444b5e7aa7cf85),FXULONG(0xa0555e361951c366),
  FXULONG(0xc86ab5c39fa63440),FXULONG(0xfa856334878fc150),FXULONG(0x9c935e00d4b9d8d2),FXULONG(0xc3b8358109e84f07),
  FXULONG(0xf4a642e14c6262c8),FXULONG(0x98e7e9cccfbd7dbd),FXULONG(0xbf21e44003acdd2c),FXULONG(0xeeea5d5004981478),
  FXULONG(0x95527a5202df0ccb),FXULONG(0xbaa718e68396cffd),FXULONG(0xe950df20247c83fd),FXULONG(0x91d28b7416cdd27e),
  FXULONG(0xb6472e511c81471d),FXULONG(0xe3d8f9e563a198e5),FXULONG(0x8e679c2f5e44ff8f),
  };

/*******************************************************************************/

// Multiply 64bit x 64bit -> 128bit result

#if defined(_MSC_VER) && defined(_M_X64)
static inline void umul128(FXulong x,FXulong y,FXulong* hi,FXulong* lo){
  *lo=_umul128(x,y,hi);
  }
#elif defined(_MSC_VER) && defined(_M_ARM64)
static inline void umul128(FXulong x,FXulong y,FXulong* hi,FXulong* lo){
  *lo=x*y;
  *hi=__umulh(x,y);
  }
#elif defined(_MSC_VER) && defined(_M_IX86)
static inline void umul128(FXulong ab,FXulong cd,FXulong *hi,FXulong *lo){
  FXuint b=(FXuint)ab;
  FXuint a=(FXuint)(ab>>32);
  FXuint d=(FXuint)cd;
  FXuint c=(FXuint)(cd>>32);
  FXulong bd=__emulu(b,d);
  FXulong bc=__emulu(b,c);
  FXulong ad=__emulu(a,d);
  FXulong ac=__emulu(a,c);
  FXulong cy=(FXuint)(((bd>>32)+(FXuint)bc+(FXuint)ad)>>32);
  *lo=bd+(bc<<32)+(ad<<32);
  *hi=ac+(bc>>32)+(ad>>32)+cy;
  }
#elif defined(__GNUC__) && defined(__SIZEOF_INT128__)
static inline void umul128(FXulong x,FXulong y,FXulong* hi,FXulong* lo){
  __uint128_t r=(__uint128_t)x*y;
  *lo=(FXulong)r;
  *hi=(FXulong)(r>>64);
  }
#elif defined(__GNUC__) && defined(__x86_64__)
static inline void umul128(FXulong x,FXulong y,FXulong* hi,FXulong* lo){
  __asm__ (
      "movq  %2, %%rax;\n\t"
      "mulq %3;\n\t"
      "movq  %%rax, %0;\n\t"
      "movq  %%rdx, %1;\n\t"
      : "=m" (*lo), "=m" (*hi)
      : "r"(x), "r"(y)
      : "%rax", "%rdx");
  }
#else
static inline void umul128(FXulong x,FXulong y,FXulong *hi,FXulong *lo){
  FXulong xlo=(FXuint)x;
  FXulong xhi=(FXuint)(x>>32);
  FXulong ylo=(FXuint)y;
  FXulong yhi=(FXuint)(y>>32);
  FXulong p0=xlo*ylo;
  FXulong p1=xlo*yhi;
  FXulong p2=xhi*ylo;
  FXulong p3=xhi*yhi;
  FXulong cy=(FXuint)(((p0>>32)+(FXuint)p1+(FXuint)p2)>>32);
  *lo=p0+(p1<<32)+(p2<<32);
  *hi=p3+(p1>>32)+(p2>>32)+cy;
  }
#endif

/*******************************************************************************/

// "Number Parsing at a Gigabyte per Second," Daniel Lemire, 2021.
// "Fast number parsing without fallback", Noble Mushtak, Daniel Lemire, 2022.
FXdouble floatFromDec64(FXulong value,FXint decex){
  if(value!=0){
    FXulong xhi,xlo,yhi,ylo;
    FXint   binex,msb,ov,lz;

    // Underflow
    if(__unlikely(decex<-308-34)){      // -308-log10(2)*(64+52)
      return 0.0;
      }

    // Overflow
    if(__unlikely(decex>308)){
      return Math::fpMake(FXULONG(0x0000000000000000),2047);
      }

    // Fast convert via W.D. Clinger 1990.
    // "How to read floating point numbers accurately," W.D. Clinger,
    // ACM SIGPLAN Notices, June 20-22, 1990.
    if(decex<=22 && -22<=decex && !(value>>53)){        // value < 2**53
      FXdouble result=(FXdouble)(FXlong)value;
      if(0<=decex){
        result*=tenTo64[decex];
        }
      else{
        result/=tenTo64[-decex];
        }
      return result;
      }

    // Move msb to bit 63, keep the shift in lz
    lz=clz64(value);
    value<<=lz;

    // Binary exponent, with bias, adjusted for shift above.
    //
    // Use integer approximation:
    //
    //  log(10)/log(2) = 3.3219280948873623479
    //
    // and:
    //
    //  (217706*decex)/65536 = 3.321929931640625
    //
    binex=((217706*decex)>>16)+1024+63-lz;

    // 64 x 64 -> 128 multiply
    umul128(value,hifive[decex+342],&xhi,&xlo);

    // Check for carry from lower 64-bits
    if((xhi&0x1ff)==0x1ff){

      // 64 x 64 -> 128 multiply
      umul128(value,lofive[decex+342],&yhi,&ylo);

      // "Fast Number Parsing Without Fallback" by N. Mushtak and D. Lemire
      // says no fallback code needed here, it can't affect result.
      // GCC recognizes the code below as addq, adcq; neat!
      xlo+=yhi;
      xhi+=(xlo<yhi);
      }

    // Shift either 9 places or 10 places, depending on msb of xhi.
    // This brings mantissa to 54 bits (we need one extra lsb for
    // rounding purposes).
    msb=xhi>>63;
    value=xhi>>(msb+9);
    binex-=1^msb;

    // Abby Normal
    if(__unlikely(binex<=0)){

      // Can't budge the exponent, so shift mantissa, keeping one extra
      // lsb for rounding.
      // If binex is much more negative, i.e. all bits shift out, we'll
      // get zero; we could early out, but testing isn't free either so
      // instead we plough ahead...
      value>>=(1-binex);

      // Round with lsb now, then drop lsb
      value+=(value&1);
      value>>=1;

      // The rounding may have made things normal again, flipping
      // 0x3fffffffffffff to 0x40000000000000, for example.
      // If this happens, we need to return 1.0000x2^-1023 i.e.
      // make the binary exponent 1 instead of 0, like it would
      // be for denormals.
      binex=(value>>52);

      // Return denormalized result
      return Math::fpMake(value,binex);
      }

    // Special rounding cases
    if(xlo<=1 && -4<=decex && decex<=23 && (value&3)==1){
      if((value<<(msb+64-53-2))==xhi){
        value&=~1;             // flip it so that we do not round up
        }
      }

    // Round using lowest bit, then shift the lsb out.
    value+=(value&1);
    value>>=1;

    // The rounding may have caused an overflow; a right shift
    // and exponent increment brings it back in shape.
    ov=value>>53;
    value>>=ov;
    binex+=ov;

    // Too big is also possible
    if(__unlikely(2047<=binex)){
      return Math::fpMake(FXULONG(0x0000000000000000),2047);
      }

    // Return normalized result
    return Math::fpMake(value,binex);
    }
  return 0.0;
  }

/*******************************************************************************/

// Convert hexadecimal input to double
FXdouble floatFromHex64(FXulong value,FXint binex){
  if(value!=0){
    FXint lz,ov;

    // Shift msb up to bit 63; the value of the msb is
    // the value of lsb plus (63-clz)
    lz=clz64(value);
    value<<=lz;
    binex+=1023+63-lz;

    // Shift msb down by 10 bits, round lsb, then
    // shift one more to get msb down to bit 52
    value>>=10;
    value+=(value&1);
    value>>=1;

    // The rounding may have caused an overflow to bit 53,
    // A right shift and exponent increment fixes it
    ov=value>>53;
    value>>=ov;
    binex+=ov;

    // Underflow
    if(__likely(-52<=binex)){

      // Check for overflow
      if(__unlikely(2047<=binex)){
        return Math::fpMake(FXULONG(0x0000000000000000),2047);
        }

      // Denormal
      if(binex<=0){
        value>>=-binex;
        value+=(value&1);
        value>>=1;
        binex=(value>>52);
        }
        
      // Normal return
      return Math::fpMake(value,binex);
      }
    }
  return 0.0;
  }

/*******************************************************************************/

// Convert string to double
// Off by 1 lsb bit if >19 significant digits, otherwise exact.
// Fast path ~29 cycles (5750GE), ~83 normal path.
FXdouble __strtod(const FXchar *s,const FXchar** e,FXbool* ok){
  FXdouble result=0.0;
  FXulong  value=0;
  FXint    exponent=0;
  FXint    expo=0;
  FXint    dig=0;
  FXint    neg;
  FXint    nex;
  FXint    c;

  // Assume the worst
  if(ok) *ok=false;

  // No characters consumed
  if(e) *e=s;

  // Skip whitespace
  c=*s;
  while(Ascii::isSpace(c)){
    c=*++s;
    }

  // Handle sign
  if((neg=(c=='-')) || (c=='+')){
    c=*++s;
    }

  // Decimal
  if(c=='.') goto dot1;

  // Decimal digit
  if(c<='9' && '0'<=c){

    // Decimal for sure
    if('1'<=c) goto gtz;

    // Eat first zero
    c=*++s;

    // Decimal float
    if((c|32)!='x'){
      dig=1;
      while(c=='0'){                    // Leading zero before decimal
        c=*++s;
        }
      if(c=='.'){                       // Got a dot
dot1:   c=*++s;
        while(c=='0'){                  // More leading zeros after decimal
          exponent--;
          dig=1;
          c=*++s;
          }
        goto dot2;
        }
gtz:  while('0'<=c && c<='9'){          // Decimal digits
        if(value<DECIMAX){ value=value*10+(c-'0'); exponent--; }
        exponent++;
        dig=1;
        c=*++s;
        }
      if(c=='.'){                       // Got a dot
        c=*++s;
dot2:   while('0'<=c && c<='9'){        // Decimal digits
          if(value<DECIMAX){ value=value*10+(c-'0'); exponent--; }
          dig=1;
          c=*++s;
          }
        }

      // Got digits
      if(dig){
        if(e) *e=s;
        if((c|32)=='e'){                // Exponent start
          c=*++s;
          if((nex=(c=='-')) || (c=='+')){
            c=*++s;
            }
          if('0'<=c && c<='9'){         // Parse exponent
            do{
              expo=expo*10+(c-'0');
              c=*++s;
              }
            while('0'<=c && s[0]<='9');
            exponent+=nex?-expo:expo;
            if(e) *e=s;
            }
          }

        // Tentatively recognized
        if(ok) *ok=true;

        // Convert decimal to double
        result=floatFromDec64(value,exponent);
        }
      }

    // Hexadecimal float
    else{
      c=*++s;
      while(c=='0'){                    // Leading zero before decimal
        dig=1;
        c=*++s;
        }
      if(c=='.'){                       // Got a dot
        c=*++s;
        while(c=='0'){                  // More leading zeros after decimal
          exponent-=4;
          dig=1;
          c=*++s;
          }
        goto dot3;
        }
      while(Ascii::isHexDigit(c)){      // Hex digits
        if(value<HEXAMAX){ value=(value<<4)+Ascii::digitValue(c); exponent-=4; }
        exponent+=4;
        dig=1;
        c=*++s;
        }
      if(c=='.'){                       // Got a dot
        c=*++s;
dot3:   while(Ascii::isHexDigit(c)){    // Hex digits
          if(value<HEXAMAX){ value=(value<<4)+Ascii::digitValue(c); exponent-=4; }
          dig=1;
          c=*++s;
          }
        }

      // Got digits
      if(dig){
        if(e) *e=s;
        if((c|32)=='p'){                // Exponent start
          c=*++s;
          if((nex=(c=='-')) || (c=='+')){
            c=*++s;
            }
          if('0'<=c && c<='9'){         // Parse exponent
            do{
              expo=expo*10+(c-'0');
              c=*++s;
              }
            while('0'<=c && c<='9');
            exponent+=nex?-expo:expo;
            if(e) *e=s;
            }
          }

        // Tentatively recognized
        if(ok) *ok=true;

        // Convert hexadecimal to double
        result=floatFromHex64(value,exponent);
        }
      }
    }

  // Infinite
  else if((c|32)=='i' && (s[1]|32)=='n' && (s[2]|32)=='f'){
    result=Math::fpMake(FXULONG(0x0000000000000000),2047);
    if(ok) *ok=true;
    if(e){
      s+=3;
      if((s[0]|32)=='i' && (s[1]|32)=='n' && (s[2]|32)=='i' && (s[3]|32)=='t' && (s[4]|32)=='y') s+=5;
      *e=s;
      }
    }

  // NaN
  else if((c|32)=='n' && (s[1]|32)=='a' && (s[2]|32)=='n'){
    result=Math::fpMake(FXULONG(0x000fffffffffffff),2047);
    if(ok) *ok=true;
    if(e){
      s+=3;
      *e=s;
      }
    }

  return neg ? -result : result;
  }

/*******************************************************************************/

// "Number Parsing at a Gigabyte per Second," Daniel Lemire, 2021.
// "Fast number parsing without fallback", Noble Mushtak, Daniel Lemire, 2022.
FXfloat floatFromDec32(FXulong value,FXint decex){
  if(value!=0){
    FXulong xhi,xlo,yhi,ylo;
    FXint   binex,msb,ov,lz;

    // Underflow
    if(__unlikely(decex<-39-26)){       // -39-log10(2)*(64+23)
      return 0.0f;
      }

    // Overflow
    if(__unlikely(decex>38)){
      return Math::fpMake(0x00000000u,255);
      }

    // Fast convert via W.D. Clinger 1990.
    // "How to read floating point numbers accurately," W.D. Clinger,
    // ACM SIGPLAN Notices, June 20-22, 1990.
    if(decex<=10 && -10<=decex && !(value>>23)){        // value < 2**23
      FXfloat result=(FXfloat)(FXlong)value;
      if(0<=decex){
        result*=tenTo32[decex];
        }
      else{
        result/=tenTo32[-decex];
        }
      return result;
      }

    // Move msb to bit 63, keep the shift in lz
    lz=clz64(value);
    value<<=lz;

    // Binary exponent, with bias, adjusted for shift above.
    //
    // Use integer approximation:
    //
    //  log(10)/log(2) = 3.3219280948873623479
    //
    // and:
    //
    //  (217706*decex)/65536 = 3.321929931640625
    //
    binex=((217706*decex)>>16)+128+63-lz;

    // 64 x 64 -> 128 multiply
    umul128(value,hifive[decex+342],&xhi,&xlo);

    // Check for carry from lower 64-bits
    if((xhi&FXULONG(0x3fffffffff))==FXULONG(0x3fffffffff)){

      // 64 x 64 -> 128 multiply
      umul128(value,lofive[decex+342],&yhi,&ylo);

      // "Fast Number Parsing Without Fallback" by N. Mushtak and D. Lemire
      // says no fallback code needed here, it can't affect result.
      // GCC recognizes the code below as addq, adcq; neat!
      xlo+=yhi;
      xhi+=(xlo<yhi);
      }

    // Shift either 37 places or 38 places, depending on msb of xhi.
    // This brings mantissa to 54 bits (we need one extra lsb for
    // rounding purposes).
    msb=xhi>>63;
    value=xhi>>(msb+38);
    binex-=1^msb;

    // Abby Normal
    if(__unlikely(binex<=0)){

      // Can't budge the exponent, so shift mantissa, keeping one extra
      // lsb for rounding.
      // If binex is much more negative, i.e. all bits shift out, we'll
      // get zero; we could early out, but testing isn't free either so
      // instead we plough ahead...
      value>>=(1-binex);

      // Round with lsb now, then drop lsb
      value+=(value&1);
      value>>=1;

      // The rounding may have made things normal again, flipping
      // 0x007FFFFF to 0x00800000, for example.
      // If this happens, we need to return 1.0000x2^-127 i.e.
      // make the binary exponent 1 instead of 0, like it would
      // be for denormals.
      binex=(value>>23);

      // Return denormalized result
      return Math::fpMake((FXuint)value,binex);
      }

    // Special rounding cases
    if(xlo<=1 && -17<=decex && decex<=10 && (value&3)==1){
      if((value<<(msb+64-24-2))==xhi){
        value&=~1;             // flip it so that we do not round up
        }
      }

    // Round using lowest bit, then shift the lsb out.
    value+=(value&1);
    value>>=1;

    // The rounding may have caused an overflow to bit 24.
    // A right shift and exponent increment brings it back.
    ov=value>>24;
    value>>=ov;
    binex+=ov;

    // Too big is also possible
    if(__unlikely(255<=binex)){
      return Math::fpMake(0x00000000u,255);
      }

    // Return normalized result
    return Math::fpMake((FXuint)value,binex);
    }
  return 0.0f;
  }

/*******************************************************************************/

// Convert hexadecimal input to float
FXfloat floatFromHex32(FXulong value,FXint binex){
  if(value!=0){
    FXint lz,ov;

    // Shift msb up to bit 63; the value of the msb is
    // the value of lsb plus (63-clz)
    lz=clz64(value);
    value<<=lz;
    binex+=127+63-lz;

    // Shift msb down by 39 bits, round lsb, then
    // shift one more to get msb to bit 23
    value>>=39;
    value+=(value&1);
    value>>=1;

    // The rounding may have caused an overflow to bit 24.
    // A right shift and exponent increment fixes it
    ov=value>>24;
    value>>=ov;
    binex+=ov;

    // Underflow
    if(__likely(-22<=binex)){
  
      // Check for overflow
      if(__unlikely(255<=binex)){
        return Math::fpMake(0x00000000u,255);
        }

      // Check for underflow
      if(binex<=0){
        value>>=-binex;
        value+=(value&1);
        value>>=1;
        binex=(value>>23);
        }
  
      // Normal return
      return Math::fpMake((FXuint)value,binex);
      }
    }
  return 0.0f;
  }

/*******************************************************************************/

// Convert string to float
// Off by 1 lsb bit if >19 significant digits, otherwise exact.
// Fast path, ~36 cycles, ~81 normal path.
FXfloat __strtof(const FXchar *s,const FXchar** e,FXbool* ok){
  FXfloat  result=0.0f;
  FXulong  value=0;
  FXint    exponent=0;
  FXint    expo=0;
  FXint    dig=0;
  FXint    neg;
  FXint    nex;
  FXint    c;

  // Assume the worst
  if(ok) *ok=false;

  // No characters consumed
  if(e) *e=s;

  // Skip whitespace
  c=*s;
  while(Ascii::isSpace(c)){
    c=*++s;
    }

  // Handle sign
  if((neg=(c=='-')) || (c=='+')){
    c=*++s;
    }

  // Decimal
  if(c=='.') goto dot1;

  // Decimal digit
  if(c<='9' && '0'<=c){

    // Decimal for sure
    if('1'<=c) goto gtz;

    // Eat first zero
    c=*++s;

    // Decimal float
    if((c|32)!='x'){
      dig=1;
      while(c=='0'){                    // Leading zero before decimal
        c=*++s;
        }
      if(c=='.'){                       // Got a dot
dot1:   c=*++s;
        while(c=='0'){                  // More leading zeros after decimal
          exponent--;
          dig=1;
          c=*++s;
          }
        goto dot2;
        }
gtz:  while('0'<=c && c<='9'){          // Decimal digits
        if(value<DECIMAX){ value=value*10+(c-'0'); exponent--; }
        exponent++;
        dig=1;
        c=*++s;
        }
      if(c=='.'){                       // Got a dot
        c=*++s;
dot2:   while('0'<=c && c<='9'){        // Decimal digits
          if(value<DECIMAX){ value=value*10+(c-'0'); exponent--; }
          dig=1;
          c=*++s;
          }
        }

      // Got digits
      if(dig){
        if(e) *e=s;
        if((c|32)=='e'){                // Exponent start
          c=*++s;
          if((nex=(c=='-')) || (c=='+')){
            c=*++s;
            }
          if('0'<=c && c<='9'){         // Parse exponent
            do{
              expo=expo*10+(c-'0');
              c=*++s;
              }
            while('0'<=c && s[0]<='9');
            exponent+=nex?-expo:expo;
            if(e) *e=s;
            }
          }

        // Tentatively recognized
        if(ok) *ok=true;

        // Convert decimal to double
        result=floatFromDec32(value,exponent);
        }
      }

    // Hexadecimal float
    else{
      c=*++s;
      while(c=='0'){                    // Leading zero before decimal
        dig=1;
        c=*++s;
        }
      if(c=='.'){                       // Got a dot
        c=*++s;
        while(c=='0'){                  // More leading zeros after decimal
          exponent-=4;
          dig=1;
          c=*++s;
          }
        goto dot3;
        }
      while(Ascii::isHexDigit(c)){      // Hex digits
        if(value<HEXAMAX){ value=(value<<4)+Ascii::digitValue(c); exponent-=4; }
        exponent+=4;
        dig=1;
        c=*++s;
        }
      if(c=='.'){                       // Got a dot
        c=*++s;
dot3:   while(Ascii::isHexDigit(c)){    // Hex digits
          if(value<HEXAMAX){ value=(value<<4)+Ascii::digitValue(c); exponent-=4; }
          dig=1;
          c=*++s;
          }
        }

      // Got digits
      if(dig){
        if(e) *e=s;
        if((c|32)=='p'){                // Exponent start
          c=*++s;
          if((nex=(c=='-')) || (c=='+')){
            c=*++s;
            }
          if('0'<=c && c<='9'){         // Parse exponent
            do{
              expo=expo*10+(c-'0');
              c=*++s;
              }
            while('0'<=c && c<='9');
            exponent+=nex?-expo:expo;
            if(e) *e=s;
            }
          }

        // Tentatively recognized
        if(ok) *ok=true;

        // Convert hexadecimal to double
        result=floatFromHex32(value,exponent);
        }
      }
    }

  // Infinite
  else if((c|32)=='i' && (s[1]|32)=='n' && (s[2]|32)=='f'){
    result=Math::fpMake(0x00000000u,255);
    if(ok) *ok=true;
    if(e){
      s+=3;
      if((s[0]|32)=='i' && (s[1]|32)=='n' && (s[2]|32)=='i' && (s[3]|32)=='t' && (s[4]|32)=='y') s+=5;
      *e=s;
      }
    }

  // NaN
  else if((c|32)=='n' && (s[1]|32)=='a' && (s[2]|32)=='n'){
    result=Math::fpMake(0x7fffffu,255);
    if(ok) *ok=true;
    if(e){
      s+=3;
      *e=s;
      }
    }

  return neg ? -result : result;
  }

}
