﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/accessanalyzer/model/PrincipalType.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>

using namespace Aws::Utils;

namespace Aws {
namespace AccessAnalyzer {
namespace Model {
namespace PrincipalTypeMapper {

static const int IAM_ROLE_HASH = HashingUtils::HashString("IAM_ROLE");
static const int IAM_USER_HASH = HashingUtils::HashString("IAM_USER");

PrincipalType GetPrincipalTypeForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == IAM_ROLE_HASH) {
    return PrincipalType::IAM_ROLE;
  } else if (hashCode == IAM_USER_HASH) {
    return PrincipalType::IAM_USER;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<PrincipalType>(hashCode);
  }

  return PrincipalType::NOT_SET;
}

Aws::String GetNameForPrincipalType(PrincipalType enumValue) {
  switch (enumValue) {
    case PrincipalType::NOT_SET:
      return {};
    case PrincipalType::IAM_ROLE:
      return "IAM_ROLE";
    case PrincipalType::IAM_USER:
      return "IAM_USER";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace PrincipalTypeMapper
}  // namespace Model
}  // namespace AccessAnalyzer
}  // namespace Aws
