/*
 * This file is part of OpenTTD.
 * OpenTTD is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, version 2.
 * OpenTTD is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details. You should have received a copy of the GNU General Public License along with OpenTTD. If not, see <https://www.gnu.org/licenses/old-licenses/gpl-2.0>.
 */

/* @file midifile.hpp Parser for standard MIDI files */

#ifndef MUSIC_MIDIFILE_HPP
#define MUSIC_MIDIFILE_HPP

#include "../fileio_type.h"
#include "midi.h"

struct MusicSongInfo;

struct MidiFile {
	struct DataBlock {
		uint32_t ticktime; ///< tick number since start of file this block should be triggered at
		int64_t realtime = 0; ///< real-time (microseconds) since start of file this block should be triggered at
		std::vector<uint8_t> data; ///< raw midi data contained in block
		DataBlock(uint32_t _ticktime = 0) : ticktime(_ticktime) { }
	};
	struct TempoChange {
		uint32_t ticktime; ///< tick number since start of file this tempo change occurs at
		uint32_t tempo;    ///< new tempo in microseconds per tick
		TempoChange(uint32_t _ticktime, uint32_t _tempo) : ticktime(_ticktime), tempo(_tempo) { }
	};

	std::vector<DataBlock> blocks{}; ///< sequential time-annotated data of file, merged to a single track
	std::vector<TempoChange> tempos{}; ///< list of tempo changes in file
	uint16_t tickdiv = 0; ///< ticks per quarter note

	MidiFile();
	~MidiFile();

	bool LoadFile(const std::string &filename);
	bool LoadMpsData(const uint8_t *data, size_t length);
	bool LoadSong(const MusicSongInfo &song);
	void MoveFrom(MidiFile &other);

	bool WriteSMF(const std::string &filename);

	static std::string GetSMFFile(const MusicSongInfo &song);
	static bool ReadSMFHeader(const std::string &filename, SMFHeader &header);
	static bool ReadSMFHeader(FileHandle &file, SMFHeader &header);
};

#endif /* MUSIC_MIDIFILE_HPP */
