/*
 * audbin - save audit bin files
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <sys/types.h>
#include <unistd.h>
#include <stdint.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/vfs.h>
#include <time.h>
#include <dirent.h>
#include <stdarg.h>
#include <stdio.h>
#include <syslog.h>

#include <linux/laus_audit.h>
#include <laussrv.h>
#include <laus.h>

static char *	opt_saveto = NULL;
static char *	opt_compress = NULL;
static int	opt_clear = 0,
		opt_overwrite = 0,
		opt_append = 0,
		opt_verbose = 1,
                opt_threshold_percent=0;
static float    opt_threshold=0.0;
static char *   opt_notify=0L;
static char *	cur_saveto;

static void	process_log(const char *logfile);
static void	usage(int);
static void	fatal(const char *fmt, ...);
static char *   make_notify_command(char *, const char *);

int
main(int argc, char **argv)
{
	int	c, l;

	while ((c = getopt(argc, argv, "aCc:hS:oqT:N:")) != -1) {
		switch (c) {
		case 'a':
			opt_append = 1;
			break;
		case 'C':
			opt_clear = 1;
			break;
		case 'c':
			opt_compress = optarg;
			if (strcasecmp(opt_compress, "gzip")
			 && strcasecmp(opt_compress, "bzip2")
			 && strcasecmp(opt_compress, "none"))
				usage(1);
			break;
		case 'o':
			opt_overwrite = 1;
			break;
		case 'q':
			opt_verbose = 0;
			break;
		case 'S':
			opt_saveto = optarg;
			break;
		case 'T':
		{       char percent[2]="";
		        sscanf(optarg,"%f%1[%]", &opt_threshold, &percent[0]);
			opt_threshold_percent = (percent[0] == '%');
		}       break;
                case 'N':
			opt_notify = optarg;
			break;
		case 'h':
		case '?':
			usage(0);
		default:
			usage(1);
		}
	}

	if (!opt_saveto && !opt_clear) {
		fprintf(stderr, "Don't know what to do.\n");
		usage(1);
	}
	if (opt_overwrite && opt_append) {
		fprintf(stderr, "Overwrite and append options are "
				"mutually exclusive\n");
		usage(1);
	}
	if (opt_compress && strcasecmp(opt_compress, "none")) {
		fprintf(stderr,
			"Compression not yet supported, option ignored\n");
		opt_compress = NULL;
	}
	
	if (opt_notify != 0L) 
	{
		if (opt_threshold <= 0)
		{
			fprintf(stderr, "-N notify command given without -T threshold option.\n");
			usage(1);
		}
		if(  (strstr(opt_notify,"%f") != 0L)
	          && ( opt_saveto == 0L )
		   )
		{
			fprintf(stderr, "-N notify command contains \%f, but no -S option.\n");
			usage(1);				
		}
	}
	if ( opt_threshold < 0 )
	{
		fprintf(stderr, "Invalid negative -T threshold option given.\n");
		usage(1);
	}
	if (optind == argc)
		usage(1);

	openlog("audbin", LOG_PID, LOG_DAEMON);

	while (optind < argc)
		process_log(argv[optind++]);
	return 0;
}

/*
 * Make a save file, expending %* substituions
 *  %u		unique integer
 *  %t		timestamp (seconds since epoch)
 */
static int
savefile_open(const char *destination, struct laus_file_header *header)
{
	static unsigned int uniq = 0;
	static char	path[PATH_MAX];
	struct stat	stb;
	int		fd, flags;
	struct statfs	stf;
	long		free_blocks;
	char		*notify_command, *last_notify_command = "",
			*destdir = ".", *p;
	int		notify_status, l;


	/* When writing to stdout, include header of first file
	 * only */
	if (!strcmp(destination, "-")) {
		static int count = 0;

		fd = 1;
		if (count++)
			goto out;
		goto include_header;
	}

	while (1) {
		const char	*sp = destination;
		char		*dp = path, c;
		long		bufleft;

		do {
			c = *sp++;

			bufleft = sizeof(path) - (dp - path);
			if (bufleft < 3)
				fatal("%s: expanded file name too long\n",
				      destination);

			if (c != '%' || *sp == '\0') {
				*dp++ = c;
			} else {
				c = *sp++;
				switch (c) {
				case '%':
					*dp++ = '%';
					continue;
				case 'h':
					snprintf(dp, bufleft, "%s",
							header->h_hostname);
					break;
				case 't':
					snprintf(dp, bufleft, "%lu",
							(unsigned long) time(NULL));
					break;
				case 'u':
					snprintf(dp, bufleft, "%u", uniq++);
					break;
				default:
					fatal("Unknown substitution %%%c "
					      "in destination file \"%s\"\n",
					      c, destination);
				}
				dp += strlen(dp);
			}
		} while (c != '\0');

		flags = O_WRONLY|O_CREAT;
		if (uniq)
			flags |= O_EXCL;
		if (opt_overwrite)
			flags |= O_TRUNC;
		else if (opt_append)
			flags |= O_APPEND;
		else
			flags |= O_EXCL;

		if ((fd = open(path, flags, 0600)) >= 0)
			break;

		if (errno != EEXIST) {
			fatal("%s: open failed: %m\n", path);
		} else if (!uniq) {
			fatal("%s: pathname not unique\n", path);
		}
	}

	if (fstat(fd, &stb) < 0)
		fatal("%s: fstat failed: %m\n", path);

	if (flock(fd, LOCK_EX|LOCK_NB) < 0) {
		if (errno == EWOULDBLOCK) {
			fprintf(stderr,
				"%s: currently locked by some other process, "
				"waiting...\n", path);
			if (flock(fd, LOCK_EX) >= 0)
				goto locked;
		}
		fatal("%s: unable to lock: %m\n", path);
	}


locked:
	if (stb.st_size == 0) {
include_header:
		header->h_count = -1;
		if (write(fd, header, sizeof(*header)) != sizeof(*header))
			fatal("%s: short write\n", path);
	}

out:
	cur_saveto = path;

	return fd;
}

static void
save_log(const char *logfile, const char *destination)
{
	struct laus_file_header	header;
	unsigned char	buffer[8192];
	struct stat	stb;
	size_t		copy;
	int		fd, ofd = -1;

	if ((fd = open(logfile, O_RDONLY)) < 0)
		fatal("%s: %m\n", logfile);

	if (fstat(fd, &stb) < 0)
		fatal("%s: fstat failed: %m\n", logfile);

	if (stb.st_size == 0) {
		fprintf(stderr, "%s: empty, ignored\n", logfile);
		close(fd);
		return;
	}

	/* threshold test */
	if (opt_threshold > 0.0)
	{
		if ( (p = strrchr(destination,'/')) != 0L )
		{
			l = (p - destination);
			destdir = (char*) malloc( l + 2 );
			strncpy(destdir, destination, l );
			sprintf(destdir+l,"/.");
		}
		notify_status = 0;
		while ( statfs(destdir, &stf) == 0 )
		{
			free_blocks =  (  stf.f_bfree 
				       - ( ( stb.st_size / stf.f_bsize ) 
				         + ((stb.st_size % stf.f_bsize) ? 1 : 0 ) 
				         )
				       );
			if ( ( free_blocks <  0 ) 
			   ||( stf.f_ffree <= 1 )
			   ||(   opt_threshold_percent  
			      && ((((float)free_blocks / (float)stf.f_blocks) * 100.0) <= opt_threshold)
			     )
			   ||(   (!opt_threshold_percent)
			      && ( free_blocks <= opt_threshold)	
			     )
			  )
			{  /* threshold exceeded */				
				if ( opt_notify == 0L )
				{
					fatal( "threshold %.2f exceeded for filesystem %s - free blocks down to %.2f%%",
					       opt_threshold, destdir,
					       (((float)free_blocks / (float)stf.f_blocks) * 100.0)
					     );
					break;
				}
			        if ( (notify_command = make_notify_command( opt_notify, destination )) == 0L)
				{
					fatal( "threshold %.2f exceeded for filesystem %s - free blocks down to %.2f%% - "
					       "notify command cannot be run - no old save file to process.",
					       opt_threshold, destdir,
					       (((float)free_blocks / (float)stf.f_blocks) * 100.0)
					     );
					break;
				}						      
				if ( strcmp(notify_command, last_notify_command) != 0)
				{
					syslog(LOG_INFO,"audbin: threshold %.2f exceeded for filesystem %s - "
					       " free blocks down to %.2f%% "
					       ": running notify command: %s",
					       opt_threshold, destdir,
					       (((float)free_blocks / (float)stf.f_blocks) * 100.0),
					       notify_command
					      );
					notify_status = WEXITSTATUS( system( notify_command ));
					if( notify_status != 0)
					        break;
					last_notify_command = notify_command;
				}else
				{
					notify_status = 1;
					break;
				}
			}else
			{
				notify_status = 0;
				break;
			}
		}
		if( notify_status != 0 )
			fatal( "threshold %.2f exceeded for filesystem %s : notify command %s failed: %d %s",
			       opt_threshold, destdir, notify_command, notify_status, strerror(notify_status)
			     );
	}
	

	if (read(fd, &header, sizeof(header)) != sizeof(header))
		fatal("%s: truncated file\n", logfile);
	copy = stb.st_size - sizeof(header);

	if (header.h_count != -1) {
		if (copy < header.h_count)
			fatal("%s: truncated file\n", logfile);
		copy = header.h_count;
	}

	if (strchr(destination, '@')) {
		fatal("Saving to remote host not yet implemented\n");
	} else {
		ofd = savefile_open(destination, &header);
		if (opt_verbose)
			printf("Saving %s -> %s\n", logfile, cur_saveto);
	}

	/* Copy bin file to new destination */
	while (copy) {
		unsigned int	count;
		int		n;

		if ((count = sizeof(buffer)) > copy)
			count = copy;
		n = read(fd, buffer, count);
		if (n < 0)
			fatal("%s: Uable to read from file (%s)\n", logfile, strerror(errno));
		if (n == 0)
			fatal("%s: Unexpected end of file\n", logfile);
		count = n;

		/* Writes to pipes may send less than we want
		 * to send, so spin until we're rid of everything */
		copy -= count;
		while (count) {
			n = write(ofd, buffer, count);
			if (n < 0)
				fatal("%s: write failed: %m\n", cur_saveto);
			if (n == 0)
				fatal("%s: Zero byte write?! Should not happen...\n",
						cur_saveto);
			count -= n;
		}
	}

	if (fsync(ofd) < 0)
		fatal("%s: fsync failed: %m\n", cur_saveto);

	close(ofd);
	close(fd);
}

static void
clear_log(const char *logfile)
{
	struct laus_file_header header;
	int	fd, n, r;

	if (opt_verbose)
		printf("Clearing %s\n", logfile);
	if ((fd = open(logfile, O_WRONLY)) < 0)
		fatal("%s: cannot open for writing: %m\n", logfile);

	n = sizeof(header);
	memset(&header, 0, n);
	if ((r = write(fd, &header, n)) < 0)
		fatal("%s: error on write: %m\n", logfile);
	if (r != n)
		fatal("%s: short write when clearing logfile\n", logfile);
	close(fd);
}

void
process_log(const char *logfile)
{
	if (opt_saveto)
	{
	        syslog(LOG_INFO, "saving binary audit log %s", logfile);
		save_log(logfile, opt_saveto);
	}
	if (opt_clear)
	{
	        syslog(LOG_INFO,"clearing binary audit log %s", logfile);
		clear_log(logfile);
	}
}

void
usage(int exval)
{
	fprintf(stderr,
		"usage: audbin [-h?] [-Coa] [-S saveto] [-c compress] [-T threshold [-N notify command]] logfile ...\n"
		"  compression modes: gzip bzip2 none\n"
	       );
	exit(exval);
}

void
fatal(const char *fmt, ...)
{
	va_list	ap;

	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	vsyslog(LOG_CRIT,fmt,ap);
	va_end(ap);
	exit(1);
}

char *make_notify_command( char *notify, const char *saveto )
{
	char   *path ="./", *oldest, *pp, *pu, *pd, *pf=(char*)saveto, *pfa=0L, *nc=notify, *nc2;
	DIR    *dir;
	struct dirent *dp;
	uint32_t l;
	struct stat st;
	time_t oldest_mtime=0;

	nc = notify;
 	if ((pfa=strstr(notify,"%f")) == 0L)
		return nc;
	if ( (pp=strrchr(saveto,'/')) != 0L )
	{
		l = (uint32_t)(pp - saveto);
		path = (char*)malloc(l+1);
		strncpy(path, saveto, l);
		path[ l ] = '\0';
		pf = pp + 1;
	}
	if ( ( dir = opendir(path) ) != 0L )
	{
		if((pu = strstr(pf, "%u"))!=0L)
			*pu='\0';
		l = strlen(pf);
		oldest=0L;
		oldest_mtime=0;
		while( (dp = readdir(dir)) != 0L )
		{
			if( strncmp(dp->d_name, pf, l) == 0 )
			{
				l = strlen(path) + 1 + strlen(dp->d_name);
				pd = malloc(l+1);
				sprintf(pd,"%s/%s", path, dp->d_name);
				if( (stat(pd, &st) == 0) && 
				    ((oldest_mtime == 0) || ( st.st_mtime <= oldest_mtime ))
				  )
				{
					oldest = pd;
					oldest_mtime = st.st_mtime;
				}else
					free(pd);				       
			}
		}
		closedir(dir);
		if(pu != 0L)
			*pu='%';
	}
	if( oldest != 0L )
	{
                nc2=nc;
	        do
		{
			l = (uint32_t) (pfa - nc);
			nc2=nc;
			nc = malloc(l + strlen(oldest)+ strlen(pfa+2) + 1);
			strncpy(nc, nc2, l );
			sprintf(nc + l, "%s%s", oldest, pfa+2);
			pfa = strstr(nc+l+strlen(oldest), "%f");
			if(nc2 != notify)
				free(nc2);
		}while(pfa != 0L);
		
	}else	/* no "save" files found to process. */
		return 0L;
	return nc;
}
