/*
 * Handle system call names
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define _GNU_SOURCE

#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/un.h>
#include <sys/resource.h>
#include <stropts.h>
#include <sched.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <net/if_arp.h>
#include <net/route.h>

/* This is quite messy */
#ifdef HAVE_SYS_CAPABILITY_H
#include <sys/capability.h>
#else
#include <linux/capability.h>
/* Cleaning up for ppc */
#undef htonl
#undef ntohl
#undef htons
#undef ntohs
#endif

#include <laus.h>
#include <laussrv.h>
#include "syscall.h"

#include <asm/unistd.h>
#ifdef __powerpc64__
	typedef unsigned int u32;
	typedef signed int s32;
#endif
#if !(defined(__ia64__) || defined(__hppa__))
	#include <asm/ipc.h>
#endif
#include <linux/sys.h>
#include <linux/net.h>

#if defined (__x86_64__)
	#warning "Will do workaround for NR_syscalls on __x86_64__"
	#define NR_syscalls __NR_syscall_max
#endif

#if defined (__powerpc64__)
	#warning "Will do workaround for NR_syscalls on __powerpc64__"
	#define NR_syscalls __NR_syscalls
#endif

#if defined (__powerpc__)
	#warning "Will do workaround for NR_syscalls on __powerpc__"
	#define NR_syscalls __NR_syscalls
#endif

#if defined (__hppa__)
	#warning "Will do workaround for NR_syscalls on __hppa__"
	#define NR_syscalls __NR_Linux_syscalls
#endif

typedef int		syscall_print_fn_t(int, const void *, size_t);

#define MAX_ARGS	8
struct syscall_info {
	const char *		name;
	syscall_print_fn_t	*arg[MAX_ARGS];
};


static int		print_open_mode(int, const void *, size_t);
static int		print_octal(int, const void *, size_t);
static int		print_hex(int, const void *, size_t);
static int		print_signal(int, const void *, size_t);
static int		print_uid(int, const void *, size_t);
static int		print_gid(int, const void *, size_t);
static int		print_groups(int, const void *, size_t);
static int		print_groups32(int, const void *, size_t);
static int		print_rlim_name(int, const void *, size_t);
static int		print_rlimit(int, const void *, size_t);
static int		print_cap_header(int, const void *, size_t);
static int		print_cap_data(int, const void *, size_t);
static int		print_ecap(int, const void *, size_t);
static int		print_icap(int, const void *, size_t);
static int		print_pcap(int, const void *, size_t);
static int		print_clone_flags(int, const void *, size_t);
static int		print_af(int, const void *, size_t);
static int		print_sockaddr(int, const void *, size_t);
static int		print_string(int, const void *, size_t);
static int		print_ifc_addr(int, const void *, size_t);
static int		print_ifc_mtu(int, const void *, size_t);
static int		print_ifc_flags(int, const void *, size_t);
static int		print_rtentry(int, const void *, size_t);
static int		print_arpreq(int, const void *, size_t);
static int		print_socktype(int, const void *, size_t);
static int		print_timeref(int, const void *, size_t);
static int		print_timespec(int, const void *, size_t);
static int		print_timeval(int, const void *, size_t);
static int		print_timezone(int, const void *, size_t);
static int		print_ref_off_t(int, const void *, size_t);
static int		print_default(int type, const void *p, size_t len);
static void		print_ioctl_args(const struct syscall_info *,
				const struct syscall_data *);

/*
 * System call table. This table does the mapping of
 * system call numbers to names and vice versa, and holds
 * additional information for pretty-printing arguments.
 *
 * We try to handle the __NR_socketcall and __NR_ipc
 * kludges transparently here.
 *
 * Note that NR_syscalls seems to be off in ia64 user space
 */
#if defined  (__ia64__)
#define __syscall_end	(1300+1)
#else
#define __syscall_end	(NR_syscalls + 1)
#endif

#define ABSTRACT_SYSCALLS (laus_api_version() > 0x20040000)

/*
 * Define bogus __NR_xxx values for socketcall and ipc
 */
enum {
	__extra_syscall_base = __syscall_end + 1,

	__socketcall_base,
#ifdef __NR_socketcall
	__NR_socket = __socketcall_base + SYS_SOCKET,
	__NR_bind = __socketcall_base + SYS_BIND,
	__NR_connect = __socketcall_base + SYS_CONNECT,
	__NR_listen = __socketcall_base + SYS_LISTEN,
	__NR_accept = __socketcall_base + SYS_ACCEPT,
	__NR_socketpair = __socketcall_base + SYS_SOCKETPAIR,
	__NR_setsockopt = __socketcall_base + SYS_SETSOCKOPT,
	__NR_getsockopt = __socketcall_base + SYS_GETSOCKOPT,
	__NR_getsockname = __socketcall_base + SYS_GETSOCKNAME,
	__NR_getpeername = __socketcall_base + SYS_GETPEERNAME,
	__NR_send = __socketcall_base + SYS_SEND,
	__NR_recv = __socketcall_base + SYS_RECV,
	__NR_sendto = __socketcall_base + SYS_SENDTO,
	__NR_recvfrom = __socketcall_base + SYS_RECVFROM,
	__NR_sendmsg = __socketcall_base + SYS_SENDMSG,
	__NR_recvmsg = __socketcall_base + SYS_RECVMSG,
	__socketcall_end = __socketcall_base + 32,
#else
	__socketcall_end = __socketcall_base,
#endif

	__ipccall_base,
#ifdef __NR_ipc
	__NR_semop  = __ipccall_base + SEMOP,
	__NR_semtimedop  = __ipccall_base + SEMTIMEDOP,
	__NR_semget  = __ipccall_base + SEMGET,
	__NR_semctl  = __ipccall_base + SEMCTL,
	__NR_msgsnd  = __ipccall_base + MSGSND,
	__NR_msgrcv  = __ipccall_base + MSGRCV,
	__NR_msgget  = __ipccall_base + MSGGET,
	__NR_msgctl  = __ipccall_base + MSGCTL,
	__NR_shmat  = __ipccall_base + SHMAT,
	__NR_shmdt  = __ipccall_base + SHMDT,
	__NR_shmget  = __ipccall_base + SHMGET,
	__NR_shmctl  = __ipccall_base + SHMCTL,
	__ipccall_end = __ipccall_base + 32,
#else
	__ipccall_end = __ipccall_base,
#endif

	__last_syscall
};

#define syscall(n, args...)	[__NR_##n] = { #n, { args } }

static const struct syscall_info syscall_table[__last_syscall] = {
#ifdef __NR_exit
	syscall(exit),
#endif
#ifdef __NR_fork
	syscall(fork), 
#endif
#ifdef __NR_read
	syscall(read), 
#endif
#ifdef __NR_write
	syscall(write), 
#endif
#ifdef __NR_open
	syscall(open, NULL, print_open_mode, print_octal), 
#endif
#ifdef __NR_close
	syscall(close), 
#endif
#ifdef __NR_waitpid
	syscall(waitpid), 
#endif
#ifdef __NR_creat
	syscall(creat, NULL, print_octal), 
#endif
#ifdef __NR_link
	syscall(link), 
#endif
#ifdef __NR_unlink
	syscall(unlink), 
#endif
#ifdef __NR_execve
	syscall(execve), 
#endif
#ifdef __NR_chdir
	syscall(chdir), 
#endif
#ifdef __NR_time
	syscall(time), 
#endif
#ifdef __NR_mknod
	syscall(mknod, NULL, print_octal, print_hex), 
#endif
#ifdef __NR_chmod
	syscall(chmod, NULL, print_octal), 
#endif
#ifdef __NR_chown
	syscall(chown, NULL, print_uid, print_gid),
#endif
#ifdef __NR_lchown
	syscall(lchown, NULL, print_uid, print_gid),
#endif
#ifdef __NR_break
	syscall(break), 
#endif
#ifdef __NR_oldstat
	syscall(oldstat), 
#endif
#ifdef __NR_lseek
	syscall(lseek), 
#endif
#ifdef __NR_getpid
	syscall(getpid), 
#endif
#ifdef __NR_mount
	syscall(mount), 
#endif
#ifdef __NR_umount
	syscall(umount), 
#endif
#ifdef __NR_setuid
	syscall(setuid), 
#endif
#ifdef __NR_getuid
	syscall(getuid), 
#endif
#ifdef __NR_stime
	syscall(stime, print_timeref), 
#endif
#ifdef __NR_ptrace
	syscall(ptrace), 
#endif
#ifdef __NR_alarm
	syscall(alarm), 
#endif
#ifdef __NR_oldfstat
	syscall(oldfstat), 
#endif
#ifdef __NR_pause
	syscall(pause), 
#endif
#ifdef __NR_utime
	syscall(utime), 
#endif
#ifdef __NR_stty
	syscall(stty), 
#endif
#ifdef __NR_gtty
	syscall(gtty), 
#endif
#ifdef __NR_access
	syscall(access), 
#endif
#ifdef __NR_nice
	syscall(nice), 
#endif
#ifdef __NR_ftime
	syscall(ftime), 
#endif
#ifdef __NR_sync
	syscall(sync), 
#endif
#ifdef __NR_kill
	syscall(kill, NULL, print_signal), 
#endif
#ifdef __NR_rename
	syscall(rename), 
#endif
#ifdef __NR_mkdir
	syscall(mkdir, NULL, print_octal), 
#endif
#ifdef __NR_rmdir
	syscall(rmdir), 
#endif
#ifdef __NR_dup
	syscall(dup), 
#endif
#ifdef __NR_pipe
	syscall(pipe), 
#endif
#ifdef __NR_times
	syscall(times), 
#endif
#ifdef __NR_prof
	syscall(prof), 
#endif
#ifdef __NR_brk
	syscall(brk), 
#endif
#ifdef __NR_setgid
	syscall(setgid), 
#endif
#ifdef __NR_getgid
	syscall(getgid), 
#endif
#ifdef __NR_signal
	syscall(signal), 
#endif
#ifdef __NR_geteuid
	syscall(geteuid), 
#endif
#ifdef __NR_getegid
	syscall(getegid), 
#endif
#ifdef __NR_acct
	syscall(acct), 
#endif
#ifdef __NR_umount2
	syscall(umount2), 
#endif
#ifdef __NR_lock
	syscall(lock), 
#endif
#ifdef __NR_ioctl
	syscall(ioctl), 
#endif
#ifdef __NR_fcntl
	syscall(fcntl), 
#endif
#ifdef __NR_mpx
	syscall(mpx), 
#endif
#ifdef __NR_setpgid
	syscall(setpgid), 
#endif
#ifdef __NR_ulimit
	syscall(ulimit), 
#endif
#ifdef __NR_oldolduname
	syscall(oldolduname), 
#endif
#ifdef __NR_umask
	syscall(umask, print_octal), 
#endif
#ifdef __NR_chroot
	syscall(chroot), 
#endif
#ifdef __NR_ustat
	syscall(ustat), 
#endif
#ifdef __NR_dup2
	syscall(dup2), 
#endif
#ifdef __NR_getppid
	syscall(getppid), 
#endif
#ifdef __NR_getpgrp
	syscall(getpgrp), 
#endif
#ifdef __NR_setsid
	syscall(setsid), 
#endif
#ifdef __NR_sigaction
	syscall(sigaction), 
#endif
#ifdef __NR_sgetmask
	syscall(sgetmask), 
#endif
#ifdef __NR_ssetmask
	syscall(ssetmask), 
#endif
#ifdef __NR_setreuid
	syscall(setreuid, print_uid, print_uid), 
#endif
#ifdef __NR_setregid
	syscall(setregid, print_gid, print_gid), 
#endif
#ifdef __NR_sigsuspend
	syscall(sigsuspend), 
#endif
#ifdef __NR_sigpending
	syscall(sigpending), 
#endif
#ifdef __NR_sethostname
	syscall(sethostname, print_string), 
#endif
#ifdef __NR_setrlimit
	syscall(setrlimit, print_rlim_name, print_rlimit), 
#endif
#ifdef __NR_getrlimit
	syscall(getrlimit), 
#endif
#ifdef __NR_getrusage
	syscall(getrusage), 
#endif
#ifdef __NR_gettimeofday
	syscall(gettimeofday), 
#endif
#ifdef __NR_settimeofday
	syscall(settimeofday, print_timeval, print_timezone), 
#endif
#ifdef __NR_getgroups
	syscall(getgroups), 
#endif
#ifdef __NR_setgroups
	syscall(setgroups, NULL, print_groups), 
#endif
#ifdef __NR_select
	syscall(select), 
#endif
#ifdef __NR_symlink
	syscall(symlink), 
#endif
#ifdef __NR_oldlstat
	syscall(oldlstat), 
#endif
#ifdef __NR_readlink
	syscall(readlink), 
#endif
#ifdef __NR_uselib
	syscall(uselib), 
#endif
#ifdef __NR_swapon
	syscall(swapon), 
#endif
#ifdef __NR_reboot
	syscall(reboot), 
#endif
#ifdef __NR_readdir
	syscall(readdir), 
#endif
#ifdef __NR_mmap
	syscall(mmap), 
#endif
#ifdef __NR_munmap
	syscall(munmap), 
#endif
#ifdef __NR_truncate
	syscall(truncate), 
#endif
#ifdef __NR_ftruncate
	syscall(ftruncate), 
#endif
#ifdef __NR_fchmod
	syscall(fchmod, NULL, print_octal), 
#endif
#ifdef __NR_fchown
	syscall(fchown, NULL, print_uid, print_gid), 
#endif
#ifdef __NR_getpriority
	syscall(getpriority), 
#endif
#ifdef __NR_setpriority
	syscall(setpriority), 
#endif
#ifdef __NR_profil
	syscall(profil), 
#endif
#ifdef __NR_statfs
	syscall(statfs), 
#endif
#ifdef __NR_fstatfs
	syscall(fstatfs), 
#endif
#ifdef __NR_ioperm
	syscall(ioperm, print_hex), 
#endif
#ifdef __NR_socketcall
	syscall(socketcall), 
#endif
#ifdef __NR_syslog
	syscall(syslog), 
#endif
#ifdef __NR_setitimer
	syscall(setitimer), 
#endif
#ifdef __NR_getitimer
	syscall(getitimer), 
#endif
#ifdef __NR_stat
	syscall(stat), 
#endif
#ifdef __NR_lstat
	syscall(lstat), 
#endif
#ifdef __NR_fstat
	syscall(fstat), 
#endif
#ifdef __NR_olduname
	syscall(olduname), 
#endif
#ifdef __NR_iopl
	syscall(iopl), 
#endif
#ifdef __NR_vhangup
	syscall(vhangup), 
#endif
#ifdef __NR_idle
	syscall(idle), 
#endif
#ifdef __NR_vm86old
	syscall(vm86old), 
#endif
#ifdef __NR_wait4
	syscall(wait4), 
#endif
#ifdef __NR_swapoff
	syscall(swapoff), 
#endif
#ifdef __NR_sysinfo
	syscall(sysinfo), 
#endif
#ifdef __NR_ipc
	syscall(ipc), 
#endif
#ifdef __NR_fsync
	syscall(fsync), 
#endif
#ifdef __NR_sigreturn
	syscall(sigreturn), 
#endif
#ifdef __NR_clone
	syscall(clone, print_clone_flags), 
#endif
#ifdef __NR_clone2
	syscall(clone2, print_clone_flags), 
#endif
#ifdef __NR_setdomainname
	syscall(setdomainname, print_string), 
#endif
#ifdef __NR_uname
	syscall(uname), 
#endif
#ifdef __NR_modify_ldt
	syscall(modify_ldt), 
#endif
#ifdef __NR_adjtimex
	syscall(adjtimex), 
#endif
#ifdef __NR_mprotect
	syscall(mprotect), 
#endif
#ifdef __NR_sigprocmask
	syscall(sigprocmask), 
#endif
#ifdef __NR_create_module
	syscall(create_module), 
#endif
#ifdef __NR_init_module
	syscall(init_module), 
#endif
#ifdef __NR_delete_module
	syscall(delete_module), 
#endif
#ifdef __NR_get_kernel_syms
	syscall(get_kernel_syms), 
#endif
#ifdef __NR_quotactl
	syscall(quotactl), 
#endif
#ifdef __NR_getpgid
	syscall(getpgid), 
#endif
#ifdef __NR_fchdir
	syscall(fchdir), 
#endif
#ifdef __NR_bdflush
	syscall(bdflush), 
#endif
#ifdef __NR_sysfs
	syscall(sysfs), 
#endif
#ifdef __NR_personality
	syscall(personality), 
#endif
#ifdef __NR_afs_syscall
	syscall(afs_syscall), 
#endif
#ifdef __NR_setfsuid
	syscall(setfsuid), 
#endif
#ifdef __NR_setfsgid
	syscall(setfsgid), 
#endif
#ifdef __NR__llseek
	syscall(_llseek), 
#endif
#ifdef __NR_getdents
	syscall(getdents), 
#endif
#ifdef __NR__newselect
	syscall(_newselect), 
#endif
#ifdef __NR_flock
	syscall(flock), 
#endif
#ifdef __NR_msync
	syscall(msync), 
#endif
#ifdef __NR_readv
	syscall(readv), 
#endif
#ifdef __NR_writev
	syscall(writev), 
#endif
#ifdef __NR_getsid
	syscall(getsid), 
#endif
#ifdef __NR_fdatasync
	syscall(fdatasync), 
#endif
#ifdef __NR__sysctl
	syscall(_sysctl), 
#endif
#ifdef __NR_mlock
	syscall(mlock), 
#endif
#ifdef __NR_munlock
	syscall(munlock), 
#endif
#ifdef __NR_mlockall
	syscall(mlockall), 
#endif
#ifdef __NR_munlockall
	syscall(munlockall), 
#endif
#ifdef __NR_sched_setparam
	syscall(sched_setparam), 
#endif
#ifdef __NR_sched_getparam
	syscall(sched_getparam), 
#endif
#ifdef __NR_sched_setscheduler
	syscall(sched_setscheduler), 
#endif
#ifdef __NR_sched_getscheduler
	syscall(sched_getscheduler), 
#endif
#ifdef __NR_sched_yield
	syscall(sched_yield), 
#endif
#ifdef __NR_sched_get_priority_max
	syscall(sched_get_priority_max), 
#endif
#ifdef __NR_sched_get_priority_min
	syscall(sched_get_priority_min), 
#endif
#ifdef __NR_sched_rr_get_interval
	syscall(sched_rr_get_interval), 
#endif
#ifdef __NR_nanosleep
	syscall(nanosleep), 
#endif
#ifdef __NR_mremap
	syscall(mremap), 
#endif
#ifdef __NR_setresuid
	syscall(setresuid, print_uid, print_uid, print_uid), 
#endif
#ifdef __NR_getresuid
	syscall(getresuid), 
#endif
#ifdef __NR_vm86
	syscall(vm86), 
#endif
#ifdef __NR_query_module
	syscall(query_module), 
#endif
#ifdef __NR_poll
	syscall(poll), 
#endif
#ifdef __NR_nfsservctl
	syscall(nfsservctl), 
#endif
#ifdef __NR_setresgid
	syscall(setresgid, print_gid, print_gid, print_gid), 
#endif
#ifdef __NR_getresgid
	syscall(getresgid), 
#endif
#ifdef __NR_prctl
	syscall(prctl), 
#endif
#ifdef __NR_rt_sigreturn
	syscall(rt_sigreturn), 
#endif
#ifdef __NR_rt_sigaction
	syscall(rt_sigaction), 
#endif
#ifdef __NR_rt_sigprocmask
	syscall(rt_sigprocmask), 
#endif
#ifdef __NR_rt_sigpending
	syscall(rt_sigpending), 
#endif
#ifdef __NR_rt_sigtimedwait
	syscall(rt_sigtimedwait), 
#endif
#ifdef __NR_rt_sigqueueinfo
	syscall(rt_sigqueueinfo), 
#endif
#ifdef __NR_rt_sigsuspend
	syscall(rt_sigsuspend), 
#endif
#ifdef __NR_pread
	syscall(pread), 
#endif
#ifdef __NR_pwrite
	syscall(pwrite), 
#endif
#ifdef __NR_chown
	syscall(chown, NULL, print_uid, print_gid), 
#endif
#ifdef __NR_getcwd
	syscall(getcwd), 
#endif
#ifdef __NR_capget
	syscall(capget), 
#endif
#ifdef __NR_capset
	syscall(capset, print_cap_header, print_cap_data), 
#endif
#ifdef __NR_sigaltstack
	syscall(sigaltstack), 
#endif
#ifdef __NR_sendfile
	syscall(sendfile, NULL, NULL, print_ref_off_t), 
#endif
#ifdef __NR_getpmsg
	syscall(getpmsg), 
#endif
#ifdef __NR_putpmsg
	syscall(putpmsg), 
#endif
#ifdef __NR_vfork
	syscall(vfork), 
#endif
#ifdef __NR_ugetrlimit
	syscall(ugetrlimit), 
#endif
#ifdef __NR_mmap2
	syscall(mmap2), 
#endif
#ifdef __NR_truncate64
	syscall(truncate64), 
#endif
#ifdef __NR_ftruncate64
	syscall(ftruncate64), 
#endif
#ifdef __NR_stat64
	syscall(stat64), 
#endif
#ifdef __NR_lstat64
	syscall(lstat64), 
#endif
#ifdef __NR_fstat64
	syscall(fstat64), 
#endif
#ifdef __NR_pciconfig_read
	syscall(pciconfig_read),
#endif
#ifdef __NR_pciconfig_write
	syscall(pciconfig_write),
#endif
#ifdef __NR_pciconfig_iobase
	syscall(pciconfig_iobase),
#endif
#ifdef __NR_multiplexer
	syscall(multiplexer),
#endif
#ifdef __NR_lchown32
	syscall(lchown32, NULL, print_uid, print_gid), 
#endif
#ifdef __NR_getuid32
	syscall(getuid32), 
#endif
#ifdef __NR_getgid32
	syscall(getgid32), 
#endif
#ifdef __NR_geteuid32
	syscall(geteuid32), 
#endif
#ifdef __NR_getegid32
	syscall(getegid32), 
#endif
#ifdef __NR_setreuid32
	syscall(setreuid32, print_uid, print_uid), 
#endif
#ifdef __NR_setregid32
	syscall(setregid32, print_gid, print_gid), 
#endif
#ifdef __NR_getgroups32
	syscall(getgroups32), 
#endif
#ifdef __NR_setgroups32
	syscall(setgroups32, NULL, print_groups32), 
#endif
#ifdef __NR_fchown32
	syscall(fchown32, NULL, print_uid, print_gid), 
#endif
#ifdef __NR_setresuid32
	syscall(setresuid32, print_uid, print_uid, print_uid), 
#endif
#ifdef __NR_getresuid32
	syscall(getresuid32), 
#endif
#ifdef __NR_setresgid32
	syscall(setresgid32, print_gid, print_gid, print_gid), 
#endif
#ifdef __NR_getresgid32
	syscall(getresgid32), 
#endif
#ifdef __NR_chown32
	syscall(chown32, NULL, print_uid, print_gid), 
#endif
#ifdef __NR_setuid32
	syscall(setuid32), 
#endif
#ifdef __NR_setgid32
	syscall(setgid32), 
#endif
#ifdef __NR_setfsuid32
	syscall(setfsuid32), 
#endif
#ifdef __NR_setfsgid32
	syscall(setfsgid32), 
#endif
#ifdef __NR_pivot_root
	syscall(pivot_root), 
#endif
#ifdef __NR_mincore
	syscall(mincore), 
#endif
#ifdef __NR_madvise
	syscall(madvise), 
#endif
#ifdef __NR_madvise1
	syscall(madvise1), 
#endif
#ifdef __NR_getdents64
	syscall(getdents64), 
#endif
#ifdef __NR_fcntl64
	syscall(fcntl64), 
#endif
#ifdef __NR_security
	syscall(security), 
#endif
#ifdef __NR_tuxcall
	syscall(tuxcall),
#endif
#ifdef __NR_io_setup
	syscall(io_setup),
#endif
#ifdef __NR_io_destroy
	syscall(io_destroy),
#endif
#ifdef __NR_io_getevents
	syscall(io_getevents),
#endif
#ifdef __NR_io_submit
	syscall(io_submit),
#endif
#ifdef __NR_io_cancel
	syscall(io_cancel),
#endif
#ifdef __NR_set_tid_address
	syscall(set_tid_address),
#endif
#ifdef __NR_fadvise64
	syscall(fadvise64),
#endif
#ifdef __NR_exit_group
	syscall(exit_group),
#endif
#ifdef __NR_lookup_dcookie
	syscall(lookup_dcookie),
#endif
#ifdef __NR_epoll_create
	syscall(epoll_create),
#endif
#ifdef __NR_epoll_ctl
	syscall(epoll_ctl),
#endif
#ifdef __NR_epoll_wait
	syscall(epoll_wait),
#endif
/* Some platforms have __NR_sys_epoll_* instead of __NR_epoll_* */
#ifdef __NR_sys_epoll_create
	syscall(sys_epoll_create),
#endif
#ifdef __NR_sys_epoll_ctl
	syscall(sys_epoll_ctl),
#endif
#ifdef __NR_sys_epoll_wait
	syscall(sys_epoll_wait),
#endif
#ifdef __NR_remap_file_pages
	syscall(remap_file_pages),
#endif
#ifdef __NR_timer_create
	syscall(timer_create),
#endif
#ifdef __NR_timer_settime
	syscall(timer_settime),
#endif
#ifdef __NR_timer_gettime
	syscall(timer_gettime),
#endif
#ifdef __NR_timer_getoverrun
	syscall(timer_getoverrun),
#endif
#ifdef __NR_timer_delete
	syscall(timer_delete),
#endif
#ifdef __NR_clock_settime
	syscall(clock_settime),
#endif
#ifdef __NR_clock_gettime
	syscall(clock_gettime),
#endif
#ifdef __NR_clock_getres
	syscall(clock_getres),
#endif
#ifdef __NR_clock_nanosleep
	syscall(clock_nanosleep),
#endif
#ifdef __NR_swapcontext
	syscall(swapcontext),
#endif
#ifdef __NR_tgkill
	syscall(tgkill),
#endif
#ifdef __NR_utimes
	syscall(utimes),
#endif
#ifdef __NR_statfs64
	syscall(statfs64),
#endif
#ifdef __NR_fstatfs64
	syscall(fstatfs64),
#endif
#ifdef __NR_gettid
	syscall(gettid), 
#endif
#ifdef __NR_readahead
	syscall(readahead), 
#endif
#ifdef __NR_setxattr
	syscall(setxattr, NULL, NULL, print_string), 
#endif
#ifdef __NR_lsetxattr
	syscall(lsetxattr), 
#endif
#ifdef __NR_fsetxattr
	syscall(fsetxattr, NULL, NULL, print_string), 
#endif
#ifdef __NR_getxattr
	syscall(getxattr), 
#endif
#ifdef __NR_lgetxattr
	syscall(lgetxattr), 
#endif
#ifdef __NR_fgetxattr
	syscall(fgetxattr), 
#endif
#ifdef __NR_listxattr
	syscall(listxattr), 
#endif
#ifdef __NR_llistxattr
	syscall(llistxattr), 
#endif
#ifdef __NR_flistxattr
	syscall(flistxattr), 
#endif
#ifdef __NR_removexattr
	syscall(removexattr), 
#endif
#ifdef __NR_lremovexattr
	syscall(lremovexattr), 
#endif
#ifdef __NR_fremovexattr
	syscall(fremovexattr), 
#endif
#ifdef __NR_tkill
	syscall(tkill), 
#endif
#ifdef __NR_sendfile64
	syscall(sendfile64), 
#endif
#ifdef __NR_futex
	syscall(futex), 
#endif
#ifdef __NR_sched_setaffinity
	syscall(sched_setaffinity), 
#endif
#ifdef __NR_sched_getaffinity
	syscall(sched_getaffinity), 
#endif
#ifdef __NR_set_thread_area
	syscall(set_thread_area),
#endif
#ifdef __NR_get_thread_area
	syscall(get_thread_area),
#endif
#ifdef __NR_alloc_hugepages
	syscall(alloc_hugepages),
#endif
#ifdef __NR_free_hugepages
	syscall(free_hugepages),
#endif

/*
 * Socket calls
 */
syscall(socket, print_af, print_socktype),
syscall(bind, NULL, print_sockaddr),
syscall(connect, NULL, print_sockaddr),
syscall(listen),
syscall(accept),
syscall(socketpair, print_af),
syscall(setsockopt),
syscall(getsockopt),
syscall(getsockname),
syscall(getpeername),
#ifdef __NR_send
syscall(send),
#endif
#ifdef __NR_recv
syscall(recv),
#endif
syscall(sendto),
syscall(recvfrom),
syscall(sendmsg),
syscall(recvmsg),

/*
 * SysV IPC calls
 */
syscall(semop), 
syscall(semtimedop),
syscall(semget, print_hex), 
syscall(semctl, print_hex, NULL, NULL /* cmd */), 
syscall(msgsnd, NULL, NULL, NULL, print_hex), 
syscall(msgrcv, NULL, NULL, NULL, print_hex, print_hex), 
syscall(msgget, print_hex, NULL /* flags */ ), 
syscall(msgctl, print_hex, NULL, NULL /* msgqid_ds */), 
syscall(shmat, print_hex), 
syscall(shmdt, print_hex), 
syscall(shmget, print_hex), 
syscall(shmctl, print_hex), 
};

#undef syscall

#define function(n, args...)	[AUDIT_##n] = { #n, { args } }

static const struct syscall_info function_table[audit_NUM_CALLS] = {
	function(access),
	function(acct),
	function(adjtimex),
	function(bind, NULL, print_sockaddr),
	function(brk), 
	function(capset, NULL, NULL, print_ecap, print_icap, print_pcap),
	function(chdir),
	function(chmod, NULL, print_octal),
	function(chown, NULL, print_uid, print_gid),
	function(chroot), 
	function(clone, print_clone_flags), 
	function(delete_module),
	function(execve), 
	function(fchdir),
	function(fchmod, NULL, print_octal),
	function(fchown, NULL, print_uid, print_gid),
	function(fremovexattr),
	function(fsetxattr),
	function(ftruncate),
	function(init_module),
	function(ioctl),
	function(ioperm, print_hex),
	function(iopl),
	function(kill, NULL, print_signal), 
	function(lchmod, NULL, print_octal),
	function(lchown, NULL, print_uid, print_gid),
	function(link),
	function(lremovexattr),
	function(lsetxattr),
	function(mkdir, NULL, print_octal),
	function(mknod, NULL, print_octal, print_hex),
	function(mount),
	function(msgctl, print_hex), 
	function(msgget, print_hex, NULL /* flags */ ), 
	function(msgrcv, NULL, NULL, NULL, print_hex, print_hex), 
	function(msgsnd, NULL, NULL, NULL, print_hex), 
	function(open, NULL, print_open_mode, print_octal),
	function(ptrace), 
	function(quotactl),
	function(reboot),
	function(removexattr),
	function(rename),
	function(rmdir),
	function(semctl, print_hex, NULL, NULL /* cmd */),
	function(semget, print_hex),
	function(semop),
	function(semtimedop, NULL, NULL, NULL, print_timespec),
	function(setdomainname, print_string), 
	function(setfsgid, print_gid),
	function(setfsuid, print_uid),
	function(setgid, print_gid),
	function(setgroups, print_groups32),
	function(sethostname, print_string), 
	function(setpgid),
	function(setpriority),
	function(setregid, print_gid, print_gid),
	function(setresgid, print_gid, print_gid, print_gid),
	function(setresuid, print_uid, print_uid, print_uid),
	function(setreuid, print_uid, print_uid),
	function(setrlimit, print_rlim_name, print_rlimit), 
	function(setsid),
	function(settimeofday, print_timeval, print_timezone),
	function(setuid, print_uid),
	function(setxattr),
	function(shmat, print_hex),
	function(shmctl, print_hex),
	function(shmdt),
	function(shmget, print_hex),
	function(socket, print_af, print_socktype),
	function(swapoff),
	function(swapon),
	function(symlink),
	function(syslog),
	function(tgkill, NULL, NULL, print_signal), 
	function(tkill, NULL, print_signal), 
	function(truncate),
	function(umask, print_octal), 
	function(umount),
	function(unlink),
	function(utimes),
};

#undef function

/*
 * Handle system call/code translations
 */
static __inline__ const char *
__code_to_name(unsigned int code, unsigned int low, unsigned int high)
{
	code += low;
	return (code < high)? syscall_table[code].name : NULL;
}

static __inline__ int
__name_to_code(const char *name, const struct syscall_info *table, unsigned int low, unsigned int high)
{
	unsigned int	n;

	for (n = low; n < high; n++) {
		if (table[n].name && !strcmp(table[n].name, name))
			return n - low;
	}
	return -1;
}

unsigned int
syscall_max(void)
{
	return ABSTRACT_SYSCALLS ? audit_NUM_CALLS : __syscall_end;
}

const char *
syscall_code_to_name(unsigned int code)
{
	if (ABSTRACT_SYSCALLS)
		return code < ARRAY_SIZE(function_table) ? function_table[code].name : NULL;
	return __code_to_name(code, 0, __syscall_end);
}

int
syscall_name_to_code(const char *name)
{
	if (ABSTRACT_SYSCALLS)
		return __name_to_code(name, function_table, 0, ARRAY_SIZE(function_table));
	return __name_to_code(name, syscall_table, 0, __syscall_end);
}

unsigned int
socketcall_max(void)
{
	return ABSTRACT_SYSCALLS ? 0 : __socketcall_end - __socketcall_base;
}

const char *
socketcall_code_to_name(unsigned int code)
{
	if (ABSTRACT_SYSCALLS)
		return NULL;
	return __code_to_name(code, __socketcall_base, __socketcall_end);
}

int
socketcall_name_to_code(const char *name)
{
	if (ABSTRACT_SYSCALLS)
		return -1;
	return __name_to_code(name, syscall_table, __socketcall_base, __socketcall_end);
}

unsigned int
ipccall_max(void)
{
	return ABSTRACT_SYSCALLS ? 0 : __ipccall_end - __ipccall_base;
}

const char *
ipccall_code_to_name(unsigned int code)
{
	if (ABSTRACT_SYSCALLS)
		return NULL;
	return __code_to_name(code, __ipccall_base, __ipccall_end);
}

int
ipccall_name_to_code(const char *name)
{
	if (ABSTRACT_SYSCALLS)
		return -1;
	return __name_to_code(name, syscall_table, __ipccall_base, __ipccall_end);
}

#define EVENT_BIAS (__AUD_POLICY_LAST_SYSCALL + 1)
#define event(code, name) [code - EVENT_BIAS] = #name

static const char *const event_table[] = {
	event(AUD_POLICY_FORK, process-create),
	event(AUD_POLICY_EXIT, process-exit),
	event(AUD_POLICY_LOGIN, process-login),
	event(AUD_POLICY_NETLINK, network-config),
	event(AUD_POLICY_USERMSG, user-message),
};

#undef event

unsigned int
event_min(void)
{
	return EVENT_BIAS;
}

unsigned int
event_max(void)
{
	return EVENT_BIAS + ARRAY_SIZE(event_table);
}

const char *
event_code_to_name(unsigned int code)
{
	return code >= EVENT_BIAS && code < EVENT_BIAS + ARRAY_SIZE(event_table) ? event_table[code - EVENT_BIAS] : NULL;
}

int
event_name_to_code(const char *name)
{
	unsigned i;

	for (i = 0; i < ARRAY_SIZE(event_table); ++i) {
		if (event_table[i] && !strcmp(event_table[i], name))
			return i + EVENT_BIAS;
	}
	return -1;
}

#undef EVENT_BIAS

/*
 * Print various types of system call arguments
 */
static const void *
get_value(const void *p, size_t len, u_int64_t *res)
{
	const unsigned char	*addr;

	addr = (const unsigned char *) p;
	if (len == 1) {
		*res = *addr;
	} else if (len == 2) {
		u_int16_t tmp;

		memcpy(&tmp, addr, 2);
		*res = tmp;
	} else if (len == 4) {
		u_int32_t tmp;

		memcpy(&tmp, addr, 4);
		*res = tmp;
	} else if (len == 8) {
		u_int64_t tmp;

		memcpy(&tmp, addr, 8);
		*res = tmp;
	}

	return addr + len;
}

static int
get_immediate(int type, const void *p, size_t len, u_int64_t *res)
{
	if (type != AUDIT_ARG_IMMEDIATE)
		return -1;

	get_value(p, len, res);
	return 0;
}

int
print_error(int type, const void *p, size_t len)
{
	u_int64_t	code;

	if (type != AUDIT_ARG_ERROR)
		return -1;

	get_value(p, len, &code);
	printf("[error=%s]", strerror(code));
	return 0;
}

void
__print_integer(const char *fmt, const void *p, size_t len)
{
	u_int64_t	value;

	if (get_immediate(AUDIT_ARG_IMMEDIATE, p, len, &value) < 0) {
		printf("[immediate,len %d]", len);
	} else {
		printf(fmt, value);
	}
}

int
print_integer(int type, const void *p, size_t len)
{
	if (type != AUDIT_ARG_IMMEDIATE)
		return -1;

	__print_integer("%lld", p, len);
	return 0;
}

int
print_string(int type, const void *p, size_t len)
{
	unsigned int	j;

	if (type != AUDIT_ARG_STRING
	 && type != AUDIT_ARG_PATH
	 && type != AUDIT_ARG_POINTER)
		return -1;

	printf("\"");
	for (j = 0; j < len && j < 128; j++) {
		unsigned char	c;

		c = *((unsigned char *) p++);
		if (!isprint(c)) {
			printf("\\%03o", c);
		} else
		if (c == '"' || c == '\\') {
			printf("\\%c", c);
		} else {
			printf("%c", c);
		}
	}
	printf("\"");
	if (j < len)
		printf("...");

	return 0;
}

int
print_opaque(int type, const void *p, size_t len)
{
	if (type != AUDIT_ARG_STRING
	 && type != AUDIT_ARG_PATH
	 && type != AUDIT_ARG_POINTER)
		return -1;

	printf("[data, len=%lu]", (unsigned long) len);
	return 0;
}

static int
print_vector(int type, const void *p, size_t len)
{
	u_int32_t	e_type, e_size;
	size_t		used;

	if (type != AUDIT_ARG_VECTOR)
		return -1;

	printf("[");
	for (used = 0; used + 8 < len; ) {
		const caddr_t	e = (const caddr_t) p + used;

		memcpy(&e_type, e, 4);
		memcpy(&e_size, e + 4, 4);

		if (used)
			printf(", ");
		if (used + e_size > len) {
			printf("...");
			break;
		}
		print_default(e_type, e + 8, e_size);
		used += e_size + 8;
	}
	printf("]");

	return 0;
}

int
print_default(int type, const void *p, size_t len)
{
	switch (type) {
	case AUDIT_ARG_IMMEDIATE:
		print_integer(type, p, len);
		break;

	case AUDIT_ARG_STRING:
	case AUDIT_ARG_PATH:
		print_string(type, p, len);
		break;

	case AUDIT_ARG_POINTER:
		/* Print unknown data as opaque. If you want
		 * to print an argument the kernel flags as
		 * "pointer" (e.g. in the case of sethostname)
		 * you need to force the string interpretation
		 * in the syscall_table above */
		print_opaque(type, p, len);
		break;

	case AUDIT_ARG_NULL:
		printf("NULL");
		break;

	case AUDIT_ARG_ERROR:
		print_error(type, p, len);
		break;

	case AUDIT_ARG_VECTOR:
		print_vector(type, p, len);
		break;

	default:
		printf("???");
		break;
	}

	return 0;
}

/*
 * Print primitives
 */
void
__print_bits(u_int64_t value, const struct bitname *bit)
{
	const char	*sep = "";

	for (; bit->name; bit++) {
		if (!((bit->value ^ value) & bit->mask)) {
			printf("%s%s", sep, bit->name);
			value &= ~bit->value;
			sep = "|";
		}
	}
	if (value)
		printf("%s0x%llx", sep, value);
	else if (*sep == '\0')
		printf("0");
}

void
__print_symbolic(u_int64_t value, const struct symbol *sym)
{
	for (; sym->name; sym++) {
		if (value == sym->value) {
			printf("%s", sym->name);
			return;
		}
	}
	printf("%llu", value);
}

void
__print_hwaddr(const struct sockaddr *ap, size_t len)
{
	const unsigned char *data;
	unsigned int	i, hwlen;

	switch (ap->sa_family) {
	case 0:	/* XXX: should guess hw type based on device name.
		 * For now, fall through to ethernet case */
	case ARPHRD_ETHER:
		hwlen = 6;
		break;

	default:
		/* as good as any other default */
		hwlen = 6;
		break;
	}

	len -= offsetof(struct sockaddr, sa_data);
	if (len > hwlen)
		len = hwlen;

	data = ap->sa_data;
	for (i = 0; i < len; i++) {
		if (i)
			printf(":");
		printf("%02x", data[i]);
	}
}

void
__print_sockaddr(const struct sockaddr *ap, size_t len)
{
	const struct sockaddr_in	*sin;
	const struct sockaddr_in6	*six;
	const struct sockaddr_un	*sux;
	unsigned int			i;
	char				abuf[INET6_ADDRSTRLEN+1];

	/* assume family is first member in struct.
	 * This is a safe assumption, as AF_LOCAL handling
	 * relies on it */
	if (len < offsetof(struct sockaddr, sa_data))
		goto truncated;

	switch (ap->sa_family) {
	case AF_UNSPEC:
		/* XXX we have no way of finding out the length of
		 * the HW addr, assume it's Ethernet */
		__print_hwaddr(ap, len);
		break;
	case AF_INET:
		if (len < sizeof(*sin))
			goto truncated;
		sin = (const struct sockaddr_in *) ap;
		printf("%s", inet_ntoa(sin->sin_addr));
		if (sin->sin_port)
			printf(":%u", ntohs(sin->sin_port));
		break;
	case AF_INET6:
		if (len < sizeof(*six))
			goto truncated;
		six = (const struct sockaddr_in6 *) ap;
		inet_ntop(AF_INET6, &six->sin6_addr, abuf, sizeof(abuf));
		printf("%s", abuf);
		if (six->sin6_port)
			printf(":%u", ntohs(six->sin6_port));
		/* XXX print scope */
		break;
	case AF_LOCAL:
		if (len < offsetof(struct sockaddr_un, sun_path))
			goto truncated;
		sux = (const struct sockaddr_un *) ap;
		print_string(AUDIT_ARG_STRING, sux->sun_path,
			       	len-offsetof(struct sockaddr_un, sun_path));
		break;
	default:
		printf("[");
		if (print_af(AUDIT_ARG_IMMEDIATE,
			       	&ap->sa_family, sizeof(ap->sa_family)) < 0)
			printf("AF_%u", ap->sa_family);
		printf("]");
		break;
	}

	return;

truncated:
	printf("[TRUNCATED SOCKADDR:");
	if (len == 0)
		printf(" len=%u", len);
	for (i = 0; i < len; i++)
		printf(" %02x", ((const unsigned char *) ap)[i]);
	printf("]");
}

void
__print_netaddr(int af, const void *p, size_t len)
{
	char	namebuf[INET6_ADDRSTRLEN+1];

	if ((af == AF_INET && len != 4)
	 || (af == AF_INET6 && len != 16)) {
		printf("[BAD ADDRESS, af=");
		__print_af(af);
		if (len == 0) {
			printf(", no data");
		} else {
			unsigned int	i;

			printf(", data=");
			for (i = 0; i < len; i++) {
				printf(" %02x",
					((const unsigned char *) p)[i]);
			}
		}
		printf("]");
		return;
	}

	printf("%s", inet_ntop(af, p, namebuf, sizeof(namebuf)));
}

/*
 * Print functions for system call arguments
 */
int
print_bits(int type, const void *p, size_t len, const struct bitname *bit)
{
	u_int64_t	value;

	if (get_immediate(type, p, len, &value) < 0)
		return -1;

	__print_bits(value, bit);
	return 0;
}

int
print_symbolic(int type, const void *p, size_t len, const struct symbol *sym)
{
	u_int64_t	value;

	if (get_immediate(type, p, len, &value) < 0)
		return -1;

	__print_symbolic(value, sym);
	return 0;
}

int
print_array(int type, const void *p, size_t len,
		syscall_print_fn_t *print,
	       	unsigned int esize)
{
	unsigned int	j, count;

	if (type != AUDIT_ARG_POINTER || (len % esize))
		return -1;

	count = len / esize;
	printf("[");
	for (j = 0; j < count; j++) {
		if (j)
			printf(", ");
		if (print(AUDIT_ARG_IMMEDIATE, p, esize) < 0)
			print_default(AUDIT_ARG_IMMEDIATE, p, esize);
		p = (const unsigned char *) p + esize;
	}
	printf("]");
	return 0;
}

static const struct bitname	open_bits[] = {
	{ "O_RDONLY",	O_RDONLY,	O_ACCMODE	},
	{ "O_WRONLY",	O_WRONLY,	O_ACCMODE	},
	{ "O_RDWR",	O_RDWR,		O_ACCMODE	},
	defbit(O_CREAT),
	defbit(O_EXCL),
	defbit(O_NOCTTY),
	defbit(O_TRUNC),
	defbit(O_APPEND),
	defbit(O_NONBLOCK),
	defbit(O_SYNC),
	defbit(O_ASYNC),
	defbit(O_DIRECTORY),
#ifdef O_DIRECT
	defbit(O_DIRECT),
#endif
#ifdef O_DIRECTORY
	defbit(O_DIRECTORY),
#endif
	defbit(O_NOFOLLOW),
	defbit(O_LARGEFILE),

	{ NULL, }
};

int
print_open_mode(int type, const void *p, size_t len)
{
	return print_bits(type, p, len, open_bits);
}

int
print_octal(int type, const void *p, size_t len)
{
	u_int64_t	value;

	if (get_immediate(type, p, len, &value) < 0)
		return -1;
	if (value)
		printf("0%llo", value);
	else
		printf("0");
	return 0;
}

int
print_hex(int type, const void *p, size_t len)
{
	u_int64_t	value;

	if (get_immediate(type, p, len, &value) < 0)
		return -1;
	printf("0x%llx", value);
	return 0;
}

static const struct symbol	af_names[] = {
	defsym(AF_INET),
	defsym(AF_INET6),
	defsym(AF_LOCAL),
	defsym(AF_PACKET),
	defsym(AF_NETLINK),

	{ NULL }
};

void
__print_af(int af)
{
	__print_symbolic(af, af_names);
}

int
print_af(int type, const void *p, size_t len)
{
	return print_symbolic(type, p, len, af_names);
}

static const struct symbol	socktype_names[] = {
	defsym(SOCK_STREAM),
	defsym(SOCK_DGRAM),
	defsym(SOCK_RAW),
	defsym(SOCK_RDM),
	defsym(SOCK_SEQPACKET),

	{ NULL }
};

int
print_socktype(int type, const void *p, size_t len)
{
	return print_symbolic(type, p, len, socktype_names);
}

int
print_sockaddr(int type, const void *p, size_t len)
{
	if (type != AUDIT_ARG_POINTER)
		return -1;

	__print_sockaddr((const struct sockaddr *) p, len);
	return 0;
}

int
print_gid(int type, const void *p, size_t len)
{
	u_int64_t	gid;

	if (get_immediate(type, p, len, &gid) < 0)
		return -1;

	/* XXX: look up group name */
	printf("%d", (int) gid);
	return 0;
}

int
print_uid(int type, const void *p, size_t len)
{
	u_int64_t	uid;

	if (get_immediate(type, p, len, &uid) < 0)
		return -1;

	/* XXX: look up user name */
	printf("%d", (int) uid);
	return 0;
}

int
print_groups32(int type, const void *p, size_t len)
{
	return print_array(type, p, len, print_gid, sizeof(u_int32_t));
}

int
print_groups(int type, const void *p, size_t len)
{
#if defined(i386)
	return print_array(type, p, len, print_gid, sizeof(u_int16_t));
#else
	return print_array(type, p, len, print_gid, sizeof(gid_t));
#endif
}

static const struct symbol	rlim_names[] = {
	defsym(RLIMIT_CPU),
	defsym(RLIMIT_FSIZE),
	defsym(RLIMIT_DATA),
	defsym(RLIMIT_STACK),
	defsym(RLIMIT_CORE),
	defsym(RLIMIT_RSS),
	defsym(RLIMIT_NPROC),
	defsym(RLIMIT_NOFILE),
	defsym(RLIMIT_LOCKS),
	defsym(RLIMIT_AS),
	defsym(RLIMIT_MEMLOCK),

	{ NULL }
};

int
print_rlim_name(int type, const void *p, size_t len)
{
	return print_symbolic(type, p, len, rlim_names);
}

#ifdef emulate32bit
struct rlimit32 {
	uint32_t	rlim_cur;
	uint32_t	rlim_max;
};

static const void *
conv32_rlimit(const void *p, size_t *lenp)
{
	static struct rlimit	copy;
	const struct rlimit32	*rlim;

	if (*lenp != sizeof(*rlim))
		return NULL;
	rlim = (const struct rlimit32 *) p;
	copy.rlim_cur = rlim->rlim_cur;
	copy.rlim_max = rlim->rlim_max;
	*lenp = sizeof(copy);
	return &copy;
}
#else
#define conv32_rlimit(p, lenp)	(p)
#endif

int
print_rlimit(int type, const void *p, size_t len)
{
	const struct rlimit	*rlim;

	if (type != AUDIT_ARG_POINTER
	 || !(p = conv32_rlimit(p, &len))
	 || len != sizeof(struct rlimit))
		return -1;

	rlim = (const struct rlimit *) p;
	if (rlim->rlim_cur == RLIM_INFINITY)
		printf("cur=infinite, ");
	else
		printf("cur=%lu, ", (unsigned long) rlim->rlim_cur);
	if (rlim->rlim_max == RLIM_INFINITY)
		printf("max=infinite]");
	else
		printf("max=%lu]", (unsigned long) rlim->rlim_max);
	return 0;
}

int
print_cap_header(int type, const void *p, size_t len)
{
	cap_user_header_t c;

	if (len != sizeof(*c))
		return -1;
	c = (cap_user_header_t) p;
	if (c->version != _LINUX_CAPABILITY_VERSION)
		return -1;
	printf("[pid=%d]", c->pid);
	return 0;
}

static const struct bitname	cap_names[] = {
	defbit(CAP_CHOWN),
	defbit(CAP_DAC_OVERRIDE),
	defbit(CAP_DAC_READ_SEARCH),
	defbit(CAP_FOWNER),
	defbit(CAP_FSETID),
	defbit(CAP_FS_MASK),
	defbit(CAP_KILL),
	defbit(CAP_SETGID),
	defbit(CAP_SETUID),
	defbit(CAP_SETPCAP),
	defbit(CAP_LINUX_IMMUTABLE),
	defbit(CAP_NET_BIND_SERVICE),
	defbit(CAP_NET_BROADCAST),
	defbit(CAP_NET_ADMIN),
	defbit(CAP_NET_RAW),
	defbit(CAP_IPC_LOCK),
	defbit(CAP_IPC_OWNER),
	defbit(CAP_SYS_MODULE),
	defbit(CAP_SYS_RAWIO),
	defbit(CAP_SYS_CHROOT),
	defbit(CAP_SYS_PTRACE),
	defbit(CAP_SYS_PACCT),
	defbit(CAP_SYS_ADMIN),
	defbit(CAP_SYS_BOOT),
	defbit(CAP_SYS_NICE),
	defbit(CAP_SYS_RESOURCE),
	defbit(CAP_SYS_TIME),
	defbit(CAP_SYS_TTY_CONFIG),
	defbit(CAP_MKNOD),
	defbit(CAP_LEASE),
	{ NULL }
};

int
print_cap_data(int type, const void *p, size_t len)
{
	cap_user_data_t c;

	if (len != sizeof(*c))
		return -1;
	c = (cap_user_data_t) p;
	printf("[effective=");
	__print_bits(c->effective, cap_names);
	printf(", permitted=");
	__print_bits(c->permitted, cap_names);
	printf(", inheritable=");
	__print_bits(c->permitted, cap_names);
	printf("]");
	return 0;
}

int
print_ecap(int type, const void *p, size_t len)
{
	if (len != sizeof(uint32_t))
		return -1;
	printf("[effective] ");
	__print_bits(*(const uint32_t *)p, cap_names);
	return 0;
}

int
print_icap(int type, const void *p, size_t len)
{
	if (len != sizeof(uint32_t))
		return -1;
	printf("[inheritable] ");
	__print_bits(*(const uint32_t *)p, cap_names);
	return 0;
}

int
print_pcap(int type, const void *p, size_t len)
{
	if (len != sizeof(uint32_t))
		return -1;
	printf("[permitted] ");
	__print_bits(*(const uint32_t *)p, cap_names);
	return 0;
}

static const struct bitname	clone_bits[] = {
	defbit(CLONE_VM),
	defbit(CLONE_FS),
	defbit(CLONE_FILES),
	defbit(CLONE_SIGHAND),
#ifdef CLONE_IDLETASK
	defbit(CLONE_IDLETASK),
#endif
#ifdef CLONE_PID
	defbit(CLONE_PID),
#endif
	defbit(CLONE_PTRACE),
	defbit(CLONE_VFORK),
#ifdef CLONE_PARENT
	/* These are not defined in older glibc headers */
	defbit(CLONE_PARENT),
	defbit(CLONE_THREAD),
	defbit(CLONE_NEWNS),
	defbit(CLONE_SYSVSEM),
	defbit(CLONE_SETTLS),
	defbit(CLONE_PARENT_SETTID),
	defbit(CLONE_CHILD_CLEARTID),
	defbit(CLONE_DETACHED),
	defbit(CLONE_UNTRACED),
	defbit(CLONE_CHILD_SETTID),
#endif
#ifdef CLONE_STOPPED
	defbit(CLONE_STOPPED),
#endif
	{ NULL }
};

int
print_clone_flags(int type, const void *p, size_t len)
{
	return print_bits(type, p, len, clone_bits);
}

static const struct symbol	sig_names[] = {
	defsym(SIGHUP),
	defsym(SIGINT),
	defsym(SIGQUIT),
	defsym(SIGILL),
	defsym(SIGTRAP),
	defsym(SIGABRT),
	defsym(SIGBUS),
	defsym(SIGFPE),
	defsym(SIGKILL),
	defsym(SIGUSR1),
	defsym(SIGSEGV),
	defsym(SIGUSR2),
	defsym(SIGPIPE),
	defsym(SIGALRM),
	defsym(SIGTERM),
	defsym(SIGCHLD),
	defsym(SIGCONT),
	defsym(SIGSTOP),
	defsym(SIGTSTP),
	defsym(SIGTTIN),
	defsym(SIGTTOU),
	defsym(SIGURG),
	defsym(SIGXCPU),
	defsym(SIGXFSZ),
	defsym(SIGVTALRM),
	defsym(SIGPROF),
	defsym(SIGWINCH),
	defsym(SIGIO),
	defsym(SIGPWR),
	defsym(SIGSYS),
	{ NULL }
};

int
print_signal(int type, const void *p, size_t len)
{
	return print_symbolic(type, p, len, sig_names);
}

int
print_ifc_addr(int type, const void *p, size_t len)
{
	const struct ifreq	*ifr;

	if (type != AUDIT_ARG_POINTER
	 || len != sizeof(struct ifreq))
		return -1;

	ifr = (struct ifreq *) p;
	printf("[%s; ", ifr->ifr_name);
	if (print_sockaddr(type, &ifr->ifr_addr, sizeof(ifr->ifr_ifru)) < 0)
		printf(" ???");
	printf("]");
	return 0;
}

int
print_ifc_mtu(int type, const void *p, size_t len)
{
	const struct ifreq	*ifr;

	if (type != AUDIT_ARG_POINTER
	 || len != sizeof(struct ifreq))
		return -1;

	ifr = (struct ifreq *) p;
	printf("[%s; ", ifr->ifr_name);
	print_default(AUDIT_ARG_IMMEDIATE, &ifr->ifr_mtu,
				sizeof(ifr->ifr_mtu));
	printf("]");
	return 0;
}

static const struct bitname	if_flag_names[] = {
	defbit(IFF_UP),
	defbit(IFF_BROADCAST),
	defbit(IFF_LOOPBACK),
	defbit(IFF_POINTOPOINT),
	defbit(IFF_RUNNING),
	defbit(IFF_NOARP),
	defbit(IFF_PROMISC),

	{ NULL }
};
static const struct bitname	rt_flag_names[] = {
	defbit(RTF_UP),
	defbit(RTF_GATEWAY),
	defbit(RTF_HOST),
	defbit(RTF_REINSTATE),
	defbit(RTF_DYNAMIC),
	defbit(RTF_MODIFIED),
	defbit(RTF_MTU),
	defbit(RTF_MSS),
	defbit(RTF_WINDOW),
	defbit(RTF_IRTT),
	defbit(RTF_REJECT),
	defbit(RTF_STATIC),
	defbit(RTF_XRESOLVE),
	defbit(RTF_NOFORWARD),
	defbit(RTF_THROW),
	defbit(RTF_NOPMTUDISC),
	defbit(RTF_DEFAULT),
	defbit(RTF_ALLONLINK),
	defbit(RTF_ADDRCONF),
	defbit(RTF_LINKRT),
	defbit(RTF_NONEXTHOP),
	defbit(RTF_CACHE),
	defbit(RTF_FLOW),
	defbit(RTF_POLICY),

	{ NULL }
};

void
__print_ifc_flags(int flags)
{
	print_bits(AUDIT_ARG_IMMEDIATE, &flags, sizeof(flags), if_flag_names);
}

int
print_ifc_flags(int type, const void *p, size_t len)
{
	const struct ifreq	*ifr;

	if (type != AUDIT_ARG_POINTER
	 || len != sizeof(struct ifreq))
		return -1;

	ifr = (struct ifreq *) p;
	printf("[%s; ", ifr->ifr_name);
	__print_ifc_flags(ifr->ifr_flags);
	printf("]");
	return 0;
}

int
print_rtentry(int type, const void *p, size_t len)
{
	const struct rtentry	*rt;
	int			flags;

	if (type != AUDIT_ARG_POINTER
	 || len != sizeof(*rt))
		return -1;

	rt = (const struct rtentry *) p;
	printf("[dst=");
	__print_sockaddr(&rt->rt_dst, sizeof(rt->rt_dst));
	if (!(rt->rt_flags & RTF_HOST)) {
		struct sockaddr netmask = rt->rt_genmask;

		printf("/");
		/* Hack - route(8) doesn't seem to set the address family */
		netmask.sa_family = rt->rt_dst.sa_family;
		__print_sockaddr(&netmask, sizeof(netmask));
	}
	if (rt->rt_flags & RTF_GATEWAY) {
		printf("; gw=");
		__print_sockaddr(&rt->rt_gateway, sizeof(rt->rt_gateway));
	}
	if (rt->rt_flags & RTF_MTU)
		printf("; mtu=%lu", rt->rt_mtu);
	if (rt->rt_flags & RTF_WINDOW)
		printf("; mtu=%lu", rt->rt_window);
	if (rt->rt_flags & RTF_IRTT)
		printf("; mtu=%u", rt->rt_irtt);

	flags = rt->rt_flags & ~(RTF_HOST|RTF_GATEWAY|RTF_MTU|RTF_WINDOW|RTF_IRTT);
	if (flags) {
		printf("; flags=");
		print_bits(AUDIT_ARG_IMMEDIATE, &flags, sizeof(flags),
				rt_flag_names);
	}

	return 0;
}

int
print_arpreq(int type, const void *p, size_t len)
{
	static const struct bitname	arp_bits[] = {
		defbit(ATF_COM),
		defbit(ATF_PERM),
		defbit(ATF_PUBL),
		defbit(ATF_USETRAILERS),
		defbit(ATF_NETMASK),
		defbit(ATF_DONTPUB),
		defbit(ATF_MAGIC),

		{ NULL }
	};
	const struct arpreq	*ar;

	if (type != AUDIT_ARG_POINTER || len != sizeof(*ar))
		return -1;

	ar = (const struct arpreq *) p;
	printf("[addr=");
	__print_sockaddr(&ar->arp_pa, sizeof(ar->arp_pa));
	if (ar->arp_flags & ATF_NETMASK) {
		printf("; netmask=");
		__print_sockaddr(&ar->arp_netmask,
				sizeof(ar->arp_netmask));
	}
	if (ar->arp_dev[0])
		printf("; dev=%.*s", sizeof(ar->arp_dev), ar->arp_dev);
	printf("; hw=");
	__print_hwaddr(&ar->arp_ha, sizeof(ar->arp_ha));
	printf("; flags=");
	print_bits(AUDIT_ARG_IMMEDIATE, &ar->arp_flags, sizeof(ar->arp_flags), arp_bits);
	printf("]");

	return 0;
}

static int
__print_time(time_t timestamp, unsigned long nsec)
{
	char		timestr[64];
	struct tm	*tm;

	tm = localtime(&timestamp);
	strftime(timestr, sizeof(timestr), "%Y-%m-%dT%H:%M:%S", tm);
	printf("%s", timestr);
	if (nsec % 1000)
		printf(".%09lu", nsec);
	else if (nsec % 1000000)
		printf(".%06lu", nsec / 1000);
	else if (nsec)
		printf(".%03lu", nsec / 1000000);
	return 0;
}

int
print_timeref(int type, const void *p, size_t len)
{
	u_int64_t	value;

	/* The argument to stime(2) is a time_t pointer */
	if (type == AUDIT_ARG_POINTER && len <= 8)
		type = AUDIT_ARG_IMMEDIATE;
	if (get_immediate(type, p, len, &value) < 0)
		return -1;

	return __print_time(value, 0);
}

int
print_timeval(int type, const void *p, size_t len)
{
	const struct timeval *tv;

	if (type != AUDIT_ARG_POINTER || len != sizeof(*tv))
		return -1;
	tv = (const struct timeval *) p;
	return __print_time(tv->tv_sec, tv->tv_usec * 1000);
};

int
print_timespec(int type, const void *p, size_t len)
{
	const struct timespec *ts;

	if (type != AUDIT_ARG_POINTER || len != sizeof(*ts))
		return -1;
	ts = (const struct timespec *) p;
	return __print_time(ts->tv_sec, ts->tv_nsec);
};

int
print_timezone(int type, const void *p, size_t len)
{
	const struct timezone *tz;

	if (type != AUDIT_ARG_POINTER || len != sizeof(*tz))
		return -1;
	tz = (const struct timezone *) p;
	printf("\"GMT");
	if (tz->tz_minuteswest)
		printf("%+05d", tz->tz_minuteswest);
	printf("\"");
	return 0;
}

/* sendfile takes a "off_t *" argument */
int
print_ref_off_t(int type, const void *p, size_t len)
{
	if (type == AUDIT_ARG_POINTER && len <= 8) {
		__print_integer("%llu", p, len);
		return 0;
	}
	return print_default(type, p, len);
}

void
__print_result(long result)
{
	if (result < -1024) {
		/* Looks like an address */
		printf("; result=%p", (void *) result);
	} else {
		printf("; result=%ld", result);
		if (result < 0)
			printf(" [\"%s\"]", strerror(-result));
	}
}

static const struct syscall_info *
syscall_print_name(unsigned int major, unsigned int minor)
{
	const struct syscall_info *table = NULL;
	const struct syscall_info *info = NULL;
	unsigned int	num = major;

	if (ABSTRACT_SYSCALLS) {
		if (num < audit_NUM_CALLS)
			table = function_table;
	}
	else {
#ifdef __NR_socketcall
		if (major == __NR_socketcall) {
			num = __socketcall_base + minor;
			if (num >= __socketcall_end) {
				printf("socketcall#%u", minor);
				return NULL;
			}
			table = syscall_table;
		} else
#endif
#ifdef __NR_ipc
		if (major == __NR_ipc) {
			num = __ipccall_base + minor;
			if (num >= __ipccall_end) {
				printf("ipc#%u", minor);
				return NULL;
			}
			table = syscall_table;
		} else
#endif
		if (num < __syscall_end)
			table = syscall_table;
	}

	if (table && table[num].name) {
		info = &table[num];
		printf("%s", info->name);
	} else if (minor) {
		printf("syscall#%u,%u", major, minor);
	} else {
		printf("syscall#%u", major);
	}

	return info;
}

void
syscall_print_argument(const struct syscall_info *info,
		unsigned int index, const struct syscall_arg *arg)
{
	syscall_print_fn_t *fn;

	if (info
	 && (fn = info->arg[index])
	 && fn(arg->type, arg->data, arg->len) >= 0)
		return;

	print_default(arg->type, arg->data, arg->len);
}

void
syscall_print(struct syscall_data *sc)
{
	const struct syscall_info *info;
	u_int64_t	value;
	unsigned int	j;
	int		r;

	info = syscall_print_name(sc->major, sc->minor);

	printf("(");
	if (sc->major == (ABSTRACT_SYSCALLS ? AUDIT_ioctl : __NR_ioctl))
		print_ioctl_args(info, sc);
	else {
		if(sc->major == (ABSTRACT_SYSCALLS ? AUDIT_open : __NR_open)) {
			/* For the open system call, if the caller didn't
			 * give the O_CREAT argument, do not print the mode */
			r = get_immediate(sc->args[1].type, sc->args[1].data,
					sc->args[1].len, &value);
			if (r >= 0 && !(value & O_CREAT)) {
				sc->nargs--;
			}
		}
		for (j = 0; j < sc->nargs; j++) {
			if (j)
				printf(", ");

			syscall_print_argument(info, j, &sc->args[j]);
		}
	}
	printf(")");

	if (ABSTRACT_SYSCALLS) {
		switch (sc->major) {
		case AUDIT_execve:
			if (sc->result == 0)
				break;
		default:
			__print_result(sc->result);
			break;
		}
	}
	else {
		switch (sc->major) {
		case __NR_exit:
			/* Nothing */
			break;
		case __NR_execve:
			if (sc->result == 0)
				break;
		default:
			__print_result(sc->result);
			break;
		}
	}
}

/*
 * Special case handling
 */
struct ioctl_info {
	const char *	name;
	unsigned int	code;
	syscall_print_fn_t *fn;
};
#define defioctl(ioc, func)	{ #ioc, ioc, func }
static const struct ioctl_info ioctl_info[] = {
	defioctl(SIOCGIFCONF,		NULL),
	defioctl(SIOCSIFADDR,		print_ifc_addr),
	defioctl(SIOCGIFADDR,		NULL),
	defioctl(SIOCGIFFLAGS,		NULL),
	defioctl(SIOCSIFFLAGS,		print_ifc_flags),
	defioctl(SIOCSIFDSTADDR,	print_ifc_addr),
	defioctl(SIOCSIFBRDADDR,	print_ifc_addr),
	defioctl(SIOCSIFHWADDR,		print_ifc_addr),
	defioctl(SIOCSIFNETMASK,	print_ifc_addr),
	defioctl(SIOCSIFMTU,		print_ifc_mtu),
	defioctl(SIOCADDRT,		print_rtentry),
	defioctl(SIOCDELRT,		print_rtentry),
	defioctl(SIOCSARP,		print_arpreq),
	defioctl(SIOCDARP,		print_arpreq),
	/* XXX add more network ioctls */

	/* Some of the more common ioctls we don't much care
	 * about, but which crop often, so help the admin
	 * by making these more intelligible */
	defioctl(TCGETS,		NULL),
	defioctl(TCSETS,		NULL),
	defioctl(TCSETSW,		NULL),
	defioctl(TCSETSF,		NULL),
	defioctl(TCGETA,		NULL),
	defioctl(TCSETA,		NULL),
	defioctl(TCSETAW,		NULL),
	defioctl(TCSETAF,		NULL),
	defioctl(TCXONC,		NULL),
	defioctl(TIOCEXCL,		NULL),
	defioctl(TIOCNXCL,		NULL),
	defioctl(TIOCSCTTY,		NULL),
	defioctl(TIOCGPGRP,		NULL),
	defioctl(TIOCSPGRP,		NULL),
	defioctl(TIOCGWINSZ,		NULL),
	defioctl(TIOCSWINSZ,		NULL),
	defioctl(FIONREAD,		NULL),
	defioctl(TIOCSETD,		NULL),
	defioctl(TIOCGETD,		NULL),
	defioctl(TIOCGPTN,		NULL),
	defioctl(TIOCSPTLCK,		NULL),
	defioctl(TIOCGDEV,		NULL),

	defioctl(AUIOCATTACH,		NULL),
	defioctl(AUIOCDETACH,		NULL),
	defioctl(AUIOCIAMAUDITD,	NULL),
	defioctl(AUIOCSETAUDITID,	NULL),
	defioctl(AUIOCLOGIN,		NULL),
	defioctl(AUIOCUSERMESSAGE,	NULL),

	/* Something is actually trying to I_PUSH stuff in glibc */
	defioctl(I_PUSH,		NULL),

	{ NULL }
};

static void
print_ioctl_args(const struct syscall_info *info, const struct syscall_data *sc)
{
	const struct syscall_arg *arg = &sc->args[0];
	const struct ioctl_info *ioc = NULL;
	u_int64_t	code;
	unsigned int	j;

	/* File descriptor */
	syscall_print_argument(info, 0, arg++);
	printf(", ");

	if (get_immediate(arg->type, arg->data, arg->len, &code) >= 0) {
		arg++;

		for (ioc = ioctl_info; ioc->name; ioc++) {
			if (ioc->code == code)
		       		break;
		}

		if (ioc->name) {
			/* We have a match here */
			printf("%s", ioc->name);
		} else {
			printf("0x%llx", code);
			ioc = NULL;
		}
	} else {
		/* Hm, strange argument type? */
		syscall_print_argument(info, 1, arg++);
	}

	/* Print 3rd argument as is */
	for (j = 2; j < sc->nargs; j++) {
		printf(", ");
		if (ioc && ioc->fn
		 && ioc->fn(arg->type, arg->data, arg->len) >= 0) {
			/* success */
		} else {
			syscall_print_argument(info, 2, arg++);
		}
	}
}

