/*
 * Handle disk usage thresholds
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 */

#include <sys/vfs.h>
#include <sys/stat.h>
#include <sys/reboot.h>
#include <stdlib.h>
#include <syslog.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>

#include "auditd.h"
#include "filter.h"

typedef struct disk_thresh {
	struct disk_thresh *next;
	time_t		next_check;
	unsigned long	spaceleft;
	unsigned int	interval;
	action_t *	actions;
} disk_thresh_t;

static disk_thresh_t *	disk_thresholds = NULL;

static void		perform_threshold_actions(disk_thresh_t *, const char *);
static unsigned long	parse_size(const char *);

/*
 * Check thresholds
 */
void
check_disk_thresholds(const char *path)
{
	struct statfs	fstb;
	struct stat	stb;
	disk_thresh_t	*thresh, *low = NULL;
	time_t		now;

	time(&now);
	if (statfs(path, &fstb) < 0 || stat(path, &stb) < 0) {
		static time_t	groan_again = 0;

		if (groan_again > now)
			return;

		log_err(LOG_WARNING,
			"check_disk_thresholds: unable to stat %s: %m",
			path);
		groan_again = now + 300;
		return;
	}

	if (opt_debug > 3)
		log_dbg("Checking for disk space on %s, "
			"currently %lu blocks left",
			path, fstb.f_bfree);

	/* If free disk space is too low, select the lowest threshold,
	 * so we don't print N different warnings (space dropped below
	 * 10GB, 1GB, ...) */
	for (thresh = disk_thresholds; thresh; thresh = thresh->next) {
		unsigned long	blocksleft;

		if (thresh->next_check >= now)
			continue;

		thresh->next_check = now + thresh->interval;
		blocksleft = thresh->spaceleft / stb.st_blksize;

		if (fstb.f_bfree < blocksleft
		 && (low == NULL || thresh->spaceleft < low->spaceleft))
			low = thresh;
	}

	if (low)
		perform_threshold_actions(low, path);
}

/*
 * Force a check of all thresholds, regardless of when they
 * were last checked. This function should be called e.g.
 * after reopening the log file
 */
void
recheck_disk_thresholds(const char *path)
{
	disk_thresh_t	*thresh;

	for (thresh = disk_thresholds; thresh; thresh = thresh->next)
		thresh->next_check = 0;

	check_disk_thresholds(path);
}


/*
 * Configure thresholds from config file
 */
void
configure_thresholds(void)
{
	cf_node_t	*np;

	np = cf_node_find(cf_root, "threshold");
	while (np != NULL) {
		disk_thresh_t	*thresh;

		thresh = (disk_thresh_t *) calloc(1, sizeof(*thresh));
		thresh->spaceleft = parse_size(cf_node_value(np, "space-left"));
		thresh->actions = configure_actions(np);
		thresh->interval = 30;

		if (opt_debug > 2) {
			log_dbg("Configured threshold %s, space-left = %lu",
					cf_node_value(np, NULL),
				       	thresh->spaceleft);
		}
		thresh->next = disk_thresholds;
		disk_thresholds = thresh;

		np = cf_node_find_next(np, "threshold");
	}
}

/*
 * perform actions configured when reaching a threshold
 */
void
perform_threshold_actions(disk_thresh_t *thresh, const char *path)
{
	char		sysmsg[1024], audmsg[1024];

	snprintf(sysmsg, sizeof(sysmsg),
			"Free disk space dropped to %lu bytes or less "
			"on filesystem holding %s",
			thresh->spaceleft, path);
	snprintf(audmsg, sizeof(audmsg),
			"Disk space dropped below %lu:%s",
			thresh->spaceleft,
			path);

	perform_actions(thresh->actions, sysmsg, audmsg);
}

unsigned long
parse_size(const char *arg)
{
	unsigned long	value;
	char		*mult;

	value = strtoul(arg, &mult, 10);
	if (mult[0]) {
		if (mult[1])
			goto bad;
		switch (tolower(mult[0])) {
		case 'g': value *= 1024;
		case 'm': value *= 1024;
		case 'k': value *= 1024;
			  break;
		default:
			  goto bad;
		}
	}
	return value;

bad:	cf_error("Bad disk space constant \"%s\"", arg);
	exit(1);
}
