/*
 * Declarations for the Audit Daemon
 *
 * Copyright (C) 2003, SuSE Linux AG
 */

#ifndef AUDITD_H
#define AUDITD_H

#include <stdarg.h>
#include <syslog.h>

extern int		opt_debug;
extern int		default_syscalls[];

/*
 * Handle disk usage thresholds
 */
extern void		configure_thresholds(void);
extern void		check_disk_thresholds(const char *);
extern void		recheck_disk_thresholds(const char *);

/*
 * Send events to the audit log directly
 */
extern void		log_audit(const char *, const char *, ...);

/*
 * Logging stuff
 */
extern void		log_open(void);
extern void		log_err(int, const char *, ...);
extern void		log_fatal(const char *, ...);
extern void		vlog_err(int, const char *, va_list);
extern void		log_dbg(const char *, ...);
extern void		log_close(void);

/*
 * Configuration file handling
 */
typedef struct cf_array	cf_array_t;
typedef struct cf_node	cf_node_t;

typedef struct cf_object {
	char *		name;
	unsigned int	num;
	struct {
		int		line;
		const char *	file;
	} defined_in;
} cf_object_t;

extern cf_node_t *	cf_root;

extern int		audit_parse_daemon_config(const char *filename);
extern int		audit_parse_filter_config(const char *filename);
extern void		audit_set_priority(int);

extern cf_array_t *	cf_array_new(const char *, size_t);
extern cf_object_t *	cf_array_alloc(cf_array_t *, const char *);
extern cf_object_t *	cf_array_find(const cf_array_t *, const char *);
extern void		cf_array_grow(cf_array_t *);
extern cf_object_t *	cf_array_at(cf_array_t *, unsigned int);
extern unsigned int	cf_array_size(const cf_array_t *);

extern cf_node_t *	cf_node_new(cf_node_t *, const char *, const char *);
extern cf_node_t *	cf_node_find(cf_node_t *, const char *);
extern cf_node_t *	cf_node_find_next(cf_node_t *, const char *);
extern const char *	cf_node_name(cf_node_t *);
extern const char *	cf_node_value(cf_node_t *, const char *);
extern long		cf_node_atoi(cf_node_t *, const char *, long);
extern size_t		cf_node_atofs(cf_node_t *, const char *, long);
extern int		cf_node_atob(cf_node_t *, const char *, int);

#ifdef __GNUC__
extern void		cf_error(const char *, ...) __attribute__ ((__noreturn__));
#else
extern void		cf_error(const char *, ...);
#endif

/*
 * Process output
 */
extern int		configure_output(void);
extern void		output_reopen(void);
extern void		output_write_record(void *data, size_t len);
extern void		output_child_complete(pid_t, int);

/*
 * Process event actions
 */
enum { ACTION_AUDIT, ACTION_SYSLOG, ACTION_NOTIFY, ACTION_SUSPEND, ACTION_PANIC };
typedef struct action {
	struct action *	ac_next;
	int		ac_type;
	union {
	    struct {
		int	facility, priority;
	    } syslog;
	    const char *command;
	    const char *event;
	} u;
} action_t;
#define ac_event	u.event
#define ac_command	u.command
#define ac_facility	u.syslog.facility
#define ac_priority	u.syslog.priority

extern action_t *	configure_actions(cf_node_t *);
extern void		perform_actions(action_t *, const char *, const char *);

/*
 * Pid file handling
 */
extern int		make_pidfile(const char *, int);

#endif /* AUDITD_H */
