/*
 * Process actions attached to certain events
 *
 * Copyright (C) 2003, SuSE Linux AG
 * Written by okir@suse.de
 */

#include <sys/vfs.h>
#include <sys/stat.h>
#include <sys/reboot.h>
#include <stdlib.h>
#include <syslog.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>

#include <laus.h>
#include "auditd.h"
#include "filter.h"

static unsigned int	get_qlen(void);
static void		set_qlen(unsigned int);

action_t *
configure_actions(cf_node_t *np)
{
	action_t	*ac, *list = NULL;
	const char	*nag;

	np = cf_node_find(np, "action");
	while (np != NULL) {
		const char	*value;

		ac = (action_t *) calloc(1, sizeof(*ac));
		if (!(value = cf_node_value(np, "type"))
		 && !(value = cf_node_value(np, NULL))) {
			nag = "unnamed action";
			goto bad;
		}
		if (!strcmp(value, "audit")) {
			ac->ac_type = ACTION_AUDIT;
			ac->ac_event = cf_node_value(np, "event");
			if (!ac->ac_event) {
				nag = "no event defined for audit action";
				goto bad;
			}
		} else
		if (!strcmp(value, "notify")) {
			ac->ac_type = ACTION_NOTIFY;
			ac->ac_command = cf_node_value(np, "command");
			if (!ac->ac_command) {
				nag = "no command defined for notify action";
				goto bad;
			}
		} else
		if (!strcmp(value, "syslog")) {
			ac->ac_type = ACTION_SYSLOG;
			ac->ac_facility = cf_syslog_facility(cf_node_value(np,
						"facility"));
			ac->ac_priority = cf_syslog_priority(cf_node_value(np,
						"priority"));
			if (ac->ac_facility < 0 || ac->ac_priority < 0) {
				nag = "bad syslog facility or priority";
				goto bad;
			}
		} else
		if (!strcmp(value, "shutdown")) {
			ac->ac_type = ACTION_PANIC;
		} else
		if (!strcmp(value, "suspend")) {
			ac->ac_type = ACTION_SUSPEND;
		} else {
			nag = "unknown action type";
			goto bad;
		}

		ac->ac_next = list;
		list = ac;

		np = cf_node_find_next(np, "action");
	}

	return list;
bad:
	cf_error("%s (defined in %s:%u)", nag,
			((cf_object_t *) np)->defined_in.file,
			((cf_object_t *) np)->defined_in.line);
	exit(1);
}

/*
 * perform actions configured when reaching a threshold
 */
void
perform_actions(action_t *ac, const char *sysmsg, const char *audmsg)
{
	while (ac != NULL) {
		switch (ac->ac_type) {
		case ACTION_SYSLOG:
			/* Close our syslog, open different facility */
			log_close();
			openlog("auditd", LOG_PID, ac->ac_facility);

			/* Now log the message */
			syslog(ac->ac_priority, "%s", sysmsg);

			/* Reopen our syslog */
			closelog();
			log_open();
			break;
		case ACTION_NOTIFY:
			system(ac->ac_command);
			break;
		case ACTION_AUDIT:
			log_audit(ac->ac_event, "%s", audmsg);
			break;
		case ACTION_PANIC:
			syslog(LOG_CRIT, "%s; shutdown forced", sysmsg);
			sleep(1); /* allow syslogd to do its stuff */
			if (reboot(RB_HALT_SYSTEM) < 0) {
				syslog(LOG_CRIT,
					"%s: unable to halt machine!",
					ac->ac_event);
				while (1)
					pause();
			}
			break;
		case ACTION_SUSPEND: {
				sigset_t	sigset;
				unsigned int	qlen;

				sigfillset(&sigset);
				sigdelset(&sigset, SIGHUP);
				sigdelset(&sigset, SIGTERM);

				log_err(LOG_CRIT, "%s; suspending execution", sysmsg);

				/* This stops all processes about to generate records */
				qlen = get_qlen();
				set_qlen(0);

				sigsuspend(&sigset);

				/* Allow processes to queue records again */
				set_qlen(qlen);
			}
		}
		ac = ac->ac_next;
	}
}

/*
 * Get/set the max queue length
 */
unsigned int
get_qlen(void)
{
	unsigned int	qlen;
	FILE		*fp;

	if (!(fp = fopen(AUDIT_QLENCFG_PATH, "r"))) {
		log_err(LOG_ERR, "fopen(%s) failed: %m", AUDIT_QLENCFG_PATH);
		return 1024;
	}
	if (fscanf(fp, "%d", &qlen) != 1 || qlen < 10) {
		qlen = 10;
		log_err(LOG_ERR, "error parsing %s: assuming qlen of %u",
				AUDIT_QLENCFG_PATH, qlen);
	}
	fclose(fp);
	return qlen;
}

void
set_qlen(unsigned int qlen)
{
	FILE	*fp;

	if (!(fp = fopen(AUDIT_QLENCFG_PATH, "w"))) {
		log_err(LOG_ERR, "fopen(%s) failed: %m", AUDIT_QLENCFG_PATH);
		return;
	}
	fprintf(fp, "%u", qlen);
	fclose(fp);
}
