/**********************************************************************
**   Copyright (C) International Business Machines  Corp., 2003
**
**   This program is free software;  you can redistribute it and/or modify
**   it under the terms of the GNU General Public License as published by
**   the Free Software Foundation; either version 2 of the License, or
**   (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY;  without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
**   the GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public License
**   along with this program;  if not, write to the Free Software
**   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
**
**
**
**  FILE   : aestest.c
**
**  PURPOSE: To perform a basic functional test of the AES encryption
**           implementation in the OpenSSL library.
**          
**
**  HISTORY:
**    06/03 Originated by Michael A. Halcrow <mhalcrow@cs.byu.edu>
**    06/03 AES test methodology and vector (taken from the AES FIPS)
**          written by Peter Gutmann in cryptlib
**
**********************************************************************/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
/* To test the cryptlib implementation, #define CRYPTLIBTEST and include the libcl library in the linking stage of the build */
//#define CRYPTLIBTEST
#ifdef CRYPTLIBTEST
#include "aes.h"
#else
#include <openssl/aes.h>
#endif

#define AES_EXPANDED_KEYSIZE sizeof( AES_CTX )
#define AES_BLOCKSIZE 16
#define AES_KEYSIZE 32

#ifdef CRYPTLIBTEST
#define AES_KEY	aes_ctx
#define AES_2KEY AES_CTX
#define AES_ERROR aes_bad
#endif

typedef struct {
  AES_KEY	encKey, decKey;
} AES_CTX;

typedef struct {
  const int keySize;
  const unsigned char key[ AES_KEYSIZE ];
  const unsigned char plaintext[ AES_BLOCKSIZE ];
  const unsigned char ciphertext[ AES_BLOCKSIZE ];
} AES_TEST;

/* Vectors from cryptlib; originated from AES FIPS */
static const AES_TEST testAES[] = {
  { 16,
    { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
      0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F },
    { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 
      0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF },
    { 0x69, 0xC4, 0xE0, 0xD8, 0x6A, 0x7B, 0x04, 0x30, 
      0xD8, 0xCD, 0xB7, 0x80, 0x70, 0xB4, 0xC5, 0x5A } },
  { 24,
    { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
      0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 
      0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17 },
    { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 
      0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF },
    { 0xDD, 0xA9, 0x7C, 0xA4, 0x86, 0x4C, 0xDF, 0xE0, 
      0x6E, 0xAF, 0x70, 0xA0, 0xEC, 0x0D, 0x71, 0x91 } },
  { 32,
    { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 
      0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 
      0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 
      0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F },
    { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 
      0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF },
    { 0x8E, 0xA2, 0xB7, 0xCA, 0x51, 0x67, 0x45, 0xBF, 
      0xEA, 0xFC, 0x49, 0x90, 0x4B, 0x49, 0x60, 0x89 } }
};

/* Test cryptlib using cryptlib's test */
#ifdef CRYPTLIBTEST

/* Test the AES code against the test vectors from the AES FIPS */
int aesSelfTest( void ) {
  int i;

  for( i = 0; i < sizeof( testAES ) / sizeof( AES_TEST ); i++ ) {
      AES_KEY aesKey;
      unsigned char temp[ AES_BLOCKSIZE ];

      memcpy( temp, testAES[ i ].plaintext, AES_BLOCKSIZE );
      aes_enc_key( testAES[ i ].key, testAES[ i ].keySize, &aesKey );
      aes_enc_blk( temp, temp, &aesKey );
      if( memcmp( testAES[ i ].ciphertext, temp, AES_BLOCKSIZE ) )
	return( -1 );
      aes_dec_key( testAES[ i ].key, testAES[ i ].keySize, &aesKey );
      aes_dec_blk( temp, temp, &aesKey );
      if( memcmp( testAES[ i ].plaintext, temp, AES_BLOCKSIZE ) )
	return( -1 );
    }

  return( 0 );
}

#else

/* Test OpenSSL using cryptlib's test methodology and AES FIPS vectors */
int aesOpenSSLTest( void ) {
  int x;

  for( x = 0; x < sizeof( testAES ) / sizeof( AES_TEST ); x++ ) {
      AES_KEY aesKey;
      unsigned char tmp[ AES_BLOCKSIZE ];

      memcpy( tmp, testAES[ x ].plaintext, AES_BLOCKSIZE );
      AES_set_encrypt_key( testAES[ x ].key, testAES[ x ].keySize * 8, &aesKey );
      AES_encrypt( tmp, tmp, &aesKey );
      if( memcmp( testAES[ x ].ciphertext, tmp, AES_BLOCKSIZE ) ) {
	return -1;
      }
      AES_set_decrypt_key( testAES[ x ].key, testAES[ x ].keySize * 8, &aesKey );
      AES_decrypt( tmp, tmp, &aesKey );
      if( memcmp( testAES[ x ].plaintext, tmp, AES_BLOCKSIZE ) ) {
	return -1;
      }
    }

  return 0;  
}

#endif

void printBlockHex( unsigned char* block, int size ) {
  int x = 0;
  while( x < size ) {
    printf( "%02X", block[ x ] );
    x++;
  }
}

/**
 * parseCbcDFileAndVerify
 * 
 * Parses a file that is assumed to conform to the specification given
 * on pages 12 and 13 of the document entitled, ``Description of Known
 * Answer Tests and Monte Carlo Tests for Advanced Encryption Standard
 * (AES) Candidate Algorithm Submissions'' dated January 7, 1998.
 *
 * @author Michael A. Halcrow <mike@halcrow.us>
 */
int parseCbcDFileAndVerify( char* filename ) {
  int keysize;
  unsigned char plaintextBlock[ AES_BLOCKSIZE ];
  unsigned char keyBlock[ 32 ];
  unsigned char initializationVectorBlock[ AES_BLOCKSIZE ];
  unsigned char chainingValueBlock[ AES_BLOCKSIZE ];
  unsigned char ciphertextBlock[ AES_BLOCKSIZE ];
  unsigned char inputBlock[ AES_BLOCKSIZE ];
  unsigned char outputBlock[ AES_BLOCKSIZE ];
  unsigned char readBuffer[ 100 ];
  unsigned char tmp[ AES_BLOCKSIZE ];
  AES_KEY aesKey;
  FILE* fPtr;
  int state = 0;
  int i = 0;
  int j, k;

  if( ( fPtr = fopen( filename, "r" ) ) == NULL ) {
    printf( "Error opening file [%s] for read access: errno = [%i]\n", filename, errno );
    return -1;
  }

  while( fgets( readBuffer, 99, fPtr ) ) {
    switch( state ) {
    case 0:
      if( strncmp( readBuffer, "KEY=", 4 ) == 0 ) {
	int x = 0;
	bzero( keyBlock, 32 );
	while( x < (keysize/8) ) {
	  char save = readBuffer[ 4 + x*2 + 2 ];
	  readBuffer[ 4 + x*2 + 2 ] = '\0';
	  keyBlock[ x ] = (char)strtoul( &readBuffer[ 4 + x*2 ], NULL, 16 );
	  readBuffer[ 4 + x*2 + 2 ] = save;
	  x++;
	}
	state = 1;
      } else if( strncmp( readBuffer, "KEYSIZE=", 8 ) == 0 ) {
	keysize = atoi( &readBuffer[ 8 ] );
	if( keysize > 256 ) {
	  printf( "Invalid keysize: [%d]\n", keysize );
	  return -1;
	}
	state = 0;
      }
      break;
    case 1:
      if( strncmp( readBuffer, "IV=", 3 ) == 0 ) {
	int x = 0;
	while( x < AES_BLOCKSIZE ) {
	  char save = readBuffer[ 3 + x*2 + 2 ];
	  readBuffer[ 3 + x*2 + 2 ] = '\0';
	  initializationVectorBlock[ x ] = (char)strtoul( &readBuffer[ 3 + x*2 ], NULL, 16 );
	  readBuffer[ 3 + x*2 + 2 ] = save;
	  x++;
	}
	state = 2;
      }
      break;
    case 2:
      if( strncmp( readBuffer, "CT=", 3 ) == 0 ) {
	int x = 0;
	while( x < AES_BLOCKSIZE ) {
	  char save = readBuffer[ 3 + x*2 + 2 ];
	  readBuffer[ 3 + x*2 + 2 ] = '\0';
	  ciphertextBlock[ x ] = (char)strtoul( &readBuffer[ 3 + x*2 ], NULL, 16 );
	  readBuffer[ 3 + x*2 + 2 ] = save;
	  x++;
	}
	state = 3;
      }
      break;
    case 3:
      if( strncmp( readBuffer, "PT=", 3 ) == 0 ) {
	int x = 0;
	while( x < AES_BLOCKSIZE ) {
	  char save = readBuffer[ 3 + x*2 + 2 ];
	  readBuffer[ 3 + x*2 + 2 ] = '\0';
	  plaintextBlock[ x ] = (char)strtoul( &readBuffer[ 3 + x*2 ], NULL, 16 );
	  readBuffer[ 3 + x*2 + 2 ] = save;
	  x++;
	}
	state = 0;
      
	// We are ready to do something with our values
	AES_set_decrypt_key( keyBlock, keysize, &aesKey );

	if( i%400 == 0 ) {
	  memcpy( chainingValueBlock, initializationVectorBlock, AES_BLOCKSIZE );
	}

	bzero( tmp, AES_BLOCKSIZE );
	bzero( inputBlock, AES_BLOCKSIZE );
	bzero( outputBlock, AES_BLOCKSIZE );

	for( j = 0; j <= 9999; j++ ) {
	  
	  memcpy( inputBlock, ciphertextBlock, AES_BLOCKSIZE );

	  AES_decrypt( inputBlock, outputBlock, &aesKey );	  
	  
	  for( k = 0; k < AES_BLOCKSIZE; k++ ) {
	    tmp[ k ] = outputBlock[ k ] ^ chainingValueBlock[ k ];
	  }

	  memcpy( chainingValueBlock, ciphertextBlock, AES_BLOCKSIZE );

	  memcpy( ciphertextBlock, tmp, AES_BLOCKSIZE );
	}

	if( memcmp( plaintextBlock, tmp, AES_BLOCKSIZE ) ) {
	  printf( "ERROR DECRYPTING: I = [%d]\n", i );
	  printf( "PT=" );
	  printBlockHex( plaintextBlock, AES_BLOCKSIZE );
	  printf( "\n:(=" );
	  printBlockHex( tmp, AES_BLOCKSIZE );
	  printf( "\n" );
	  return -1;
	}

	i++;
	
      }
      break;
    default:
      printf( "Error: We just got into an undefined state\n" );
      return -1;
      break;
    }
  }
  fclose( fPtr );
  return 0;
}

/**
 * parseCbcEFileAndVerify
 * 
 * Parses a file that is assumed to conform to the specification given
 * on pages 10 and 11 of the document entitled, ``Description of Known
 * Answer Tests and Monte Carlo Tests for Advanced Encryption Standard
 * (AES) Candidate Algorithm Submissions'' dated January 7, 1998.
 *
 * @author Michael A. Halcrow <mike@halcrow.us>
 */
int parseCbcEFileAndVerify( char* filename ) {
  int keysize;
  unsigned char plaintextBlock[ AES_BLOCKSIZE ];
  unsigned char keyBlock[ 32 ];
  unsigned char initializationVectorBlock[ AES_BLOCKSIZE ];
  unsigned char chainingValueBlock[ AES_BLOCKSIZE ];
  unsigned char ciphertextBlock[ AES_BLOCKSIZE ];
  unsigned char inputBlock[ AES_BLOCKSIZE ];
  unsigned char outputBlock[ AES_BLOCKSIZE ];
  unsigned char readBuffer[ 100 ];
  unsigned char tmp[ AES_BLOCKSIZE ];
  AES_KEY aesKey;
  FILE* fPtr;
  int state = 0;
  int i = 0;
  int j, k;

  if( ( fPtr = fopen( filename, "r" ) ) == NULL ) {
    printf( "Error opening file [%s] for read access: errno = [%i]\n", filename, errno );
    return -1;
  }

  while( fgets( readBuffer, 99, fPtr ) ) {
    switch( state ) {
    case 0:
      if( strncmp( readBuffer, "KEY=", 4 ) == 0 ) {
	int x = 0;
	bzero( keyBlock, 32 );
	while( x < (keysize/8) ) {
	  char save = readBuffer[ 4 + x*2 + 2 ];
	  readBuffer[ 4 + x*2 + 2 ] = '\0';
	  keyBlock[ x ] = (char)strtoul( &readBuffer[ 4 + x*2 ], NULL, 16 );
	  readBuffer[ 4 + x*2 + 2 ] = save;
	  x++;
	}
	state = 1;
      } else if( strncmp( readBuffer, "KEYSIZE=", 8 ) == 0 ) {
	keysize = atoi( &readBuffer[ 8 ] );
	if( keysize > 256 ) {
	  printf( "Invalid keysize: [%d]\n", keysize );
	  return -1;
	}
	state = 0;
      }
      break;
    case 1:
      if( strncmp( readBuffer, "IV=", 3 ) == 0 ) {
	int x = 0;
	while( x < AES_BLOCKSIZE ) {
	  char save = readBuffer[ 3 + x*2 + 2 ];
	  readBuffer[ 3 + x*2 + 2 ] = '\0';
	  initializationVectorBlock[ x ] = (char)strtoul( &readBuffer[ 3 + x*2 ], NULL, 16 );
	  readBuffer[ 3 + x*2 + 2 ] = save;
	  x++;
	}
	state = 2;
      }
      break;
    case 2:
      if( strncmp( readBuffer, "PT=", 3 ) == 0 ) {
	int x = 0;
	while( x < AES_BLOCKSIZE ) {
	  char save = readBuffer[ 3 + x*2 + 2 ];
	  readBuffer[ 3 + x*2 + 2 ] = '\0';
	  plaintextBlock[ x ] = (char)strtoul( &readBuffer[ 3 + x*2 ], NULL, 16 );
	  readBuffer[ 3 + x*2 + 2 ] = save;
	  x++;
	}
	state = 3;
      }
      break;
    case 3:
      if( strncmp( readBuffer, "CT=", 3 ) == 0 ) {
	int x = 0;
	while( x < AES_BLOCKSIZE ) {
	  char save = readBuffer[ 3 + x*2 + 2 ];
	  readBuffer[ 3 + x*2 + 2 ] = '\0';
	  ciphertextBlock[ x ] = (char)strtoul( &readBuffer[ 3 + x*2 ], NULL, 16 );
	  readBuffer[ 3 + x*2 + 2 ] = save;
	  x++;
	}
	state = 0;
      
	// We are ready to do something with our values
	AES_set_encrypt_key( keyBlock, keysize, &aesKey );

	if( i%400 == 0 ) {
	  memcpy( chainingValueBlock, initializationVectorBlock, AES_BLOCKSIZE );
	}

	bzero( tmp, AES_BLOCKSIZE );
	bzero( inputBlock, AES_BLOCKSIZE );
	bzero( outputBlock, AES_BLOCKSIZE );

	memcpy( tmp, ciphertextBlock, AES_BLOCKSIZE );

	for( j = 0; j <= 9999; j++ ) {
	  
	  for( k = 0; k < AES_BLOCKSIZE; k++ ) {
	    inputBlock[ k ] = plaintextBlock[ k ] ^ chainingValueBlock[ k ];
	  }

	  AES_encrypt( inputBlock, outputBlock, &aesKey );	  

	  if( j == 0 ) {
	    memcpy( plaintextBlock, chainingValueBlock, AES_BLOCKSIZE );
	  } else {
	    memcpy( plaintextBlock, tmp, AES_BLOCKSIZE );
	  }

	  memcpy( tmp, outputBlock, AES_BLOCKSIZE );

	  memcpy( chainingValueBlock, tmp, AES_BLOCKSIZE );

	}

	if( memcmp( ciphertextBlock, tmp, AES_BLOCKSIZE ) ) {
	  printf( "ERROR DECRYPTING: I = [%d]\n", i );
	  printf( "PT=" );
	  printBlockHex( plaintextBlock, AES_BLOCKSIZE );
	  printf( "\n:(=" );
	  printBlockHex( tmp, AES_BLOCKSIZE );
	  printf( "\n" );
	  return -1;
	}

	i++;
	
      }
      break;
    default:
      printf( "Error: We just got into an undefined state\n" );
      return -1;
      break;
    }
  }
  fclose( fPtr );
  return 0;
}

/**
 * parseEcbVKFileAndVerify
 * 
 * Parses a file that is assumed to conform to the specification given
 * on page 4 of the document entitled, ``Description of Known Answer
 * Tests and Monte Carlo Tests for Advanced Encryption Standard (AES)
 * Candidate Algorithm Submissions'' dated January 7, 1998.
 *
 * @author Michael A. Halcrow <mike@halcrow.us>
 */
int parseEcbVKFileAndVerify( char* filename ) {
  int keysize;
  unsigned char plaintextBlock[ AES_BLOCKSIZE ];
  unsigned char keyBlock[ 32 ];
  unsigned char ciphertextBlock[ AES_BLOCKSIZE ];
  unsigned char readBuffer[ 100 ];
  unsigned char tmp[ AES_BLOCKSIZE ];
  AES_KEY aesKey;
  FILE* fPtr;
  int state = 0;
  int i = 1;

  if( ( fPtr = fopen( filename, "r" ) ) == NULL ) {
    printf( "Error opening file [%s] for read access: errno = [%i]\n", filename, errno );
    return -1;
  }

  while( fgets( readBuffer, 99, fPtr ) ) {
    switch( state ) {
    case 0:
      if( strncmp( readBuffer, "KEYSIZE=", 8 ) == 0 ) {
	keysize = atoi( &readBuffer[ 8 ] );
	if( keysize > 256 ) {
	  printf( "Invalid keysize: [%d]\n", keysize );
	  return -1;
	}
	state = 1;
      }
      break;
    case 1:
      if( strncmp( readBuffer, "PT=", 3 ) == 0 ) {
	int x = 0;
	while( x < AES_BLOCKSIZE ) {
	  char save = readBuffer[ 3 + x*2 + 2 ];
	  readBuffer[ 3 + x*2 + 2 ] = '\0';
	  plaintextBlock[ x ] = (char)strtoul( &readBuffer[ 3 + x*2 ], NULL, 16 );
	  readBuffer[ 3 + x*2 + 2 ] = save;
	  x++;
	}
	state = 2;
      }
      break;
    case 2:
      if( strncmp( readBuffer, "KEY=", 4 ) == 0 ) {
	int x = 0;
	bzero( keyBlock, 32 );
	while( x < (keysize/8) ) {
	  char save = readBuffer[ 4 + x*2 + 2 ];
	  readBuffer[ 4 + x*2 + 2 ] = '\0';
	  keyBlock[ x ] = (char)strtoul( &readBuffer[ 4 + x*2 ], NULL, 16 );
	  readBuffer[ 4 + x*2 + 2 ] = save;
	  x++;
	}
	state = 3;
      } else if( strncmp( readBuffer, "KEYSIZE=", 8 ) == 0 ) {
	keysize = atoi( &readBuffer[ 8 ] );
	if( keysize > 256 ) {
	  printf( "Invalid keysize: [%d]\n", keysize );
	  return -1;
	}
	state = 1;
      }
      break;
    case 3:
      if( strncmp( readBuffer, "CT=", 3 ) == 0 ) {
	int x = 0;
	while( x < AES_BLOCKSIZE ) {
	  char save = readBuffer[ 3 + x*2 + 2 ];
	  readBuffer[ 3 + x*2 + 2 ] = '\0';
	  ciphertextBlock[ x ] = (char)strtoul( &readBuffer[ 3 + x*2 ], NULL, 16 );
	  readBuffer[ 3 + x*2 + 2 ] = save;
	  x++;
	}
	state = 2;

	// We are ready to do something with our values
	memcpy( tmp, plaintextBlock, AES_BLOCKSIZE );
	AES_set_encrypt_key( keyBlock, keysize, &aesKey );
	AES_ecb_encrypt( tmp, tmp, &aesKey, AES_ENCRYPT );
	if( memcmp( ciphertextBlock, tmp, AES_BLOCKSIZE ) ) {
	  printf( "ERROR ENCRYPTING: I = [%d]\n", i );
	  return -1;
	}
	AES_set_decrypt_key( keyBlock, keysize, &aesKey );
	AES_ecb_encrypt( tmp, tmp, &aesKey, AES_DECRYPT );
	if( memcmp( plaintextBlock, tmp, AES_BLOCKSIZE ) ) {
	  printf( "ERROR DECRYPTING: I = [%d]\n", i );
	  return -1;
	}

	i++;
	
      }
      break;
    default:
      printf( "Error: We just got into an undefined state\n" );
      return -1;
      break;
    }
  }
  fclose( fPtr );
  return 0;
}

int main( int argc, char* argv[] ) {
  char* ecbFilename;
  char* cbcDFilename;
  char* cbcEFilename;

  if( argc == 1 ) {
    ecbFilename = (char*)malloc( strlen( "ecb_vk.txt" ) + 1 );
    cbcDFilename = (char*)malloc( strlen( "cbc_d_m.txt" ) + 1 );
    cbcEFilename = (char*)malloc( strlen( "cbc_e_m.txt" ) + 1 );
    strcpy( ecbFilename, "ecb_vk.txt" );
    strcpy( cbcDFilename, "cbc_d_m.txt" );
    strcpy( cbcEFilename, "cbc_e_m.txt" );
  } else if( argc == 2 ) {
    ecbFilename = (char*)malloc( strlen( "ecb_vk.txt" ) + strlen( argv[1] ) + 2 );
    cbcDFilename = (char*)malloc( strlen( "cbc_d_m.txt" ) + strlen( argv[1] ) + 2 );
    cbcEFilename = (char*)malloc( strlen( "cbc_e_m.txt" ) + strlen( argv[1] ) + 2 );
    sprintf( ecbFilename, "%s/ecb_vk.txt", argv[1] );
    sprintf( cbcDFilename, "%s/cbc_d_m.txt", argv[1] );
    sprintf( cbcEFilename, "%s/cbc_e_m.txt", argv[1] );
  } else {
    printf( "Usage: aestest [PATH_TO_TEST_VECTOR_FILES]\n" );
    return 0;
  }

#ifdef CRYPTLIBTEST
  if( aesSelfTest() == 0 ) {
    printf( "Pass\n" );
    return 0;
  } else {
    printf( "Fail\n" );
  }
#else

  printf( "Performing basic AES test\n" );

  if( aesOpenSSLTest() != 0 ) {
    printf( "Fail\n" );
    return -1;
  }

  printf( "Testing AES ECB VK\n" );

  if( parseEcbVKFileAndVerify( ecbFilename ) != 0 ) {
    printf( "Fail\n" );
    return -1;
  }

  printf( "Testing AES CBC Decoding\n" );

  if( parseCbcDFileAndVerify( cbcDFilename ) != 0 ) {
    printf( "Fail\n" );
    return -1;
  }

  printf( "Testing AES CBC Encoding\n" );

  if( parseCbcEFileAndVerify( cbcEFilename ) != 0 ) {
    printf( "Fail\n" );
    return -1;
  }

  printf( "Pass\n" );

  free( ecbFilename );
  free( cbcDFilename );
  free( cbcEFilename );

  return 0;

#endif

  return -1;
}
