/**********************************************************************
**   Copyright (C) International Business Machines  Corp., 2003
**
**   This program is free software;  you can redistribute it and/or modify
**   it under the terms of the GNU General Public License as published by
**   the Free Software Foundation; either version 2 of the License, or
**   (at your option) any later version.
**
**   This program is distributed in the hope that it will be useful,
**   but WITHOUT ANY WARRANTY;  without even the implied warranty of
**   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
**   the GNU General Public License for more details.
**
**   You should have received a copy of the GNU General Public License
**   along with this program;  if not, write to the Free Software
**   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
**
**
**
**  FILE   : libpam.c
**
**  PURPOSE: This file contains the main body of the Linux Auditing 
**           System test suite for Pluggable Authentication Modules.  
**           
**           This file, as well as 
**           the accompanying headers in the ../includes/ directory and the
**           helper utility functions in the ../utils/ directory, form a
**           framework that facilitates a robustly modular test suite.
**
**           Each test is written as an independent function to be
**           called by the main body in this file.  This design 
**           conveniently allows multiple developers to implement
**           test functions for separate pam programs (located in the
**           ./tests/ directory) and these functions simply plug into
**           this framework.  
**           
**           As the tests become available, pointers to these
**           functions are added to the pamTests[] array.  Each
**           function in this array is called to execute the tests
**           contained therein.  Documentation for each test can be
**           found in the source code prolog of each test file.
**
**           Aside from the test functions themselves, there are several
**           helper utility functions that are used.  Documentation for
**           each utility can be found in the source code prolog of each
**           utility file.
**          
**
**  HISTORY:
**    08/2003 originated by Dustin Kirkland (k1rkland@us.ibm.com)
**
**********************************************************************/

#include "../include/includes.h"
#include "../include/utils.h"
#include "include/libpam.h"
#include <pwd.h>
#include <getopt.h>
#include <string.h>

int test_rc = 0;
int pass_testcases = 0;
int fail_testcases = 0;
int skip_testcases = 0;
int debug = 2;
uid_t login_uid = 0;
char cwd[PATH_MAX];

/*
** main
*/
int main(int argc, char **argv)
{

  int rc    = 0;
  int k     = 0;
  uid_t uid = 0;
  gid_t gid = 0;
  char* testcase = NULL;
  char* defaultUser = "nobody";
  char* usage = "Usage:\n\
  laustest [-u $TESTUSER] [-t $TRUSTEDPROGRAM] [-d $DEBUG]\n\n\
Arguments:\n\
  -u $TESTUSER:  Specify the user to run the tests as $TESTUSER.\n\
                 This user must actually exist on the system.\n\
                 By default, this is 'nobody'\n\
  -t $PAM:       Optionally specify a single test case to run.\n\
                 $PAM is the name of the pam program call to test.\n\
                 By default, all test cases are executed.\n\
  -d $DEBUG:     Optionally specify the debug level 0-5.\n\
                 0  No messages\n\
                 1  + Environment messages\n\
                 2  + Test messages (Default)\n\
                 3  + Warning messages\n\
                 4  + Info messages\n\
                 5  + Full debug messages\n\n";


  struct passwd* passwd_data = NULL;
  struct passwd* login_uid_data = NULL;
  laus_data* successDataPtr = NULL;

#ifdef __IX86
  int arch = AUDIT_ARCH_I386;
#endif
#ifdef __PPC32
  int arch = AUDIT_ARCH_PPC;
#endif
#ifdef __PPC64
  int arch = AUDIT_ARCH_PPC64;
#endif
#ifdef __S390
  int arch = AUDIT_ARCH_S390;
#endif
#ifdef __S390X
  int arch = AUDIT_ARCH_S390X;
#endif
#ifdef __X86_64
  int arch =  AUDIT_ARCH_X86_64;
#endif

  // Array contains data for each pam program test.
  pam_data pamTests[] = {
    /* fnptr, testname, dataptr */
    //{&test_login, "login", NULL},
    {&test_sshd, "sshd", NULL},
    {&test_su, "su", NULL},
    {&test_vsftpd, "vsftpd", NULL}
  };

  backupFile("/etc/audit/filter.conf");

  // Iterate through command line options
  while (1) {
    int option_index = 0;
    // BUGBUG: Not sure if this is actually necessary
    static struct option long_options[] = {
      {"user", 1, 0, 0},
      {"testcase", 1, 0, 0},
      {"debug", 1, 0, 0},
      {"login", 1, 0, 0},
      {0, 0, 0, 0}
    };
    int c = getopt_long (argc, argv, "u:t:d:l:",
                         long_options, &option_index);
    if (c == -1) break;
    switch (c) {
      case 'u':
        // User option specified, try to get passwd info, exit if none
        if ((passwd_data = getpwnam(optarg)) == NULL) {
          printf1("ERROR: Unable to get %s passwd info.\n", optarg);
          goto EXIT_ERROR;
        }
        break;
      case 'l':
        // login uid specified, try to get passwd info, exit if none
        if ((login_uid_data = getpwnam(optarg)) == NULL) {
          printf1("ERROR: Unable to get %s passwd info.\n", optarg);
          goto EXIT_ERROR;
        }
	login_uid = login_uid_data->pw_uid;
        break;
      case 't':
        // Individual test case specified, exit if invalid value
        testcase = (char *) malloc(strlen(optarg) + 1);	
        sprintf(testcase, "%s", optarg);				// optarg is not null-terminated
        for (k = 0; k < sizeof(pamTests)/sizeof(pam_data); k++) {
          if ((rc = strcmp(testcase, pamTests[k].testName)) == 0 ) {
            break;
          }
        }
        if ( rc != 0 ) {
          printf1("ERROR: System call '%s' not tested by laustest\n", testcase);
          goto EXIT_ERROR;
        }
        break;
      case 'd':
        // Debug level specified
        debug = atoi(optarg);
        // BUGBUG: This if statement needs to be modified to handle
        // non-numeric input correctly.
        if ( (debug < 0) || (debug > 9) ) {
          printf1("ERROR: Debug level %s is invalid\n", optarg);
          goto EXIT_ERROR;
        }
        break;
      default :
        // Help option or unhandled option specified, display usage text and exit
        printf("%s", usage);
        rc = 1;
        goto EXIT_HELP;
        break;
    }
  }

  if (passwd_data == NULL) {
    // If no user specified, try defaultUser
    passwd_data = getpwnam(defaultUser);
  }

  if (passwd_data == NULL) {
    // Must enter a valid user to run tests as
    printf1("ERROR: Please enter a test user name with the -u option.\n");
    goto EXIT_ERROR;
  }

  if (login_uid_data == NULL) {
    // Get login uid from system
    login_uid = getLoginUID();
  }

  uid = passwd_data->pw_uid;
  gid = passwd_data->pw_gid;

  // Save the CWD for setFilterDomain
  getcwd(cwd, PATH_MAX);

  /*
  ** For these tests, we want to log all success and failures
  */
  if ( ( rc = system("echo \"event user-message = always;\nevent process-login = always;\" > /etc/audit/filter.conf") ) != 0 ) {
    printf1("Could not configure filter.conf\n");
    goto EXIT_ERROR;
  }

  if ((rc = reloadAudit()) != 0) {
    goto EXIT_ERROR;
  }

  /*
  ** Loop through pam programs
  **
  */
  successDataPtr = (laus_data*)malloc(sizeof(laus_data));

  for (k = 0; k < sizeof(pamTests)/sizeof(pam_data); k++) {
    // Run exactly one test case?
    if ((testcase != NULL) && 
	(strcmp(testcase, pamTests[k].testName) != 0 )) {
      continue;
    }

    printf2("%s()\n", pamTests[k].testName);

    memset(successDataPtr,'\0',sizeof(laus_data));

    successDataPtr->successCase = TRUE;
    successDataPtr->msg_arch = arch;
    successDataPtr->msg_type = AUDIT_MSG_TEXT;
    successDataPtr->msg_euid = uid;
    successDataPtr->msg_egid = gid;
    successDataPtr->testName = pamTests[k].testName;
    
    printf4("Performing test on %s\n", pamTests[k].testName);

    if ((rc = pamTests[k].testPtr(successDataPtr)) != 0) {
      goto EXIT_ERROR;
    }
    
  }

  free(successDataPtr);

  if (test_rc != 0) {
    printf2("ERROR: At least 1 pam program did not execute as expected\n");
    rc = -1;
  }

  printf2("PASSED = %i, FAILED = %i\n", pass_testcases, fail_testcases);

  restoreFile("/etc/audit/filter.conf");
  reloadAudit();

EXIT_HELP:
  if (testcase != NULL)
    free (testcase);
  return rc;

EXIT_ERROR:
  restoreFile("/etc/audit/filter.conf");
  reloadAudit();
  printf1("ERROR: Test aborted: errno = %i\n", errno);
  return rc;

}
