/*
** Copyright (C) 1998-2002 Martin Roesch <roesch@sourcefire.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* Snort Preprocessor for Telnet Negotiation Normalization*/
/* $Id: spp_telnet_negotiation.c,v 1.3.4.1 2002/03/15 14:42:32 chrisgreen Exp $ */

/* spp_telnet_negotiation.c 
 * 
 * Purpose:  Telnet and FTP sessions can contain telnet negotiation strings 
 *           that can disrupt pattern matching.  This plugin detects 
 *           negotiation strings in stream and "normalizes" them much like
 *           the http_decode preprocessor normalizes encoded URLs
 *
 * Arguments:  None
 *   
 * Effect:  The telnet nogiation data is removed from the payload
 *
 * Comments:
 *
 */

/* your preprocessor header file goes here */
#include "spp_telnet_negotiation.h"

/* external globals from rules.c */
extern char *file_name;
extern int file_line;

/*
 * Function: SetupTelNeg()
 *
 * Purpose: Registers the preprocessor keyword and initialization 
 *          function into the preprocessor list.  
 *
 * Arguments: None.
 *
 * Returns: void function
 *
 */
void SetupTelNeg()
{
    /* Telnet negotiation has many names, but we only implement this
     * plugin for Bob Graham's benefit...
     */ 
    RegisterPreprocessor("telnet_neg", TelNegInit);
    RegisterPreprocessor("telnet_negotiation", TelNegInit);
    RegisterPreprocessor("telnet_decode", TelNegInit);

#ifdef DEBUG
    printf("Preprocessor: Telnet Negotiation Decode is setup...\n");
#endif
}


/*
 * Function: TelNegInit(u_char *)
 *
 * Purpose: Calls the argument parsing function, performs final setup on data
 *          structs, links the preproc function into the function list.
 *
 * Arguments: args => ptr to argument string
 *
 * Returns: void function
 *
 */
void TelNegInit(u_char *args)
{
#ifdef DEBUG
    printf("Preprocessor: TelNeg Initialized\n");
#endif

    /* Set the preprocessor function into the function list */
    AddFuncToPreprocList(NormalizeTelnet);
}


/*
 * Function: PreprocFunction(Packet *)
 *
 * Purpose: Perform the preprocessor's intended function.  This can be
 *          simple (statistics collection) or complex (IP defragmentation)
 *          as you like.  Try not to destroy the performance of the whole
 *          system by trying to do too much....
 *
 * Arguments: p => pointer to the current packet data struct 
 *
 * Returns: void function
 *
 */
void NormalizeTelnet(Packet *p)
{
    char *read_ptr;
    char *write_ptr;
    char *end;
    int telneg_present = 0;
    
    /* check for TCP traffic that's part of an established session */
    if(!PacketIsTCP(p) || !IsTcpSessionTraffic(p))
    {
        return;
    }

    /* we're only interested in inbound telnet and ftp traffic */
    if(p->dp != 23 && p->dp != 21)
    {
        return;
    }

    /* negotiation strings are at least 3 bytes long */
    if(p->dsize < TNC_STD_LENGTH)
    {
        return;
    }

    /* setup the pointers */
    write_ptr = read_ptr = (char*)p->data;
    end = (char*)(p->data + p->dsize);
    
    /* look to see if we have any telnet negotiaion codes in the payload */
    while(!telneg_present && (read_ptr++ < end))
    {
        /* look for the start of a negotiation string */
        if(*read_ptr == (char) TNC_IAC)
        {
            /* set a flag for stage 2 normalization */
            telneg_present = 1;
        }
    }

    /* if we found telnet negotiation strings, zap them */
    if(telneg_present)
    {
        /* setup for overwriting the negotaiation strings with 
         * the follow-on data
         */ 
        write_ptr = read_ptr;

        /* walk thru the remainder of the packet */
        while(read_ptr < end)
        {
            /* if the following byte isn't a subnegotiation initialization */
            if(*read_ptr == (char) TNC_IAC && *(read_ptr + 1) != (char) TNC_SB)
            {
                /* NOPs are two bytes long */
                if(*(read_ptr+1) == (char) TNC_NOP)
                {
                    read_ptr += 2;
                    p->dsize -= 2;
                }
                else
                {
                    /* move the read ptr up 3 bytes */
                    read_ptr += TNC_STD_LENGTH;
                    p->dsize -= TNC_STD_LENGTH;
                }
            } /* check for subnegotiation */
            else if(*(read_ptr+1) == (char) TNC_SB) 
            {
                /* move to the end of the subneg */
                do
                {
                    read_ptr++;
                    p->dsize--;
                } while((*read_ptr != (char) TNC_SE) && (read_ptr < end));
            }
            else
            {
                DebugMessage(DEBUG_PLUGIN, "overwriting %2X(%c) with %2X(%c)\n",
                            (char)(*write_ptr&0xFF), *write_ptr, 
                            (char)(*read_ptr & 0xFF), *read_ptr);

                /* overwrite the negotiation bytes with the follow-on bytes */
                *write_ptr++ = *read_ptr++;
            }
        }

        DebugMessage(DEBUG_PLUGIN, 
                     "Converted buffer after telnet normalization:\n");
#ifdef DEBUG        
        PrintNetData(stdout, p->data, p->dsize);
#endif
    }
}
