/* $Id: spp_defrag.c,v 1.21.4.3 2002/03/15 14:42:32 chrisgreen Exp $ */
/*
** Copyright (C) 1998-2002 Martin Roesch <roesch@sourcefire.com>
** Copyright (C) 2000,2001 Dragos Ruiu <dr@kyx.net>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
/*
 * spp_defrag v1.4 - 28 June 2001
 * 
 * Purpose: IP defragmentation preprocessor
 * Author:  Dragos Ruiu (dr@dursec.com/dr@v-wave.com)
 * Acknowledgements:    Code skeleton from diphen@agitation.net
 *                  There's a good paper about splay trees
 *                  by the developer and they rock.
 *                  Marty Roesch(roesch@md.prestige.net)
 *                  and Ron Gula(rgula@networksecuritywizards.com)
 *                  helped and did something few in the security
 *                  community do, shared knowledge.  Thanks.
 *
 * notes:
 *            This defragger implementation differs from the usual
 *            hash table and linked list technique in that it uses
 *            self caching splay trees. My hypothesis is that this
 *            provides substantial performance improvements.  I
 *            hope this module will be able to demonstrate or debunk
 *            this hypothesis. 
 * 
 * Splay Tree:
 * `A self-organizing data structure which uses rotations to move an
 *  accessed key to the root. This leaves recently accessed nodes near
 *  the top of the tree, making them very quickly searchable (Skiena 1997, p. 177). 
 * 
 * All splay tree operations run in O(log n) time _on_average_, where n is the
 * number of items in the tree, assuming you start with an empty tree.  Any single
 * operation can take Theta(n) time in the worst-case, but operations slower than
 * O(log n) time happen rarely enough that they don't affect the average.
 * 
 * Although 2-3-4 trees make a stronger guarantee (_every_ operation on a 2-3-4
 * tree takes O(log n) time), splay trees have several advantages.  Splay trees
 * are simpler and easier to program, and they can take advantage of circumstances
 * in which lots of find operations occur on a small number of items.  Because of
 * their simplicity, splay tree insertions and deletions are typically faster in
 * practice (sometimes by a constant factor, sometimes asymptotically).  Find
 * operations can be faster or slower, depending on circumstances.  Splay trees
 * really excel in applications where a fraction of the items are the targets of
 * most of the find operations, because they're designed to give especially fast
 * access to items that have been accessed recently.
 *
 * a good reference on splay trees is:
 * http://www.cs.umbc.edu/courses/undergraduate/341/fall98/frey/ClassNotes/Class17/splay.html
 * 
 * References 
 * Skiena, S. S. The Algorithm Design Manual. New York: Springer-Verlag, pp. 177 and 179, 1997. 
 * Sleator, D. and Tarjan, R. "Self-Adjusting Binary Search Trees." J. ACM 32, 652-686, 1985. 
 * Tarjan, R. Data Structures and Network Algorithms. Philadelphia, PA: SIAM Press, 1983. 
 * Wood, D. Data Structures, Algorithms, and Performance. Reading, MA: Addison-Wesley, 1993. 
 * 
 */

/* Changelog....
*  16/06/01 v1.1 --dr
*          Fixed reassembledgram ip length to include IP header
*	   (Thanks Aaron Richard Walters <awalters@nd.edu> for tracking this down)
*	   Redid timeout flushing and garbage sweeping.
*	   New algorithm uses progrssively more aggressive timeouts
*          as mem utilization approaches ~32Mb hard limit.
*	   Also flushes related fragments together.
*  17/06/01 v1.2 --dr
*	   Pseudorandom splay tree balancing added.
*  28/06/06 v1.3 --dr
*	   Memory hard limit added
*  28/06/06 v1.4 --dr
*	   Corner case memory leak fixed.
*/

#include "snort.h"

/* Warning these are only valid for IPv4
   also note that these are still 
   in raw network order (see htons)  
   and these may fail on 64 bit machines 
*/
#define ID(x)       *((u_int16_t *)((u_int8_t *)x->iph+4))
#define PROTO(x)    *((u_int8_t *)x->iph+9)
#define SADDR(x)    ( x->iph->ip_src.s_addr )
#define DADDR(x)    ( x->iph->ip_dst.s_addr )
#define DATA(x)     ((u_int8_t *)x->iph+20)

/* Uh-oh hope this wierdness is right :-) */
#define FOFF(x)     (u_int32_t)((x->frag_offset)<<3)
#define DF(x)       (x->df)
#define MF(x)       (x->mf)


/* fragment ID structure  */
typedef Packet *frag;

typedef struct tree_node Tree;
struct tree_node
{
    Tree * left, * right;
    frag key;
    int size;   /* maintained to be the number of nodes rooted here */
};

Tree *froot;

u_int32_t mem_locked;
u_int32_t mem_freed;

/* Garbage collection stack (linked list) */

typedef struct list_node List;
struct list_node
{
	List * next;
	frag key;
};

List *garbagelist;
char trashejectspinlock;
	

/*  These next declarations are for the fragment timeout and 
    and cleanup/sweeping process... time math routines are from
    an obscure piece of old code for a defunct video camera product
*/

#define FRAGTIMEOUTSEC      10      /* 10 seconds let's play safe for now */
#define FRAGTIMEOUTUSEC      0      /* 0 micro seconds                  */
#define MEMHARDLIM     	  32000000  /* memory hardlimit threshold */

int fragmemuse;
/* the packet timeout / garbage collection stuff  */

int spacealert;  /* user already alerted about condition */

typedef struct _timestruct
{
    u_int32_t tv_sec;
    u_int32_t tv_usec;
} time_struct;

time_struct fragtimeout;
time_struct last_frag_time;
time_struct timecheck;



/******Timestamp Routines******/

#define TIME_LT(x,y) (x tv_sec<y tv_sec||(x tv_sec==y tv_sec&&x tv_usec<y tv_usec))

void addtime(time_struct *op1, time_struct *op2, time_struct *result)
{
    result->tv_usec = op1->tv_usec+op2->tv_usec;
    if(result->tv_usec > 999999)
    {
        result->tv_usec -= 1000000;
        op1->tv_sec++;
    }
    result->tv_sec = op1->tv_sec+op2->tv_sec;
}



/*********Memory mangement**********/
void *DefragAlloc(unsigned long size)
{
    void *tmp;

    fragmemuse += size;

    /* if we use up all of our RAM, try to free up some stale frags */
    if(fragmemuse > MEMHARDLIM)
    {
        /* do something here */
    }

    tmp = (void *) calloc(size, sizeof(char));

    if(tmp == NULL)
    {
        FatalError("spp_defrag: Unable to allocate memory! "
                   "(%lu bytes in use)\n", fragmemuse);
    }

    return tmp;
}




/******Splay Tree Stuff******/

/* Function: fragcompare(i,j)                            */
/* This is the splay tree comparison.           */
/* Returns 1 if i>j; 0 if i=j; -1 if i<j;       */

int fragcompare(i,j)
frag i,j;
{
    if(!j)
    {
        if(!i)
            return(0);
        else
            return(1);
    }
    else
    {
        if(!trashejectspinlock && TIME_LT(timecheck. , j->pkth->ts. ))
	{  /* bzzt you lose... too much time.... the answer is.... */
		/* toss that puppy later by adding it to garbagelist */
		List *new;
		new = (List *) DefragAlloc(sizeof(List));
		new->next = garbagelist;
		new->key = j;
		garbagelist = new;
#ifdef DEBUG
    fprintf(stderr, "+++++++++++++++++++++++++\n");
    fprintf(stderr, "%p + List Alloc\n", new);
    fprintf(stderr, "+++++++++++++++++++++++++\n");
    fflush(stderr);
    mem_locked += sizeof(List);
#endif

	} 
        if(!i)
            return(-1);
    }
    if(j->iph == NULL)
    {
        if(i->iph == NULL)
        {
            return 0;
        }
        else
        {
            return 1;
        }
    }
    else
    {
        if(i->iph == NULL)
        {
            return -1;
        }
    }

    if(SADDR(i) > SADDR(j))
    {
        return(1);
    }
    else if(SADDR(i) < SADDR(j))
    {
        return(-1);
    }
    else if(DADDR(i) > DADDR(j))
    {
        return(1);
    }
    else if(DADDR(i) < DADDR(j))
    {
        return(-1);
    }
    else if(PROTO(i) > PROTO(j))
    {
        return(1);
    }
    else if(PROTO(i) < PROTO(j))
    {
        return(-1);
    }
    else if(ID(i) > ID(j))
    {
        return(1);
    }
    else if(ID(i) < ID(j))
    {
        return(-1);
    }
    else if((i->frag_offset)<<3 > (j->frag_offset)<<3)
    {
        return(1);
    }
    else if((i->frag_offset)<<3 < (j->frag_offset)<<3)
    {
        return(-1);
    }
    else if(i->dsize > j->dsize)
    {
        return(1);
    }
    else if(i->dsize < j->dsize)
    {
        return(-1);
    }
    return(0);
}

/* This macro returns the size of a node.  Unlike "x->size",     */
/* it works even if x=NULL.  The test could be avoided by using  */
/* a special version of NULL which was a real node with size 0.  */

#define node_size(x) ((!x) ? 0 : ((x)->size))

/* Function: fragsplay(i, t)                              */
/* Splay using the key i (which may or may not be in the tree.) */
/* The starting root is t, size fields are maintained            */

Tree *fragsplay(frag i, Tree *t) 
{
    Tree N, *l, *r, *y;
    int comp, root_size, l_size, r_size;
    if(!t) return t;
    N.left = N.right = NULL;
    l = r = &N;
    root_size = node_size(t);
    l_size = r_size = 0;

    for(;;)
    {
        comp = fragcompare(i, t->key);
        if(comp < 0)
        {
            if(!t->left) break;
            if(fragcompare(i, t->left->key) < 0)
            {
                y = t->left;                           /* rotate right */
                t->left = y->right;
                y->right = t;
                t->size = node_size(t->left) + node_size(t->right) + 1;
                t = y;
                if(!t->left) break;
            }
            r->left = t;                               /* link right */
            r = t;
            t = t->left;
            r_size += 1+node_size(r->right);
        }
        else if(comp > 0)
        {
            if(!t->right) break;
            if(fragcompare(i, t->right->key) > 0)
            {
                y = t->right;                          /* rotate left */
                t->right = y->left;
                y->left = t;
                t->size = node_size(t->left) + node_size(t->right) + 1;
                t = y;
                if(!t->right) break;
            }
            l->right = t;                              /* link left */
            l = t;
            t = t->right;
            l_size += 1+node_size(l->left);
        }
        else
        {
            break;
        }
    }
    l_size += node_size(t->left);  /* Now l_size and r_size are the sizes of */
    r_size += node_size(t->right); /* the left and right trees we just built.*/
    t->size = l_size + r_size + 1;

    l->right = r->left = NULL;

    /* The following two loops correct the size fields of the right path  */
    /* from the left child of the root and the right path from the left   */
    /* child of the root.                                                 */
    for(y = N.right; y; y = y->right)
    {
        y->size = l_size;
        l_size -= 1+node_size(y->left);
    }
    for(y = N.left; y; y = y->left)
    {
        y->size = r_size;
        r_size -= 1+node_size(y->right);
    }

    l->right = t->left;                                /* assemble */
    r->left = t->right;
    t->left = N.right;
    t->right = N.left;

    return t;
}

/* Function: Tree * fraginsert(frag i, Tree * t)             */
/* Insert frag i into the tree t, if it is not already there.       */
/* Return a pointer to the resulting tree.                         */

Tree *fraginsert(frag i, Tree * t)
{
    Tree * new_tree_node;
    if(t)
    {
        t = fragsplay(i,t);
        if(fragcompare(i, t->key)==0)
        {
            return t;  /* it's already there */
        }
    }

    new_tree_node = (Tree *) DefragAlloc (sizeof (Tree));
#ifdef DEBUG
    fprintf(stderr, "+++++++++++++++++++++++++\n");
    fprintf(stderr, "%p + Tree Alloc\n", new_tree_node);
    fprintf(stderr, "+++++++++++++++++++++++++\n");
    fflush(stderr);
    mem_locked += sizeof(Tree);
#endif

    if(!new_tree_node || fragmemuse > MEMHARDLIM)
    {
        ErrorMessage("Ran out of space\n");
        spacealert++;
        return(t);
    }

    /* alert hysterisis(?) */
    if (spacealert)
        spacealert -= ((unsigned int)spacealert) >> 1;
    if (spacealert < 0)
        spacealert = 0;
    if(!t)
    {
        new_tree_node->left = new_tree_node->right = NULL;
    }
    else
        if(fragcompare(i, t->key) < 0)
        {
            new_tree_node->left = t->left;
            new_tree_node->right = t;
            t->left = NULL;
            t->size = 1+node_size(t->right);
        }
        else
        {
            new_tree_node->right = t->right;
            new_tree_node->left = t;
            t->right = NULL;
            t->size = 1+node_size(t->left);
        }
    new_tree_node->key = i;
    new_tree_node->size = 1 + node_size(new_tree_node->left) + node_size(new_tree_node->right);
    return new_tree_node;
}

/* Function: Tree * fragdelete(frag i, Tree *t)       */
/* Deletes i from the tree if it's there.               */
/* Return a pointer to the resulting tree.              */

Tree *fragdelete(frag i, Tree *t)
{
    Tree * x;
    int tsize;

    if(!t) return NULL;
    tsize = t->size;
    t = fragsplay(i,t);
    if(fragcompare(i, t->key) == 0)
    {               /* found it */
        if(!t->left)
        {
            x = t->right;
        }
        else
        {
            x = fragsplay(i, t->left);
            x->right = t->right;
        }

        if(t->key != NULL)
        {
#ifdef DEBUG
            fprintf(stderr, "------------------------\n");
            fprintf(stderr,"%p - p-struct Free\n", t->key);
            fprintf(stderr, "------------------------\n");
            fflush(stderr);        
            mem_freed += sizeof(Packet);
#endif
            fragmemuse -= sizeof(Packet);
            free(t->key);
        }

#ifdef DEBUG
        fprintf(stderr, "------------------------\n");
        fprintf(stderr,"%p - Tree Free\n", t);
        fprintf(stderr, "------------------------\n");
        fflush(stderr);        
        mem_freed += sizeof(Tree);
#endif
        fragmemuse -= sizeof(Tree);
        free(t);

        if(x)
        {
            x->size = tsize-1;
        }
        return x;
    }
    else
    {
        return t;                         /* It wasn't there */
    }
}

/* Function: Tree *fragfind_rank(int r, Tree *t)               */ 
/* Returns a pointer to the node in the tree with the given rank.  */
/* Returns NULL if there is no such node.                          */
/* Does not change the tree.  To guarantee logarithmic behavior,   */
/* the node found here should be splayed to the root.              */

Tree *fragfind_rank(int r, Tree *t)
{
    int lsize;
    if((r < 0) || (r >= node_size(t))) return NULL;
    for(; t != 0 ;)
    {
        lsize = node_size(t->left);
        if(r < lsize)
        {
            t = t->left;
        }
        else if(r > lsize)
        {
            r = r - lsize -1;
            t = t->right;
        }
        else
        {
            return t;
        }
    }
    return NULL;
}

/* Function: fragget_rank(t)                                 */
/* returns the rank of a tree element                           */

int fragget_rank(Tree *t)
{
    if(!t)
        return(0);
    return(node_size(t->left)+1);
}


/* Function: treedump(t)                                        */
/* dumps the tree in table form                                 */
void treedump(Tree *t)
{
    int loc;
    Tree *temp;
    fprintf(stdout,"Rank Ptr      ID       Lptr      Rptr \n");
    fprintf(stdout,"========================================== \n");
    for(loc = 0; loc < node_size(t); loc++)
    {
        temp = fragfind_rank(loc, t);
        fprintf(stdout,"%3d %p %d %p %p\n", loc, temp, ID(temp->key), temp->left, temp->right);
    }
    fflush(stdout);
}

/******Snort Stuff******/

/*
 * Function: SetupDefrag()
 * Purpose:
 * Registers the preprocessor keyword and initialization function
 * into the preprocessor list.  This is the function that gets called from
 * InitPreprocessors() in plugbase.c.
 * Arguments: None.
 * Returns: void function
 */
void SetupDefrag()
{
    RegisterPreprocessor("defrag", DefragInit);
}

/*
 * Function: DefragInit(u_char *)
 * Purpose:
 * Calls the argument parsing function, performs final setup on data
 * structs, links the preproc function into the function list.
 * Arguments: args => ptr to argument string
 * Returns: void function
 */
void DefragInit(u_char *args)
{
    AddFuncToPreprocList(PreprocDefrag);

    froot = NULL;  /* initialize empty fragment tree */
    garbagelist = NULL;  /* initialize empty trash list */
    trashejectspinlock = FALSE;
    fragmemuse = 0;      /* No memory yet */
    spacealert = 0;     /* No space alerts yet */
    fragtimeout.tv_sec = FRAGTIMEOUTSEC;
    fragtimeout.tv_usec = FRAGTIMEOUTUSEC;
}


 
/*************Tree Balance Performance Enhancement**************/
 
int balancecompare(point,j)
unsigned int point;
frag j;
{
    if(!j)
    {
        return(0);
    }
    else
    {
        if(j->iph == NULL)
        {
            return 0;
        }
    }
    if(point > SADDR(j))
    {
        return(1);
    }
    else if(point < SADDR(j))
    {
        return(-1);
    }
    return(0);
}
 
 
/* shift based bit revrser --dr */
 
unsigned int bitrev(x)
unsigned int x;
{
unsigned int i, r = 0;
        for( i = 0 ; i <= sizeof(unsigned int)*8 ; i++)
                if ( x & (unsigned int)1<<i )
                        r |= (unsigned int)1<<((sizeof(unsigned int)*8)-i-1);
        return r;
}
 
 
/* Function: fragbalance(i, t)                              */
/* Splay using the key i (which may or may not be in the tree.) */
/* The starting root is t, size fields are maintained            */
 
Tree *fragbalance(unsigned int balancepoint, Tree *t)
{
    Tree N, *l, *r, *y;
    int comp, root_size, l_size, r_size;
    if(!t) return t;
    N.left = N.right = NULL;
    l = r = &N;
    root_size = node_size(t);
    l_size = r_size = 0;
    for(;;)
    {
        comp = balancecompare(balancepoint, t->key);
        if(comp < 0)
        {
            if(!t->left) break;
            if(balancecompare(balancepoint, t->left->key) < 0)
            {
                y = t->left;                           /* rotate right */
                t->left = y->right;
                y->right = t;
                t->size = node_size(t->left) + node_size(t->right) + 1;
                t = y;
                if(!t->left) break;
            }
            r->left = t;                               /* link right */
            r = t;
            t = t->left;
            r_size += 1+node_size(r->right);
        }
        else if(comp > 0)
        {
            if(!t->right) break;
            if(fragcompare(balancepoint, t->right->key) > 0)
            {
                y = t->right;                          /* rotate left */
                t->right = y->left;
                y->left = t;
                t->size = node_size(t->left) + node_size(t->right) + 1;
                t = y;
                if(!t->right) break;
            }
            l->right = t;                              /* link left */
            l = t;
            t = t->right;
            l_size += 1+node_size(l->left);
        }
        else
        {
            break;
        }
    }
    l_size += node_size(t->left);  /* Now l_size and r_size are the sizes of */
    r_size += node_size(t->right); /* the left and right trees we just built.*/
    t->size = l_size + r_size + 1;
 
    l->right = r->left = NULL;
 
    /* The following two loops correct the size fields of the right path  */
    /* from the left child of the root and the right path from the left   */
    /* child of the root.                                                 */
    for(y = N.right; y; y = y->right)
    {
        y->size = l_size;
        l_size -= 1+node_size(y->left);
    }
    for(y = N.left; y; y = y->left)
    {
        y->size = r_size;
        r_size -= 1+node_size(y->right);
    }
 
    l->right = t->left;                                /* assemble */
    r->left = t->right;
    t->left = N.right;
    t->right = N.left;
 
    return t;
}

/******Fragmentation Stuff******/

/* Function: fragaddrmatch(i,j)                         */
/* returns true if the fragments belong to the same reassembly      */
int fragaddrmatch(i,j)
frag i,j;
/* This is the comparison.                                   */
/* Returns 1 if i j have matching addresses else 0           */
{
#ifdef DEBUG
    printf("   =>  sip1 = 0x%X, sip2 = 0x%X\n", SADDR(i), SADDR(j));
    printf("   =>  dip1 = 0x%X, dip2 = 0x%X\n", DADDR(i), DADDR(j));
    printf("   =>  id1 = %d, id2 = %d\n", ID(i), ID(j));
    printf("   =>  proto1 = %d, proto2 = %d\n", PROTO(i), PROTO(j));
#endif    

    if(( SADDR(i) == SADDR(j) )
       && ( DADDR(i) == DADDR(j) )
       && ( ID(i) == ID(j) )
       && ( PROTO(i) == PROTO(j) ))
    {
        return(1);
    }

    return(0);
}

/*
 * Function: Packet *ReassembleIP(frag *froot)
 * Purpose: Generate a Packet * and pass it to ProcessPacket, then 
 * deallocate packet
 * Arguments: froot - root of the tree containing the last packet of the frame
 * Returns: tree with reassembled fragments deleted
 */
Tree *ReassembleIP(Tree *froot)
{
    Packet *p;
    int writecount = 0;
    char *np;
    u_char *tmp;
    u_int psize;
    int fragrank;
    unsigned int overhead, moreoverhead;
    IPHdr slowarisfix;
    Event event;

    if(froot == NULL)
    {
        ErrorMessage("Got NULL *froot in ReassembleIP(), please tell Dragos\n");
        return NULL;
    }

    psize = (froot->key)->dsize + ((froot->key->frag_offset)<<3); /* last frag is at top of tree */

#ifdef DEBUG
    printf("psize calculated to be %d bytes (last payload: %d)"
           "(last offset: %d)\n", psize, (froot->key)->dsize, 
           ((froot->key->frag_offset)<<3));
#endif        

    /* we've got to have a packet at least as large as an ICMP header */
    if(psize < 4)
    {
        ErrorMessage("[!] ERROR: Defrag plugin is fucked up, "
                     "calculated packet size too small\n");
    }

    /* hopefully moreoverhead here finds out about all the wierd MAC 
     * sizes like FDDI and ATM 
     */
    moreoverhead = (char*)froot->key->iph - (char*)froot->key->pkt;
    overhead = (char*)froot->key->pkt - (char*)froot->key->pkth;
    
    /* Linux fix code */
    if(overhead < sizeof(struct pcap_pkthdr) || 
       overhead > sizeof( struct pcap_pkthdr) + 2)
    {
        overhead = sizeof(struct pcap_pkthdr) + 2;
    }

#ifdef DEBUG
    printf("Overhead = %d, struct size = %d\n", overhead, 
            sizeof(struct pcap_pkthdr));
#endif        

    p = (Packet *)DefragAlloc(sizeof(Packet));

#ifdef DEBUG
    fprintf(stderr, "+++++++++++++++++++++++++\n");
    fprintf(stderr, "%p + p-struct Alloc\n", p);
    fprintf(stderr, "+++++++++++++++++++++++++\n");
    fflush(stderr);
    mem_locked += sizeof(Packet);
#endif

    if(!p)
    {
        ErrorMessage("[!] ERROR: Unable to allocate memory for "
                     "fragment rebuild!\n");
        return NULL;
    }

    /* MFR: Why are we copying this header if we're in the reassembly phase?
     *  Why not use the original and modify its data in place?
     */
    /* because the original buffer isn't big enough to reassemble in --dr */
    memcpy(p, froot->key, sizeof(Packet));

    /* SEMI BOGUS - setting up ethernet time header - required by one of the
     * decode routines - we set the timestamp equal to froot (last fragment)
     * note also that our bogus reassembledgrams may not have the bpf header
     * contiguous and datagram access through this pointer is suspect. Been 
     * warned.
     */
    p->pkth = (struct pcap_pkthdr *)
                DefragAlloc(psize + overhead + moreoverhead + sizeof(IPHdr) + 32);

#ifdef DEBUG
    fprintf(stderr, "+++++++++++++++++++++++++\n");
    fprintf(stderr, "%p + frankenpacket Alloc\n", p->pkth);
    fprintf(stderr, "+++++++++++++++++++++++++\n");
    fflush(stderr);
    mem_locked += sizeof(psize+overhead+moreoverhead+sizeof(IPHdr)+32);
#endif

    if(!p->pkth)
    {
        ErrorMessage("[!] ERROR: Unable to allocate memory for fragment "
                     "rebuild!\n");
#ifdef DEBUG
        fprintf(stderr, "------------------------\n");
        fprintf(stderr, "%p - p-struct Free\n", p);
        fprintf(stderr, "------------------------\n");
        fflush(stderr);
        mem_freed += sizeof(Packet);
#endif
        fragmemuse -= sizeof(Packet);
        free(p);
        return NULL;
    }

    p->iph = (IPHdr *)((u_char*)p->pkth + overhead + moreoverhead);
    p->pkt = (u_char*)p->iph - moreoverhead;

    /*
     * Now copy the header and fragments into the newly-allocated buffer,
     * reconstructing the packet chunk by chunk. Current algorithm is first 
     * write to an area wins, play with splay order to change.
     * we start from the last fragment and work back....
     */

    /* fill in packet header bpf header first*/
    memcpy(p->pkth, froot->key->pkth, overhead); 

    /* then the mac junk, split into two copies for Linux 
     * non-contiguous headers 
     */
    memcpy((char*)p->pkth + overhead, froot->key->pkt, moreoverhead); 

    /*
     * then the IP header just to be paranoid for debugging because in the
     * real world we would do these in one copy
     */
    tmp = (u_char *) froot->key->iph;
    memcpy(p->iph, tmp, sizeof(IPHdr)); 

    p->pkth->caplen = psize + overhead + moreoverhead + sizeof(IPHdr);
    p->pkth->len = p->pkth->caplen;

    /*
     * Clear the more fragments bit, and set the length in the ip header
     * (in network byte order).
     */
    p->iph->ip_len = htons( (u_short) (psize+sizeof(IPHdr)));
    p->iph->ip_off = 0;
    p->frag_flag = 0;

    fragrank = fragget_rank(froot);

#ifdef DEBUG
    printf("fragrank = %d, froot = %p, fragaddrmatch = %d\n", fragrank, froot, 
            fragaddrmatch(p, froot->key));
#endif        

    while(fragrank > 0 && froot && fragaddrmatch(p,froot->key))
    {
        if( (u_int)(((froot->key->frag_offset)<<3) + froot->key->dsize) <= psize)
        {
#ifdef DEBUG
            printf("Writing %d bytes from %p at %p\n", froot->key->dsize, 
                    DATA(froot->key), 
                    (u_int8_t *) DATA(p)+((froot->key->frag_offset)<<3));        
#endif            
            memcpy((u_int8_t *)DATA(p)+((froot->key->frag_offset)<<3), 
                    DATA(froot->key), froot->key->dsize);

            writecount += froot->key->dsize;

#ifdef DEBUG
            printf("reassembly writecount: %d\n", writecount);
#endif                        
        }
        else
        {
#ifdef DEBUG
            printf("Overflow attack in rebuild!\n");
#endif                        
            SetEvent(&event, GENERATOR_SPP_DEFRAG, 
                    DEFRAG_FRAG_OVERFLOW, 1, 0, 0, 0);
            CallAlertFuncs(p, "Fragmentation Overflow Attack", NULL, &event);
        }

        /* clear the fragment store of the frag that was just put into the
         * reassembly
         */
        if(froot)
        {
            np = (char *)froot->key->pkth;  /* address for free later */

            fragrank = fragget_rank(froot);

            froot = fragdelete(froot->key, froot);

            if(np)
            {
#ifdef DEBUG
                fprintf(stderr, "------------------------\n");
                fprintf(stderr, "%p - packet Free\n", np);
                fprintf(stderr, "------------------------\n");
                fflush(stderr);
                mem_freed += ((struct pcap_pkthdr *)np)->caplen + overhead + 20;
#endif                
                /* this fragment was an element of a rebuilt packet */
                pc.rebuild_element++;
                fragmemuse -= (((struct pcap_pkthdr *)np)->caplen + 
                              overhead + 20);
                free(np);
            }
        }
    }

    memcpy(&slowarisfix, p->iph, sizeof(IPHdr));
    p->iph = &slowarisfix; 

    /* and now some jolt2proofing */
    /* psize check increased from 8192 to 65535-mtusize, these packets can get
     * big legitimately - MFR */
    /* moved back down to 8192 to account for swisscheesegram attack --dr */
    if(psize > 8192)  /* only kick in for monstergrams */
    {
        /*packets have to be half full for us to look at them */
        if( ((u_int)writecount) > (psize>>1)) 
        {
            if(p)
            {
#ifdef DEBUG
                printf("Fragmented packet rebuilt, processing...\n");
                printf("BPF header (%d bytes):\n", overhead);
                PrintNetData(stdout,(char *) p->pkth, overhead);
                ClearDumpBuf();
                printf("Packet dump (%d bytes):\n", p->pkth->caplen);
                PrintNetData(stdout, (char *)p->pkt, p->pkth->caplen);
                ClearDumpBuf();
#endif                                    
                /* keep stats for the packet counter struct -MFR */
                pc.rebuilt_frags++;  /* Marty... Huh?? --dr */
                ProcessPacket(NULL,p->pkth, p->pkt);
            }
        }
        else
        {
            SetEvent(&event, GENERATOR_SPP_DEFRAG, 
                    DEFRAG_FRAGS_DISCARDED, 1, 0, 0, 0);
            CallAlertFuncs(p, "Incomplete Packet Fragments Discarded", 
                    NULL, &event);
        }
    }
    else
    {
        if(p)
        {
#ifdef DEBUG
            printf("Fragmented packet rebuilt, processing...\n");
            printf("BPF header (%d bytes):\n", overhead);
            ClearDumpBuf();
            PrintNetData(stdout,(char *) p->pkth, overhead);
            ClearDumpBuf();
            printf("Packet dump (%d bytes):\n", p->pkth->caplen);
            PrintNetData(stdout,(char *) p->pkt, p->pkth->caplen);
            ClearDumpBuf();
#endif                            
            /* keep stats for the packet counter struct -MFR */
            pc.rebuilt_frags++;  /* Marty... Huh?? --dr */
            ProcessPacket(NULL, p->pkth, p->pkt);
        }
    }

#ifdef DEBUG
    fprintf(stderr, "------------------------\n");
    fprintf(stderr, "%p - frankenpacket Free\n",(p->pkth));
    fprintf(stderr, "%p - p-struct Free\n", p);
    fprintf(stderr, "------------------------\n");
    fflush(stderr);
    treedump(froot);
    mem_freed += psize+overhead+moreoverhead+sizeof(IPHdr)+32;
    mem_freed += sizeof(Packet);
#endif    
    fragmemuse -= (psize+overhead+moreoverhead+sizeof(IPHdr)+32+sizeof(Packet));
    free(p->pkth);      /* set the frankensteingram free.... */
    free(p);

    return(froot);
}

/*
 * Function: PreprocDefrag(Packet *)
 * Purpose:
 * Driver function for the IP defragmentation preprocessor.
 * Arguments: p => pointer to the current packet data struct
 * Returns: void function
 */
void PreprocDefrag(Packet *p)
{
    Packet *packet_copy;
    struct pcap_pkthdr *freetemp;
    int overhead;
    int cap;
    u_char *tmp;
    Tree *new_froot;

#ifdef DEBUG
    printf("fragments =>\n");
    printf("    mem used: %u\n", mem_locked);
    printf("    mem freed: %u\n", mem_freed);
    printf("    fragmemuse: %d\n", fragmemuse);
#endif    

    if(!p || !p->pkth || !p->pkt)
    {
        if(pv.verbose_flag)
        {
            ErrorMessage("%s\n","Garbage Packet with Null Pointer discarded!");
        }

        return;
    }

    /* check to make sure the IP header exists and that 
     * there isn't a bad IP checksum
     */
    if(!p->iph || (p->csum_flags & CSE_IP))
    {
        return;      
    }

    if(p->frag_flag)
    {  /* heads up, live fragments inbound  */
        overhead = (char*)p->pkt - (char*)p->pkth; /* +4 mystery points */
        /* Linux fix code */
        if(overhead < 1 || overhead > sizeof(struct pcap_pkthdr) + 2)
            overhead = sizeof(struct pcap_pkthdr) + 2;

        packet_copy = (Packet *) DefragAlloc(sizeof(Packet));

#ifdef DEBUG
        fprintf(stderr, "++++++++++++++++++++++++\n");
        fprintf(stderr, "%p + p-struct Alloc (%d bytes) ", 
                packet_copy, sizeof(Packet));

        fflush(stderr);
        mem_locked += sizeof(Packet);
#endif

        if(!packet_copy)
        {
            ErrorMessage("[!] ERROR: Cannot allocate fragment "
                         "buffer(usage 0x%X)\n",fragmemuse);
            return;
        }


        memcpy(packet_copy, p, sizeof(Packet));
        cap = p->pkth->caplen + overhead;
        tmp = (u_char *) DefragAlloc(cap + 20);
        packet_copy->pkth = (struct pcap_pkthdr *) tmp;

#ifdef DEBUG
        fprintf(stderr, " && %p + packet Alloc\n", packet_copy->pkth);
        fprintf(stderr, "++++++++++++++++++++++++\n");
        fflush(stderr);
        mem_locked += cap+20;
#endif
        if(!packet_copy->pkth)
        {
            free(packet_copy);
            fragmemuse -= sizeof(Packet);
            ErrorMessage("[!] ERROR: Cannot allocate fragment "
                         "buffer(usage %X)\n",fragmemuse);
            return;
        }

        packet_copy->pkt = (u_char*)packet_copy->pkth + overhead;
        packet_copy->iph = (IPHdr*)((u_char*)packet_copy->pkt + 
                           ((u_char*)p->iph - (u_char*)p->pkt));

        /* cleared by MFR, accounted for below */
        /*fragmemuse += p->pkth->caplen + overhead;*/
        last_frag_time.tv_sec = p->pkth->ts.tv_sec;
        last_frag_time.tv_usec = p->pkth->ts.tv_usec;
        /****** update the timeout threshold global so fragcompare can
         ****** use it.
         ******/

        if(fragmemuse < 0)
            fragmemuse = 0;
        fragtimeout.tv_sec = 
            (int)((float)FRAGTIMEOUTSEC*((MEMHARDLIM-fragmemuse)/
                                         (float)MEMHARDLIM));
        fragtimeout.tv_usec = 
            (int)((float)FRAGTIMEOUTUSEC*((MEMHARDLIM-fragmemuse)/
                                          (float)MEMHARDLIM));
#ifdef DEBUG                    
        fprintf(stderr, "Timeout=%d,%d \n", fragtimeout.tv_sec, 
                fragtimeout.tv_usec);
#endif
        addtime(&last_frag_time, &fragtimeout, &timecheck);

        /* we do this with two memcopies to cope with Linux 
         * non-contiguous bpf headers 
         */
        memcpy(packet_copy->pkth, p->pkth, overhead);
        memcpy(packet_copy->pkt, p->pkt, p->pkth->caplen);
        new_froot = fraginsert(packet_copy, froot); 
        if(new_froot == froot)
        {
            /* out of memory toss the copies */
            free(packet_copy->pkth);
            fragmemuse -= (cap + 20);
            free(packet_copy);
            fragmemuse -= sizeof(Packet);
        }
        froot = new_froot;

        /* now check if we have to reassemble anything... */
        if(!MF(p))
            froot = ReassembleIP(froot);

        /* OK now eject the trash (fragment timeout :-) */
        if(froot)
        {
            while(garbagelist)
            {
                List *trash;
                trashejectspinlock = TRUE;
#ifdef DEBUG                    
                fprintf(stderr, "Timeout cleanup in progress...\n");
#endif
                freetemp = garbagelist->key->pkth;
#ifdef DEBUG    
                fprintf(stderr, "------------------------\n");
                fprintf(stderr, "%p - packet Free\n", freetemp);
                fprintf(stderr, "------------------------\n");
                fprintf(stderr, "------------------------\n");
                fprintf(stderr, "%p - List  Free\n", garbagelist);
                fprintf(stderr, "------------------------\n");
                fflush(stderr);
                mem_freed += freetemp->caplen + overhead + 20;
#endif
                fragmemuse -= freetemp->caplen + overhead + 20;
                froot = fragdelete(garbagelist->key, froot);
                free(freetemp);
                pc.frag_timeout++;
                trash = garbagelist;
                garbagelist = garbagelist->next;
                free(trash);
                fragmemuse -= sizeof(List);
            }
            trashejectspinlock = FALSE;

        }
    }

    return;
}



