/*
** Copyright (C) 1998-2002 Martin Roesch <roesch@sourcefire.com>
** Copyright (C) 2000,2001 Andrew R. Baker <andrewb@uab.edu>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* $Id: parser.c,v 1.26.2.5 2002/05/12 20:47:30 chrisgreen Exp $ */

#include "parser.h"
extern char *file_name;
extern int file_line;
extern RuleListNode *RuleLists;
extern int rule_count;
extern ListHead *head_tmp;

/****************************************************************************
 *
 * Function: ProcessClassificationConfig(char *)
 *
 * Purpose: parses the classification configuration
 *
 * Arguments: filespec => the file specification
 *
 * Returns: void function
 *
 ***************************************************************************/
void ProcessClassificationConfig(char *args)
{
    char **ctoks;
    int num_ctoks;
    int i;
    char *data;
    int class_id = -1;
    ClassTypes *newNode;
    ClassTypes *current = pv.ct;

    ctoks = mSplit(args, ",",3, &num_ctoks, '\\');

    if(num_ctoks < 1)
    {
        ErrorMessage("WARNING %s(%d): You must supply at least ONE"
                " classification arguement\n", file_name, file_line);
        ErrorMessage("WARNING %s(%d): Ignoring configuration directive (%s: %d)"
                "\"config classification: %s\"\n", file_name, file_line, args);
        return;
    }
    
    data = ctoks[0];
    while(isspace((int)*data)) data++;

    while(current != NULL)
    {
        if(strlen(current->type) == strlen(data))
        {
            if(!strncasecmp(current->type, data, strlen(current->type)))
            {
                ErrorMessage("WARNING %s(%d): Duplicate classification \"%s\""
                        "found, ignoring this line\n", file_name, file_line, data);
                return;
            }
        }

        class_id = current->id;
        current = current->next;
    }

    /* Create the new node */
    if((newNode = (ClassTypes *) calloc(sizeof(ClassTypes), sizeof(char)))
            == NULL)
    {
        FatalError("ERROR => Can't add a new class type, calloc failed\n");
    }

    newNode->type = strdup(ctoks[0]);
    newNode->id = ++class_id;

    if(num_ctoks == 2)
    {
        data = ctoks[1];
        while (isspace((int)*data)) data++;
        newNode->priority = atoi(data);

        data = ctoks[0];
        while (isspace((int)*data)) data++;
        newNode->name = strdup(data);
    }
    else
    {
        data = ctoks[1];
        while (isspace((int)*data)) data++;
        newNode->name = strdup(data);


        data = ctoks[2];
        while (isspace((int)*data)) data++;
        newNode->priority = atoi(data);
    }
        

    /* Add the node to the list */
    if(pv.ct == NULL)
    {
        pv.ct = newNode;
    }
    else
    {
        current = pv.ct;

        while(current->next != NULL)
            current = current->next;
        
        current->next = newNode;
    }

    for(i=0; i<num_ctoks; i++)
    {
        free(ctoks[i]);
    }

#ifdef DEBUG
    printf("classification list:\n");
    current = pv.ct;
    i = 0;
    while(current != NULL)
    {
        printf("Node %d   type: %s   name: %s   pri: %d\n", i, current->type,
                current->name, current->priority);
        i++;
        current = current->next;
    }
#endif

    return;
}



/****************************************************************************
 *
 * Function: ProcessAlertFileOption(char *)
 *
 * Purpose: define the alert file
 *
 * Arguments: filespec => the file specification
 *
 * Returns: void function
 *
 ***************************************************************************/
void ProcessAlertFileOption(char *filespec)
{
    pv.alert_filename = ProcessFileOption(filespec);

#ifdef DEBUG
    printf("alertfile set to: %s\n", pv.alert_filename);
#endif

    return;
}

char *ProcessFileOption(char *filespec)
{
    char *filename;
    char buffer[STD_BUF];

    if(filespec == NULL)
    {
        FatalError("ERROR: no arguement in this file option, remove extra ':' at the end of the alert option\n");
    }

    /* look for ".." in the string and complain and exit if it is found */
    if(strstr(filespec, "..") != NULL)
    {
        FatalError("ERROR: file definition contains \"..\".  Do not do that!\n");
    }

    if(filespec[0] == '/')
    {
        /* absolute filespecs are saved as is */
        filename = strdup(filespec);
    }
    else
    {
        /* relative filespec is considered relative to the log directory */
        /* or /var/log if the log directory has not been set */
        if(pv.log_flag)
        {
            strlcpy(buffer, pv.log_dir, STD_BUF);
        }
        else
        {
            strlcpy(buffer, "/var/log/snort", STD_BUF);
        }

        strlcat(buffer, "/", STD_BUF - strlen(buffer));
        strlcat(buffer, filespec, STD_BUF - strlen(buffer));
        filename = strdup(buffer);
    }

    if(!pv.quiet_flag)
        printf("ProcessFileOption: %s\n", filename);

    return filename;
}

void ParseConfig(char *rule)
{
    char **toks;
    char **config_decl;
    char *args = NULL;
    char *config;
    int num_toks;

    toks = mSplit(rule, ":", 2, &num_toks, 0);

    if(num_toks > 1)
    {
        args = toks[1];
    }

    config_decl = mSplit(toks[0], " ", 2, &num_toks, '\\');

    if(num_toks != 2)
    {
        FatalError("Error parsing config: %s\n", rule);
    }

    config = config_decl[1];

#ifdef DEBUG
    printf("Config: %s\n", config);
    printf("Args: %s\n", args);
#endif

    if(!strcasecmp(config, "order"))
    {
        if(!pv.rules_order_flag)
            OrderRuleLists(args);
        else
            printf("Commandline option overiding rule file config\n");
        return;
    }
    else if(!strcasecmp(config, "alertfile"))
    {
        toks = mSplit(args, " ", 1, &num_toks, 0);

        ProcessAlertFileOption(toks[0]);

        return;
    }
    else if(!strcasecmp(config, "classification"))
    {
        ProcessClassificationConfig(args);
        return;
    }
    else if(!strcasecmp(config, "decode_arp"))
    {
        /* show ARP packets */
        DebugMessage(DEBUG_INIT, "Show ARP is active\n");
        pv.showarp_flag = 1;
        return;
    }
    else if(!strcasecmp(config, "dump_chars_only"))
    {
        /* dump the application layer as text only */
        DebugMessage(DEBUG_INIT, "Character payload dump set\n");
        pv.char_data_flag = 1;
        return;
    }
    else if(!strcasecmp(config, "dump_payload"))
    {
        /* dump the application layer */
        DebugMessage(DEBUG_INIT, "Payload dump set\n");
        pv.data_flag = 1;
        return;
    }
    else if(!strcasecmp(config, "decode_data_link"))
    {
        /* dump the data link layer as text only */
        DebugMessage(DEBUG_INIT, "Decode DLL set\n");
        pv.show2hdr_flag = 1;
        return;
    }
    else if(!strcasecmp(config, "bpf_file"))
    {
        /* Read BPF filters from a file */
        DebugMessage(DEBUG_INIT, "BPF file set\n");
        /* suck 'em in */
        pv.pcap_cmd = read_infile(args);
        return;
    }
    else if(!strcasecmp(config, "set_gid"))
    {
#ifdef WIN32
        FatalError("[!] ERROR: Setting the group id is not supported in the WIN32 port of snort!\n");
#else
        if((groupname = calloc(strlen(args) + 1, 1)) == NULL)
            FatalPrintError("calloc");

        bcopy(args, groupname, strlen(args));

        if((groupid = atoi(groupname)) == 0)
        {
            gr = getgrnam(groupname);

            if(gr == NULL)
            {
                ErrorMessage("ERROR %s(%d) => Group \"%s\" unknown\n", 
                             groupname);
            }

            groupid = gr->gr_gid;
        }
#endif
        return;
    }
    else if(!strcasecmp(config, "daemon"))
    {
        DebugMessage(DEBUG_INIT, "Daemon mode flag set\n");
        pv.daemon_flag = 1;
        pv.quiet_flag = 1;

    }
    else if(!strcasecmp(config, "ghetto_msg"))
    {
        if(!strncasecmp(args, "basic", 5))
        {
            pv.ghetto_msg_flag = GHETTO_BASIC;
        }
        else if(!strncasecmp(args, "url", 3))
        {
            pv.ghetto_msg_flag = GHETTO_URL;
        }
        else
        {
            ErrorMessage("ERROR %s(%d) => Unknown command line Ghetto "
                         "option: %s\n", file_name, file_line, args);
            return;
        }
        return;
    }
    else if(!strcasecmp(config, "reference_net"))
    {
        GenHomenet(args);
        return;
    }
    else if(!strcasecmp(config, "interface"))
    {
        if(ifr_count == MAX_INTERFACES)
        {
            ErrorMessage(
                    "\nMaximum number of interfaces (%i) exceeded."
                    "Please recompile to extend it (oops)\n",
                    MAX_INTERFACES);
            return;
        }
        pv.interfaces[ifr_count] = (char *) malloc(strlen(args) + 1);
        strlcpy(pv.interfaces[ifr_count], args, strlen(args)+1);
        ifr_count++;
        DebugMessage(DEBUG_INIT, "Interface = %s\n", PRINT_INTERFACE(pv.interfaces[ifr_count - 1]));
        if(!ifr_count)
            ifr_count++;

        if(!pv.readmode_flag)
        {
            if(pd != NULL)
            {
                pcap_close(pd);
            }

            DebugMessage(DEBUG_INIT, "Opening interface: %s\n", 
                         PRINT_INTERFACE(pv.interfaces[0]));
            /* open up our libpcap packet capture interface */
            InitializeInterfaces();
        }

        return;
    }
    else if(!strcasecmp(config, "alert_with_interface_name"))
    {
        pv.alert_interface_flag = 1;
        return;
    }
    else if(!strcasecmp(config, "logdir"))
    {
        printf("Found logdir config directive (%s)\n", args);
        strlcpy(pv.log_dir, args, STD_BUF);
        DebugMessage(DEBUG_INIT, "Log directory = %s\n", pv.log_dir);
        pv.log_flag = 1;
        return;
    }
    else if(!strcasecmp(config, "umask"))
    {
        char *p;
        long val = 0;
        int umaskchange = 1;
        int defumask = 0;

        umaskchange = 0;

        val = strtol(args, &p, 8);
        if (*p != '\0' || val < 0 || (val & ~FILEACCESSBITS))
        {
            ErrorMessage("ERROR: bad umask %s\n", args);
            return;
        }
        else
        {
            defumask = val;
        }

        /* if the umask arg happened, set umask */
        if (umaskchange)
        {
            umask(077);           /* set default to be sane */
        }
        else
        {
            umask(defumask);
        }

        return;
    }
    else if(!strcasecmp(config, "pkt_count"))
    {
        pv.pkt_cnt = atoi(args);
        DebugMessage(DEBUG_INIT, "Exiting after %d packets\n", pv.pkt_cnt);
        return;
    }
    else if(!strcasecmp(config, "nolog"))
    {
        pv.nolog_flag = 1;
        pv.log_cmd_override = 1;

        return;
    }
    else if(!strcasecmp(config, "obfuscate"))
    {
        pv.obfuscation_flag = 1;
        return;
    }
    else if(!strcasecmp(config, "no_promisc"))
    {
        pv.promisc_flag = 0;
        DebugMessage(DEBUG_INIT, "Promiscuous mode disabled!\n");
        return;
    }
    else if(!strcasecmp(config, "snaplen"))
    {
        pv.pkt_snaplen = atoi(optarg);
        DebugMessage(DEBUG_INIT, "Snaplength of Packets set to: %d\n", 
                     pv.pkt_snaplen);
        return;
    }
    else if(!strcasecmp(config, "quiet"))
    {
        pv.quiet_flag = 1;
        return;
    }
    else if(!strcasecmp(config, "read_bin_file"))
    {
        strlcpy(pv.readfile, optarg, STD_BUF);
        pv.readmode_flag = 1;
        DebugMessage(DEBUG_INIT, "Opening file: %s\n", pv.readfile);

        /* open the packet file for readback */
        OpenPcap(pv.readfile, 0);

        return;
    }
    else if(!strcasecmp(config, "chroot"))
    {
#ifdef WIN32
		FatalError("[!] ERROR: Setting the chroot directory is not supported in the WIN32 port of snort!\n");
#else
        if((chrootdir = calloc(strlen(args) + 2, 1)) == NULL)
            FatalPrintError("calloc");

        /* make sure '/' is appended */
        sprintf(chrootdir, "%s/", args);
        if(chrootdir != NULL)
        {
            if(chdir(chrootdir) < 0)
                FatalError("Can not chdir to \"%s\"\n", chrootdir);
            if(chroot(chrootdir) < 0)
                FatalError("Can not chroot to %s\n", chrootdir);
            if(chdir("/") < 0)
                FatalError("Can not chdir to \"/\"\n");

            free(chrootdir);
            chrootdir = NULL;        /* we don't need chrootdir anymore so all
                                      * other routines should use fullpath. */
        }
#endif
        return;
    }
    else if(!strcasecmp(config, "checksum_mode"))
    {
	if(args == NULL || !strcasecmp(args, "all"))
	{
	    pv.checksums_mode = DO_IP_CHECKSUMS | DO_TCP_CHECKSUMS |
		DO_UDP_CHECKSUMS | DO_ICMP_CHECKSUMS;
	}
	else if(!strcasecmp(args, "noip")) 
	{
	    pv.checksums_mode ^= DO_IP_CHECKSUMS;
	}
	else if(!strcasecmp(args, "notcp"))
	{
	    pv.checksums_mode ^= DO_TCP_CHECKSUMS;
	}
	else if(!strcasecmp(args, "noudp"))
	{
	    pv.checksums_mode ^= DO_UDP_CHECKSUMS;
	}
	else if(!strcasecmp(args, "noicmp"))
	{
	    pv.checksums_mode ^= DO_ICMP_CHECKSUMS;
	}
	else if(!strcasecmp(args, "none"))
	{
	    pv.checksums_mode = 0;
	}

        return;
    }
    else if(!strcasecmp(config, "set_uid"))
    {
#ifdef WIN32
        FatalError("[!] ERROR: Setting the user id is not supported in the WIN32 port of snort!\n");
#else
        if((username = calloc(strlen(args) + 1, 1)) == NULL)
            FatalPrintError("malloc");

        bcopy(args, username, strlen(args));

        if((userid = atoi(username)) == 0)
        {
            pw = getpwnam(username);
            if(pw == NULL)
                FatalError("User \"%s\" unknown\n", username);

            userid = pw->pw_uid;
        }
        else
        {
            pw = getpwuid(userid);
            if(pw == NULL)
                FatalError(
                        "Can not obtain username for uid: %lu\n",
                        (u_long) userid);
        }

        if(groupname == NULL)
        {
            char name[256];

            snprintf(name, 255, "%lu", (u_long) pw->pw_gid);

            if((groupname = calloc(strlen(name) + 1, 1)) == NULL)
            {
                FatalPrintError("malloc");
            }
            groupid = pw->pw_gid;
        }

        DebugMessage(DEBUG_INIT, "UserID: %lu GroupID: %lu\n",
                (unsigned long) userid, (unsigned long) groupid);
#endif
        return;
    }
    else if(!strcasecmp(config, "utc"))
    {
        pv.use_utc = 1;
        return;
    }
    else if(!strcasecmp(config, "verbose"))
    {
        pv.verbose_flag = 1;
        DebugMessage(DEBUG_INIT, "Verbose Flag active\n");
        return;
    }
    else if(!strcasecmp(config, "dump_payload_verbose"))
    {
        DebugMessage(DEBUG_INIT, "Verbose packet bytecode dumps enabled\n");
        pv.verbose_bytedump_flag = 1;
        return;
    }
    else if(!strcasecmp(config, "show_year"))
    {
        pv.include_year = 1;
        DebugMessage(DEBUG_INIT, "Enabled year in timestamp\n");
        return;
    }
    else if(!strcasecmp(config, "stateful")) /* this one's for Johnny! */
    {
        pv.assurance_mode = ASSURE_EST;
        return;
    }


    FatalError("Error: Unknown config: %s\n", config);

    return;
}

/* verify that we are not reusing some other keyword */
int checkKeyword(char *keyword)
{
    RuleListNode *node = RuleLists;

    if(RuleType(keyword) != RULE_UNKNOWN)
    {
        return 1;
    }

    /* check the declared ruletypes now */
    while(node != NULL)
    {
        if(!strcasecmp(node->name, keyword))
        {
            return 1;
        }

        node = node->next;
    }

    return 0;
}

void ParseRuleTypeDeclaration(FILE* rule_file, char *rule)
{
    char *input;
    char *keyword;
    char **toks;
    int num_toks;
    int type;
    int rval = 1;
    int i;
    ListHead *listhead = NULL;

    toks = mSplit(rule, " ", 10, &num_toks, 0);
    keyword = strdup(toks[1]);

    /* Verify keyword is unique */
    if(checkKeyword(keyword))
    {
        FatalError("ERROR line %s (%d): Duplicate keyword: %s\n",
                   file_name, file_line, keyword);
    }

#ifdef DEBUG
    printf("Declaring new rule type: %s\n", keyword);
#endif

    if(num_toks > 2)
    {
        if(strcasecmp("{", toks[2]) != 0)
        {
            FatalError("ERROR line %s (%d): Syntax error: %s\n",
                       file_name, file_line, rule);
        }
    }
    else
    {
        input = ReadLine(rule_file);
        for(i=0;i<num_toks;i++)
        {
            free(toks[i]);
        }
        toks = mSplit(input, " ", 2, &num_toks, 0);
        free(input);
    }

    input = ReadLine(rule_file);
    for(i=0;i<num_toks;i++)
    {
        free(toks[i]);
    }
    toks = mSplit(input, " ", 10, &num_toks, 0);

    /* read the type field */
    if(!strcasecmp("type", toks[0]))
    {
        type = RuleType(toks[1]);
        /* verify it is a valid ruletype */
        if((type != RULE_LOG) && (type != RULE_PASS) && (type != RULE_ALERT) &&
           (type != RULE_ACTIVATE) && (type != RULE_DYNAMIC))
        {
            FatalError("ERROR line %s (%d): Invalid type for rule type declaration: %s\n", file_name, file_line, toks[1]);
        }

#ifdef DEBUG
        printf("\ttype(%i): %s\n", type, toks[1]);
#endif

        if(type == RULE_PASS)
        {
            rval = 0;
        }

        listhead = CreateRuleType(keyword, type, rval, NULL);
    }
    else
    {
        FatalError("ERROR line %s (%d): Type not defined for rule file declaration: %s\n", file_name, file_line, keyword);
    }

    free(input);
    input = ReadLine(rule_file);
    for(i=0;i<num_toks;i++)
    {
        free(toks[i]);
    }

    toks = mSplit(input, " ", 2, &num_toks, 0);

    while(strcasecmp("}", toks[0]) != 0)
    {
        if(RuleType(toks[0]) != RULE_OUTPUT)
        {
            FatalError("ERROR line %s (%d): Not an output plugin declaration: %s\n", file_name, file_line, keyword);
        }

        head_tmp = listhead;
        ParseOutputPlugin(input);
        head_tmp = NULL;
        free(input);
        input = ReadLine(rule_file);

        for(i=0;i<num_toks;i++)
        {
            free(toks[i]);
        }
        toks = mSplit(input, " ", 2, &num_toks, 0);
    }

    return;
}


/* Adapted from ParseRule in rules.c */
void ParseDeclaredRuleType(char *rule)
{
    char **toks;
    int num_toks;
    RuleListNode *node;
    int protocol;
    int i;
    RuleTreeNode proto_node;

    toks = mSplit(rule, " ", 10, &num_toks, 0);
    node = RuleLists;

    while(node != NULL)
    {
        if(!strcasecmp(node->name, toks[0]))
            break;
        node = node->next;
    }

    /* if we did not find a match, then there is no such ruletype */
    if(node == NULL)
    {
        FatalError("ERROR line %s (%d) => Unknown rule type: %s\n",
                   file_name, file_line, toks[0]);
    }

    printf("[**] Rule start\n");
    printf("Rule id: %s\n", toks[0]);
    printf("Rule type: ");

    switch(node->mode)
    {
        case RULE_PASS:
            printf("Pass\n");
            break;
        case RULE_LOG:
            printf("Log\n");
            break;
        case RULE_ALERT:
            printf("Alert\n");
            break;
        default:
            printf("Unknown\n");
    }

    /* the rest of this function is almost identical to code in ParseRule */
    bzero((char *) &proto_node, sizeof(RuleTreeNode));

    proto_node.type = node->mode;

    /* set the rule protocol */
    protocol = WhichProto(toks[1]);

    /* Process the IP address and CIDR netmask */
    /* changed version 1.2.1 */
    /*
     * "any" IP's are now set to addr 0, netmask 0, and the normal rules are
     * applied instead of checking the flag
     */
    /*
     * if we see a "!<ip number>" we need to set a flag so that we can
     * properly deal with it when we are processing packets
     */
    /*
   if( *toks[2] == '!' )    
   {
       proto_node.flags |= EXCEPT_SRC_IP;
       ParseIP(&toks[2][1], (u_long *) & proto_node.sip,
               (u_long *) & proto_node.smask);
   }
   else
   {
       ParseIP(toks[2], (u_long *) & proto_node.sip,
               (u_long *) & proto_node.smask);
   }*/

    ProcessIP(toks[2], &proto_node, SRC);

    /* do the same for the port */
    if(ParsePort(toks[3], (u_short *) & proto_node.hsp,
                 (u_short *) & proto_node.lsp, toks[1],
                 (int *) &proto_node.not_sp_flag))
    {
        proto_node.flags |= ANY_SRC_PORT;
    }

    if(proto_node.not_sp_flag)
        proto_node.flags |= EXCEPT_SRC_PORT;

    /* New in version 1.3: support for bidirectional rules */
    /*
     * this checks the rule "direction" token and sets the bidirectional flag
     * if the token = '<>'
     */
    if(!strncmp("<>", toks[4], 2))
    {
#ifdef DEBUG
        printf("Bidirectional rule!\n");
#endif
        proto_node.flags |= BIDIRECTIONAL;
    }

    /* changed version 1.2.1 */
    /*
     * "any" IP's are now set to addr 0, netmask 0, and the normal rules are
     * applied instead of checking the flag
     */
    /*
     * if we see a "!<ip number>" we need to set a flag so that we can
     * properly deal with it when we are processing packets
     */
    /*
   if( *toks[5] == '!' )    
   {
#ifdef DEBUG
       printf("setting exception flag for dest IP\n");
#endif
       proto_node.flags |= EXCEPT_DST_IP;
       ParseIP(&toks[5][1], (u_long *) & proto_node.dip,
               (u_long *) & proto_node.dmask);
   }
   else
       ParseIP(toks[5], (u_long *) & proto_node.dip,
               (u_long *) & proto_node.dmask);
*/

    ProcessIP(toks[5], &proto_node, DST);

    if(ParsePort(toks[6], (u_short *) & proto_node.hdp,
                 (u_short *) & proto_node.ldp, toks[1],
                 (int *) &proto_node.not_dp_flag))
    {
        proto_node.flags |= ANY_DST_PORT;
    }
    if(proto_node.not_dp_flag)
        proto_node.flags |= EXCEPT_DST_PORT;

#ifdef DEBUG
    printf("proto_node.flags = 0x%X\n", proto_node.flags);
#endif

    if(&proto_node == NULL)
        printf("NULL proto_node\n");

    ProcessHeadNode(&proto_node, node->RuleList, protocol);
    rule_count++;
    ParseRuleOptions(rule, node->mode, protocol);
    for(i=0;i<num_toks;i++)
    {
        free(toks[i]);
    }

    return;
}


/* adapted from ParseRuleFule in rules.c */
char *ReadLine(FILE * file)
{
    char buf[STD_BUF];
    char *index;

    bzero((char *) buf, STD_BUF);

    /*
     * Read a line from file and return it. Skip over lines beginning with #,
     * ;, or a newline
     */
    while((fgets(buf, STD_BUF, file)) != NULL)
    {
        file_line++;
        index = buf;

#ifdef DEBUG2
        printf("Got line %s (%d): %s", file_name, file_line, buf);
#endif
        /* if it's not a comment or a <CR>, we return it */
        if((*index != '#') && (*index != 0x0a) && (*index != ';')
           && (index != NULL))
        {
            /* advance through any whitespace at the beginning of ther line */
            while(isspace((int) *index))
                ++index;

            /* return a copy of the line */
            return strdup(index);
        }
    }

    return NULL;
}
