#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <ctype.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>

#include "licq_filetransfer.h"
#include "licq_log.h"
#include "licq_constants.h"
#include "licq_icqd.h"
#include "licq_translate.h"
#include "support.h"

#define DEBUG_THREADS(x)

const unsigned short FT_STATE_DISCONNECTED = 0;
const unsigned short FT_STATE_HANDSHAKE = 1;
const unsigned short FT_STATE_WAITxFORxCLIENTxINIT = 2;
const unsigned short FT_STATE_WAITxFORxSERVERxINIT = 3;
const unsigned short FT_STATE_WAITxFORxSTART = 4;
const unsigned short FT_STATE_WAITxFORxFILExINFO = 5;
const unsigned short FT_STATE_RECEIVINGxFILE = 6;
const unsigned short FT_STATE_SENDINGxFILE = 7;
const unsigned short FT_STATE_CONFIRMINGxFILE = 8;



//=====FILE==================================================================


//-----FileInitClient-----------------------------------------------------------
CPFile_InitClient::CPFile_InitClient(char *_szLocalName,
                                    unsigned long _nNumFiles,
                                    unsigned long _nTotalSize)
{
  m_nSize = 20 + strlen(_szLocalName);
  InitBuffer();

  buffer->PackChar(0x00);
  buffer->PackUnsignedLong(0);
  buffer->PackUnsignedLong(_nNumFiles);
  buffer->PackUnsignedLong(_nTotalSize);
  buffer->PackUnsignedLong(0x64);
  buffer->PackString(_szLocalName);
}


//-----FileInitServer-----------------------------------------------------------
CPFile_InitServer::CPFile_InitServer(char *_szLocalName)
{
  m_nSize = 8 + strlen(_szLocalName);
  InitBuffer();

  buffer->PackChar(0x01);
  buffer->PackUnsignedLong(0x64);
  buffer->PackString(_szLocalName);
}


//-----FileBatch----------------------------------------------------------------
CPFile_Info::CPFile_Info(const char *_szFileName)
{
  m_bValid = true;
  m_nError = 0;

  char *pcNoPath = NULL;
  struct stat buf;

  // Remove any path from the filename
  if ( (pcNoPath = strrchr(_szFileName, '/')) != NULL)
    m_szFileName = strdup(pcNoPath + 1);
  else
    m_szFileName = strdup(_szFileName);

  if (stat(_szFileName, &buf) < 0)
  {
    m_bValid = false;
    m_nError = errno;
    return;
  }
  m_nFileSize = buf.st_size;

  m_nSize = strlen(m_szFileName) + 21;
  InitBuffer();

  buffer->PackUnsignedShort(0x02);
  // Add all the file names
  buffer->PackString(m_szFileName);
  // Add the empty file name
  buffer->PackString("");
  //Add the file length
  buffer->PackUnsignedLong(m_nFileSize);
  buffer->PackUnsignedLong(0x00);
  buffer->PackUnsignedLong(0x64);
}


CPFile_Info::~CPFile_Info()
{
  free (m_szFileName);
}


//-----FileStart----------------------------------------------------------------
CPFile_Start::CPFile_Start(unsigned long nFilePos, unsigned long nFile)
{
  m_nSize = 17;
  InitBuffer();

  buffer->PackChar(0x03);
  buffer->PackUnsignedLong(nFilePos);
  buffer->PackUnsignedLong(0x00);
  buffer->PackUnsignedLong(0x64);
  buffer->PackUnsignedLong(nFile);
}


//-----FileSpeed----------------------------------------------------------------
CPFile_SetSpeed::CPFile_SetSpeed(unsigned long nSpeed)
{
  m_nSize = 5;
  InitBuffer();

  buffer->PackChar(0x05);
  buffer->PackUnsignedLong(nSpeed);
}


//=====FileTransferManager===========================================================
CFileTransferEvent::CFileTransferEvent(unsigned char t, char *d)
{
  m_nCommand = t;
  if (d != NULL)
    m_szData = strdup(d);
  else
    m_szData = NULL;
}




FileTransferManagerList CFileTransferManager::ftmList;


CFileTransferManager::CFileTransferManager(CICQDaemon *d, unsigned long nUin)
{
  // Create the plugin notification pipe
  pipe(pipe_thread);
  pipe(pipe_events);

  m_nUin = nUin;
  m_nSession = rand();
  licqDaemon = d;

  ICQOwner *o = gUserManager.FetchOwner(LOCK_R);
  strncpy(m_szLocalName, o->GetAlias(), sizeof(m_szLocalName) - 1);
  gUserManager.DropOwner();

  m_nCurrentFile = m_nBatchFiles = 0;
  m_nFileSize = m_nBatchSize = m_nFilePos = m_nBatchPos = 0;
  m_nBytesTransfered = m_nBatchBytesTransfered = 0;
  m_nStartTime = m_nBatchStartTime = 0;
  m_nFileDesc = -1;

  m_szFileName[0] = m_szPathName[0] = '\0';
  sprintf(m_szRemoteName, "%ld", m_nUin);

  ftmList.push_back(this);
}


//-----CFileTransferManager::StartFileTransferServer-----------------------------------------
bool CFileTransferManager::StartFileTransferServer()
{
  if (licqDaemon->StartTCPServer(&ftServer) == -1)
  {
    gLog.Warn("%sNo more ports available, add more or close open chat/file sessions.\n", L_WARNxSTR);
    return false;
  }

  // Add the server to the sock manager
  sockman.AddSocket(&ftServer);
  sockman.DropSocket(&ftServer);

  return true;
}



bool CFileTransferManager::ReceiveFiles(const char *szDirectory)
{
  m_nDirection = D_RECEIVER;

  if (szDirectory == NULL)
  {
    snprintf(m_szDirectory, MAX_FILENAME_LEN, "%s/%ld", BASE_DIR, m_nUin);
    if (access(BASE_DIR, F_OK) < 0 && mkdir(m_szDirectory, 0700) == -1 &&
        errno != EEXIST)
    {
      gLog.Warn("%sUnable to create directory %s for file transfer.\n",
         L_WARNxSTR, m_szDirectory);
      strncpy(m_szDirectory, BASE_DIR, MAX_FILENAME_LEN - 1);
    }
  }
  else
    strncpy(m_szDirectory, szDirectory, MAX_FILENAME_LEN - 1);

  struct stat buf;
  stat(m_szDirectory, &buf);
  if (!S_ISDIR(buf.st_mode))
  {
    gLog.Warn("%sPath %s is not a directory.\n", L_WARNxSTR, m_szDirectory);
    return false;
  }

  if (!StartFileTransferServer()) return false;

  // Create the socket manager thread
  if (pthread_create(&thread_ft, NULL, &FileTransferManager_tep, this) == -1)
    return false;

  return true;
}


//-----CFileTransferManager::StartAsClient-------------------------------------------
bool CFileTransferManager::SendFiles(ConstFileList lPathNames, unsigned short nPort)
{
  m_nDirection = D_SENDER;

  // Validate the pathnames
  if (lPathNames.size() == 0) return false;

  struct stat buf;
  ConstFileList::iterator iter;
  for (iter = lPathNames.begin(); iter != lPathNames.end(); iter++)
  {
    if (stat(*iter, &buf) == -1)
    {
      gLog.Warn("%sFile Transfer: File access error %s:\n%s%s.\n", L_WARNxSTR,
         *iter, L_BLANKxSTR, strerror(errno));
      return false;
    }
    m_lPathNames.push_back(strdup(*iter));
    m_nBatchFiles++;
    m_nBatchSize += buf.st_size;
  }
  m_iPathName = m_lPathNames.begin();
  strcpy(m_szPathName, *m_iPathName);

  if (!ConnectToFileServer(nPort)) return false;

  // Create the socket manager thread
  if (pthread_create(&thread_ft, NULL, &FileTransferManager_tep, this) == -1)
    return false;

  return true;
}


//-----CFileTransferManager::ConnectToFileServer-----------------------------
bool CFileTransferManager::ConnectToFileServer(unsigned short nPort)
{
  gLog.Info("%sFile Transfer: Connecting to server.\n", L_TCPxSTR);
  if (!licqDaemon->OpenConnectionToUser(m_nUin, &ftSock, nPort))
  {
    return false;
  }

  gLog.Info("%sFile Transfer: Shaking hands.\n", L_TCPxSTR);

  // Send handshake packet:
  ICQUser *u = gUserManager.FetchUser(m_nUin, LOCK_R);
  unsigned short nVersion = u->ConnectionVersion();
  gUserManager.DropUser(u);
  if (!CICQDaemon::Handshake_Send(&ftSock, m_nUin, LocalPort(), nVersion, false))
    return false;

  // Send init packet:
  CPFile_InitClient p(m_szLocalName, m_nBatchFiles, m_nBatchSize);
  if (!SendPacket(&p)) return false;

  gLog.Info("%sFile Transfer: Waiting for server to respond.\n", L_TCPxSTR);

  m_nState = FT_STATE_WAITxFORxSERVERxINIT;

  sockman.AddSocket(&ftSock);
  sockman.DropSocket(&ftSock);

  return true;
}


//-----CFileTransferManager::AcceptReverseConnection-------------------------
void CFileTransferManager::AcceptReverseConnection(TCPSocket *s)
{
  if (ftSock.Descriptor() != -1)
  {
    gLog.Warn("%sFile Transfer: Attempted reverse connection when already connected.\n",
       L_WARNxSTR);
    return;
  }

  ftSock.TransferConnectionFrom(*s);
  sockman.AddSocket(&ftSock);
  sockman.DropSocket(&ftSock);

  m_nState = FT_STATE_WAITxFORxCLIENTxINIT;

  // Reload socket info
  write(pipe_thread[PIPE_WRITE], "R", 1);

  gLog.Info("%sFile Transfer: Received reverse connection.\n", L_TCPxSTR);
}


//-----CFileTransferManager::ProcessPacket-------------------------------------------
bool CFileTransferManager::ProcessPacket()
{
  if (!ftSock.RecvPacket())
  {
    char buf[128];
    if (ftSock.Error() == 0)
      gLog.Info("%sFile Transfer: Remote end disconnected.\n", L_TCPxSTR);
    else
      gLog.Warn("%sFile Transfer: Lost remote end:\n%s%s\n", L_WARNxSTR,
                L_BLANKxSTR, ftSock.ErrorStr(buf, 128));
    if (m_nState == FT_STATE_WAITxFORxFILExINFO)
      m_nResult = FT_DONExBATCH;
    else
      m_nResult = FT_ERRORxCLOSED;
    return false;
  }

  if (!ftSock.RecvBufferFull()) return true;
  CBuffer &b = ftSock.RecvBuffer();

  switch(m_nState)
  {
    // Server States

    case FT_STATE_HANDSHAKE:
    {
      if (!CICQDaemon::Handshake_Recv(&ftSock, LocalPort(), false)) break;
      gLog.Info("%sFile Transfer: Received handshake.\n", L_TCPxSTR);
      m_nState = FT_STATE_WAITxFORxCLIENTxINIT;
      break;
    }

    case FT_STATE_WAITxFORxCLIENTxINIT:
    {
      unsigned char nCmd = b.UnpackChar();
      if (nCmd == 0x05)
      {
        unsigned long nSpeed = b.UnpackUnsignedLong();
        gLog.Info("%sFile Transfer: Speed set to %ld%%.\n", L_TCPxSTR, nSpeed);
        break;
      }
      if (nCmd != 0x00)
      {
        char *pbuf;
        gLog.Error("%sFile Transfer: Invalid client init packet:\n%s%s\n",
                   L_ERRORxSTR, L_BLANKxSTR, b.print(pbuf));
        delete [] pbuf;
        m_nResult = FT_ERRORxHANDSHAKE;
        return false;
      }
      b.UnpackUnsignedLong();
      m_nBatchFiles = b.UnpackUnsignedLong();
      m_nBatchSize = b.UnpackUnsignedLong();
      m_nSpeed = b.UnpackUnsignedLong();
      b.UnpackString(m_szRemoteName, sizeof(m_szRemoteName));

      m_nBatchStartTime = time(TIME_NOW);
      m_nBatchBytesTransfered = m_nBatchPos = 0;

      PushFileTransferEvent(FT_STARTxBATCH);

      // Send speed response
      CPFile_SetSpeed p1(100);
      if (!SendPacket(&p1))
      {
        m_nResult = FT_ERRORxCLOSED;
        return false;
      }

      // Send response
      CPFile_InitServer p(m_szLocalName);
      if (!SendPacket(&p))
      {
        m_nResult = FT_ERRORxCLOSED;
        return false;
      }

      gLog.Info("%sFile Transfer: Waiting for file info.\n", L_TCPxSTR);
      m_nState = FT_STATE_WAITxFORxFILExINFO;
      break;
    }

    case FT_STATE_WAITxFORxFILExINFO:
    {
      unsigned char nCmd = b.UnpackChar();
      if (nCmd == 0x05)
      {
        unsigned long nSpeed = b.UnpackUnsignedLong();
        gLog.Info("%sFile Transfer: Speed set to %ld%%.\n", L_TCPxSTR, nSpeed);
        break;
      }
      if (nCmd != 0x02)
      {
        char *pbuf;
        gLog.Error("%sFile Transfer: Invalid file info packet:\n%s%s\n",
                   L_ERRORxSTR, L_BLANKxSTR, b.print(pbuf));
        delete [] pbuf;
        m_nResult = FT_ERRORxHANDSHAKE;
        return false;
      }
      b.UnpackChar();
      b.UnpackString(m_szFileName, sizeof(m_szFileName));

      // Remove any preceeding path info from the filename for security
      // reasons
      char *pTmp, *pNoPath;
      for (pTmp = m_szFileName + strlen(m_szFileName);
           *pTmp != '/' && pTmp >= m_szFileName;
           pTmp--);
      if (pTmp >= m_szFileName && *pTmp == '/')
      {
        pNoPath = strdup(pTmp + 1);
        strcpy(m_szFileName, pNoPath);
        free(pNoPath);
      }
      
      b.UnpackUnsignedShort(); // 0 length string...?
      b.UnpackChar();
      m_nFileSize = b.UnpackUnsignedLong();
      b.UnpackUnsignedLong();
      m_nSpeed = b.UnpackUnsignedLong();

      m_nBytesTransfered = 0;
      m_nCurrentFile++;
      
      gLog.Info("File Transfer: Waiting for plugin to confirm file receive.\n");
      
      m_nState = FT_STATE_CONFIRMINGxFILE;
      PushFileTransferEvent(new CFileTransferEvent(FT_CONFIRMxFILE, m_szPathName));
      break;
    }
    
    case FT_STATE_CONFIRMINGxFILE:
    {
      // Still waiting for the plugin to confirm
      gLog.Warn("File Transfer: Still waiting for the plugin to confirm file receive...");
      break;
    }

    case FT_STATE_RECEIVINGxFILE:
    {
      // if this is the first call to this function...
      if (m_nBytesTransfered == 0)
      {
        m_nStartTime = time(TIME_NOW);
        m_nBatchPos += m_nFilePos;
        gLog.Info("%sFile Transfer: Receiving %s (%ld bytes).\n", L_TCPxSTR,
           m_szFileName, m_nFileSize);
        PushFileTransferEvent(new CFileTransferEvent(FT_STARTxFILE, m_szPathName));
        gettimeofday(&tv_lastupdate, NULL);
      }

      // Write the new data to the file and empty the buffer
      CBuffer &b = ftSock.RecvBuffer();
      char nCmd = b.UnpackChar();
      if (nCmd == 0x05)
      {
        unsigned long nSpeed = b.UnpackUnsignedLong();
        gLog.Info("%sFile Transfer: Speed set to %ld%%.\n", L_TCPxSTR, nSpeed);
        break;
      }

      if (nCmd != 0x06)
      {
        gLog.Unknown("%sFile Transfer: Invalid data (%c) ignoring packet.\n",
           L_UNKNOWNxSTR, nCmd);
        break;
      }

      errno = 0;
      size_t nBytesWritten = write(m_nFileDesc, b.getDataPosRead(), b.getDataSize() - 1);
      if (nBytesWritten != b.getDataSize() - 1)
      {
        gLog.Error("%sFile Transfer: Write error:\n%s%s.\n", L_ERRORxSTR, L_BLANKxSTR,
           errno == 0 ? "Disk full (?)" : strerror(errno));
        m_nResult = FT_ERRORxFILE;
        return false;
      }

      m_nFilePos += nBytesWritten;
      m_nBytesTransfered += nBytesWritten;
      m_nBatchPos += nBytesWritten;
      m_nBatchBytesTransfered += nBytesWritten;

      // Check if we need to send an update notification
      if (m_nUpdatesEnabled)
      {
        struct timeval tv_now;
        gettimeofday(&tv_now, NULL);
        if (tv_now.tv_sec >= tv_lastupdate.tv_sec + m_nUpdatesEnabled)
        {
          PushFileTransferEvent(FT_UPDATE);
          tv_lastupdate = tv_now;
        }
      }

      int nBytesLeft = m_nFileSize - m_nFilePos;
      if (nBytesLeft > 0)
        break;

      close(m_nFileDesc);
      m_nFileDesc = -1;
      if (nBytesLeft == 0) // File transfer done perfectly
      {
        gLog.Info("%sFile Transfer: %s received.\n", L_TCPxSTR, m_szFileName);
      }
      else // nBytesLeft < 0
      {
        // Received too many bytes for the given size of the current file
        gLog.Warn("%sFile Transfer: %s received %d too many bytes.\n", L_WARNxSTR,
           m_szFileName, -nBytesLeft);
      }
      // Notify Plugin
      PushFileTransferEvent(new CFileTransferEvent(FT_DONExFILE, m_szPathName));

      // Now wait for a disconnect or another file
      m_nState = FT_STATE_WAITxFORxFILExINFO;
      break;
    }


    // Client States

    case FT_STATE_WAITxFORxSERVERxINIT:
    {
      char nCmd = b.UnpackChar();
      if (nCmd == 0x05)
      {
        unsigned long nSpeed = b.UnpackUnsignedLong();
        gLog.Info("%sFile Transfer: Speed set to %ld%%.\n", L_TCPxSTR, nSpeed);
        break;
      }
      if (nCmd != 0x01)
      {
        char *pbuf;
        gLog.Error("%sFile Transfer: Invalid server init packet:\n%s%s\n",
                   L_ERRORxSTR, L_BLANKxSTR, b.print(pbuf));
        delete [] pbuf;
        m_nResult = FT_ERRORxHANDSHAKE;
        return false;
      }
      m_nSpeed = b.UnpackUnsignedLong();
      b.UnpackString(m_szRemoteName, sizeof(m_szRemoteName));

      // Send file info packet
      CPFile_Info p(*m_iPathName);
      if (!p.IsValid())
      {
        gLog.Warn("%sFile Transfer: Read error for %s:\n%s\n", L_WARNxSTR,
           *m_iPathName, p.ErrorStr());
        m_nResult = FT_ERRORxFILE;
        return false;
      }
      if (!SendPacket(&p))
      {
        m_nResult = FT_ERRORxCLOSED;
        return false;
      }

      m_nFileSize = p.GetFileSize();
      strcpy(m_szFileName, p.GetFileName());

      m_nBatchStartTime = time(TIME_NOW);
      m_nBatchBytesTransfered = m_nBatchPos = 0;

      PushFileTransferEvent(FT_STARTxBATCH);

      m_nState = FT_STATE_WAITxFORxSTART;
      break;
    }

    case FT_STATE_WAITxFORxSTART:
    {
      // contains the seek value
      char nCmd = b.UnpackChar();
      if (nCmd == 0x05)
      {
        unsigned long nSpeed = b.UnpackUnsignedLong();
        gLog.Info("%sFile Transfer: Speed set to %ld%%.\n", L_TCPxSTR, nSpeed);
        break;
      }
      if (nCmd != 0x03)
      {
        char *pbuf;
        gLog.Error("%sFile Transfer: Invalid start packet:\n%s%s\n",
                   L_ERRORxSTR, L_BLANKxSTR, b.print(pbuf));
        delete [] pbuf;
        m_nResult = FT_ERRORxCLOSED;
        return false;
      }

      m_nBytesTransfered = 0;
      m_nCurrentFile++;

      m_nFilePos = b.UnpackUnsignedLong();

      m_nFileDesc = open(*m_iPathName, O_RDONLY);
      if (m_nFileDesc == -1)
      {
        gLog.Error("%sFile Transfer: Read error '%s':\n%s%s\n.", L_ERRORxSTR,
           *m_iPathName, L_BLANKxSTR, strerror(errno));
        m_nResult = FT_ERRORxFILE;
        return false;
      }

      if (lseek(m_nFileDesc, m_nFilePos, SEEK_SET) == -1)
      {
        gLog.Error("%sFile Transfer: Seek error '%s':\n%s%s\n.", L_ERRORxSTR,
                   m_szFileName, L_BLANKxSTR, strerror(errno));
        m_nResult = FT_ERRORxFILE;
        return false;
      }

      m_nState = FT_STATE_SENDINGxFILE;
      break;
    }

    case FT_STATE_SENDINGxFILE:
    {
      char nCmd = b.UnpackChar();
      if (nCmd == 0x05)
      {
        unsigned long nSpeed = b.UnpackUnsignedLong();
        gLog.Info("%sFile Transfer: Speed set to %ld%%.\n", L_TCPxSTR, nSpeed);
        break;
      }
      char *p;
      gLog.Unknown("%sFile Transfer: Unknown packet received during file send:\n%s\n",
         L_UNKNOWNxSTR, b.print(p));
      delete [] p;
      break;
    }


    default:
    {
      gLog.Error("%sInternal error: FileTransferManager::ProcessPacket(), invalid state (%d).\n",
         L_ERRORxSTR, m_nState);
      break;
    }

  } // switch

  ftSock.ClearRecvBuffer();

  return true;
}

// This function gives a callback opportunity for the plugin, just before
// the actual transfer begins
bool CFileTransferManager::StartReceivingFile(char *szFileName)
{
  gLog.Info("%sFile Transfer: Received plugin confirmation.\n", L_ERRORxSTR);
      
  if (m_nState != FT_STATE_CONFIRMINGxFILE)
  {
     gLog.Warn("%sFile Transfer: StartReceivingFile called without a pending confirmation.\n",
        L_ERRORxSTR);
     return false;
  }

  // If a different filename was specified, use it
  if (szFileName != NULL)
  {
    strncpy(m_szFileName, szFileName, sizeof(m_szFileName)-1);
    m_szFileName[sizeof(m_szFileName)-1] = '\0';
  }

  // Get the local filename and set the file offset (for resume)
  struct stat buf;
  m_nFileDesc = -1;
  snprintf(m_szPathName, MAX_FILENAME_LEN, "%s/%s", m_szDirectory, m_szFileName);
  while (m_nFileDesc == -1)
  {
    if (stat(m_szPathName, &buf) != -1)
    {
      if ((unsigned long)buf.st_size >= m_nFileSize)
        snprintf(m_szPathName, MAX_FILENAME_LEN, "%s/%s.%ld", m_szDirectory, m_szFileName, (unsigned long)time(NULL));
      m_nFileDesc = open(m_szPathName, O_WRONLY | O_CREAT | O_APPEND, 00664);
      m_nFilePos = buf.st_size;
    }
    else
    {
      m_nFileDesc = open(m_szPathName, O_WRONLY | O_CREAT, 00664);
      m_nFilePos = 0;
    }
    if (m_nFileDesc == -1)
    {
      gLog.Error("%sFile Transfer: Unable to open %s for writing:\n%s%s.\n",
         L_ERRORxSTR, m_szPathName, L_BLANKxSTR, strerror(errno));
      m_nResult = FT_ERRORxFILE;
      return false;
    }
  }

  // Send response
  CPFile_Start p(m_nFilePos, m_nCurrentFile);
  if (!SendPacket(&p))
  {
    gLog.Error("%sFile Transfer: Unable to send file receive start packet.\n", L_ERRORxSTR);
    m_nResult = FT_ERRORxCLOSED;
    return false;
  }

  m_nState = FT_STATE_RECEIVINGxFILE;
  return true;
}

//-----CFileTransferManager::SendFilePacket----------------------------------
bool CFileTransferManager::SendFilePacket()
{
  static char pSendBuf[2048];

  if (m_nBytesTransfered == 0)
  {
    m_nStartTime = time(TIME_NOW);
    m_nBatchPos += m_nFilePos;
    gLog.Info("%sFile Transfer: Sending %s (%ld bytes).\n", L_TCPxSTR,
       m_szPathName, m_nFileSize);
    PushFileTransferEvent(new CFileTransferEvent(FT_STARTxFILE, m_szPathName));
    gettimeofday(&tv_lastupdate, NULL);
  }

  int nBytesToSend = m_nFileSize - m_nFilePos;
  if (nBytesToSend > 2048) nBytesToSend = 2048;
  if (read(m_nFileDesc, pSendBuf, nBytesToSend) != nBytesToSend)
  {
    gLog.Error("%sFile Transfer: Error reading from %s:\n%s%s.\n", L_ERRORxSTR,
       m_szPathName, L_BLANKxSTR, strerror(errno));
    m_nResult = FT_ERRORxFILE;
    return false;
  }
  CBuffer xSendBuf(nBytesToSend + 1);
  xSendBuf.PackChar(0x06);
  xSendBuf.Pack(pSendBuf, nBytesToSend);
  if (!SendBuffer(&xSendBuf))
  {
    m_nResult = FT_ERRORxCLOSED;
    return false;
  }

  m_nFilePos += nBytesToSend;
  m_nBytesTransfered += nBytesToSend;

  m_nBatchPos += nBytesToSend;
  m_nBatchBytesTransfered += nBytesToSend;

  // Check if we need to send an update notification
  if (m_nUpdatesEnabled)
  {
    struct timeval tv_now;
    gettimeofday(&tv_now, NULL);
    if (tv_now.tv_sec >= tv_lastupdate.tv_sec + m_nUpdatesEnabled)
    {
      PushFileTransferEvent(FT_UPDATE);
      tv_lastupdate = tv_now;
    }
  }

  int nBytesLeft = m_nFileSize - m_nFilePos;
  if (nBytesLeft > 0)
  {
    // More bytes to send so go away until the socket is free again
    return true;
  }

  // Only get here if we are done
  close(m_nFileDesc);
  m_nFileDesc = -1;

  if (nBytesLeft == 0)
  {
    gLog.Info("%sFile Transfer: Sent %s.\n", L_TCPxSTR, m_szFileName);
  }
  else // nBytesLeft < 0
  {
    gLog.Info("%sFile Transfer: Sent %s, %d too many bytes.\n", L_TCPxSTR,
       m_szFileName, -nBytesLeft);
  }
  PushFileTransferEvent(new CFileTransferEvent(FT_DONExFILE, m_szPathName));

  // Go to the next file, if no more then close connections
  m_iPathName++;
  if (m_iPathName == m_lPathNames.end())
  {
    m_nResult = FT_DONExBATCH;
    return false;
  }
  else
  {
    // Send file info packet
    CPFile_Info p(*m_iPathName);
    if (!p.IsValid())
    {
      gLog.Warn("%sFile Transfer: Read error for %s:\n%s\n", L_WARNxSTR,
         *m_iPathName, p.ErrorStr());
      m_nResult = FT_ERRORxFILE;
      return false;
    }
    if (!SendPacket(&p))
    {
      m_nResult = FT_ERRORxCLOSED;
      return false;
    }

    m_nFileSize = p.GetFileSize();
    strcpy(m_szFileName, p.GetFileName());
    strcpy(m_szPathName, *m_iPathName);

    m_nState = FT_STATE_WAITxFORxSTART;
  }

  return true;
}



//-----CFileTransferManager::PopChatEvent------------------------------------
CFileTransferEvent *CFileTransferManager::PopFileTransferEvent()
{
  if (ftEvents.size() == 0) return NULL;

  CFileTransferEvent *e = ftEvents.front();
  ftEvents.pop_front();

  return e;
}


//-----CFileTransferManager::PushChatEvent-------------------------------------------
void CFileTransferManager::PushFileTransferEvent(unsigned char t)
{
  PushFileTransferEvent(new CFileTransferEvent(t));
}

void CFileTransferManager::PushFileTransferEvent(CFileTransferEvent *e)
{
  ftEvents.push_back(e);
  write(pipe_events[PIPE_WRITE], "*", 1);
}


//-----CFileTransferManager::SendPacket----------------------------------------------
bool CFileTransferManager::SendPacket(CPacket *p)
{
  return SendBuffer(p->getBuffer());
}


//-----CFileTransferManager::SendBuffer----------------------------------------------
bool CFileTransferManager::SendBuffer(CBuffer *b)
{
  if (!ftSock.SendPacket(b))
  {
    char buf[128];
    gLog.Warn("%sFile Transfer: Send error:\n%s%s\n", L_WARNxSTR, L_BLANKxSTR,
       ftSock.ErrorStr(buf, 128));
    return false;
  }
  return true;
}


void CFileTransferManager::ChangeSpeed(unsigned short nSpeed)
{
  if (nSpeed > 100)
  {
    gLog.Warn("%sInvalid file transfer speed: %d%%.\n", L_WARNxSTR, nSpeed);
    return;
  }

  //CPFile_ChangeSpeed p(nSpeed);
  //SendPacket(&p);
  m_nSpeed = nSpeed;
}



//----CFileTransferManager::CloseFileTransfer--------------------------------
void CFileTransferManager::CloseFileTransfer()
{
  // Close the thread
  if (pipe_thread[PIPE_WRITE] != -1)
  {
    write(pipe_thread[PIPE_WRITE], "X", 1);
    pthread_join(thread_ft, NULL);

    close(pipe_thread[PIPE_READ]);
    close(pipe_thread[PIPE_WRITE]);

    pipe_thread[PIPE_READ] = pipe_thread[PIPE_WRITE] = -1;
  }

  CloseConnection();
}


//----CFileTransferManager::CloseConnection----------------------------------
void CFileTransferManager::CloseConnection()
{
  sockman.CloseSocket(ftServer.Descriptor(), false, false);
  sockman.CloseSocket(ftSock.Descriptor(), false, false);
  m_nState = FT_STATE_DISCONNECTED;
}



void *FileTransferManager_tep(void *arg)
{
  CFileTransferManager *ftman = (CFileTransferManager *)arg;

  fd_set f_recv, f_send;
  struct timeval *tv;
  struct timeval tv_updates = { 2, 0 };
  int l, nSocketsAvailable, nCurrentSocket;
  char buf[2];

  while (true)
  {
    f_recv = ftman->sockman.SocketSet();
    l = ftman->sockman.LargestSocket() + 1;

    // Add the new socket pipe descriptor
    FD_SET(ftman->pipe_thread[PIPE_READ], &f_recv);
    if (ftman->pipe_thread[PIPE_READ] >= l)
      l = ftman->pipe_thread[PIPE_READ] + 1;

    // Set up the send descriptor
    FD_ZERO(&f_send);
    if (ftman->m_nState == FT_STATE_SENDINGxFILE)
    {
      FD_SET(ftman->ftSock.Descriptor(), &f_send);
      // No need to check "l" as ftSock is already in the read list
    }

    // Prepare max timeout if necessary
    if (ftman->m_nUpdatesEnabled &&
        (ftman->m_nState == FT_STATE_SENDINGxFILE ||
         ftman->m_nState == FT_STATE_RECEIVINGxFILE) )
    {
      tv_updates.tv_sec = ftman->m_nUpdatesEnabled;
      tv_updates.tv_usec = 0;
      tv = &tv_updates;
    }
    else
    {
      tv = NULL;
    }

    nSocketsAvailable = select(l, &f_recv, &f_send, NULL, tv);

    // Check if we timed out
    if (tv != NULL && nSocketsAvailable == 0)
    {
      ftman->PushFileTransferEvent(FT_UPDATE);
      gettimeofday(&ftman->tv_lastupdate, NULL);
    }

    nCurrentSocket = 0;
    while (nSocketsAvailable > 0 && nCurrentSocket < l)
    {
      if (FD_ISSET(nCurrentSocket, &f_recv))
      {
        // New socket event ----------------------------------------------------
        if (nCurrentSocket == ftman->pipe_thread[PIPE_READ])
        {
          read(ftman->pipe_thread[PIPE_READ], buf, 1);
          if (buf[0] == 'S')
          {
            DEBUG_THREADS("[FileTransferManager_tep] Reloading socket info.\n");
          }
          else if (buf[0] == 'X')
          {
            DEBUG_THREADS("[FileTransferManager_tep] Exiting.\n");
            pthread_exit(NULL);
          }
        }

        // Connection on the server port ---------------------------------------
        else if (nCurrentSocket == ftman->ftServer.Descriptor())
        {
          if (ftman->ftSock.Descriptor() != -1)
          {
            gLog.Warn("%sFile Transfer: Receiving repeat incoming connection.\n", L_WARNxSTR);
          }
          else
          {
            ftman->ftServer.RecvConnection(ftman->ftSock);
            ftman->sockman.AddSocket(&ftman->ftSock);
            ftman->sockman.DropSocket(&ftman->ftSock);

            ftman->m_nState = FT_STATE_HANDSHAKE;
            gLog.Info("%sFile Transfer: Received connection.\n", L_TCPxSTR);
          }
        }

        // Message from connected socket----------------------------------------
        else if (nCurrentSocket == ftman->ftSock.Descriptor())
        {
          ftman->ftSock.Lock();
          bool ok = ftman->ProcessPacket();
          ftman->ftSock.Unlock();
          if (!ok)
          {
            ftman->CloseConnection();
            ftman->PushFileTransferEvent(ftman->m_nResult);
          }
        }

        else
        {
          gLog.Warn("%sFile Transfer: No such socket.\n", L_WARNxSTR);
        }

        nSocketsAvailable--;
      }
      else if (FD_ISSET(nCurrentSocket, &f_send))
      {
        if (nCurrentSocket == ftman->ftSock.Descriptor())
        {
          ftman->ftSock.Lock();
          bool ok = ftman->SendFilePacket();
          ftman->ftSock.Unlock();
          if (!ok)
          {
            ftman->CloseConnection();
            ftman->PushFileTransferEvent(ftman->m_nResult);
          }
        }
        nSocketsAvailable--;
      }

      nCurrentSocket++;
    }
  }
  return NULL;
}


CFileTransferManager *CFileTransferManager::FindByPort(unsigned short p)
{
  FileTransferManagerList::iterator iter;
  for (iter = ftmList.begin(); iter != ftmList.end(); iter++)
  {
    if ( (*iter)->LocalPort() == p ) return *iter;
  }
  return NULL;
}


CFileTransferManager::~CFileTransferManager()
{
  CloseFileTransfer();

  // Delete any pending events
  CFileTransferEvent *e = NULL;
  while (ftEvents.size() > 0)
  {
    e = ftEvents.front();
    delete e;
    ftEvents.pop_front();
  }

  FileList::iterator iter;
  for (iter = m_lPathNames.begin(); iter != m_lPathNames.end(); iter++)
  {
    free(*iter);
  }

  FileTransferManagerList::iterator fiter;
  for (fiter = ftmList.begin(); fiter != ftmList.end(); fiter++)
  {
    if (*fiter == this) break;
  }
  if (fiter != ftmList.end()) ftmList.erase(fiter);
}

