//
//   File : kvi_window.cpp
//   Creation date : Sat Sep 01 2001 17:13:12 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_console.h"
#include "kvi_options.h"
#include "kvi_ircsocket.h"
#include "kvi_frame.h"
#include "kvi_locale.h"
#include "kvi_app.h"
#include "kvi_error.h"

#include <qasciidict.h>

// kvi_app.cpp
extern KVIRC_API QAsciiDict<KviWindow> * g_pGlobalWindowDict;

// $window.caption $window.x $window.y $window.width $window.height $window.isActive $window.type
// $window.input.text $window.input.cursorpos $window.input.textlen

static KviWindow * get_window_by_first_param(KviCommand *c,KviParameterList *parms)
{
	KviStr * id = parms->safeFirst();
	if(id->isEmpty())return c->window();
	return g_pApp->findWindow(id->ptr());
}



/*
	@doc: window.exists
	@type:
		function
	@title:
		$window.exists
	@short:
		Checks for the existence of a window
	@syntax:
		$window.exists(<window_id>)
	@description:
	@seealso:
*/

static bool window_module_fnc_exists(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	buffer.append(g_pApp->findWindow(parms->safeFirstParam()) ? '1' : '0');
	return true;
}


/*
	@doc: window.type
	@type:
		function
	@title:
		$window.type
	@short:
		Returns the type of a window
	@syntax:
		$window.type
		$window.type(<window_id>)
	@descritpion:
		Returns the type of the window with <window_id>.[br]
		The form with no parameters returns the type of the current window.[br]
		If the window with the specified id does not exist, an empty string is returned.[br]
	@seealso:
*/


static bool window_module_fnc_type(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"window_module_fnc_type");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)buffer.append(wnd->typeString());
	return c->leaveContext();
}



/*
	@doc: window.caption
	@type:
		function
	@title:
		$window.caption
	@short:
		Returns the caption of a window
	@syntax:
		$window.caption
		$window.caption(<window_id>)
	@descritpion:
		Returns the caption of the window with <window_id>.[br]
		The form with no parameters returns the caption of the current window.[br]
		If the window with the specified id does not exist, an empty string is returned.[br]
	@seealso:
*/


static bool window_module_fnc_caption(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"window_module_fnc_caption");
	KviWindow * wnd = get_window_by_first_param(c,parms);
	if(wnd)buffer.append(wnd->plainTextCaption());
	return c->leaveContext();
}

/*
	@doc: window.listtypes
	@type:
		command
	@title:
		window.listtypes
	@short:
		Lists available types of windows
	@syntax:
		window.listtypes
	@description:
		Lists the types of windows that are built in the current release of KVIrc.[br]
		This is actually a command and not a static list just because new window
		types may be added in subsequent releases.[br]
	@seealso:
		[cmd]window.listtypes[/cmd]
*/

static bool window_module_cmd_listtypes(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"window_module_cmd_listtypes");

	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
	c->window()->listWindowTypes();
	return c->leaveContext();
}


/*
	@doc: window.list
	@type:
		function
	@title:
		$window.list
	@short:
		Generates lists of windows
	@syntax:
		$window.list(<type>[,<irc_context_id>])
	@description:
		Returns an array of window identifiers with a speficied type and eventually belonging to a specified
		irc context.[br]
		<type> is a window type such as 'query' or 'channel'.[br]
		See [cmd]window.listtypes[/cmd] for a list of available window types in this KVIrc release.[br]
		If <type> is the special word 'all', all the window types are listed.[br]
		<irc_context_id> specifies the irc context in which the windows are searched.[br]
		If no <irc_context_id> is specified , the current one is used.[br]
		If <irc_context_id> is the special word 'all' , all the irc context are searched.[br]
		The windows that do not belong to any irc context (such as DCC windows), must be searched
		by using 'all' as <irc_context_id>.
	@examples:
		[example]
			[comment]# List all the queries of the current irc context[/comment]
			[cmd]echo[/cmd] $window.list(query)
			[comment]# Equivalent to the above[/comment]
			[cmd]echo[/cmd] $window.list(query,[fnc]$ic[/fnc])
			[comment]# List all the channels in all the irc contexts[/comment]
			[cmd]echo[/cmd] $window.list(channel,all)
			[comment]# List all the windows in the current irc context[/comment]
			[cmd]echo[/cmd] $window.list(all)
			[comment]# List all the windows in all irc contexts[/comment]
			[cmd]echo[/cmd] $window.list(all,all)
			[comment]# List all the DCC Send windows[/comment]
			[cmd]echo[/cmd] $window.list(dccsend,all)
		[/example]
	@seealso:
		[cmd]window.listtypes[/cmd]
*/

static bool window_module_fnc_list(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"window_module_fnc_list");

	// all , all
	// all , context
	// type , all
	// type , context

	KviStr * type = parms->safeFirst();
	KviStr * cntx = parms->safeNext();

	if(type->isEmpty())return c->error(KviError_notEnoughParameters,__tr("Window type or 'all' expected as first parameter"));

	c->beginListArrayOrDictionaryReturnIdentifier();

	int idx = 0;

	if(kvi_strEqualCI(cntx->ptr(),"all"))
	{
		// all contexts
		bool bAllWindows = kvi_strEqualCI(type->ptr(),"all");
		QAsciiDictIterator<KviWindow> it(*g_pGlobalWindowDict);

		while(KviWindow * wnd = it.current())
		{
			if(bAllWindows)
			{
				c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
			} else {
				if(kvi_strEqualCI(wnd->typeString(),type->ptr()))
				{
					c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
				}
			}
			++it;
		}
	} else {
		// some specified context
		unsigned int uId = 0;

		if(cntx->hasData())
		{
			// specific context
			bool bOk;
			uId = cntx->toUInt(&bOk);
			if(!bOk)return c->error(KviError_invalidParameter,__tr("Invalid IRC context id '%s'"),cntx->ptr());
		} else {
			// current irc context
			if(!c->window()->console())return c->noIrcContext();
			uId = c->window()->console()->ircContextId();
		}

		bool bAllWindows = kvi_strEqualCI(type->ptr(),"all");
		QAsciiDictIterator<KviWindow> it(*g_pGlobalWindowDict);

		while(KviWindow * wnd = it.current())
		{
			if(wnd->console())
			{
				if(wnd->console()->ircContextId() == uId)
				{
					if(bAllWindows)
					{
						c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
					} else {
						if(kvi_strEqualCI(wnd->typeString(),type->ptr()))
						{
							c->addListArrayOrDictionaryReturnValue(idx++,wnd->id(),buffer);
						}
					}
				}
			}
			++it;
		}
	}

	return c->leaveContext();
}



static bool window_module_init(KviModule *m)
{
	m->registerFunction("list",window_module_fnc_list);
	m->registerCommand("listtypes",window_module_cmd_listtypes);

	m->registerFunction("exists",window_module_fnc_exists);
	m->registerFunction("type",window_module_fnc_type);
	m->registerFunction("caption",window_module_fnc_caption);

/*
	m->registerFunction("geometry",window_module_fnc_geometry);
	m->registerCommand("setGeometry",window_module_cmd_setGeometry);

	m->registerFunction("isActive",window_module_fnc_isActive);
	m->registerCommand("activate",window_module_cmd_activate);

	m->registerCommand("close",window_module_cmd_close);

	// restored , maximized , minimized
	m->registerFunction("state",window_module_fnc_state);
	m->registerCommand("setState",window_module_cmd_setState);

	m->registerFunction("caption",window_module_fnc_caption);
	m->registerCommand("setCaption",window_module_cmd_setCaption);

	m->registerFunction("dockState",window_module_fnc_dockState);
	m->registerCommand("setDockState",window_module_cmd_setDockState);

	// Input stuff
	m->registerFunction("hasInput",window_module_fnc_hasInput);

	m->registerFunction("inputText",window_module_fnc_inputText);
	m->registerCommand("setInputText",window_module_cmd_setInputText);

	// Output stuff
	m->registerFunction("hasOutput",window_module_fnc_hasOutput);

	m->registerCommand("clearOutput",window_module_cmd_clear);

	m->registerFunction("outputBufferSize",window_module_fnc_outputBufferSize);
	m->registerCommand("setOutputBufferSize",window_module_cmd_setOutputBufferSize);

	m->registerFunction("outputShowsImages",window_module_fnc_outputShowsImages);
	m->registerCommand("setOutputShowsImages",window_module_cmd_setOutputShowsImages);
	
	m->registerFunction("outputTimestamp",window_module_fnc_outputTimestamp);
	m->registerCommand("setOutputTimestamp",window_module_cmd_setOutputTimestamp);

	m->registerFunction("hasOutputBackgroundPixmap",window_module_fnc_hasOutputBackgroundPixmap);
	m->registerCommand("setOutputBackgroundPixmap",window_module_cmd_setOutputBackgroundPixmap);

*/
	return true;
}

static bool window_module_cleanup(KviModule *m)
{
	return true;
}

static bool window_module_can_unload(KviModule *m)
{
	return true;
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Window",                                               // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)", // author & (C)
	"KVIrc window management functions",
	window_module_init,
	window_module_can_unload,
	0,
	window_module_cleanup
};
