//
//   This file is part of the KVIrc irc client distribution
//   Copyright (C) 1999-2002 Andrea Parrella (yap@kvirc.net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//



#include "kvi_module.h"
#include "libkviurl.h"
//#include "kvi_process.h"
#include "kvi_app.h"
//#include "kvi_mdi.h"
#include "kvi_frame.h"
#include "kvi_event.h"
#include "kvi_menubar.h"
#include "kvi_internalcmd.h"
#include "kvi_iconmanager.h"
#include "kvi_taskbar.h"
#include "icons.h"
#include "kvi_list.h"
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qcursor.h>

typedef struct _UrlDlgList
{
	KviFrame *frm;
	UrlDialog *dlg;
	UrlToolBar *toolbar;
	int menu_id;
} UrlDlgList;

const char *g_pUrlListFilename = "/list.kviurl";
const char *g_pBanListFilename = "/list.kviban";

KviPtrList<KviUrl> *g_pList;
KviPtrList<UrlDlgList> *g_pUrlDlgList;
KviPtrList<KviStr> *g_pBanList;
ConfigDialog *g_pConfigDialog;

KviStr szConfigPath;

void saveUrlList();
void loadUrlList();
void saveBanList();
void loadBanList();
UrlDlgList *findFrame(KviFrame *frame);
bool urllist(KviFrame *frame);
void url_module_help();

/*	extern QPixmap * g_pShadedChildGlobalDesktopBackground;

#include <qpainter.h>

KviListView::KviListView(QWidget *parent,const char *name)
:QListView(parent,name)
{

}

KviListView::~KviListView()
{

}

void KviListView::drawContents(QPainter *p)
{
	if(g_pShadedChildGlobalDesktopBackground)
	{
		QPoint pnt = mapToGlobal(contentsRect().topLeft());
		p->drawTiledPixmap(contentsRect(),*g_pShadedChildGlobalDesktopBackground,pnt);
	}
}*/

// ---------------------------- CLASS URLDIALOG ------------------------begin //

UrlDialog::UrlDialog(KviPtrList<KviUrl> *g_pList, KviFrame *lpFrm)
:KviWindow(KVI_WINDOW_TYPE_ICQ,lpFrm,"URL List")
{
	m_pMenuBar = new KviTalMenuBar(this,"url menu");
	m_pUrlList = new QListView(this,"list");
	//m_pUrlList = new KviListView(this,"list");
	KviConfig cfg(szConfigPath.ptr());

	QPopupMenu *pop;

	pop = new QPopupMenu(this);
	pop->insertItem(__tr("&Configure"),this,SLOT(config()));
	pop->insertItem(__tr("&Help"),this,SLOT(help()));
	pop->insertItem(__tr("Close &Window"),this,SLOT(close_slot()));
	m_pMenuBar->insertItem(__tr("&Module"),pop);

	pop = new QPopupMenu(this);
	pop->insertItem(__tr("&Load"),this,SLOT(loadList()));
	pop->insertItem(__tr("&Save"),this,SLOT(saveList()));
	pop->insertItem(__tr("&Clear"),this,SLOT(clear()));
	m_pMenuBar->insertItem(__tr("&List"),pop);

	m_pUrlList->setShowSortIndicator(true);
	m_pUrlList->addColumn(QString(__tr("URL")));
	m_pUrlList->addColumn(QString(__tr("Window")));
	m_pUrlList->addColumn(QString(__tr("Count")));
	m_pUrlList->addColumn(QString(__tr("Timestamp")));

	cfg.setGroup("colsWidth");
	m_pUrlList->setColumnWidth(0,cfg.readIntEntry("Url",170));
	m_pUrlList->setColumnWidth(1,cfg.readIntEntry("Window",130));
	m_pUrlList->setColumnWidth(2,cfg.readIntEntry("Count",70));
	m_pUrlList->setColumnWidth(3,cfg.readIntEntry("Timestamp",70));

	connect(m_pUrlList,SIGNAL(doubleClicked(QListViewItem *)),SLOT(dblclk_url(QListViewItem *)));
	connect(m_pUrlList,SIGNAL(rightButtonPressed(QListViewItem *, const QPoint &, int)),SLOT(popup(QListViewItem *, const QPoint &, int)));

	setFocusHandlerNoChildren(m_pUrlList);
	m_pUrlList->setFocus();
}

void UrlDialog::config()
{
	if (!g_pConfigDialog) g_pConfigDialog = new ConfigDialog();
}

void UrlDialog::help()
{
//#warning "help"
//	m_pFrm->requestHelpOn("doc_plugin_url.kvihelp");
}

void UrlDialog::saveList()
{
	saveUrlList();
}

void UrlDialog::loadList()
{
	loadUrlList();
}

void UrlDialog::clear()
{
	g_pList->clear();
	for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (tmpitem->dlg) tmpitem->dlg->m_pUrlList->clear();
	}
}

/*
void UrlDialog::saveProperties()
{

	KviWindowProperty p;
	p.rect = externalGeometry();
	p.isDocked = isAttached();
	p.splitWidth1 = 0;
	p.splitWidth2 = 0;
	p.timestamp = 0;
	p.imagesVisible = 0;
	KviWindow * w = m_pFrm->activeWindow();
	p.isMaximized = isAttached() && w ? w->isMaximized() : isMaximized();
	p.topSplitWidth1 = 0;
	p.topSplitWidth2 = 0;
	p.topSplitWidth3 = 0;
	g_pOptions->m_pWinPropertiesList->setProperty(caption(),&p);
}
*/

void UrlDialog::close_slot()
{
	close();
}

void UrlDialog::remove()
{
	if (!m_pUrlList->currentItem()) {
		QMessageBox::warning(0,__tr("Warning"),__tr("Select an URL"),QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}
	for(KviUrl *tmp=g_pList->first();tmp;tmp=g_pList->next())
	{
		if (tmp->url == KviStr(m_pUrlList->currentItem()->text(0))) {
			g_pList->find(tmp);
			g_pList->remove();
			m_pUrlList->takeItem(m_pUrlList->currentItem());
			return;
		}
	}
}

void UrlDialog::findtext()
{
//#warning "find text"
/*
	if (!m_pUrlList->currentItem()) {
		kvirc_plugin_warning_box(__tr("Select an URL"));
		return;
	}
	for(KviUrl *tmp=g_pList->first();tmp;tmp=g_pList->next())
	{
		if (tmp->url == KviStr(m_pUrlList->currentItem()->text(0))) {
			g_pList->find(tmp);
			KviStr ft="findtext %";
			ft.replaceAll('%',tmp->url.ptr());
			KviWindow *wnd = m_pFrm->findWindow(tmp->window.ptr());
			if (wnd) {
				if (kvirc_plugin_execute_command(wnd,ft.ptr())) {
					if (wnd->mdiParent()) m_pFrm->m_pMdi->setTopChild(wnd->mdiParent(),true);
				}
			} else kvirc_plugin_warning_box(__tr("Window not found"));
		}

	}
*/
}

void UrlDialog::dblclk_url(QListViewItem *item)
{
	KviStr cmd(KviStr::Format,"openurl %s",item->text(0).latin1());
	g_pUserParser->parseCommandBuffer(cmd.ptr(),this);
}

void UrlDialog::popup(QListViewItem *item, const QPoint &point, int col)
{
	if (col == 0) {
		m_szUrl = item->text(0);
		QPopupMenu p(0,"menu");
		p.insertItem(__tr("&Remove"),this,SLOT(remove()));
		p.insertItem(__tr("&Find Text"),this,SLOT(findtext()));
		p.insertSeparator();
		m_pListPopup = new QPopupMenu(0,"list");
		int i=0;
		for(KviWindow *w=m_pFrm->windowList()->first();w;w=m_pFrm->windowList()->next()){
			if ((w->type() <= 2) || (w->type() == 2) || (w->type() == 6)) {	// values defined in kvi_define.h (console,channel,query,chat,uwindow)
				m_pListPopup->insertItem(QString(w->plainTextCaption()),i);
				m_pListPopup->connectItem(i,this,SLOT(sayToWin(int)));
				i++;
			}
		}
		p.insertItem(__tr("&Say to Window"),m_pListPopup);
		p.exec(QCursor::pos());
	}
}

void UrlDialog::sayToWin(int itemID)
{
	KviStr say="PRIVMSG %1 %2";
	KviWindow *wnd = g_pApp->findWindowByCaption(m_pListPopup->text(itemID).latin1());
	say.replaceAll("%1",wnd->name());
	say.replaceAll("%2",m_szUrl.ptr());
	if (wnd) {
		g_pUserParser->parseCommandBuffer(say.ptr(),wnd);
		wnd->raise();
		wnd->setActiveWindow();
		wnd->setFocus();
	} else QMessageBox::warning(0,__tr("Warning"),__tr("Window not found"),QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
}

QPixmap *UrlDialog::myIconPtr()
{
	QPixmap *icon = new QPixmap(url_icon_xpm);
	return icon;
}

void UrlDialog::addUrl(QString url, QString window, QString count, QString timestamp)
{
	QListViewItem *UrlItem = new QListViewItem(m_pUrlList);

	UrlItem->setText(0, url);
	UrlItem->setText(1, window);
	UrlItem->setText(2, count);
	UrlItem->setText(3, timestamp);
}

void UrlDialog::resizeEvent(QResizeEvent *)
{
	int hght = m_pMenuBar->heightForWidth(width());
	m_pMenuBar->setGeometry(0,0,width(),hght);
	m_pUrlList->setGeometry(0,hght,width(),height() - hght);
}

UrlDialog::~UrlDialog()
{
	KviConfig cfg(szConfigPath.ptr());
	cfg.setGroup("ConfigDialog");
	if (cfg.readBoolEntry("SaveColumnWidthOnClose",false)) {
		cfg.setGroup("ColsWidth");
		cfg.writeEntry("Url",m_pUrlList->columnWidth(0));
		cfg.writeEntry("Window",m_pUrlList->columnWidth(1));
		cfg.writeEntry("Count",m_pUrlList->columnWidth(2));
		cfg.writeEntry("Timestamp",m_pUrlList->columnWidth(3));
	}

	delete m_pUrlList;
/*	if (m_pListPopup) delete m_pListPopup;
	m_pListPopup = 0;
	if (m_pMenuBar) delete m_pMenuBar;
	m_pMenuBar = 0;*/
	UrlDlgList *tmpitem = findFrame(m_pFrm);
	tmpitem->dlg = 0;
}

// ----------------------------- CLASS URLDIALOG -------------------------end //

// --------------------------- CLASS CONFIGDIALOG ----------------------begin //

ConfigDialog::ConfigDialog()
:QDialog()
{
	setCaption(__tr("URL Module Configuration"));

	QGridLayout *g = new QGridLayout(this,4,2,10,10);

	KviConfig *cfg = new KviConfig(szConfigPath.ptr());
	cfg->setGroup("ConfigDialog");

	cb[0] = new QCheckBox(__tr("Save URL list on module unload"),this);
	cb[0]->setChecked(cfg->readBoolEntry("SaveUrlListOnUnload",false));
	g->addMultiCellWidget(cb[0],0,0,0,1);

	cb[1] = new QCheckBox(__tr("Save columns width on URL list close"),this);
	cb[1]->setChecked(cfg->readBoolEntry("SaveColumnWidthOnClose",false));
	g->addMultiCellWidget(cb[1],1,1,0,1);

	cb[2] = new QCheckBox(__tr("Show URL list button on frame window created"),this);
	cb[2]->setChecked(cfg->readBoolEntry("LoadUrlToolbarOnFrameWindowCreated",true));
	g->addMultiCellWidget(cb[2],2,2,0,1);

	bool tmp = cfg->readBoolEntry("BanEnabled",false);
	delete cfg;

	m_pBanFrame = new BanFrame(this,"banlist",tmp);
	g->addMultiCellWidget(m_pBanFrame,3,3,0,1);

	QPushButton *b;
	// configure buttons
        b = new QPushButton(__tr("Discard changes"),this,"discard");
	connect(b,SIGNAL(clicked()),this,SLOT(discardbtn()));
	g->addWidget(b,4,0);

	b = new QPushButton(__tr("Accept changes"),this,"accept");
	connect(b,SIGNAL(clicked()),this,SLOT(acceptbtn()));
	g->addWidget(b,4,1);

	show();
}

void ConfigDialog::discardbtn()
{
	delete this;
}

void ConfigDialog::acceptbtn()
{
	if (m_pBanFrame) m_pBanFrame->saveBans();

	KviConfig *cfg = new KviConfig(szConfigPath.ptr());
	cfg->setGroup("ConfigDialog");
	cfg->writeEntry("SaveUrlListOnUnload",cb[0]->isChecked());
	cfg->writeEntry("SaveColumnWidthOnClose",cb[1]->isChecked());
	cfg->writeEntry("LoadUrlToolbarOnFrameWindowCreated",cb[2]->isChecked());
	delete cfg;

	delete this;
}

void ConfigDialog::closeEvent(QCloseEvent *)
{
	delete this;
}

ConfigDialog::~ConfigDialog()
{
	for(int i=0;i<cbnum;i++) delete cb[i];
	g_pConfigDialog = 0;
}

// --------------------------- CLASS CONFIGDIALOG ------------------------end //

// ---------------------------- CLASS BANFRAME ------------------------begin //

BanFrame::BanFrame(QWidget *parent, const char *name, bool banEnabled)
:QFrame(parent,name)
{
	setFrameStyle(QFrame::Panel | QFrame::Raised);

	QGridLayout *g = new QGridLayout(this,2,2,10,10);

	m_pEnable = new QCheckBox(__tr("Enable URL ban list"),this);
	connect(m_pEnable,SIGNAL(clicked()),this,SLOT(enableClicked()));
	m_pEnable->setChecked(banEnabled);
	g->addMultiCellWidget(m_pEnable,0,0,0,1);

	m_pBanList = new QListBox(this,"listbox");
	m_pBanList->setMinimumHeight(100);
	loadBanList();
	for(KviStr *tmp=g_pBanList->first();tmp;tmp=g_pBanList->next()) m_pBanList->insertItem(tmp->ptr()); // load ban list into listbox
	m_pBanList->setEnabled(m_pEnable->isChecked());
	g->addMultiCellWidget(m_pBanList,1,1,0,1);

	m_pAddBtn = new QPushButton(__tr("Add ban"),this,"add");
	connect(m_pAddBtn,SIGNAL(clicked()),this,SLOT(addBan()));
	m_pAddBtn->setEnabled(m_pEnable->isChecked());
	g->addWidget(m_pAddBtn,2,0);

	m_pRemoveBtn = new QPushButton(__tr("Remove selected"),this,"remove");
	connect(m_pRemoveBtn,SIGNAL(clicked()),this,SLOT(removeBan()));
	m_pRemoveBtn->setEnabled(m_pEnable->isChecked());
	g->addWidget(m_pRemoveBtn,2,1);
}

void BanFrame::enableClicked()
{
	m_pBanList->setEnabled(m_pEnable->isChecked());
	m_pAddBtn->setEnabled(m_pEnable->isChecked());
	m_pRemoveBtn->setEnabled(m_pEnable->isChecked());
}

void BanFrame::addBan()
{
	bool ok = false;
	KviStr *text = new KviStr(QInputDialog::getText(__tr("URL ban list"),__tr("Add"),QLineEdit::Normal,QString::null,&ok,this));
	if (ok && !text->isEmpty()) {
		g_pBanList->append(text);
		m_pBanList->insertItem(text->ptr());
	}
}

void BanFrame::removeBan()
{
	uint i = 0;
	while ((!m_pBanList->isSelected(i)) && (i < m_pBanList->count())) i++;
	if (!m_pBanList->isSelected(i)) {
		QMessageBox::warning(0,__tr("Warning"),__tr("Select a ban"),QMessageBox::Ok,QMessageBox::NoButton,QMessageBox::NoButton);
		return;
	}
	KviStr item(m_pBanList->text(i).latin1());
	for(KviStr *tmp=g_pBanList->first();tmp;tmp=g_pBanList->next())
	{
		if (*tmp == item) g_pBanList->remove();
	}

	m_pBanList->removeItem(i);

}

void BanFrame::saveBans()
{
	if (m_pEnable->isChecked()) saveBanList();
	KviConfig *cfg = new KviConfig(szConfigPath.ptr());
	cfg->setGroup("ConfigDialog");
	cfg->writeEntry("BanEnabled",m_pEnable->isChecked());
	delete cfg;
}

BanFrame::~BanFrame()
{

}

// ---------------------------- CLASS BANFRAME ---------------------------end //

// ---------------------------- CLASS URLTOOLBAR -----------------------begin //

UrlToolBar::UrlToolBar(KviIrcToolBarAppletContainer *parent,KviFrame *pFrm)
:KviIrcToolBarGraphicalApplet(parent,"url_tool_bar")
{
	QBoxLayout *l = new QHBoxLayout(this);
	m_pParentFrame = pFrm;
	m_pParentAppCont = parent;

	m_pUrlBtn = new QToolButton(this,"url_list_btn");
	m_pUrlBtn->setPixmap(url_toolbar_xpm);
	m_pUrlBtn->setTextLabel(__tr("URL List Window"));
	m_pUrlBtn->setUsesBigPixmap(true);
	m_pUrlBtn->setAutoRaise(true);
	connect(m_pUrlBtn,SIGNAL(clicked()),this,SLOT(loadUrlListWindow()));
	l->addWidget(m_pUrlBtn);

/*	m_pConfBtn = new QToolButton(this,"configure_btn");
	m_pConfBtn->setPixmap(urlconf_toolbar_xpm);
	m_pConfBtn->setTextLabel(__tr("Configure Dialog"));
	m_pConfBtn->setUsesBigPixmap(true);
	m_pConfBtn->setAutoRaise(true);
	connect(m_pConfBtn,SIGNAL(clicked()),this,SLOT(loadConfigure()));
	l->addWidget(m_pConfBtn);*/

	m_pParentAppCont->addApplet(this);
	show();

}

QSize UrlToolBar::sizeHint() const
{
	return QSize(50,KVI_IRCTOOLBARAPPLET_MAXIMUM_HEIGHT);
}

void UrlToolBar::loadUrlListWindow()
{
	UrlDlgList *tmpitem = findFrame(m_pParentFrame);
	if (tmpitem->toolbar) tmpitem->toolbar->m_pUrlBtn->setPixmap(url_toolbar_xpm);
	if (tmpitem->dlg) {
		tmpitem->dlg->close();
	} else {
		urllist(m_pParentFrame);
	}
}

void UrlToolBar::loadConfigure()
{
	if (!g_pConfigDialog) g_pConfigDialog = new ConfigDialog();
}

UrlToolBar::~UrlToolBar()
{
	//hide();
	//m_pParentAppCont->removeApplet(this);
	delete m_pUrlBtn;
//	delete m_pConfBtn;
	UrlDlgList *tmpitem = findFrame(m_pParentFrame);
	tmpitem->toolbar = 0;
}

// ---------------------------- CLASS URLTOOLBAR -------------------------end //

void saveUrlList()
{
	KviStr urllist;
	g_pApp->getLocalKvircDirectory(urllist,KviApp::ConfigPlugins);
	urllist += g_pUrlListFilename;
	QFile file;
	file.setName(urllist.ptr());
	file.open(IO_WriteOnly);

	QTextStream stream(&file);

	stream << g_pList->count() << endl;

	for(KviUrl *tmp=g_pList->first();tmp;tmp=g_pList->next())
	{
		stream << tmp->url.ptr() << endl;
		stream << tmp->window.ptr() << endl;
		stream << tmp->count << endl;
		stream << tmp->timestamp.ptr() << endl;
	}
	file.flush();
	file.close();
}

void loadUrlList()
{
	KviStr urllist;
	g_pApp->getLocalKvircDirectory(urllist,KviApp::ConfigPlugins);
	urllist += g_pUrlListFilename;
	QFile file;
	file.setName(urllist.ptr());
	if (!file.open(IO_ReadOnly))return;
	QTextStream stream(&file);

	g_pList->clear();

	for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (tmpitem->dlg) tmpitem->dlg->m_pUrlList->clear();
	}
	KviUrl *tmp;
	int i=0;
	int num = stream.readLine().toInt();
	while ((!stream.eof()) && (i<num)){
		tmp = new KviUrl();
		tmp->url = stream.readLine();
		tmp->window = stream.readLine();
		tmp->count = stream.readLine().toInt();
		tmp->timestamp = stream.readLine();

		g_pList->append(tmp);

		for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
			if (tmpitem->dlg) {
				QString tmpCount;
				tmpCount.setNum(tmp->count);
				tmpitem->dlg->addUrl(QString(tmp->url.ptr()), QString(tmp->window.ptr()), tmpCount, QString(tmp->timestamp.ptr()));
			}
		}
		i++;
	}
	file.close();
}

void saveBanList()
{
	KviStr banlist;
	g_pApp->getLocalKvircDirectory(banlist,KviApp::ConfigPlugins);
	banlist += g_pBanListFilename;
	QFile file;
	file.setName(banlist.ptr());
	file.open(IO_WriteOnly);

	QTextStream stream(&file);

	stream << g_pBanList->count() << endl;
	for(KviStr *tmp=g_pBanList->first();tmp;tmp=g_pBanList->next())
	{
		stream << tmp->ptr() << endl;
	}
	file.flush();
	file.close();
}

void loadBanList()
{
	KviStr banlist;
	g_pApp->getLocalKvircDirectory(banlist,KviApp::ConfigPlugins);
	banlist += g_pBanListFilename;
	QFile file;
	file.setName(banlist.ptr());
	if (!file.open(IO_ReadOnly))return;
	QTextStream stream(&file);

	g_pBanList->clear();

	int i=0;
	int num = stream.readLine().toInt();
	while ((!stream.eof()) && (i<num)){
		KviStr *tmp = new KviStr(stream.readLine());
		g_pBanList->append(tmp);
		i++;
	}
	file.close();
}


/*
 	@doc: url.list
	@type:
		command
	@title:
		url.list
	@short:
		Opens url list
	@syntax:
		url.list
	@description:
		This command opens a window containing the urls' list.
		In the list there is other information:[br]
		[U]Window[/U] : window where the url has been shown last[br]
		[U]Count[/U] : number of urls shown[br]
		[U]Timestamp[/U] : date/time when the url has been shown first[br]
		Clicking right on the url column of the list a menu will popup, through it
		you can remove the selected item, find the url in the window it appeared last, and
		say it to: [I]@Console, Channels, Querys, DCC Chats[/I] and [I]User windows[/I].[br]
		The list is saved to file when you click on the menu item or when you unload the plugin
		on condition that you have checked the relative checkbox in configuration dialog.[br]
		You can also open the url in your web browser double clicking on it in the url list window.[br][br]

  */

bool urllist_command(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"urllist_command");
	urllist(c->window()->frame());
	return c->leaveContext();
}

UrlDlgList *findFrame(KviFrame *frame)
{
	UrlDlgList *tmpitem;
	for (tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (frame == tmpitem->frm) break;
	}
	if (!tmpitem) {
		UrlDlgList *udl = new UrlDlgList();
		udl->frm = frame;
		udl->dlg = 0;
		udl->toolbar = 0;
		udl->menu_id = -1;
		g_pUrlDlgList->append(udl);
		tmpitem = g_pUrlDlgList->current();
	}
	return tmpitem;
}

bool urllist(KviFrame *frame)
{
	UrlDlgList *tmpitem = findFrame(frame);
	if (tmpitem->dlg) return false;

	tmpitem->dlg = new UrlDialog(g_pList,tmpitem->frm);
	tmpitem->frm->addWindow(tmpitem->dlg);

	for(KviUrl *tmp=g_pList->first();tmp;tmp=g_pList->next())
	{
		QString tmpCount;
		tmpCount.setNum(tmp->count);
		tmpitem->dlg->addUrl(QString(tmp->url.ptr()), QString(tmp->window.ptr()), tmpCount, QString(tmp->timestamp.ptr()));
	}
	return true;
}


/*
 	@doc: url.config
	@type:
		command
	@title:
		url.config
	@short:
		Loads URL list module
	@syntax:
		url.config
	@description:
		This command opens a configuration window where it is possible
		to setup plugin's parameters. You can also open this window by
		using popup menu in the url list window or by clicking on the "configure plugin" button
		in plugins options.<BR><BR>
		<H3>Configure dialog options:</H3>
		There is an option which loads a small <U>toolbar</U> when the kvirc onStartup event is fired.<BR>
		In the toolbar you'll find 2 icons:<BR>
		<UL><LI>the first toggles the url list window and when the onUrl event is fired the button is highlighted;</LI>
		<LI>the second opens the configure dialog.</LI></UL>
		There is also a ban list widget, which allows to have a list of words that plugin mustn't catch.<BR><BR>
		<I>E.g.<BR>
		<blockquote>if the word "ftp" is inserted in the ban list and if in a window there is an output like "ftp.kvirc.net",
		the url will not be catched.</blockquote></I>
		<HR>
  */

bool urllistconfig(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"urllistconfig");
	if (!g_pConfigDialog) g_pConfigDialog = new ConfigDialog();
	return c->leaveContext();
}

int check_url(KviWindow *w,KviParameterList *p)		// return 0 if no occurence of the url were found
{
	int tmp = 0;

	for(KviStr *tmpi=g_pBanList->first();tmpi;tmpi=g_pBanList->next())
	{
		if (p->safeFirst()->contains(tmpi->ptr())) tmp++;
	}
	if (tmp > 0) return tmp;

	for(KviUrl *u = g_pList->first();u;u=g_pList->next())
	{
		if (u->url == p->safeFirst()->ptr()) {
			u->window = w->plainTextCaption();
			u->count++;
			tmp++;
		}
	}

	for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (tmpitem->dlg) {
			QListViewItemIterator lvi(tmpitem->dlg->m_pUrlList);
			for(;lvi.current();++lvi)
			{
				if (lvi.current()->text(0) == p->safeFirst()->ptr()) {
					int tmpCount = lvi.current()->text(2).toInt();
					tmpCount++;
					QString tmpStr;
					tmpStr.setNum(tmpCount);
					lvi.current()->setText(2,tmpStr);
					lvi.current()->setText(1,w->plainTextCaption());
				}
			}
		}
	}
	return tmp;
}

bool toolbar_command(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"toolbar_command");
	UrlDlgList *tmpitem = findFrame(c->window()->frame());
	if (!tmpitem->toolbar)
	{
		tmpitem->toolbar = new UrlToolBar(c->window()->frame()->frameAppletContainer(),c->window()->frame());
	}
	return c->leaveContext();
}

bool urllist_module_event_onUrl(KviModule *m,KviWindow *w,KviParameterList * p)
{
	if (check_url(w,p) == 0) {

		KviUrl *tmp = new KviUrl;
		KviStr tmpTimestamp;
		QDate d = QDate::currentDate();
		KviStr date(KviStr::Format,"%d-%d%d-%d%d",d.year(),d.month() / 10,d.month() % 10,d.day() / 10,d.day() % 10);
		tmpTimestamp = "["+date+"]"+" [";
		tmpTimestamp += QTime::currentTime().toString()+"]";
		tmp->url = *(p->safeFirst());
		tmp->window = w->plainTextCaption();
		tmp->count = 1;
		tmp->timestamp = tmpTimestamp;

		g_pList->append(tmp);
		for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
			if (tmpitem->dlg) {
				QString tmpCount;
				tmpCount.setNum(tmp->count);
				tmpitem->dlg->addUrl(QString(tmp->url.ptr()), QString(tmp->window.ptr()), tmpCount, QString(tmp->timestamp.ptr()));
				tmpitem->dlg->taskBarItem()->highlight(false);
			} else {
				if (tmpitem->toolbar) tmpitem->toolbar->m_pUrlBtn->setPixmap(urlhigh_toolbar_xpm);
			}
		}
	}
	return true;
}

bool urllist_module_event_onFrameWindowCreated(KviModule *m,KviWindow *w,KviParameterList * p)
{
	KviConfig cfg(szConfigPath.ptr());
	cfg.setGroup("ConfigDialog");
	UrlDlgList *tmpitem = findFrame(w->frame());
	if (tmpitem->menu_id == -1) {
		tmpitem->menu_id = w->frame()->mainMenuBar()->toolsMenu()->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_URL)),__c2q(__tr("URL List")),w->frame(),SLOT(executeInternalCommand(int)));
		w->frame()->mainMenuBar()->toolsMenu()->setItemParameter(tmpitem->menu_id,KVI_INTERNALCOMMAND_URL_OPEN);
	}
	if (cfg.readBoolEntry("LoadUrlToolbarOnFrameWindowCreated",true)) {
		tmpitem->toolbar = new UrlToolBar(w->frame()->frameAppletContainer(),w->frame());
	}
	return false;
}

bool urllist_module_event_onFrameWindowDestroyed(KviModule *m,KviWindow *w,KviParameterList * p)
{
	UrlDlgList *tmpitem;
	for (tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (w->frame() == tmpitem->frm) break;
	}
	if (tmpitem->menu_id != -1)
	{
		tmpitem->frm->mainMenuBar()->toolsMenu()->removeItem(tmpitem->menu_id);
		tmpitem->menu_id = -1;
	}
	if (tmpitem->toolbar)
	{
		tmpitem->toolbar->container()->removeApplet(tmpitem->toolbar);
		delete tmpitem->toolbar;
		tmpitem->toolbar = 0;
	}
	return false;
}

/*	@doc: url.load
	@type:
		command
	@title:
		url.load
	@short:
		Loads URL list module
	@syntax:
		url.load
	@description:
		Loads the URL list module which keeps track of all urls shown in kvirc windows.
*/

static bool url_module_init(KviModule *m)
{
	g_pList = new KviPtrList<KviUrl>;
	g_pList->setAutoDelete(true);

	g_pUrlDlgList = new KviPtrList<UrlDlgList>;
	g_pUrlDlgList->setAutoDelete(true);

	g_pBanList = new KviPtrList<KviStr>;
	g_pBanList->setAutoDelete(true);

	m->registerCommand("list",urllist_command);
	m->registerCommand("config",urllistconfig);
	m->registerCommand("toolbar",toolbar_command);
	m->registerEventHandler(KviEvent_OnUrl,urllist_module_event_onUrl);
	m->registerEventHandler(KviEvent_OnFrameWindowCreated,urllist_module_event_onFrameWindowCreated);
	m->registerEventHandler(KviEvent_OnFrameWindowDestroyed,urllist_module_event_onFrameWindowDestroyed);

	g_pApp->getLocalKvircDirectory(szConfigPath,KviApp::ConfigPlugins,"url.conf");

	loadUrlList();
	loadBanList();

	for(KviFrame *f = g_pApp->frameList()->first();f;f=g_pApp->frameList()->next()) {
		UrlDlgList *udl = new UrlDlgList();
		udl->frm = f;
		udl->dlg = 0;
		udl->toolbar = 0;
		udl->menu_id = f->mainMenuBar()->toolsMenu()->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_URL)),__c2q(__tr("URL List")),f,SLOT(executeInternalCommand(int)));
		g_pUrlDlgList->append(udl);
		f->mainMenuBar()->toolsMenu()->setItemParameter(udl->menu_id,KVI_INTERNALCOMMAND_URL_OPEN);
	}

	return true;
}

static bool url_module_cleanup(KviModule *m)
{
	KviConfig cfg(szConfigPath.ptr());
	cfg.setGroup("ConfigDialog");
	if (cfg.readBoolEntry("SaveUrlListOnUnload",false) == true) saveUrlList();
	for (UrlDlgList *tmpitem=g_pUrlDlgList->first();tmpitem;tmpitem=g_pUrlDlgList->next()) {
		if (tmpitem->toolbar) {
			tmpitem->toolbar->container()->removeApplet(tmpitem->toolbar);
			delete tmpitem->toolbar;
		}
		if (tmpitem->dlg) tmpitem->dlg->close();
		if (tmpitem->menu_id != -1) tmpitem->frm->mainMenuBar()->toolsMenu()->removeItem(tmpitem->menu_id);
	}

	delete g_pList;
	delete g_pBanList;
	delete g_pUrlDlgList;

	m->unregisterAllEventHandlers();
	m->unregisterAllCommands();

	m->unregisterMetaObject("UrlDialog");
	m->unregisterMetaObject("BanFrame");
	m->unregisterMetaObject("ConfigDialog");
	m->unregisterMetaObject("UrlToolBar");
	return true;
}

static bool url_module_can_unload(KviModule *m)
{
	return false;
}

void url_module_config()
{
	if (!g_pConfigDialog) g_pConfigDialog = new ConfigDialog();
}

/*
	@doc: gnutella
	@type:
		module
	@short:
		The URL list module: keeps track of all urls shown in kvirc windows
	@title:
		The URL list module
	@body:
		This plugin keeps track of all urls shown in kvirc windows.
		<H3>Exported commands:</H3>
		<B>/urllist</B> : this command opens a window containing the urls' list.
		In the list there is other information:<BR>
		<U>Window</U> : window where the url has been shown last<BR>
		<U>Count</U> : number of urls shown<BR>
		<U>Timestamp</U> : date/time when the url has been shown first<BR>
		Clicking right on the url column of the list a menu will popup, through it
		you can remove the selected item, find the url in the window it appeared last, and
		say it to: <I>@Console, Channels, Querys, DCC Chats</I> and <I>User windows</I>.<BR>
		The list is saved to file when you click on the menu item or when you unload the plugin
		on condition that you have checked the relative checkbox in configuration dialog.<BR>
		You can also open the url in your web browser double clicking on it in the url list window.<BR><BR>

		Mail me if you have any suggestion or you want to notice a bug.<BR>
		<B>Andrea 'YaP' Parrella</B> &lt;anandrea@iname.com&gt;<BR><BR>
		Thanks to:<BR>
		<B>Szymon Stefanek</B> and <B>Till Bush</B> for their help.<BR>
		<B>Ted Podgurski</B> for toolbar's icons.
*/

void url_module_help()
{
//#warning "show help"
	//kvirc_plugin_show_help_page("doc_plugin_url.kvihelp");
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info =
{
	"URL",
	"1.0.0" ,
	"Copyright (C) 2002 Andrea Parrella <yap@yapsoft.it>" ,
	"url list module for KVIrc",
	url_module_init,
	url_module_can_unload,
	0,
	url_module_cleanup
};


#include "libkviurl.moc"
