#ifndef _TRACKEDUSER_H_
#define _TRACKEDUSER_H_
//
//   File : trackeduser.h
//   Creation date : Fri Dec  7 17:08:07 2001 GMT by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_string.h"
#include "kvi_file.h"
#include "kvi_thread.h"
#include "kvi_module.h"
#include "kvi_window.h"
#include "kvi_parameterlist.h"

#include <time.h>
#include <qasciidict.h>
#include "kvi_list.h"

class KviTrackedUserDb;

#define KVI_TRACKEDUSER_MAX_OTHER_NICKNAMES 10
#define KVI_TRACKEDUSER_MAX_CHANNELS 8
#define KVI_TRACKEDUSER_MAX_USERNAMES 8
#define KVI_TRACKEDUSER_MAX_HOSTS 10
#define KVI_TRACKEDUSER_MAX_SERVERS 6
#define KVI_TRACKEDUSER_MAX_MEPRIVMSGS 5

// at this number of entries a cleanup is forcibly made when a new entry is requested
#define KVI_TRACKEDUSER_MAX_ENTRIES 2000
// at this number of entries that we want to have normally in the db
#define KVI_TRACKEDUSER_NORMAL_ENTRIES 1000


class KviTrackedUser
{
	friend class KviTrackedUserDb;
protected:
	KviTrackedUser(const char * nick);
public:
	~KviTrackedUser();
protected:
	bool            m_bDirty;

	KviStr          m_szNick;              // this record nickname

	KviPtrList<KviStr> * m_pOtherNicknames;     // other nicknames that this user uses (nick change to and from)
	KviPtrList<KviStr> * m_pChannels;           // channels that the user is on
	KviPtrList<KviStr> * m_pUsernames;
	KviPtrList<KviStr> * m_pHosts;
	KviPtrList<KviStr> * m_pServers;
	KviPtrList<KviStr> * m_pMePrivmsgs;

	KviStr          m_szLastEvent;

	time_t          m_tLastUpdated;        // last record update

	time_t          m_tCreated;            // record creation time
	// event counts
	unsigned int    m_uJoins;
	unsigned int    m_uParts;
	unsigned int    m_uQuits;
	unsigned int    m_uChanPrivmsgs;
	unsigned int    m_uMePrivmsgs;
public:
	KviPtrList<KviStr> * otherNicknames(){ return m_pOtherNicknames; };
	KviPtrList<KviStr> * channels(){ return m_pChannels; };
	KviPtrList<KviStr> * usernames(){ return m_pUsernames; };
	KviPtrList<KviStr> * hosts(){ return m_pHosts; };
	KviPtrList<KviStr> * servers(){ return m_pServers; };
	KviPtrList<KviStr> * mePrivmsgList(){ return m_pMePrivmsgs; };
	unsigned int joins(){ return m_uJoins; };
	unsigned int parts(){ return m_uParts; };
	unsigned int quits(){ return m_uQuits; };
	unsigned int chanPrivmsgs(){ return m_uChanPrivmsgs; };
	unsigned int mePrivmsgs(){ return m_uMePrivmsgs; };
	const char * lastEvent(){ return m_szLastEvent.ptr(); };
	time_t created(){ return m_tCreated; };
	time_t lastUpdated(){ return m_tLastUpdated; };
	bool dirty(){ return m_bDirty; };
	const char * nick(){ return m_szNick; };
	int nickLen(){ return m_szNick.len(); };
protected:
	bool load(KviFile * f);
	bool save(KviFile * f);

	void addNick(const KviStr &nick);
	void addUser(const KviStr &user);
	void addHost(const KviStr &host);
	void addChannel(const KviStr &chan);
	void addServer(const KviStr &serv);
	void addMePrivmsg(KviStr * pMsg);

	void increaseJoins(){ m_uJoins++; };
	void increaseParts(){ m_uParts++; };
	void increaseChanPrivmsgs(){ m_uChanPrivmsgs++; };
	void increaseMePrivmsgs(){ m_uMePrivmsgs++; };
	void increaseQuits(){ m_uQuits++; };

	KviStr & markLastEvent();
};

#define KVI_TRACKED_USER_THREAD_EVENT_JOIN (KVI_THREAD_USER_EVENT_BASE + 1001)

typedef struct _KviTrackedUserJoin
{
	KviStr szServer;
	KviStr szNick;
	KviStr szUser;
	KviStr szHost;
	KviStr szChan;
} KviTrackedUserJoin;

#define KVI_TRACKED_USER_THREAD_EVENT_PART (KVI_THREAD_USER_EVENT_BASE + 1002)

typedef struct _KviTrackedUserPart
{
	KviStr szServer;
	KviStr szNick;
	KviStr szUser;
	KviStr szHost;
	KviStr szChan;
	KviStr szReason;
} KviTrackedUserPart;

#define KVI_TRACKED_USER_THREAD_EVENT_QUIT (KVI_THREAD_USER_EVENT_BASE + 1003)

typedef struct _KviTrackedUserQuit
{
	KviStr szServer;
	KviStr szNick;
	KviStr szUser;
	KviStr szHost;
	KviPtrList<KviStr> lChans;
	KviStr szReason;
} KviTrackedUserQuit;

#define KVI_TRACKED_USER_THREAD_EVENT_NICKCHANGE (KVI_THREAD_USER_EVENT_BASE + 1004)

typedef struct _KviTrackedUserNickChange
{
	KviStr szServer;
	KviStr szNick;
	KviStr szUser;
	KviStr szHost;
	KviPtrList<KviStr> lChans;
	KviStr szNewNick;
} KviTrackedUserNickChange;


#define KVI_TRACKED_USER_THREAD_EVENT_CHANNELMESSAGE (KVI_THREAD_USER_EVENT_BASE + 1005)

typedef struct _KviTrackedUserChannelMessage
{
	KviStr szServer;
	KviStr szNick;
	KviStr szUser;
	KviStr szHost;
	KviStr szChan;
	KviStr szMessage;
} KviTrackedUserChannelMessage;

#define KVI_TRACKED_USER_THREAD_EVENT_QUERYMESSAGE (KVI_THREAD_USER_EVENT_BASE + 1006)

typedef struct _KviTrackedUserQueryMessage
{
	KviStr szServer;
	KviStr szNick;
	KviStr szUser;
	KviStr szHost;
	KviStr szMessage;
} KviTrackedUserQueryMessage;

class KviTrackedUserDb : public KviSensitiveThread
{
public:
	KviTrackedUserDb(const char *configPath,const char * dbPath);
	~KviTrackedUserDb();
protected:
	KviStr                       m_szDbPath;
	KviStr                       m_szConfigPath;
	KviMutex                   * m_pUserDictMutex;
	QAsciiDict<KviTrackedUser> * m_pUserDict;
	unsigned long int            m_uTrackingTimeBeforeStartup;
	time_t                       m_tStartup;
public:
	void lock(){ m_pUserDictMutex->lock(); };
	KviTrackedUser * findEntry(const char * nick); // returns 0 if the entry is not in memory nor disk
	void unlock(){ m_pUserDictMutex->unlock(); };
	unsigned int trackingTime();
protected:
	KviTrackedUser * getEntry(const char * nick); // if not exists , it is created
	void flush();
	KviTrackedUser * loadEntry(const char * nick);
	void dropOldEntries();
	void dropEntry(KviTrackedUser * u);
	void dropOldEntries(unsigned int tTimeDiff);
	void saveEntry(KviTrackedUser * u);
	void processEvent(KviThreadEvent * e);
	virtual void run();
};

#endif //_TRACKEDUSER_H_
