//
//   File : trackeduser.cpp
//   Creation date : Fri Dec  7 17:08:08 2001 GMT by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "trackeduser.h"
#include "kvi_config.h"
#include "kvi_fileutils.h"


KviTrackedUser::KviTrackedUser(const char * nick)
{
	m_szNick = nick;
	m_tCreated = time(0);
	m_bDirty = true;
	m_pOtherNicknames = 0;
	m_pChannels = 0;
	m_pUsernames = 0;
	m_pHosts = 0;
	m_uJoins = 0;
	m_uParts = 0;
	m_pServers = 0;
	m_uChanPrivmsgs = 0;
	m_uMePrivmsgs = 0;
	m_pMePrivmsgs = 0;
	m_uQuits = 0;
}

KviTrackedUser::~KviTrackedUser()
{
	if(m_pOtherNicknames)delete m_pOtherNicknames;
	if(m_pChannels)delete m_pChannels;
	if(m_pUsernames)delete m_pUsernames;
	if(m_pHosts)delete m_pHosts;
	if(m_pServers)delete m_pServers;
	if(m_pMePrivmsgs)delete m_pMePrivmsgs;
}

#define KVI_TU_FLAG_HAVE_OTHERNICKNAMES 1
#define KVI_TU_FLAG_HAVE_CHANNELS 2
#define KVI_TU_FLAG_HAVE_USERNAMES 4
#define KVI_TU_FLAG_HAVE_HOSTS 8
#define KVI_TU_FLAG_HAVE_SERVERS 16
#define KVI_TU_FLAG_HAVE_MEPRIVMSGS 32

bool KviTrackedUser::load(KviFile * f)
{
	m_bDirty = false;
	if(!f->load(m_szNick))return false;
	int iFlags;
	if(!f->load(iFlags))return false;
	if(iFlags & KVI_TU_FLAG_HAVE_OTHERNICKNAMES)
	{
		m_pOtherNicknames = new KviPtrList<KviStr>;
		m_pOtherNicknames->setAutoDelete(true);
		if(!f->load(m_pOtherNicknames))return false;
	}
	if(iFlags & KVI_TU_FLAG_HAVE_CHANNELS)
	{
		m_pChannels = new KviPtrList<KviStr>;
		m_pChannels->setAutoDelete(true);
		if(!f->load(m_pChannels))return false;
	}
	if(iFlags & KVI_TU_FLAG_HAVE_USERNAMES)
	{
		m_pUsernames = new KviPtrList<KviStr>;
		m_pUsernames->setAutoDelete(true);
		if(!f->load(m_pUsernames))return false;
	}
	if(iFlags & KVI_TU_FLAG_HAVE_HOSTS)
	{
		m_pHosts = new KviPtrList<KviStr>;
		m_pHosts->setAutoDelete(true);
		if(!f->load(m_pHosts))return false;
	}
	if(iFlags & KVI_TU_FLAG_HAVE_SERVERS)
	{
		m_pServers = new KviPtrList<KviStr>;
		m_pServers->setAutoDelete(true);
		if(!f->load(m_pServers))return false;
	}
	if(iFlags & KVI_TU_FLAG_HAVE_MEPRIVMSGS)
	{
		m_pMePrivmsgs = new KviPtrList<KviStr>;
		m_pMePrivmsgs->setAutoDelete(true);
		if(!f->load(m_pMePrivmsgs))return false;
	}
	if(!f->load(m_tLastUpdated))return false;
	if(!f->load(m_szLastEvent))return false;
	if(!f->load(m_tCreated))return false;
	if(!f->load(m_uJoins))return false;
	if(!f->load(m_uParts))return false;
	if(!f->load(m_uChanPrivmsgs))return false;
	if(!f->load(m_uQuits))return false;
	if(!f->load(m_uMePrivmsgs))return false;
	return true;
}

bool KviTrackedUser::save(KviFile * f)
{
	m_bDirty = false;
	int iFlags = m_pOtherNicknames ? KVI_TU_FLAG_HAVE_OTHERNICKNAMES : 0;
	if(m_pChannels)iFlags |= KVI_TU_FLAG_HAVE_CHANNELS;
	if(m_pUsernames)iFlags |= KVI_TU_FLAG_HAVE_USERNAMES;
	if(m_pHosts)iFlags |= KVI_TU_FLAG_HAVE_HOSTS;
	if(m_pServers)iFlags |= KVI_TU_FLAG_HAVE_SERVERS;
	if(m_pMePrivmsgs)iFlags |= KVI_TU_FLAG_HAVE_MEPRIVMSGS;
	if(!f->save(m_szNick))return false;
	if(!f->save(iFlags))return false;
	if(m_pOtherNicknames)if(!f->save(m_pOtherNicknames))return false;
	if(m_pChannels)if(!f->save(m_pChannels))return false;
	if(m_pUsernames)if(!f->save(m_pUsernames))return false;
	if(m_pHosts)if(!f->save(m_pHosts))return false;
	if(m_pServers)if(!f->save(m_pServers))return false;
	if(m_pMePrivmsgs)if(!f->save(m_pMePrivmsgs))return false;
	if(!f->save(m_tLastUpdated))return false;
	if(!f->save(m_szLastEvent))return false;
	if(!f->save(m_tCreated))return false;
	if(!f->save(m_uJoins))return false;
	if(!f->save(m_uParts))return false;
	if(!f->save(m_uChanPrivmsgs))return false;
	if(!f->save(m_uQuits))return false;
	if(!f->save(m_uMePrivmsgs))return false;
	return true;
}

void KviTrackedUser::addMePrivmsg(KviStr * pMsg)
{
	if(m_pMePrivmsgs)
	{
		while(m_pMePrivmsgs->count() >= KVI_TRACKEDUSER_MAX_MEPRIVMSGS)m_pMePrivmsgs->removeFirst();
	} else {
		m_pMePrivmsgs = new KviPtrList<KviStr>;
		m_pMePrivmsgs->setAutoDelete(true);
	}
	m_pMePrivmsgs->append(pMsg);
}

void KviTrackedUser::addNick(const KviStr &nick)
{
	if(m_pOtherNicknames)
	{
		for(KviStr * s = m_pOtherNicknames->first();s;s = m_pOtherNicknames->next())
		{
			if(kvi_strEqualCI(nick.ptr(),s->ptr()))return;
		}
		while(m_pOtherNicknames->count() >= KVI_TRACKEDUSER_MAX_OTHER_NICKNAMES)m_pOtherNicknames->removeFirst();
	} else {
		m_pOtherNicknames = new KviPtrList<KviStr>;
		m_pOtherNicknames->setAutoDelete(true);
	}
	m_pOtherNicknames->append(new KviStr(nick));
}

void KviTrackedUser::addUser(const KviStr &user)
{
	if(m_pUsernames)
	{
		for(KviStr * s = m_pUsernames->first();s;s = m_pUsernames->next())
		{
			if(kvi_strEqualCI(user.ptr(),s->ptr()))return;
		}
		while(m_pUsernames->count() >= KVI_TRACKEDUSER_MAX_USERNAMES)m_pUsernames->removeFirst();
	} else {
		m_pUsernames = new KviPtrList<KviStr>;
		m_pUsernames->setAutoDelete(true);
	}
	m_pUsernames->append(new KviStr(user));
}

void KviTrackedUser::addHost(const KviStr &host)
{
	if(m_pHosts)
	{
		for(KviStr * s = m_pHosts->first();s;s = m_pHosts->next())
		{
			if(kvi_strEqualCI(host.ptr(),s->ptr()))return;
		}
		while(m_pHosts->count() >= KVI_TRACKEDUSER_MAX_HOSTS)m_pHosts->removeFirst();
	} else {
		m_pHosts = new KviPtrList<KviStr>;
		m_pHosts->setAutoDelete(true);
	}
	m_pHosts->append(new KviStr(host));
}

void KviTrackedUser::addChannel(const KviStr &chan)
{
	if(m_pChannels)
	{
		for(KviStr * s = m_pChannels->first();s;s = m_pChannels->next())
		{
			if(kvi_strEqualCI(chan.ptr(),s->ptr()))return;
		}
		while(m_pChannels->count() >= KVI_TRACKEDUSER_MAX_CHANNELS)m_pChannels->removeFirst();
	} else {
		m_pChannels = new KviPtrList<KviStr>;
		m_pChannels->setAutoDelete(true);
	}
	m_pChannels->append(new KviStr(chan));
}

void KviTrackedUser::addServer(const KviStr &serv)
{
	if(m_pServers)
	{
		for(KviStr * s = m_pServers->first();s;s = m_pServers->next())
		{
			if(kvi_strEqualCI(serv.ptr(),s->ptr()))return;
		}
		while(m_pServers->count() >= KVI_TRACKEDUSER_MAX_SERVERS)m_pServers->removeFirst();
	} else {
		m_pServers = new KviPtrList<KviStr>;
		m_pServers->setAutoDelete(true);
	}
	m_pServers->append(new KviStr(serv));
}

KviStr & KviTrackedUser::markLastEvent()
{
	m_tLastUpdated = time(0);
	m_bDirty = true;
	return m_szLastEvent;
}








KviTrackedUserDb::KviTrackedUserDb(const char * configPath,const char * dbPath)
: KviSensitiveThread()
{
	m_szDbPath = dbPath;
	m_szConfigPath = configPath;
	kvi_adjustFilePath(m_szDbPath);
	m_szDbPath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	m_pUserDict = new QAsciiDict<KviTrackedUser>(3037,false);
	m_pUserDict->setAutoDelete(true);
	m_tStartup = time(0);
	KviConfig cfg(m_szConfigPath);
	m_uTrackingTimeBeforeStartup = cfg.readUIntEntry("TrackingTime",0);
	m_pUserDictMutex = new KviMutex();
}

KviTrackedUserDb::~KviTrackedUserDb()
{
	terminate();
	flush();
	m_pUserDictMutex->lock();
	delete m_pUserDict;
	m_pUserDictMutex->unlock();
	m_pUserDict = 0;
	delete m_pUserDict;
	time_t tNow = time(0);
	unsigned int tDiff = ((unsigned int)tNow - (unsigned int)m_tStartup);
	KviConfig cfg(m_szConfigPath);
	unsigned int uSecs = cfg.readUIntEntry("TrackingTime",0);
	uSecs += tDiff;
	cfg.writeEntry("TrackingTime",uSecs);
}

unsigned int KviTrackedUserDb::trackingTime()
{
	return (m_uTrackingTimeBeforeStartup + ((unsigned int)(time(0)) - (unsigned int)m_tStartup));
}

KviTrackedUser * KviTrackedUserDb::getEntry(const char * nick)
{
	KviTrackedUser * u = findEntry(nick);
	if(!u)
	{
		u = new KviTrackedUser(nick);
		m_pUserDict->insert(nick,u);
	}
	return u;
}

KviTrackedUser * KviTrackedUserDb::findEntry(const char * nick)
{
	KviTrackedUser * u = m_pUserDict->find(nick);
	if(!u)return loadEntry(nick);
	return u;
}

KviTrackedUser * KviTrackedUserDb::loadEntry(const char * nick)
{
	KviStr szPath = m_szDbPath;
	KviStr szNick = nick;
	kvi_encodeFileName(szNick);
	szPath.append(szNick);
	KviFile f(szPath.ptr());
	if(f.open(IO_ReadOnly))
	{
		KviTrackedUser * u = new KviTrackedUser(nick);
		if(u->load(&f))
		{
			m_pUserDict->insert(nick,u);
		} else {
			//debug("Broken user file (%s->%s) ?",nick,szPath);
			delete u;
			u = 0;
		}
		f.close();
		return u;
	}
	return 0;
}

void KviTrackedUserDb::dropEntry(KviTrackedUser * u)
{
	saveEntry(u);
	m_pUserDict->remove(u->nick());
}

void KviTrackedUserDb::dropOldEntries()
{
	dropOldEntries(60);
	if(m_pUserDict->count() < KVI_TRACKEDUSER_NORMAL_ENTRIES)return;
	dropOldEntries(20);
}

void KviTrackedUserDb::dropOldEntries(unsigned int uTimeDiff)
{
	QAsciiDictIterator<KviTrackedUser> it(*m_pUserDict);
	KviPtrList<KviTrackedUser> l;

	l.setAutoDelete(false);
	time_t tNow = time(0);

	while(it.current())
	{
		if(((unsigned int)(((unsigned int)tNow)- ((unsigned int)(it.current()->lastUpdated())))) >= uTimeDiff)
		{
			l.append(it.current());
		}
		++it;
	}

	for(KviTrackedUser * u=l.first();u;u = l.next())
	{
		dropEntry(u);
	}
}

void KviTrackedUserDb::saveEntry(KviTrackedUser * u)
{
	if(!u->dirty())return;
	KviStr szPath = m_szDbPath;
	KviStr szNick = u->nick();
	kvi_encodeFileName(szNick);
	szPath.append(szNick);
	KviFile f(szPath.ptr());
	if(f.open(IO_WriteOnly | IO_Truncate))
	{
		if(!u->save(&f))
			debug("Ops...can't save the entry %s",u->nick());
	}
}

void KviTrackedUserDb::flush()
{
	QAsciiDictIterator<KviTrackedUser> it(*m_pUserDict);
	while(it.current())
	{
		if(it.current()->dirty())saveEntry(it.current());
		++it;
	}
}

void KviTrackedUserDb::processEvent(KviThreadEvent * e)
{
	KviTrackedUser * u;

	switch(e->id())
	{
		case KVI_TRACKED_USER_THREAD_EVENT_JOIN:
		{
			KviTrackedUserJoin * j = ((KviThreadDataEvent<KviTrackedUserJoin> *)e)->data();
			u = getEntry(j->szNick.ptr());
			if(!u)return;
			u->addServer(j->szServer);
			u->addChannel(j->szChan);
			u->addUser(j->szUser);
			u->addHost(j->szHost);
			u->increaseJoins();
			u->markLastEvent().sprintf("[%s] %s!%s@%s join %s",
				j->szServer.ptr(),j->szNick.ptr(),j->szUser.ptr(),j->szHost.ptr(),j->szChan.ptr());
//			debug("JOIN");
		}
		break;
		case KVI_TRACKED_USER_THREAD_EVENT_PART:
		{
			KviTrackedUserPart * j = ((KviThreadDataEvent<KviTrackedUserPart> *)e)->data();
			u = getEntry(j->szNick.ptr());
			if(!u)return;
			u->addServer(j->szServer);
			u->addChannel(j->szChan);
			u->addUser(j->szUser);
			u->addHost(j->szHost);
			u->increaseParts();
			u->markLastEvent().sprintf("[%s] %s!%s@%s part %s (%s)",
				j->szServer.ptr(),j->szNick.ptr(),j->szUser.ptr(),j->szHost.ptr(),j->szChan.ptr(),j->szReason.ptr());
			// we could drop the entry if the user is not queried nor on channels
			// ... ?
		}
		break;
		case KVI_TRACKED_USER_THREAD_EVENT_QUIT:
		{
			KviTrackedUserQuit * j = ((KviThreadDataEvent<KviTrackedUserQuit> *)e)->data();
			u = getEntry(j->szNick.ptr());
			if(!u)return;
			u->addServer(j->szServer);
			u->addUser(j->szUser);
			u->addHost(j->szHost);
			KviStr szChans;
			for(KviStr * s = j->lChans.first();s;s = j->lChans.next())
			{
				u->addChannel(*s);
				if(szChans.hasData())szChans.append(',');
				szChans.append(*s);
			}
			u->increaseQuits();
			u->markLastEvent().sprintf("[%s] %s!%s@%s quit %s (%s)",
				j->szServer.ptr(),j->szNick.ptr(),j->szUser.ptr(),j->szHost.ptr(),szChans.ptr(),j->szReason.ptr());
			dropEntry(u);
		}
		break;
		case KVI_TRACKED_USER_THREAD_EVENT_NICKCHANGE:
		{
			KviTrackedUserNickChange * j = ((KviThreadDataEvent<KviTrackedUserNickChange> *)e)->data();
			u = getEntry(j->szNick.ptr());
			if(!u)return;
			u->addServer(j->szServer);
			u->addUser(j->szUser);
			u->addHost(j->szHost);
			u->addNick(j->szNewNick);
			KviStr szChans;
			for(KviStr * s = j->lChans.first();s;s = j->lChans.next())
			{
				u->addChannel(*s);
				if(szChans.hasData())szChans.append(',');
				szChans.append(*s);
			}
//			u->increaseNic();
			u->markLastEvent().sprintf("[%s] %s!%s@%s changes nick to %s (on %s)",
				j->szServer.ptr(),j->szNick.ptr(),j->szUser.ptr(),j->szHost.ptr(),j->szNewNick.ptr(),szChans.ptr());
			dropEntry(u);
			u = getEntry(j->szNewNick.ptr());
			if(!u)return;
			u->addNick(j->szNick);
			u->addServer(j->szServer);
			u->addUser(j->szUser);
			u->addHost(j->szHost);
			for(KviStr * s = j->lChans.first();s;s = j->lChans.next())
			{
				u->addChannel(*s);
			}
			u->markLastEvent().sprintf("[%s] %s!%s@%s changes nick to %s (on %s)",
				j->szServer.ptr(),j->szNick.ptr(),j->szUser.ptr(),j->szHost.ptr(),j->szNewNick.ptr(),szChans.ptr());
		}
		break;
		case KVI_TRACKED_USER_THREAD_EVENT_CHANNELMESSAGE:
		{
			KviTrackedUserChannelMessage * j = ((KviThreadDataEvent<KviTrackedUserChannelMessage> *)e)->data();
			u = getEntry(j->szNick.ptr());
			if(!u)return;
			u->addServer(j->szServer);
			u->addUser(j->szUser);
			u->addHost(j->szHost);
			u->addChannel(j->szChan);
			u->markLastEvent().sprintf("[%s] %s!%s@%s talks to %s: %s",
				j->szServer.ptr(),j->szNick.ptr(),j->szUser.ptr(),j->szHost.ptr(),j->szChan.ptr(),j->szMessage.ptr());
			u->increaseChanPrivmsgs();
		}
		break;
		case KVI_TRACKED_USER_THREAD_EVENT_QUERYMESSAGE:
		{
			KviTrackedUserQueryMessage * j = ((KviThreadDataEvent<KviTrackedUserQueryMessage> *)e)->data();
			u = getEntry(j->szNick.ptr());
			if(!u)return;
			u->addServer(j->szServer);
			u->addUser(j->szUser);
			u->addHost(j->szHost);
			u->markLastEvent().sprintf("[%s] %s!%s@%s talks to me: %s",
				j->szServer.ptr(),j->szNick.ptr(),j->szUser.ptr(),j->szHost.ptr(),j->szMessage.ptr());
			u->increaseMePrivmsgs();
			u->addMePrivmsg(new KviStr(KviStr::Format,"%u:%s",(unsigned int)(u->lastUpdated()),j->szMessage.ptr()));
		}
		break;
	}
}

void KviTrackedUserDb::run()
{
	bool bDropping = false;
	for(;;)
	{
		if(bDropping)
		{
			// We are in dropping mode
			m_pUserDictMutex->lock();
			if(m_pUserDict->count() > KVI_TRACKEDUSER_NORMAL_ENTRIES)
			{
				// still stuff to drop... ignore events too
				dropOldEntries();
			} else {
				// last drop was enough... don't ignore events
				bDropping = false;
			}
			m_pUserDictMutex->unlock();
		}

		KviThreadEvent * e = dequeueEvent();

		if(e)
		{
			if(e->id() == KVI_THREAD_EVENT_TERMINATE)
			{
				// ups... they want us to quit!
				delete e;
				return;
			}

			if(!bDropping)
			{
				m_pUserDictMutex->lock();
				processEvent(e);
				m_pUserDictMutex->unlock();
			}
			delete e;

		} else msleep(200); // 1/5 sec

		m_pUserDictMutex->lock();
		if(m_pUserDict->count() > KVI_TRACKEDUSER_MAX_ENTRIES)
		{
			// ops.. we need to drop down the entry number
			bDropping = true;
			dropOldEntries();
		}
		m_pUserDictMutex->unlock();
	}
}
