//
//   File : libkvitracker.cpp
//   Creation date : Thu Dec 06 2001 19:45:02 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"

#include "kvi_fileutils.h"
#include "kvi_app.h"
#include "kvi_event.h"
#include "kvi_window.h"
#include "kvi_console.h"
#include "kvi_channel.h"
#include "kvi_out.h"
#include "kvi_mirccntrl.h"
#include "kvi_locale.h"

#include "kvi_time.h"

#include "trackeduser.h"

#include <qdatetime.h>

static KviTrackedUserDb * g_pTrackedUserDb = 0;


static bool tracker_module_event_OnNickChange(KviModule * m,KviWindow *w,KviParameterList *p)
{
	KviTrackedUserNickChange * tuq = new KviTrackedUserNickChange;

	tuq->szServer = w->console()->currentServerName();
	tuq->szNick = *(p->safeFirst());
	tuq->szUser = *(p->safeNext());
	tuq->szHost = *(p->safeNext());
	tuq->szNewNick = *(p->safeNext());
	tuq->lChans.setAutoDelete(true);

	KviPtrList<KviChannel> * cl = w->console()->channelList();
	if(cl)
	{
		for(KviChannel * ch = cl->first();ch;ch = cl->next())
		{
			if(ch->isOn(tuq->szNick.ptr()))tuq->lChans.append(new KviStr(ch->name()));
		}
	}

	g_pTrackedUserDb->enqueueEvent(new KviThreadDataEvent<KviTrackedUserNickChange>(KVI_TRACKED_USER_THREAD_EVENT_NICKCHANGE,tuq));
	return true;
}

static bool tracker_module_event_OnJoin(KviModule * m,KviWindow * w,KviParameterList * p)
{
	KviTrackedUserJoin * tuj = new KviTrackedUserJoin;

	tuj->szServer = w->console()->currentServerName();
	tuj->szNick = *(p->safeFirst());
	tuj->szUser = *(p->safeNext());
	tuj->szHost = *(p->safeNext());
	tuj->szChan = w->name();

	g_pTrackedUserDb->enqueueEvent(new KviThreadDataEvent<KviTrackedUserJoin>(KVI_TRACKED_USER_THREAD_EVENT_JOIN,tuj));

	return true;
}

static bool tracker_module_event_OnPart(KviModule *m,KviWindow * w,KviParameterList * p)
{
	KviTrackedUserPart * tup = new KviTrackedUserPart;

	tup->szServer = w->console()->currentServerName();
	tup->szNick = *(p->safeFirst());
	tup->szUser = *(p->safeNext());
	tup->szHost = *(p->safeNext());
	tup->szReason = *(p->safeNext());
	tup->szChan = w->name();

	g_pTrackedUserDb->enqueueEvent(new KviThreadDataEvent<KviTrackedUserPart>(KVI_TRACKED_USER_THREAD_EVENT_PART,tup));

	return true;
}

static bool tracker_module_event_OnQuit(KviModule *m,KviWindow *w,KviParameterList * p)
{
	KviTrackedUserQuit * tuq = new KviTrackedUserQuit;

	tuq->szServer = w->console()->currentServerName();
	tuq->szNick = *(p->safeFirst());
	tuq->szUser = *(p->safeNext());
	tuq->szHost = *(p->safeNext());
	tuq->szReason = *(p->safeNext());
	tuq->lChans.setAutoDelete(true);

	KviPtrList<KviChannel> * cl = w->console()->channelList();
	if(cl)
	{
		for(KviChannel * ch = cl->first();ch;ch = cl->next())
		{
			if(ch->isOn(tuq->szNick.ptr()))tuq->lChans.append(new KviStr(ch->name()));
		}
	}

	g_pTrackedUserDb->enqueueEvent(new KviThreadDataEvent<KviTrackedUserQuit>(KVI_TRACKED_USER_THREAD_EVENT_QUIT,tuq));

	return true;
}

static bool tracker_module_event_OnChannelMessage(KviModule *m,KviWindow *w,KviParameterList * p)
{
	KviTrackedUserChannelMessage * tuq = new KviTrackedUserChannelMessage;

	tuq->szServer = w->console()->currentServerName();
	tuq->szNick = *(p->safeFirst());
	tuq->szUser = *(p->safeNext());
	tuq->szHost = *(p->safeNext());
	tuq->szMessage = *(p->safeNext());
	tuq->szChan = w->name();

	g_pTrackedUserDb->enqueueEvent(new KviThreadDataEvent<KviTrackedUserChannelMessage>(KVI_TRACKED_USER_THREAD_EVENT_CHANNELMESSAGE,tuq));

	return true;
}

static bool tracker_module_event_OnQueryMessage(KviModule *m,KviWindow *w,KviParameterList * p)
{
	KviTrackedUserQueryMessage * tuq = new KviTrackedUserQueryMessage;

	tuq->szServer = w->console()->currentServerName();
	tuq->szNick = *(p->safeFirst());
	tuq->szUser = *(p->safeNext());
	tuq->szHost = *(p->safeNext());
	tuq->szMessage = *(p->safeNext());

	g_pTrackedUserDb->enqueueEvent(new KviThreadDataEvent<KviTrackedUserQueryMessage>(KVI_TRACKED_USER_THREAD_EVENT_QUERYMESSAGE,tuq));

	return true;
}


//static bool tracker_module_event_OnMePart(KviModule *m,KviWindow *w,KviParameterList * p)
//{
//	return true;
//}

/*
	@doc: tracker.info
	@type:
		command
	@title:
		tracker.info
	@keyterms:
		tracking irc users
	@short:
		Dumps a tracker database entry
	@syntax:
		tracker.info <nickname>
	@description:
	@examples:
	@seealso:
*/

static void merge_list(KviPtrList<KviStr> * l,KviStr &buffer,const char * prepend,const char * append)
{
	buffer = "";
	if(!l)return;
	for(KviStr * s = l->first();s;s = l->next())
	{
		if(buffer.hasData())buffer.append(", ");
		if(prepend)buffer.append(prepend);
		buffer.append(*s);
		if(append)buffer.append(append);
	}
}

static bool tracker_module_cmd_info(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"tracker_module_cmd_info");

	KviStr nick;
	if(!g_pUserParser->parseCmdFinalPart(c,nick))return false;

	if(nick.isEmpty())c->warning(__tr("No nickname given"));
	else {
		g_pTrackedUserDb->lock();
		KviTrackedUser * u = g_pTrackedUserDb->findEntry(nick.ptr());
		if(u)
		{
			c->window()->output(KVI_OUT_WHOISUSER,__tr("%cTracking database entry for \r!n\r%s\r"),KVI_TEXT_BOLD,nick.ptr());
			QDateTime tim;
			tim.setTime_t(u->created());
			KviStr tmp = tim.toString();
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Created on %c%s"),KVI_TEXT_BOLD,tmp.ptr());
			tim.setTime_t(u->lastUpdated());
			tmp = tim.toString();
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Last updated on %c%s"),KVI_TEXT_BOLD,tmp.ptr());
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Last event: %c%s"),KVI_TEXT_BOLD,u->lastEvent());
			merge_list(u->otherNicknames(),tmp,
					__tr("\r![!dbl]tracker.info $0[!txt]Double click the nickname to see the info entry\r"),"\r");
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Chained nicknames: %c%s"),KVI_TEXT_BOLD,tmp.ptr());
			merge_list(u->usernames(),tmp,0,0);
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Usernames: %c%s"),KVI_TEXT_BOLD,tmp.ptr());
			merge_list(u->hosts(),tmp,"\r!h\r","\r");
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Hosts: %c%s"),KVI_TEXT_BOLD,tmp.ptr());
			merge_list(u->channels(),tmp,"\r!c\r","\r");
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Channels: %c%s"),KVI_TEXT_BOLD,tmp.ptr());
			merge_list(u->servers(),tmp,"\r!s\r","\r");
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Was observed from: %c%s"),KVI_TEXT_BOLD,tmp.ptr());
			unsigned int uD,uH,uM,uS;
			kvi_secondsToDaysHoursMinsSecs(g_pTrackedUserDb->trackingTime(),&uD,&uH,&uM,&uS);
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Event counts (total tracking time: %u d %u h %u m %u s)"),uD,uH,uM,uS);
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Joins count: %c%u"),KVI_TEXT_BOLD,u->joins());
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Parts count: %c%u"),KVI_TEXT_BOLD,u->parts());
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Quits count: %c%u"),KVI_TEXT_BOLD,u->quits());
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Chan privmsgs count: %c%u"),KVI_TEXT_BOLD,u->chanPrivmsgs());
			c->window()->output(KVI_OUT_WHOISUSER,__tr("Me privmsgs count: %c%u"),KVI_TEXT_BOLD,u->mePrivmsgs());
			KviPtrList<KviStr> * l = u->mePrivmsgList();
			if(l)
			{
				int cnt = l->count();
				if(cnt > 0)
				{
					c->window()->output(KVI_OUT_WHOISUSER,__tr("Last %d private messages directed to me"),cnt);
					for(KviStr * s = l->first();s;s = l->next())
					{
						int idx = s->findFirstIdx(':');
						if(idx != -1)
						{
							KviStr num(s->left(idx));
							bool bOk;
							unsigned int uTime = num.toInt(&bOk);
							if(bOk)
							{
								QDateTime dtt;
								dtt.setTime_t(uTime);
								KviStr msg(*s);
								msg.cutToFirst(':');
								tmp = dtt.toString();
								c->window()->output(KVI_OUT_WHOISUSER,__tr("[%s]: %s"),tmp.ptr(),msg.ptr());
							}
						}
					}
				}
			}
			c->window()->output(KVI_OUT_WHOISUSER,__tr("%cEnd of tracking database entry for \r!n\r%s\r"),KVI_TEXT_BOLD,nick.ptr());
		} else {
			c->window()->output(KVI_OUT_WHOISUSER,__tr("No database entry for %s"),nick.ptr());
		}
		g_pTrackedUserDb->unlock();
	}


	return c->leaveContext();
}

static bool tracker_module_init(KviModule * m)
{
	KviStr szDir;
	g_pApp->getLocalKvircDirectory(szDir,KviApp::ConfigPlugins,"trackdb");
	kvi_makeDir(szDir.ptr());
	szDir.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	kvi_adjustFilePath(szDir);
	KviStr szConfig;
	m->getDefaultConfigFileName(szConfig);
	g_pTrackedUserDb = new KviTrackedUserDb(szConfig.ptr(),szDir.ptr());
	g_pTrackedUserDb->start();

	m->registerEventHandler(KviEvent_OnJoin,tracker_module_event_OnJoin);
	m->registerEventHandler(KviEvent_OnPart,tracker_module_event_OnPart);
	m->registerEventHandler(KviEvent_OnQuit,tracker_module_event_OnQuit);
	m->registerEventHandler(KviEvent_OnNickChange,tracker_module_event_OnNickChange);
	m->registerEventHandler(KviEvent_OnChannelMessage,tracker_module_event_OnChannelMessage);
	m->registerEventHandler(KviEvent_OnQueryMessage,tracker_module_event_OnQueryMessage);

	m->registerCommand("info",tracker_module_cmd_info);

	return true;
}

static bool tracker_module_cleanup(KviModule *m)
{
	m->unregisterAllEventHandlers();
	delete g_pTrackedUserDb;
	return true;
}

static bool tracker_module_can_unload(KviModule *m)
{
	// never unload automatically
	return false;
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Tracker",                                                // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)", // author & (C)
	"User tracker module for KVIrc",
	tracker_module_init,
	tracker_module_can_unload,
	0,
	tracker_module_cleanup
};
