//
//   File : libkvisystem.cpp
//   Creation date : Fri Nov 16 03:50:12 2001 GMT by Szymon Stefanek
//
//   This system is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_settings.h"
#include "kvi_module.h"
#include "kvi_string.h"
#include "kvi_uparser.h"
//#include "kvi_systemutils.h"
#include "kvi_locale.h"
#include "kvi_command.h"
//#include "kvi_malloc.h"
//#include "kvi_app.h"
#include "kvi_env.h"

#ifndef COMPILE_ON_WINDOWS
	#include <sys/utsname.h>
	#include <stdlib.h>
	#include <unistd.h>
#endif


/*
	@doc: system.ostype
	@type:
		function
	@title:
		$system.ostype
	@short:
		Returns the type of the operating system
	@syntax:
		$system.ostype
	@description:
		Returns the current type of operating system: unix or windows.[br]
*/

static bool system_module_fnc_ostype(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	buffer.append("unix");
#endif
	return true; //c->leaveContext();
}

/*
	@doc: system.osname
	@type:
		function
	@title:
		$system.osname
	@short:
		Returns the name of the operating system
	@syntax:
		$system.osname
	@description:
		Returns the name of the operating system.[br]
*/

static bool system_module_fnc_osname(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	struct utsname uts;
	if(uname(&uts) == 0)buffer.append(uts.sysname);
#endif
	return true; //c->leaveContext();
}

/*
	@doc: system.osversion
	@type:
		function
	@title:
		$system.osversion
	@short:
		Returns the version of the operating system
	@syntax:
		$system.osversion
	@description:
		Returns the version of the operating system.[br]
*/

static bool system_module_fnc_osversion(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	struct utsname uts;
	if(uname(&uts) == 0)buffer.append(uts.version);
#endif
	return true; //c->leaveContext();
}


/*
	@doc: system.osrelease
	@type:
		function
	@title:
		$system.osrelease
	@short:
		Returns the release of the operating system
	@syntax:
		$system.osrelease
	@description:
		Returns the release of the operating system.[br]
*/

static bool system_module_fnc_osrelease(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	struct utsname uts;
	if(uname(&uts) == 0)buffer.append(uts.release);
#endif
	return true; //c->leaveContext();
}

/*
	@doc: system.osmachine
	@type:
		function
	@title:
		$system.osmachine
	@short:
		Returns the machine of the operating system
	@syntax:
		$system.osmachine
	@description:
		Returns the machine of the operating system.[br]
*/

static bool system_module_fnc_osmachine(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	struct utsname uts;
	if(uname(&uts) == 0)buffer.append(uts.machine);
#endif
	return true; //c->leaveContext();
}

/*
	@doc: system.osnodename
	@type:
		function
	@title:
		$system.osnodename
	@short:
		Returns the nodename of the operating system
	@syntax:
		$system.osnodename
	@description:
		Returns the nodename of the operating system.[br]
*/

static bool system_module_fnc_osnodename(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	struct utsname uts;
	if(uname(&uts) == 0)buffer.append(uts.nodename);
#endif
	return true; //c->leaveContext();
}

/*
	@doc: system.getenv
	@keyterms:
		Enviroinement variables
	@type:
		function
	@title:
		$system.getenv
	@short:
		Returns the value of an enviroinement variable
	@syntax:
		$system.getenv(<variable>)
	@description:
		Returns the value of the enviroinement <variable>.[br]
*/

static bool system_module_fnc_getenv(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"system_module_fnc_size");
	char * b = kvi_getenv(parms->safeFirstParam());
	if(b)buffer.append(b);
	return true; //c->leaveContext();
}



/*
	@doc: system.hostname
	@keyterms:
		System information
	@type:
		function
	@title:
		$system.hostname
	@short:
		Returns the hostname of the machine that KVIrc is running on
	@syntax:
		$system.hostname
	@description:
		Returns the hostname of the machine that KVIrc is running on.[br]
*/

static bool system_module_fnc_hostname(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_CONTEXT(c,"system_module_fnc_size");
	char hbuffer[1024];
	if(gethostname(hbuffer,1024) == 0)buffer.append(hbuffer);
	return true; //c->leaveContext();
}

/*
	@doc: system.setenv
	@type:
		command
	@title:
		system.setenv
	@keyterms:
		Enviroinement variables
	@short:
		Sets an enviroinement variable
	@syntax:
		system.setenv <variable> [<value>]
	@description:
		Sets the enviroinement <variable> to the <value> string.[br]
		If <value> is not given , the <variable> is unset.[br]
	@seealso:
		[fnc]$system.getenv[/fnc]()
*/

static bool system_module_cmd_setenv(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"system_module_cmd_setenv");
	KviStr szVariable,szValue;
	if(!g_pUserParser->parseCmdSingleToken(c,szVariable))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szValue))return false;

	if(szVariable.hasData())
	{
		if(szValue.hasData())
		{
			kvi_setenv(szVariable.ptr(),szValue.ptr());
		} else {
			kvi_unsetenv(szVariable.ptr());
		}
	} else {
		c->warning(__tr("No environment variable name given"));
	}

	return c->leaveContext();
}

static bool system_module_init(KviModule * m)
{
	m->registerFunction("ostype",system_module_fnc_ostype);
	m->registerFunction("osname",system_module_fnc_osname);
	m->registerFunction("osversion",system_module_fnc_osversion);
	m->registerFunction("osrelease",system_module_fnc_osrelease);
	m->registerFunction("osmachine",system_module_fnc_osmachine);
	m->registerFunction("osnodename",system_module_fnc_osnodename);
	m->registerFunction("getenv",system_module_fnc_getenv);
	m->registerFunction("hostname",system_module_fnc_hostname);

	m->registerCommand("setenv",system_module_cmd_setenv);

	return true;
}

static bool system_module_cleanup(KviModule *m)
{
	return true;
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"System",                                                 // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)", // author & (C)
	"System informations module",
	system_module_init,
	0,
	0,
	system_module_cleanup
};
