//   File : libkvispaste.cpp
//   Creation date : Thu Dec 27 2002 17:13:12 GMT by Juanjo lvarez
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2002 Juanjo lvarez (juanjux@yahoo.es)
//   Copyright (C) 2002 Szymon Stefanek (kvirc@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "libkvispaste.h"
#include "controller.h"

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_fileutils.h"
#include "kvi_app.h"
#include "kvi_locale.h"
#include "kvi_command.h"
#include "kvi_uparser.h"
#include "kvi_console.h"
#include "kvi_options.h"
#include "kvi_out.h"

#include <qfile.h>
#include <qclipboard.h>

#ifndef COMPILE_ON_WINDOWS
    #include <unistd.h>
#endif

KviPtrList<SPasteController> * g_pControllerList = 0;
int ctrlId = 0;

static KviWindow * spaste_module_find_window(KviStr &win, KviCommand * c)
{
    KviWindow * w;
    if(!win.isEmpty())w = g_pApp->findWindow(win.ptr());
    else w = c->window();
    if(!w)
    {
        c->warning(__tr("Can't find the window with id '%s'"),win.ptr());
        return 0;
    }
    if((w->type() == KVI_WINDOW_TYPE_CHANNEL) || (w->type() == KVI_WINDOW_TYPE_QUERY) || (w->type() == KVI_WINDOW_TYPE_DCCCHAT))return w;
    c->warning(__tr("The specified window (%s) is not a channel/query/dcc"),win.ptr());
    return 0;
}
//-------------------------------------------------
/*
    @doc: spaste.file
    @type:
        command
    @title:
        spaste.file
    @short:
        Send the contents of a file to a window, with a delay between each line
    @syntax:
        spaste.file <filename>
        spaste.file <filename> [window]
    @description:
        Send the contents of a file to a conversation window doing a pause between each line. [br]
        the optional window parameter must be a channel, query or DCC chat window, if [br]
        no window is specified the text will be send to the current window.
    @seealso:
        [cmd]spaste.clipboard[/cmd],
        [cmd]spaste.stop[/cmd],
        [cmd]spaste.list[/cmd],
        [cmd]spaste.setdelay[/cmd]
*/
//-------------------------------------------
// spaste.file
//-------------------------------------------
static bool spaste_module_cmd_file(KviModule *m,KviCommand *c)
{
 	ENTER_CONTEXT(c,"spaste_module_cmd_file");
    KviStr fname;
    if(!g_pUserParser->parseCmdSingleToken(c,fname))return false;
    if(fname.isEmpty())
    {
        c->warning(__tr("No file specified"));
        return false;
    }
    
    KviStr winName;
    QString fileStr(fname.ptr());
    if(!g_pUserParser->parseCmdFinalPart(c,winName))return false;
    KviWindow * window = spaste_module_find_window(winName,c);
    if( (!window) || window->console()->isNotConnected())return false;

	if(fileStr.isEmpty() || (!kvi_fileExists(__q2ascii(fileStr))))
	{
        c->warning(__tr("File not found or empty"));
		return false;
	}

    QFile tmp(__q2ascii(fileStr));
    if(!tmp.open(IO_ReadOnly)) {
        c->warning(__tr("I can't open that file"));
        return false;
    }
    tmp.close();

    SPasteController * controller = new SPasteController(window,++ctrlId);
    if(!controller->pasteFileInit(fileStr)) {
        c->warning(__tr("Could not paste file"));
        return false;
    }

	return c->leaveContext();
}
/*
    @doc: spaste.clipboard
    @type:
        command
    @title:
        spaste.clipboard
    @short:
        Send the contents of the clipboard to a window, with a delay between each line
    @syntax:
        spaste.clipboard
        spaste.clipboard [window]
    @description:
        Send the contents of the clipboard to a conversation window doing a pause between each line. [br]
        the optional window parameter must be a channel, query or DCC chat window. [br]
        If no window is specified the text will be send to the current window.
    @seealso:
        [cmd]spaste.file[/cmd],
        [cmd]spaste.stop[/cmd],
        [cmd]spaste.list[/cmd],
        [cmd]spaste.setdelay[/cmd]
*/
//-------------------------------------------------
// spaste.clipboard
//-------------------------------------------------
static bool spaste_module_cmd_clipboard(KviModule *m,KviCommand *c)
{
    ENTER_CONTEXT(c,"spaste_module_cmd_clipboard");
    KviStr winName;
    if(!g_pUserParser->parseCmdFinalPart(c,winName))return false;
    
    KviWindow * window = spaste_module_find_window(winName,c);
    if( (!window) || window->console()->isNotConnected())return false;

    SPasteController * controller = new SPasteController(window,++ctrlId);
    controller->pasteClipboardInit();
    
    return c->leaveContext();
}

/*
    @doc: spaste.stop
    @type:
        command
    @title:
        spaste.stop
    @short:
        Stops one or more slow-paste process.
    @syntax:
        spaste.stop
        spaste.stop [-a] [id]
    @description:
        This commands stop one or more slow-paste process. It can operate in three ways. The firt, [br]
        without any parameter or switch, stops all slow-paste processes running in the same window [br]
        the command is called. The second, using the switch -a stops all slow paste processes running [br]
        on all windows. The third form, without the switch and specifying a numerical slow paste process id [br]
        (which you can obtain with the [cmd]spaste.list[/cmd] command) stops only that process id.
    @seealso:
        [cmd]spaste.clipboard[/cmd],
        [cmd]spaste.file[/cmd],
        [cmd]spaste.list[/cmd],
        [cmd]spaste.setdelay[/cmd]

*/
//--------------------------------------------------
// spaste.stop
//--------------------------------------------------
static bool spaste_module_cmd_stop(KviModule *m,KviCommand *c)
{
    ENTER_CONTEXT(c,"spaste_module_cmd_stop");
    KviStr id;
    if(!g_pUserParser->parseCmdSingleToken(c,id))return false;
    
    if(c->hasSwitch('a')) //Delete all spaste's
    {
        while(g_pControllerList->first()) delete g_pControllerList->first();
        return c->leaveContext();
    } else 
    {
        KviPtrListIterator<SPasteController> it(*g_pControllerList);
        SPasteController *item;
        
        if(id.isEmpty()) //Delete all spaste's from the current window
        {
            if((c->window()->type() != KVI_WINDOW_TYPE_CHANNEL) && (c->window()->type() != KVI_WINDOW_TYPE_QUERY) && (c->window()->type() != KVI_WINDOW_TYPE_DCCCHAT))
            {
                c->warning(__tr("The specified window (%s) is not a channel/query/dcc"),c->window()->id());
                return false;
            } else 
            {
                
                while( (item = it.current()) != 0)
                {
                    ++it;
                    if(kvi_strEqualCS(item->window()->id(),c->window()->id()))delete item;
                }
            }
        } else //Delete the spaste with the given id
        { 
            while( (item = it.current()) != 0) 
            { 
                ++it;
                if(item->getId() == id.toInt())delete item;
            }
        }
        return c->leaveContext();
    } 
}
/*
    @doc: spaste.list
    @type:
        command
    @title:
        spaste.list
    @short:
        List all the running spaste process.
    @syntax:
        spaste.list
    @description:
        This command will list in the window where it is executed all the current slow-paste [br]
        process including their identification number and the window where they are running. [br]
    @seealso:
        [cmd]spaste.clipboard[/cmd],
        [cmd]spaste.file[/cmd],
        [cmd]spaste.stop[/cmd],
        [cmd]spaste.setdelay[/cmd]
*/
//--------------------------------------------------
// spaste.list
//--------------------------------------------------
static bool spaste_module_cmd_list(KviModule *m,KviCommand *c)
{
    ENTER_CONTEXT(c,"spaste_module_cmd_list");
    
    KviPtrListIterator<SPasteController> it(*g_pControllerList);
    SPasteController *item;
    
    while( (item = it.current()) != 0)
    {
        ++it;
        c->window()->output(KVI_OUT_NONE,__tr("Slow-paste Id:%d Window:%s"),item->getId(),item->window()->id());
    }
    return c->leaveContext();
}
/*
    @doc: spaste.setdelay
    @type:
        command
    @title:
        spaste.setdelay
    @short:
        Set the delay time in miliseconds for the spaste module commands delay
    @syntax:
        spaste.setdelay <time_in_msecs>
    @description:
        Set the delay tiem in miliseconds for the spaste module commands delay.
    @seealso:
        [cmd]spaste.clipboard[/cmd],
        [cmd]spaste.file[/cmd],
        [cmd]spaste.stop[/cmd],
        [cmd]spaste.list[/cmd]
*/
//-------------------------------------------------
// spaste.setdelay
//-------------------------------------------------

static bool spaste_module_cmd_setdelay(KviModule *m,KviCommand *c)
{
    KviStr msecs;
    if(!g_pUserParser->parseCmdFinalPart(c,msecs))return false;
    KVI_OPTION_UINT(KviOption_uintPasteDelay) = msecs.toLong();
    return true;
}

//-------------------------------------------------    
static bool spaste_module_init(KviModule * m)
{
    g_pControllerList = new KviPtrList<SPasteController>;
    g_pControllerList->setAutoDelete(false);
    
	m->registerCommand("file", spaste_module_cmd_file);
    m->registerCommand("clipboard", spaste_module_cmd_clipboard);
    m->registerCommand("setdelay", spaste_module_cmd_setdelay);
    m->registerCommand("stop", spaste_module_cmd_stop);
    m->registerCommand("list", spaste_module_cmd_list);
	return true;
}
//-------------------------------------------------
static bool spaste_module_cleanup(KviModule *m)
{
    while(g_pControllerList->first()) delete g_pControllerList->first();
    delete g_pControllerList;
    
	return true;
}
//-------------------------------------------------
static bool spaste_module_can_unload(KviModule *m)
{
    return (g_pControllerList->isEmpty());
}
//-------------------------------------------------
KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Sound",                                                 // module name
	"1.0.0",                                                // module version
	"          (C) 2002 Juanjo Alvarez (juanjux@yahoo.es)", // author & (C)
	"Delayed paste commands",
	spaste_module_init,
	0,
	0,
	spaste_module_cleanup
};
