#include "dirview.h"

#include "kvi_fileutils.h"
#include "kvi_list.h"

#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qpixmap.h>
#include <qevent.h>
#include <qpoint.h>
#include <qmessagebox.h>
#include <qdragobject.h>
#include <qmime.h>
#include <qstrlist.h>
#include <qstringlist.h>
#include <qapplication.h>
#include <qheader.h>

static const char* folder_closed_xpm[]={
    "16 16 9 1",
    "g c #808080",
    "b c #c0c000",
    "e c #c0c0c0",
    "# c #000000",
    "c c #ffff00",
    ". c None",
    "a c #585858",
    "f c #a0a0a4",
    "d c #ffffff",
    "..###...........",
    ".#abc##.........",
    ".#daabc#####....",
    ".#ddeaabbccc#...",
    ".#dedeeabbbba...",
    ".#edeeeeaaaab#..",
    ".#deeeeeeefe#ba.",
    ".#eeeeeeefef#ba.",
    ".#eeeeeefeff#ba.",
    ".#eeeeefefff#ba.",
    ".##geefeffff#ba.",
    "...##gefffff#ba.",
    ".....##fffff#ba.",
    ".......##fff#b##",
    ".........##f#b##",
    "...........####."};

static const char* folder_open_xpm[]={
    "16 16 11 1",
    "# c #000000",
    "g c #c0c0c0",
    "e c #303030",
    "a c #ffa858",
    "b c #808080",
    "d c #a0a0a4",
    "f c #585858",
    "c c #ffdca8",
    "h c #dcdcdc",
    "i c #ffffff",
    ". c None",
    "....###.........",
    "....#ab##.......",
    "....#acab####...",
    "###.#acccccca#..",
    "#ddefaaaccccca#.",
    "#bdddbaaaacccab#",
    ".eddddbbaaaacab#",
    ".#bddggdbbaaaab#",
    "..edgdggggbbaab#",
    "..#bgggghghdaab#",
    "...ebhggghicfab#",
    "....#edhhiiidab#",
    "......#egiiicfb#",
    "........#egiibb#",
    "..........#egib#",
    "............#ee#"};

static const char * folder_locked[]={
    "16 16 10 1",
    "h c #808080",
    "b c #ffa858",
    "f c #c0c0c0",
    "e c #c05800",
    "# c #000000",
    "c c #ffdca8",
    ". c None",
    "a c #585858",
    "g c #a0a0a4",
    "d c #ffffff",
    "..#a#...........",
    ".#abc####.......",
    ".#daa#eee#......",
    ".#ddf#e##b#.....",
    ".#dfd#e#bcb##...",
    ".#fdccc#daaab#..",
    ".#dfbbbccgfg#ba.",
    ".#ffb#ebbfgg#ba.",
    ".#ffbbe#bggg#ba.",
    ".#fffbbebggg#ba.",
    ".##hf#ebbggg#ba.",
    "...###e#gggg#ba.",
    ".....#e#gggg#ba.",
    "......###ggg#b##",
    ".........##g#b##",
    "...........####."};



Directory::Directory( Directory * parent, const QString& filename )
: QListViewItem( parent ), f(filename), pix( 0 )
{
 	p = parent;
	readable = QDir(fullName()).isReadable();
	if(!readable )setPixmap(((DirectoryView *)listView())->folderLocked);
	else setPixmap(((DirectoryView *)listView())->folderClosed);
}


Directory::Directory( QListView * parent, const QString& filename)
: QListViewItem( parent ), f(filename), pix( 0 )
{
	p = 0;
 	readable = QDir( fullName() ).isReadable();
}

void Directory::setPixmap( QPixmap *px )
{
	pix = px;
	setup();
	widthChanged( 0 );
	invalidateHeight();
	repaint();
}

const QPixmap *Directory::pixmap(int i) const
{
	return pix;
}

void Directory::setOpen( bool o )
{
	if ( o )setPixmap( ((DirectoryView *)listView())->folderOpen );
 	else  setPixmap( ((DirectoryView *)listView())->folderClosed );

	if ( o && !childCount() )
	{
		QString s( fullName() );
		QDir thisDir( s );
		if(!thisDir.isReadable() )
		{
			readable = FALSE;
			setExpandable( FALSE );
			return;
		}

		listView()->setUpdatesEnabled( FALSE );
		const QFileInfoList * files = thisDir.entryInfoList();
		if ( files )
		{
			QFileInfoListIterator it( *files );
			QFileInfo * fi;
			while( (fi=it.current()) != 0 )
			{
				++it;
				if((fi->fileName() != ".") && (fi->fileName() != "..") && fi->isDir())
							(void)new Directory( this, fi->fileName() );
			}
		}
		listView()->setUpdatesEnabled( TRUE );
	}
	QListViewItem::setOpen( o );
}


void Directory::setup()
{
	setExpandable( TRUE );
 	QListViewItem::setup();
}


QString Directory::fullName()
{
	QString s;
	if(p)
	{
		s = p->fullName();
		s.append( f.name() );
		s.append(KVI_PATH_SEPARATOR);
	} else {
		s = f.name();
	}
    return s;
}


QString Directory::text( int column ) const
{
	return f.name();
}

/*****************************************************************************
 *
 * Class DirectoryView
 *
 *****************************************************************************/

DirectoryView::DirectoryView(QWidget *parent,const char *name)
: QListView( parent, name ) , oldCurrent( 0 ), mousePressed( FALSE )
{
    addColumn( "Name" );
    //setTreeStepSize( 20 );

	folderLocked = new QPixmap( folder_locked );
	folderClosed = new QPixmap( folder_closed_xpm );
	folderOpen = new QPixmap( folder_open_xpm );
	connect( this, SIGNAL( doubleClicked( QListViewItem * ) ),this, SLOT( slotFolderSelected( QListViewItem * )));
	connect( this, SIGNAL( returnPressed( QListViewItem * ) ),this, SLOT( slotFolderSelected( QListViewItem * )));

    const QFileInfoList* roots = QDir::drives();
    KviPtrListIterator<QFileInfo> i(*roots);
    QFileInfo* fi;
    while ( (fi = *i) )
	{
		KviStr szPath = fi->filePath();
		++i;
		// Skip the floppy-disks
		if(!(kvi_strEqualCIN(szPath.ptr(),"a:",2) || kvi_strEqualCIN(szPath.ptr(),"b:",2)))
		{
			Directory * root = new Directory(this, fi->filePath() );
			if( roots->count() <= 1)root->setOpen( TRUE ); // be interesting
		}
    }

}

DirectoryView::~DirectoryView()
{
	delete folderLocked;
	delete folderClosed;
	delete folderOpen;
}

void DirectoryView::slotFolderSelected( QListViewItem *i )
{
 	emit folderSelected(((Directory *)i)->fullName() );
}




QString DirectoryView::fullPath(QListViewItem* item)
{
    QString fullpath = item->text(0);
    while ( (item=item->parent()) ) {
	if ( item->parent() )
	    fullpath = item->text(0) + KVI_PATH_SEPARATOR + fullpath;
	else
	    fullpath = item->text(0) + fullpath;
    }
    return fullpath;
}

void DirectoryView::contentsMousePressEvent( QMouseEvent* e )
{
    QListView::contentsMousePressEvent(e);
    QPoint p( contentsToViewport( e->pos() ) );
    QListViewItem *i = itemAt( p );
    if ( i ) {
	// if the user clicked into the root decoration of the item, don't try to start a drag!
	if ( p.x() > header()->cellPos( header()->mapToActual( 0 ) ) +
	     treeStepSize() * ( i->depth() + ( rootIsDecorated() ? 1 : 0) ) + itemMargin() ||
	     p.x() < header()->cellPos( header()->mapToActual( 0 ) ) ) {
	    presspos = e->pos();
	    mousePressed = TRUE;
	}
    }
}


void DirectoryView::contentsMouseReleaseEvent( QMouseEvent * )
{
    mousePressed = FALSE;
}

void DirectoryView::setDir( const QString &s )
{
    QListViewItemIterator it( this );
    ++it;
    for ( ; it.current(); ++it ) {
	it.current()->setOpen( FALSE );
    }

    QStringList lst( QStringList::split( "/", s ) );
    QListViewItem *item = firstChild();
    QStringList::Iterator it2 = lst.begin();
    for ( ; it2 != lst.end(); ++it2 ) {
	while ( item ) {
	    if ( item->text( 0 ) == *it2 ) {
		item->setOpen( TRUE );
		break;
	    }
	    item = item->itemBelow();
	}
    }

    if ( item )
	setCurrentItem( item );
}


#include "dirview.moc"
