//
//   File : popupeditor.cpp
//   Creation date : Sat Jul 7 2001 15:06:10 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "popupeditor.h"

#include <qsplitter.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qtooltip.h>
#include <qasciidict.h>

#include "kvi_scripteditor.h"

#include "kvi_debug.h"
#include "kvi_locale.h"
#include "kvi_app.h"
#include "kvi_iconmanager.h"
#include "kvi_popupmanager.h"
#include "kvi_filedialog.h"
#include "kvi_fileutils.h"

//KviPopupEntryItem
KviPopupListViewItem::KviPopupListViewItem(QListView * pListView,KviPopupListViewItem * after,Type t)
: QListViewItem(pListView,after)
{
	m_type = t;
	init();
}

KviPopupListViewItem::KviPopupListViewItem(KviPopupListViewItem * parent,KviPopupListViewItem * after,Type t)
: QListViewItem(parent,after)
{
	m_type = t;
	init();
}

void KviPopupListViewItem::init()
{
	switch(m_type)
	{
		case Item:
			setText(1,__c2q(__tr("Item")));
		break;
		case Menu:
			setText(1,__c2q(__tr("Submenu")));
		break;
		case ExtMenu:
			setText(1,__c2q(__tr("External menu")));
		break;
		case Separator:
			setText(0,"-----------------------");
			setText(1,__c2q(__tr("Separator")));
		break;
		case Label:
			setText(1,__c2q(__tr("Label")));
		break;
		case Epilogue:
			setText(0,__c2q(__tr("### Epilogue ###")));
			setText(1,__c2q(__tr("Epilogue")));
		break;
		case Prologue:
			setText(0,__c2q(__tr("### Prologue ###")));
			setText(1,__c2q(__tr("Prologue")));
		break;
	}
}

void KviPopupListViewItem::setItemText(const char * szText)
{
	switch(m_type)
	{
		case Item:
		case Menu:
		case Label:
		case ExtMenu:
			m_szText = szText;
			setText(0,szText);
		break;
	}
}

void KviPopupListViewItem::setCondition(const char * szCondition)
{
	switch(m_type)
	{
		case Item:
		case Menu:
		case Label:
		case ExtMenu:
			m_szCondition = szCondition;
		break;
	}
}

void KviPopupListViewItem::setCode(const char * szCode)
{
	switch(m_type)
	{
		case Item:
		case Epilogue:
		case Prologue:
		case ExtMenu:
			m_szCode = szCode;
		break;
	}
}

void KviPopupListViewItem::setIcon(const char * szIcon)
{
	switch(m_type)
	{
		case Item:
		case Menu:
		case ExtMenu:
			m_szIcon = szIcon;
			{
				QPixmap * pix = g_pIconManager->getImage(szIcon);
				if(pix)setPixmap(0,*pix);
				else setPixmap(0,QPixmap());
			}
		break;
	}
}












KviSinglePopupEditor::KviSinglePopupEditor(QWidget * par)
: QWidget(par)
{
	m_pLastSelectedItem = 0;
	m_pContextPopup = new QPopupMenu(this);
	m_pClipboard = 0;

	QGridLayout * g = new QGridLayout(this,6,3,2,2);

	m_pNameEditor = new QLineEdit(this);
	QToolTip::add(m_pNameEditor,__c2q(__tr("Popup name")));

	g->addMultiCellWidget(m_pNameEditor,0,0,0,1);

	m_pMenuButton = new QToolButton(DownArrow,this);
	g->addWidget(m_pMenuButton,0,2);

	QPopupMenu * pop = new QPopupMenu(this);
	m_pMenuButton->setPopup(pop);
	m_pMenuButton->setPopupDelay(0);

	pop->insertItem(__c2q(__tr("Generate defpopup script snippet")),this,SLOT(menuGenerateDefPopup()));


	QSplitter * spl = new QSplitter(QSplitter::Vertical,this);

	m_pListView = new QListView(spl);
	m_pListView->addColumn(__c2q(__tr("Item")));
	m_pListView->addColumn(__c2q(__tr("Type")));
	m_pListView->setMultiSelection(false);
	m_pListView->setAllColumnsShowFocus(true);
	m_pListView->setRootIsDecorated(true);
	m_pListView->setShowSortIndicator(false);
	m_pListView->setSorting(-1);

	connect(m_pListView,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(selectionChanged(QListViewItem *)));
	connect(m_pListView,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),
		this,SLOT(itemPressed(QListViewItem *,const QPoint &,int)));

	m_pEditor = kvi_createScriptEditor(spl);

	g->addMultiCellWidget(spl,1,1,0,2);

	QLabel * l = new QLabel(__c2q(__tr("Text:")),this);
	l->setMargin(2);
	g->addWidget(l,2,0);

	m_pTextEditor = new QLineEdit(this);
	QToolTip::add(m_pTextEditor,
		__c2q(__tr("<center><b>Visible text</b><br>May contain identifiers that will be evaluated at popup call time.<br>For labels this text can contain also limited html tags.</center>")));
	g->addMultiCellWidget(m_pTextEditor,2,2,1,2);

	l = new QLabel(__c2q(__tr("Condition:")),this);
	l->setMargin(2);
	g->addWidget(l,3,0);

	m_pConditionEditor = new QLineEdit(this);
	QToolTip::add(m_pConditionEditor,
		__c2q(__tr("<center><b>Boolean condition</b><br>Will be evaluated at popup call time in order to decide if this entry has to be shown.<br>An empty condition evaluates to true.</center>")));
	g->addMultiCellWidget(m_pConditionEditor,3,3,1,2);

	l = new QLabel(__c2q(__tr("Icon:")),this);
	l->setMargin(2);
	g->addWidget(l,4,0);

	m_pIconEditor = new QLineEdit(this);
	QToolTip::add(m_pIconEditor,
		__c2q(__tr("<center><b>Icon identifier</b><br>May be an internal icon id, an absolute path or a relative path.<br>Portable scripts should never use absolute paths.</center>")));
	g->addMultiCellWidget(m_pIconEditor,4,4,1,2);

	l = new QLabel(__c2q(__tr("External menu:")),this);
	l->setMargin(2);
	g->addWidget(l,5,0);

	m_pExtNameEditor = new QLineEdit(this);
	QToolTip::add(m_pExtNameEditor,
		__c2q(__tr("<center><b>External menu name</b><br>This allows to nest externally defined popup menus. The popup menu with the specified name will be looked up at menu setup time.</center>")));
	g->addMultiCellWidget(m_pExtNameEditor,5,5,1,2);

	g->setColStretch(1,1);
	g->setRowStretch(1,1);

}

KviSinglePopupEditor::~KviSinglePopupEditor()
{
	if(m_pClipboard)delete m_pClipboard;
}

void KviSinglePopupEditor::menuGenerateDefPopup()
{
	KviStr fName;
	if(KviFileDialog::askForSaveFileName(fName,__tr("Choose a save file name"),"defpopup.kvs",__tr("KVIrc scripts (*.kvs);;All files (*)"),false,true))
	{
		KviPopupMenu * m = getMenu();
		KviStr buffer;
		m->generateDefPopup(buffer);
		kvi_writeFile(fName.ptr(),buffer);
		delete m;
	}
}

KviPopupListViewItem * KviSinglePopupEditor::findPrologue(KviPopupListViewItem * par)
{
	return findFirst(par,KviPopupListViewItem::Prologue);
}

KviPopupListViewItem * KviSinglePopupEditor::findEpilogue(KviPopupListViewItem * par)
{
	return findFirst(par,KviPopupListViewItem::Epilogue);
}

KviPopupListViewItem * KviSinglePopupEditor::findFirst(KviPopupListViewItem * par,KviPopupListViewItem::Type t)
{
	KviPopupListViewItem * it = par ? (KviPopupListViewItem *)par->firstChild() : (KviPopupListViewItem *)m_pListView->firstChild();

	while(it)
	{
		if(it->m_type == t)return it;
		it = (KviPopupListViewItem *)it->nextSibling();
	}
	return 0;
}

void KviSinglePopupEditor::itemPressed(QListViewItem *it,const QPoint &pnt,int col)
{
	m_pContextPopup->clear();

	KviPopupListViewItem * parentMenu = 0;

	if(it)
	{
		parentMenu = (KviPopupListViewItem *) (((KviPopupListViewItem *)it)->parent());

		bool bIsMenu = ((KviPopupListViewItem *)it)->m_type == KviPopupListViewItem::Menu;

		m_pContextPopup->insertItem(
				*(g_pIconManager->getSmallIcon(KVI_SMALLICON_CUT)),
				__c2q(__tr("Cut")),
				this,SLOT(contextCut()));
		m_pContextPopup->insertItem(
				*(g_pIconManager->getSmallIcon(KVI_SMALLICON_COPY)),
				__c2q(__tr("Copy")),
				this,SLOT(contextCopy()));
		if(m_pClipboard)
		{
			m_pContextPopup->insertItem(
				*(g_pIconManager->getSmallIcon(KVI_SMALLICON_PASTE)),
				__c2q(__tr("Paste below")),this,SLOT(contextPasteBelow()));
			m_pContextPopup->insertItem(
				*(g_pIconManager->getSmallIcon(KVI_SMALLICON_PASTE)),
				__c2q(__tr("Paste above")),this,SLOT(contextPasteAbove()));
			if(((KviPopupListViewItem *)it)->m_type == KviPopupListViewItem::Menu)
			{
				m_pContextPopup->insertItem(
					*(g_pIconManager->getSmallIcon(KVI_SMALLICON_PASTE)),
					__c2q(__tr("Paste inside")),this,SLOT(contextPasteInside()));
			}
		}

		m_pContextPopup->insertSeparator();

		m_pContextPopup->insertItem(__c2q(__tr("New separator above")),this,SLOT(contextNewSeparatorAbove()));
		m_pContextPopup->insertItem(__c2q(__tr("New separator below")),this,SLOT(contextNewSeparatorBelow()));
		if(bIsMenu)m_pContextPopup->insertItem(__c2q(__tr("New separator inside")),this,SLOT(contextNewSeparatorInside()));

		m_pContextPopup->insertSeparator();

		m_pContextPopup->insertItem(__c2q(__tr("New label above")),this,SLOT(contextNewLabelAbove()));
		m_pContextPopup->insertItem(__c2q(__tr("New label below")),this,SLOT(contextNewLabelBelow()));
		if(bIsMenu)m_pContextPopup->insertItem(__c2q(__tr("New label inside")),this,SLOT(contextNewLabelInside()));

		m_pContextPopup->insertSeparator();

		m_pContextPopup->insertItem(__c2q(__tr("New item above")),this,SLOT(contextNewItemAbove()));
		m_pContextPopup->insertItem(__c2q(__tr("New item below")),this,SLOT(contextNewItemBelow()));
		if(bIsMenu)m_pContextPopup->insertItem(__c2q(__tr("New item inside")),this,SLOT(contextNewItemInside()));

		m_pContextPopup->insertSeparator();

		m_pContextPopup->insertItem(__c2q(__tr("New menu above")),this,SLOT(contextNewMenuAbove()));
		m_pContextPopup->insertItem(__c2q(__tr("New menu below")),this,SLOT(contextNewMenuBelow()));
		if(bIsMenu)m_pContextPopup->insertItem(__c2q(__tr("New menu inside")),this,SLOT(contextNewMenuInside()));

		m_pContextPopup->insertSeparator();

		m_pContextPopup->insertItem(__c2q(__tr("New external menu above")),this,SLOT(contextNewExtMenuAbove()));
		m_pContextPopup->insertItem(__c2q(__tr("New external menu below")),this,SLOT(contextNewExtMenuBelow()));
		if(bIsMenu)m_pContextPopup->insertItem(__c2q(__tr("New external menu inside")),this,SLOT(contextNewExtMenuInside()));


	} else {
		if(m_pClipboard)
		{
			m_pContextPopup->insertItem(
				*(g_pIconManager->getSmallIcon(KVI_SMALLICON_PASTE)),
				__c2q(__tr("Paste")),this,SLOT(contextPasteBelow()));
			m_pContextPopup->insertSeparator();
		}

		m_pContextPopup->insertItem(__c2q(__tr("New separator")),this,SLOT(contextNewSeparatorBelow()));
		m_pContextPopup->insertItem(__c2q(__tr("New label")),this,SLOT(contextNewLabelBelow()));
		m_pContextPopup->insertItem(__c2q(__tr("New item")),this,SLOT(contextNewItemBelow()));
		m_pContextPopup->insertItem(__c2q(__tr("New menu")),this,SLOT(contextNewMenuBelow()));
		m_pContextPopup->insertItem(__c2q(__tr("New external menu")),this,SLOT(contextNewExtMenuBelow()));

	}

	bool bSeparatorInserted = false;

	if(!findPrologue(parentMenu))
	{
		m_pContextPopup->insertSeparator();
		bSeparatorInserted = true;
		m_pContextPopup->insertItem(
				*(g_pIconManager->getSmallIcon(KVI_SMALLICON_PROLOGUE)),
				__c2q(__tr("New prologue at this level")),this,SLOT(contextNewPrologue()));
	}

	if(!findEpilogue(parentMenu))
	{
		if(!bSeparatorInserted)m_pContextPopup->insertSeparator();
		m_pContextPopup->insertItem(
				*(g_pIconManager->getSmallIcon(KVI_SMALLICON_EPILOGUE)),
				__c2q(__tr("New epilogue at this level")),this,SLOT(contextNewEpilogue()));
	}

	m_pContextPopup->popup(pnt);
}

void KviSinglePopupEditor::createNewItemAboveLastSelected(KviPopupListViewItem::Type t)
{
	m_pListView->setCurrentItem(newItemAbove(m_pLastSelectedItem,t));
}

void KviSinglePopupEditor::createNewItemBelowLastSelected(KviPopupListViewItem::Type t)
{
	m_pListView->setCurrentItem(newItemBelow(m_pLastSelectedItem,t));
}

void KviSinglePopupEditor::createNewItemInsideLastSelected(KviPopupListViewItem::Type t)
{
	if(m_pLastSelectedItem)m_pLastSelectedItem->setOpen(true);
	m_pListView->setCurrentItem(newItemInside(m_pLastSelectedItem,t));
}

void KviSinglePopupEditor::contextNewSeparatorAbove()
{
	createNewItemAboveLastSelected(KviPopupListViewItem::Separator);
}

void KviSinglePopupEditor::contextNewSeparatorBelow()
{
	createNewItemBelowLastSelected(KviPopupListViewItem::Separator);
}

void KviSinglePopupEditor::contextNewSeparatorInside()
{
	createNewItemInsideLastSelected(KviPopupListViewItem::Separator);
}

void KviSinglePopupEditor::contextNewItemBelow()
{
	createNewItemBelowLastSelected(KviPopupListViewItem::Item);
}

void KviSinglePopupEditor::contextNewItemAbove()
{
	createNewItemAboveLastSelected(KviPopupListViewItem::Item);
}

void KviSinglePopupEditor::contextNewItemInside()
{
	createNewItemInsideLastSelected(KviPopupListViewItem::Item);
}

void KviSinglePopupEditor::contextNewMenuBelow()
{
	createNewItemBelowLastSelected(KviPopupListViewItem::Menu);
}

void KviSinglePopupEditor::contextNewMenuAbove()
{
	createNewItemAboveLastSelected(KviPopupListViewItem::Menu);
}

void KviSinglePopupEditor::contextNewMenuInside()
{
	createNewItemInsideLastSelected(KviPopupListViewItem::Menu);
}

void KviSinglePopupEditor::contextNewExtMenuBelow()
{
	createNewItemBelowLastSelected(KviPopupListViewItem::ExtMenu);
}

void KviSinglePopupEditor::contextNewExtMenuAbove()
{
	createNewItemAboveLastSelected(KviPopupListViewItem::ExtMenu);
}
void KviSinglePopupEditor::contextNewExtMenuInside()
{
	createNewItemInsideLastSelected(KviPopupListViewItem::ExtMenu);
}


void KviSinglePopupEditor::contextNewLabelBelow()
{
	createNewItemBelowLastSelected(KviPopupListViewItem::Label);
}

void KviSinglePopupEditor::contextNewLabelAbove()
{
	createNewItemAboveLastSelected(KviPopupListViewItem::Label);
}

void KviSinglePopupEditor::contextNewLabelInside()
{
	createNewItemInsideLastSelected(KviPopupListViewItem::Label);
}

KviPopupListViewItem * KviSinglePopupEditor::newItem(KviPopupListViewItem * par,KviPopupListViewItem * after,KviPopupListViewItem::Type t)
{
	if(par)return new KviPopupListViewItem(par,after,t);
	return new KviPopupListViewItem(m_pListView,after,t);
}

KviPopupListViewItem * KviSinglePopupEditor::newItemBelow(KviPopupListViewItem * it,KviPopupListViewItem::Type t)
{
	if(!it)return newItem(0,0,t);
	return newItem((KviPopupListViewItem *)it->parent(),it,t);
}

KviPopupListViewItem * KviSinglePopupEditor::newItemAbove(KviPopupListViewItem * it,KviPopupListViewItem::Type t)
{
	if(!it)return newItem(0,0,t);
	return newItem((KviPopupListViewItem *)it->parent(),(KviPopupListViewItem *)it->itemAbove(),t);
}

KviPopupListViewItem * KviSinglePopupEditor::newItemInside(KviPopupListViewItem * it,KviPopupListViewItem::Type t)
{
	if(it)
	{
		if(it->m_type != KviPopupListViewItem::Menu)
		{
			return newItemBelow(it,t);
		}
	}
	return newItem(it,it,t);
}

void KviSinglePopupEditor::contextNewPrologue()
{
	KviPopupListViewItem * it = m_pLastSelectedItem ? (KviPopupListViewItem *)m_pLastSelectedItem->parent() : 0;
	if(!findPrologue(it))
	{
		m_pListView->setCurrentItem(newItem(it,it,KviPopupListViewItem::Prologue));
	}
}

void KviSinglePopupEditor::contextNewEpilogue()
{
	KviPopupListViewItem * it = m_pLastSelectedItem ? (KviPopupListViewItem *)m_pLastSelectedItem->parent() : 0;
	if(!findEpilogue(it))
	{
		KviPopupListViewItem * after = it ? (KviPopupListViewItem *)it->firstChild() : (KviPopupListViewItem *)m_pListView->firstChild();
		if(after)
		{
			while(after->nextSibling())after = (KviPopupListViewItem *)after->nextSibling();
		} else {
			after = it;
		}
		m_pListView->setCurrentItem(newItem(it,after,KviPopupListViewItem::Epilogue));
	}
}

void KviSinglePopupEditor::contextCopy()
{
	if(!m_pLastSelectedItem)return;
	if(m_pClipboard)delete m_pClipboard;
	m_pClipboard = new KviPopupMenu("clipboard");
	addItemToMenu(m_pClipboard,m_pLastSelectedItem);
}

void KviSinglePopupEditor::contextCut()
{
	if(!m_pLastSelectedItem)return;
	contextCopy();

	KviPopupListViewItem * it = m_pLastSelectedItem;
	m_pLastSelectedItem = 0;
	delete it;
	if(!m_pLastSelectedItem)selectionChanged(0);
}


void KviSinglePopupEditor::contextPasteBelow()
{
	if(!m_pClipboard)return;
	KviPopupListViewItem * par = m_pLastSelectedItem ? (KviPopupListViewItem *)m_pLastSelectedItem->parent() : 0;
	populateMenu(m_pClipboard,par,m_pLastSelectedItem);
}

void KviSinglePopupEditor::contextPasteAbove()
{
	if(!m_pClipboard)return;
	KviPopupListViewItem * par = m_pLastSelectedItem ? (KviPopupListViewItem *)m_pLastSelectedItem->parent() : 0;
	KviPopupListViewItem * above = m_pLastSelectedItem ? (KviPopupListViewItem *)m_pLastSelectedItem->itemAbove() : 0;
	populateMenu(m_pClipboard,par,above);

}

void KviSinglePopupEditor::contextPasteInside()
{
	if(!m_pClipboard)return;
	if(m_pLastSelectedItem)
	{
		if(m_pLastSelectedItem->m_type != KviPopupListViewItem::Menu)
		{
			contextPasteBelow();
			return;
		}
		m_pLastSelectedItem->setOpen(true);
	}
	populateMenu(m_pClipboard,m_pLastSelectedItem,0);
}

void KviSinglePopupEditor::saveLastSelectedItem()
{
	if(!m_pLastSelectedItem)return;

	KviStr tmp;

	switch(m_pLastSelectedItem->m_type)
	{
		case KviPopupListViewItem::Prologue:
		case KviPopupListViewItem::Epilogue:
		case KviPopupListViewItem::Item:
		{
			QCString tmpx;
			m_pEditor->getText(tmpx);
			m_pLastSelectedItem->setCode(tmpx.data());
		}
		break;
	}

	switch(m_pLastSelectedItem->m_type)
	{
		case KviPopupListViewItem::Menu:
		case KviPopupListViewItem::ExtMenu:
		case KviPopupListViewItem::Item:
			tmp = m_pIconEditor->text();
			m_pLastSelectedItem->setIcon(tmp.ptr());
		break;
	}

	switch(m_pLastSelectedItem->m_type)
	{
		case KviPopupListViewItem::Menu:
		case KviPopupListViewItem::Item:
		case KviPopupListViewItem::Label:
		case KviPopupListViewItem::ExtMenu:
			tmp = m_pTextEditor->text();
			m_pLastSelectedItem->setItemText(tmp.ptr());
			tmp = m_pConditionEditor->text();
			m_pLastSelectedItem->setCondition(tmp.ptr());
		break;
	}

	if(m_pLastSelectedItem->m_type == KviPopupListViewItem::ExtMenu)
	{
		tmp = m_pExtNameEditor->text();
		m_pLastSelectedItem->setCode(tmp.ptr());
	}
}

void KviSinglePopupEditor::addItemToMenu(KviPopupMenu * p,KviPopupListViewItem *it)
{
	switch(it->m_type)
	{
		case KviPopupListViewItem::Prologue:
			it->m_szCode.stripWhiteSpace();
			p->setPrologue(it->m_szCode.ptr());
		break;
		case KviPopupListViewItem::Epilogue:
			it->m_szCode.stripWhiteSpace();
			p->setEpilogue(it->m_szCode.ptr());
		break;
		case KviPopupListViewItem::Separator:
			p->addItem(new KviPopupMenuItem(KviPopupMenuItem::Separator));
		break;
		case KviPopupListViewItem::Label:
			it->m_szText.stripWhiteSpace();
			it->m_szCondition.stripWhiteSpace();
			p->addItem(new KviPopupMenuItem(KviPopupMenuItem::Label,it->m_szText.ptr(),0,it->m_szCondition.ptr()));
		break;
		case KviPopupListViewItem::Item:
			it->m_szText.stripWhiteSpace();
			it->m_szIcon.stripWhiteSpace();
			it->m_szCondition.stripWhiteSpace();
			it->m_szCode.stripWhiteSpace();
			p->addItem(new KviPopupMenuItem(KviPopupMenuItem::Item,it->m_szText.ptr(),it->m_szIcon.ptr(),it->m_szCondition.ptr(),it->m_szCode.ptr()));
		break;
		case KviPopupListViewItem::ExtMenu:
			it->m_szText.stripWhiteSpace();
			it->m_szIcon.stripWhiteSpace();
			it->m_szCondition.stripWhiteSpace();
			it->m_szCode.stripWhiteSpace();
			p->addItem(new KviPopupMenuItem(KviPopupMenuItem::ExtMenu,it->m_szText.ptr(),it->m_szIcon.ptr(),it->m_szCondition.ptr(),it->m_szCode.ptr()));
		break;
		case KviPopupListViewItem::Menu:
		{
			KviPopupMenu * menu = new KviPopupMenu("submenu");
			for(KviPopupListViewItem * item = (KviPopupListViewItem *)it->firstChild();item;item = (KviPopupListViewItem *)item->nextSibling())
			{
				addItemToMenu(menu,item);
			}
			it->m_szText.stripWhiteSpace();
			it->m_szIcon.stripWhiteSpace();
			it->m_szCondition.stripWhiteSpace();
			p->addItem(new KviPopupMenuItem(KviPopupMenuItem::Menu,it->m_szText.ptr(),it->m_szIcon.ptr(),it->m_szCondition.ptr(),0,menu));
		}
		break;
	}
}

KviPopupMenu * KviSinglePopupEditor::getMenu()
{
	saveLastSelectedItem();

	KviStr tmp = m_pNameEditor->text();
	tmp.stripWhiteSpace();

	KviPopupMenu * p = new KviPopupMenu(tmp.ptr());
	for(KviPopupListViewItem * it = (KviPopupListViewItem *)m_pListView->firstChild();it;it = (KviPopupListViewItem *)it->nextSibling())
	{
		addItemToMenu(p,it);
	}
	return p;
}

void KviSinglePopupEditor::selectionChanged(QListViewItem * it)
{
	saveLastSelectedItem();

	bool bEditorEnabled = false;
	bool bIconEditorEnabled = false;
	bool bConditionEditorEnabled = false;
	bool bTextEditorEnabled = false;
	bool bNameEditorEnabled = false;

	if(it)
	{
		switch(((KviPopupListViewItem *)it)->m_type)
		{
			case KviPopupListViewItem::Prologue:
			case KviPopupListViewItem::Epilogue:
			case KviPopupListViewItem::Item:
				m_pEditor->setText(((KviPopupListViewItem *)it)->m_szCode.ptr());
				bEditorEnabled = true;
			break;
		}

		switch(((KviPopupListViewItem *)it)->m_type)
		{
			case KviPopupListViewItem::Menu:
			case KviPopupListViewItem::Item:
			case KviPopupListViewItem::ExtMenu:
				m_pIconEditor->setText(((KviPopupListViewItem *)it)->m_szIcon.ptr());
				bIconEditorEnabled = true;
			break;
		}

		switch(((KviPopupListViewItem *)it)->m_type)
		{
			case KviPopupListViewItem::Menu:
			case KviPopupListViewItem::Item:
			case KviPopupListViewItem::Label:
			case KviPopupListViewItem::ExtMenu:
				m_pConditionEditor->setText(((KviPopupListViewItem *)it)->m_szCondition.ptr());
				bConditionEditorEnabled = true;
				m_pTextEditor->setText(((KviPopupListViewItem *)it)->m_szText.ptr());
				bTextEditorEnabled = true;
			break;
		}

		if(((KviPopupListViewItem *)it)->m_type == KviPopupListViewItem::ExtMenu)
		{
			m_pExtNameEditor->setText(((KviPopupListViewItem *)it)->m_szCode.ptr());
			bNameEditorEnabled =true;
		}
	}

	m_pLastSelectedItem = (KviPopupListViewItem *)it;

	if(!bEditorEnabled)m_pEditor->setText("");
	m_pEditor->setEnabled(bEditorEnabled);
	if(!bIconEditorEnabled)m_pIconEditor->setText("");
	m_pIconEditor->setEnabled(bIconEditorEnabled);
	if(!bConditionEditorEnabled)m_pConditionEditor->setText("");
	m_pConditionEditor->setEnabled(bConditionEditorEnabled);
	if(!bTextEditorEnabled)m_pTextEditor->setText("");
	m_pTextEditor->setEnabled(bTextEditorEnabled);
	m_pExtNameEditor->setEnabled(bNameEditorEnabled);
	if(!bNameEditorEnabled)m_pExtNameEditor->setText("");
}

void KviSinglePopupEditor::populateMenu(KviPopupMenu * pop,KviPopupListViewItem * par,KviPopupListViewItem * theItem)
{
	if(!pop)return;

	if(pop->hasPrologue())
	{
		if(par)theItem = new KviPopupListViewItem(par,theItem,KviPopupListViewItem::Prologue);
		else theItem = new KviPopupListViewItem(m_pListView,theItem,KviPopupListViewItem::Prologue);
		theItem->setCode(pop->prologue());
	}

	for(KviPopupMenuItem * item = pop->m_pItemList->first();item;item = pop->m_pItemList->next())
	{
		switch(item->type())
		{
			case KviPopupMenuItem::Item:
				if(par)theItem = new KviPopupListViewItem(par,theItem,KviPopupListViewItem::Item);
				else theItem = new KviPopupListViewItem(m_pListView,theItem,KviPopupListViewItem::Item);
				theItem->setIcon(item->icon());
				theItem->setItemText(item->text());
				theItem->setCondition(item->expr());
				theItem->setCode(item->code());
			break;
			case KviPopupMenuItem::ExtMenu:
				if(par)theItem = new KviPopupListViewItem(par,theItem,KviPopupListViewItem::ExtMenu);
				else theItem = new KviPopupListViewItem(m_pListView,theItem,KviPopupListViewItem::ExtMenu);
				theItem->setIcon(item->icon());
				theItem->setItemText(item->text());
				theItem->setCondition(item->expr());
				theItem->setCode(item->code());
			break;
			case KviPopupMenuItem::Label:
				if(par)theItem = new KviPopupListViewItem(par,theItem,KviPopupListViewItem::Label);
				else theItem = new KviPopupListViewItem(m_pListView,theItem,KviPopupListViewItem::Label);
				theItem->setItemText(item->text());
				theItem->setCondition(item->expr());
			break;
			case KviPopupMenuItem::Separator:
				if(par)theItem = new KviPopupListViewItem(par,theItem,KviPopupListViewItem::Separator);
				else theItem = new KviPopupListViewItem(m_pListView,theItem,KviPopupListViewItem::Separator);
			break;
			case KviPopupMenuItem::Menu:
				if(par)theItem = new KviPopupListViewItem(par,theItem,KviPopupListViewItem::Menu);
				else theItem = new KviPopupListViewItem(m_pListView,theItem,KviPopupListViewItem::Menu);
				theItem->setIcon(item->icon());
				theItem->setItemText(item->text());
				theItem->setCondition(item->expr());
				populateMenu(item->menu(),theItem,0);
			break;
		}
	}

	if(pop->hasEpilogue())
	{
		if(par)theItem = new KviPopupListViewItem(par,theItem,KviPopupListViewItem::Epilogue);
		else theItem = new KviPopupListViewItem(m_pListView,theItem,KviPopupListViewItem::Epilogue);
		theItem->setCode(pop->epilogue());
	}

}



void KviSinglePopupEditor::edit(KviMenuListViewItem * it)
{
	saveLastSelectedItem();

	m_pLastSelectedItem = 0;

	m_pListView->clear();

	selectionChanged(0);

	if(it)
	{
		m_pNameEditor->setText(it->m_pPopup->name());

		populateMenu(it->m_pPopup,0,0);

	} else {
		m_pIconEditor->setText("");
		m_pIconEditor->setEnabled(false);
		m_pConditionEditor->setText("");
		m_pConditionEditor->setEnabled(false);
		m_pTextEditor->setText("");
		m_pTextEditor->setEnabled(false);
		m_pEditor->setText("");
		m_pEditor->setEnabled(false);
		m_pNameEditor->setText("");
		m_pExtNameEditor->setText("");
		m_pExtNameEditor->setEnabled(false);
	}
	m_pListView->setEnabled(it);
	m_pNameEditor->setEnabled(it);
	m_pMenuButton->setEnabled(it);

}

















KviMenuListViewItem::KviMenuListViewItem(QListView * par,KviPopupMenu * popup)
: QListViewItem(par)
{
	setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_POPUP)));
	setText(0,popup->name());
	m_pPopup = popup;
}

KviMenuListViewItem::~KviMenuListViewItem()
{
	delete m_pPopup;
}

void KviMenuListViewItem::replacePopup(KviPopupMenu * popup)
{
	delete m_pPopup;
	m_pPopup = popup;
}


KviPopupEditor::KviPopupEditor(QWidget * par)
: QWidget(par)
{
	QGridLayout * l = new QGridLayout(this,1,1,0,2);

	QSplitter * spl = new QSplitter(QSplitter::Horizontal,this);

	l->addWidget(spl,0,0);

	m_pListView = new QListView(spl);
	m_pListView->addColumn(__c2q(__tr("Popup")));
	m_pListView->setMultiSelection(false);
	m_pListView->setShowSortIndicator(true);

	m_pEditor = new KviSinglePopupEditor(spl);

	m_bOneTimeSetupDone = false;
	m_pLastEditedItem = 0;

	m_pContextPopup = new QPopupMenu(this);

	selectionChanged(0);
}

KviPopupEditor::~KviPopupEditor()
{
}

void KviPopupEditor::oneTimeSetup()
{
	if(m_bOneTimeSetupDone)return;
	m_bOneTimeSetupDone = true;

	const QAsciiDict<KviPopupMenu> * a = g_pPopupManager->popupDict();
	if(!a)return;

	QAsciiDictIterator<KviPopupMenu> it(*a);

	KviMenuListViewItem * item;

	while(it.current())
	{
		KviPopupMenu * popup = it.current();
		KviPopupMenu * copy = new KviPopupMenu(popup->name());
		copy->copyFrom(popup);
		item = new KviMenuListViewItem(m_pListView,copy);
		++it;
	}

	connect(m_pListView,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(selectionChanged(QListViewItem *)));
	connect(m_pListView,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),
		this,SLOT(itemPressed(QListViewItem *,const QPoint &,int)));

}

void KviPopupEditor::itemPressed(QListViewItem *it,const QPoint &pnt,int col)
{
	__range_valid(m_bOneTimeSetupDone);
	m_pContextPopup->clear();

	if(it)
	{
		m_pContextPopup->insertItem(
				*(g_pIconManager->getSmallIcon(KVI_SMALLICON_QUIT)),
				__c2q(__tr("Remove this popup")),
				this,SLOT(removeCurrentPopup()));
		m_pContextPopup->insertSeparator();
	}

	m_pContextPopup->insertItem(
		*(g_pIconManager->getSmallIcon(KVI_SMALLICON_POPUP)),
		__c2q(__tr("Add new popup")),
		this,SLOT(newPopup()));

	m_pContextPopup->popup(pnt);
}

void KviPopupEditor::removeCurrentPopup()
{
	if(m_pLastEditedItem)
	{
		KviMenuListViewItem * it = m_pLastEditedItem;
		m_pLastEditedItem = 0;
		delete it;
		if(!m_pLastEditedItem)selectionChanged(0);
	}
}

void KviPopupEditor::newPopup()
{
	KviStr newName;
	getUniquePopupName(0,newName);
	KviMenuListViewItem * it = new KviMenuListViewItem(m_pListView,new KviPopupMenu(newName));
	m_pListView->setCurrentItem(it);
}

void KviPopupEditor::saveLastEditedItem()
{
	if(m_pLastEditedItem == 0)return;

	KviPopupMenu * m = m_pEditor->getMenu();
	KviStr tmp = m->name();
	KviStr old = m_pLastEditedItem->m_pPopup->name();
	if(!kvi_strEqualCI(old.ptr(),tmp.ptr()))
	{
		getUniquePopupName(m_pLastEditedItem,tmp);
		m->setName(tmp.ptr());
	}

	m_pLastEditedItem->replacePopup(m);
	m_pLastEditedItem->setText(0,m->name());
}

void KviPopupEditor::selectionChanged(QListViewItem *it)
{
	saveLastEditedItem();

	m_pLastEditedItem = (KviMenuListViewItem *)it;

	m_pEditor->edit(m_pLastEditedItem);
}

void KviPopupEditor::showEvent(QShowEvent *e)
{
	oneTimeSetup();
	QWidget::showEvent(e);
}

void KviPopupEditor::commit()
{
	if(!m_bOneTimeSetupDone)return;

	saveLastEditedItem();

	//g_pPopupManager->clear();

	KviMenuListViewItem * it = (KviMenuListViewItem *)m_pListView->firstChild();

	// Copy the original popup dict
	QAsciiDict<KviPopupMenu> copy(*(g_pPopupManager->popupDict()));
	copy.setAutoDelete(false);

	while(it)
	{
		KviPopupMenu * p = g_pPopupManager->getPopup(it->m_pPopup->name());
		p->doClear();
		p->copyFrom(it->m_pPopup);
		it = (KviMenuListViewItem *)it->nextSibling();
		// remove it from the original copy
		copy.remove(p->name());
	}

	// the remaining elements in the copy need to be removed from
	// the "new" dictionary (they are no longer used)
	QAsciiDictIterator<KviPopupMenu> iter(copy);

	while(iter.current())
	{
		g_pPopupManager->removePopup(iter.currentKey());
		++iter;
	}
	

	g_pApp->savePopups();
}


void KviPopupEditor::getUniquePopupName(KviMenuListViewItem *item,KviStr &buffer)
{
	__range_valid(m_bOneTimeSetupDone);

	if(buffer.isEmpty())buffer = __tr("unnamed");
	KviStr newName = buffer;

	bool bFound = true;
	int idx = 1;

	while(bFound)
	{
		bFound = false;

		for(KviMenuListViewItem * ch = (KviMenuListViewItem *)(m_pListView->firstChild());ch;ch = (KviMenuListViewItem *)ch->nextSibling())
		{
			if(kvi_strEqualCI(newName.ptr(),ch->m_pPopup->name()) && (ch != item))
			{
				bFound = true;
				newName.sprintf("%s.%d",buffer.ptr(),idx);
				idx++;
				break;
			}
		}

	}

	buffer = newName;
}

#include "m_popupeditor.moc"
