//
//   File : eventeditor.cpp
//   Creation date : Sun Oct 10 2000 23:25:10 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "eventeditor.h"

#include <qsplitter.h>
#include <qlayout.h>
#include <qvbox.h>
#include <qpopupmenu.h>
#include <qtooltip.h>

#include "kvi_scripteditor.h"

#include "kvi_debug.h"
#include "kvi_event.h"
#include "kvi_locale.h"
#include "kvi_iconmanager.h"
#include "kvi_app.h"
#include "kvi_cmdformatter.h"
#include "kvi_eventhandler.h"

const QPixmap * KviEventListViewItem::pixmap(int col) const
{
	return g_pIconManager->getSmallIcon(firstChild() ? KVI_SMALLICON_EVENT : KVI_SMALLICON_EVENTNOHANDLERS);
}

const QPixmap * KviEventHandlerListViewItem::pixmap(int col) const
{
	return g_pIconManager->getSmallIcon(m_bEnabled ? KVI_SMALLICON_HANDLER : KVI_SMALLICON_HANDLERDISABLED);
}



KviEventEditor::KviEventEditor(QWidget * par)
: QWidget(par,"event_editor")
{
	QGridLayout * l = new QGridLayout(this,1,1,2,2);

	QSplitter * spl = new QSplitter(QSplitter::Horizontal,this);

	l->addWidget(spl,0,0);

	m_pListView = new QListView(spl);
	m_pListView->addColumn(__c2q(__tr("Event")));
	m_pListView->setMultiSelection(false);
	m_pListView->setShowSortIndicator(true);
	m_pListView->setRootIsDecorated(true);

	QVBox * box = new QVBox(spl);
	m_pNameEditor = new QLineEdit(box);
	QToolTip::add(m_pNameEditor,__c2q(__tr("Handler name")));
	m_pEditor = kvi_createScriptEditor(box);

	m_bOneTimeSetupDone = false;
	m_pLastEditedItem = 0;
}

KviEventEditor::~KviEventEditor()
{
}

void KviEventEditor::oneTimeSetup()
{
	//debug("ONE TIME SETUP!");
	if(m_bOneTimeSetupDone)return;
	m_bOneTimeSetupDone = true;

	//debug("YES");

	KviEventListViewItem * it;
	KviEventHandlerListViewItem * ch;

	for(int i = 0;i < KVI_NUM_SCRIPT_EVENTS;i++)
	{
		//debug("SETTING UP %d",i);
		KviEventHandlerList * l = g_pEventManager->handlers(i);
		it = new KviEventListViewItem(m_pListView,g_pEventManager->eventName(i),g_pEventManager->eventParams(i));
		if(l)
		{
			//debug("THERE ARE HANDLERS!");
			for(KviEventHandler * s = l->first();s;s = l->next())
			{
				//debug("Handler");
				if(s->type() == KviEventHandler::Script)
				{
					//debug("Its a script ! Handler %s",((KviScriptEventHandler *)s)->name());
					ch = new KviEventHandlerListViewItem(it,((KviScriptEventHandler *)s)->name(),
					((KviScriptEventHandler *)s)->code(),((KviScriptEventHandler *)s)->enabled());
				}
			}
			it->setOpen(true);
		}
	}


	m_pContextPopup = new QPopupMenu(this);

	connect(m_pListView,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(selectionChanged(QListViewItem *)));
	connect(m_pListView,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),
		this,SLOT(itemPressed(QListViewItem *,const QPoint &,int)));
}

void KviEventEditor::itemPressed(QListViewItem *it,const QPoint &pnt,int col)
{
	__range_valid(m_bOneTimeSetupDone);

	if(it)
	{
		m_pContextPopup->clear();
		if(it->parent())
		{
			if(((KviEventHandlerListViewItem *)it)->m_bEnabled)
			{
				m_pContextPopup->insertItem(
					*(g_pIconManager->getSmallIcon(KVI_SMALLICON_HANDLERDISABLED)),
					__c2q(__tr("Disable this handler")),
					this,SLOT(disableCurrentHandler()));
			} else {
				m_pContextPopup->insertItem(
					*(g_pIconManager->getSmallIcon(KVI_SMALLICON_HANDLER)),
					__c2q(__tr("Enable this handler")),
					this,SLOT(enableCurrentHandler()));
			}
			m_pContextPopup->insertItem(
					*(g_pIconManager->getSmallIcon(KVI_SMALLICON_QUIT)),
					__c2q(__tr("Remove this handler")),
					this,SLOT(removeCurrentHandler()));
		} else {
			m_pContextPopup->insertItem(
				*(g_pIconManager->getSmallIcon(KVI_SMALLICON_HANDLER)),
				__c2q(__tr("Add handler")),
				this,SLOT(addHandlerForCurrentEvent()));
		}

		m_pContextPopup->popup(pnt);
	}
}

void KviEventEditor::getUniqueHandlerName(KviEventListViewItem *it,KviStr &buffer)
{
	__range_valid(m_bOneTimeSetupDone);

	KviStr newName = buffer;
	if(newName.isEmpty())newName = __c2q(__tr("unnamed"));

	bool bFound = true;
	int idx = 1;

	while(bFound)
	{
		bFound = false;

		for(KviEventHandlerListViewItem * ch = (KviEventHandlerListViewItem *)(it->firstChild());ch;ch = (KviEventHandlerListViewItem *)ch->nextSibling())
		{
			if(kvi_strEqualCI(newName.ptr(),ch->m_szName.ptr()))
			{
				bFound = true;
				newName.sprintf("%s_%d",buffer.ptr(),idx);
				idx++;
				break;
			}
		}

	}

	buffer = newName;
}

void KviEventEditor::addHandlerForCurrentEvent()
{
	__range_valid(m_pOneTimeSetupDone);

	QListViewItem * it = m_pListView->selectedItem();
	if(it)
	{
		if(it->parent() == 0)
		{
			KviStr buffer = __c2q(__tr("default"));
			getUniqueHandlerName((KviEventListViewItem *)it,buffer);
			QListViewItem * ch = new KviEventHandlerListViewItem(it,buffer.ptr(),"",true);
			it->setOpen(true);
			m_pListView->setSelected(ch,true);
		}
	}
}

void KviEventEditor::removeCurrentHandler()
{
	__range_valid(m_pOneTimeSetupDone);
	if(m_pLastEditedItem)
	{
		QListViewItem * it = m_pLastEditedItem;
		m_pLastEditedItem = 0;
		delete it;
		m_pEditor->setEnabled(false);
		m_pNameEditor->setEnabled(false);
	}
}

void KviEventEditor::enableCurrentHandler()
{
	__range_valid(m_pOneTimeSetupDone);
	if(m_pLastEditedItem)
	{
		m_pLastEditedItem->m_bEnabled = true;
		m_pListView->repaintItem(m_pLastEditedItem);
		selectionChanged(m_pLastEditedItem);
	}
}

void KviEventEditor::disableCurrentHandler()
{
	__range_valid(m_pOneTimeSetupDone);
	if(m_pLastEditedItem)
	{
		m_pLastEditedItem->m_bEnabled = false;
		m_pListView->repaintItem(m_pLastEditedItem);
		selectionChanged(m_pLastEditedItem);
	}
}

void KviEventEditor::commit()
{
	if(!m_bOneTimeSetupDone)return; // nothing to commit

	saveLastEditedItem();
	g_pEventManager->clearScriptHandlers();
	for(QListViewItem * it = m_pListView->firstChild();it;it = it->nextSibling())
	{
		if(it->firstChild())
		{
			for(QListViewItem * ch = it->firstChild();ch;ch = ch->nextSibling())
			{
				g_pEventManager->registerScriptHandler(
					((KviEventListViewItem *)it)->m_szName.ptr(),
					((KviEventHandlerListViewItem *)ch)->m_szName.ptr(),
					((KviEventHandlerListViewItem *)ch)->m_szBuffer.ptr(),
					((KviEventHandlerListViewItem *)ch)->m_bEnabled
				);
			}
		}
	}

	g_pApp->saveEvents(); // otherwise this page has never been shown
}

void KviEventEditor::saveLastEditedItem()
{
	__range_valid(m_bOneTimeSetupDone);
	if(!m_pLastEditedItem)return;

	KviStr buffer = m_pNameEditor->text();
	if(!kvi_strEqualCI(buffer.ptr(),m_pLastEditedItem->m_szName.ptr()))
	{
		getUniqueHandlerName((KviEventListViewItem *)(m_pLastEditedItem->parent()),buffer);
	}

	m_pLastEditedItem->m_szName = buffer.ptr();
	QCString tmp;
	m_pEditor->getText(tmp);


	m_pLastEditedItem->m_szBuffer = tmp.data();
}

void KviEventEditor::selectionChanged(QListViewItem * it)
{
	__range_valid(m_bOneTimeSetupDone);
	saveLastEditedItem();
	if(it->parent())
	{
		m_pLastEditedItem = (KviEventHandlerListViewItem *)it;
		m_pNameEditor->setEnabled(true);
		m_pNameEditor->setText(it->text(0));
		m_pEditor->setEnabled(true);
		m_pEditor->setText(((KviEventHandlerListViewItem *)it)->m_szBuffer.ptr());
	} else {
		m_pLastEditedItem = 0;
		m_pNameEditor->setEnabled(false);
		m_pNameEditor->setText("");
		m_pEditor->setEnabled(false);
		KviStr parms = ((KviEventListViewItem *)it)->m_szParams;
		if(parms.isEmpty())parms = __tr("none");
		KviCommandFormatter::indent(parms);
		KviCommandFormatter::indent(parms);
		KviStr tmp(KviStr::Format,__tr("\n\n	Event:\n		%s\n\n	Parameters:\n%s"),
			((KviEventListViewItem *)it)->m_szName.ptr(),parms.ptr());
		m_pEditor->setText(tmp.ptr());
	}
}

void KviEventEditor::showEvent(QShowEvent *e)
{
	oneTimeSetup();
	QWidget::showEvent(e);
}

#include "m_eventeditor.moc"
