//
//   File : libkvirijndael.cpp
//   Creation date : Sat Now 4 2000 15:33:12 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Till Bush (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "libkvirijndael.h"

#include "rijndael.h"

#include "kvi_module.h"
#include "kvi_debug.h"
#include "kvi_locale.h"


//#warning "Other engines: mircStrip koi2win colorizer lamerizer etc.."

/*
	@doc: rijndael
	@type:
		module
	@short:
		The Rijndael cryptographic engines
	@title:
		The rijndael module
	@body:
		The rijndael module exports six [doc:crypt_engines]cryptographic engines[/doc] based
		on the Advanced Encryptiong Standard algorithm called Rijndael. Rijndael was
		originally written by Joan Daemen and Vincent Rijmen. The original Rijndael
		description is available at http://www.esat.kuleuven.ac.be/~rijmen/rijndael/.[br]
		It is a private key block cipher that has been designed to replace
		the widely used DES, and it should provide at leas a decent security agains
		common attacks. Theoretically the best attack that one can perform on this cipher
		is the "brute force" attack that requires a really massive parallel computation:
		actually out of the possibilities of a common "hacker".[br]
		My implementation allows the usage of 128, 192 and 256 bit keys
		on 128 bit data blocks. The encrypted binary data buffer is then converted
		into an ascii-string by using the base64 conversion or hex-digit-string rappresentation.
		The six engines are the six possible combinations of the key lengths and ascii-string
		conversions.
*/



#ifdef COMPILE_CRYPT_SUPPORT
	
	#include "kvi_memmove.h"
	#include "kvi_malloc.h"

	#include "kvi_list.h"

	static KviPtrList<KviCryptEngine> * g_pEngineList = 0;

	KviRijndaelEngine::KviRijndaelEngine()
	: KviCryptEngine()
	{
		g_pEngineList->append(this);
		m_pEncryptCipher = 0;
		m_pDecryptCipher = 0;
	}

	KviRijndaelEngine::~KviRijndaelEngine()
	{
		g_pEngineList->removeRef(this);
		if(m_pEncryptCipher)delete m_pEncryptCipher;
		if(m_pDecryptCipher)delete m_pDecryptCipher;
	}

	bool KviRijndaelEngine::init(const char *encKey,int encKeyLen,const char *decKey,int decKeyLen)
	{
		if(m_pEncryptCipher)
		{
			delete m_pEncryptCipher;
			m_pEncryptCipher = 0;
		}
		if(m_pDecryptCipher)
		{
			delete m_pDecryptCipher;
			m_pDecryptCipher = 0;
		}

		if(encKey && (encKeyLen > 0))
		{
			if(!(decKey && (decKeyLen > 0)))
			{
				decKey = encKey;
				decKeyLen = encKeyLen;
			} // else all 
		} else {
			// no encrypt key specified...
			if(decKey && decKeyLen)
			{
				encKey = decKey;
				encKeyLen = decKeyLen;
			} else {
				// both keys missing
				setLastError(__tr("Missing both encrypt and decrypt key: at least one is needed"));
				return false;
			}
		}

		int defLen = getKeyLen();

		char * encryptKey = (char *)kvi_malloc(defLen);
		char * decryptKey = (char *)kvi_malloc(defLen);

		if(encKeyLen > defLen)encKeyLen = defLen;
		kvi_memmove(encryptKey,encKey,encKeyLen);
		if(encKeyLen < defLen)kvi_memset(encryptKey + encKeyLen,'0',defLen - encKeyLen);

		if(decKeyLen > defLen)decKeyLen = defLen;
		kvi_memmove(decryptKey,decKey,decKeyLen);
		if(decKeyLen < defLen)kvi_memset(decryptKey + decKeyLen,'0',defLen - decKeyLen);

		m_pEncryptCipher = new Rijndael();
		int retVal = m_pEncryptCipher->init(Rijndael::CBC,Rijndael::Encrypt,(unsigned char *)encryptKey,getKeyLenId());
		kvi_free(encryptKey);
		if(retVal != RIJNDAEL_SUCCESS)
		{
			kvi_free(decryptKey);
			delete m_pEncryptCipher;
			m_pEncryptCipher = 0;
			setLastErrorFromRijndaelErrorCode(retVal);
			return false;
		}

		m_pDecryptCipher = new Rijndael();
		retVal = m_pDecryptCipher->init(Rijndael::CBC,Rijndael::Decrypt,(unsigned char *)decryptKey,getKeyLenId());
		kvi_free(decryptKey);
		if(retVal != RIJNDAEL_SUCCESS)
		{
			delete m_pEncryptCipher;
			m_pEncryptCipher = 0;
			delete m_pDecryptCipher;
			m_pDecryptCipher = 0;
			setLastErrorFromRijndaelErrorCode(retVal);
			return false;
		}

		return true;
	}

	void KviRijndaelEngine::setLastErrorFromRijndaelErrorCode(int errCode)
	{
		switch(errCode)
		{
			case RIJNDAEL_SUCCESS: setLastError(__tr("Error 0: Success ?")); break;
			case RIJNDAEL_UNSUPPORTED_MODE: setLastError(__tr("Unsupported crypt mode")); break;
			case RIJNDAEL_UNSUPPORTED_DIRECTION: setLastError(__tr("Unsupported direction")); break;
			case RIJNDAEL_UNSUPPORTED_KEY_LENGTH: setLastError(__tr("Unsupported key length")); break;
			case RIJNDAEL_BAD_KEY: setLastError(__tr("Bad key data")); break;
			case RIJNDAEL_NOT_INITIALIZED: setLastError(__tr("Engine not initialized")); break;
			case RIJNDAEL_BAD_DIRECTION: setLastError(__tr("Invalid direction for this engine")); break;
			case RIJNDAEL_CORRUPTED_DATA: setLastError(__tr("Corrupted message data or invalid decrypt key")); break;
			default: setLastError(__tr("Unknown error")); break;
		}
	}

	bool KviRijndaelEngine::encrypt(const char * plainText,KviStr &outBuffer)
	{
		if(!m_pEncryptCipher)
		{
			setLastError(__tr("Ops...encrypt cipher not initialized"));
			return false;
		}
		int len = kvi_strLen(plainText);
		char * buf = (char *)kvi_malloc(len + 16);
		
		int retVal = m_pEncryptCipher->padEncrypt((const unsigned char *)plainText,len,(unsigned char *)buf);
		if(retVal < 0)
		{
			kvi_free(buf);
			setLastErrorFromRijndaelErrorCode(retVal);
			return false;
		}

		if(!binaryToAscii(buf,retVal,outBuffer))
		{
			kvi_free(buf);
			return false;
		}
		kvi_free(buf);

		if(outBuffer.len() > maxEncryptLen())
		{
			if(maxEncryptLen() > 0)
			{
				setLastError(__tr("Data buffer too long"));
				return false;
			}
		}
		return true;
	}

	bool KviRijndaelEngine::decrypt(const char * inBuffer,KviStr &plainText)
	{
		if(!m_pDecryptCipher)
		{
			setLastError(__tr("Ops...decrypt cipher not initialized"));
			return false;
		}

		int len;
		char * binary;

		if(!asciiToBinary(inBuffer,&len,&binary))return false;

		char * buf = (char *)kvi_malloc(len + 1);
		
		int retVal = m_pDecryptCipher->padDecrypt((const unsigned char *)binary,len,(unsigned char *)buf);
		kvi_free(binary);

		if(retVal < 0)
		{
			kvi_free(buf);
			setLastErrorFromRijndaelErrorCode(retVal);
			return false;
		}

		buf[retVal] = '\0';

		plainText = buf;

		kvi_free(buf);
		return true;
	}

	bool KviRijndaelHexEngine::binaryToAscii(const char * inBuffer,int len,KviStr &outBuffer)
	{
		outBuffer.bufferToHex(inBuffer,len);
		return true;
	}

	bool KviRijndaelHexEngine::asciiToBinary(const char * inBuffer,int * len,char ** outBuffer)
	{
		KviStr hex(inBuffer);
		char * tmpBuf;
		*len = hex.hexToBuffer(&tmpBuf,false);
		if(*len < 0)
		{
			setLastError(__tr("The message is not a hexadecimal string: this is not my stuff"));
			return false;
		} else {
			if(len > 0)
			{
				*outBuffer = (char *)kvi_malloc(*len);
				kvi_memmove(*outBuffer,tmpBuf,*len);
				KviStr::freeBuffer(tmpBuf);
			}
		}
		return true;
	}

	bool KviRijndaelBase64Engine::binaryToAscii(const char * inBuffer,int len,KviStr &outBuffer)
	{
		outBuffer.bufferToBase64(inBuffer,len);
		return true;
	}

	bool KviRijndaelBase64Engine::asciiToBinary(const char * inBuffer,int * len,char ** outBuffer)
	{
		KviStr base64(inBuffer);
		char * tmpBuf;
		*len = base64.base64ToBuffer(&tmpBuf,false);
		if(*len < 0)
		{
			setLastError(__tr("The message is not a base64 string: this is not my stuff"));
			return false;
		} else {
			if(len > 0)
			{
				*outBuffer = (char *)kvi_malloc(*len);
				kvi_memmove(*outBuffer,tmpBuf,*len);
				KviStr::freeBuffer(tmpBuf);
			}
		}
		return true;
	}

	static KviCryptEngine * allocRijndael128HexEngine()
	{
		return new KviRijndael128HexEngine();
	}

	static KviCryptEngine * allocRijndael192HexEngine()
	{
		return new KviRijndael192HexEngine();
	}

	static KviCryptEngine * allocRijndael256HexEngine()
	{
		return new KviRijndael256HexEngine();
	}

	static KviCryptEngine * allocRijndael128Base64Engine()
	{
		return new KviRijndael128Base64Engine();
	}

	static KviCryptEngine * allocRijndael192Base64Engine()
	{
		return new KviRijndael192Base64Engine();
	}

	static KviCryptEngine * allocRijndael256Base64Engine()
	{
		return new KviRijndael256Base64Engine();
	}

	static void deallocRijndaelCryptEngine(KviCryptEngine * e)
	{
		delete e;
	}

#endif


// =======================================
// module routines
// =======================================
static bool rijndael_module_init(KviModule * m)
{
#ifdef COMPILE_CRYPT_SUPPORT
	g_pEngineList = new KviPtrList<KviCryptEngine>;
	g_pEngineList->setAutoDelete(false);

	KviStr format = __tr("Cryptographic engine based on the\n" \
		"Advanced Encryption Standard (AES)\n" \
		"algorithm called Rijndael.\n" \
		"The text is first encrypted with rijndael\n" \
		"and then converted to %s notation.\n" \
		"The keys used are %d bit long and will be padded\n" \
		"with zeros if you provide shorter ones.\n" \
		"If only one key is provided, this engine\n" \
		"will use it for both encrypting and decrypting.\n" \
		"See the rijndael module documentation\n" \
		"for more info on the algorithm used.\n");

	// FIXME: Maybe convert this repeated code to a function eh ?

	KviCryptEngineDescription * d = new KviCryptEngineDescription;
	d->szName = "Rijndael128Hex";
	d->szAuthor = "Szymon Stefanek";
	d->szDescription.sprintf(format.ptr(),__tr("hexadecimal"),128);
	d->iFlags = KVI_CRYPTENGINE_CAN_ENCRYPT | KVI_CRYPTENGINE_CAN_DECRYPT |
			KVI_CRYPTENGINE_WANT_ENCRYPT_KEY | KVI_CRYPTENGINE_WANT_DECRYPT_KEY;
	d->allocFunc = allocRijndael128HexEngine;
	d->deallocFunc = deallocRijndaelCryptEngine;
	m->registerCryptEngine(d);

	d = new KviCryptEngineDescription;
	d->szName = "Rijndael192Hex";
	d->szAuthor = "Szymon Stefanek";
	d->szDescription.sprintf(format.ptr(),__tr("hexadecimal"),192);
	d->iFlags = KVI_CRYPTENGINE_CAN_ENCRYPT | KVI_CRYPTENGINE_CAN_DECRYPT |
			KVI_CRYPTENGINE_WANT_ENCRYPT_KEY | KVI_CRYPTENGINE_WANT_DECRYPT_KEY;
	d->allocFunc = allocRijndael192HexEngine;
	d->deallocFunc = deallocRijndaelCryptEngine;
	m->registerCryptEngine(d);

	d = new KviCryptEngineDescription;
	d->szName = "Rijndael256Hex";
	d->szAuthor = "Szymon Stefanek";
	d->szDescription.sprintf(format.ptr(),__tr("hexadecimal"),256);
	d->iFlags = KVI_CRYPTENGINE_CAN_ENCRYPT | KVI_CRYPTENGINE_CAN_DECRYPT |
			KVI_CRYPTENGINE_WANT_ENCRYPT_KEY | KVI_CRYPTENGINE_WANT_DECRYPT_KEY;
	d->allocFunc = allocRijndael256HexEngine;
	d->deallocFunc = deallocRijndaelCryptEngine;
	m->registerCryptEngine(d);

	d = new KviCryptEngineDescription;
	d->szName = "Rijndael128Base64";
	d->szAuthor = "Szymon Stefanek";
	d->szDescription.sprintf(format.ptr(),__tr("base64"),128);
	d->iFlags = KVI_CRYPTENGINE_CAN_ENCRYPT | KVI_CRYPTENGINE_CAN_DECRYPT |
			KVI_CRYPTENGINE_WANT_ENCRYPT_KEY | KVI_CRYPTENGINE_WANT_DECRYPT_KEY;
	d->allocFunc = allocRijndael128Base64Engine;
	d->deallocFunc = deallocRijndaelCryptEngine;
	m->registerCryptEngine(d);

	d = new KviCryptEngineDescription;
	d->szName = "Rijndael192Base64";
	d->szAuthor = "Szymon Stefanek";
	d->szDescription.sprintf(format.ptr(),__tr("base64"),192);
	d->iFlags = KVI_CRYPTENGINE_CAN_ENCRYPT | KVI_CRYPTENGINE_CAN_DECRYPT |
			KVI_CRYPTENGINE_WANT_ENCRYPT_KEY | KVI_CRYPTENGINE_WANT_DECRYPT_KEY;
	d->allocFunc = allocRijndael192Base64Engine;
	d->deallocFunc = deallocRijndaelCryptEngine;
	m->registerCryptEngine(d);

	d = new KviCryptEngineDescription;
	d->szName = "Rijndael256Base64";
	d->szAuthor = "Szymon Stefanek";
	d->szDescription.sprintf(format.ptr(),__tr("base64"),256);
	d->iFlags = KVI_CRYPTENGINE_CAN_ENCRYPT | KVI_CRYPTENGINE_CAN_DECRYPT |
			KVI_CRYPTENGINE_WANT_ENCRYPT_KEY | KVI_CRYPTENGINE_WANT_DECRYPT_KEY;
	d->allocFunc = allocRijndael256Base64Engine;
	d->deallocFunc = deallocRijndaelCryptEngine;
	m->registerCryptEngine(d);

    return true;
#else
	return false;
#endif
}

static bool rijndael_module_cleanup(KviModule *m)
{
#ifdef COMPILE_CRYPT_SUPPORT
	while(g_pEngineList->first())delete g_pEngineList->first();
	delete g_pEngineList;
	m->unregisterCryptEngines();
	m->unregisterMetaObject("KviRijndaelEngine");
	m->unregisterMetaObject("KviRijndaelHexEngine");
	m->unregisterMetaObject("KviRijndael128HexEngine");
	m->unregisterMetaObject("KviRijndael192HexEngine");
	m->unregisterMetaObject("KviRijndael256HexEngine");
	m->unregisterMetaObject("KviRijndaelBase64Engine");
	m->unregisterMetaObject("KviRijndael128Base64Engine");
	m->unregisterMetaObject("KviRijndael192Base64Engine");
	m->unregisterMetaObject("KviRijndael256Base64Engine");
	return true;
#else
	return false;
#endif
}

static bool rijndael_module_can_unload(KviModule *)
{
#ifdef COMPILE_CRYPT_SUPPORT
	return g_pEngineList->isEmpty();
#else
	return true;
#endif
}

// =======================================
// plugin definition structure
// =======================================
KVIMODULEEXPORT KviModuleInfo kvirc_module_info =
{
    "Rijndael crypt engine",
	"1.0.0",
	"Szymon Stefanek <stefanek@tin.it>" ,
	"Exports the rijndael crypt engine",
    rijndael_module_init ,
    rijndael_module_can_unload,
    0,
	rijndael_module_cleanup
};

#ifdef COMPILE_CRYPT_SUPPORT
	#include "libkvirijndael.moc"
#endif
