//
//   File : libkvireguser.cpp
//   Creation date : Fri Dec 01 2000 14:53:10 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_regusersdb.h"
#include "kvi_command.h"
#include "kvi_out.h"
#include "kvi_mirccntrl.h"
#include "kvi_window.h"
#include "kvi_locale.h"
#include "kvi_app.h"
#include "kvi_window.h"
#include "kvi_frame.h"

#include "edituser.h"

#include "kvi_list.h"
#include <qsplitter.h> // FIXME: REmove this!

//#warning "$reguser.matches..."
//#warning "$reguser.clear"

//#warning "THIS MODULE HAS TO BE REVISED (notify list consistency!!!)"

// kvi_app.cpp
extern KVIRC_API KviRegisteredUserDataBase * g_pRegisteredUserDataBase;

KviUserEditor * g_pRegisteredUserEditor = 0;

/*
	@doc: reguser
	@type:
		module
	@short:
		Interface to the registered users database
	@title:
		The reguser module
	@body:
		The reguser module is the scripting interface to the [doc:registered_users]registered users database[/doc].[br]
		It provides the following set of commands:[br]
		[cmd]reguser.add[/cmd]: adds an user entry to the database[br]
		[cmd]reguser.remove[/cmd]: removes an user entry from the database[br]
		[cmd]reguser.list[/cmd]: lists the entries[br]
		[cmd]reguser.setproperty[/cmd]: sets a property associated to an entry[br]
		[cmd]reguser.edit[/cmd]: shws an editor for an user entry
		[fnc]$reguser.match[/fnc](): finds an entry by matching masks[br]
		[fnc]$reguser.matchExact[/fnc](): finds an entry by comparing masks[br]
		[fnc]$reguser.property[/fnc](): finds a property associated to an entry[br]
		[fnc]$reguser.matchProperty[/fnc](): shortcut for [fnc]$reguser.property[/fnc]([fnc]$reguser.match[/fnc]())[br]
		[fnc]$reguser.mask[/fnc](): returns the registration masks for an entry
*/

/*
	@doc: reguser.edit
	@type:
		command
	@title:
		reguser.edit
	@keyterms:
		registering users
	@short:
		Edits an user entry
	@syntax:
		reguser.edit [-t] [-f] [name]
	@description:
		Shows the registered user database editor.[br]
		If [name] is passed, the user [name] entry editor is shown. If the user with [name] is not in the database
		and the [-f] switch is passed, a new entry with [name] is added and the user [name] entry editor is shown.[br]
		If the [-t] switch is passed, the editor is created as a toplevel window (dialog), otherwise
		it is created as part of the current main window.[br]
	@examples:
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.add[/cmd],
		[cmd]reguser.remove[/cmd],
		[cmd]reguser.addmask[/cmd],
		[cmd]reguser.list[/cmd]
*/

static bool reguser_module_cmd_edit(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"reguser_module_cmd_edit");

	KviStr name,dummy;
	if(!g_pUserParser->parseCmdSingleToken(c,name))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	///
	if(g_pRegisteredUserEditor)
	{
		if(c->hasSwitch('t'))
		{
			if(g_pRegisteredUserEditor->parent())
			{
//#warning "CENTER THE DIALOG ? Remember its position ?"
				g_pRegisteredUserEditor->reparent(0,QPoint(0,0),true);
			}
		} else {
			if(g_pRegisteredUserEditor->parent() != c->window()->frame()->splitter())
			{
				g_pRegisteredUserEditor->reparent(c->window()->frame()->splitter(),QPoint(0,0),true);
			}
		}
	} else {
		if(c->hasSwitch('t'))
		{
			g_pRegisteredUserEditor = new KviUserEditor(0,"reguser_editor");
		} else {
			g_pRegisteredUserEditor = new KviUserEditor(c->window()->frame()->splitter(),"reguser_editor");
		}
	}

	g_pRegisteredUserEditor->show();
	g_pRegisteredUserEditor->raise();
	g_pRegisteredUserEditor->setFocus();
	///

	if(name.hasData())
	{
		g_pRegisteredUserEditor->editUser(name.ptr(),c->hasSwitch('f'));
	}

	return c->leaveContext();
}


/*
	@doc: reguser.add
	@type:
		command
	@title:
		reguser.add
	@keyterms:
		registering users
	@short:
		Registers an user
	@syntax:
		reguser.add [-r] [-f] [-q] <name> [mask]
	@description:
		Adds a new entry with the specified <name> to the database.[br]
		If the database contains an entry with the same <name>, this command just prints
		a warning and returns.[br]
		If no [mask] is given the new entry has no registration masks: so it can't be "matched" yet. You have to add
		at least one mask with [cmd]reguser.addmask[/cmd].
		If [mask] is given, then it is added to the entry mask list.[br]
		The <name> parameter may contain any character: even spaces are allowed (obviously you have to
		use quotes in that case).[br]
		If the '-r' switch is given , the new entry replaces any previous one with the same <name> (the old entry is removed).[br]
		If the '-f' switch is given , and there is an existing entry with the same name , no warning is printed
		and the old entry is treated just like it has been just added (thus the [mask] is eventually added to its mask list).[br]
		The '-q' switch causes the command to run in "quiet" mode and print no warning.[br]
	@examples:
		[example]
			reguser.add Pragma
		[/example]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.remove[/cmd],
		[cmd]reguser.addmask[/cmd],
		[cmd]reguser.list[/cmd]
*/

static bool reguser_module_cmd_add(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"reguser_module_cmd_add");

	KviStr name,mask;
	if(!g_pUserParser->parseCmdSingleToken(c,name))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,mask))return false;

	if(name.isEmpty())
	{
		if(!c->hasSwitch('q'))c->warning(__tr("No name specified"));
		return c->leaveContext();
	}

	if(c->hasSwitch('r'))g_pRegisteredUserDataBase->removeUser(name.ptr());


	KviRegisteredUser * u = g_pRegisteredUserDataBase->addUser(name.ptr());

	if(u == 0)
	{
		if(c->hasSwitch('f'))
		{
			u = g_pRegisteredUserDataBase->findUserByName(name.ptr());
		} else {
			if(!c->hasSwitch('q'))c->warning(__tr("User already registered: found exact name match"));
		}
	}


	if(u)
	{
		if(mask.hasData())
		{
			KviIrcMask * m = new KviIrcMask(mask.ptr());
			u = g_pRegisteredUserDataBase->addMask(u,m);
			if(u != 0)
			{
				if(!c->hasSwitch('q'))c->warning(__tr("Mask %s is already used to identify user %s"),mask.ptr(),u->name());
			}
		}
	}

	// no need to restart notify list for now

	return c->leaveContext();
}

/*
	@doc: reguser.remove
	@type:
		command
	@title:
		reguser.remove
	@keyterms:
		unregistering users
	@short:
		Removes an entry from the regusers database
	@syntax:
		reguser.remove [-q] [-n] <name>
	@description:
		Removes the regusers database entry with the specified <name>.[br]
		If the -n switch is given, and there is a removed entry,
		this command will restart all the running "notify lists".
		If the -q switch is used , no warning is printed if the <name>
		does not identify an existing entry.
	@examples:
		[example]
			[cmd]reguser.add Pragma[/cmd]
			reguser.remove Pragma
		[example]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.add[cmd],
		[cmd]reguser.addmask[/cmd],
		[cmd]reguser.list[/cmd]
*/

static bool reguser_module_cmd_remove(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"reguser_module_cmd_remove");

	KviStr name;
	if(!g_pUserParser->parseCmdFinalPart(c,name))return false;

	if(name.isEmpty())
	{
		if(!c->hasSwitch('q'))c->warning(__tr("No name specified"));
		return c->leaveContext();
	}

	if(!(g_pRegisteredUserDataBase->removeUser(name.ptr())))
	{
		c->warning(__tr("User not found (%s)"),name.ptr());
	} else {
		if(c->hasSwitch('n'))g_pApp->restartNotifyLists();
	}

	return c->leaveContext();
}

/*
	@doc: reguser.addmask
	@type:
		command
	@title:
		reguser.addmask
	@keyterms:
		registering users
	@short:
		Adds a mask to a registered user
	@syntax:
		reguser.addmask [-f] <name> <mask>
	@description:
		Adds a registration mask to the database entry with name <name>.[br]
		The <mask> has the common [doc:irc_masks]irc mask[/doc] format.[br]
		The only restriction on the <mask> is that it must be unique inside the
		regusers database: you can't match two users with the same mask (it would
		have rather undefined results).[br]
		Once at least one mask has been added, the user can be "matched"
		by using the [fnc]$reguser.match[/fnc]() function.[br]
		Each database entry can have an unlimited number of masks.[br]
	@examples:
		[example]
			[cmd]reguser.add[/cmd] "Szymon Stefanek"
			reguser.addmask "Szymon Stefanek" Pragma!*@*.it
		[/example]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.add[cmd],
		[cmd]reguser.remove[/cmd],
		[cmd]reguser.delmask[/cmd],
		[cmd]reguser.list[/cmd],
		[fnc]reguser.match[/fnc],
		[fnc]reguser.exactMatch[/fnc]
*/

static bool reguser_module_cmd_addmask(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"reguser_module_cmd_addmask");

	KviStr name,mask;
	if(!g_pUserParser->parseCmdSingleToken(c,name))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,mask))return false;

	if(name.isEmpty())
	{
		c->warning(__tr("No name specified"));
		return c->leaveContext();
	}

	if(mask.isEmpty())
	{
		c->warning(__tr("No mask specified"));
		return c->leaveContext();
	}

	KviRegisteredUser * u = g_pRegisteredUserDataBase->findUserByName(name.ptr());
	if(!u)
	{
		c->warning(__tr("User %s not found"),name.ptr());
		return c->leaveContext();
	}

	KviIrcMask * mk = new KviIrcMask(mask.ptr());

	if(c->hasSwitch('f'))g_pRegisteredUserDataBase->removeMask(*mk);

	u = g_pRegisteredUserDataBase->addMask(u,mk);

	if(u != 0)c->warning(__tr("Mask %s already used to identify user %s"),mask.ptr(),u->name());

	return c->leaveContext();
}


/*
	@doc: reguser.delmask
	@type:
		command
	@title:
		reguser.delmask
	@keyterms:
		registering users
	@short:
		Removes a mask from the regusers database
	@syntax:
		reguser.delmask <mask>
	@description:
		Removes a mask from the regusers database.[br]
		Since masks are unique in the database , you don't need to pass
		an <user> parameter: if an <user> entry has the <mask>, removing that <mask>
		will surely remove it from that <user> entry. (contorsions ?)[br]
	@examples:
		[example]
			[cmd]reguser.add[/cmd] "Szymon Stefanek"
			[cmd]reguser.addmask[/cmd] "Szymon Stefanek" Pragma!*@*.it
			reguser.delmask Pragma!*@*.it
			[cmd]reguser.list[/cmd]
		[/example]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.add[cmd],
		[cmd]reguser.remove[/cmd],
		[cmd]reguser.addmask[/cmd],
		[cmd]reguser.list[/cmd],
		[fnc]reguser.match[/fnc],
		[fnc]reguser.exactMatch[/fnc]
*/

static bool reguser_module_cmd_delmask(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"reguser_module_cmd_delmask");

	KviStr mask;
	if(!g_pUserParser->parseCmdFinalPart(c,mask))return false;

	if(mask.isEmpty())
	{
		c->warning(__tr("No mask specified"));
		return c->leaveContext();
	}

	if(!g_pRegisteredUserDataBase->removeMask(mask.ptr()))
	{
		c->warning(__tr("Mask %s not found"),mask.ptr());
	}

	return c->leaveContext();
}


/*
	@doc: reguser.setproperty
	@type:
		command
	@title:
		reguser.setproperty
	@keyterms:
		setting user properties
	@short:
		Sets a property of a registered user
	@syntax:
		reguser.setproperty [-n] [-q] <name> <property> [value]
	@description:
		Adds a property to the registered users database entry with name <name>.[br]
		The property <property> is set to the specified <value> or is unset if <value>
		is not given (or is an empty string).[br]
		If the '-n' switch is used , and the user entry has been found in the database,
		all the [doc:notify_list]notify lists[/doc] are restarted.[br]
		This is useful when you're changing someone's notify property.[br]
		If the -q switch is used , the command runs in "quiet" mode and prints no warnings.[br]
	@examples:
		[example]
			[cmd]reguser.add[/cmd] "Szymon Stefanek"
			reguser.setproperty -n "Szymon Stefanek" notify Pragma
			[cmd]reguser.list[/cmd]
		[/example]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[fnc]reguser.matchProperty[/fnc],
		[fnc]reguser.property[/fnc]
*/

static bool reguser_module_cmd_setproperty(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"reguser_module_cmd_setproperty");

	KviStr name,property,value;
	if(!g_pUserParser->parseCmdSingleToken(c,name))return false;
	if(!g_pUserParser->parseCmdSingleToken(c,property))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,value))return false;

	if(name.isEmpty())
	{
		if(!c->hasSwitch('q'))c->warning(__tr("No name specified"));
		return c->leaveContext();
	}

	if(property.isEmpty())
	{
		if(!c->hasSwitch('q'))c->warning(__tr("No property specified"));
		return c->leaveContext();
	}

	KviRegisteredUser * u = g_pRegisteredUserDataBase->findUserByName(name.ptr());
	if(!u)
	{
		if(!c->hasSwitch('q'))c->warning(__tr("User %s not found"),name.ptr());
	} else {
		u->setProperty(property.ptr(),value.ptr());
		if(c->hasSwitch('n'))g_pApp->restartNotifyLists();
	}

	return c->leaveContext();
}


/*
	@doc: reguser.list
	@type:
		command
	@title:
		reguser.list [mask]
	@keyterms:
		registered users
	@short:
		Lists the registered users
	@syntax:
		reguser.list [mask]
	@description:
		Lists the entries in the registered users database.[br]
		If <mask> is specified, only entries matching the <mask>
		and the entries with no registration masks are listed.[br]
		Please note that <mask> is a wildcard string that will match
		wildcard strings... don't get messed with that :D[br]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.add[cmd],
		[cmd]reguser.remove[/cmd],
		[fnc]reguser.match[/fnc],
		[fnc]reguser.exactMatch[/fnc]
*/

static bool reguser_module_cmd_list(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"reguser_module_cmd_list");

	KviStr usermask;
	if(!g_pUserParser->parseCmdFinalPart(c,usermask))return false;

	KviIrcMask mask(usermask.ptr());


//#warning "FIND A BETTER KVI_OUT_*"
	c->window()->outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("Registered users database dump:"));

	int count = 0;

	QAsciiDict<KviRegisteredUser> * d = g_pRegisteredUserDataBase->userDict();
	QAsciiDictIterator<KviRegisteredUser> it(*d);
	while(KviRegisteredUser * u = it.current())
	{
		KviPtrList<KviIrcMask> * ml = u->maskList();
		if(u->matches(mask) || (ml->count() == 0))
		{
			c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr(" User: %c%s"),KVI_TEXT_BOLD,u->name());

			if(ml->count() == 0)
			{
				c->window()->output(KVI_OUT_SYSTEMWARNING,__tr("    Warning: this user has no registration masks"));
			} else {
				for(KviIrcMask * m = ml->first();m;m = ml->next())
				{
					c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr("    Mask: %s!%s@%s"),m->nick(),m->user(),m->host());
				}
			}

			QAsciiDict<KviStr> * pd = u->propertyDict();
			if(pd)
			{
				QAsciiDictIterator<KviStr> pdit(*pd);
				while(pdit.current())
				{
					c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr("    Property: %s=%s"),pdit.currentKey(),pdit.current()->ptr());
					++pdit;
				}
			} else c->window()->outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("    No properties"));
			count++;
		}
		++it;
	}

	c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr("Total: %d matching users (of %d in the database)"),count,d->count());

	return c->leaveContext();
}


/*
	@doc: reguser.match
	@type:
		function
	@title:
		$reguser.match
	@short:
		Returns an entry in the registered user database
	@syntax:
		$reguser.match(<user_mask>)
	@description:
		Returns the name of the first entry matching <user_mask> in the registered user database.[br]
		The <user_mask> has the common [doc:irc_masks]irc mask[/doc] format but can't contain wildcards.[br]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.add[cmd],
		[cmd]reguser.remove[/cmd],
		[cmd]reguser.addmask[/cmd],
		[cmd]reguser.list[/cmd],
		[fnc]reguser.exactMatch[/fnc]
*/

static bool reguser_module_fnc_match(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"reguser_module_fnc_match");
	KviIrcMask mask(parms->safeFirstParam());
	KviRegisteredUser * u = g_pRegisteredUserDataBase->findMatchingUser(mask.nick(),mask.user(),mask.host());
	if(u)buffer.append(u->name());
	return c->leaveContext();
}


/*
	@doc: reguser.exactmatch
	@type:
		function
	@title:
		$reguser.exactMatch
	@short:
		Returns an entry in the registered user database
	@syntax:
		$reguser.exactMatch(<user_mask>)
	@description:
		Returns the name of the registered user database entry that has <user_mask> in its registration mask list.[br]
		The <user_mask> has the common [doc:irc_masks]wild irc mask[/doc] format and can contain wildcards.[br]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.add[cmd],
		[cmd]reguser.remove[/cmd],
		[cmd]reguser.addmask[/cmd],
		[cmd]reguser.list[/cmd],
		[fnc]reguser.match[/fnc]
*/

static bool reguser_module_fnc_exactMatch(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"reguser_module_fnc_exactmatch");
	KviIrcMask mask(parms->safeFirstParam());
	KviRegisteredUser * u = g_pRegisteredUserDataBase->findUserWithMask(mask);
	if(u)buffer.append(u->name());
	return c->leaveContext();
}

///*
//	@doc: reguser.matches
//	@type:
//		function
//	@title:
//		$reguser.matches
//	@short:
//		Returns entries in the registered user database
//	@syntax:
//		$reguser.matches(<user_mask>)
//	@description:
//		Returns a comma separated list of registered user masks that match <user_mask>.[br]
//		User mask must be in the format <nick>!<user>@<host> , where <user> and
//		<host> may contain '*' wildcards.[br]
//		If no registered mask matches the <user_mask> an empty string is returned.[br]
//	@seealso:
//		[module:reguser]reguser module documentation[/module]
//*/

//static void search_reguser_list(KviRegisteredUserList * l,KviWindow * w,const char * mask,KviStr & buffer)
//{
//	for(KviRegisteredUser * u = l->first();u;u = l->next())
//	{
//		if(u->matches(mask))
//		{
//			if(buffer.hasData())buffer.append(',');
//			buffer.append(u->mask());
//		}
//	}
//}

//static bool reguser_module_fnc_matches(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
//{
//	ENTER_CONTEXT(c,"reguser_module_fnc_matches");
//
//	KviStr list;
//
//	const QAsciiDict<KviRegisteredUserList> * d = g_pRegisteredUserDataBase->nickDict();
//	QAsciiDictIterator<KviRegisteredUserList> it(*d);
//	while(KviRegisteredUserList * l = it.current())
//	{
//		search_reguser_list(l,c->window(),parms->safeFirstParam(),list);
//		++it;
//	}
//
//	search_reguser_list(g_pRegisteredUserDataBase->wildList(),c->window(),parms->safeFirstParam(),list);
//
//	buffer.append(list);
//
//	return c->leaveContext();
//}


/*
	@doc: reguser.mask
	@type:
		function
	@title:
		$reguser.mask
	@short:
		Returns the registration masks for a db entry
	@syntax:
		$reguser.mask(<name>[,<N>])
	@description:
		Returns the <N>th registration mask for the registered user database record identified
		by <name>. If <N> is not given , returns an array of registration masks for that entry.[br]
		If <name> does not identify a valid entry an empty string is returned;
		if <N> is out of range (or there are no masks at all) , an empty string is returned.[br]
		<N> is a ZERO BASED index.[br]
	@examples:
		[example]
			[cmd]reguser.add Pragma[/cmd]
			[cmd]reguser.addmask[/cmd] Pragma Pragma!*@*.it
			[cmd]echo[/cmd] $reguser.mask(Pragma,0)
		[/example]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.add[cmd],
		[cmd]reguser.remove[/cmd],
		[cmd]reguser.addmask[/cmd],
		[cmd]reguser.delmask[/cmd],
		[cmd]reguser.list[/cmd],
		[fnc]reguser.match[/fnc],
		[fnc]reguser.matchProperty[/fnc]
*/

static bool reguser_module_fnc_mask(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"reguser_module_fnc_mask");
	KviRegisteredUser * u = g_pRegisteredUserDataBase->findUserByName(parms->safeFirstParam());
	if(u)
	{
		KviStr n = parms->safeNextParam();
		if(n.hasData() && n.isUnsignedNum())
		{
			KviIrcMask * m = u->maskList()->at(n.toInt());
			if(m)buffer.append(KviStr::Format,"%s!%s@%s",m->nick(),m->user(),m->host());
		} else {
			c->beginListArrayOrDictionaryReturnIdentifier();
			int id=0;
			for(KviIrcMask * m = u->maskList()->first();m;m = u->maskList()->next())
			{
				KviStr tmp(KviStr::Format,"%s!%s@%s",m->nick(),m->user(),m->host());
				c->addListArrayOrDictionaryReturnValue(id++,tmp,buffer);
			}
		}
	} //else c->warning(__tr("User %s not found"),parms->safeFirstParam());
	return c->leaveContext();
}

/*
	@doc: reguser.property
	@type:
		function
	@title:
		$reguser.property
	@short:
		Returns a property of an entry in the registered user database
	@syntax:
		$reguser.property(<user_name>,<property_name>)
	@description:
		Returns the value of the property identified by <property_name> and
		bound to the entry identified by <user_name>.[br]
		If the property is not set or if there is no entry with <user_name> an empty string is returned.[br]
	@examples:
		[example]
			[cmd]reguser.add Pragma[/cmd]
			[cmd]reguser.addmask[/cmd] Pragma Pragma!*@*.it
			[cmd]reguser.setproperty[/cmd] Pragma isDev 1
			[cmd]echo[/cmd] $reguser.property(Pragma,isDev)
		[/example]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.add[cmd],
		[cmd]reguser.remove[/cmd],
		[cmd]reguser.addmask[/cmd],
		[cmd]reguser.list[/cmd],
		[fnc]reguser.match[/fnc],
		[fnc]reguser.matchProperty[/fnc]
*/

static bool reguser_module_fnc_property(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"reguser_module_fnc_property");
	KviRegisteredUser * u = g_pRegisteredUserDataBase->findUserByName(parms->safeFirstParam());
	if(u)
	{
		KviStr tmp;
		u->getProperty(parms->safeNextParam(),tmp);
		buffer.append(tmp.ptr());
	} //else c->warning(__tr("User %s not found"),parms->safeFirstParam());
	return c->leaveContext();
}


/*
	@doc: reguser.matchproperty
	@type:
		function
	@title:
		$reguser.matchProperty
	@short:
		Returns a property of an entry in the registered user database
	@syntax:
		$reguser.matchProperty(<user_mask>,<property_name>)
	@description:
		Returns the value of the property identified by <property_name> and
		bound to the first entry matched by by <user_mask>.[br]
		If the property is not set, an empty string is returned.[br]
		If no entry matches <user_mask> this function returns an empty string
		and does not print any error.[br]
		[example]
			%property = $reguser.matchProperty(<user_mask>,<property_name>)
		[/example]
		Is actually a shortcut for:[br]
		[example]
			%tmp = [fnc]$reguser.match[/fnc](<user_mask>)
			if("%tmp" != "")%property = $reguser.property(%tmp,<property_name>)
		[/example]
	@seealso:
		[module:reguser]Registered users database interface[/module],
		[doc:registered_users]Registered users database[/doc],
		[cmd]reguser.add[cmd],
		[cmd]reguser.remove[/cmd],
		[cmd]reguser.addmask[/cmd],
		[cmd]reguser.list[/cmd],
		[fnc]reguser.match[/fnc],
		[fnc]reguser.property[/fnc]
*/

static bool reguser_module_fnc_matchProperty(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"reguser_module_fnc_matchProperty");
	KviIrcMask mask(parms->safeFirstParam());

	KviRegisteredUser * u = g_pRegisteredUserDataBase->findMatchingUser(mask.nick(),mask.user(),mask.host());
	if(u)
	{
		KviStr tmp;
		u->getProperty(parms->safeNextParam(),tmp);
		buffer.append(tmp.ptr());
	}
	return c->leaveContext();
}

static bool reguser_module_init(KviModule * m)
{
	m->registerCommand("add",reguser_module_cmd_add);
	m->registerCommand("remove",reguser_module_cmd_remove);
	m->registerCommand("addmask",reguser_module_cmd_addmask);
	m->registerCommand("delmask",reguser_module_cmd_delmask);
	m->registerCommand("edit",reguser_module_cmd_edit);
	m->registerCommand("list",reguser_module_cmd_list);
	m->registerCommand("setproperty",reguser_module_cmd_setproperty); // setproperty <mask> <name> <value>
	m->registerFunction("match",reguser_module_fnc_match);
	m->registerFunction("exactmatch",reguser_module_fnc_exactMatch);
	m->registerFunction("mask",reguser_module_fnc_mask);
	m->registerFunction("property",reguser_module_fnc_property);
	m->registerFunction("matchProperty",reguser_module_fnc_matchProperty);

	return true;
}

static bool reguser_module_cleanup(KviModule *m)
{
	if(g_pRegisteredUserEditor)delete g_pRegisteredUserEditor;
//	g_pRegisteredUserEditor = 0;
	m->unregisterMetaObject("KviUserEditorWidget");
	m->unregisterMetaObject("KviUserEditor");
	m->unregisterMetaObject("KviUserEditorIndex");
	return true;
}

static bool reguser_module_can_unload(KviModule *)
{
	return (g_pRegisteredUserEditor == 0);
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Reguser",                                              // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2002 Szymon Stefanek (stefanek@tin.it)", // author & (C)
	"Script interface to the registered users database",
	reguser_module_init,
	reguser_module_can_unload,
	0,
	reguser_module_cleanup
};
