//
//   File : edituser.cpp
//   Creation date : Tue Dec 26 2000 12:24:12 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "edituser.h"

#include "kvi_regusersdb.h"

#include "kvi_locale.h"
#include "kvi_ircmask.h"
#include "kvi_debug.h"
#include "kvi_iconmanager.h"
#include "kvi_app.h"
#include "kvi_options.h"
#include "kvi_file.h"
#include "kvi_filedialog.h"
#include "kvi_msgbox.h"
#include "kvi_fileutils.h"

#include <qlayout.h>
#include <qlabel.h>
#include <qheader.h>
#include <qasciidict.h>
#include <qimage.h>
#include <qstring.h>
#include <qcombobox.h>

// libkvireguser.cpp
extern KviUserEditor * g_pRegisteredUserEditor;

// kvi_app.cpp
extern KVIRC_API KviRegisteredUserDataBase * g_pRegisteredUserDataBase;


static KviRegisteredUserDataBase * g_pLocalRegisteredUserDataBase = 0; // local copy!

KviUserEditorWidget::KviUserEditorWidget(QWidget * par,KviRegisteredUser *u,const char * name)
: QWidget(par,name)
{
	m_szOriginalName = u->name();

	if(g_pRegisteredUserEditor)g_pRegisteredUserEditor->registerEditor(this);

	QGridLayout * g = new QGridLayout(this,13,3,2,2);

	QLabel * l = new QLabel(__tr("Name:"),this);
	g->addWidget(l,0,0);

	m_pNameEdit = new QLineEdit(this);
	m_pNameEdit->setText(u->name());
	m_pNameEdit->setEnabled(false); 
	g->addWidget(m_pNameEdit,0,1);

	m_pChangeButton = new QPushButton(__tr("Change"),this);
	g->addWidget(m_pChangeButton,0,2);
	connect(m_pChangeButton,SIGNAL(clicked()),this,SLOT(changeClicked()));

	QFrame * f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(f,1,1,0,2);

	l = new QLabel(__tr("Masks:"),this);
	g->addWidget(l,2,0);

	m_pMaskTable = new QTable(this);
	g->addMultiCellWidget(m_pMaskTable,2,5,1,2);

	m_pMaskTable->setNumCols(3);
	m_pMaskTable->setSelectionMode(QTable::NoSelection);

	m_pMaskTable->horizontalHeader()->setLabel(0,__tr("Nick"));
	m_pMaskTable->horizontalHeader()->setLabel(1,__tr("User"));
	m_pMaskTable->horizontalHeader()->setLabel(2,__tr("Host"));
	connect(m_pMaskTable,SIGNAL(valueChanged(int,int)),this,SLOT(maskValueChanged(int,int)));
	
	m_pDelMaskButton = new QPushButton(__tr("Delete mask"),this);
	connect(m_pDelMaskButton,SIGNAL(clicked()),this,SLOT(delMaskClicked()));
	g->addWidget(m_pDelMaskButton,3,0);

	m_pAddMaskButton = new QPushButton(__tr("Add mask"),this);
	connect(m_pAddMaskButton,SIGNAL(clicked()),this,SLOT(addMaskClicked()));
	g->addWidget(m_pAddMaskButton,4,0);

	if(m_pMaskTable->numRows() == 0)m_pDelMaskButton->setEnabled(false);

	f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(f,6,6,0,2);

	l = new QLabel(__tr("Properties:"),this);
	g->addWidget(l,7,0);


	m_pPropertyTable = new QTable(this);
	g->addMultiCellWidget(m_pPropertyTable,7,10,1,2);

	m_pPropertyTable->setNumCols(2);
	m_pPropertyTable->setSelectionMode(QTable::NoSelection);

	m_pPropertyTable->horizontalHeader()->setLabel(0,__tr("Property"));
	m_pPropertyTable->horizontalHeader()->setLabel(1,__tr("Value"));
	connect(m_pPropertyTable,SIGNAL(valueChanged(int,int)),this,SLOT(propertyValueChanged(int,int)));
	m_pDelPropertyButton = new QPushButton(__tr("Delete property"),this);
	connect(m_pDelPropertyButton,SIGNAL(clicked()),this,SLOT(delPropertyClicked()));
	g->addWidget(m_pDelPropertyButton,8,0);

	m_pAddPropertyButton = new QPushButton(__tr("Add property"),this);
	connect(m_pAddPropertyButton,SIGNAL(clicked()),this,SLOT(addPropertyClicked()));
	g->addWidget(m_pAddPropertyButton,9,0);

	g->setRowStretch(5,1);
	g->setRowStretch(10,1);

	f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(f,11,11,0,2);

	QPushButton * b = new QPushButton(__tr("Close"), this);
	g->addWidget(b,12,2);
	connect(b,SIGNAL(clicked()),this,SLOT(closeClicked()));

	g->setColStretch(2,1);
	refillData();

}

KviUserEditorWidget::~KviUserEditorWidget()
{
	if(g_pRegisteredUserEditor)g_pRegisteredUserEditor->unregisterEditor(this);
}
void KviUserEditorWidget::closeClicked()
{
	delete this; // Yahoo!!
}

void KviUserEditorWidget::changeClicked()
{
	disconnect(m_pChangeButton,SIGNAL(clicked()),this,SLOT(changeClicked()));
	m_pNameEdit->setEnabled(true);
	m_pChangeButton->setText(__tr("Finish"));
	connect(m_pChangeButton,SIGNAL(clicked()),this,SLOT(finishClicked()));	
}

void KviUserEditorWidget::finishClicked()
{
	disconnect(m_pChangeButton,SIGNAL(clicked()),this,SLOT(finishClicked()));
	KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->findUserByName(originalName());
	KviStr tmp = m_pNameEdit->text();
	KviRegisteredUser * u2 = g_pLocalRegisteredUserDataBase->findUserByName(tmp.ptr());
	if((u2 != u) && u2)
	{
		// the name changed... and the new one is alredy in the db!
		// ask the user what to do...
//#warning "Maybe ask the user what he wants to do ?"
		g_pLocalRegisteredUserDataBase->removeUser(tmp.ptr()); // remove...
	}
	// the name has notbeen changed...(so u == u2)
	// and/or the new name is NOT in the database (so u==u2==0 or only u is non zero)
	// in sum...there is eventually ONE obsolete entry in the db now and it is pointed by u
	if(u)g_pLocalRegisteredUserDataBase->removeUser(originalName());
	u = g_pLocalRegisteredUserDataBase->addUser(tmp.ptr());
	if(!u)debug("Ops... failed to add the new user %s (already there but he shoudn't be!)",tmp.ptr());
	int num = m_pMaskTable->numRows();
	int i;
	for(i = 0;i < num;i++)
	{
		KviStr nick = m_pMaskTable->text(i,0);
		KviStr user = m_pMaskTable->text(i,1);
		KviStr host = m_pMaskTable->text(i,2);
		if(nick.hasData())
		{
			if(user.isEmpty())user = "*";
			if(host.isEmpty())host = "*";

			tmp.sprintf("%s!%s@%s",nick.ptr(),user.ptr(),host.ptr());
			KviIrcMask * msk = new KviIrcMask(tmp.ptr());
			if(g_pLocalRegisteredUserDataBase->addMask(u,msk))
			{
				// and here?
				// #warning "Ask the user?"
				debug("Ops, failed to add the mask (%s!%s@%s) (mask already owned)",nick.ptr(),user.ptr(),host.ptr());
			}
		}
	}
	
	KviStr value;
	num = m_pPropertyTable->numRows();
	for(i=0;i<num;i++)
	{
		tmp = m_pPropertyTable->text(i,0);
		value= m_pPropertyTable->text(i,1);
		if(tmp.hasData() && value.hasData())u->setProperty(tmp.ptr(),value.ptr());
		// else notify it?
		// #warning "else notify it?"
	}
	m_szOriginalName = u->name();
	emit changed();

	m_pNameEdit->setEnabled(false);
	m_pChangeButton->setText(__tr("Change"));
	connect(m_pChangeButton,SIGNAL(clicked()),this,SLOT(changeClicked()));
	
}

void KviUserEditorWidget::delMaskClicked()
{
//#warning "Popup a dialog here if the last mask is going to be removed?"
	int i = m_pMaskTable->currentRow();

	KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->findUserByName(originalName());	
	if(!u)
	{
		debug("Ops.. no such user ?");
		return;
	}
	KviPtrList<KviIrcMask> * ml = u->maskList();
	KviStr tmp, nick, user, host;

	nick = m_pMaskTable->text(i,0);
	user = m_pMaskTable->text(i,1);
	host = m_pMaskTable->text(i,2);
	tmp.sprintf("%s!%s@%s",nick.ptr(),user.ptr(),host.ptr());	
	KviIrcMask msk(tmp.ptr());

	int sw = false;
	for(KviIrcMask * m = ml->first();m;m = ml->next())
	{
		if(*m == msk)
		{
			sw = true;
			if(!ml->remove(m))
			{
				debug("KviUserEditorWidget::delMaskClicked(): Ups, could not delete mask!");
				return;
			}else emit changed();
			break;
		}
	}
	
	if(!sw)
	{
		//Hopefully an empty row, should we do something special here?
	}

	if((i > -1) && (i < m_pMaskTable->numRows()))
	{
		// remove row i
		m_pMaskTable->clearCell(i,0);
		m_pMaskTable->clearCell(i,1);
		m_pMaskTable->clearCell(i,2);

		for(;i < (m_pMaskTable->numRows() - 1);i++)
		{
			m_pMaskTable->swapRows(i,i+1);
		}

		m_pMaskTable->setNumRows(m_pMaskTable->numRows() - 1);
		if(m_pMaskTable->numRows() == 0)m_pDelMaskButton->setEnabled(false);

		
	}
}

void KviUserEditorWidget::addMaskClicked()
{
	m_pMaskTable->setNumRows(m_pMaskTable->numRows() + 1);
	m_pMaskTable->setItem(m_pMaskTable->numRows() - 1,0,new QTableItem(m_pMaskTable,QTableItem::OnTyping,""));
	m_pMaskTable->setItem(m_pMaskTable->numRows() - 1,1,new QTableItem(m_pMaskTable,QTableItem::OnTyping,""));
	m_pMaskTable->setItem(m_pMaskTable->numRows() - 1,2,new QTableItem(m_pMaskTable,QTableItem::OnTyping,""));
	m_pDelMaskButton->setEnabled(true);
}
 
void KviUserEditorWidget::delPropertyClicked()
{
	int i = m_pPropertyTable->currentRow();

	KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->findUserByName(originalName());
	if(!u)
	{
		debug("Ops.. no such user ?");
		return;
	}
	QAsciiDict<KviStr> * d = u->propertyDict();
	KviStr name = m_pPropertyTable->text(i,0);

	if(!d->remove(name.ptr()))
	{
		//Hopefully and empty row, should we do something special here?
	}
	emit changed();

	if((i > -1) && (i < m_pPropertyTable->numRows()))
	{
		// remove row i
		m_pPropertyTable->clearCell(i,0);
		m_pPropertyTable->clearCell(i,1);

		for(;i < (m_pPropertyTable->numRows() - 1);i++)
		{
			m_pPropertyTable->swapRows(i,i+1);
		}

		m_pPropertyTable->setNumRows(m_pPropertyTable->numRows() - 1);
		if(m_pPropertyTable->numRows() == 0)m_pDelPropertyButton->setEnabled(false);
	}
}

void KviUserEditorWidget::refillData()
{
	int row = 0;
	KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->findUserByName(originalName());
	m_pMaskTable->setNumRows(u->maskList()->count());
	for(KviIrcMask * m = u->maskList()->first();m;m = u->maskList()->next())
	{
		m_pMaskTable->setItem(row,0,new QTableItem(m_pMaskTable,QTableItem::OnTyping,m->nick()));
		m_pMaskTable->setItem(row,1,new QTableItem(m_pMaskTable,QTableItem::OnTyping,m->user()));
		m_pMaskTable->setItem(row,2,new QTableItem(m_pMaskTable,QTableItem::OnTyping,m->host()));
		row++;
	}
	
	if(m_pMaskTable->numRows() == 0)m_pDelMaskButton->setEnabled(false);

	QAsciiDict<KviStr> * pd = u->propertyDict();
	if(pd)
	{
		m_pPropertyTable->setNumRows(pd->count());
		QAsciiDictIterator<KviStr> it(*pd);
		row = 0;
		while(it.current())
		{
			m_pPropertyTable->setItem(row,0,new QTableItem(m_pPropertyTable,QTableItem::OnTyping,it.currentKey()));
			m_pPropertyTable->setItem(row,1,new QTableItem(m_pPropertyTable,QTableItem::OnTyping,it.current()->ptr()));
			++row;
			++it;
		}
	}
	if(m_pPropertyTable->numRows() == 0)m_pDelPropertyButton->setEnabled(false);
	m_pNameEdit->setText(u->name());
}

void KviUserEditorWidget::addPropertyClicked()
{
	m_pPropertyTable->setNumRows(m_pPropertyTable->numRows() + 1);
	m_pPropertyTable->setItem(m_pPropertyTable->numRows() - 1,0,new QTableItem(m_pPropertyTable,QTableItem::OnTyping,""));
	m_pPropertyTable->setItem(m_pPropertyTable->numRows() - 1,1,new QTableItem(m_pPropertyTable,QTableItem::OnTyping,""));
	m_pDelPropertyButton->setEnabled(true);
}

void KviUserEditorWidget::propertyValueChanged(int row, int)
{
	KviStr propname = m_pPropertyTable->text(row,0);
	KviStr prop = m_pPropertyTable->text(row,1);
	if(propname.hasData() && prop.hasData())
	{
		KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->findUserByName(originalName());
		u->setProperty(propname.ptr(),prop.ptr());
		emit changed();
	}
}

void KviUserEditorWidget::maskValueChanged(int,int)
{
	/// AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAARGH!
	/*
	KviStr tmp, nick, user, host;

	KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->findUserByName(originalName());
	KviPtrList<KviIrcMask> * ml = u->maskList();
	if(ml)ml->clear();
	int num = m_pMaskTable->numRows();
	for(int i = 0;i < num;i++)
	{
		nick = m_pMaskTable->text(i,0);
		user = m_pMaskTable->text(i,1);
		host = m_pMaskTable->text(i,2);
		if(nick.hasData())
		{
			if(user.isEmpty())user = "*";
			if(host.isEmpty())host = "*";

			tmp.sprintf("%s!%s@%s",nick.ptr(),user.ptr(),host.ptr());
			KviIrcMask * msk = new KviIrcMask(tmp.ptr());
			ml->append(msk);
		}
	}
	emit changed();
	*/
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
KviNotifyListEditorWidget::KviNotifyListEditorWidget(QWidget * par,const char * name) : QWidget(par,name)
{
	QGridLayout * g = new QGridLayout(this,4,2,2,2);
	
	m_pNickListBox = new QListBox(this);
	connect(m_pNickListBox,SIGNAL(clicked(QListBoxItem *)),this,SLOT(itemClicked(QListBoxItem *)));
	g->addMultiCellWidget(m_pNickListBox,3,5,1,2);
	
	QLabel * l = new QLabel(__tr("Nickname:"),this);
	g->addWidget(l,0,0);

	m_pNickEdit = new QLineEdit(this);
	connect(m_pNickEdit,SIGNAL(returnPressed()),this,SLOT(addClicked()));
	g->addMultiCellWidget(m_pNickEdit,0,0,1,2);		

	l = new QLabel(__tr("Registered user:"),this);
	g->addWidget(l,1,0);

	m_pUserCombo = new QComboBox(this);
	m_pUserCombo->setEditable(true);
	m_pUserCombo->lineEdit()->setText("");
	connect(m_pUserCombo->lineEdit(),SIGNAL(returnPressed()),this,SLOT(addClicked()));
	g->addMultiCellWidget(m_pUserCombo,1,1,1,2);
	

	QFrame * f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(f,2,2,0,2);
	
	m_pAddButton = new QPushButton(__tr("Add nickname"),this);
	connect(m_pAddButton,SIGNAL(clicked()),this,SLOT(addClicked()));
	g->addWidget(m_pAddButton,3,0);

	m_pRemoveButton = new QPushButton(__tr("Remove nickname"),this);
	connect(m_pRemoveButton,SIGNAL(clicked()),this,SLOT(removeClicked()));
	g->addWidget(m_pRemoveButton,4,0);

	refillNotifyList();
}

KviNotifyListEditorWidget::~KviNotifyListEditorWidget()
{
}

void KviNotifyListEditorWidget::addClicked()
{
	KviStr nick = m_pNickEdit->text();
	KviStr notify;
	KviStr newNotify;

	nick.stripWhiteSpace();
	if(nick.isEmpty())return;
	notify=nick;
	
	KviStr findUser = m_pUserCombo->lineEdit()->text();

	KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->findUserByName(findUser.ptr());

	if (u)
	{   //Existing user
		if(u->getProperty("notify",notify)) //Has a notify property;append
		{
			notify.append(" ");
			notify.append(nick);
		}
		u->setProperty("notify",notify.ptr());
	} 
	else 
	{   //No user name: We create a new user with name==nick, if we can
		int idx = -1;
		KviStr tmp = nick;
		KviRegisteredUser * u;

		while(u = g_pLocalRegisteredUserDataBase->findUserByName(tmp.ptr()))
		{
			idx++;
			tmp.sprintf("%s-%d",nick.ptr(),idx);
		}
		
		u = g_pLocalRegisteredUserDataBase->addUser(tmp.ptr());
		u->setProperty("notify",notify.ptr());

		//A default Mask (nick!*@*)
		KviStr tmpMask;
		tmpMask.sprintf("%s!*@*",nick.ptr());
		KviIrcMask * msk = new KviIrcMask(tmpMask.ptr());
		if(!g_pLocalRegisteredUserDataBase->addMask(u,msk))
			debug("KviNotifyListEditor::addClicked: Could not add mask for user!!");
	} 
	refillNotifyList();
	emit changed();
}

void KviNotifyListEditorWidget::removeClicked()
{
	KviStr nick = m_pNickEdit->text();
	KviStr user = m_pUserCombo->lineEdit()->text();
	KviStr origNotify, destNotify, token;
	
	KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->findUserByName(user.ptr());
	
	if(u->getProperty("notify", origNotify))
	{
		bool bFirst=true;
		while(origNotify.getToken(token, ' ')) //The user has more than this notify nick
		{
			if(!kvi_strEqualCI(token,nick))
			{
				if(bFirst)bFirst=false;
				else destNotify.append(" ");
				destNotify.append(token); 
			}
		}
		//Now the last (or the only) token
		if(!kvi_strEqualCI(token,nick))
		{
			if(!bFirst)destNotify.append(" ");
			destNotify.append(token);
		}
	} else 
		debug("KviNotifyListEditorWidget::removeClicked(): Trying to remove an unexistent notify nick!", destNotify.ptr());
	u->setProperty("notify", destNotify);
	refillNotifyList();
	emit changed();
}

void KviNotifyListEditorWidget::refillNotifyList()
{
	
	m_pNickListBox->clear();
	m_pUserCombo->clear();
	KviStr token, notify;

	QAsciiDict<KviRegisteredUser> * d = g_pLocalRegisteredUserDataBase->userDict();
	QAsciiDictIterator<KviRegisteredUser> it(*d);
	while(KviRegisteredUser * u = it.current())
	{
		m_pUserCombo->insertItem(u->name());
		
		if(!u->getProperty("notify",notify))
		{
			++it;
			continue; 		
		}
		if(notify.getToken(token,' '))
		{
			do			
			{	
					m_pNickListBox->insertItem(token.ptr());
			} while(notify.getToken(token,' '));
		}
		//Get the last token (or the only one)
		m_pNickListBox->insertItem(token.ptr());
		++it;
	}
	//m_pUserCombo->lineEdit()->setText("");
	m_pUserCombo->lineEdit()->clear();
	m_pNickEdit->clear();
	m_pNickListBox->clearSelection();
}

void KviNotifyListEditorWidget::itemClicked(QListBoxItem * item)
{
	if(!item)return;
	m_pNickEdit->setText(item->text());

	//We have to find which user the clicked nick belongs to
	QAsciiDict<KviRegisteredUser> * d = g_pLocalRegisteredUserDataBase->userDict();
	QAsciiDictIterator<KviRegisteredUser> it(*d);
	
	KviStr txt = item->text();

	while(KviRegisteredUser * u = it.current())
	{
		KviStr notify, token;
		if(u->getProperty("notify",notify))
		{
			while(notify.getToken(token,' '))
			{	// This user have more than one notified nick 
				if(kvi_strEqualCI(token,txt.ptr())) 
				{ 
						m_pUserCombo->lineEdit()->setText(u->name());
						goto next_user;
				}
			}
			//One nick in the notify
			if (kvi_strEqualCI(token,txt.ptr()))  
					m_pUserCombo->lineEdit()->setText(u->name());
		}
		
next_user:
		++it;
	}
}
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
KviUserEditorIndex::KviUserEditorIndex(QWidget * par,const char * name)
: QWidget(par,name)
{
	QGridLayout * g = new QGridLayout(this,8,2,2,2);

	m_pListView = new QListView(this);
	m_pListView->addColumn(__tr("User"));
	m_pListView->addColumn(__tr("Masks"));
	m_pListView->addColumn(__tr("Properties"));
	m_pListView->setAllColumnsShowFocus(true);
	m_pListView->setSelectionMode(QListView::Extended);

	g->addMultiCellWidget(m_pListView,0,7,0,0);

	m_pAddButton = new QPushButton(__tr("Add"),this);
	g->addWidget(m_pAddButton,0,1);
	connect(m_pAddButton,SIGNAL(clicked()),this,SLOT(addNewUser()));


	m_pRemoveButton = new QPushButton(__tr("Remove"),this);
	g->addWidget(m_pRemoveButton,1,1);
	m_pRemoveButton->setEnabled(false);
	connect(m_pRemoveButton,SIGNAL(clicked()),this,SLOT(removeUser()));

	QFrame * f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addWidget(f,2,1);

	m_pEditButton = new QPushButton(__tr("Edit"),this);
	g->addWidget(m_pEditButton,3,1);
	m_pEditButton->setEnabled(false);
	connect(m_pEditButton,SIGNAL(clicked()),this,SLOT(editUser()));

	f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addWidget(f,4,1);

	m_pExportButton = new QPushButton(__tr("Export selected"),this);
	g->addWidget(m_pExportButton,5,1);
	m_pExportButton->setEnabled(false);
	connect(m_pExportButton,SIGNAL(clicked()),this,SLOT(exportSelected()));

	m_pImportButton = new QPushButton(__tr("Import"),this);
	g->addWidget(m_pImportButton,6,1);
	connect(m_pImportButton,SIGNAL(clicked()),this,SLOT(importFromFile()));


	connect(m_pListView,SIGNAL(selectionChanged()),this,SLOT(selectionChanged()));

	g->setRowStretch(7,1);

	refillUserList();
}

KviUserEditorIndex::~KviUserEditorIndex()
{
}

#define KVI_REGUSER_DB_FILE_MAGIC 0x5334DBDB
#define KVI_REGUSER_DB_FILE_VERSION 1

typedef struct _KviReguserDbFileHeader
{
	unsigned int magic;
	unsigned int version;
	unsigned int nentries;
} KviReguserDbFileHeader;

void KviUserEditorIndex::exportSelected()
{
	unsigned int nEntries = 0;

	QListViewItem * it;

	for(it = m_pListView->firstChild();it;it = it->nextSibling())
	{
		if(it->isSelected())nEntries++;
	}

	if(nEntries < 1)
	{
		kvi_warningBox(__tr("No entries selected"));
		return;
	}

	KviStr buffer;

	if(!KviFileDialog::askForSaveFileName(buffer,__tr("Choose the name of the db file"),0,0,true,true))return;

	KviFile f(buffer.ptr());
	if(!f.open(IO_WriteOnly | IO_Truncate))
	{
		kvi_warningBox(__tr("Can't open the file %s for writing"),buffer.ptr());
		return;
	}

	KviReguserDbFileHeader hf;
	hf.magic = KVI_REGUSER_DB_FILE_MAGIC;
	hf.version = KVI_REGUSER_DB_FILE_VERSION;
	hf.nentries = nEntries;

	if(f.writeBlock((const char *)&hf,sizeof(KviReguserDbFileHeader)) != sizeof(KviReguserDbFileHeader))goto write_error;

	for(it = m_pListView->firstChild();it;it = it->nextSibling())
	{
		if(it->isSelected())
		{
			KviStr szName = it->text(0);
			KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->findUserByName(szName.ptr());
			if(u)
			{
				if(!f.save(szName))goto write_error;
				QAsciiDict<KviStr> * pd = u->propertyDict();
				if(pd)
				{
					if(!f.save(pd->count()))goto write_error;
					QAsciiDictIterator<KviStr> it(*pd);
					while(it.current())
					{
						KviStr key = it.currentKey();
						if(!f.save(key))goto write_error;
						if(!f.save(*(it.current())))goto write_error;
						++it;
					}
				} else {
					if(!f.save(0))goto write_error;
				}

				KviPtrList<KviIrcMask> * ml = u->maskList();
				if(ml)
				{
					if(!f.save(ml->count()))goto write_error;
					for(KviIrcMask * m = ml->first();m;m = ml->next())
					{
						KviStr fullMask;
						m->mask(fullMask);
						if(!f.save(fullMask))goto write_error;
					}
				} else {
					if(!f.save(0))goto write_error;
				}

				KviStr avatar;
				if(u->getProperty("avatar",avatar))
				{
					KviAvatar * av = g_pIconManager->getAvatar(avatar.ptr());
					if(av)
					{
						if(!av->pixmap()->isNull())
						{
							if(!f.save(1))goto write_error;
							QImageIO io;
							io.setImage(av->pixmap()->convertToImage());
							io.setIODevice(&f);
							io.setFormat("PNG");
							if(!io.write())goto write_error;
						} else {
							if(!f.save(0))goto write_error;
							debug("Got null avatar for %s",u->name());
						}
					} else {
						if(!f.save(0))goto write_error;
					}
				} else {
					if(!f.save(0))goto write_error;
				}
			}
		}
	}

	goto succesfull_export;


write_error:
	kvi_warningBox(__tr("Can't export the reguser database: write error"));
	f.close();
	return;

succesfull_export:

	f.close();
}

void KviUserEditorIndex::importFromFile()
{
	KviStr buffer;

	if(!KviFileDialog::askForOpenFileName(buffer,__tr("Choose the name of the db file")))return;

	KviFile f(buffer.ptr());
	if(!f.open(IO_ReadOnly))
	{
		kvi_warningBox(__tr("Can't open the file %s for reading"),buffer.ptr());
		return;
	}

	KviReguserDbFileHeader hf;
	unsigned int idx;

	if(f.readBlock((char *)&hf,sizeof(KviReguserDbFileHeader)) != sizeof(KviReguserDbFileHeader))goto read_error;

	if(hf.magic != KVI_REGUSER_DB_FILE_MAGIC)
	{
		kvi_warningBox(__tr("The file %s doesn't appear to be a valid reguser database"),buffer.ptr());
		f.close();
		return;
	}

	if(hf.version != KVI_REGUSER_DB_FILE_VERSION)
	{
		kvi_warningBox(__tr("The file %s contains an invalid reguser database version"),buffer.ptr());
		f.close();
		return;
	}



	for(idx = 0;idx < hf.nentries;idx++)
	{
		KviStr szName;
		if(!f.load(szName))goto read_error;
		KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->getUser(szName.ptr());
		unsigned int count = 0;
		if(!f.load(count))goto read_error;
		for(unsigned int up = 0;up < count;up++)
		{
			KviStr szKey,szValue;
			if(!f.load(szKey))goto read_error;
			if(!f.load(szValue))goto read_error;
			u->setProperty(szKey.ptr(),szValue.ptr());
		}
		if(!f.load(count))goto read_error;
		for(unsigned int um = 0;um < count;um++)
		{
			KviStr szMask;
			if(!f.load(szMask))goto read_error;
			if(szMask.hasData())
			{
				KviIrcMask * m = new KviIrcMask(szMask.ptr());
				g_pLocalRegisteredUserDataBase->addMask(u,m);
			}
		}
		if(!f.load(count))goto read_error;
		if(count)
		{
			// there is an avatar
			QImageIO io;
			QImage img;
			io.setImage(img);
			io.setIODevice(&f);
			io.setFormat("PNG");

			if(!io.read())goto read_error;

			img = io.image();

			if(img.isNull())debug("Ops.. readed a null image ?");

			KviStr fName = u->name();
			kvi_encodeFileName(fName);

			KviStr fPath;
			int rnm = 0 ;
			do
			{
				g_pApp->getLocalKvircDirectory(fPath,KviApp::Avatars,fName);
				fPath.append(KviStr::Format,"%d.png",rnm);
				rnm++;
			} while(kvi_fileExists(fPath.ptr()));

			if(!img.save(fPath.ptr(),"PNG"))
			{
				debug("Can't save image %s",fPath.ptr());
			} else {
				u->setProperty("avatar",fPath.ptr());
			}
		}
	}

	goto succesfull_import;


read_error:
	kvi_warningBox(__tr("Can't import the reguser database: read error"));
	f.close();
	return;

succesfull_import:

	f.close();
	refillUserList();
	emit changed();
}

void KviUserEditorIndex::selectionChanged()
{
	bool bSelected = false;
	for(QListViewItem * it = m_pListView->firstChild();it;it = it->nextSibling())
	{
		if(it->isSelected())
		{
			bSelected = true;
			break;
		}
	}
	m_pEditButton->setEnabled(bSelected);
	m_pRemoveButton->setEnabled(bSelected);
	m_pExportButton->setEnabled(bSelected);
}

void KviUserEditorIndex::editUser()
{
	for(QListViewItem * it = m_pListView->firstChild();it;it = it->nextSibling())
	{
		if(it->isSelected())
		{
			KviStr tmp = it->text(0);
			g_pRegisteredUserEditor->editUser(tmp.ptr());
		}
	}
}

void KviUserEditorIndex::removeUser()
{
	KviPtrList<KviStr> l;
	l.setAutoDelete(true);
	for(QListViewItem * it = m_pListView->firstChild();it;it = it->nextSibling())
	{
		if(it->isSelected())
		{
			l.append(new KviStr(it->text(0)));
		}
	}
	for(KviStr * s = l.first();s;s = l.next())
	{
		g_pRegisteredUserEditor->removeUser(s->ptr());
	}
	emit changed();
}

void KviUserEditorIndex::addNewUser()
{
	int idx = 0;
	KviStr tmp = __tr("New User 0");

	KviRegisteredUser * u;

	while(u = g_pLocalRegisteredUserDataBase->findUserByName(tmp.ptr()))
	{
		idx++;
		tmp.sprintf(__tr("New User %d"),idx);
	}

	u = g_pLocalRegisteredUserDataBase->addUser(tmp.ptr());

	m_pListView->clearSelection();
	QListViewItem * item = new QListViewItem(m_pListView,tmp.ptr(),"0","0");
	m_pListView->setSelected(item,true);
	emit changed();
}


void KviUserEditorIndex::refillUserList()
{
	m_pListView->clear();

	QAsciiDict<KviRegisteredUser> * d = g_pLocalRegisteredUserDataBase->userDict();
	QAsciiDictIterator<KviRegisteredUser> it(*d);
	QListViewItem * item;
	while(KviRegisteredUser * u = it.current())
	{
		KviStr masks(KviStr::Format,"%d",u->maskList()->count());
		QAsciiDict<KviStr> * pd = u->propertyDict();
		KviStr props(KviStr::Format,"%d",pd ? pd->count() : 0);
		item = new QListViewItem(m_pListView,u->name(),masks.ptr(),props.ptr());
		item->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_LINUX)));
		++it;
	}
	if(m_pListView->firstChild())m_pListView->setSelected(m_pListView->firstChild(),true);

}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

KviUserEditor::KviUserEditor(QWidget * par,const char * name)
: QWidget(par,name)
{
//	g_pRegisteredUserEditor = this;
	g_pLocalRegisteredUserDataBase = new KviRegisteredUserDataBase();
	g_pLocalRegisteredUserDataBase->copyFrom(g_pRegisteredUserDataBase);

	setIcon(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_REGUSERS)));
	setCaption(__tr("Registered users"));


	QGridLayout * g = new QGridLayout(this,2,4,2,2);

	m_pTabWidget = new QTabWidget(this);
	g->addMultiCellWidget(m_pTabWidget,0,0,0,3);

	g->setRowStretch(0,1);

	m_pEditors = new KviPtrList<KviUserEditorWidget>;
	m_pEditors->setAutoDelete(false);

	m_pIndex = new KviUserEditorIndex(m_pTabWidget);

	m_pTabWidget->addTab(m_pIndex,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_REGUSERS)),__tr("Registered users"));

	m_pNotify = new KviNotifyListEditorWidget(m_pTabWidget);
	connect(m_pNotify,SIGNAL(changed()),m_pIndex,SLOT(refillUserList()));	
	connect(m_pIndex,SIGNAL(changed()),m_pNotify,SLOT(refillNotifyList()));

	m_pTabWidget->addTab(m_pNotify,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_NOTIFYONLINE)),__c2q(__tr("Notify list")));	

	QPushButton * b = new QPushButton(__c2q(__tr("What is this ?")),this);
//#warning "What is this button! (help)"
	b->setEnabled(false);
	g->addWidget(b,1,1);

	b = new QPushButton(__c2q(__tr("Cancel")),this);
	g->addWidget(b,1,2);
	connect(b,SIGNAL(clicked()),this,SLOT(cancelClicked()));

	b = new QPushButton(__tr("Ok"),this);
	g->addWidget(b,1,3);
	connect(b,SIGNAL(clicked()),this,SLOT(okClicked()));

	if(!parent())setGeometry(KVI_OPTION_RECT(KviOption_rectRegisteredUsersDialogGeometry));
}

KviUserEditor::~KviUserEditor()
{
	if(!parent())KVI_OPTION_RECT(KviOption_rectRegisteredUsersDialogGeometry) = QRect(pos().x(),pos().y(),
			size().width(),size().height());

	while(m_pEditors->first())delete m_pEditors->first();
	delete m_pEditors;
	g_pRegisteredUserEditor = 0;
	delete g_pLocalRegisteredUserDataBase;
	g_pLocalRegisteredUserDataBase = 0;
}

void KviUserEditor::cancelClicked()
{
	delete this;
}

void KviUserEditor::okClicked()
{
	while(KviUserEditorWidget * e = m_pEditors->first())
	{
		e->finishClicked();
		delete e;
	}
//#warning "Update avatars ?"
	g_pRegisteredUserDataBase->copyFrom(g_pLocalRegisteredUserDataBase);
	g_pApp->restartNotifyLists();
	delete this;
}

void KviUserEditor::closeEvent(QCloseEvent *e)
{
	e->ignore();
	delete this;
}

KviUserEditorWidget * KviUserEditor::findUserPage(const char * uName)
{
	for(KviUserEditorWidget * w = m_pEditors->first();w;w = m_pEditors->next())
	{
		if(kvi_strEqualCI(w->originalName(),uName))return w;
	}
	return 0;
}

void KviUserEditor::registerEditor(KviUserEditorWidget * w)
{
	m_pTabWidget->addTab(w,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_LINUX)),w->originalName());
	m_pEditors->append(w);
}

void KviUserEditor::unregisterEditor(KviUserEditorWidget *w)
{
	m_pTabWidget->removePage(w);
	m_pEditors->removeRef(w);
	m_pTabWidget->showPage(m_pIndex);
}

void KviUserEditor::editUser(const char * uName,bool bAddIfNonExisting)
{
	KviUserEditorWidget * w = findUserPage(uName);
	if(!w)
	{
		KviRegisteredUser * u = g_pLocalRegisteredUserDataBase->findUserByName(uName);
		if(!u)
		{
			if(!bAddIfNonExisting)return;
			u = g_pLocalRegisteredUserDataBase->addUser(uName);
			m_pIndex->refillUserList();
			m_pNotify->refillNotifyList();
		}
	
		w = new KviUserEditorWidget(m_pTabWidget,u);

		// Ensure everything is in sync and happy ;^)
		connect(w,SIGNAL(changed()),m_pIndex,SLOT(refillUserList()));
		connect(w,SIGNAL(changed()),m_pNotify,SLOT(refillNotifyList()));		
		connect(m_pIndex,SIGNAL(changed()),w,SLOT(refillData()));
		connect(m_pNotify,SIGNAL(changed()),w,SLOT(refillData()));
	}
	m_pTabWidget->showPage(w);
}

void KviUserEditor::removeUser(const char * uName)
{
	KviUserEditorWidget * w = findUserPage(uName);
	if(w)delete w; // remove the editor! (if there)
	g_pLocalRegisteredUserDataBase->removeUser(uName);
	m_pIndex->refillUserList();
	m_pNotify->refillNotifyList();
}

#include "edituser.moc"
