//
//   File : optw_servers.cpp
//   Creation date : Wed Jun 12 2000 03:29:51 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "optw_servers.h"
#include "kvi_query.h"
#include "kvi_channel.h"
#include "kvi_locale.h"
#include "kvi_iconmanager.h"
#include "kvi_ircserverdb.h"
#include "kvi_ipeditor.h"
#include "kvi_netutils.h"
#include "kvi_settings.h"
#include "kvi_filedialog.h"
#include "kvi_config.h"
#include "kvi_scripteditor.h"
#include "kvi_options.h"
#include "kvi_uparser.h"
#include "kvi_app.h"
#include "kvi_console.h"
#include "kvi_ircurl.h"

#include <qtoolbutton.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qtooltip.h>
#include <qmessagebox.h>
#include <qhbox.h>


KviServerDetailsWidget::KviServerDetailsWidget(QWidget * par,KviIrcServer * s)
: QDialog(par,"server_details",true)
{
	QGridLayout * g = new QGridLayout(this,12,3,8,2);

	setCaption(__tr("Server details"));
	setIcon(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SERVER)));

	KviStr tmp(KviStr::Format,__tr("Details for server %s"),s->m_szHostname.ptr());

	QLabel * l = new QLabel(tmp.ptr(),this);
	g->addMultiCellWidget(l,0,0,0,2);



	l = new QLabel(__c2q(__tr("Description:")),this);
	g->addWidget(l,1,0);
	m_pDescEditor = new QLineEdit(this);
	g->addMultiCellWidget(m_pDescEditor,1,1,1,2);
	m_pDescEditor->setText(s->description());
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pDescEditor,__c2q(__tr("<center>This is a brief description of the server: " \
				"this field has no restrictions but many server lists use it to describe the server physical location</center>")));
#endif


	l = new QLabel(__tr("Ip address:"),this);
	g->addWidget(l,2,0);
	m_pIpEditor = new KviIpEditor(this,KviIpEditor::IpV4);
	g->addMultiCellWidget(m_pIpEditor,2,2,1,2);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pIpEditor,__c2q(__tr("<center>This is the <b>IP address</b> of the server, it is here for caching purposes.<br>" \
				"If you leave this field empty, KVIrc will fill it the first time that it connects to the server. " \
				"If you enable the \"cache Ip address\" option below, KVirc will use it as a \"cached result\" " \
				"and avoid looking it up again.</center>")));
#endif

#ifdef COMPILE_IPV6_SUPPORT
	m_pIpEditor->setAddressType(s->isIpV6() ? KviIpEditor::IpV6 : KviIpEditor::IpV4);
#else
	m_pIpEditor->setAddressType(KviIpEditor::IpV4);
#endif

	if(!m_pIpEditor->setAddress(s->m_szIp.ptr()))
	{
#ifdef COMPILE_IPV6_SUPPORT
		m_pIpEditor->setAddress(s->isIpV6() ? "0:0:0:0:0:0:0:0" : "0.0.0.0");
#else
		m_pIpEditor->setAddress("0.0.0.0");
#endif
	}


	m_pCacheIpCheck = new QCheckBox(__c2q(__tr("Cache IP address")),this);
	g->addMultiCellWidget(m_pCacheIpCheck,3,3,0,2);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pCacheIpCheck,__c2q(__tr("<center>This check will enable <b>IP address caching</b> for this server:<br>" \
				"DNS lookups can be time consuming and might be blocking on some platforms; " \
				"this option will cause KViIrc to lookup the server hostname only once.</center>")));
#endif
	m_pCacheIpCheck->setChecked(s->cacheIp());

	m_pUseSSLCheck = new QCheckBox(__c2q(__tr("Use SSL")),this);
	g->addMultiCellWidget(m_pUseSSLCheck,4,4,0,2);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pCacheIpCheck,__c2q(__tr("<center>This check will cause the connection to use the <b>Secure Socket Layer<b><br>" \
				"support. Obviously the server must have the support too :)</center>")));
#endif
#ifndef COMPILE_SSL_SUPPORT
	m_pUseSSLCheck->setEnabled(false);
#endif
	m_pUseSSLCheck->setChecked(s->useSSL());

	l = new QLabel(__c2q(__tr("Username:")),this);
	g->addWidget(l,5,0);
	m_pUserEditor = new QLineEdit(this);
	g->addMultiCellWidget(m_pUserEditor,5,5,1,2);
	m_pUserEditor->setText(s->user());
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pUserEditor,__c2q(__tr("<center>This is the <b>username</b> that KVIrc will use to login to this server.\n" \
				"If you leave this field empty (most common case), KVIrc will use the default username " \
				"specified in the \"Identity\" options tab.</center>")));
#endif

	l = new QLabel(__c2q(__tr("Password:")),this);
	g->addWidget(l,6,0);
	m_pPassEditor = new QLineEdit(this);
	g->addMultiCellWidget(m_pPassEditor,6,6,1,2);
	m_pPassEditor->setText(s->pass());
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pPassEditor,__c2q(__tr("<center>If the server requires a password, put it in this field, otherwise leave it empty.</center>")));
#endif

	l = new QLabel(__tr("Nickname:"),this);
	g->addWidget(l,7,0);
	m_pNickEditor = new QLineEdit(this);
	g->addMultiCellWidget(m_pNickEditor,7,7,1,2);
	m_pNickEditor->setText(s->nick());
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pNickEditor,__c2q(__tr("<center>You can specify here a \"special\" <b>nickname</b> that can be used to login with this server.<br>" \
				"If you leave this field empty (most common case), the default nickname (specified in the \"Identity\" tab) will be used.</center>")));
#endif

	l = new QLabel(__c2q(__tr("Real Name:")),this);
	g->addWidget(l,8,0);
	m_pRealEditor = new QLineEdit(this);
	g->addMultiCellWidget(m_pRealEditor,8,8,1,2);
	m_pRealEditor->setText(s->realName());
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pRealEditor,__c2q(__tr("<center>You can specify here a \"special\" <b>real name</b> that can be used to login with this server.<br>" \
				"If you leave this field empty (most common case), the default \"real name\" (specified in the \"Identity\" tab) will be used.</center>")));
#endif

	l = new QLabel(__c2q(__tr("On connect execute:")),this);
	g->addMultiCellWidget(l,9,9,0,2);
	m_pCommandEditor = kvi_createScriptEditor(this);
	g->addMultiCellWidget(m_pCommandEditor,10,10,0,2);
	m_pCommandEditor->setText(s->onConnectCommand());

	m_pCommandEditor->setMinimumHeight(150);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(l,__c2q(__tr("<center>The following commands will be executed after the connection has been started.<br>" \
				"KVirc will first send the USER command , then eventually PASS and NICK and then execute this<br>" \
				"command sequence.<br>This is particularly useful with bouncers</center>")));
#endif




	QPushButton * b = new QPushButton(__c2q(__tr("Cancel")),this);
	g->addWidget(b,11,1);
	connect(b,SIGNAL(clicked()),this,SLOT(reject()));
	b = new QPushButton(__tr("OK"),this);
	b->setDefault(true);
	g->addWidget(b,11,2);
	connect(b,SIGNAL(clicked()),this,SLOT(accept()));

	g->setRowStretch(10,1);

	setMinimumWidth(300);
}

KviServerDetailsWidget::~KviServerDetailsWidget()
{
}

void KviServerDetailsWidget::fillData(KviIrcServer * s)
{
	s->m_szUser = m_pUserEditor->text();
	s->m_szPass = m_pPassEditor->text();
	s->m_szNick = m_pNickEditor->text();
	s->m_szRealName = m_pRealEditor->text();
	s->m_szDescription = m_pDescEditor->text();
	s->m_szIp = "";

	s->setCacheIp(m_pCacheIpCheck->isChecked());
	s->setUseSSL(m_pUseSSLCheck->isChecked());

	KviStr tmpAddr = m_pIpEditor->address();

	if(!m_pIpEditor->hasEmptyFields())
	{
#ifdef COMPILE_IPV6_SUPPORT
		if(s->isIpV6())
		{
			if((!kvi_strEqualCI(tmpAddr.ptr(),"0:0:0:0:0:0:0:0")) && kvi_isValidStringIp_V6(tmpAddr.ptr()))
			{
				s->m_szIp = tmpAddr;
			} else {
				s->setCacheIp(false);
				s->m_szIp = "";
			}
		} else {
#endif
			if((!kvi_strEqualCI(tmpAddr.ptr(),"0.0.0.0")) && kvi_isValidStringIp(tmpAddr.ptr()))
			{
				s->m_szIp = tmpAddr;
			} else {
				s->setCacheIp(false);
				s->m_szIp = "";
			}
#ifdef COMPILE_IPV6_SUPPORT
		}
#endif
	} else {
		s->setCacheIp(false);
		s->m_szIp = "";
	}

	QCString tmp;
	m_pCommandEditor->getText(tmp);
	s->m_szOnConnectCommand = tmp.data();
}

void KviServerDetailsWidget::closeEvent(QCloseEvent *e)
{
	e->ignore();
	reject();
}


// kvi_app.cpp
extern KVIRC_API KviIrcServerDataBase * g_pIrcServerDataBase;

KviServerOptionsListViewItem::KviServerOptionsListViewItem(QListView *parent,const QPixmap &pm,const char * netname)
: QListViewItem(parent,QString(netname))
{
	setPixmap(0,pm);
	m_pServerData = 0;
}

KviServerOptionsListViewItem::KviServerOptionsListViewItem(QListViewItem *parent,const QPixmap &pm,KviIrcServer *s)
: QListViewItem(parent)
{
	setPixmap(0,pm);
	m_pServerData = new KviIrcServer(*s);
	setText(0,s->m_szHostname.ptr());
	//debug("Desc: %s",s->m_szDescription.ptr());
	setText(1,s->m_szDescription.ptr());
}

KviServerOptionsListViewItem::~KviServerOptionsListViewItem()
{
	if(m_pServerData)delete m_pServerData;
}









KviServerOptionsWidget::KviServerOptionsWidget(QWidget * parent)
: KviOptionsWidget(parent,"server_options_widget")
{
	createLayout(6,3);

	m_pDetailsDialog = 0;

	m_pListView = new QListView(this);
	layout()->addMultiCellWidget(m_pListView,0,0,0,2);
	m_pListView->addColumn(__c2q(__tr("Network/Server")));
	m_pListView->addColumn(__c2q(__tr("Description")));
	m_pListView->setRootIsDecorated(true);
	m_pListView->setAllColumnsShowFocus(true);
	m_pListView->setSelectionMode(QListView::Single);
	connect(m_pListView,SIGNAL(selectionChanged(QListViewItem *)),
		this,SLOT(listViewItemSelectionChanged(QListViewItem *)));
	connect(m_pListView,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),
		this,SLOT(listViewRightButtonPressed(QListViewItem *,const QPoint &,int)));

#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pListView,__c2q(__tr("<center>This is the list of available IRC servers.<br>" \
				"Right click on the list to add or remove servers and for other options</center>")));
#endif

	m_pSrvNetLabel = new QLabel(__tr("Server:"),this);
	layout()->addWidget(m_pSrvNetLabel,1,0);

	m_pSrvNetEdit = new QLineEdit(this);
	layout()->addWidget(m_pSrvNetEdit,1,1);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pSrvNetEdit,__c2q(__tr("<center>This is the <b>hostname</b> of the currently selected server.</center>")));
#endif


	m_pIpV6Check = new QCheckBox(__c2q(__tr("Uses IpV6 protocol")),this);

	layout()->addWidget(m_pIpV6Check,1,2);
#ifndef COMPILE_IPV6_SUPPORT
	m_pIpV6Check->setEnabled(false);
#endif

#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pIpV6Check,__c2q(__tr("<center>This check identifies IPV6 servers.<br>If enabled, KVIrc will attempt to use the IPV6 protocol " \
				"(thus your kernel <b>must have</b> a working IPV6 stack and you <b>must have</b> an IPV6 connection).</center>")));
#endif


	m_pPortLabel = new QLabel(__tr("Port:"),this);
	layout()->addWidget(m_pPortLabel,2,0);

	m_pPortEdit = new QLineEdit(this);
	layout()->addWidget(m_pPortEdit,2,1);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pPortEdit,__c2q(__tr("<center>This is the default <b>port</b> that this server will be contacted on.<br>Usually <b>6667</b> is OK.</center>")));
#endif

	m_pDetailsButton = new QPushButton(__tr("Advanced ..."),this);
	connect(m_pDetailsButton,SIGNAL(clicked()),this,SLOT(detailsClicked()));
	layout()->addWidget(m_pDetailsButton,2,2);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pDetailsButton,__c2q(__tr("<center>Click here to edit advanced options for this server...</center>")));
#endif

	addSeparator(0,3,2,3);

	QHBox * box = new QHBox(this);
	layout()->addMultiCellWidget(box,4,4,0,2);


	m_pConnectCurrent = new QPushButton(__c2q(__tr("Connect in current irc context")),box);
	connect(m_pConnectCurrent,SIGNAL(clicked()),this,SLOT(connectCurrentClicked()));

#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pConnectCurrent,__c2q(__tr("<center>Hit this button to connect to the currently selected server in the current irc context</center>")));
#endif

	m_pConnectNew = new QPushButton(__c2q(__tr("Connect in new irc context")),box);
	connect(m_pConnectNew,SIGNAL(clicked()),this,SLOT(connectNewClicked()));

#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pConnectNew,__c2q(__tr("<center>Hit this button to connect to the currently selected server in a new irc context.<br>(as a new additional connection)</center>")));
#endif

	m_pRecentPopup = new QPopupMenu(this);
	connect(m_pRecentPopup,SIGNAL(aboutToShow()),this,SLOT(recentServersPopupAboutToShow()));
	connect(m_pRecentPopup,SIGNAL(activated(int)),this,SLOT(recentServersPopupClicked(int)));

	QToolButton * tb = new QToolButton(DownArrow,box);
	tb->setPopup(m_pRecentPopup);
	tb->setPopupDelay(1);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(tb,__c2q(__tr("<center>This button shows a list of recently used servers, it allows you to quickly find them in the list</center>")));
#endif


	box->setStretchFactor(m_pConnectCurrent,10);
	box->setStretchFactor(m_pConnectNew,10);
	box->setStretchFactor(tb,0);

	KviBoolSelector * b = addBoolSelector(0,5,2,5,__tr("Show the servers dialog on startup"),KviOption_boolShowServersConnectDialogOnStart);

#ifdef COMPILE_INFO_TIPS
	QToolTip::add(b,__c2q(__tr("<center>If you mark this check , you will see this dialog popping up every time you start KVIrc</center>")));
#endif

//	KviBoolSelector * c = addBoolSelector(0,6,2,6,__tr("Close after connection"),KviOption_boolCloseServerWidgetAfterConnect);

	m_pLastEditedItem = 0;

	m_pClipboard = 0;

	fillServerList();

	layout()->setRowStretch(0,1);
	layout()->setColStretch(1,1);

	m_pContextPopup = new QPopupMenu(this);
}

KviServerOptionsWidget::~KviServerOptionsWidget()
{
	if(m_pClipboard)delete m_pClipboard;
	if(m_pDetailsDialog)delete m_pDetailsDialog;
}

void KviServerOptionsWidget::recentServersPopupAboutToShow()
{
	g_pApp->fillRecentServersPopup(m_pRecentPopup);
}

void KviServerOptionsWidget::recentServersPopupClicked(int id)
{
	KviStr data = m_pRecentPopup->text(id);
	data.cutToFirst('/');
	while(data.firstCharIs('/'))data.cutLeft(1);
	data.cutFromLast(':');
	// we should have a full server name now , with no port

	for(QListViewItem * net = m_pListView->firstChild();net;net = net->nextSibling())
	{
		for(QListViewItem * srv = net->firstChild();srv;srv = srv->nextSibling())
		{
			if(kvi_strEqualCI(((KviServerOptionsListViewItem *)srv)->m_pServerData->hostname(),data.ptr()))
			{
				net->setOpen(true);
				m_pListView->setCurrentItem(srv);
				m_pListView->ensureItemVisible(srv);
				return;
			}
		}
	}
	// FIXME: add the server if it isn't there ?
}

/*
void KviServerOptionsWidget::parseCommand(const KviStr & szCommand,KviStr & szServer,KviStr & szPort)
{
    KviStr szServerAndPort = szCommand;
    szServerAndPort.cutLeft(7); // "server"
    szServerAndPort.cutRight(1); // ";"
    bool bOk;
    int port = szServerAndPort.right(5).toUShort(&bOk);
    if(bOk)
    {
        szServer = szServerAndPort.cutRight(5);
        szPort.sprintf("%d",port);
    }
}
*/
/*
void KviServerOptionsWidget::itemRecentClicked(QListBoxItem * it)
{
//	saveLastItem();
	m_pLastEditedItem = 0;

	m_pSrvNetLabel->setEnabled(false);
	m_pSrvNetEdit->setEnabled(false);
	m_pDetailsButton->setEnabled(false);
	m_pPortLabel->setEnabled(false);
	m_pPortEdit->setEnabled(false);
	m_pIpV6Check->setEnabled(false); 
	m_pConnectCurrent->setEnabled(true);
	m_pConnectNew->setEnabled(true);
	
	m_pSrvNetLabel->setText(__tr("Server:"));
	//FIXME XXX: Put the IPV6 check in the correct state
	

	if (it == 0)return;
	m_pListView->clearSelection(); 
	KviStr text = it->text();
	KviStr szServer = "";
	KviStr szPort = "";
	if(text.hasData())
	{
		KviStr szCommand;
		if(parseIrcUrl(text.ptr(),szCommand,KVI_IRCURL_CONTEXT_THIS))
		{
			parseCommand(szCommand,szServer,szPort);
			m_pSrvNetEdit->setText(szServer.ptr());
			m_pPortEdit->setText(szPort.ptr());
		}
	}
}
		
*/
/*
void KviServerOptionsWidget::itemRecentDoubleClicked(QListBoxItem * it)
{
    if (it == 0)return;
    m_pListView->clearSelection();
    KviStr text = it->text();
    KviStr szServer = "";
    KviStr szPort = "";
    if(text.hasData())
    {
        KviStr szCommand;
        if(parseIrcUrl(text.ptr(),szCommand,KVI_IRCURL_CONTEXT_THIS))
        {
            parseCommand(szCommand,szServer,szPort);
            m_pSrvNetEdit->setText(szServer.ptr());
            m_pPortEdit->setText(szPort.ptr());
            g_pUserParser->parseCommandBuffer(szCommand.ptr(),g_pApp->activeWindow());
        }
    }
}
*/

void KviServerOptionsWidget::connectCurrentClicked()
{
	saveLastItem();
	commit();
	KviStr szServer = m_pSrvNetEdit->text();
//	KviStr szPort = m_pPortEdit->text();
	if (szServer.isEmpty())return;
	
//	KviStr command;
//	command.sprintf("server %s %s",szServer.ptr(),szPort.ptr());
	g_pUserParser->parseCommandBuffer("server",g_pApp->activeConsole());
//  FIXME: How to destroy this when is a toplevel widget?	
//	if (KVI_OPTION_BOOL(KviOption_boolCloseServerWidgetAfterConnect))delete this;
}

void KviServerOptionsWidget::connectNewClicked()
{
	saveLastItem();
	commit();
	KviStr szServer = m_pSrvNetEdit->text();
//	KviStr szPort = m_pPortEdit->text();
	if (szServer.isEmpty())return;
	
//	KviStr command;
//	command.sprintf("server -n",szServer.ptr(),szPort.ptr());
	g_pUserParser->parseCommandBuffer("server -n",g_pApp->activeConsole());
}

void KviServerOptionsWidget::fillServerList()
{
	KviServerOptionsListViewItem * net;
	KviServerOptionsListViewItem * srv;

	KviPtrList<KviIrcNetwork> * l = g_pIrcServerDataBase->networkList();

	for(KviIrcNetwork * n = l->first();n;n = l->next())
	{
		net = new KviServerOptionsListViewItem(m_pListView,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_WORLD)),n->name());
		KviPtrList<KviIrcServer> * sl = n->serverList();
		net->setOpen(n == g_pIrcServerDataBase->currentNetwork());
		//debug("Current server = %s",n->currentServer() ? n->currentServer()->m_szHostname.ptr() : "non");
		for(KviIrcServer * s = sl->first();s;s = sl->next())
		{
			srv = new KviServerOptionsListViewItem(net,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SERVER)),s);
			
			if((s == n->currentServer()) && (n == g_pIrcServerDataBase->currentNetwork()))
			{
				m_pListView->setSelected(srv,true);
				m_pListView->ensureItemVisible(srv);
			}
		}
	}
}

void KviServerOptionsWidget::listViewItemSelectionChanged(QListViewItem *it)
{
//	m_pRecentListBox->clearSelection();
	saveLastItem();
	m_pLastEditedItem = (KviServerOptionsListViewItem *)it;

	if(m_pLastEditedItem)
	{
		m_pSrvNetLabel->setEnabled(true);
		m_pSrvNetEdit->setEnabled(true);
		m_pDetailsButton->setEnabled(m_pLastEditedItem->m_pServerData);
		m_pPortLabel->setEnabled(m_pLastEditedItem->m_pServerData);
		m_pPortEdit->setEnabled(m_pLastEditedItem->m_pServerData);
#ifdef COMPILE_IPV6_SUPPORT
		m_pIpV6Check->setEnabled(m_pLastEditedItem->m_pServerData);
#else
		m_pIpV6Check->setEnabled(false);
#endif
		m_pConnectCurrent->setEnabled(m_pLastEditedItem->m_pServerData);
		m_pConnectNew->setEnabled(m_pLastEditedItem->m_pServerData);

		if(m_pLastEditedItem->m_pServerData)
		{
			m_pSrvNetLabel->setText(__tr("Server:"));
			m_pSrvNetEdit->setText(m_pLastEditedItem->m_pServerData->m_szHostname.ptr());
			KviStr tmp(KviStr::Format,"%u",m_pLastEditedItem->m_pServerData->m_uPort);
			m_pPortEdit->setText(tmp.ptr());
#ifdef COMPILE_IPV6_SUPPORT
			m_pIpV6Check->setChecked(m_pLastEditedItem->m_pServerData->isIpV6());
#else
			m_pIpV6Check->setChecked(false);
#endif

		} else {
			m_pSrvNetLabel->setText(__c2q(__tr("Network:")));
			m_pSrvNetEdit->setText(it->text(0));
			m_pPortEdit->setText("");
		}
	} else {
		m_pConnectCurrent->setEnabled(false);
		m_pConnectNew->setEnabled(false);
		m_pIpV6Check->setEnabled(false);
		m_pSrvNetLabel->setEnabled(false);
		m_pSrvNetEdit->setEnabled(false);
		m_pSrvNetEdit->setText(__tr("No selection"));
		m_pDetailsButton->setEnabled(false);
		m_pPortLabel->setEnabled(false);
		m_pPortEdit->setEnabled(false);
		m_pPortEdit->setText("");
	}
}



void KviServerOptionsWidget::saveLastItem()
{
	if(m_pLastEditedItem)
	{
		if(m_pLastEditedItem->m_pServerData)
		{
			KviStr tmp = m_pSrvNetEdit->text();
			if(tmp.isEmpty())tmp = "irc.unknown.net";
			m_pLastEditedItem->setText(0,tmp.ptr());
			m_pLastEditedItem->m_pServerData->m_szHostname = tmp;
			m_pLastEditedItem->setText(1,m_pLastEditedItem->m_pServerData->m_szDescription.ptr());
			tmp = m_pPortEdit->text();
			bool bOk;
			unsigned short int uPort = tmp.toUShort(&bOk);
			if(!bOk)uPort = 6667;
			m_pLastEditedItem->m_pServerData->m_uPort = uPort;
#ifdef COMPILE_IPV6_SUPPORT
			m_pLastEditedItem->m_pServerData->setIpV6(m_pIpV6Check->isChecked());
#else
			m_pLastEditedItem->m_pServerData->setIpV6(false);
#endif
		} else {
			KviStr tmp = m_pSrvNetEdit->text();
			if(tmp.isEmpty())tmp = __c2q(__tr("UnknownNet"));
			m_pLastEditedItem->setText(0,tmp.ptr());
		}
	}
}

void KviServerOptionsWidget::commit()
{
	//debug("COMMITTING");
	saveLastItem();
	g_pIrcServerDataBase->clear();
	KviServerOptionsListViewItem * it = (KviServerOptionsListViewItem *)m_pListView->firstChild();
	while(it)
	{
		KviStr tmp = it->text(0);
		if(tmp.hasData())
		{
			KviIrcNetwork * net = g_pIrcServerDataBase->findNetwork(tmp.ptr());
			if(!net)
			{
				net = new KviIrcNetwork(tmp.ptr());
				g_pIrcServerDataBase->insertNetwork(net);
			}
			if(it == m_pLastEditedItem)g_pIrcServerDataBase->setCurrentNetwork(net);

			KviServerOptionsListViewItem * ch = (KviServerOptionsListViewItem *)it->firstChild();
			KviIrcServer *srv;
			while(ch)
			{
				if(ch->m_pServerData)
				{
					if(ch->m_pServerData->m_szHostname.hasData())
					{
						srv = net->findServer(ch->m_pServerData->m_szHostname.ptr());
						if(!srv)
						{
							srv = new KviIrcServer(*(ch->m_pServerData));
							net->insertServer(srv);
						} else *srv = *(ch->m_pServerData);
						if(ch == m_pLastEditedItem)
						{
							g_pIrcServerDataBase->setCurrentNetwork(net);
							net->setCurrentServer(srv);
						}
					}
				}
				ch = (KviServerOptionsListViewItem *)ch->nextSibling();
			}
	
			if(net->currentServer() == 0)net->setCurrentServer(net->serverList()->first());
		}
		it = (KviServerOptionsListViewItem *)it->nextSibling();
	}
	if(g_pIrcServerDataBase->currentNetwork() == 0)
	{
		g_pIrcServerDataBase->setCurrentNetwork(g_pIrcServerDataBase->networkList()->first());
	}

	KviOptionsWidget::commit();
}

void KviServerOptionsWidget::listViewRightButtonPressed(QListViewItem *it,const QPoint &pnt,int col)
{
	m_pContextPopup->clear();
	m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_WORLD)),__c2q(__tr("Add new network")),this,SLOT(newNetwork()));
	if(it)
	{
		if(((KviServerOptionsListViewItem *)it)->m_pServerData)
		{
			m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_CUT)),__c2q(__tr("Remove sever")),this,SLOT(removeCurrent()));
			m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_COPY)),__c2q(__tr("Copy server")),this,SLOT(copyServer()));
		} else {
			m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_CUT)),__c2q(__tr("Remove network")),this,SLOT(removeCurrent()));
		}
		m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SERVER)),__c2q(__tr("Add new server")),this,SLOT(newServer()));
		if(m_pClipboard)m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_PASTE)),__c2q(__tr("Paste server")),this,SLOT(pasteServer()));
	}
	m_pContextPopup->insertSeparator();
	m_pContextPopup->insertItem(__c2q(__tr("Merge list from server.ini")),this,SLOT(importFromIni()));
	m_pContextPopup->insertItem(__c2q(__tr("Clear list")),this,SLOT(clearList()));
	m_pContextPopup->popup(QCursor::pos());
}

void KviServerOptionsWidget::newNetwork()
{
	KviServerOptionsListViewItem * it = new KviServerOptionsListViewItem(m_pListView,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_WORLD)),__tr("New Network"));
	it->setOpen(true);
	m_pListView->setSelected(it,true);
	m_pListView->ensureItemVisible(it);
}

void KviServerOptionsWidget::newServer()
{
	if(m_pLastEditedItem)
	{
		KviServerOptionsListViewItem * net;
		if(m_pLastEditedItem->m_pServerData)
		{
			net = (KviServerOptionsListViewItem *)m_pLastEditedItem->parent();
			if(!net)return;
		} else net = m_pLastEditedItem;

		KviIrcServer tmp;
		tmp.m_szHostname = __tr("irc.unknown.net");
		tmp.setCacheIp(true);

		KviServerOptionsListViewItem * it = new KviServerOptionsListViewItem(net,
			*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SERVER)),&tmp);

		net->setOpen(true);

		m_pListView->setSelected(it,true);
		m_pListView->ensureItemVisible(it);
	}
}

void KviServerOptionsWidget::copyServer()
{
	if(m_pLastEditedItem)
	{
		if(m_pLastEditedItem->m_pServerData)
		{
			if(!m_pClipboard)m_pClipboard = new KviIrcServer;
			*m_pClipboard = *(m_pLastEditedItem->m_pServerData);
		}
	}	
}

void KviServerOptionsWidget::pasteServer()
{
	if(m_pClipboard)
	{
		if(m_pLastEditedItem)
		{
			KviServerOptionsListViewItem * net;
			if(m_pLastEditedItem->m_pServerData)
			{
				net = (KviServerOptionsListViewItem *)m_pLastEditedItem->parent();
				if(!net)return;
			} else net = m_pLastEditedItem;
	
			KviServerOptionsListViewItem * it = new KviServerOptionsListViewItem(net,
				*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SERVER)),m_pClipboard);
	
			net->setOpen(true);
	
			m_pListView->setSelected(it,true);
			m_pListView->ensureItemVisible(it);
		}
	}
}

void KviServerOptionsWidget::removeCurrent()
{
	if(m_pLastEditedItem)
	{
		delete m_pLastEditedItem;
		m_pLastEditedItem = 0;
		QListViewItem * it = m_pListView->firstChild();
		if(it)
		{
			m_pListView->setSelected(it,true);
			m_pListView->ensureItemVisible(it);
		}
	}
}

void KviServerOptionsWidget::importFromIni()
{
	KviStr buffer;
	if(KviFileDialog::askForOpenFileName(buffer,__tr("Choose a servers.ini file"),0,"*.ini"))
	{
		importFromIni(buffer.ptr());
	}
}

void KviServerOptionsWidget::importFromIni(const char * filename)
{
	saveLastItem();

	KviConfig cfg(filename);
	if(cfg.hasGroup("servers"))
	{
		cfg.setGroup("servers");
		int i = 0;
		KviStr key;
		KviStr entry;
		do {
			key.sprintf("n%d",i);
			entry = cfg.readEntry(key.ptr(),"");
			if(entry.hasData())
			{
				KviStr description;
				KviStr server;
				KviStr port;
				unsigned short int uPort;
				// <description>SERVER:<server:port>GROUP:<network>
				int idx = entry.findFirstIdx("SERVER:");
				if(idx != -1)
				{
					description = entry.left(idx);
					entry.cutLeft(idx + 7);
					idx = entry.findFirstIdx("GROUP:");
					if(idx != -1)
					{
						port = entry.left(idx);
						entry.cutLeft(idx + 6);
					}
					idx = port.findFirstIdx(':');
					if(idx != -1)
					{
						server = port.left(idx);
						port.cutLeft(idx + 1);
						bool bOk;
						uPort = port.toUInt(&bOk);
						if(!bOk)uPort = 6667;
					} else {
						server = port;
						uPort = 6667;
					}
				}
				if(entry.isEmpty())entry = "orphan_servers";
				if(server.hasData())
				{
					KviIrcServer s;
					s.m_szHostname = server;
					s.m_szDescription = description;
					s.m_uPort = uPort;

					KviServerOptionsListViewItem * net = findNetItem(entry.ptr());
					if(!net)net = new KviServerOptionsListViewItem(m_pListView,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_WORLD)),entry.ptr());

					KviServerOptionsListViewItem * srv = new KviServerOptionsListViewItem(net,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SERVER)),&s);
				}
				++i;
			}
		} while(entry.hasData());
	} else {
		KviStr tmp(KviStr::Format,__tr("%s doesn't look like a servers.ini file..."),filename);
		QMessageBox::warning(0,__c2q(__tr("KVIrc : Warning")),__c2q(tmp.ptr()));
	}
}

KviServerOptionsListViewItem * KviServerOptionsWidget::findNetItem(const char * netname)
{
	KviServerOptionsListViewItem * it = (KviServerOptionsListViewItem *)m_pListView->firstChild();
	while(it)
	{
		KviStr szNetName = it->text(0);
		if(kvi_strEqualCI(szNetName.ptr(),netname))return it;
		it = (KviServerOptionsListViewItem *)it->nextSibling();
	}
	return 0;
}

void KviServerOptionsWidget::clearList()
{
	m_pListView->clear();
	m_pLastEditedItem = 0;
	listViewItemSelectionChanged(0);
}

void KviServerOptionsWidget::detailsClicked()
{
	if(!m_pLastEditedItem)return;
	if(!(m_pLastEditedItem->m_pServerData))return;

	saveLastItem();

	m_pDetailsDialog = new KviServerDetailsWidget(this,m_pLastEditedItem->m_pServerData);

	int retCode = m_pDetailsDialog->exec();
	if(retCode == QDialog::Accepted)
	{
		if(m_pLastEditedItem)
		{
			if(m_pLastEditedItem->m_pServerData)
			{
				m_pDetailsDialog->fillData(m_pLastEditedItem->m_pServerData);
				saveLastItem();
			}
		}
	}


	delete m_pDetailsDialog;
	m_pDetailsDialog = 0;
}

#include "m_optw_servers.moc"
