//
//   File : optw_nickserv.cpp
//   Creation date : Fri Aug 10 2001 03:38:10 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "optw_nickserv.h"

#include "kvi_settings.h"

#include <qlayout.h>
#include <qtooltip.h>
#include <qasciidict.h>

#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_nickserv.h"
#include "kvi_ircmask.h"

// kvi_app.cpp
extern KVIRC_API KviNickServDataBase * g_pNickServDataBase;


KviNickServOptionsWidget::KviNickServOptionsWidget(QWidget * parent)
: KviOptionsWidget(parent,"nickserv_options_widget")
{
	m_pCurrentItem = 0;

	m_pContextPopup = new QPopupMenu(this);

	createLayout(8,2);

	m_pEnableSelector = addBoolSelector(0,0,1,0,__tr("Enable automatic NickServ identification"),KviOption_boolEnableAutomaticNickServIdentification);

	connect(m_pEnableSelector,SIGNAL(toggled(bool)),this,SLOT(enableToggled(bool)));

	addSeparator(0,1,1,1);

	m_pListViewLabel = addLabel(0,2,1,2,__tr("NickServ identification rules:"));

	m_pListView = new QListView(this);
	m_pListView->addColumn(__tr("Nick"));
	m_pListView->addColumn(__tr("Server"));
	m_pListView->addColumn(__tr("NickServ mask"));
	m_pListView->addColumn(__tr("Identify command"));
	m_pListView->setAllColumnsShowFocus(true);
	m_pListView->setSelectionMode(QListView::Single);

	connect(m_pListView,SIGNAL(mouseButtonPressed(int,QListViewItem *,const QPoint &,int)),this,SLOT(itemClicked(int,QListViewItem *,const QPoint &,int)));
	connect(m_pListView,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(selectionChanged(QListViewItem *)));

	addWidgetToLayout(m_pListView,0,3,1,3);

#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pListView,
		__tr("<center>This is a list of NickServ identification rules." \
			"KVIrc will use them to model its automatic interaction with NickServ.<br>" \
			"Please be aware that this feature can cause your NickServ passwords to be stolen," \
			"if you use it without care: please follow the suggestions in the tooltips and " \
			"make sure that you fully understand the NickServ authentication protocol.<br>" \
			"In other words: be sure to know what you're doing.<br>" \
			"Also note that the password that you provide is stored as PLAIN TEXT in the KVIrc " \
			"configuration file.<br>" \
			"Right click to add/remove rule lines.</center>"));
#endif

	m_pListView->setEnabled(KVI_OPTION_BOOL(KviOption_boolEnableAutomaticNickServIdentification));

	m_pNicknameLabel = addLabel(0,4,0,4,__tr("Nickname"),false);
	m_pNicknameEdit = addLineEdit(1,4,1,4,false);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pNicknameEdit,
		__tr("<center>This is the nickname that this rule will attempt to authenticate with NickServ:" \
			"e.g. the nickname that you have registered with NickServ</center>"));
#endif

	m_pServerMaskLabel = addLabel(0,5,0,5,__tr("Server"),false);
	m_pServerMaskEdit = addLineEdit(1,5,1,5,false);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pServerMaskEdit,
		__tr("<center>This is a mask that defines the server(s) that this identification rule will work on." \
			"For a single server, this option will be just something as <b>irc.funet.fi</b>." \
			"You can use wildcards to match multiple servers: <b>*.openprojects.net</b> will work on the whole OpenProjects network." \
			"(It matches all the server names that end in \".openprojects.net\").<br>" \
			"Please note that some servers have more than one name: for example , irc.eu.dal.net refers a \"random\" " \
			"DalNet server, but the real server name will be actually different. You must provide the name that " \
			"the server will use to \"identify\" himself. You can look up this name by using the SocketSpy.</center>"));
#endif


	m_pNickServMaskLabel = addLabel(0,6,0,6,__tr("NickServ mask"),false);
	m_pNickServMaskEdit = addLineEdit(1,6,1,6,false);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pNickServMaskEdit,
		__tr("<center>This is the mask that NickServ must match to be correctly identified as NickServ service."  \
			"This usually will be something like <b>NickServ!service@services.dalnet</b>..." \
			"You can use wildcards for this field, but generally it is a security flaw." \
			"If you're 100% sure that NO user on the network can change his nickname to NickServ then " \
			"probably <b>NickServ!*@*</b> will be ok for this field.</center>"));
#endif

	m_pIdentifyCommandLabel = addLabel(0,7,0,7,__tr("Identify command"),false);
	m_pIdentifyCommandEdit = addLineEdit(1,7,1,7,false);
#ifdef COMPILE_INFO_TIPS
	QToolTip::add(m_pIdentifyCommandEdit,
		__tr("<center>This is the command that will be executed when the NickServ will request authentication " \
			"for the nickname described in this rule (and the server and NickServ mask match)." \
			"This will be usually something as <b>msg NickServ identify &lt;yourpassword&gt;</b>." \
			"You can use \"msg -q\" if you don't want the password to be echoed on the screen." \
			"Please note that there is no leading slash in this command.</center>"));
#endif

	layout()->setRowStretch(3,1);



	QListViewItem * item = 0;

	QAsciiDictIterator<KviNickServEntryList> it(*g_pNickServDataBase->entryDict());
	while(KviNickServEntryList * l = it.current())
	{
		for(KviNickServEntry * e = l->first();e;e = l->next())
		{
			item = new QListViewItem(m_pListView,it.currentKey(),e->szServerMask.ptr(),e->szNickServMask.ptr(),e->szIdentifyCommand.ptr());
		}
		++it;
	}
}

KviNickServOptionsWidget::~KviNickServOptionsWidget()
{
}

void KviNickServOptionsWidget::enableToggled(bool)
{
	enableItems();
}

void KviNickServOptionsWidget::enableItems()
{
	bool b = m_pEnableSelector->isChecked();
	m_pListView->setEnabled(b);
	m_pListViewLabel->setEnabled(b);
	m_pNicknameEdit->setEnabled(b && m_pCurrentItem);
	m_pNickServMaskEdit->setEnabled(b && m_pCurrentItem);
	m_pServerMaskEdit->setEnabled(b && m_pCurrentItem);
	m_pIdentifyCommandEdit->setEnabled(b && m_pCurrentItem);
	m_pNicknameLabel->setEnabled(b && m_pCurrentItem);
	m_pNickServMaskLabel->setEnabled(b && m_pCurrentItem);
	m_pServerMaskLabel->setEnabled(b && m_pCurrentItem);
	m_pIdentifyCommandLabel->setEnabled(b && m_pCurrentItem);
}

void KviNickServOptionsWidget::saveLastItem()
{
	if(!m_pCurrentItem)return;
	KviStr tmp = m_pNicknameEdit->text();
	if(tmp.isEmpty())tmp = "???";
	m_pCurrentItem->setText(0,tmp.ptr());

	tmp = m_pServerMaskEdit->text();
	if(tmp.isEmpty())tmp = "???";
	m_pCurrentItem->setText(1,tmp.ptr());

	tmp = m_pNickServMaskEdit->text();
	KviIrcMask mask(tmp.ptr());
	tmp.sprintf("%s!%s@%s",mask.nick(),mask.user(),mask.host());
	m_pCurrentItem->setText(2,tmp.ptr());

	m_pCurrentItem->setText(3,m_pIdentifyCommandEdit->text());
}

void KviNickServOptionsWidget::selectionChanged(QListViewItem * it)
{
	saveLastItem();
	m_pCurrentItem = it;
	enableItems();
	if(it)
	{
		m_pNicknameEdit->setText(it->text(0));
		m_pServerMaskEdit->setText(it->text(1));
		m_pNickServMaskEdit->setText(it->text(2));
		m_pIdentifyCommandEdit->setText(it->text(3));
	} else {
		m_pNicknameEdit->setText("");
		m_pServerMaskEdit->setText("");
		m_pNickServMaskEdit->setText("");
		m_pIdentifyCommandEdit->setText("");
	}
}

void KviNickServOptionsWidget::itemClicked(int btn,QListViewItem * it,const QPoint &pnt,int)
{
	if(btn & RightButton)
	{
		m_pContextPopup->clear();
		m_pContextPopup->insertItem(__tr("New rule"),this,SLOT(newRule()));
		if(it)m_pContextPopup->insertItem(__tr("Delete this rule"),this,SLOT(deleteRule()));

		// context
		m_pContextPopup->popup(pnt);
	}
}


void KviNickServOptionsWidget::commit()
{
	saveLastItem();
	g_pNickServDataBase->clear();

	for(QListViewItem * it = m_pListView->firstChild();it;it = it->nextSibling())
	{
		KviStr szNick = it->text(0);
		KviStr szServ = it->text(1);
		KviStr szNSer = it->text(2);
		KviStr szICmd = it->text(3);
		g_pNickServDataBase->insertEntry(szNick.ptr(),szServ.ptr(),szNSer.ptr(),szICmd.ptr());
	}
	KviOptionsWidget::commit();
}

void KviNickServOptionsWidget::newRule()
{
	saveLastItem();
	QListViewItem * it = new QListViewItem(m_pListView,"YourNick","irc.unknown.net","NickServ!s@services.unknown.net","msg NickServ IDENTIFY yourpassword");
	m_pListView->setCurrentItem(it);
	m_pListView->setSelected(it,true);
}

void KviNickServOptionsWidget::deleteRule()
{
	if(m_pCurrentItem)
	{
		QListViewItem * it = m_pCurrentItem;
		m_pCurrentItem = 0;
		delete it;
	}
	if(!m_pCurrentItem)
	{
		QListViewItem * f = m_pListView->firstChild();
		if(f)
		{
			m_pListView->setSelected(f,true);
			m_pListView->setCurrentItem(f);
		} else selectionChanged(0);
	}
}

#include "m_optw_nickserv.moc"
