//
//   File : libkvioptions.cpp
//   Creation date : Fri Aug 18 2000 15:04:09 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_options.h"
#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_window.h"
#include "kvi_locale.h"
#include "kvi_mirccntrl.h"
#include "kvi_out.h"

#include "container.h"
#include "instances.h"

#include "dlg_options.h"

#include <qsplitter.h>


KviGeneralOptionsDialog * g_pGeneralOptionsDialog = 0;
KviOptionsInstanceManager * g_pOptionsInstanceManager = 0;

extern int g_iOptionWidgetInstances;

/*
	@doc: options.save
	@type:
		command
	@title:
		options.save
	@short:
		Saves the options to disk
	@syntax:
		options.save
	@description:
		Saves the options to disk; this includes aliases , popups ,
		events and user preferences.
*/


static bool options_module_cmd_save(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"options_module_cmd_save");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
	g_pApp->saveOptions();
	return c->leaveContext();
}


/*
	@doc: options.dialog
	@type:
		command
	@title:
		options.dialog
	@short:
		Shows the "options dialog"
	@syntax:
		options.dialog [-t]
	@description:
		Shows the KVIrc "general options dialog". If the dialog
		is already open it is simply reparented to the current
		frame.[br]
		If the [-t] switch is used , the dialog is opened as toplevel window,
		otherwise it is opened as part of the current frame window.[br]
		This command is exported by the "options" module.
	@seealso:
		[fnc]$options.isdialog[/fnc]
*/


static bool options_module_cmd_dialog(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"options_module_cmd_dialog");

	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	if(g_pGeneralOptionsDialog)
	{
		if(c->hasSwitch('t'))
		{
			if(g_pGeneralOptionsDialog->parent())
			{
				g_pGeneralOptionsDialog->reparent(0,QPoint(0,0),true);
			}
		} else {
			if(g_pGeneralOptionsDialog->parent() != c->window()->frame()->splitter())
			{
				g_pGeneralOptionsDialog->reparent(c->window()->frame()->splitter(),QPoint(0,0),true);
			}
		}
	} else {
		if(c->hasSwitch('t'))
		{
			g_pGeneralOptionsDialog = new KviGeneralOptionsDialog(0);
		} else {
			g_pGeneralOptionsDialog = new KviGeneralOptionsDialog(c->window()->frame()->splitter());
		}
	}

	g_pGeneralOptionsDialog->show();
	g_pGeneralOptionsDialog->raise();
	g_pGeneralOptionsDialog->setFocus();

	return c->leaveContext();
}


/*
	@doc: options.pages
	@type:
		command
	@title:
		options.pages
	@short:
		Lists the option pages
	@syntax:
		options.pages
	@description:
		Lists the option pages available for editing by the means of [cmd]options.edit[/cmd].
	@seealso:
*/


static bool options_module_cmd_pages(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"options_module_cmd_pages");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	KviPtrList<KviOptionsWidgetInstanceEntry> * l = g_pOptionsInstanceManager->instanceEntryTree();

	for(KviOptionsWidgetInstanceEntry * e = l->first();e;e = l->next())
	{
		c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr("Page %c%s%c  (%s)"),KVI_TEXT_BOLD,e->szClassName,KVI_TEXT_BOLD,e->szName);
	}

	return c->leaveContext();
}


/*
	@doc: options.edit
	@type:
		command
	@title:
		options.edit
	@short:
		Shows a single options page
	@syntax:
		options.edit [-t] <"options page class name">
	@description:
		Shows an options page as toplevel dialog (if -t is used) or
		as a docked window.[br]
		The available option pages can be listed by using [cmd]option.pages[/cmd].
	@seealso:
*/


static bool options_module_cmd_edit(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"options_module_cmd_edit");

	KviStr clName;
	if(!g_pUserParser->parseCmdFinalPart(c,clName))return false;

	KviOptionsWidgetInstanceEntry * e = g_pOptionsInstanceManager->findInstanceEntry(clName.ptr());
	if(!e)
	{
		c->warning(__tr("No such options page class name"));
		return c->leaveContext();
	}

	if(e->pWidget)
	{
		c->warning(__tr("The editor page is already open"));
		return c->leaveContext();
	}

	KviOptionsWidgetContainer * wc = new KviOptionsWidgetContainer(c->hasSwitch('t') ? 0 : c->window()->frame()->splitter());

	wc->setup(g_pOptionsInstanceManager->getInstance(e,wc));

	wc->show();
	wc->raise();

	return c->leaveContext();
}

/*
	@doc: options.isdialog
	@type:
		function
	@title:
		$options.isdialog
	@short:
		Returns the options dialog state
	@syntax:
		$options.isdialog
	@description:
		Returns '1' if the general options dialog is open, '0' otherwise.[br]
		This command is exported by the "options" module.
	@seealso:
		[cmd]options.dialog[/cmd]
*/

static bool options_module_fnc_isdialog(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_CONTEXT(c,"options_module_fnc_isdialog");
	buffer.append(g_pGeneralOptionsDialog ? '1' : '0');
	return c->leaveContext();
}

static bool options_module_init(KviModule * m)
{
	g_pOptionsInstanceManager = new KviOptionsInstanceManager();

	m->registerCommand("dialog",options_module_cmd_dialog);
	m->registerCommand("save",options_module_cmd_save);
	m->registerFunction("isDialog",options_module_fnc_isdialog);
	m->registerCommand("pages",options_module_cmd_pages);
	m->registerCommand("edit",options_module_cmd_edit);
	return true;
}

static bool options_module_cleanup(KviModule *m)
{
//	debug("CLEANUP ROUTINE CALLED");

	if(g_pGeneralOptionsDialog)delete g_pGeneralOptionsDialog;

	g_pOptionsInstanceManager->cleanup(m);

	delete g_pOptionsInstanceManager;

//	debug("DELETED");

	m->unregisterMetaObject("KviMessageListView");
	m->unregisterMetaObject("KviServerDetailsWidget");
	m->unregisterMetaObject("KviOptionsInstanceManager");
	m->unregisterMetaObject("KviOptionsWidgetContainer");

	return true;
}

static bool options_module_can_unload(KviModule *m)
{
	return ((!g_pGeneralOptionsDialog) && (g_iOptionWidgetInstances == 0));
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Options",                                              // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2000 Szymon Stefanek (stefanek@tin.it)", // author & (C)
	"Options dialog",
	options_module_init,
	options_module_can_unload,
	0,
	options_module_cleanup
};
