//
//   File : libkvioffer.cpp
//   Creation date : Wed Sep 27 2000 20:59:02 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_filetrader.h"
#include "kvi_ircmask.h"
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "kvi_command.h"
#include "kvi_out.h"
#include "kvi_mirccntrl.h"
#include "kvi_window.h"
#include "kvi_frame.h"

#include "offerwindow.h"

#include <time.h>
#include <qasciidict.h>

extern KVIRC_API KviFileTrader * g_pFileTrader;

KviOfferWindow * g_pOfferWindow = 0;

/*
	@doc: offer.window
	@type:
		command
	@title:
		offer.window
	@short:
		Opens the offer list window
	@syntax:
		offer.window
	@description:
		Opens the offer list window.
	@seealso:
		[cmd]offer.list[/cmd]
*/

static bool offer_module_cmd_window(KviModule *,KviCommand *c)
{
	ENTER_CONTEXT(c,"offer_module_cmd_window");

	KviStr doc;
	if(!g_pUserParser->parseCmdFinalPart(c,doc))return false;


	if(g_pOfferWindow)g_pOfferWindow->delayedAutoRaise();
	else {
		g_pOfferWindow = new KviOfferWindow(c->window()->frame());
		c->window()->frame()->addWindow(g_pOfferWindow);
	}

	return c->leaveContext();
}

/*
	@doc: offer.add
	@type:
		command
	@title:
		offer.add
	@keyterms:
		trading files by dcc
	@short:
		Adds a file offer
	@syntax:
		offer.add [-t=<timeout>] [-n=<visible name>] <filename> [user_mask]
	@description:
		Adds <filename> to the list of the active offers.
		The users will be able to request the file via [cmd]dcc.get[/cmd].
		If [user_mask] is specified , the access to the file is limited
		to the users that match this mask, otherwise the mask will
		be automatically set to '*!*@*'.[br]
		If the 't' switch is used, the offer will be removed after
		<timeout> seconds. If the 'n' switch is used, the offer
		will be visible to the oter users as <visible name> instead
		of the real <filename> (stripped of the leading path).
		<filename> must be an absolute path.
	@seealso:
		[cmd]offer.remove[/cmd], [cmd]offer.list[/cmd],
		[cmd]dcc.get[/cmd]
*/

static bool offer_module_cmd_add(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"offer_module_cmd_add");

	KviStr filename;
	KviStr usermask;
	if(!g_pUserParser->parseCmdSingleToken(c,filename))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,usermask))return false;

	if(filename.isEmpty())
	{
		c->warning(__tr("No filename specified"));
		return c->leaveContext();
	}

	if(!kvi_fileIsReadable(filename.ptr()))
	{
		c->warning(__tr("The file '%s' is not readable"),filename.ptr());
		return c->leaveContext();
	}

	if(usermask.isEmpty())usermask="*!*@*";
	KviIrcMask u(usermask.ptr());
	u.mask(usermask);

	int timeout = 0;

	if(c->hasSwitch('t'))
	{
		KviStr tmo;
		c->getSwitchValue('t',tmo);
		bool bOk;
		timeout = tmo.toInt(&bOk);
		if(!bOk)
		{
			c->warning(__tr("Invalid timeout, ignoring"));
			timeout = 0;
		}
	}

	KviStr vName = filename;
	vName.cutToLast('/');

	if(c->hasSwitch('n'))
	{
		KviStr tmp;
		c->getSwitchValue('n',tmp);
		if(tmp.isEmpty())
		{
			c->warning(__tr("Invalid visible name: using default"));
		} else vName = tmp;
	}

	if(!g_pFileTrader->addOffer(vName.ptr(),filename.ptr(),usermask.ptr(),timeout))
	{
		c->warning(__tr("Ops..failed to add the offer..."));
	}

	return c->leaveContext();
}

/*
	@doc: offer.remove
	@type:
		command
	@title:
		offer.remove
	@keyterms:
		trading files by dcc
	@short:
		Removes a file offer
	@syntax:
		offer.remove <visible name> <user mask> [filesize]
	@description:
		Removes the file offer that matches <visible name> and <user mask>.
		If [filesize] is specified, then it must be matched by the entry
		to be removed.
	@seealso:
		[cmd]offer.add[/cmd], [cmd]offer.list[/cmd], [cmd]offer.clear[/cmd]
*/

static bool offer_module_cmd_remove(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"offer_module_cmd_remove");
	KviStr vName,uMask,fSize;
	if(!g_pUserParser->parseCmdSingleToken(c,vName))return false;
	if(!g_pUserParser->parseCmdSingleToken(c,uMask))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,fSize))return false;
	unsigned int uSize = 0;
	if(fSize.hasData())
	{
		bool bOk;
		uSize = fSize.toUInt(&bOk);
		if(!bOk)
		{
			c->warning(__tr("The specified file size is invalid"));
			return c->leaveContext();
		}
	}
	if(!g_pFileTrader->removeOffer(vName.ptr(),uMask.ptr(),uSize))
		c->warning(__tr("No offer with visible name '%s' and user mask '%s'"),vName.ptr(),uMask.ptr());
	return c->leaveContext();
}

/*
	@doc: offer.clear
	@type:
		command
	@title:
		offer.clear
	@keyterms:
		trading files by dcc, offer list
	@short:
		Clears the offer list
	@syntax:
		offer.clear
	@description:
		Clears the offer list
	@seealso:
		[cmd]offer.add[/cmd], [cmd]offer.list[/cmd], [cmd]offer.remove[/cmd]
*/

static bool offer_module_cmd_clear(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"offer_module_cmd_clear");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
	g_pFileTrader->clear();
	return c->leaveContext();
}

/*
	@doc: offer.list
	@type:
		command
	@title:
		offer.list
	@keyterms:
		trading files by dcc
	@short:
		Lists the active file offers
	@syntax:
		offer.list
	@description:
		Lists the active file offers.
	@seealso:
		[cmd]offer.add[/cmd], [cmd]offer.remove[/cmd]
*/

static bool offer_module_cmd_list(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"offer_module_cmd_list");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	QAsciiDictIterator<KviFileOfferList> it(*(g_pFileTrader->offerListDict()));

	int idx = 0;

	while(KviFileOfferList * l = it.current())
	{
		for(KviFileOffer * o = l->first();o;o = l->next())
		{
			c->window()->output(KVI_OUT_NONE,__tr("%c%d. %s"),
				KVI_TEXT_BOLD,idx + 1,it.currentKey());
			c->window()->output(KVI_OUT_NONE,__tr("    File: %s (%u bytes)"),
				o->absFilePath(),o->fileSize());
			c->window()->output(KVI_OUT_NONE,__tr("    Mask: %s"),
				o->userMask());
			if(o->expireTime() > 0)
			{
				int secs = ((int)(o->expireTime())) - ((int)(time(0)));
				int hour = secs / 3600;
				secs = secs % 3600;
				int mins = secs / 60;
				secs = secs % 60;
				c->window()->output(KVI_OUT_NONE,__tr("    Expires in %d hours %d minutes %d seconds"),
					hour,mins,secs);
			}
			++idx;
		}
		++it;
	}

//#warning "FIND A BETTER KVI_OUT_*"

	if(idx == 0)c->window()->outputNoFmt(KVI_OUT_NONE,__tr("No active file offers"));
	else c->window()->output(KVI_OUT_NONE,__tr("Total: %d offers"),idx);

	return c->leaveContext();
}

static bool offer_module_init(KviModule * m)
{
	m->registerCommand("window",offer_module_cmd_window);
	m->registerCommand("add",offer_module_cmd_add);
	m->registerCommand("remove",offer_module_cmd_remove);
	m->registerCommand("list",offer_module_cmd_list);
	m->registerCommand("clear",offer_module_cmd_clear);
	return true;
}

static bool offer_module_can_unload(KviModule *m)
{
	return (!g_pOfferWindow);
}

static bool offer_module_cleanup(KviModule *m)
{
	if(g_pOfferWindow)delete g_pOfferWindow;
	return true;
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Offer",                                                // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2000 Szymon Stefanek (stefanek@tin.it)", // author & (C)
	"User interface to the file trading system (file offer list)",
	offer_module_init,
	offer_module_can_unload,
	0,
	offer_module_cleanup
};
