//vim: ts=8
//   File : class_urllabel.cpp
//   Creation date : Thu Feb 18 1:27:44 CEST 2001 by Krzysztof Godlewski
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Krzysztof Godlewski
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "class_urllabel.h"

#include "kvi_debug.h"
#include "kvi_error.h"
#include "kvi_command.h"
#include "kvi_app.h"
#include "kvi_uparser.h"

#warning "Functions to change colors for different states"
#warning "$setUnderline(), $underline()"

extern KviUserParser * g_pUserParser;
extern KviApp * g_pApp;

static KviScriptObjectClass * g_pUrlLabelClass = 0;

static KviScriptObject * urlLabelClassCreateInstance(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n)
{
	return new KviScriptUrlLabel(c, p, n);
}

KviScriptUrlLabel::KviScriptUrlLabel(KviScriptObjectClass * c, \
KviScriptObject * p, const char *n) : KviScriptLabel(c, p, n)
{
	m_normalClr = QColor("black");
	m_enterClr = QColor("blue");
	m_activeClr = QColor("red");
	m_useSingleClick = true;
//	((QLabel *)widget())->setAutoResize(true);
	//widget()->setCursor(Qt::pointingHandCursor);
	//((KviScriptWidget * )p)->widget()->setCursor(Qt::pointingHandCursor);
}

KviScriptUrlLabel::~KviScriptUrlLabel()
{
}

bool KviScriptUrlLabel::eventFilter(QObject * o, QEvent * e)
{
	QPalette pal = ((QLabel *)widget())->palette();

	switch(e->type())
	{
		case QEvent::Enter:
			pal.setColor(QColorGroup::Foreground, m_enterClr);
			break;
		case QEvent::Leave:
			// this doesn't work...
			if(((QMouseEvent *)e)->state() & QEvent::LeftButton)
				pal.setColor(QColorGroup::Foreground, \
					m_activeClr);
			else
				pal.setColor(QColorGroup::Foreground, \
					m_normalClr);
			break;
		case QEvent::MouseButtonRelease:
			pal.setColor(QColorGroup::Foreground, m_enterClr);
			// perform actions here
			if(!m_lastClickWasDbl && !m_useSingleClick)
				break;
			if(!m_action.isEmpty())
			{
				g_pUserParser->parseUserCommand( m_action, \
					g_pApp->activeWindow());
					
			}
			break;
		case QEvent::MouseButtonPress:
			m_lastClickWasDbl = false;
			if(!((QMouseEvent * )e)->button() & \
			  QEvent::LeftButton || !m_useSingleClick)
				break;
			pal.setColor(QColorGroup::Foreground, m_activeClr);
			break;
		case QEvent::MouseButtonDblClick:
			m_lastClickWasDbl = true;
			if(m_useSingleClick)
				break;
			pal.setColor(QColorGroup::Foreground, m_activeClr);
			break;
	}
	
	((QLabel *)widget())->setPalette(pal);

	return false;
}

#warning "Docs for urllabel"

/*
	@doc:	urllabel
	@keyterms:
		display url, link
	@title:
		urllabel class
	@type:
		class
	@short:
		Displays an URI allowing to perform actions on click
	@inherits:
		[class]object[/class]
		[class]widget[/class]
		[class]label[/class]
	@description:
		This widget allows you to bind specific action on click. It can
		open an url, or execute any KVI++ code. The text color reacts on
		cursor movements and clicks.
	@functions:
		!fn: $setUrl(<text>)
		Sets the url to be opened when the links is activated. The way
		the label opens the url is defined in KVirc config dialog.
		!fn: $url()
		Returns the currently set url.
		!fn: $setAction(<text>)
		Sets the action to be performed when the link is triggered. It
		can be any valid KVI++ code. Note that all KVirc commands must
		be prefixed with a '/', just as if you typed them in the input
		widget.
		!fn: $action()
		Returns currently set action string.
		!fn: $setText(<text>)
		Sets the label's text. Reimplemented from internal reasons.
		!fn: $setCursorChange(<bool>)
		Tells the label wheather to change or not the shape of cursor,
		when it is above the widget. Default is false.
		!fn: $cursorChange()
		Returns true if the cursor changes over the label, false if not.
		!fn: $setUseSingleClick(<bool>)
		If <bool> is true, the label will react on single clicks,
		otherwise only double clicks will trigger the link.
		!fn: $useSingleClick()
		Returns true if the label reacts on single clicks, false if it
		reacts only on double clicks.
*/

#define urlLabelFuncReg(__nam, __func) \
	g_pUrlLabelClass->registerFunctionHandler(__nam, \
	(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptUrlLabel::__func)),\
	0, true);

void KviScriptUrlLabel::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController-> \
		lookupClass("label");
	__range_valid(base);

	g_pUrlLabelClass = new KviScriptObjectClass(base, "urllabel",
		urlLabelClassCreateInstance, true);

	urlLabelFuncReg("setUrl", functionSetUrl);
	urlLabelFuncReg("url", functionUrl);
	urlLabelFuncReg("setAction", functionSetAction);
	urlLabelFuncReg("action", functionAction);
	urlLabelFuncReg("setText", functionSetText);
	urlLabelFuncReg("setCursorChange", functionSetCursorChange);
	urlLabelFuncReg("cursorChange", functionCursorChange);
	urlLabelFuncReg("setUseSingleClick", functionSetUseSingleClick);
	urlLabelFuncReg("useSingleClick", functionUseSingleClick);
}

void KviScriptUrlLabel::unregisterSelf()
{
	delete g_pUrlLabelClass;
}

bool KviScriptUrlLabel::functionSetUrl(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(!widget())
		return true;

	ENTER_CONTEXT(c, "urllabel::setUrl");

	KviStr * url = p->first();
	if(url->isEmpty())
		return c->error(KviError_notEnoughParameters);

	m_url = url->ptr();
	return c->leaveContext();
}

bool KviScriptUrlLabel::functionUrl(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	b.append(m_url);
	return true;
}

bool KviScriptUrlLabel::functionSetAction(KviCommand * c, KviParameterList * p,\
	KviStr & b)
{
	if(!widget())
		return true;

	ENTER_CONTEXT(c, "urllabel::setAction");

	KviStr * act = p->first();
	if(act->isEmpty())
		return c->error(KviError_notEnoughParameters);

	m_action = act->ptr();
	return c->leaveContext();
}

bool KviScriptUrlLabel::functionAction(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	b.append(m_action);
	return true;
}


bool KviScriptUrlLabel::functionSetText(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
	{
		if(p->first())
			((QLabel *)widget())->setText(p->first()->ptr());
		else
			((QLabel *)widget())->setText("[!-NULL-!]");
	}

	((QLabel *)widget())->setFixedSize(((QLabel *)widget())->sizeHint());

	return true;
}

bool KviScriptUrlLabel::functionSetCursorChange(KviCommand * c, \
	KviParameterList *p, KviStr & b)
{
	if(!widget())
		return true;

	if(m_changeCursor = p->getBool())
		widget()->setCursor(Qt::pointingHandCursor);
	else
		widget()->setCursor(Qt::arrowCursor);

	return true;
}

bool KviScriptUrlLabel::functionCursorChange(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	b.append((m_changeCursor ? '1' : '0'));
	return true;
}

bool KviScriptUrlLabel::functionSetUseSingleClick(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	m_useSingleClick = p->getBool();
	return true;
}

bool KviScriptUrlLabel::functionUseSingleClick(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	b.append((m_useSingleClick ? '1' : '0'));
	return true;
}

#include "m_class_urllabel.moc"
