//vim: ts=8
//   File : class_multilineedit.cpp
//   Creation date : Mon Feb 19 00:45:34 CET by Krzysztof Godlewski
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_debug.h"
#include "kvi_scriptobject.h"
#include "kvi_locale.h"

#include "class_multilineedit.h"

static KviScriptObjectClass * g_pMultiLineEditClass = 0;

static KviScriptObject * multiLineEditClassCreateInstance( \
	KviScriptObjectClass * c, KviScriptObject * p, const char * n)
{
	return new KviScriptMLEdit(c, p, n);
}

KviScriptMLEdit::KviScriptMLEdit(KviScriptObjectClass * c, KviScriptObject * p,\
	const char * n) : KviScriptWidget(c, p, n)
{
}

KviScriptMLEdit::~KviScriptMLEdit()
{
}

#warning "docs !"

/*
	@doc:	multilineedit
	@keyterms:
		multilineedit object class, line editor, input
	@title:
		multilineedit class
	@type:
		class
	@short:
		An editor for inputting larger portions of text
	@inherits:
		[class]object[/class]
		[class]widget[/class]
	@description:
		This is class an editor for inputting larger portions of text.
		Besides that, it has unfinished docs.
	@functions:
		!fn: $textLine(<line>)
		Returns the string at line <line>. Note that it can be an empty
		string.
		!fn: $insertLine(<text>, <line>)
		Inserts <text> at line number <line>. If line is less than
		zero, or larger than the number of rows, the new text is
		put at the end. If txt contains newline characters, several
		lines are inserted. The <line> parameter is optional. If it is
		ommited, the line is put at end.
		!fn: $removeLine(<line>)
		Removes line number <line>.
		!fn: $insertAt(<txt>, <line>, <col>)
		Inserts txt at line number line, after character number col in
		the line. If txt contains newline characters, new lines are
		inserted.
		!fn: $numLines()
		Returns number of lines in the widget.
		!fn: $setCursorPosition(<line>, <col>, <mark>)
		Sets the cursor position to character number col in line
		number line. The parameters are adjusted to lie within the
		legal range.[br]
		If mark is FALSE, the selection is cleared. otherwise it
		is extended
		!fn: $cursorPosition()
		Returns current cursor position in "<line>, <col>" format.

*/

void KviScriptMLEdit::init()
{
	setObject(new QMultiLineEdit(parentScriptWidget(), name()), true);
}

#define mlEditFuncReg(__nam, __fnc) \
	g_pMultiLineEditClass->registerFunctionHandler(__nam, \
        	(KviScriptObjectFunctionHandlerProc)( \
		KVI_PTR2MEMBER(KviScriptMLEdit::__fnc)), 0, true);

void KviScriptMLEdit::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController-> \
		lookupClass("widget");
	__range_valid(base);

	g_pMultiLineEditClass = new KviScriptObjectClass(base, "multilineedit",
		multiLineEditClassCreateInstance, true);

	mlEditFuncReg("textLine", functionTextLine);
	mlEditFuncReg("insertLine", functionInsertLine);
	mlEditFuncReg("removeLine", functionRemoveLine);
	mlEditFuncReg("insertAt", functionInsertAt);
	mlEditFuncReg("numLines", functionNumLines);
	mlEditFuncReg("setCursorPosition", functionSetCursorPosition);
	mlEditFuncReg("cursorPosition", functionCursorPosition);
	mlEditFuncReg("atBeginning", functionAtBeginning);
	mlEditFuncReg("atEnd", functionAtEnd);
	mlEditFuncReg("setWordWrap", functionSetWordWrap);
	mlEditFuncReg("wordWrap", functionWordWrap);
	mlEditFuncReg("text", functionText);
	mlEditFuncReg("length", functionLength);
	mlEditFuncReg("setMaxLines", functionSetMaxLines);
	mlEditFuncReg("maxLines", functionMaxLines);
//	mlEditFuncReg("setMaxLength", functionSetMaxLength);
//	mlEditFuncReg("maxLength", functionMaxLength);
	mlEditFuncReg("insert", functionInsert);
	mlEditFuncReg("append", functionAppend);
	mlEditFuncReg("copy", functionCopy);
	mlEditFuncReg("cut", functionCut);
	mlEditFuncReg("paste", functionPaste);
}

void KviScriptMLEdit::unregisterSelf()
{
	delete g_pMultiLineEditClass;
}

bool KviScriptMLEdit::functionTextLine(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_CONTEXT(c, "multilineedit::textLine");

	if(widget())
	{
		bool bOk;
		int lineNum = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);
			
		if(lineNum > ((QMultiLineEdit *)widget())->numLines() || \
			lineNum < 0)
			c->warning("No such line number");
			
		KviStr line = ((QMultiLineEdit *)widget())->textLine(lineNum);
		b.append(line);
	}
	
	return c->leaveContext();
}

bool KviScriptMLEdit::functionInsertLine(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_CONTEXT(c, "multilineedit::insertLine");

	if(widget())
	{
		bool bOk;
		KviStr * txt = p->safeFirst();
		if(txt->isEmpty())
			return c->error(KviError_notEnoughParameters);

		int line = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);

		((QMultiLineEdit *)widget())->insertLine(txt->ptr(), line);
	}
	
	return c->leaveContext();
}

bool KviScriptMLEdit::functionRemoveLine(KviCommand * c, KviParameterList * p,\
	KviStr & b)
{
	ENTER_CONTEXT(c, "multilineedit::removeLine");
	
	if(widget())
	{
		bool bOk;
		int line = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);

		if(line > ((QMultiLineEdit *)widget())->numLines() || line < 0)
			c->warning("No such line number");

		((QMultiLineEdit *)widget())->removeLine(line);
	}
	
	return c->leaveContext();
}

bool KviScriptMLEdit::functionInsertAt(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_CONTEXT(c, "ultilineedit::insertAt");

	if(widget())
	{
		if(p->count() < 3)
			return c->error(KviError_notEnoughParameters);
			
		KviStr * txt = p->safeFirst();
		bool bOk1, bOk2;
		int line = p->getInt(&bOk1);
		int col = p->getInt(&bOk2);

		if(line < 0 || col < 0 || \
		  line > ((QMultiLineEdit *)widget())->numLines())
			c->warning("No such line number");	

		((QMultiLineEdit *)widget())->insertAt(txt->ptr(), line, col);
	}
	
	return c->leaveContext();
}

bool KviScriptMLEdit::functionNumLines(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", \
			((QMultiLineEdit *)widget())->numLines());

	return true;
}

bool KviScriptMLEdit::functionSetCursorPosition(KviCommand * c, \
	KviParameterList * p, KviStr &)
{
	ENTER_CONTEXT(c, "multilineedit::setCursorPosition");

	if(widget())
	{
		if(p->count() < 3)
			return c->error(KviError_notEnoughParameters);

		bool bOk[2];
		int line = p->getInt(&bOk[1]);
		int col = p->getInt(&bOk[2]);
		bool m = p->getBool();

		if(!bOk[1] || !bOk[2])
			return c->error(KviError_integerParameterExpected);

		((QMultiLineEdit *)widget())->setCursorPosition(line, col, m);
	}
	
	return c->leaveContext();
}

bool KviScriptMLEdit::functionCursorPosition(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
	{
		int line, col;
#if QT_VERSION >= 300
		((QMultiLineEdit *)widget())->getCursorPosition(&line, &col);
#else
		((QMultiLineEdit *)widget())->cursorPosition(&line, &col);
#endif
		b.append(KviStr::Format, "%d, %d", line, col);
	}

	return true;
}

bool KviScriptMLEdit::functionAtBeginning(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
		b.append((((QMultiLineEdit *)widget())->atBeginning())?'1':'0');

	return true;
}

bool KviScriptMLEdit::functionAtEnd(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
		b.append((((QMultiLineEdit *)widget())->atEnd()) ? '1' : '0');

	return true;
}

bool KviScriptMLEdit::functionSetWordWrap(KviCommand *, KviParameterList * p, \
	KviStr &)
{
	if(widget())
	{
		bool on;
		if(!p->count())
			on = false;
		else
			on = p->getBool();

		((QMultiLineEdit *)widget())->setWordWrap( \
			QMultiLineEdit::WidgetWidth);
	}

	return true;
}

bool KviScriptMLEdit::functionWordWrap(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
	{
		bool on = ((QMultiLineEdit *)widget())->wordWrap() & \
			QMultiLineEdit::WidgetWidth;
		b.append(on ? '1' : '0');
	}

	return true;
}

bool KviScriptMLEdit::functionText(KviCommand *, KviParameterList *, KviStr & b)
{
	if(widget())
		b.append(((QMultiLineEdit *)widget())->text());

	return true;
}

bool KviScriptMLEdit::functionLength(KviCommand *, KviParameterList *, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", \
			((QMultiLineEdit *)widget())->length());

	return true;
}

bool KviScriptMLEdit::functionSetMaxLines(KviCommand * c, KviParameterList * p,\
	KviStr & b)
{
	ENTER_CONTEXT(c, "multilineedit::setMaxLines");

	if(widget())
	{
		bool bOk;
		int lines = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);

		((QMultiLineEdit *)widget())->setMaxLines(lines);
	}
	
	return c->leaveContext();
}

bool KviScriptMLEdit::functionMaxLines(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", \
			((QMultiLineEdit *)widget())->maxLines());

	return true;
}
/*
bool KviScriptMLEdit::functionMaxLength(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", \
			((QMultiLineEdit *)widget())->maxLength());

	return true;
}

bool KviScriptMLEdit::functionSetMaxLength(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_CONTEXT(c, "multilineedit::setMaxLength");
	
	if(widget())
	{
		bool bOk;
		int lines = p->getInt(&bOk);
		
		if(!bOk)
			return c->error(KviError_integerParameterExpected);

		((QMultiLineEdit *)widget())->setMaxLength(lines);
	}

	return c->leaveContext();
}
*/
bool KviScriptMLEdit::functionInsert(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_CONTEXT(c, "multilineedit::insert");

	if(widget())
	{
		KviStr * pStr;
		if(!(pStr = p->safeFirst())->isEmpty())
			((QMultiLineEdit *)widget())->insert(pStr->ptr());
		else
			return c->error(KviError_notEnoughParameters);
	}
	
	return c->leaveContext();
}

bool KviScriptMLEdit::functionAppend(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_CONTEXT(c, "multilineedit::append");

	if(widget())
	{
		KviStr * pStr;
		if(!(pStr = p->safeFirst())->isEmpty())
			((QMultiLineEdit *)widget())->append(pStr->ptr());
		else
			return c->error(KviError_notEnoughParameters);
	}

	return c->leaveContext();
}

bool KviScriptMLEdit::functionCopy(KviCommand *, KviParameterList *, KviStr &)
{
	if(widget())
		((QMultiLineEdit *)widget())->copy();

	return true;
}

bool KviScriptMLEdit::functionCut(KviCommand *, KviParameterList *, KviStr &)
{
	if(widget())
		((QMultiLineEdit *)widget())->cut();

	return true;
}

bool KviScriptMLEdit::functionPaste(KviCommand *, KviParameterList *, KviStr &)
{
	if(widget())
		((QMultiLineEdit *)widget())->paste();

	return true;
}

#include "m_class_multilineedit.moc"
