//vim: ts=8
//   File : class_lineedit.cpp
//   Creation date : Wed 20 Sep 2000 04:34:40 CEST by Krzysztof Godlewski
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_debug.h"
#include "kvi_scriptobject.h"
#include "kvi_locale.h"

#include "class_lineedit.h"

static const char * mode_tbl[] = {
			"Normal",
			"NoEcho",
			"Password"
			  };

static const int mode_cod[] =  {
		QLineEdit::Normal,
		QLineEdit::NoEcho,
		QLineEdit::Password
			};

#define mode_num	(sizeof(mode_tbl) / sizeof(mode_tbl[0]))

static KviScriptObjectClass * g_pLineEditClass = 0;

static KviScriptObject * lineEditClassCreateInstance(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n)
{
	return new KviScriptLineEdit(c, p, n);
}

KviScriptLineEdit::KviScriptLineEdit(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n) : KviScriptWidget(c,p,n)
{
}

KviScriptLineEdit::~KviScriptLineEdit()
{
}

/*
	@doc:	lineedit
	@keyterms:
		lineedit object class, line editor, input
	@title:
		lineedit class
	@type:
		class
	@short:
		Simple line editor for inputting short text
	@inherits:
		[class]object[/class]
		[class]widget[/class]
	@description:
		This is a simple, one-line editor usually used for inputting
		short text like names, phone numbers etc. It can display text 
		in three echo modes (see [classfnc]$setEchoMode[/classfnc]() 
		for details).
	@functions:
		!fn: $text()
		Returns text contained by the widget.[br]
		See also [classfnc]$setText[/classfnc]().
		!fn: $setText(<text>)
		Sets editor text to <text>.[br]
		See also [classfnc]$text[/classfnc]().
		!fn: $maxLength()
		Returns the current maximum length of the text that can be typed
		 in the editor.[br]
		See also [classfnc]$setMaxLength[/classfnc]().
		!fn: $setMaxLength(<value>)
		Sets maximum length of the text that can be typed in the 
		editor.[br]
		See also [classfnc]$maxLength[/classfnc]().
		!fn: $frame()
		Returns '1' if the editor is drawn inside a frame, which is 2 
		pixels wide, '0' otherwise.[br]
		See also [classfnc]$setFrame[/classfnc]().
		!fn: $setFrame(<bool>)
		Sets editor frame drawing to <bool>.[br]
		See also [classfnc]$frame[/classfnc]().
		!fn: $cursorPosition()
		Returns current cursor position in the editor.[br]
		See also [classfnc]$setCursorPosition[/classfnc]().
		!fn: $setCursorPosition(<index>)
		Moves the cursor to place pointed by <index>.[br]
		See also [classfnc]$cursorPosition[/classfnc]().
		!fn: $selectAll()
		Selects whole text inside the editor.[br]
		See also [classfnc]$setSelection[/classfnc](), 
		[classfnc]$cut[/classfnc](), [classfnc]$copy[/classfnc](), 
		[classfnc]$paste[/classfnc]().
		!fn: $setSelection(<start>, <length>)
		Selects <length> characters in the editor starting at position 
		<start>.[br]
		See also [classfnc]$selectAll[/classfnc](), 
		[classfnc]$cut[/classfnc](), [classfnc]$copy[/classfnc](), 
		[classfnc]$paste[/classfnc]().
		!fn: $copy()
		Copies selected text in the editor to the clipboard.[br]
		See also [classfnc]$setSelection[/classfnc](), 
		[classfnc]$cut[/classfnc](), [classfnc]$paste[/classfnc]().
		!fn: $cut()
		Cuts selected text from the editor to the clipboard.[br]   
		See also [classfnc]$setSelection[/classfnc](), 
		[classfnc]$copy[/classfnc](), [classfnc]$paste[/classfnc]().
		!fn: $paste()
		Pastes text from the clipboard into the editor. The pasted text 
		will start at cursor position.[br]
		See also [classfnc]$setSelection[/classfnc](), 
		[classfnc]$copy[/classfnc](), [classfnc]$cut[/classfnc]().
		!fn: $clear()
		Deletes all text from the editor. Equal to calling 
		[classfnc]$setText[/classfnc]("").
*/

void KviScriptLineEdit::init()
{
	setObject(new QLineEdit(parentScriptWidget(), name()), true);
}

#define lineeditFuncReg(__nam, __func) \
        g_pLineEditClass->registerFunctionHandler(__nam, \
        (KviScriptObjectFunctionHandlerProc)( \
	KVI_PTR2MEMBER(KviScriptLineEdit::__func)), 0, true);

void KviScriptLineEdit::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController-> \
		lookupClass("widget");
	__range_valid(base);

	g_pLineEditClass = new KviScriptObjectClass(base, "lineedit",
		lineEditClassCreateInstance, true);

	lineeditFuncReg("text", functionText);
	lineeditFuncReg("setText", functionSetText);
	lineeditFuncReg("maxLength", functionMaxLength);
	lineeditFuncReg("setMaxLength", functionSetMaxLength);
	lineeditFuncReg("frame", functionFrame);
	lineeditFuncReg("setFrame", functionSetFrame);
	lineeditFuncReg("cursorPosition", functionCursorPosition);
	lineeditFuncReg("setCursorPosition", functionSetCursorPosition);
	lineeditFuncReg("selectAll", functionSelectAll);
	lineeditFuncReg("setSelection", functionSetSelection);
	lineeditFuncReg("copy", functionCopy);
	lineeditFuncReg("cut", functionCut);
	lineeditFuncReg("paste", functionPaste);
	lineeditFuncReg("echoMode", functionEchoMode);
	lineeditFuncReg("setEchoMode", functionSetEchoMode);
	lineeditFuncReg("clear", functionClear);
}

void KviScriptLineEdit::unregisterSelf()
{
	delete g_pLineEditClass;
}

bool KviScriptLineEdit::functionText(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(widget())
		b.append(((QLineEdit *)widget())->text().latin1());
	return true;
}

bool KviScriptLineEdit::functionSetText(KviCommand *, KviParameterList * p,
	KviStr &)
{
	if(widget())
	{
		if(p->count())
			((QLineEdit *)widget())->setText(p->safeFirst()->ptr());
		else
			((QLineEdit *)widget())->setText("");
	}
	return true;
}

bool KviScriptLineEdit::functionMaxLength(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", ((QLineEdit *)widget())-> \
			maxLength());
	else
		b.append("-1");
	return true;
}

bool KviScriptLineEdit::functionSetMaxLength(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	ENTER_CONTEXT(c, "lineedit::setMaxLength");
	if(widget())
	{
		bool bOk;
		int val = p->getInt(&bOk);
		if(bOk)
		{
			((QLineEdit *)widget())->setMaxLength(val);
			return c->leaveContext();
		}
		else
			return c->error(KviError_integerParameterExpected);
	}
	return c->leaveContext();
}

bool KviScriptLineEdit::functionFrame(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(widget())
		b.append(((QLineEdit *)widget())->frame() ? '1' : '0');
	return true;
}

bool KviScriptLineEdit::functionSetFrame(KviCommand *, KviParameterList * p,
	KviStr & b)
{
	if(widget())
		((QLineEdit *)widget())->setFrame(p->getBool());
	return true;
}

bool KviScriptLineEdit::functionCursorPosition(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", ((QLineEdit *)widget())-> \
			cursorPosition());
	return true;
}

bool KviScriptLineEdit::functionSetCursorPosition(KviCommand * c,
	KviParameterList * p, KviStr & b)
{
	ENTER_CONTEXT(c, "lineedit::setCursorPosition");
	if(widget())
	{
		bool bOk;
		int val = p->getInt(&bOk);
		if(bOk)
		{
			((QLineEdit *)widget())->setCursorPosition(val);
			return c->leaveContext();
		}
		else
			return c->error(KviError_integerParameterExpected);
	}
	return c->leaveContext();
}

bool KviScriptLineEdit::functionSelectAll(KviCommand *, KviParameterList *,
	KviStr &)
{
	if(widget())
		((QLineEdit *)widget())->selectAll();
	return true;
}

bool KviScriptLineEdit::functionSetSelection(KviCommand * c, \
	KviParameterList * p, KviStr &)
{
	ENTER_CONTEXT(c, "lineedit::setSelection");
	if(!widget())
		return true;
	
	bool bOk1, bOk2;
	int start = p->getInt(&bOk1);
	int len = p->getInt(&bOk2);
	if(!bOk1 || !bOk2)
		return c->error(KviError_integerParameterExpected);

	((QLineEdit *)widget())->setSelection(start, len);
	
	return c->leaveContext();
}

bool KviScriptLineEdit::functionCopy(KviCommand *, KviParameterList *, KviStr &)
{
	if(widget())
		((QLineEdit *)widget())->copy();
	return true;
}

bool KviScriptLineEdit::functionCut(KviCommand *, KviParameterList *, KviStr &)
{
	if(widget())
		((QLineEdit *)widget())->cut();
	return true;
}

bool KviScriptLineEdit::functionPaste(KviCommand *, KviParameterList *,KviStr &)
{
	if(widget())
		((QLineEdit *)widget())->paste();
	return true;
}

bool KviScriptLineEdit::functionEchoMode(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(!widget())
	{
		b.append('0');
		return true;
	}

	int mode = ((QLineEdit *)widget())->echoMode();
	for(int i = 0; i < mode_num; i++)
	{
		if(mode == mode_cod[i])
		{
			b.append(mode_tbl[i]);
			return true;
		}
	}

	b.append(__tr("Hey ! You wanna know my echo mode or what ?"));
	return true;
}

bool KviScriptLineEdit::functionSetEchoMode(KviCommand * c,KviParameterList * p,
	KviStr &)
{
	ENTER_CONTEXT(c, "lineedit::setEchoMode");
	if(!widget())
		return c->leaveContext();

	for(int i = 0; i < mode_num; i++)
	{
		if(kvi_strEqualCI(p->safeFirst()->ptr(), mode_tbl[i]))
		{
			((QLineEdit *)widget())->setEchoMode( \
				((QLineEdit::EchoMode)mode_cod[i]));
			return c->leaveContext();
		}
	}

	c->warning(__tr("Unknown echo mode"));
	return c->leaveContext();
}

bool KviScriptLineEdit::functionClear(KviCommand *, KviParameterList *,KviStr &)
{
	if(widget())
		((QLineEdit *)widget())->clear();
	return true;
}

#include "m_class_lineedit.moc"
