//vim: ts=8
//   File : class_label.cpp
//   Creation date : Mon Sep 18 14:21:48 CEST 2000 by Krzysztof Godlewski
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Krzysztof Godlewski
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include <qpixmap.h>

#include "class_label.h"
//#include "kvi_fileutils.h"
#include "kvi_error.h"
#include "kvi_debug.h"
#include "kvi_command.h"
#include "kvi_locale.h"
#include "kvi_iconmanager.h"

// Tables used in $setAlignment & $alignment
const char * const align_tbl[] = {
			"Left", 
			"Right",
			"HCenter",
			"VCenter",
			"Center",
			"Top",
			"Bottom",
			"WordBreak"
			   };
const int align_cod[] = {
		QLabel::AlignLeft,
		QLabel::AlignRight,
		QLabel::AlignHCenter,
		QLabel::AlignVCenter,
		QLabel::AlignCenter,
		QLabel::AlignTop,
		QLabel::AlignBottom,
		QLabel::WordBreak
			};

#define align_num	(sizeof(align_tbl) / sizeof(align_tbl[0]))
		  
// used in $frameStyle & $setFrameStyle
const char * const frame_tbl[] = {
				"NoFrame",
				"Box",
				"Panel",
				"WinPanel",
			// shadow styles
				"Plain",
				"Raised",
				"Sunken"
			   };

const int frame_cod[] = {
				QFrame::NoFrame,
				QFrame::Box,
				QFrame::Panel,
				QFrame::WinPanel,
				QFrame::Plain,
				QFrame::Raised,
				QFrame::Sunken
			 };

#define frame_num	(sizeof(frame_tbl) / sizeof(frame_tbl[0]))

static KviScriptObjectClass * g_pLabelClass = 0;

static KviScriptObject * labelClassCreateInstance(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n)
{
	return new KviScriptLabel(c, p, n);
}

KviScriptLabel::KviScriptLabel(KviScriptObjectClass * c, KviScriptObject * p,
	const char *n) : KviScriptWidget(c, p, n)
{
}

KviScriptLabel::~KviScriptLabel()
{
}

/*
	@doc:	label
	@keyterms:
		label object class, show image
	@title:
		label class
	@type:
		class
	@short:
		Displays text or an image
	@inherits:
		[class]object[/class]
		[class]widget[/class]
	@description:
		This widget can be used to display a text or an image. It can
		have different frame styles and text/image alignment.
	@functions:
		!fn: $setText(<text>)
		Sets the text to be displayed by the label.
		The text can contain limited html tags.
		See also [classfnc]$text[/classfnc]().
		!fn: $text()
		Returns the text currently displayed by the label. 
		See also [classfnc]$setText[/classfnc]().
		!fn: $margin()
		Returns current value of margin for this label (default is 0).
		See also [classfnc]$setMargin[/classfnc]().
		!fn: $setMargin(<value>)
		Sets margin width / height to <value>.
		See also [classfnc]$margin[/classfnc]().
		!fn: $autoResize()
		Returns 1, if auto-resize ability of the label is enabled. 
		See also [classfnc]$setAutoResize[/classfnc]().
		!fn: $setAutoResize(<bool>)
		Sets auto-resize ability to enabled (if <bool> is 1) or disabled
		 (if <bool> is 0). If auto-resize is enabled, the label will 
		automagically resize itself accordingly to the 'size' of the
		text it contains. 
		See also [classfnc]$autoResize[/classfnc]().
		!fn: $alignment()
		Returns a string containing alignment flags that are set for 
		this label. The flags are separated by commas. An example output
		could look like this:[br]
		[pre]Bottom, Right[/pre][br]
		See [classfnc]$setAlignment[/classfnc]() for explanation of all
		alignment flags.
		!fn: $setAlignment(<flag1>, <flag2>, ...)
		This function sets alignment flags, given as parameters, for 
		this label. Valid flags are:
		[pre]
		Right     - Text is aligned to right border[br]
		Left      - Text is aligned to left border[br]
		Top       - Text is aligned to the top border[br]
		Bottom    - Text is aligned to the bottom border[br]
		HCenter   - Text is horizontally centered[br]
		VCenter   - Text is vertically centered[br]
		Center    - Equals HCenter + VCenter[br]
		WordBreak - Enables automatic word breaking[br]
		[/pre]
		It is obvious that you can not set for example [i]Right[/i] 
		and [i]Left[/i] simoultaneously - this will [b]NOT[/b] 
		result in an error message - this will simply not work :)
		!fn: $clear()
		Clears the label. Equal to calling 
		[classfnc]$setText[/classfnc]("")
		!fn: $frameStyle()
		Returns a string containing this label's frame-style flags,
		separated with commas. Output from this function could look like
		this:
		[pre]Panel, Raised[/pre]
		See [classfnc]$setFrameStyle[/classfnc]() for a list of all
		frame-style flags and their explenation.
		!fn: $setFrameStyle(<flag1>, <flag2>, ...)
		Sets the frame-style flags to the ones passed as arguments.
		The flags can either decide of the shape or shadow of the
		label's frame. Valid shape flags are:[br]
		[pre]
		NoFrame     - draw no frame. You shouldn't specify a shadow when
		using this.[br]
		Box         - draws a rectangular box. Its borders can be 
		[i]Raised[/i] or [i]Sunken[/i][br]
		Panel       - draws a rectangular panel which can be 
		[i]Raised[/i] or [i]Sunken[/i][br]
		WinPanel    - similar to [i]Panel[/i], but is more in Win95
		style[br]
		[/pre]
		Valid shadow flags are:[br]
		[pre]
		Plain       - no 3D effect (draws using foreground color)[br]
		Raised      - makes the label look like it was raised above the
		parent widget[br]
		Sunken      - makes the label look like it was "pushed" inside
		the parent widget[br]
		[/pre]
		!fn: $setImage(<image_id>)
		Sets the image to be displayed on this label. 
		Giving empty argument clears the pixmap[br]
		See the [doc:image_id]image identifier[/doc] documentation for
		the explaination of the <image_id> parameter.
*/

void KviScriptLabel::init()
{
	setObject(new QLabel(parentScriptWidget(), name()), true);
}

#define labelFuncReg(__nam, __func) \
	g_pLabelClass->registerFunctionHandler(__nam, \
	(KviScriptObjectFunctionHandlerProc)(KVI_PTR2MEMBER(KviScriptLabel::__func)), \
	0, true);

void KviScriptLabel::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController-> \
		lookupClass("widget");
	__range_valid(base);

	g_pLabelClass = new KviScriptObjectClass(base, "label",
		labelClassCreateInstance, true);

	labelFuncReg("setText", functionSetText);
	labelFuncReg("text", functionText);
	labelFuncReg("margin", functionMargin);
	labelFuncReg("setMargin", functionSetMargin);
	labelFuncReg("autoResize", functionAutoResize);
	labelFuncReg("setAutoResize", functionSetAutoResize);
	labelFuncReg("alignment", functionAlignment);
	labelFuncReg("setAlignment", functionSetAlignment);
	labelFuncReg("clear", functionClear);
	labelFuncReg("frameStyle", functionFrameStyle);
	labelFuncReg("setFrameStyle", functionSetFrameStyle);
	labelFuncReg("setImage", functionSetImage);
}

void KviScriptLabel::unregisterSelf()
{
	delete g_pLabelClass;
}

bool KviScriptLabel::functionSetText(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
	{
		if(p->first())
			((QLabel *)widget())->setText(p->first()->ptr());
		else
			((QLabel *)widget())->setText("[!-NULL-!]");
	}
	return true;
}

bool KviScriptLabel::functionText(KviCommand *, KviParameterList *, KviStr & b)
{
	if(widget())
		b.append(widget()->caption().latin1());
	return true;
}

bool KviScriptLabel::functionMargin(KviCommand *, KviParameterList *,KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", ((QLabel *)widget())->margin());
	return true;
}

bool KviScriptLabel::functionSetMargin(KviCommand * c, KviParameterList * p, 
	KviStr & b)
{
	ENTER_CONTEXT(c, "label::setMargin");
	if(!widget())
		return c->leaveContext();
	bool bOk;
	int val = p->getInt(&bOk);
	bOk ? ((QLabel *)widget())->setMargin(val) :
		(void)c->warning(__tr("Invalid margin parameter"));
	return c->leaveContext();
}

bool KviScriptLabel::functionAutoResize(KviCommand *, KviParameterList *, 
	KviStr & b)
{
	if(widget())
		((QLabel *)widget())->autoResize() ? b.append('1') :
			b.append('0');
	return true;
}

bool KviScriptLabel::functionSetAutoResize(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())
	{
		((QLabel *)widget())->setAutoResize(p->getBool());
	}
	return true;
}

bool KviScriptLabel::functionAlignment(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(widget())
	{
		KviStr ret;
		int align = ((QLabel *)widget())->alignment();
		for(int i = 0; i < align_num; i++)
		{
			if( align & align_cod[i] )
			{
				ret.append(align_tbl[i]);
				ret.append(",");
			}
		}
		ret.cutRight(1);	// cut last ","
		b.append(ret);
		return true;
	}
	b.append('0');
	return true;
}

bool KviScriptLabel::functionSetAlignment(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(!widget())
		return true;

	ENTER_CONTEXT(c, "label::setAlignment");
	int cnt = p->count();
	int align;
	int sum = 0;
	for(int i = 0; i < cnt; i++)
	{
		align = 0;
		
		for(int j = 0; j < align_num; j++ )
		{
			if(kvi_strEqualCI(p->at(i)->ptr(), align_tbl[j]))
			{
				align = align_cod[j];
				break;
			}
		}
		
		if(align)
			sum |= align;
		else
			c->warning( __tr("Unknown alignment: %s"), \
				p->at(i)->ptr());
	}
	((QLabel *)widget())->setAlignment(sum);
	return c->leaveContext();
}

bool KviScriptLabel::functionClear(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(widget())
		((QLabel *)widget())->clear();
	return true;
}

bool KviScriptLabel::functionFrameStyle(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	if(!widget())
	{
		b.append('0');
		return true;
	}

	KviStr ret;
	int fstyle = ((QLabel *)widget())->frameStyle();
	for(int i = 0; i < frame_num; i++)
	{
		if(fstyle & frame_cod[i])
		{
			ret.append(frame_tbl[i]);
			ret.append(",");
		}
	}
	ret.cutRight(1);	// get rid of last ","
	b.append(ret);
	return true;
}

bool KviScriptLabel::functionSetFrameStyle(KviCommand * c, KviParameterList *p,
	KviStr & b)
{
	if(!widget())
		return true;

	ENTER_CONTEXT(c, "label::setFrameStyle");
	int cnt = p->count();
	int fstyle;
	int sum = 0;
	for(int i = 0; i < cnt; i++)
	{
		fstyle = 0;
		
		for(int j = 0; j < frame_num; j++ )
		{
			if(kvi_strEqualCI(p->at(i)->ptr(), frame_tbl[j]))
			{
				fstyle = frame_cod[j];
				break;
			}
		}
		
		if(fstyle)
			sum |= fstyle;
		else
			c->warning(__tr("Unknown frame / shadow style: %s"), \
				p->at(i)->ptr());
	}
	((QLabel *)widget())->setFrameStyle(sum);
	return c->leaveContext();
}

bool KviScriptLabel::functionSetImage(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	QPixmap *pix = NULL;
//#warning "Add a method to stretch the image on the fly to a desired size ?"
// Not needed: the pixmap will resize with the label.
	if(!widget())
		return true;
	if(p->first()) {
    	    pix = g_pIconManager->getImage(p->first()->ptr());
	} else {
	    pix = new QPixmap(0,0);
	}
	
	if(pix) {
		((QLabel *)widget())->setPixmap(*pix);
	}
	return true;
}

#include "m_class_label.moc"
