//
//   File : libkvilog.cpp
//   Creation date : Sat Jan 06 2001 13:28:02 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"
#include "kvi_window.h"
#include "kvi_app.h"
#include "kvi_locale.h"
#include "kvi_ircview.h"
#include "kvi_uparser.h"

//#warning "log.stats"
//#warning "log.compress (gzip -r the log directory)"

/*
	@doc: log
	@title:
		The log interface module
	@short:
		Interface to the KVirc log subsystem
	@type:
		module
	@keyterms:
		logging
	@body:
		The log module is an interface to the KVIrc logging subsystem.[br]
		It provides commands and functions that allow the management of
		logging in the KVIrc windows:[br]
		[cmd]log.start[/cmd], [cmd]log.stop[/cmd], [cmd]log.flush[/cmd],
		[fnc]$log.file[/fnc]().[br]
		NOTE:[br]
		If you want to AUTOENABLE logging in certain types of windows
		you have to use the [cmd]option[/cmd] command.[br]
		For example, [b]option boolAutoLogQueries 1[/b] enables logging in all the newly created
		query windows.
*/

/*
	@doc: log.start
	@type:
		command
	@title:
		log.start
	@keyterms:
		logging
	@short:
		Starts logging in a specified window
	@syntax:
		log.start [-w = <window_id>] [-p] [filename]
	@description:
		Starts logging in the current window or in the window specified by the -w switch.[br]
		If a logging session is already running in the specified window, it is stopped first.[br]
		If [filename] is specified, this filename is used as log file, otherwise a default
		filename is used.[br]
		If the -p switch is used, all the already existing contents of the window are
		stored to the log file before starting the log action.[br]
		Please note that some windows do not have logging capabilities...this command
		will print a warning in that case.[br]
	@seealso:
		[fnc]$window[/fnc],
		[cmd]log.stop[/cmd],
		[fnc]$log.file[/fnc],
		[doc:window_naming_conventions]window naming conventions documentation[/doc]
*/

static bool log_module_cmd_start(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"log_module_cmd_start");

	KviStr filename;
	if(!g_pUserParser->parseCmdFinalPart(c,filename))return false;

	KviWindow * pWnd = c->window();

	if(c->hasSwitch('w'))
	{
		KviStr window;
		if(c->getSwitchValue('w',window))
		{
			pWnd = g_pApp->findWindow(window.ptr());
			if(!pWnd)return c->warning(__tr("Window %s not found"),window.ptr());

		} else return c->warning(__tr("Missing window id after the 'w' switch"));
	}

	if(pWnd->view())
	{
		if(filename.isEmpty())pWnd->getDefaultLogFileName(filename);
		if(!pWnd->view()->startLogging(filename.ptr(),
			c->hasSwitch('p')))c->warning(__tr("Can't log to file %s"),filename.ptr());
	} else c->warning(__tr("This window has no logging capabilities"));

	return c->leaveContext();
}


/*
	@doc: log.stop
	@type:
		command
	@title:
		log.stop
	@keyterms:
		logging
	@short:
		Stops logging in a specified window
	@syntax:
		log.stop [-w = <window_id>]
	@description:
		Stops logging in the current window or in the window specified by the -w switch.[br]
		If logging is not enabled in the specified window, this command does nothing.[br]
	@seealso:
		[fnc]$window[/fnc],
		[cmd]log.start[/cmd],
		[doc:window_naming_conventions]window naming conventions documentation[/doc]
*/

static bool log_module_cmd_stop(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"log_module_cmd_stop");

	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	KviWindow * pWnd = c->window();

	if(c->hasSwitch('w'))
	{
		KviStr window;
		if(c->getSwitchValue('w',window))
		{
			pWnd = g_pApp->findWindow(window.ptr());
			if(!pWnd)return c->warning(__tr("Window %s not found"),window.ptr());
		} else return c->warning(__tr("Missing window id after the 'w' switch"));
	}

	if(pWnd->view())pWnd->view()->stopLogging();

	return c->leaveContext();
}

/*
	@doc: log.flush
	@type:
		command
	@title:
		log.flush
	@keyterms:
		logging
	@short:
		Flushes the log file for a specified window
	@syntax:
		log.flush [-w = <window_id>]
	@description:
		Flushes the log file the current window or in the window specified by the -w switch.[br]
		If logging is not enabled in the specified window, this command does nothing.[br]
		If this command is never used, the log file is flushed at regulear file-size intervals,
		depending on the underlying IO subsystem. This is usually 4KB.[br]
		The "block flushing" is used to achieve a fast I/O on files, but may cause
		the last data block to be lost in case of a program crash.[br]
	@seealso:
		[fnc]$window[/fnc],
		[cmd]log.start[/cmd],
		[cmd]log.stop[/cmd],
		[doc:window_naming_conventions]window naming conventions documentation[/doc]
*/

static bool log_module_cmd_flush(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"log_module_cmd_flush");

	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	KviWindow * pWnd = c->window();

	if(c->hasSwitch('w'))
	{
		KviStr window;
		if(c->getSwitchValue('w',window))
		{
			pWnd = g_pApp->findWindow(window.ptr());
			if(!pWnd)return c->warning(__tr("Window %s not found"),window.ptr());
		} else return c->warning(__tr("Missing window id after the 'w' switch"));
	}

	if(pWnd->view())pWnd->view()->flushLog();

	return c->leaveContext();
}


/*
	@doc: log.file
	@type:
		function
	@title:
		$log.file
	@short:
		Returns the current log filename
	@syntax:
		$log.file
		$log.file(<window id>)
	@description:
		The form with the <window id> parameter returns the log file name
		of the window that has the specified id. If logging is not enabled
		in the specified window, this function returns an empty string.[br]
		The form without parameters returns the log file name of the current window,
		thus it is equivalent to calling $log.file([fnc]$window[/fnc])
	@examples:
		[example]
		[cmd]log[/cmd] mylogfilename.log
		[cmd]echo[/cmd] $log
		[cmd]echo[/cmd] $log([fnc]$console[/fnc])
		[/example]
	@seealso:
		[fnc]$window[/fnc],
		[cmd]log.start[/cmd],
		[cmd]log.stop[/cmd],
		[doc:window_naming_conventions]window naming conventions documentation[/doc]
*/


static bool log_module_fnc_file(KviModule *m,KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_CONTEXT(c,"log_module_fnc_file");

	KviStr * pStr = params->first();

	KviWindow * wnd = c->window();

	if(pStr)
	{
		wnd = g_pApp->findWindow(pStr->ptr());
		if(!wnd)return c->warning(__tr("Window with id '%s' not found, returning empty string"),pStr->ptr());
	}

	if(wnd->view())wnd->view()->getLogFileName(buffer);

	return c->leaveContext();
}


static bool log_module_init(KviModule * m)
{
	m->registerCommand("start",log_module_cmd_start);
	m->registerCommand("stop",log_module_cmd_stop);
	m->registerCommand("flush",log_module_cmd_flush);
	m->registerFunction("file",log_module_fnc_file);
	return true;
}

static bool log_module_cleanup(KviModule *m)
{
	return true;
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Log",                                                  // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)", // author & (C)
	"User interface to the logging system for KVIrc",
	log_module_init,
	0,
	0,
	log_module_cleanup
};
