//
//   File : listwindow.cpp
//   Creation date : Thu Oct 7 2001 13:27:55 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the lists of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#include "listwindow.h"

#include "kvi_debug.h"
#include "kvi_iconmanager.h"
#include "kvi_ircview.h"
#include "kvi_out.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_out.h"
#include "kvi_mirccntrl.h"
#include "kvi_themedlabel.h"
#include "kvi_options.h"
#include "kvi_mirccntrl.h"
#include "kvi_ircsocket.h"

#include <qpainter.h>
#include <qfontmetrics.h>
#include <qsplitter.h>
#include <qtooltip.h>
#include <qhbox.h>

extern KviPtrList<KviListWindow> * g_pListWindowList;

// kvi_ircview.cpp
extern KVIRC_API const char * getColorBytes(const char *data_ptr,unsigned char *byte_1,unsigned char *byte_2);



KviChannelListViewItem::KviChannelListViewItem(QListView * v,const char * szChan,const char * szUsers,const char * szTopic)
: QListViewItem(v,QString::null,szUsers,szTopic)
{
	m_szChan = szChan;
	m_szUsers = szUsers;
	m_szTopic = szTopic;
	m_szUsersKey = szUsers;
	KviStr szC = szChan;
	szC.toUpper();
	setText(0,szC.ptr());
	while(m_szUsersKey.length() < 6)m_szUsersKey.prepend("0");
}


KviChannelListViewItem::~KviChannelListViewItem()
{
}


#define KVI_LABEL_DEF_BACK 100
#define KVI_LABEL_DEF_FORE 101

void KviChannelListViewItem::paintCell(QPainter * p,const QColorGroup &cg,int col,int wdth,int align)
{
	p->fillRect(0, 0, wdth, height(), cg.brush( QColorGroup::Base ));

	const char * t;
	switch(col)
	{
		case 0:  t = m_szChan.ptr();   break;
		case 1:  t = m_szUsers.ptr();  break;
		default: t = m_szTopic.ptr();  break;
	}

//	debug("topic : %s",m_szTopic.ptr());

	QFontMetrics fm(p->font());

	bool curBold      = false;
	bool curUnderline = false;
	unsigned char curFore      = KVI_LABEL_DEF_FORE; //default fore
	unsigned char curBack      = KVI_LABEL_DEF_BACK; //default back
	int baseline = ((height() + fm.ascent() - fm.descent() + 1) >> 1);

	int curX = 2; //2 is the margin

	while(*t && (curX < wdth))
	{
		const char * aux = t;
		while(*aux && (*aux != KVI_TEXT_COLOR) && (*aux != KVI_TEXT_BOLD) &&
					(*aux != KVI_TEXT_UNDERLINE) && (*aux != KVI_TEXT_REVERSE) &&
					(*aux != KVI_TEXT_RESET))++aux;

		int len = aux - t;
		int wdth;

		if(len > 0)
		{
			wdth = fm.width(t,len);
	
			if(curFore == KVI_LABEL_DEF_FORE)
			{
				p->setPen(cg.text());
			} else {
				if(curFore > 16)p->setPen(cg.base());
				else p->setPen(KVI_OPTION_MIRCCOLOR(curFore));
			}
		
			if(curBack != KVI_LABEL_DEF_BACK)
			{
				if(curBack > 16)
				{
					p->fillRect(curX,2,wdth,height() - 4,cg.text());
				} else {
					p->fillRect(curX,2,wdth,height() - 4,KVI_OPTION_MIRCCOLOR(curBack));
				}
			}

			p->drawText(curX,baseline,t,len);
	
			if(curBold)p->drawText(curX+1,baseline,t,len);
			if(curUnderline)p->drawLine(curX,baseline + 1,curX+wdth,baseline + 1);

		} else {
			wdth = 0;
		}

	
		switch(*aux)
		{
			case KVI_TEXT_BOLD: curBold = !curBold; ++aux; break;
			case KVI_TEXT_UNDERLINE: curUnderline = !curUnderline; ++aux; break;
			case KVI_TEXT_REVERSE:
				{
					char auxBack = curBack;
					curBack = curFore;
					curFore = auxBack;
				}
				++aux;
			break;
			case KVI_TEXT_RESET:
				curFore = KVI_LABEL_DEF_FORE;
				curBack = KVI_LABEL_DEF_BACK;
				curBold = false;
				curUnderline = false;
				++aux;
			break;
			case KVI_TEXT_COLOR:
			{
				++aux;
				unsigned char fore;
				unsigned char back;
				aux = getColorBytes(aux,&fore,&back);
				if(fore != KVI_NOCHANGE)curFore = fore;
				else curFore = KVI_LABEL_DEF_FORE;
				if(back != KVI_NOCHANGE)curBack = back;
				else curBack = KVI_LABEL_DEF_BACK;
			}
			break;
		}
		t = aux;
		curX += wdth;
	}

	if(isSelected())
	{
		p->setRasterOp(Qt::NotROP);
		p->fillRect(0, 0, wdth, height(), Qt::black);
		p->setRasterOp(Qt::CopyROP);
	}
}
	
QString KviChannelListViewItem::key(int col,bool) const
{
	switch(col)
	{
		case 0:
			return text(0);
		break;
		case 1:
			return m_szUsersKey;
		break;
		case 2:
			return text(2);
		break;
	}
	QString ret;
	return ret;
}



KviListWindow::KviListWindow(KviFrame * lpFrm,KviConsole * lpConsole)
: KviWindow(KVI_WINDOW_TYPE_LIST,lpFrm,"list",lpConsole) , KviExternalServerDataParser()
{
	g_pListWindowList->append(this);

	m_pSplitter = new QSplitter(QSplitter::Horizontal,this,"splitter");
	m_pTopSplitter = new QSplitter(QSplitter::Horizontal,this,"top_splitter");
	m_pVertSplitter = new QSplitter(QSplitter::Vertical,m_pSplitter,"vsplitter");

	QHBox * box = new QHBox(m_pTopSplitter);

	m_pRequestButton = new QToolButton(box,"request_button");
	m_pRequestButton->setUsesBigPixmap(false);
	m_pRequestButton->setPixmap(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_LIST)));
	connect(m_pRequestButton,SIGNAL(clicked()),this,SLOT(requestList()));
	QToolTip::add(m_pRequestButton,__tr("Request list"));

	m_pParamsEdit = new QLineEdit(box);
	box->setStretchFactor(m_pParamsEdit,1);
	QToolTip::add(m_pParamsEdit,__tr("/LIST command parameters\nMany servers accept special parameters that" \
						"allow you to control (usually reduce) the returned entries.\n" \
						"Commonly , masked channel names are accepted as parameters as well as strings\n" \
						"like \"c<n\" or \"c>n\" where n is the minimum or maximum of users on the channel."));

	m_pInfoLabel = new KviThemedLabel(m_pTopSplitter,"info_label");

	m_pListView  = new QListView(m_pVertSplitter);
	m_pListView->addColumn(__tr("Channel"));
	m_pListView->addColumn(__tr("Users"));
	m_pListView->addColumn(__tr("Topic"));
	m_pListView->setAllColumnsShowFocus(true);
	connect(m_pListView,SIGNAL(doubleClicked(QListViewItem *)),this,SLOT(itemDoubleClicked(QListViewItem *)));

	m_pIrcView = new KviIrcView(m_pVertSplitter,lpFrm,this);

	setFocusHandlerNoClass(m_pListView,this,"QLineEdit");

	m_pConsole->setListWindowPointer(this);

	connect(m_pConsole,SIGNAL(stateChange(KviConsole::ConnectionState)),
		this,SLOT(connectionStateChange(KviConsole::ConnectionState)));

	connectionStateChange(lpConsole->state());
}

KviListWindow::~KviListWindow()
{
	g_pListWindowList->removeRef(this);
	m_pConsole->setListWindowPointer(0);
}

void KviListWindow::getBaseLogFileName(KviStr &buffer)
{
	buffer.sprintf("LIST_%d",console()->ircContextId());
}


void KviListWindow::requestList()
{
	if(m_pConsole->isConnected())
	{
		KviStr parms = m_pParamsEdit->text();
		if(parms.isEmpty())m_pConsole->socket()->sendFmtData("list");
		else m_pConsole->socket()->sendFmtData("list %s",parms.ptr());
		outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("Sent list request: waiting for reply..."));
		m_pRequestButton->setEnabled(false);
	} else {
		outputNoFmt(KVI_OUT_SYSTEMERROR,__tr("Can't request list: no active connection"));
	}
}

void KviListWindow::connectionStateChange(KviConsole::ConnectionState st)
{
	m_pRequestButton->setEnabled(st == KviConsole::Connected);
	if(st == KviConsole::Connected)
	{
		KviStr tmp(KviStr::Format,__tr("You're connected to %s (%s)"),m_pConsole->currentServerName(),m_pConsole->currentNetworkName());
		m_pInfoLabel->setText(tmp.ptr());
	} else {
		m_pInfoLabel->setText(__tr("You're not connected to a server: list cannot be requested"));
	}
}

QPixmap * KviListWindow::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_LIST);
}

void KviListWindow::resizeEvent(QResizeEvent *e)
{
	int hght = m_pTopSplitter->sizeHint().height();
	m_pTopSplitter->setGeometry(0,0,width(),hght);
	m_pSplitter->setGeometry(0,hght,width(),height() - hght);
}

QSize KviListWindow::sizeHint() const
{
	return m_pSplitter->sizeHint();
}

void KviListWindow::fillCaptionBuffers()
{
	m_szPlainTextCaption.sprintf(__tr("Channel list [IRC Context %u]"),m_pConsole->ircContextId());

	m_szHtmlActiveCaption.sprintf(
		__tr("<nobr><font color=\"%s\"><b>Channel list</b></font> " \
			"<font color=\"%s\">[IRC Context %u]</font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive).name().ascii(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive2).name().ascii(),
		m_pConsole->ircContextId());

	m_szHtmlInactiveCaption.sprintf(
		__tr("<nobr><font color=\"%s\"><b>Channel list</b></font> " \
			"<font color=\"%s\">[IRC Context %u]</font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive).name().ascii(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive2).name().ascii(),
		m_pConsole->ircContextId());
}

void KviListWindow::die()
{
	close();
}

void KviListWindow::control(int message)
{
	switch(message)
	{
		case EXTERNAL_SERVER_DATA_PARSER_CONTROL_RESET:       reset();       break;
		case EXTERNAL_SERVER_DATA_PARSER_CONTROL_STARTOFDATA: startOfList(); break;
		case EXTERNAL_SERVER_DATA_PARSER_CONTROL_ENDOFDATA:   endOfList();   break;
	}
}

void KviListWindow::reset()
{
	outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("Reset"));
}

void KviListWindow::endOfList()
{
	m_pRequestButton->setEnabled(true);
}

void KviListWindow::startOfList()
{
	m_pListView->clear();
	m_pRequestButton->setEnabled(false);
}

void KviListWindow::processData(KviIrcMessage *msg)
{
	(void)new KviChannelListViewItem(m_pListView,msg->safeParam(1),msg->safeParam(2),msg->safeTrailing());

	if(KVI_OPTION_BOOL(KviOption_boolBeVerbose))
	{
		output(KVI_OUT_LIST,__tr("Processing list: %s"),msg->allParams());
	}
}

void KviListWindow::itemDoubleClicked(QListViewItem *it)
{
	m_pConsole->socket()->sendFmtData("join %s",((KviChannelListViewItem *)it)->m_szChan.ptr());
}


//
//#warning "Load & save properties of this kind of window"

//void KviListWindow::saveProperties()
//{
//	KviWindowProperty p;
//	p.rect = externalGeometry();
//	p.isDocked = isAttacched();
//	QValueList<int> l(m_pSplitter->sizes());
//	if(l.count() >= 1)p.splitWidth1 = *(l.at(0));
//	if(l.count() >= 2)p.splitWidth2 = *(l.at(1));
//	p.timestamp = m_pView->timestamp();
//	p.imagesVisible = m_pView->imagesVisible();
//	p.isMaximized = isAttacched() && isMaximized();
//	p.topSplitWidth1 = 0;
//	p.topSplitWidth2 = 0;
//	p.topSplitWidth3 = 0;
//	g_pOptions->m_pWinPropertiesList->setProperty(caption(),&p);
//}
//
//void KviListWindow::setProperties(KviWindowProperty *p)
//{
//	QValueList<int> l;
//	l.append(p->splitWidth1);
//	l.append(p->splitWidth2);
//	m_pVertSplitter->setSizes(l);
//	m_pIrcView->setTimestamp(p->timestamp);
//	m_pIrcView->setShowImages(p->imagesVisible);
//}

void KviListWindow::applyOptions()
{
	m_pIrcView->applyOptions();
	KviWindow::applyOptions();
}

#include "listwindow.moc"
