//
//   File : linkswindow.cpp
//   Creation date : Thu Dec 21 2001 12:41:18 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the linkss of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#include "linkswindow.h"

#include "kvi_debug.h"
#include "kvi_iconmanager.h"
#include "kvi_ircview.h"
#include "kvi_out.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_out.h"
#include "kvi_mirccntrl.h"
#include "kvi_themedlabel.h"

#include <qsplitter.h>
#include <qtooltip.h>
#include <qhbox.h>

extern KviPtrList<KviLinksWindow> * g_pLinksWindowList;

KviLinksWindow::KviLinksWindow(KviFrame * lpFrm,KviConsole * lpConsole)
: KviWindow(KVI_WINDOW_TYPE_LINKS,lpFrm,"links",lpConsole) , KviExternalServerDataParser()
{
	g_pLinksWindowList->append(this);

	m_pTopSplitter = new QSplitter(QSplitter::Horizontal,this,"top_splitter");

	// The button box on the left
	QHBox * box = new QHBox(m_pTopSplitter,"button_box");

	m_pRequestButton = new QToolButton(box,"request_button");
	m_pRequestButton->setUsesBigPixmap(false);
	m_pRequestButton->setPixmap(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_LINKS)));
	connect(m_pRequestButton,SIGNAL(clicked()),this,SLOT(requestLinks()));
	QToolTip::add(m_pRequestButton,__tr("Request links"));

	m_pInfoLabel = new KviThemedLabel(m_pTopSplitter,"info_label");

	connect(m_pConsole,SIGNAL(stateChange(KviConsole::ConnectionState)),
		this,SLOT(connectionStateChange(KviConsole::ConnectionState)));

	connectionStateChange(lpConsole->state());

	m_pSplitter = new QSplitter(QSplitter::Horizontal,this,"splitter");
	m_pVertSplitter = new QSplitter(QSplitter::Vertical,m_pSplitter,"vsplitter");

	m_pListView  = new QListView(m_pVertSplitter);
	m_pListView->addColumn(__tr("Link"));
	m_pListView->addColumn(__tr("Hops"));
	m_pListView->addColumn(__tr("Description"));
	m_pListView->setRootIsDecorated(true);
	m_pListView->setAllColumnsShowFocus(true);
	connect(m_pListView,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),
			this,SLOT(showHostPopup(QListViewItem *,const QPoint &,int)));

	m_pIrcView = new KviIrcView(m_pVertSplitter,lpFrm,this);

	m_pLinkList = new KviPtrList<KviLink>;
	m_pLinkList->setAutoDelete(true);

	m_pHostPopup = new QPopupMenu();
	connect(m_pHostPopup,SIGNAL(activated(int)),this,SLOT(hostPopupClicked(int)));

	setFocusHandler(m_pListView,this);

	m_pConsole->setLinksWindowPointer(this);

	m_szRootServer = __tr("(None)");
}

KviLinksWindow::~KviLinksWindow()
{
	g_pLinksWindowList->removeRef(this);
	m_pConsole->setLinksWindowPointer(0);
	delete m_pLinkList;
	delete m_pHostPopup;
}

void KviLinksWindow::getBaseLogFileName(KviStr &buffer)
{
	buffer.sprintf("LINKS_%d",console()->ircContextId());
}


void KviLinksWindow::requestLinks()
{
	if(m_pConsole->isConnected())
	{
		m_pConsole->socket()->sendFmtData("links");
		outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("Sent links request: waiting for reply..."));
		m_pRequestButton->setEnabled(false);
	} else {
		outputNoFmt(KVI_OUT_SYSTEMERROR,__tr("Can't request links: no active connection"));
	}
}

void KviLinksWindow::connectionStateChange(KviConsole::ConnectionState st)
{
	m_pRequestButton->setEnabled(st == KviConsole::Connected);
	if(st == KviConsole::Connected)
	{
		KviStr tmp(KviStr::Format,__tr("You're connected to %s (%s)"),m_pConsole->currentServerName(),m_pConsole->currentNetworkName());
		m_pInfoLabel->setText(tmp.ptr());
	} else {
		m_pInfoLabel->setText(__tr("You're not connected to a server: links cannot be requested"));
	}
}

QPixmap * KviLinksWindow::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_LINKS);
}

void KviLinksWindow::resizeEvent(QResizeEvent *e)
{
	int hght2 = m_pTopSplitter->sizeHint().height();
	m_pTopSplitter->setGeometry(0,0,width(),hght2);
	m_pSplitter->setGeometry(0,hght2,width(),height() - hght2);
}

QSize KviLinksWindow::sizeHint() const
{
	QSize ret(m_pSplitter->sizeHint().width(),
		m_pSplitter->sizeHint().height() + m_pTopSplitter->sizeHint().height());
	return ret;
}

void KviLinksWindow::fillCaptionBuffers()
{
	m_szPlainTextCaption.sprintf(__tr("Links for %s [IRC Context %u]"),m_szRootServer.ptr(),m_pConsole->ircContextId());

	m_szHtmlActiveCaption.sprintf(
		__tr("<nobr><font color=\"%s\"><b>Links for %s</b></font> " \
			"<font color=\"%s\">[IRC Context %u]</font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive).name().ascii(),
		m_szRootServer.ptr(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive2).name().ascii(),
		m_pConsole->ircContextId());

	m_szHtmlInactiveCaption.sprintf(
		__tr("<nobr><font color=\"%s\"><b>Links for %s</b></font> " \
			"<font color=\"%s\">[IRC Context %u]</font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive).name().ascii(),
		m_szRootServer.ptr(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive2).name().ascii(),
		m_pConsole->ircContextId());
}

void KviLinksWindow::die()
{
	close();
}

void KviLinksWindow::control(int message)
{
	switch(message)
	{
		case EXTERNAL_SERVER_DATA_PARSER_CONTROL_RESET:     reset(); break;
		case EXTERNAL_SERVER_DATA_PARSER_CONTROL_ENDOFDATA: endOfLinks(); break;
	}
}

void KviLinksWindow::endOfLinks()
{
	m_pRequestButton->setEnabled(true);

	m_pListView->clear();
	m_pListView->setSorting(-1);

	outputNoFmt(KVI_OUT_SYSTEMMESSAGE,"======================");
	outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("Received end of links."));
	outputNoFmt(KVI_OUT_SYSTEMMESSAGE,"======================");

	QListViewItem * it   = 0;
	QListViewItem * root = 0;

	int totalHosts  = 0;
	int totalHops   = 0;
	int maxHops     = 0;
	int avgHops     = 0;
	int directLinks = 0;
	int nearLinks   = 0;
	int brokenLinks = 0;
	int maxLinks    = 0;
	int farLinks    = 0;
	int wildServers = 0;

	KviStr szMaxHop,szMaxLinks;

	m_pListView->setUpdatesEnabled(false);
	for(KviLink *l=m_pLinkList->first();l;l=m_pLinkList->next()){
		totalHosts++;
		if(l->hops == 0)root = new QListViewItem(m_pListView,QString(l->host.ptr()),"0",QString(l->description.ptr()));
		else {
			totalHops += l->hops;
			if(l->hops < 4){
				nearLinks++;
				if(l->hops == 1)directLinks++;
			} else {
				if(l->hops >= 7)farLinks++;
			}
			if(l->hops == maxHops){
				szMaxHop.append(',');
				szMaxHop.append(l->host);
			} else if(l->hops > maxHops){
				maxHops = l->hops;
				szMaxHop = l->host;
			}
			if(l->host.contains('*'))wildServers++;
			it = insertLink(l);
			if(!it){
				output(KVI_OUT_SYSTEMERROR,__tr("Broken link : missing parent (%s) for %s (%d hops): %s (used /LINKS <mask> ?)"),
					l->parent.ptr(),l->host.ptr(),l->hops,l->description.ptr());
				brokenLinks++;
				KviStr tmp(KviStr::Format,__tr("%s : Parent link %s"),l->description.ptr(),l->parent.ptr());
				KviStr tmp2(KviStr::Format,"%d",l->hops);
				if(root)it = new QListViewItem(m_pListView,root,QString(l->host.ptr()),QString(tmp2.ptr()),QString(tmp.ptr()));
				else {
					outputNoFmt(KVI_OUT_SYSTEMERROR,__tr("Warning : No root link was sent by the server : the stats may be invalid."));
					it = new QListViewItem(m_pListView,QString(l->host.ptr()),QString(tmp2.ptr()),QString(tmp.ptr()));
				}
			} else {
				it = it->parent();
				if(it)
				{
					int links = it->childCount() + 1;
					if(links == maxLinks)
					{
						szMaxLinks.append(',');
						szMaxLinks.append(it->text(0));
					} else if(links > maxLinks)
					{
						maxLinks = links;
						szMaxLinks = it->text(0);
					}
				}
			}
		}
	}
	
	avgHops = ((totalHosts > 0) ? ((totalHops * 100) / totalHosts) : 0 );
	int nearProcent = ((totalHosts > 0) ? ((nearLinks * 10000) / totalHosts) : 0);
	int directProcent = ((totalHosts > 0) ? ((directLinks * 10000) / totalHosts) : 0);
	int midLinks = totalHosts - (farLinks + nearLinks + 1 + brokenLinks);
	if(midLinks < 0)midLinks = 0;
	int midProcent = ((totalHosts > 0) ? ((midLinks * 10000) / totalHosts) : 0);
	int farProcent = ((totalHosts > 0) ? ((farLinks * 10000) / totalHosts) : 0);

	outputNoFmt(KVI_OUT_LINKS,"======================");

//	if(!root)root = m_pListView->firstChild();
	if(root)
	{
		m_szRootServer = root->text(0);
		output(KVI_OUT_LINKS,__tr("%c%cLinks for %s"),KVI_TEXT_BOLD,KVI_TEXT_UNDERLINE,m_szRootServer.ptr());
		outputNoFmt(KVI_OUT_LINKS,"======================");
		output(KVI_OUT_LINKS,__tr("Total hosts %s : %d"),wildServers ? __tr("listed") : __tr("in the network"),totalHosts);
		if(wildServers)output(KVI_OUT_LINKS,__tr("Wild servers (hubs ?) : %d"),wildServers);
		output(KVI_OUT_LINKS,__tr("Direct links : %d (~%d.%d %)"),directLinks,directProcent / 100, directProcent % 100);
		output(KVI_OUT_LINKS,__tr("Near links (1 <= hops <= 3) : %d (~%d.%d %)"),nearLinks,nearProcent / 100,nearProcent % 100);
		output(KVI_OUT_LINKS,__tr("Mid range links (4 <= hops  <= 6) : %d (~%d.%d %)"),midLinks,midProcent / 100,midProcent % 100);
		output(KVI_OUT_LINKS,__tr("Far links (7 <= hops) : %d (~%d.%d %)"),farLinks,farProcent / 100,farProcent % 100);
		output(KVI_OUT_LINKS,__tr("Broken (unknown) links : %d"),brokenLinks);
		output(KVI_OUT_LINKS,__tr("Max links per host : %d [%s]"),maxLinks,szMaxLinks.ptr());
		output(KVI_OUT_LINKS,__tr("Total links : %d"),totalHops);
		output(KVI_OUT_LINKS,__tr("Maximum hops : %d [%s]"),maxHops,szMaxHop.ptr());
		output(KVI_OUT_LINKS,__tr("Average hops : ~%d.%d"),avgHops / 100,avgHops % 100);
	} else {
		m_szRootServer = __tr("(Unknown)");
		outputNoFmt(KVI_OUT_LINKS,__tr("Partial LINKS result : No stats available"));
	}
	outputNoFmt(KVI_OUT_LINKS,"======================");

	updateCaption();


	while(!m_pLinkList->isEmpty())m_pLinkList->removeFirst();

	m_pListView->setUpdatesEnabled(true);
	m_pListView->repaint();
}

QListViewItem * KviLinksWindow::insertLink(KviLink *l)
{
	__range_valid(l->hops > 0);
	QListViewItem * i = getItemByHost(l->parent.ptr(),0);
	QListViewItem * it = 0;
	if(!i)return 0;
	else {
		KviStr hops(KviStr::Format,"%d",l->hops);
		it = new QListViewItem(i,QString(l->host.ptr()),QString(hops.ptr()),QString(l->description.ptr()));
		i->setOpen(true);
	}
	return it;
}

QListViewItem * KviLinksWindow::getItemByHost(const char *host,QListViewItem * par)
{
	QListViewItem * i = (par ? par->firstChild() : m_pListView->firstChild());
	if(!i)return 0;
	while(i){
		KviStr tmp = i->text(0);
		if(kvi_strEqualCI(tmp.ptr(),host))return i;
		QListViewItem * ch = getItemByHost(host,i);
		if(ch)return ch;
		i = i->nextSibling();
	}
	return 0;
}

void KviLinksWindow::showHostPopup(QListViewItem *i,const QPoint &p,int)
{
	if(!i)return;
	KviStr host=i->text(0);
	if(host.isEmpty())return;
	m_pHostPopup->clear();
	KviStr tmp(KviStr::Format,"LINKS %s *",host.ptr());
	m_pHostPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_LINKS)),tmp.ptr());
	m_pHostPopup->insertSeparator();
	tmp.sprintf("TIME %s",host.ptr());
	m_pHostPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TIME)),tmp.ptr());
	tmp.sprintf("ADMIN %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("INFO %s",host.ptr());
	m_pHostPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SERVERINFO)),tmp.ptr());
	tmp.sprintf("MOTD %s",host.ptr());
	m_pHostPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_MOTD)),tmp.ptr());
	tmp.sprintf("VERSION %s",host.ptr());
	m_pHostPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_KVIRC)),tmp.ptr());
	tmp.sprintf("TRACE %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("USERS %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	m_pHostPopup->insertSeparator();
	tmp.sprintf("STATS c %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS d %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS h %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS i %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS k %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS l %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS m %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS o %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS t %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS u %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS y %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	tmp.sprintf("STATS z %s",host.ptr());
	m_pHostPopup->insertItem(tmp.ptr());
	m_pHostPopup->popup(p);
}

void KviLinksWindow::hostPopupClicked(int id)
{
	KviStr tmp = m_pHostPopup->text(id);
	if(tmp.hasData())
	{
		if(!m_pConsole->socket()->sendData(tmp.ptr()))output(KVI_OUT_SYSTEMERROR,__tr("You're not connected to a server"));
	}
}

void KviLinksWindow::reset()
{
	outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr("Reset"));
	while(!m_pLinkList->isEmpty())m_pLinkList->removeFirst();
}

void KviLinksWindow::processData(KviIrcMessage *msg)
{
	output(KVI_OUT_SYSTEMMESSAGE,__tr("Processing link: %s"),msg->allParams());
	KviLink * l = new KviLink;

	l->host = msg->safeParam(1);
	l->parent = msg->safeParam(2);

	const char * tr = msg->safeTrailing();

	if(isdigit(*tr))
	{
		const char *aux = tr;
		while(*tr && (isdigit(*tr)))tr++;
		KviStr tmp(aux,tr - aux);
		l->hops = tmp.toInt();
	} else {
		outputNoFmt(KVI_OUT_SYSTEMERROR,__tr("Broken message syntax: can't extract hops number, assuming 0"));
		l->hops = 0;
	}
	while(*tr && (*tr == ' '))tr++;
	l->description = tr;
	uint idx=0;
	for(KviLink *m=m_pLinkList->first();m;m=m_pLinkList->next())
	{
		if(m->hops >= l->hops)
		{
			m_pLinkList->insert(idx,l);
			return;
		}
		idx++;
	}
	m_pLinkList->append(l);
}

//#warning "Load & save properties of this kind of window"

//void KviLinksWindow::saveProperties()
//{
//	KviWindowProperty p;
//	p.rect = externalGeometry();
//	p.isDocked = isAttacched();
//	QValueList<int> l(m_pSplitter->sizes());
//	if(l.count() >= 1)p.splitWidth1 = *(l.at(0));
//	if(l.count() >= 2)p.splitWidth2 = *(l.at(1));
//	p.timestamp = m_pView->timestamp();
//	p.imagesVisible = m_pView->imagesVisible();
//	p.isMaximized = isAttacched() && isMaximized();
//	p.topSplitWidth1 = 0;
//	p.topSplitWidth2 = 0;
//	p.topSplitWidth3 = 0;
//	g_pOptions->m_pWinPropertiesList->setProperty(caption(),&p);
//}
//
//void KviLinksWindow::setProperties(KviWindowProperty *p)
//{
//	QValueList<int> l;
//	l.append(p->splitWidth1);
//	l.append(p->splitWidth2);
//	m_pVertSplitter->setSizes(l);
//	m_pIrcView->setTimestamp(p->timestamp);
//	m_pIrcView->setShowImages(p->imagesVisible);
//}

void KviLinksWindow::applyOptions()
{
	m_pIrcView->applyOptions();
	KviWindow::applyOptions();
}

#include "m_linkswindow.moc"
