//
//   File : libkvihelp.cpp
//   Creation date : Sun Aug 13 2000 03:00:00 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#include "libkvihelp.h"

#include "kvi_module.h"
#include "kvi_uparser.h"

#include "helpwidget.h"
#include "helpwindow.h"

#include "kvi_frame.h"

#include <qsplitter.h>

KviPtrList<KviHelpWidget> * g_pHelpWidgetList = 0;
KviPtrList<KviHelpWindow> * g_pHelpWindowList = 0;

/*
	@doc: help.search
	@type:
		command
	@title:
		help.search
	@short:
		Searches the documentation
	@syntax:
		help.search [-n] [-m] <key terms>
	@description:
		Finds the first available help browser in the current frame
		then searches the documentation for the specified <key terms>.
		If no help browser is available it creates one first:
		if the -m switch is present, the created browser is a MDI window,
		otherwise it is a static window.
		If the -n switch is present, the window creation is forced even
		if there are other help browsers already open.[br]
		The <key terms> are [b]space separated words[/b]
		to be matched inside the document body (logical AND).[br]
		This command is exported by the "help" module.
*/

static bool help_module_cmd_search(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"help_module_cmd_search");

	KviStr keys;
	if(!g_pUserParser->parseCmdFinalPart(c,keys))return false;

	if(keys.isEmpty())keys = "kvirc";

	if(!c->hasSwitch('n'))
	{
		// look for an already open help widget in this frame
		KviHelpWidget * w = (KviHelpWidget *)c->window()->frame()->child(
			"help_widget","KviHelpWidget");

		if(w)
		{
			w->doExactSearchFor(keys.ptr());
			return c->leaveContext();
		}
	}

	if(c->hasSwitch('m'))
	{
		KviHelpWindow *w = new KviHelpWindow(c->window()->frame(),"Help browser");
		w->helpWidget()->doExactSearchFor(keys.ptr());
		c->window()->frame()->addWindow(w);
	} else {
		KviHelpWidget *w = new KviHelpWidget(c->window()->frame()->splitter(),
			c->window()->frame(),true);
		w->doExactSearchFor(keys.ptr());
		w->show();
	}

	return c->leaveContext();
}

/*
	@doc: help.open
	@type:
		command
	@title:
		help.open
	@short:
		Shows a help document
	@syntax:
		help.open [-n] [-m] [document]
	@description:
		Finds the first available help browser in the current frame
		then opens the specified [document].
		If no [document] is specified it the documentation index is shown.
		If no help browser is available , a new one is created:
		if the -m switch is present, the created browser is a MDI window,
		otherwise it is a static window.
		If the -n switch is present, the window creation is forced even
		if there are other help browsers already open.[br]
		[document] can be an absolute path or a relative one: in this case
		the document is searched in the KVIrc documentation directory.[br]
		The help browser has limited html browsing capabilities: you can
		use it to view simple html files on your filesystem.[br]
		This command is exported by the "help" module.
*/

static bool help_module_cmd_open(KviModule *m,KviCommand *c)
{
	ENTER_CONTEXT(c,"help_module_cmd_open");

	KviStr doc;
	if(!g_pUserParser->parseCmdFinalPart(c,doc))return false;

	if(!c->hasSwitch('n'))
	{
		// look for an already open help widget in this frame
		KviHelpWidget * w = (KviHelpWidget *)c->window()->frame()->child(
			"help_widget","KviHelpWidget");

		if(w)
		{
			w->textBrowser()->setSource(doc.hasData() ? doc.ptr() : "index.html");
			return c->leaveContext();
		}
	}

	if(c->hasSwitch('m'))
	{
		KviHelpWindow *w = new KviHelpWindow(c->window()->frame(),"Help browser");
		w->textBrowser()->setSource(doc.hasData() ? doc.ptr() : "index.html");
		c->window()->frame()->addWindow(w);
	} else {
		KviHelpWidget *w = new KviHelpWidget(c->window()->frame()->splitter(),
			c->window()->frame(),true);
		w->textBrowser()->setSource(doc.hasData() ? doc.ptr() : "index.html");
		w->show();
	}

	return c->leaveContext();
}


static bool help_module_init(KviModule * m)
{
	g_pHelpWidgetList = new KviPtrList<KviHelpWidget>;
	g_pHelpWidgetList->setAutoDelete(false);
	g_pHelpWindowList = new KviPtrList<KviHelpWindow>;
	g_pHelpWindowList->setAutoDelete(false);

	m->registerCommand("open",help_module_cmd_open);
	m->registerCommand("search",help_module_cmd_search);
	return true;
}

static bool help_module_cleanup(KviModule *m)
{
	while(g_pHelpWidgetList->first())delete g_pHelpWidgetList->first();
	delete g_pHelpWidgetList;
	while(g_pHelpWindowList->first())g_pHelpWindowList->first()->close();
	delete g_pHelpWindowList;
	m->unregisterMetaObject("KviHelpWidget");
	m->unregisterMetaObject("KviHelpWindow");
	return true;
}

static bool help_module_can_unload(KviModule *m)
{
	return (g_pHelpWidgetList->isEmpty() && g_pHelpWindowList->isEmpty());
}

KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Help",                                                 // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2000 Szymon Stefanek (stefanek@tin.it)", // author & (C)
	"Help browser extension",
	help_module_init,
	help_module_can_unload,
	0,
	help_module_cleanup
};
