//
//   File : libkvignutella.cpp
//   Creation date : Mon Apr 16 2001 05:16:02 CEST by Szymon Stefanek
//
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


//
// For the mighty programmer that attempts to look at this code:
//
//   This is a simple Gnutella protocol implementation based
//   on the protocol specifications found commonly on the web.
//   I have written it "quickly", without taking care of commenting too much,
//   so it might be not suitable as "sample gnutella implementation".
//
//   Anyway , if you still want to understand something from this code
//   just be aware that this stuff runs on 3 + n concurrent threads
//
//   The GUI is on the main KVIrc thread
//   KviGnutellaThread is the Gnutella network routing thread (singleton)
//   KviGnutellaSearchThread is a search slave that serves Query requests (again a singleton)
//   KviGnutellaTransferThread is the thread that takes care of a single
//      upload or download transfer.
//

#define _LIBKVIGNUTELLA_CPP_

#include "kvi_app.h"
#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_config.h"

#include "gnutellawindow.h"
#include "gnutellashares.h"
#include "gnutellaoptions.h"

#include "kvi_frame.h"
#include "kvi_window.h"
#include "kvi_locale.h"
#include "kvi_thread.h"

#include "kvi_list.h"
#include <qsplitter.h>

KviGnutellaWindow              * g_pGnutellaWindow = 0;

KviMutex                       * g_pGnutellaOptionsMutex = 0;
KviGnutellaOptions             * g_pGnutellaOptions = 0;

unsigned int                     g_iGnutellaNextSharedFileId = 0;
KviPtrList<KviGnutellaSharedFile>   * g_pGnutellaSharedFiles = 0;
KviMutex                       * g_pGnutellaSharedFilesMutex = 0;
unsigned int                     g_uGnutellaSharedFilesCount = 0;
unsigned int                     g_uGnutellaSharedBytes = 0;

KviPtrList<KviGnutellaSearchQuery>  * g_pGnutellaSearchQueryList = 0;
KviMutex                       * g_pGnutellaSearchQueryMutex = 0;
KviPtrList<KviGnutellaSearchResult> * g_pGnutellaSearchResultList = 0;
KviMutex                       * g_pGnutellaSearchResultMutex = 0;

unsigned int                     g_uGnutellaLocalIpAddress = 0;

KviGnutellaOptionsDialog       * g_pGnutellaOptionsDialog = 0;

KviMutex                       * g_pGnutellaDownloadFilesMutex = 0;

unsigned int                     g_uGnutellaCurrentUploadTransfers = 0;
unsigned int                     g_uGnutellaCurrentDownloadTransfers = 0;
KviMutex                       * g_pGnutellaTransferCountersMutex = 0;

/*
	@doc: gnutella.open
	@type:
		command
	@title:
		gnutella.open
	@short:
		Opens a Gnutella window
	@syntax:
		gnutella.open
	@description:
		Opens (or raises) the gnutella window.
		You can find more informations in the [module:gnutella]gnutella module documentation[/module].
*/

static bool gnutella_module_cmd_open(KviModule *,KviCommand *c)
{
	ENTER_CONTEXT(c,"socketspy_module_cmd_open");


	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	if(g_pGnutellaWindow)g_pGnutellaWindow->delayedAutoRaise();
	else {
		g_pGnutellaWindow = new KviGnutellaWindow(c->window()->frame());
		c->window()->frame()->addWindow(g_pGnutellaWindow);
	}

	return c->leaveContext();
}

/*
	@doc: gnutella.connect
	@type:
		command
	@title:
		gnutella.connect
	@short:
		Attempts a new gnutella connection to the specified node
	@syntax:
		gnutella.connect <ip> <port>
	@description:
		Attempts a new gnutella connection to the node specified
		by the <ip> address and the <port>[br]
		You can find more informations in the [module:gnutella]gnutella module documentation[/module].
*/

static bool gnutella_module_cmd_connect(KviModule *,KviCommand *c)
{
	ENTER_CONTEXT(c,"socketspy_module_cmd_connect");


	KviStr ip;
	KviStr port;
	if(!g_pUserParser->parseCmdSingleToken(c,ip))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,port))return false;

	if(!g_pGnutellaWindow)c->warning(__tr("No Gnutella window: use gnutella.open first"));
	else {
		if(!g_pGnutellaWindow->connectTo(ip.ptr(),port.ptr()))c->warning(__tr("Invalid node address"));
	}

	return c->leaveContext();
}

static void gnutella_load_options()
{
	g_pGnutellaOptions->load();

	KviStr buf;

	g_pApp->getLocalKvircDirectory(buf,KviApp::ConfigPlugins,"libkvignutellashared.conf");

	KviConfig cfg2(buf.ptr());

	g_uGnutellaSharedFilesCount = 0;
	g_uGnutellaSharedBytes = 0;

	unsigned int uCount = cfg2.readUIntEntry("uCount",0);
	while(uCount > 0)
	{
		KviStr tmp(KviStr::Format,"file%u",uCount);
		KviStr fEntry = cfg2.readEntry(tmp.ptr(),"");
		gnutella_add_shared_file(fEntry.ptr());
		uCount--;
	}

}

static void gnutella_save_options()
{
	g_pGnutellaOptions->save();


	KviStr buf;

	g_pApp->getLocalKvircDirectory(buf,KviApp::ConfigPlugins,"libkvignutellashared.conf");

	KviConfig cfg2(buf.ptr());

	unsigned int uCount = (unsigned int)g_pGnutellaSharedFiles->count();

	cfg2.writeEntry("uCount",uCount);

	for(KviGnutellaSharedFile * f = g_pGnutellaSharedFiles->first();f && (uCount > 0);f = g_pGnutellaSharedFiles->next())
	{
		KviStr tmp(KviStr::Format,"file%u",uCount);
		KviStr szFileName = f->szPath;
		szFileName.append(f->szFileName);
		cfg2.writeEntry(tmp.ptr(),szFileName.ptr());
		uCount--;
	}
}

static bool gnutella_module_init(KviModule * m)
{
	g_pGnutellaWindow = 0;
	g_pGnutellaOptions = new KviGnutellaOptions();

	g_pGnutellaOptionsDialog = 0;

	g_pGnutellaOptionsMutex = new KviMutex();
	g_pGnutellaSharedFiles = new KviPtrList<KviGnutellaSharedFile>;
	g_pGnutellaSharedFiles->setAutoDelete(true);
	g_pGnutellaSharedFilesMutex = new KviMutex();
	g_pGnutellaSearchQueryList = new KviPtrList<KviGnutellaSearchQuery>;
	g_pGnutellaSearchQueryList->setAutoDelete(false);
	g_pGnutellaSearchQueryMutex = new KviMutex();
	g_pGnutellaSearchResultList = new KviPtrList<KviGnutellaSearchResult>;
	g_pGnutellaSearchResultList->setAutoDelete(false);
	g_pGnutellaSearchResultMutex = new KviMutex();
	g_pGnutellaDownloadFilesMutex = new KviMutex();
	g_pGnutellaTransferCountersMutex = new KviMutex();
	g_uGnutellaCurrentUploadTransfers = 0;
	g_uGnutellaCurrentDownloadTransfers = 0;


	gnutella_load_options();

	m->registerCommand("open",gnutella_module_cmd_open);
	m->registerCommand("connect",gnutella_module_cmd_connect);
	return true;
}

static bool gnutella_module_cleanup(KviModule *m)
{
	if(g_pGnutellaOptionsDialog)delete g_pGnutellaOptionsDialog;

	if(g_pGnutellaWindow)delete g_pGnutellaWindow; // the g_pGnutellaOptionsMutex MUST be unlocked now

	gnutella_save_options();

	delete g_pGnutellaOptions;
	g_pGnutellaOptions = 0;
	delete g_pGnutellaOptionsMutex;
	g_pGnutellaOptionsMutex = 0;
	delete g_pGnutellaSharedFiles;
	g_pGnutellaSharedFiles = 0;
	delete g_pGnutellaSharedFilesMutex;
	g_pGnutellaSharedFilesMutex = 0;
	g_pGnutellaSearchQueryList->setAutoDelete(true);
	delete g_pGnutellaSearchQueryList;
	g_pGnutellaSearchQueryList = 0;
	delete g_pGnutellaSearchQueryMutex;
	g_pGnutellaSearchQueryMutex = 0;
	g_pGnutellaSearchResultList->setAutoDelete(true);
	delete g_pGnutellaSearchResultList;
	g_pGnutellaSearchResultList = 0;
	delete g_pGnutellaSearchResultMutex;
	g_pGnutellaSearchResultMutex = 0;
	delete g_pGnutellaDownloadFilesMutex;
	g_pGnutellaDownloadFilesMutex = 0;
	delete g_pGnutellaTransferCountersMutex;
	g_pGnutellaTransferCountersMutex = 0;

	m->unregisterMetaObject("KviGnutellaWindow");
	m->unregisterMetaObject("KviGnutellaOptionsDialog");
	m->unregisterMetaObject("KviGnutellaTransferTab");
//	m->unregisterMetaObject("KviGnutellaSmartDownloadTab");
	m->unregisterMetaObject("KviGnutellaSearchTab");
	return true;
}

static bool gnutella_module_can_unload(KviModule *m)
{
	return (g_pGnutellaWindow == 0);
}

/*
	@doc: gnutella
	@type:
		module
	@short:
		The Gnutella module: a gnutella node implementation
	@title:
		The Gnutella module
	@body:
		[big]The Gnutella network[/big][br]
		Gnutella is a widely-used distributed file-sharing system.[br]
		Basically, the protocol defines a gnutella-network as a set of interconnected nodes.
		A node (almost) always acts as client, server and router:[br]
		- as a client, when it is downloading a file from a server.[br]
		- as a server, when it is uploading a file to a client.[br]
		- as a router, at any time (when connected to the gnutella-network at all).[br]
		You can find a lot of informations about this protocol by simply
		searching for "gnutella" in your favorite search engine.[br]

		[big]The KVIrc implementation[/big]
		First of all, you have to open the gnutella window with the command:[br]
		[example]
			[cmd]gnutella.open[/cmd]
		[/example]
		The window is splitted in 4 tabs that control all the module actions:[br]
		The first tab is the [b]"Network" controller[/b].[br]
		The list on the left side shows the currently active (or attempted) network connections.[br]
		By double clicking on a connection item, you will be able to see some statistics
		about the node it refers to.[br]
		The buttons in the left part allow you to kill connections.[br]
		The list on the right contains a set of ip addresses of the hosts that
		are have a gnutella-application running and are (probably) connected to some part
		of the gnutella-network.[br]
		You can add ip addresses to this list by typing them in the edit box just below
		and pressing enter.[br]
		A Gnutella-session usually starts by connecting to a Gnutella-hostcache machine
		(in order to get a list of gnutella nodes): the button with the arrow on the right
		shows a menu with a list of the widely known gnutella-hostcaches.
		You can select a hostcache from that list to automatically add it to the ip address list.[br]
		Once you have some ip addreses in the list, you can select some of these , and click
		"Connect to selected".
		After a first connection has been estabilished, you're officially on the gnutella-network.[br]
		If you check the corresponding option in the Network controller, KVIrc will
		attempt to keep up a number of network connections (the number is specified in the options dialog).[br]
		Take care in choosing a good number of connections: if you set it too low (1-2),
		your searches will lead to a small amount of results, if you set it too big (this depends on
		your available bandwidth), you might get overloaded with data to route, and your
		bandwidth will soon saturate.[br]
		Personally I choose 4 as number of minimum connections to keep up, but I've seen clients
		that have 25 as default (!!!).[br]
		[br]
		The second tab is the [b]"Search" window[/b].[br]
		From here you can trigger your search queries and check the results.[br]
		Since every client has its own rules about matching the search criteria (and some clients
		send out garbage query-hits at all), you will
		probably get more search results than expected. For this reason , a local
		filter can be applied to the current search results.[br]
		By double-clicking on a search result, you can attempt to start a download session
		for the file.[br]
		[br]
		The third tab is the [b]"Shares" window[/b].[br]
		Here you can see the list of the files that you are offering for download.[br]
		You can add files to this list by clicking the "add" button on the right.[br]
		[br]
		The last tab is ther [b]"Transfers" window[/b].[br]
		Here you can monitor the currently running transfers.[br]
		[br]
		[br]
		[big]Smart download[/big]
		This plugin has a basic "smart download" feature. Basically, you choose
		a file to download and the engine goes and downloads it , full stop :)).[br]
		No , it is not exactly like that: this program has no human brain emulation yet.[br]
		Anyway, with some luck, it will be able to download the file without any human interaction:
		it will retry interrupted or failed (for non critical reasons) downloads, up to
		a specified number of attempts. This is useful when a server from which you want
		to download replies with a "503 Server Busy" or "503 Service Unavailable" HTTP message:
		this usually means that the server has actually too much work to satisfy the request,
		but later might be able to. You don't need to manually retry the download, the engine
		will do it for you.[br]
		The plugin will retry also in some other cases; for example, when the download
		speed drops under a certain (specified) rate for a certain amount of time (this is called "stall"):
		the connection will be dropped and restarted again: this often helps (don't ask me why).[br]
		When there is no way to download the file by retrying the transfer from the same host (there
		are some "hidden" rules to decide this, including the maximum retry amount),
		the smart download agent will look for another query hit with the same filename and file size;
		if found, it will attempt to download from that host, if not found, it will perform a query
		and look again for a hit in 30 seconds.[br]
		That's all the intelligence that I have put in there; that's not too much: you still need
		a lot of luck.[br]
		[big]The clue[/big][br]
		Once you have readed about the gnutella protocol, and understood the basic
		meaning of the four tabs, you should be ready to start sharing.[br]
		It is just a matter of trying.[br]
		I had a lot of fun in implementing this module: a sort of personal challenge.[br]
		Enjoy using it :)
*/


KVIMODULEEXPORTDATA KviModuleInfo kvirc_module_info=
{
	"Gnutella",                                             // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)", // author & (C)
	"A Simple Gnutella protocl implementation",
	gnutella_module_init,
	gnutella_module_can_unload,
	0,
	gnutella_module_cleanup
};
