//
//   File : gnutella.cpp
//   Creation date : Mon Apr 16 2001 05:23:22 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the socketspys of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "gnutellawindow.h"
#include "gnutellathread.h"
#include "gnutellashares.h"
#include "gnutellatransfer.h"
#include "gnutellaoptions.h"
#include "gnutellasearch.h"
//#include "gnutellasmart.h"

#include "kvi_iconmanager.h"
#include "kvi_ircview.h"
#include "kvi_out.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_console.h"
#include "kvi_themedlabel.h"
#include "kvi_netutils.h"
#include "kvi_memmove.h"
#include "kvi_mirccntrl.h"
#include "kvi_filedialog.h"
#include "kvi_dns.h"
#include "kvi_error.h"

#include <qsplitter.h>
#include <qvbox.h>
#include <qtooltip.h>
#include <qtabwidget.h>
#include <qcheckbox.h>
#include <qdatetime.h>
#include <qfileinfo.h>
#include <qtoolbutton.h>

extern KviPtrList<KviGnutellaSharedFile> * g_pGnutellaSharedFiles;
extern KviMutex                     * g_pGnutellaSharedFilesMutex;
extern unsigned int                   g_uGnutellaSharedFilesCount;
extern unsigned int                   g_uGnutellaSharedBytes;
extern KviGnutellaOptionsDialog     * g_pGnutellaOptionsDialog;


KviGnutellaNodeItem::KviGnutellaNodeItem(QListView * par,int id,KviStr szDescription,KviStr szIp,KviStr szPort)
: QListViewItem(par)
{
	m_iNodeId = id;
	m_bOnNetwork = false;
	KviStr tmp(KviStr::Format,"%d",id);
	setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SERVER)));
	setText(0,tmp.ptr());
	setText(1,szDescription.ptr());
	setText(2,szIp.ptr());
	setText(3,szPort.ptr());
}

KviGnutellaNodeItem::~KviGnutellaNodeItem()
{
}

void KviGnutellaNodeItem::setStatus(const char *stat)
{
	setText(4,stat);
}

void KviGnutellaNodeItem::setBandwidthStats(const char * stat)
{
	setText(5,stat);
}



KviGnutellaWindow::KviGnutellaWindow(KviFrame * lpFrm)
: KviWindow(KVI_WINDOW_TYPE_GNUTELLA,lpFrm,"gnutella",0)
{
	g_pGnutellaWindow = this;

	m_iConnectedNodes = 0;
	m_pGnutellaThread = 0;

	m_pDnsList = new KviPtrList<KviDns>;
	m_pDnsList->setAutoDelete(true);

	m_pTopSplitter = new QSplitter(QSplitter::Horizontal,this,"top_splitter");
	m_pStatusLabel = new KviThemedLabel(m_pTopSplitter,"status_label");
	m_pNetworkBandwidthLabel = new KviThemedLabel(m_pTopSplitter,"network_bandwidth_label");

	QHBox * tbox = new QHBox(m_pTopSplitter);

	m_pOptionsButton = new QToolButton(tbox,"options_button");
	m_pOptionsButton->setUsesBigPixmap(false);
	m_pOptionsButton->setIconSet(
		*(g_pIconManager->getSmallIcon(KVI_SMALLICON_HELP)));
	connect(m_pOptionsButton,SIGNAL(clicked()),this,SLOT(showOptionsDialog()));

	QToolTip::add(m_pOptionsButton,__tr("Options"));


	m_pSplitter = new QSplitter(QSplitter::Horizontal,this,"splitter");

	m_pVertSplitter = new QSplitter(QSplitter::Vertical,m_pSplitter,"vert_splitter");


	QTabWidget * tab = new QTabWidget(m_pVertSplitter);


	////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
	// Network tab
	//

	m_pNetworkSplitter = new QSplitter(QSplitter::Horizontal,tab,"network_splitter");


	QVBox * vbox = new QVBox(m_pNetworkSplitter);
	vbox->setSpacing(2);

	m_pNetworkView = new QListView(vbox);
	m_pNetworkView->setShowSortIndicator(true);
	m_pNetworkView->setAllColumnsShowFocus(true);
	m_pNetworkView->setSelectionMode(QListView::Extended);
	connect(m_pNetworkView,SIGNAL(selectionChanged()),this,SLOT(networkViewSelectionChanged()));
	connect(m_pNetworkView,SIGNAL(doubleClicked(QListViewItem *)),this,SLOT(networkViewDoubleClicked(QListViewItem *)));
	m_pNetworkView->addColumn(__tr("Id"));
	m_pNetworkView->addColumn(__tr("Type"));
	m_pNetworkView->addColumn(__tr("Host"));
	m_pNetworkView->addColumn(__tr("Port"));
	m_pNetworkView->addColumn(__tr("Status"));
	m_pNetworkView->addColumn(__tr("Bandwidth (bytes/sec)"));

	QHBox * hbox = new QHBox(vbox);
	hbox->setSpacing(2);

	m_pRemoveConnectionsButton = new QPushButton(__c2q(__tr("Remove selected nodes")),hbox);
	connect(m_pRemoveConnectionsButton,SIGNAL(clicked()),this,SLOT(removeSelectedConnections()));
	m_pRemoveConnectionsButton->setEnabled(false);

	QPushButton * b = new QPushButton(__c2q(__tr("Remove non connected nodes")),hbox);
	connect(b,SIGNAL(clicked()),this,SLOT(removeNonConnectedNodes()));

	hbox = new QHBox(vbox);
	hbox->setSpacing(2);

	QCheckBox * cbox = new QCheckBox(__c2q(__tr("Accept incoming connections")),hbox);
	cbox->setChecked(g_pGnutellaOptions->m_bAcceptConnections);
	connect(cbox,SIGNAL(toggled(bool)),this,SLOT(acceptConnectionsToggled(bool)));

	cbox = new QCheckBox(__c2q(__tr("Auto-connect to reach min-connections")),hbox);
	cbox->setChecked(g_pGnutellaOptions->m_bAutoConnectToReachMinConnections);
	connect(cbox,SIGNAL(toggled(bool)),this,SLOT(autoConnectToggled(bool)));

	vbox = new QVBox(m_pNetworkSplitter);
	vbox->setSpacing(2);

	m_pHostList = new QListView(vbox);
	m_pHostList->setShowSortIndicator(true);
	m_pHostList->setSelectionMode(QListView::Extended);
	connect(m_pHostList,SIGNAL(selectionChanged()),this,SLOT(hostListSelectionChanged()));
	m_pHostList->setAllColumnsShowFocus(true);
	m_pHostList->addColumn(__tr("Host"));
	m_pHostList->addColumn(__tr("Port"));
	m_pHostList->addColumn(__tr("Hops"));

	hbox = new QHBox(vbox);
	hbox->setSpacing(2);

	QLabel * l = new QLabel(__tr(" Host (ip:port) :"),hbox);

	m_pHostInput = new QLineEdit(hbox);
	connect(m_pHostInput,SIGNAL(textChanged(const QString &)),this,SLOT(hostInputTextChanged(const QString &)));
	connect(m_pHostInput,SIGNAL(returnPressed()),this,SLOT(addHostClicked()));

	m_pAddHostButton = new QPushButton(__tr("Add"),hbox);
	m_pAddHostButton->setEnabled(false);
	connect(m_pAddHostButton,SIGNAL(clicked()),this,SLOT(addHostClicked()));

	hbox = new QHBox(vbox);
	hbox->setSpacing(2);

	m_pConnectToSelectedButton = new QPushButton(__c2q(__tr("Connect to selected")),hbox);
	m_pConnectToSelectedButton->setEnabled(false);
	connect(m_pConnectToSelectedButton,SIGNAL(clicked()),this,SLOT(connectToSelectedClicked()));

	QToolButton * btn = new QToolButton(DownArrow,hbox);
	m_pHostCachePopup = new QPopupMenu(btn);

	for(QStringList::Iterator it = g_pGnutellaOptions->m_cacheHosts.begin();it != g_pGnutellaOptions->m_cacheHosts.end(); ++it)
	{
		m_pHostCachePopup->insertItem(*it);
	}

	connect(m_pHostCachePopup,SIGNAL(activated(int)),this,SLOT(connectToHostcache(int)));

	btn->setPopup(m_pHostCachePopup);
	btn->setPopupDelay(1);
	
	hbox->setStretchFactor(m_pConnectToSelectedButton,1);

	tab->addTab(m_pNetworkSplitter,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_GNUTELLA)),__c2q(__tr("Network")));


	// Transfer tab

	m_pSearchTab = new KviGnutellaSearchTab(tab,this);

	tab->addTab(m_pSearchTab,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SEARCH)),__c2q(__tr("Search")));


	// Search tab
/*
	QHBox * searchbox = new QHBox(tab);
	searchbox->setSpacing(2);

	vbox = new QVBox(searchbox);
	vbox->setSpacing(2);

	searchbox->setStretchFactor(vbox,5);

	m_pSearchResults = new QListView(vbox);
	m_pSearchResults->setShowSortIndicator(true);
	m_pSearchResults->setAllColumnsShowFocus(true);
	m_pSearchResults->setSelectionMode(QListView::Extended);

	m_pSearchResults->addColumn(__tr("File"),200);
	m_pSearchResults->addColumn(__tr("Size (Bytes)"));
	m_pSearchResults->addColumn(__tr("Speed (KBit/sec)"));
	m_pSearchResults->addColumn(__tr("Host"));
	m_pSearchResults->addColumn(__tr("Port"));

	// People use horrible filenames...
	m_pSearchResults->setColumnWidthMode(0,QListView::Manual);

	connect(m_pSearchResults,SIGNAL(doubleClicked(QListViewItem *)),this,SLOT(searchResultDoubleClicked(QListViewItem *)));
	connect(m_pSearchResults,SIGNAL(selectionChanged()),this,SLOT(searchResultsSelectionChanged()));

	hbox = new QHBox(vbox);

	hbox->setSpacing(2);

	m_pClearSearchResults = new QPushButton(__tr("Clear all results"),hbox);
	connect(m_pClearSearchResults,SIGNAL(clicked()),this,SLOT(clearSearchResults()));

	m_pClearSearchResultsSelected = new QPushButton(__tr("Clear selected"),hbox);
	m_pClearSearchResultsSelected->setEnabled(false);
	connect(m_pClearSearchResultsSelected,SIGNAL(clicked()),this,SLOT(clearSearchResultsSelected()));

	m_pDownloadSelected = new QPushButton(__tr("Download selected"),hbox);
	m_pDownloadSelected->setEnabled(false);
	connect(m_pDownloadSelected,SIGNAL(clicked()),this,SLOT(downloadSelected()));


	hbox = new QHBox(vbox);

	hbox->setSpacing(2);

	l = new QLabel(__tr("Search string:"),hbox);

	m_pSearchString = new QLineEdit(hbox);
	connect(m_pSearchString,SIGNAL(textChanged(const QString &)),this,SLOT(searchStringChanged(const QString &)));
	connect(m_pSearchString,SIGNAL(returnPressed()),this,SLOT(doSearch()));

	l = new QLabel(__tr("Min speed (KBit/sec):"),hbox);

	m_pSearchMinSpeed = new QLineEdit(hbox);
	connect(m_pSearchMinSpeed,SIGNAL(textChanged(const QString &)),this,SLOT(searchMinSpeedChanged(const QString &)));
	connect(m_pSearchMinSpeed,SIGNAL(returnPressed()),this,SLOT(doSearch()));

	m_pSearchButton = new QPushButton(__tr("Search"),hbox);
	connect(m_pSearchButton,SIGNAL(clicked()),this,SLOT(doSearch()));
	m_pSearchButton->setEnabled(false);

	vbox = new QVBox(searchbox);
	vbox->setSpacing(2);

//#warning "APPLY LOCAL FILTER AUTOMATICALLY"
//	m_pEnableLocalFilter = new QCheckBox(__tr("Apply local filter automatically"),vbox);

//	QFrame * f = new QFrame(vbox);
//	f->setFrameStyle(QFrame::HLine | QFrame::Sunken);

	l = new QLabel(__tr("File name regexp:"),vbox);

	m_pFilterRegexpEdit = new QLineEdit(vbox);

	l = new QLabel(__tr("Minimum size:"),vbox);

	m_pMinimumSizeEdit = new QLineEdit(vbox);

	l = new QLabel(__tr("Maximum size:"),vbox);

	m_pMaximumSizeEdit = new QLineEdit(vbox);

	l = new QLabel(__tr("Minimum speed:"),vbox);

	m_pMinimumSpeedEdit = new QLineEdit(vbox);

	m_pApplyFilterNowButton = new QPushButton(__tr("Apply filter now"),vbox);
	connect(m_pApplyFilterNowButton,SIGNAL(clicked()),this,SLOT(applySearchFilter()));

	l = new QLabel("",vbox);
	vbox->setStretchFactor(l,1);


	tab->addTab(searchbox,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SEARCH)),__tr("Search"));
*/
	// Shared files tab

	vbox = new QVBox(tab);

	vbox->setSpacing(2);

	m_pSharedFilesList = new QListView(vbox);
	m_pSharedFilesList->setSelectionMode(QListView::Extended);
	m_pSharedFilesList->setAllColumnsShowFocus(true);
	m_pSharedFilesList->addColumn(__tr("ID"));
	m_pSharedFilesList->addColumn(__tr("File name"));
	m_pSharedFilesList->addColumn(__c2q(__tr("File size (bytes)")));
	m_pSharedFilesList->addColumn(__tr("Path"));
	connect(m_pSharedFilesList,SIGNAL(selectionChanged()),this,SLOT(sharedFilesSelectionChanged()));

	hbox = new QHBox(vbox);
	hbox->setSpacing(2);

	m_pSharedFilesCountLabel = new QLabel(__c2q(__tr("No shared files")),hbox);


	m_pRemoveSelectedFilesButton = new QPushButton(__c2q(__tr("Remove selected")),hbox);
	m_pRemoveSelectedFilesButton->setEnabled(false);
	connect(m_pRemoveSelectedFilesButton,SIGNAL(clicked()),this,SLOT(removeSelectedFilesClicked()));

	b = new QPushButton(__tr("Add..."),hbox);
	connect(b,SIGNAL(clicked()),this,SLOT(addSharedFiles()));

	hbox = new QHBox(vbox);

	hbox->setSpacing(2);

	cbox = new QCheckBox(__c2q(__tr("Enable file sharing")),hbox);
	cbox->setChecked(g_pGnutellaOptions->m_bEnableFileSharing);
	connect(cbox,SIGNAL(toggled(bool)),this,SLOT(enableFileSharingToggled(bool)));

	cbox = new QCheckBox(__c2q(__tr("Spy local search results")),hbox);
	cbox->setChecked(g_pGnutellaOptions->m_bSpyLocalSearchResults);
	connect(cbox,SIGNAL(toggled(bool)),this,SLOT(spyLocalSearchResultsToggled(bool)));

	tab->addTab(vbox,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_FILES)),__c2q(__tr("Shared files")));

	// Transfer tab

	m_pTransferTab = new KviGnutellaTransferTab(tab,this);

	tab->addTab(m_pTransferTab,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_TRANSFER)),__tr("Transfer"));

/*
	// Smart download tab

	m_pSmartDownloadTab = new KviGnutellaSmartDownloadTab(tab,this);

	tab->addTab(m_pSmartDownloadTab,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_IDEA)),__tr("Smart download"));
*/

	fillSharedFilesList();

	m_pIrcView = new KviIrcView(m_pVertSplitter,lpFrm,this);

	// Ensure proper focusing
	setFocusHandlerNoClass(m_pIrcView,this,"QLineEdit");

	m_pGnutellaThread = new KviGnutellaThread();
	if(!m_pGnutellaThread->start())
		outputNoFmt(KVI_OUT_SYSTEMERROR,__tr("ops..can't start the slave gnutella thread"));

	updateStatusLabel();
}

KviGnutellaWindow::~KviGnutellaWindow()
{
	while(KviDns *dns = m_pDnsList->first())
	{
		KviStr * port = (KviStr *)dns->releaseAuxData();
		delete port;
		m_pDnsList->removeFirst();
	}
	delete m_pDnsList;
//	debug("KviGnutellaWindow::~KviGnutellaWindow destroying the gnutella thread (%d)",m_pGnutellaThread);
	delete m_pGnutellaThread; // will terminate it (is a KviSensitiveThread)
//	debug("KviGnutellaWindow::~KviGnutellaWindow gnutella thread dead");
	KviThreadManager::killPendingEvents(this);
	g_pGnutellaWindow = 0;
}

void KviGnutellaWindow::showOptionsDialog()
{
	if(!g_pGnutellaOptionsDialog)g_pGnutellaOptionsDialog = new KviGnutellaOptionsDialog();
	g_pGnutellaOptionsDialog->show();
	g_pGnutellaOptionsDialog->raise();
}

void KviGnutellaWindow::connectToHostcache(int id)
{
	KviStr t = m_pHostCachePopup->text(id);
	addHostToList(t);
}
/*
void KviGnutellaWindow::applySearchFilter()
{
	bool bApplyRegex = true;
	QString regex = m_pFilterRegexpEdit->text();

	if(regex.isEmpty() || regex.isNull())bApplyRegex = false;

	KviStr minsiz = m_pMinimumSizeEdit->text();

	bool bApplyUMinSize;
	unsigned int uMinSize = minsiz.toUInt(&bApplyUMinSize);

	KviStr maxsiz = m_pMaximumSizeEdit->text();

	bool bApplyUMaxSize;
	unsigned int uMaxSize = maxsiz.toUInt(&bApplyUMaxSize);

	KviStr minspd = m_pMinimumSpeedEdit->text();

	bool bApplyMinSpeed;
	unsigned int uMinSpeed = minspd.toUInt(&bApplyMinSpeed);


	if(!(bApplyMinSpeed || bApplyUMaxSize || bApplyUMinSize || bApplyRegex))return;

	KviPtrList<KviGnutellaHitItem> l;
	l.setAutoDelete(true);

	QRegExp re(regex,false);

	KviGnutellaHitItem * it = (KviGnutellaHitItem *)m_pSearchResults->firstChild();
	while(it)
	{
		bool bMatch = true;

		if(bApplyRegex)bMatch = (re.find(it->text(0),0) != -1);
		if(bMatch)
		{
			if(bApplyUMinSize)bMatch = (it->m_info.uSize >= uMinSize);
			if(bMatch)
			{
				if(bApplyUMaxSize)bMatch = (it->m_info.uSize <= uMaxSize);
				if(bMatch)
				{
					if(bApplyMinSpeed)bMatch = (it->m_info.uSpeed >= uMinSpeed);
				}
			}
		}

		if(!bMatch)l.append(it);

		it = (KviGnutellaHitItem *)it->nextSibling();
	}
}
*/
void KviGnutellaWindow::updateStatusLabel()
{
	if(m_iConnectedNodes > 0)
	{
		KviStr tmp(KviStr::Format,__tr("On network: %d nodes connected"),m_iConnectedNodes);
		m_pStatusLabel->setText(tmp.ptr());
		m_pSearchTab->updateSearchButtonState(QString::null);
	} else {
		m_pStatusLabel->setText(__tr("Not on network"));
		m_pSearchTab->updateSearchButtonState(QString::null);
	}
}

void KviGnutellaWindow::hostInputTextChanged(const QString &text)
{
	KviStr tmp = text;
	int idx = tmp.findFirstIdx(':');
	if(idx != -1)
	{
		KviStr host = tmp.left(idx);
		tmp.cutLeft(idx + 1);
		bool bOk = true;
		struct in_addr ia;
		bOk = kvi_stringIpToBinaryIp(host.ptr(),&ia);
		if(bOk)
		{
			unsigned short int uPort = tmp.toUShort(&bOk);
		}
		m_pAddHostButton->setEnabled(bOk);
	}
}

void KviGnutellaWindow::addHostToList(const char * host,const char * port)
{
	QListViewItem * it = new QListViewItem(m_pHostList,host,port,"-");
}

void KviGnutellaWindow::addHostToList(KviStr &buf)
{
	int idx = buf.findFirstIdx(':');
	if(idx != -1)
	{
		KviStr host = buf.left(idx);
		buf.cutLeft(idx + 1);
		bool bOk = true;
		struct in_addr ia;
		unsigned short int uPort = buf.toUShort(&bOk);
		if(bOk)
		{
			bOk = kvi_stringIpToBinaryIp(host.ptr(),&ia);
		}
		if(bOk)addHostToList(host.ptr(),buf.ptr());
		else {
			// lookup!
			KviDns * dns = new KviDns();
			if(dns->lookup(host.ptr(),KviDns::IpV4))
			{
				dns->setAuxData(new KviStr(buf));
				connect(dns,SIGNAL(lookupDone(KviDns *)),this,SLOT(hostLookupDone(KviDns *)));
				m_pDnsList->append(dns);
				output(KVI_OUT_SYSTEMMESSAGE,__tr("Looking up host %s"),host.ptr());
			} else delete dns;
		}
	}
}

void KviGnutellaWindow::hostLookupDone(KviDns *dns)
{
	KviStr * port = (KviStr *)dns->releaseAuxData();
	if(dns->state() != KviDns::Success)
	{
		output(KVI_OUT_SYSTEMERROR,__tr("Failed to lookup host %s: %s"),dns->query(),kvi_getErrorString(dns->error()));
	} else {
		addHostToList(dns->firstIpAddress(),port->ptr());
	}
	delete port;
	m_pDnsList->removeRef(dns);
}

void KviGnutellaWindow::addHostClicked()
{
	KviStr tmp = m_pHostInput->text();
	addHostToList(tmp);
	m_pHostInput->setText("");
	m_pAddHostButton->setEnabled(false);
}

void KviGnutellaWindow::hostListSelectionChanged()
{
	int iSelCount = 0;
	QListViewItem * it = m_pHostList->firstChild();
	while(it)
	{
		if(it->isSelected())iSelCount++;
		it = it->nextSibling();
	}
	m_pConnectToSelectedButton->setEnabled(iSelCount > 0);
}

void KviGnutellaWindow::connectToSelectedClicked()
{
	KviPtrList<QListViewItem> l;
	l.setAutoDelete(true);
	QListViewItem * it = m_pHostList->firstChild();
	while(it)
	{
		if(it->isSelected())
		{
			KviStr host = it->text(0);
			KviStr port = it->text(1);
			connectTo(host.ptr(),port.ptr());
			l.append(it);
		}
		it = it->nextSibling();
	}
//	hostListSelectionChanged();
}

QPixmap * KviGnutellaWindow::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_GNUTELLA);
}

void KviGnutellaWindow::resizeEvent(QResizeEvent *e)
{
	int hght2 = m_pOptionsButton->sizeHint().height();
	m_pTopSplitter->setGeometry(0,0,width(),hght2);
	m_pSplitter->setGeometry(0,hght2,width(),height() - hght2);
}

QSize KviGnutellaWindow::sizeHint() const
{
	QSize ret(m_pSplitter->sizeHint().width(),
		m_pIrcView->sizeHint().height() + m_pStatusLabel->sizeHint().height());

	return ret;
}

void KviGnutellaWindow::getBaseLogFileName(KviStr &buffer)
{
	buffer = "Gnutella_window";
}

void KviGnutellaWindow::fillCaptionBuffers()
{
	m_szPlainTextCaption.sprintf(__tr("Gnutella"));

	m_szHtmlActiveCaption.sprintf(
		__tr("<nobr><font color=\"%s\"><b>Gnutella</b></font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive).name().ascii(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive2).name().ascii());

	m_szHtmlInactiveCaption.sprintf(
		__tr("<nobr><font color=\"%s\"><b>Gnutella</b></font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive).name().ascii(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive2).name().ascii());
}


void KviGnutellaWindow::applyOptions()
{
	m_pIrcView->applyOptions();
	m_pStatusLabel->applyOptions();
	KviWindow::applyOptions();
}


KviGnutellaNodeItem * KviGnutellaWindow::findNodeItem(int id)
{
	KviGnutellaNodeItem * it = (KviGnutellaNodeItem *)m_pNetworkView->firstChild();
	while(it)
	{
		if(it->id() == id)return it;
		it = (KviGnutellaNodeItem *)it->nextSibling();
	}
	return 0;
}

void KviGnutellaWindow::handleEvent(KviThreadEvent *e)
{
	switch(e->id())
	{
		case KVI_GNUTELLA_THREAD_EVENT_COMPOSITE:
		{
//			debug("Composite event");
			KviGnutellaCompositeThreadEvent * ev = (KviGnutellaCompositeThreadEvent *)e;

			for(KviThreadEvent *te = ev->m_pEventList->first();te;te = ev->m_pEventList->next())
			{
				handleEvent(te);
			}
		}
		break;
		case KVI_THREAD_EVENT_WARNING:
		{
			KviStr * s = ((KviThreadDataEvent<KviStr> *)e)->getData();
			outputNoFmt(KVI_OUT_SYSTEMWARNING,__tr_no_xgettext(s->ptr()));
			delete s;
		}
		break;
		case KVI_THREAD_EVENT_MESSAGE:
		{
			KviStr * s = ((KviThreadDataEvent<KviStr> *)e)->getData();
			outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr_no_xgettext(s->ptr()));
			delete s;
		}
		break;
		case KVI_THREAD_EVENT_ERROR:
		{
			KviStr * s = ((KviThreadDataEvent<KviStr> *)e)->getData();
			outputNoFmt(KVI_OUT_SYSTEMERROR,__tr_no_xgettext(s->ptr()));
			delete s;
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_QUERY_HIT:
		{
			KviGnutellaHitThreadEvent * ev = (KviGnutellaHitThreadEvent *)e;
		//	output(KVI_OUT_SYSTEMMESSAGE,__tr("Hit packet with %u entries from %s:%u"),ev->m_pHitList->count(),
				//ev->m_szIp.ptr(),ev->m_uPort);
			for(KviGnutellaHit * h = ev->m_pHitList->first();h;h = ev->m_pHitList->next())
			{
//				KviGnutellaHitItem * it = new KviGnutellaHitItem(m_pSearchResults,h,ev);

				m_pSearchTab->addQueryHit(h,ev);
			}
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_BANDWIDTH_STATS:
		{
			if(((KviGnutellaThreadEvent *)e)->m_iNodeId != -1)
			{
				KviGnutellaNodeItem * it = findNodeItem(((KviGnutellaThreadEvent *)e)->m_iNodeId);
				if(it)it->setBandwidthStats(((KviGnutellaThreadEvent *)e)->m_szData.ptr());
			} else {
				// Total network bandwidth
				m_pNetworkBandwidthLabel->setText(__tr_no_xgettext(((KviGnutellaThreadEvent *)e)->m_szData.ptr()));
			}
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_INCOMING_TRANSFER:
		{
			KviGnutellaIncomingTransferInfo * inf = ((KviThreadDataEvent<KviGnutellaIncomingTransferInfo> *)e)->getData();
			m_pTransferTab->addIncomingTransfer(inf);
			delete inf;
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_PUSH_REQUEST:
		{
			KviGnutellaPushRequestInfo * inf = ((KviThreadDataEvent<KviGnutellaPushRequestInfo> * )e)->getData();
			m_pTransferTab->addPushTransfer(inf);
			delete inf;
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_NEW_NODE:
		{
			KviStr tmp;
			tmp.setNum(((KviGnutellaThreadEvent *)e)->m_uPort);

			KviStr dir = ((KviGnutellaThreadEvent *)e)->m_bIncoming ? __tr("incoming") : __tr("outgoing");

			KviGnutellaNodeItem * it = new KviGnutellaNodeItem(m_pNetworkView,
				((KviGnutellaThreadEvent *)e)->m_iNodeId,
				dir.ptr(),
				((KviGnutellaThreadEvent *)e)->m_szIp.ptr(),tmp.ptr());

			if(((KviGnutellaThreadEvent *)e)->m_bIncoming)it->setStatus(__tr("Connected (Incoming handshake)"));
			else it->setStatus(__tr("Just born"));
			output(KVI_OUT_SYSTEMMESSAGE,__tr("New %s node: %s:%s"),dir.ptr(),
				((KviGnutellaThreadEvent *)e)->m_szIp.ptr(),tmp.ptr());
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_NODE_DEAD:
		{
			KviGnutellaNodeItem * it = findNodeItem(((KviGnutellaThreadEvent *)e)->m_iNodeId);
			if(it)
			{
				KviStr szHost = it->text(2);
				KviStr szPort = it->text(3);
				output(KVI_OUT_SYSTEMMESSAGE,__tr("Lost node: %s:%s : %s"),
					szHost.ptr(),szPort.ptr(),__tr_no_xgettext(((KviGnutellaThreadEvent *)e)->m_szData.ptr()));

				if(it->onNetwork())
				{
					m_iConnectedNodes--;
					updateStatusLabel();
				}
				delete it;
			}
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTING_V4:
		{
			KviGnutellaNodeItem * it = findNodeItem(((KviGnutellaThreadEvent *)e)->m_iNodeId);
			if(it)it->setStatus(__tr("Connecting (proto 0.4)"));
		}
		case KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTING_V6:
		{
			KviGnutellaNodeItem * it = findNodeItem(((KviGnutellaThreadEvent *)e)->m_iNodeId);
			if(it)it->setStatus(__tr("Connecting (proto 0.6)"));
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_PUSH_FAILURE:
		{
//			debug("Push failure here");
			KviGnutellaPushFailureInfo * inf = ((KviThreadDataEvent<KviGnutellaPushFailureInfo> *)e)->getData();
			output(KVI_OUT_SYSTEMERROR,__tr("[transfer %u]: %s"),inf->uTransferId,__tr_no_xgettext(inf->szErrorString.ptr()));
			m_pTransferTab->pushFailure(inf);
			delete inf;
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_NODE_HANDSHAKE:
		{
			KviGnutellaNodeItem * it = findNodeItem(((KviGnutellaThreadEvent *)e)->m_iNodeId);
			if(it)it->setStatus(__tr("Connected (Handshake)"));
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTED_V4:
		{
			KviGnutellaNodeItem * it = findNodeItem(((KviGnutellaThreadEvent *)e)->m_iNodeId);
			if(it)
			{
				it->setStatus(__tr("Connected (On Network) (0.4)"));
				it->setOnNetwork();
				m_iConnectedNodes++;
				updateStatusLabel();
			}
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTED_V6:
		{
			KviGnutellaNodeItem * it = findNodeItem(((KviGnutellaThreadEvent *)e)->m_iNodeId);
			if(it)
			{
				it->setStatus(__tr("Connected (On Network) (0.6)"));
				it->setOnNetwork();
				m_iConnectedNodes++;
				updateStatusLabel();
			}
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_NODES_CAUGHT:
		{
			KviGnutellaNodesCaughtThreadEvent * ev = (KviGnutellaNodesCaughtThreadEvent *)e;
//#warning "If verbose...output it"
			//output(KVI_OUT_SYSTEMMESSAGE,__tr("Caught %u new nodes"),ev->m_pNodeList->count());
			for(KviGnutellaNodeInfo * i = ev->m_pNodeList->first();i;i = ev->m_pNodeList->next())
			{
				QListViewItem * it = new QListViewItem(m_pHostList,i->szIp.ptr(),i->szPort.ptr(),i->szHops.ptr());
			}
//				KviStr tmp(KviStr::Format,"%u",((KviGnutellaThreadEvent *)e)->m_uPort);
//				QListViewItem * it = new QListViewItem(m_pHostList,((KviGnutellaThreadEvent *)e)->m_szIp.ptr(),
//					tmp.ptr(),((KviGnutellaThreadEvent *)e)->m_szData.ptr());
		}
		break;
		case KVI_GNUTELLA_THREAD_EVENT_NODE_STATS:
		{
			KviGnutellaNodeStatsThreadEvent * ev = (KviGnutellaNodeStatsThreadEvent *)e;
			output(KVI_OUT_SERVERINFO,__tr("Stats for node %d       %c: %s:%u"),ev->m_iNodeId,KVI_TEXT_BOLD,ev->m_szIp.ptr(),ev->m_uPort);
			output(KVI_OUT_SERVERINFO,__tr("  Connection type       %c: %s"),KVI_TEXT_BOLD,ev->m_bIncoming ? __tr("incoming") : __tr("outgoing"));
			output(KVI_OUT_SERVERINFO,__tr("  Protocol              %c: %s"),KVI_TEXT_BOLD,ev->m_szProtocol.ptr());
			ev->m_szPublicHeaders.replaceAll("\r\n","\n--- ");
			output(KVI_OUT_SERVERINFO,__tr("  Public headers        :\n%s"),ev->m_szPublicHeaders.ptr());
//			QDateTime dt;
//			dt.setTime_t(ev->m_statsStartTime);
//			KviStr tmp = dt.toString();
//			int secs   = dt.secsTo(QDateTime::currentDateTime());
//			if(secs == 0)secs = 1;
//			int avg    = ev->m_stats.uTotalPacketsReceived / secs;
//			output(KVI_OUT_SERVERINFO,__tr("  Connection started at %c: %s"),KVI_TEXT_BOLD,tmp.ptr());
			output(KVI_OUT_SERVERINFO,__tr("  Packets processed     %c: %u"),KVI_TEXT_BOLD,ev->m_stats.uTotalPacketsReceived);
			output(KVI_OUT_SERVERINFO,__tr("  Packets enqueued      %c: %u"),KVI_TEXT_BOLD,ev->m_stats.uTotalPacketsSent);
			output(KVI_OUT_SERVERINFO,__tr("  Bytes received        %c: %u"),KVI_TEXT_BOLD,ev->m_stats.uTotalBytesReceived);
			output(KVI_OUT_SERVERINFO,__tr("  Bytes sent            %c: %u"),KVI_TEXT_BOLD,ev->m_stats.uTotalBytesSent);

			outputNoFmt(KVI_OUT_SERVERINFO,__tr("  Dropped packets:"));
			output(KVI_OUT_SERVERINFO,__tr("  Duplicated            %c: %u"),KVI_TEXT_BOLD,
				ev->m_stats.uTotalPingsDuplicated + ev->m_stats.uTotalQueriesDuplicated + ev->m_stats.uTotalPushesDuplicated);
			output(KVI_OUT_SERVERINFO,__tr("  Looped-back           %c: %u"),KVI_TEXT_BOLD,
				ev->m_stats.uTotalPingsLoopedBack + ev->m_stats.uTotalQueriesLoopedBack);
			output(KVI_OUT_SERVERINFO,__tr("  Expired               %c: %u"),KVI_TEXT_BOLD,
				ev->m_stats.uTotalPingsTtlExpired + ev->m_stats.uTotalPongsTtlExpired + ev->m_stats.uTotalQueriesTtlExpired + \
				ev->m_stats.uTotalQueryHitsTtlExpired + ev->m_stats.uTotalPushesTtlExpired);
			output(KVI_OUT_SERVERINFO,__tr("  Unroutable (no path)  %c: %u"),KVI_TEXT_BOLD,
				ev->m_stats.uTotalPongsUnroutable + ev->m_stats.uTotalQueryHitsUnroutable + ev->m_stats.uTotalPushesUnroutable);
			output(KVI_OUT_SERVERINFO,__tr("  Unknown               %c: %u"),KVI_TEXT_BOLD,ev->m_stats.uTotalUnknownPackets);

			outputNoFmt(KVI_OUT_SERVERINFO,__tr("  Incoming packet types:"));


			unsigned long int uDropped = (ev->m_stats.uTotalPingsLoopedBack + ev->m_stats.uTotalPingsTtlExpired + ev->m_stats.uTotalPingsDuplicated);
			output(KVI_OUT_SERVERINFO,__tr("  Ping                  %c: %u (%u routed, %u dropped (%u looped back, %u ttl expired, %u duplicated))"),
				KVI_TEXT_BOLD,ev->m_stats.uTotalPings,ev->m_stats.uTotalPings - uDropped,uDropped,
				ev->m_stats.uTotalPingsLoopedBack,ev->m_stats.uTotalPingsTtlExpired,ev->m_stats.uTotalPingsDuplicated);

			uDropped = (ev->m_stats.uTotalPongsTtlExpired + ev->m_stats.uTotalPongsUnroutable);
			output(KVI_OUT_SERVERINFO,__tr("  Pong                  %c: %u (%u routed, %u directed to me, %u dropped (%u ttl expired, %u unroutable))"),
				KVI_TEXT_BOLD,ev->m_stats.uTotalPongs,ev->m_stats.uTotalPongs - (uDropped + ev->m_stats.uTotalPongsDirectedToMe),
				ev->m_stats.uTotalPongsDirectedToMe,uDropped,
				ev->m_stats.uTotalPongsTtlExpired,ev->m_stats.uTotalPongsUnroutable);

			uDropped = (ev->m_stats.uTotalQueriesLoopedBack + ev->m_stats.uTotalQueriesTtlExpired + ev->m_stats.uTotalQueriesDuplicated);
			output(KVI_OUT_SERVERINFO,__tr("  Query                 %c: %u (%u routed, %u dropped (%u looped back, %u ttl expired, %u duplicated))"),
				KVI_TEXT_BOLD,ev->m_stats.uTotalQueries,ev->m_stats.uTotalQueries - uDropped,uDropped,
				ev->m_stats.uTotalQueriesLoopedBack,ev->m_stats.uTotalQueriesTtlExpired,ev->m_stats.uTotalQueriesDuplicated);

			uDropped = (ev->m_stats.uTotalQueryHitsTtlExpired + ev->m_stats.uTotalQueryHitsUnroutable);
			output(KVI_OUT_SERVERINFO,__tr("  QueryHit                  %c: %u (%u routed, %u directed to me, %u dropped (%u ttl expired, %u unroutable))"),
				KVI_TEXT_BOLD,ev->m_stats.uTotalQueryHits,ev->m_stats.uTotalQueryHits - (uDropped + ev->m_stats.uTotalQueryHitsDirectedToMe),
				ev->m_stats.uTotalQueryHitsDirectedToMe,uDropped,
				ev->m_stats.uTotalQueryHitsTtlExpired,ev->m_stats.uTotalQueryHitsUnroutable);

			uDropped = (ev->m_stats.uTotalPushesTtlExpired + ev->m_stats.uTotalPushesUnroutable + ev->m_stats.uTotalPushesDuplicated);
			output(KVI_OUT_SERVERINFO,__tr("  Push                  %c: %u (%u routed, %u directed to me, %u dropped (%u ttl expired, %u unroutable, %u duplicated))"),
				KVI_TEXT_BOLD,ev->m_stats.uTotalPushes,ev->m_stats.uTotalPushes - (uDropped + ev->m_stats.uTotalPushesDirectedToMe),
				ev->m_stats.uTotalPushesDirectedToMe,uDropped,
				ev->m_stats.uTotalPushesTtlExpired,ev->m_stats.uTotalPushesUnroutable,ev->m_stats.uTotalPushesDuplicated);

		}
		break;
	}
}

bool KviGnutellaWindow::event(QEvent *e)
{
	if(((int)(e->type())) == KVI_THREAD_EVENT)
	{
		handleEvent((KviThreadEvent *)e);
		return true;
	}


	return KviWindow::event(e);
}

bool KviGnutellaWindow::connectTo(const char * szIp,const char * szPort)
{
	struct in_addr addr;
	if(!kvi_stringIpToBinaryIp(szIp,&addr))return false;
	KviStr szP = szPort;
	bool bOk;
	unsigned short int uPort = szP.toUShort(&bOk);
	if(!bOk)return false;

	KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(KVI_GNUTELLA_WINDOW_EVENT_CONNECT_TO_NODE);
	e->m_szIp = szIp;
	e->m_uPort = uPort;
	m_pGnutellaThread->enqueueEvent(e);
	return true;
}

void KviGnutellaWindow::networkViewDoubleClicked(QListViewItem *it)
{
	if(it)
	{
		KviStr tmp = it->text(0);
		bool bOk;
		int id = tmp.toInt(&bOk);
		if(bOk)
		{
			KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(KVI_GNUTELLA_WINDOW_EVENT_NODE_STATS);
			e->m_iNodeId = id;
			m_pGnutellaThread->enqueueEvent(e);
		}
	}
}

void KviGnutellaWindow::removeSelectedConnections()
{
	KviGnutellaNodeItem * it = (KviGnutellaNodeItem *)m_pNetworkView->firstChild();
	while(it)
	{
		if(it->isSelected())
		{
			KviStr szId = it->text(0);

			bool bOk;
			unsigned short int iId = szId.toUInt(&bOk);
			if(bOk)
			{
				KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(KVI_GNUTELLA_WINDOW_EVENT_KILL_NODE);
				e->m_iNodeId = iId;
				m_pGnutellaThread->enqueueEvent(e);	
			}
		}

		it = (KviGnutellaNodeItem *)it->nextSibling();
	}
}

void KviGnutellaWindow::removeNonConnectedNodes()
{
//	KviGnutellaNodeItem * it = (KviGnutellaNodeItem *)m_pNetworkView->firstChild();
//	while(it)
//	{
//		if(!(it->onNetwork()))
//		{
//			KviStr szId = it->text(0);
//
//			bool bOk;
//			unsigned short int iId = szId.toUInt(&bOk);
//			if(bOk)
//			{
//				KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(KVI_GNUTELLA_WINDOW_EVENT_KILL_NODE);
//				e->m_iNodeId = iId;
//				m_pGnutellaThread->enqueueEvent(e);	
//			}
//		}
//
//		it = (KviGnutellaNodeItem *)it->nextSibling();
//	}
	m_pGnutellaThread->enqueueEvent(new KviThreadEvent(KVI_GNUTELLA_WINDOW_EVENT_KILL_NON_CONNECTED));
}

void KviGnutellaWindow::networkViewSelectionChanged()
{
	int iSelCount = 0;
	QListViewItem * it = m_pNetworkView->firstChild();
	while(it)
	{
		if(it->isSelected())iSelCount++;
		it = it->nextSibling();
	}
	m_pRemoveConnectionsButton->setEnabled(iSelCount > 0);
}

void KviGnutellaWindow::triggerUpdateOptions()
{
	if(m_pGnutellaThread)
	{
		KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(KVI_GNUTELLA_WINDOW_EVENT_UPDATE_OPTIONS);
		m_pGnutellaThread->enqueueEvent(e);	
	}
}

void KviGnutellaWindow::acceptConnectionsToggled(bool bChecked)
{
	g_pGnutellaOptionsMutex->lock();
	g_pGnutellaOptions->m_bAcceptConnections = bChecked;
	g_pGnutellaOptionsMutex->unlock();
	triggerUpdateOptions();
}

void KviGnutellaWindow::autoConnectToggled(bool bChecked)
{
	g_pGnutellaOptionsMutex->lock();
	g_pGnutellaOptions->m_bAutoConnectToReachMinConnections = bChecked;
	g_pGnutellaOptionsMutex->unlock();
	triggerUpdateOptions();
}

//-------------------------------------------------------------------------------------
// Do Search
//
/*
void KviGnutellaWindow::enableSearchButton()
{
	KviStr tmp = m_pSearchMinSpeed->text();
	tmp.stripWhiteSpace();
	KviStr text = m_pSearchString->text();
	m_pSearchButton->setEnabled((tmp.isUnsignedNum() || tmp.isEmpty()) &&
		text.hasData() && (m_iConnectedNodes > 0));
}

void KviGnutellaWindow::doSearch()
{
	KviStr tmp = m_pSearchString->text();
	if(tmp.hasData())
	{
		KviStr num = m_pSearchMinSpeed->text();
		unsigned short int uShort;

		if(num.isEmpty())uShort = 0;
		else {
			bool bOk;
			uShort = num.toUShort(&bOk);
			if(!bOk)return;
		}

		KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(KVI_GNUTELLA_WINDOW_EVENT_DO_SEARCH);
		e->m_szData = tmp;
		e->m_uPort = uShort;
		m_pGnutellaThread->enqueueEvent(e);	
		m_pSearchString->setText("");
		m_pSearchMinSpeed->setText("");
		enableSearchButton();
	}
}

void KviGnutellaWindow::searchStringChanged(const QString &)
{
	enableSearchButton();
}

void KviGnutellaWindow::searchMinSpeedChanged(const QString &)
{
	enableSearchButton();
}

void KviGnutellaWindow::searchResultsSelectionChanged()
{
	QListViewItem * it = m_pSearchResults->firstChild();
	while(it)
	{
		if(it->isSelected())
		{
			m_pClearSearchResultsSelected->setEnabled(true);
			m_pDownloadSelected->setEnabled(true);
			return;
		}
		it = it->nextSibling();
	}
	m_pClearSearchResultsSelected->setEnabled(false);
	m_pDownloadSelected->setEnabled(false);
}

void KviGnutellaWindow::clearSearchResults()
{
	m_pSearchResults->clear();
}

void KviGnutellaWindow::clearSearchResultsSelected()
{
	KviPtrList<QListViewItem> l;
	l.setAutoDelete(true);
	QListViewItem * it = m_pSearchResults->firstChild();
	while(it)
	{
		if(it->isSelected())l.append(it);
		it = it->nextSibling();
	}
}

void KviGnutellaWindow::downloadSelected()
{
	QListViewItem * it = m_pSearchResults->firstChild();
	while(it)
	{
		if(it->isSelected())searchResultDoubleClicked(it);
		it = it->nextSibling();
	}
}

void KviGnutellaWindow::searchResultDoubleClicked(QListViewItem *it)
{
	if(!it)return;


	m_pTransferTab->addOutgoingTransfer(&(((KviGnutellaHitItem *)it)->m_info));
}
*/
void KviGnutellaWindow::startDownload(KviGnutellaQueryHitInfo * inf)
{
	m_pTransferTab->addOutgoingTransfer(inf);
}

KviGnutellaQueryHitInfo * KviGnutellaWindow::findBestQueryHit(const char * fileName,unsigned int uSize,KviPtrList<KviStr> * excludeHosts)
{
	return m_pSearchTab->findBestQueryHit(fileName,uSize,excludeHosts);
/*
	KviPtrList<KviGnutellaHitItem> l;
	l.setAutoDelete(false);

	KviGnutellaHitItem * it = (KviGnutellaHitItem *)m_pSearchResults->firstChild();
	while(it)
	{
		if(it->m_info.uSize == uSize)
		{
			if(kvi_strEqualCS(it->m_info.szName.ptr(),fileName))
			{
				bool bGotIt = false;
				if(excludeHosts)
				{
					for(KviStr * s = excludeHosts->first();s && !bGotIt;s = excludeHosts->next())
					{
						if(kvi_strEqualCS(it->m_info.szIp.ptr(),s->ptr()))bGotIt = true;
					}
				}
				if(!bGotIt)
				{
					// append with greater speeds first
					int idx = 0;
					for(KviGnutellaHitItem * h = l.first();h && !bGotIt;h = l.next())
					{
						if(h->m_info.uSpeed < it->m_info.uSpeed)
						{
							l.insert(idx,it);
							bGotIt = true;
						}
						idx++;
					}
					if(!bGotIt)l.append(it);
				}
			}
		}
		it = (KviGnutellaHitItem *)it->nextSibling();
	}
	KviGnutellaQueryHitInfo * inf = l.first() ? &(l.first()->m_info) : 0;
	return inf;
*/
}


void KviGnutellaWindow::sharedFilesSelectionChanged()
{
	QListViewItem * it = m_pSharedFilesList->firstChild();
	while(it)
	{
		if(it->isSelected())
		{
			m_pRemoveSelectedFilesButton->setEnabled(true);
			return;
		}
		it = it->nextSibling();
	}
	m_pRemoveSelectedFilesButton->setEnabled(false);
}

void KviGnutellaWindow::removeSelectedFilesClicked()
{
	KviPtrList<QListViewItem> l;
	l.setAutoDelete(true);
	QListViewItem * it = m_pSharedFilesList->firstChild();
	g_pGnutellaSharedFilesMutex->lock();
	while(it)
	{
		if(it->isSelected())
		{
			KviStr tmp = it->text(0);
			bool bOk;
			unsigned int uId = tmp.toUInt(&bOk);
			if(bOk)gnutella_remove_shared_file(uId);
			l.append(it);
		}
		it = it->nextSibling();
	}
	g_pGnutellaSharedFilesMutex->unlock();
	updateSharedFilesCount();
	sharedFilesSelectionChanged();
}

void KviGnutellaWindow::enableFileSharingToggled(bool bChecked)
{
	g_pGnutellaOptionsMutex->lock();
	g_pGnutellaOptions->m_bEnableFileSharing = bChecked;
	g_pGnutellaOptionsMutex->unlock();
	triggerUpdateOptions();
}

void KviGnutellaWindow::spyLocalSearchResultsToggled(bool bChecked)
{
	g_pGnutellaOptionsMutex->lock();
	g_pGnutellaOptions->m_bSpyLocalSearchResults = bChecked;
	g_pGnutellaOptionsMutex->unlock();
	triggerUpdateOptions();
}



void KviGnutellaWindow::fillSharedFilesList()
{
	m_pSharedFilesList->clear();
	g_pGnutellaSharedFilesMutex->lock();

	for(KviGnutellaSharedFile * f = g_pGnutellaSharedFiles->first();f;f = g_pGnutellaSharedFiles->next())
	{
		KviStr tmp(KviStr::Format,"%u",f->uId);
		KviStr tmp2(KviStr::Format,"%u",f->uFileSize);
		QListViewItem * it = new QListViewItem(m_pSharedFilesList,tmp.ptr(),f->szFileName.ptr(),tmp2.ptr(),f->szPath.ptr());
	}
	g_pGnutellaSharedFilesMutex->unlock();
	updateSharedFilesCount();
	sharedFilesSelectionChanged();
}

void KviGnutellaWindow::updateSharedFilesCount()
{
	KviStr tmp(KviStr::Format,"Sharing %d files (%u bytes)",g_uGnutellaSharedFilesCount,g_uGnutellaSharedBytes);
	m_pSharedFilesCountLabel->setText(tmp.ptr());
}

void KviGnutellaWindow::addSharedFiles()
{
	QStringList buffer;
	if(KviFileDialog::askForOpenFileNames(buffer,__tr("Select the files to share")))
	{
		g_pGnutellaSharedFilesMutex->lock();
		for(QStringList::Iterator it = buffer.begin();it != buffer.end();++it)
		{
			if(KviGnutellaSharedFile * f = gnutella_add_shared_file((*(it)).latin1()))
			{
				KviStr tmp(KviStr::Format,"%u",f->uId);
				KviStr tmp2(KviStr::Format,"%u",f->uFileSize);
				QListViewItem * it = new QListViewItem(m_pSharedFilesList,tmp.ptr(),f->szFileName.ptr(),tmp2.ptr(),f->szPath.ptr());
			}
		}
		g_pGnutellaSharedFilesMutex->unlock();
	}
	updateSharedFilesCount();
}

#include "m_gnutellawindow.moc"
