#ifndef _GNUTELLATRANSFER_H_
#define _GNUTELLATRANSFER_H_
//
//   File : gnutellatransfer.h
//   Creation date : Tue Apr 22 2001 02:51:21 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the gnutellas of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_thread.h"
#include "kvi_string.h"
#include "kvi_socket.h"

#include <qvbox.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qpopupmenu.h>
#include <qtimer.h>

typedef struct _KviGnutellaQueryHitInfo KviGnutellaQueryHitInfo;
typedef struct _KviGnutellaIncomingTransferInfo KviGnutellaIncomingTransferInfo;
typedef struct _KviGnutellaPushRequestInfo KviGnutellaPushRequestInfo;
typedef struct _KviGnutellaPushFailureInfo KviGnutellaPushFailureInfo;


typedef struct _KviGnutellaTransferRetryInfo
{
	// A transfer is retry-able
	unsigned int       uTransferId;
	bool               bRetryFromSameHost;
	unsigned int       uRetryAfterSecs;
} KviGnutellaTransferRetryInfo;

class KviGnutellaWindow;

class KviGnutellaTransferItem;

class KviGnutellaTransferTab : public QVBox
{
	Q_OBJECT
public:
	KviGnutellaTransferTab(QWidget * par,KviGnutellaWindow * wnd);
	~KviGnutellaTransferTab();
public:
	KviGnutellaWindow * m_pGnutellaWindow;

	QListView         * m_pView;
	QPushButton       * m_pRemoveSelectedButton;
	QPushButton       * m_pClearDeadButton;
	QPopupMenu        * m_pContextPopup;

	unsigned int        m_uLastClickedItemId;

	QTimer            * m_pRetryTimer;

public:
	KviGnutellaTransferItem * findTransferItem(unsigned int uId);
	KviGnutellaTransferItem * addOutgoingTransfer(KviGnutellaQueryHitInfo * inf);
	void addIncomingTransfer(KviGnutellaIncomingTransferInfo * inf);
	void addPushTransfer(KviGnutellaPushRequestInfo * inf);
	void pushFailure(KviGnutellaPushFailureInfo * inf);
protected:
	virtual bool event(QEvent *e);
	void enableClearDeadTransfers();
	void startRetryTimer();
	void stopRetryTimer();
	KviGnutellaTransferItem * retryTransfer(unsigned int uTransferId);
	void handleRetryHint(KviGnutellaTransferRetryInfo * inf);
	void searchForTransferData(unsigned int uTransferId);
protected slots:
	void selectionChanged();
	void clearDeadTransfers();
	void removeSelectedTransfers();
	void rightButtonPressed(QListViewItem *it,const QPoint &pnt,int);
	void removeCurrentTransfer();
	void killCurrentTransfer();
	void retryCurrentTransfer();
	void searchCurrentTransfer();
	void retryTimerTimeout();
};




class KviGnutellaTransferThread : public KviSensitiveThread
{
public:
	enum RetryHint          { NoRetry , RetryAnotherHost , RetrySameHost };
	enum TransferDirection  { Recv , Send };
	enum ConnectionType     { Active , Passive };
	enum TransferState      { Idle , Connecting , WaitingForPush , Handshaking , Transferring , Dead , WaitingForRetry };
public:
	KviGnutellaTransferThread(KviGnutellaTransferTab * tab,unsigned int uId);
	~KviGnutellaTransferThread();
protected:
	unsigned int             m_uId;
	KviGnutellaTransferTab * m_pTab;

	ConnectionType           m_connectionType;
	TransferDirection        m_transferDirection;
	TransferState            m_state;
	RetryHint                m_retryHint;
	unsigned int             m_uRetryDelayInSecs;

	kvi_socket_t             m_sock;

	KviStr                   m_szIp;
	unsigned short int       m_uPort;

	KviStr                   m_szFilePath;                 // path where the file is saved/loaded from
	KviStr                   m_szFileName;                 // name of the file being downloaded/uploaded
	unsigned int             m_uTransferEndPosition;       // end position of the transfer
	unsigned int             m_uTotalFileSize;             // the total size of the file (expected)
	unsigned int             m_uFileIndex;                 // index of the file in shared files (upload)
	unsigned int             m_uResumePosition;            // start position of the upload/download

	KviStr                   m_szLockFileName;             // complete filename (with path) of the lock file

	unsigned char            m_serventId[16];

	char                   * m_pInBuffer;
	unsigned int             m_uInBufferDataLen;
	unsigned int             m_uInBufferRealLen;

	KviPtrList<KviStr>          * m_pHttpHeader;

	bool                     m_bTerminateRequestSeen;
	bool                     m_bDecrementCurrentUploadTransfersOnDeath;
	bool                     m_bDecrementCurrentDownloadTransfersOnDeath;

	KviStr                   m_szStateChangeReason;

	bool                     m_bSuicideOnStall;
	unsigned int             m_uStallTransferRate;
	unsigned int             m_uStallTimeoutInMSecs;


protected:

	bool readData();
	bool waitForHttpHeader();
	bool waitForData();
	bool readHttpHeader();
	KviStr * processHttpHeaderLine();
	void postErrorEvent(const char * msg);

	bool lockIncomingFile();
	void unlockIncomingFile();
	

	bool selectForWrite(int iTimeoutInSecs);
	int  selectForWriteStep();
	bool processInternalEvents();
	bool pushAttempt();
	bool connectToRemoteHost();
	bool sendHttpGet();
	bool sendHttpOk();
	bool sendHttpGiv();
	bool expectHttpGet();
	bool sendBuffer(const char * buffer,int bufLen,int iTimeoutInSecs);
	bool selectForRead(int iTimeoutInSecs);
	int  selectForReadStep();
	bool expectHttpGiv();
	bool incomingHandshake();
	bool expectHttpOk();

	bool sendHttpError(const char * error,const char * warning);
	bool receiveFile();
	bool sendFile();


	virtual void run();

	void setState(TransferState st);

	bool closeSock(const char * error = 0,const char * reason = 0);

	void postProgressEvent(unsigned int uActualSize,unsigned int uTotalSizeToRecv,unsigned int uRate,bool bStalled = false);
	void setRetryHint(bool bSameHost,unsigned int uRetryDelay);
	void postRetryEvent(bool bSameHost,unsigned int uRetryDelay);
public:
	const char * remoteHostIp(){ return m_szIp.ptr(); };
	unsigned short int remoteHostPort(){ return m_uPort; };
	const char * fileName(){ return m_szFileName; };
	unsigned int fileSize(){ return m_uTotalFileSize; };
	const unsigned char * serventId(){ return m_serventId; };
	void setRemoteHost(const char *szIp,unsigned short int uPort)
		{ m_szIp = szIp; m_uPort = uPort; };
	void setConnectionType(ConnectionType t)
		{ m_connectionType = t; };
	void setTransferDirection(TransferDirection t)
		{ m_transferDirection = t; };
	void setFileInfo(const char * fName,unsigned int uSize,unsigned int uIndex)
		{ m_szFileName = fName; m_uTransferEndPosition = uSize; m_uTotalFileSize = uSize; m_uFileIndex = uIndex; };
	void setConnectedFd(int fd)
		{ m_sock = fd; };
	void setFilePath(const char * path)
		{ m_szFilePath = path; };
	void setServentId(unsigned char * servId);
};


class KviGnutellaTransferItem : public QListViewItem
{
	friend class KviGnutellaTransferTab;
protected:
	KviGnutellaTransferItem(QListView * par,KviGnutellaTransferTab * tab);
	~KviGnutellaTransferItem();
public:
	KviGnutellaTransferThread              * m_pThread;
	KviGnutellaTransferTab                 * m_pTab;
	bool                                     m_bCompleted;
	bool                                     m_bRetryFromSameHost;
	unsigned int                             m_uRetryAfterSecs;
	unsigned int                             m_uId;
	unsigned int                             m_uSameHostRetries;
	KviGnutellaTransferThread::TransferState m_state;
	KviGnutellaTransferThread::TransferDirection m_direction;
//	unsigned char                            m_serventId[16];
	unsigned int                             m_uFileIndex;
	int                                      m_iNodeId;
	KviPtrList<KviStr>                          * m_pExcludeHostList;
	unsigned int                             m_uSearchesWithNoHit;
public:
	bool isDead(){ return m_state == KviGnutellaTransferThread::Dead; };
	unsigned int id(){ return m_uId; };
	KviGnutellaTransferThread * thread(){ return m_pThread; };
	KviPtrList<KviStr> * excludeHostList(){ return m_pExcludeHostList; };
	void addExcludeHost(const char * host);
	void addExcludeHosts(KviPtrList<KviStr> * list);
	void clearExcludeHostList();
};



typedef struct _KviGnutellaTransferPushRequest
{
	unsigned int       uTransferId;
	unsigned short int uPort;
//	unsigned int       uAddress;
	unsigned char      serventId[16];
	unsigned int       uFileIndex;
	int                iNodeId;
} KviGnutellaTransferPushRequest;


typedef struct _KviGnutellaTransferStateChange
{
	unsigned int                             uTransferId;
	KviGnutellaTransferThread::TransferState newState;
	KviStr                                   szReason;
} KviGnutellaTransferStateChange;

typedef struct _KviGnutellaTransferProgress
{
	unsigned int uTransferId;
	KviStr       szProgress;
} KviGnutellaTransferProgress;


typedef struct _KviGnutellaTransferFileInfo
{
	KviStr       szFileName;
	unsigned int uFileSize;
	unsigned int uTransferId;
} KviGnutellaTransferFileInfo;

typedef struct _KviGnutellaTransferCompletedInfo
{
	unsigned int uTransferId;
} KviGnutellaTransferCompletedInfo;

#define KVI_GNUTELLA_TRANSFER_THREAD_EVENT_STATE_CHANGE (KVI_THREAD_USER_EVENT_BASE + 1400)
#define KVI_GNUTELLA_TRANSFER_THREAD_EVENT_PUSH_REQUEST (KVI_THREAD_USER_EVENT_BASE + 1401)
#define KVI_GNUTELLA_TRANSFER_THREAD_EVENT_PROGRESS     (KVI_THREAD_USER_EVENT_BASE + 1402)
#define KVI_GNUTELLA_TRANSFER_THREAD_EVENT_FILE_INFO    (KVI_THREAD_USER_EVENT_BASE + 1403)
#define KVI_GNUTELLA_TRANSFER_THREAD_EVENT_COMPLETED    (KVI_THREAD_USER_EVENT_BASE + 1404)
#define KVI_GNUTELLA_TRANSFER_THREAD_EVENT_RETRY        (KVI_THREAD_USER_EVENT_BASE + 1405)

#endif //_GNUTELLATRANSFER_H_
