//
//   File : gnutellatransfer.cpp
//   Creation date : Tue Apr 24 2001 15:17:45 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the gnutellas of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#warning "IDEA: Smart download: autosearch and autoresume"

#include "gnutellatransfer.h"
#include "gnutellawindow.h"
#include "gnutellathread.h"
#include "gnutellashares.h"
#include "gnutellaoptions.h"
#include "gnutelladcache.h"
#include "gnutellasearch.h"

#include "kvi_locale.h"
#include "kvi_error.h"
#include "kvi_netutils.h"
#include "kvi_out.h"
#include "kvi_memmove.h"
#include "kvi_fileutils.h"
#include "kvi_malloc.h"
#include "kvi_iconmanager.h"
#include "kvi_config.h"

#include "kvi_fileutils.h"
#include "kvi_time.h"
#include "kvi_socket.h"

#include <sys/stat.h>
//#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
//#include <sys/types.h>
//#include <sys/socket.h>
//#include <fcntl.h>


//#define _GNUTELLADEBUG

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"


extern KviConfig                      * g_pGnutellaDownloadFilesConfig;
extern KviMutex                       * g_pGnutellaDownloadFilesMutex;

extern unsigned int                     g_uGnutellaCurrentUploadTransfers;
extern unsigned int                     g_uGnutellaCurrentDownloadTransfers;
extern KviMutex                       * g_pGnutellaTransferCountersMutex;


////////////////////////////////////////////////////////////////////////////////////////////
// TRANSFER LISTVIEW ITEM
//

static unsigned int g_uNextTransferId = 1;



KviGnutellaTransferItem::KviGnutellaTransferItem(QListView * par,KviGnutellaTransferTab * tab)
: QListViewItem(par)
{
	m_uId = g_uNextTransferId;
	g_uNextTransferId++;	
	m_pTab = tab;
	m_state = KviGnutellaTransferThread::Idle;
	m_pThread = new KviGnutellaTransferThread(tab,m_uId);
	m_bCompleted = false;
	m_uSameHostRetries = 0;
	m_pExcludeHostList = 0;
	m_uSearchesWithNoHit = 0;
}

KviGnutellaTransferItem::~KviGnutellaTransferItem()
{
	clearExcludeHostList();
	if(m_pExcludeHostList)delete m_pExcludeHostList;
	delete m_pThread;
	m_pThread = 0;
	m_uId = 0;
}

void KviGnutellaTransferItem::clearExcludeHostList()
{
	if(m_pExcludeHostList)delete m_pExcludeHostList;
	m_pExcludeHostList = 0;
}

void KviGnutellaTransferItem::addExcludeHost(const char * host)
{
	if(!m_pExcludeHostList)
	{
		m_pExcludeHostList = new KviPtrList<KviStr>;
		m_pExcludeHostList->setAutoDelete(true);
	}
	for(KviStr * s = m_pExcludeHostList->first();s;s = m_pExcludeHostList->next())
	{
		if(kvi_strEqualCS(s->ptr(),host))return;
	}
	m_pExcludeHostList->append(new KviStr(host));
}

void KviGnutellaTransferItem::addExcludeHosts(KviPtrList<KviStr> * list)
{
	if(!list)return;
	for(KviStr * s = list->first();s;s = list->next())addExcludeHost(s->ptr());
}

////////////////////////////////////////////////////////////////////////////////////////////
// TRANSFER TAB
//

KviGnutellaTransferTab::KviGnutellaTransferTab(QWidget * par,KviGnutellaWindow * wnd)
: QVBox(par)
{
	m_pGnutellaWindow = wnd;

	m_pRetryTimer = 0;

	setSpacing(2);

	m_uLastClickedItemId = 0;

	m_pContextPopup = new QPopupMenu(this);

	m_pView = new QListView(this);
	m_pView->setSelectionMode(QListView::Extended);
	m_pView->setAllColumnsShowFocus(true);
	m_pView->addColumn(__tr("Index"));      // transfer index
	m_pView->addColumn(__tr("File"),200);       // file name (no path)
	m_pView->addColumn(__c2q(__tr("Size")));       // file size
	m_pView->addColumn(__tr("Host"));       // remote host name
	m_pView->addColumn(__tr("Port"));       // remote port
	m_pView->addColumn(__tr("Type"));       // active | passive | unknown
	m_pView->addColumn(__tr("Direction"));  // send | recv
	m_pView->addColumn(__tr("Status"));
	m_pView->addColumn(__tr("Progress"));
	m_pView->setColumnWidthMode(1,QListView::Manual);
	connect(m_pView,SIGNAL(selectionChanged()),this,SLOT(selectionChanged()));
	connect(m_pView,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),this,SLOT(rightButtonPressed(QListViewItem *,const QPoint &,int)));

	QHBox * hbox = new QHBox(this);
	hbox->setSpacing(2);

	m_pRemoveSelectedButton = new QPushButton(__c2q(__tr("Remove selected transfers")),hbox);
	connect(m_pRemoveSelectedButton,SIGNAL(clicked()),this,SLOT(removeSelectedTransfers()));
	m_pRemoveSelectedButton->setEnabled(false);

	m_pClearDeadButton = new QPushButton(__c2q(__tr("Clear dead transfers")),hbox);
	connect(m_pClearDeadButton,SIGNAL(clicked()),this,SLOT(clearDeadTransfers()));
	m_pClearDeadButton->setEnabled(false);

}

KviGnutellaTransferTab::~KviGnutellaTransferTab()
{
	stopRetryTimer();
	m_pView->clear();
	KviThreadManager::killPendingEvents(this);
}

void KviGnutellaTransferTab::pushFailure(KviGnutellaPushFailureInfo * inf)
{
	// We have failed to send the push request
	// Artificially kill the node
	KviGnutellaTransferItem * it = findTransferItem(inf->uTransferId);
	if(it)
	{
		if(it->m_state == KviGnutellaTransferThread::WaitingForPush)
		{
			it->thread()->enqueueEvent(new KviThreadEvent(KVI_GNUTELLA_THREAD_EVENT_PUSH_FAILURE)); // this is equivalent to calling terminate()
		}
	}
}


void KviGnutellaTransferTab::rightButtonPressed(QListViewItem *it,const QPoint &pnt,int)
{
	if(it)
	{
		m_pContextPopup->clear();

		m_uLastClickedItemId = ((KviGnutellaTransferItem *)it)->m_uId;


		if(((KviGnutellaTransferItem *)it)->m_state != KviGnutellaTransferThread::Dead)
		{
			m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_KICK)),__tr("Kill"),this,SLOT(killCurrentTransfer()));
		} else {
			m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_DISCARD)),__tr("Remove"),this,SLOT(removeCurrentTransfer()));

			if((! (((KviGnutellaTransferItem *)it)->m_bCompleted)) && 
				(((KviGnutellaTransferItem *)it)->m_direction == KviGnutellaTransferThread::Recv))
			{
				m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_RETRY)),__tr("Retry (same host)"),this,SLOT(retryCurrentTransfer()));
			}
		}

		m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_SEARCH)),__tr("Search for this filename"),this,SLOT(searchCurrentTransfer()));

		m_pContextPopup->popup(pnt);
	}
}

KviGnutellaTransferItem * KviGnutellaTransferTab::retryTransfer(unsigned int uTransferId)
{
	KviGnutellaTransferItem * it = findTransferItem(uTransferId);
	if(it)
	{
		if((it->m_state == KviGnutellaTransferThread::Dead) || (it->m_state == KviGnutellaTransferThread::WaitingForRetry))
		{
			// the thread is dead now...we can access its info safely
			KviGnutellaQueryHitInfo inf;
			inf.szName  = it->thread()->fileName();
			inf.szIp    = it->text(3);
			//inf.szIp    = it->thread()->remoteHostIp();
			inf.iNodeId = it->m_iNodeId;
			inf.uSize   = it->thread()->fileSize();
			inf.uIndex  = it->m_uFileIndex;
			inf.uSpeed  = 0;
			KviStr tmp  = it->text(4);
//			inf.uPort   = it->thread()->remoteHostPort();
			inf.uPort   = tmp.toUShort();
			kvi_memmove(inf.servId,it->thread()->serventId(),16);
			return addOutgoingTransfer(&inf);
		}

//		delete it;
	}
	return 0;
}

void KviGnutellaTransferTab::retryCurrentTransfer()
{
	retryTransfer(m_uLastClickedItemId);
}

void KviGnutellaTransferTab::searchForTransferData(unsigned int uTransferId)
{
	KviGnutellaTransferItem * it = findTransferItem(uTransferId);
	if(it)
	{
		KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(KVI_GNUTELLA_WINDOW_EVENT_DO_SEARCH);
		e->m_szData = it->text(1);
		e->m_uPort = 0; // min speed!
		m_pGnutellaWindow->mainGnutellaThread()->enqueueEvent(e);
	}
}

void KviGnutellaTransferTab::searchCurrentTransfer()
{
	searchForTransferData(m_uLastClickedItemId);
}

void KviGnutellaTransferTab::removeCurrentTransfer()
{
	KviGnutellaTransferItem * it = findTransferItem(m_uLastClickedItemId);
	if(it)delete it;
}

void KviGnutellaTransferTab::killCurrentTransfer()
{
	KviGnutellaTransferItem * it = findTransferItem(m_uLastClickedItemId);
	if(it)
	{
		if(it->m_state != KviGnutellaTransferThread::Dead)
		{
			if(it->m_state == KviGnutellaTransferThread::WaitingForRetry)
			{
				it->m_state = KviGnutellaTransferThread::Dead;
				it->setText(7,__tr("Dead (Retry killed)"));
				m_pClearDeadButton->setEnabled(true);
				it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_BLACKSQUARE)));
			} else {
				it->thread()->terminate();
				enableClearDeadTransfers();
			}
		}
	}
}


void KviGnutellaTransferTab::selectionChanged()
{
	QListViewItem * it = m_pView->firstChild();
	while(it)
	{
		if(it->isSelected())
		{
			m_pRemoveSelectedButton->setEnabled(true);
			return;
		}
		it = it->nextSibling();
	}
	m_pRemoveSelectedButton->setEnabled(false);
}

void KviGnutellaTransferTab::removeSelectedTransfers()
{
	KviPtrList<QListViewItem> l;
	l.setAutoDelete(true);

	QListViewItem * it = m_pView->firstChild();
	while(it)
	{
		if(it->isSelected())l.append(it);
		it = it->nextSibling();
	}

	while(l.first())l.removeFirst();

	enableClearDeadTransfers();
}

void KviGnutellaTransferTab::clearDeadTransfers()
{
	KviPtrList<QListViewItem> l;
	l.setAutoDelete(true);

	KviGnutellaTransferItem * it = (KviGnutellaTransferItem *)m_pView->firstChild();
	while(it)
	{
		if(it->isDead())l.append(it);
		it = (KviGnutellaTransferItem *)it->nextSibling();
	}
	m_pClearDeadButton->setEnabled(false);

}

void KviGnutellaTransferTab::enableClearDeadTransfers()
{
	KviGnutellaTransferItem * it = (KviGnutellaTransferItem *)m_pView->firstChild();
	while(it)
	{
		if(it->isDead())
		{
			m_pClearDeadButton->setEnabled(true);
			return;
		}
		it = (KviGnutellaTransferItem *)it->nextSibling();
	}
	m_pClearDeadButton->setEnabled(false);
}

void KviGnutellaTransferTab::handleRetryHint(KviGnutellaTransferRetryInfo * inf)
{
	g_pGnutellaOptionsMutex->lock();
	bool bCanRetry = g_pGnutellaOptions->m_bAutoRetryFailedTransfers;
	g_pGnutellaOptionsMutex->unlock();

	if(bCanRetry)
	{
		KviGnutellaTransferItem * it = findTransferItem(inf->uTransferId);
		if(it)
		{
			it->m_state = KviGnutellaTransferThread::WaitingForRetry;
			it->m_bRetryFromSameHost = inf->bRetryFromSameHost;
			it->m_uRetryAfterSecs    = inf->uRetryAfterSecs;
			KviStr tmp(KviStr::Format,__tr("Waiting for retry (%u secs)"),it->m_uRetryAfterSecs);
			it->setText(7,tmp.ptr());
			it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_IDEA)));
			startRetryTimer();
		}
	}
}

void KviGnutellaTransferTab::addIncomingTransfer(KviGnutellaIncomingTransferInfo * inf)
{
	KviGnutellaTransferItem * it = new KviGnutellaTransferItem(m_pView,this);

	KviStr tmp(KviStr::Format,"%u",it->id());
	it->setText(0,tmp.ptr());


	it->setText(1,"???");
	it->setText(2,"???");

	it->setText(3,inf->szIp.ptr());

	tmp.setNum(inf->uPort);
	it->setText(4,tmp.ptr());

	it->setText(5,__tr("Passive"));

	it->setText(6,__tr("Send"));

	it->setText(7,__tr("Booting"));

	it->m_direction = KviGnutellaTransferThread::Send;

	it->thread()->setRemoteHost(inf->szIp.ptr(),inf->uPort);
	it->thread()->setConnectionType(KviGnutellaTransferThread::Passive);
	it->thread()->setTransferDirection(KviGnutellaTransferThread::Send);
	it->thread()->setConnectedFd(inf->iFd);
	it->thread()->start();

}

void KviGnutellaTransferTab::addPushTransfer(KviGnutellaPushRequestInfo * inf)
{
	KviGnutellaTransferItem * it = new KviGnutellaTransferItem(m_pView,this);

	KviStr tmp(KviStr::Format,"%u",it->id());
	it->setText(0,tmp.ptr());

	it->setText(1,inf->szFileName.ptr());

	tmp.setNum(inf->uFileSize);
	it->setText(2,tmp.ptr());

	it->setText(3,inf->szIp.ptr());

	tmp.setNum(inf->uPort);
	it->setText(4,tmp.ptr());

	it->setText(5,__tr("Active"));

	it->setText(6,__tr("Send"));

	it->setText(7,__tr("Booting"));

	it->m_uFileIndex = inf->uFileIndex;
	it->m_iNodeId = 0;

	it->m_direction = KviGnutellaTransferThread::Send;

	it->thread()->setRemoteHost(inf->szIp.ptr(),inf->uPort);
	it->thread()->setConnectionType(KviGnutellaTransferThread::Active);
	it->thread()->setTransferDirection(KviGnutellaTransferThread::Send);
	it->thread()->setFileInfo(inf->szFileName.ptr(),inf->uFileSize,inf->uFileIndex);
	it->thread()->setFilePath(inf->szFilePath.ptr());
	it->thread()->setServentId(inf->localServentId);
	it->thread()->start();
}

KviGnutellaTransferItem * KviGnutellaTransferTab::addOutgoingTransfer(KviGnutellaQueryHitInfo * inf)
{
	KviGnutellaTransferItem * it = new KviGnutellaTransferItem(m_pView,this);

	KviStr tmp(KviStr::Format,"%u",it->id());
	it->setText(0,tmp.ptr());

	it->setText(1,inf->szName.ptr());

	tmp.setNum(inf->uSize);
	it->setText(2,tmp.ptr());

	it->setText(3,inf->szIp.ptr());

	tmp.setNum(inf->uPort);
	it->setText(4,tmp.ptr());

	it->setText(5,__tr("Active"));

	it->setText(6,__tr("Recv"));

	it->setText(7,__tr("Booting"));

	it->m_uFileIndex = inf->uIndex;
	it->m_iNodeId = inf->iNodeId;

	it->m_direction = KviGnutellaTransferThread::Recv;


	it->thread()->setRemoteHost(inf->szIp.ptr(),inf->uPort);
	it->addExcludeHost(inf->szIp.ptr()); // we add it here
	it->thread()->setConnectionType(KviGnutellaTransferThread::Active);
	it->thread()->setTransferDirection(KviGnutellaTransferThread::Recv);
	it->thread()->setFileInfo(inf->szName.ptr(),inf->uSize,inf->uIndex);
//#warning "We might lookup the save path for the file type"
	g_pGnutellaOptionsMutex->lock();
	it->thread()->setFilePath(g_pGnutellaOptions->m_szIncompleteDirectory.ptr());
	g_pGnutellaOptionsMutex->unlock();
	it->thread()->setServentId(inf->servId);
	it->thread()->start();

	return it;
}

void KviGnutellaTransferTab::startRetryTimer()
{
	if(m_pRetryTimer)return;
	m_pRetryTimer = new QTimer(this);
	connect(m_pRetryTimer,SIGNAL(timeout()),this,SLOT(retryTimerTimeout()));
	m_pRetryTimer->start(1000);
}

void KviGnutellaTransferTab::stopRetryTimer()
{
	if(!m_pRetryTimer)return;
	m_pRetryTimer->stop();
	delete m_pRetryTimer;
	m_pRetryTimer = 0;
}

void KviGnutellaTransferTab::retryTimerTimeout()
{
	KviGnutellaTransferItem * it = (KviGnutellaTransferItem *)m_pView->firstChild();
	bool bGotIt = false;
	while(it)
	{
		if(it->m_state == KviGnutellaTransferThread::WaitingForRetry)
		{
			if(it->m_uRetryAfterSecs > 0)it->m_uRetryAfterSecs--;
			if(it->m_uRetryAfterSecs == 0)
			{
				g_pGnutellaOptionsMutex->lock();
				unsigned int uMaxSameHostRetries = g_pGnutellaOptions->m_uMaxSameHostRetries;
				g_pGnutellaOptionsMutex->unlock();

				if(it->m_bRetryFromSameHost && (it->m_uSameHostRetries < uMaxSameHostRetries))
				{
					it->m_state = KviGnutellaTransferThread::Dead;
					it->setText(7,__tr("Dead (Failed+Retried)"));
					m_pClearDeadButton->setEnabled(true);
					it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_BLACKSQUARE)));

					KviStr tmp = it->text(1);
					it->m_uSameHostRetries++;
					m_pGnutellaWindow->output(KVI_OUT_SYSTEMMESSAGE,__tr("[smart download]: Attempting to retry transfer for file \"%s\": same host , attempt %u"),tmp.ptr(),it->m_uSameHostRetries);
					KviGnutellaTransferItem * itm = retryTransfer(it->m_uId);
					if(itm)
					{
						itm->m_uSameHostRetries = it->m_uSameHostRetries;
						itm->addExcludeHosts(it->excludeHostList());
					}
				} else {
					KviStr fName = it->text(1);
					KviStr szHost = it->text(3);
					KviStr szSize = it->text(2);
					unsigned int uSize = szSize.toUInt();
					m_pGnutellaWindow->output(KVI_OUT_SYSTEMMESSAGE,__tr("[smart download]: Attempting to retry transfer for file \"%s\": changing host"),fName.ptr());
//					it->addExcludeHost(szHost.ptr()); // re-add the exclude host: the 
					KviGnutellaQueryHitInfo * inf = m_pGnutellaWindow->findBestQueryHit(fName.ptr(),uSize,it->excludeHostList());
					if(inf)
					{
						it->m_state = KviGnutellaTransferThread::Dead;
						it->setText(7,__tr("Dead (Failed+Retried)"));
						m_pClearDeadButton->setEnabled(true);
						it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_BLACKSQUARE)));
						KviGnutellaTransferItem * itm = addOutgoingTransfer(inf);
						itm->addExcludeHosts(it->excludeHostList());
					} else {
						m_pGnutellaWindow->output(KVI_OUT_SYSTEMMESSAGE,__tr("[smart download]: No alternate hit found for file \"%s\": performing a query; will retry again in 30 secs"),fName.ptr());
						it->m_uSearchesWithNoHit++;
						if(it->m_uSearchesWithNoHit > 5)
						{
							// we had 5 consecutive searches with no hit here...
							// this is either because the network is down (so we can do nothing about it),
							// because there are NO hits at all (and again we can do nothing),
							// or because the hits come all from excluded hosts
							// the only thing that we can do is to clear the excluded host list
							// and try to cycle again thru all the hosts
							// IDEA: a nice idea would be to increase the search ttl by one , or
							// walking the network (dropping a couple of connections and gaining new ones)
//#warning "Walk the network"
							it->clearExcludeHostList();
						}
						bGotIt = true;
						searchForTransferData(it->m_uId);
						it->m_uRetryAfterSecs = 30;
						KviStr tmp(KviStr::Format,__tr("Waiting for retry (%u secs)"),it->m_uRetryAfterSecs);
						it->setText(7,tmp.ptr());
					}
				}
			} else {
				bGotIt = true;
				KviStr tmp(KviStr::Format,__tr("Waiting for retry (%u secs)"),it->m_uRetryAfterSecs);
				it->setText(7,tmp.ptr());
			}
		}
		it = (KviGnutellaTransferItem *)it->nextSibling();
	}
	if(!bGotIt)stopRetryTimer();
}

KviGnutellaTransferItem * KviGnutellaTransferTab::findTransferItem(unsigned int uId)
{
	KviGnutellaTransferItem * it = (KviGnutellaTransferItem *)m_pView->firstChild();
	while(it)
	{
		if(uId == it->id())return it;
		it = (KviGnutellaTransferItem *)it->nextSibling();
	}
	return 0;
}

bool KviGnutellaTransferTab::event(QEvent *e)
{
	if(e->type() == (QEvent::Type)KVI_THREAD_EVENT)
	{
		switch(((KviThreadEvent *)e)->id())
		{
			case KVI_THREAD_EVENT_WARNING:
			{
				KviStr * s = ((KviThreadDataEvent<KviStr> *)e)->getData();
				m_pGnutellaWindow->output(KVI_OUT_SYSTEMWARNING,__tr_no_xgettext(s->ptr()));
				delete s;
			}
			break;
			case KVI_THREAD_EVENT_ERROR:
			{
				KviStr * s = ((KviThreadDataEvent<KviStr> *)e)->getData();
				m_pGnutellaWindow->output(KVI_OUT_SYSTEMERROR,__tr_no_xgettext(s->ptr()));
				delete s;
			}
			break;
			case KVI_THREAD_EVENT_MESSAGE:
			{
				KviStr * s = ((KviThreadDataEvent<KviStr> *)e)->getData();
				m_pGnutellaWindow->output(KVI_OUT_SYSTEMMESSAGE,__tr_no_xgettext(s->ptr()));
				delete s;
			}
			break;
			case KVI_GNUTELLA_TRANSFER_THREAD_EVENT_COMPLETED:
			{
				KviGnutellaTransferCompletedInfo * inf = ((KviThreadDataEvent<KviGnutellaTransferCompletedInfo> *)e)->getData();
				KviGnutellaTransferItem * it = findTransferItem(inf->uTransferId);
				it->m_bCompleted = true;
				delete inf;
			}
			break;
			case KVI_GNUTELLA_TRANSFER_THREAD_EVENT_STATE_CHANGE:
			{
				KviGnutellaTransferStateChange * st = ((KviThreadDataEvent<KviGnutellaTransferStateChange> *)e)->getData();
				KviGnutellaTransferItem * it = findTransferItem(st->uTransferId);
				if(it)
				{
					it->m_state = st->newState;
					switch(st->newState)
					{
						case KviGnutellaTransferThread::Connecting:
							it->setText(7,__tr("Connecting"));
							it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_REDSQUARE)));
						break;
						case KviGnutellaTransferThread::WaitingForPush:
							it->setText(7,__tr("Waiting for push"));
							it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_BLUESQUARE)));
						break;
						case KviGnutellaTransferThread::Handshaking:
							it->setText(7,__tr("Handshaking"));
							it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_YELLOWSQUARE)));
						break;
						case KviGnutellaTransferThread::Transferring:
							it->setText(7,__tr("Transferring"));
							it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_GREENSQUARE)));
						break;
						case KviGnutellaTransferThread::Dead:
							if(it->m_bCompleted)
							{
								it->setText(7,__tr("Dead (Completed)"));
								m_pClearDeadButton->setEnabled(true);
								it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_PACKAGE))); //smile :)
							} else {
								it->setText(7,__tr("Dead (Failed)"));
								m_pClearDeadButton->setEnabled(true);
								it->setPixmap(0,*(g_pIconManager->getSmallIcon(KVI_SMALLICON_BLACKSQUARE)));
							}
						break;
					}
					if(st->szReason.hasData())it->setText(8,__tr_no_xgettext(st->szReason.ptr()));
				}
				delete st;
			}
			break;
			case KVI_GNUTELLA_TRANSFER_THREAD_EVENT_RETRY:
			{
				KviGnutellaTransferRetryInfo * inf = ((KviThreadDataEvent<KviGnutellaTransferRetryInfo> *)e)->getData();
				handleRetryHint(inf);
				delete inf;
			}
			break;
			case KVI_GNUTELLA_TRANSFER_THREAD_EVENT_FILE_INFO:
			{
				KviGnutellaTransferFileInfo * inf = ((KviThreadDataEvent<KviGnutellaTransferFileInfo> *)e)->getData();
				KviGnutellaTransferItem * it = findTransferItem(inf->uTransferId);
				if(it)
				{
					it->setText(1,inf->szFileName.ptr());
					KviStr tmp(KviStr::Format,"%u",inf->uFileSize);
					it->setText(2,tmp.ptr());
				}
				delete inf;
			}
			break;
			case KVI_GNUTELLA_TRANSFER_THREAD_EVENT_PUSH_REQUEST:
			{
				KviGnutellaTransferPushRequest * trr = ((KviThreadDataEvent<KviGnutellaTransferPushRequest> *)e)->getData();

				KviGnutellaTransferItem * it = findTransferItem(trr->uTransferId);
				if(it && m_pGnutellaWindow->mainGnutellaThread())
				{
					// Copy the struct... yes... this could be avoided... but we want to avoid too much mess
//					KviGnutellaTransferPushRequest * trr2 = new KviGnutellaTransferPushRequest;
//					trr2->uTransferId = trr->uTransferId;
//					trr2->uPort       = trr->uPort;
//					kvi_memmove(trr2->serventId,trr->serventId,16);
					// We're going to route this event and data structure to the main gnutella thread
					trr->uFileIndex = it->m_uFileIndex;
					trr->iNodeId  = it->m_iNodeId;
					trr->uTransferId = it->m_uId;
					KviThreadDataEvent<KviGnutellaTransferPushRequest> * ev = new KviThreadDataEvent<KviGnutellaTransferPushRequest>(KVI_GNUTELLA_TRANSFER_THREAD_EVENT_PUSH_REQUEST);
					ev->setData(trr);
					m_pGnutellaWindow->mainGnutellaThread()->enqueueEvent(ev);
				} else delete trr;
			}
			break;
			case KVI_GNUTELLA_TRANSFER_THREAD_EVENT_PROGRESS:
			{
				KviGnutellaTransferProgress * pr = ((KviThreadDataEvent<KviGnutellaTransferProgress> *)e)->getData();
				KviGnutellaTransferItem * it = findTransferItem(pr->uTransferId);
				if(it)
				{
					it->setText(8,__tr_no_xgettext(pr->szProgress.ptr()));
				}
				delete pr;
			}
			break;
		}
		return true;
	}
	return QWidget::event(e);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// TRANSFER THREAD
//

KviGnutellaTransferThread::KviGnutellaTransferThread(KviGnutellaTransferTab * tab,unsigned int uId)
: KviSensitiveThread()
{
	m_uId                                       = uId;                   // this transfer ID
	m_pTab                                      = tab;
	m_sock                                      = KVI_INVALID_SOCKET;
	m_state                                     = Idle;
	m_pInBuffer                                 = 0;
	m_uInBufferDataLen                          = 0;
	m_uInBufferRealLen                          = 0;
	m_uTransferEndPosition                      = 0;
	m_uResumePosition                           = 0;
	m_pHttpHeader                               = new KviPtrList<KviStr>;
	m_pHttpHeader->setAutoDelete(true);
	m_bTerminateRequestSeen                     = false;
	m_bDecrementCurrentUploadTransfersOnDeath   = false;
	m_bDecrementCurrentDownloadTransfersOnDeath = false;
	m_retryHint                                 = NoRetry;
	m_uRetryDelayInSecs                         = 0;
}

KviGnutellaTransferThread::~KviGnutellaTransferThread()
{
	terminate();
	closeSock();
	if(m_pInBuffer)
	{
		__range_valid(m_uInBufferRealLen);
		kvi_free(m_pInBuffer);
		m_uInBufferDataLen = 0;
		m_uInBufferRealLen = 0;
		m_pInBuffer = 0;
	}
	delete m_pHttpHeader;
}

void KviGnutellaTransferThread::setRetryHint(bool bSameHost,unsigned int uRetryDelay)
{
	m_retryHint = bSameHost ? RetrySameHost : RetryAnotherHost;
	m_uRetryDelayInSecs = uRetryDelay;
}


void KviGnutellaTransferThread::setServentId(unsigned char * servId)
{
	kvi_memmove(m_serventId,servId,16);
}

int KviGnutellaTransferThread::selectForReadStep()
{
	// calls select on the main socket
	// returns 1 if there is data available for reading
	// returns 0 if there is no data available but there was no error
	// returns -1 if there was a critical error (socket closed)
	fd_set readSet;

	FD_ZERO(&readSet);

	FD_SET(m_sock,&readSet);

	struct timeval tmv;
	tmv.tv_sec  = 0;
	tmv.tv_usec = 1000; // we wait 1000 usecs for an event


	int nRet = kvi_socket_select(m_sock + 1,&readSet,0,0,&tmv);

	if(nRet > 0)
	{
		if(FD_ISSET(m_sock,&readSet))
		{
			// ok
			return 1;
		}
	} else {
		if(nRet < 0)
		{
			int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
			if((err != EAGAIN) && (err != EINTR) && (err != WSAEWOULDBLOCK))
#else
			if((err != EAGAIN) && (err != EINTR))
#endif
			{
				KviStr tmp(KviStr::Format,__tr_no_lookup("Select error: %s (errno=%d)"),
					kvi_getErrorString(kvi_errorFromSystemError(err)),err);
				closeSock(tmp.ptr());
				return -1;
			}
		}
	}

	return 0;
}


int KviGnutellaTransferThread::selectForWriteStep()
{
	// calls select on the main socket
	// returns 1 if there is space avalable for writing
	// returns 0 if there is no space available but there was no error
	// returns -1 if there was a critical error (socket closed)
	fd_set writeSet;

	FD_ZERO(&writeSet);

	FD_SET(m_sock,&writeSet);

	struct timeval tmv;
	tmv.tv_sec  = 0;
	tmv.tv_usec = 1000; // we wait 1000 usecs for an event


	int nRet = kvi_socket_select(m_sock + 1,0,&writeSet,0,&tmv);

	if(nRet > 0)
	{
		if(FD_ISSET(m_sock,&writeSet))
		{
			// ok
			return 1;
		}
	} else {
		if(nRet < 0)
		{
			int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
			if((err != EAGAIN) && (err != EINTR) && (err != WSAEWOULDBLOCK))
#else
			if((err != EAGAIN) && (err != EINTR))
#endif
			{
				KviStr tmp(KviStr::Format,__tr_no_lookup("Select error: %s (errno=%d)"),
					kvi_getErrorString(kvi_errorFromSystemError(err)),err);
				closeSock(tmp.ptr());
				return -1;
			}
		}
	}

	return 0;
}


bool KviGnutellaTransferThread::selectForRead(int iTimeoutInSecs)
{
	// waits for some data to arrive on the socket
	// up to iTimeoutInSecs seconds
	// returns true if data is available on the socket
	// or false if there was a select() error or no data
	// was available in the specified amount of time

	time_t startTime = time(0);

	for(;;)
	{
		if(!processInternalEvents())
		{
#ifdef _GNUTELLADEBUG
			debug("[transfer %u] Received terminate event in select for read",m_uId);
#endif
			return closeSock(); // ensure that the socket is closed
		}

		int nRet = selectForReadStep();

		if(nRet < 0)return false;
		if(nRet > 0)return true;

		int diff = time(0) - startTime;
		if(diff > iTimeoutInSecs)
			return closeSock(__tr_no_lookup("Operation timed out (while selecting for read)"));

		usleep(100000); // 1/10 sec
	}

	return false;
}

bool KviGnutellaTransferThread::readData()
{
	// Reads some data from the socket and places it in m_pInBuffer
	// Returns false in case of error (critical=socket closed!)
	unsigned int expectedLen = 1024 + m_uInBufferDataLen;
	if(m_uInBufferRealLen < expectedLen)
	{
		m_pInBuffer = (char *)kvi_realloc(m_pInBuffer,expectedLen);
		m_uInBufferRealLen = expectedLen;
	}
	int readed = kvi_socket_read(m_sock,m_pInBuffer + m_uInBufferDataLen,1024);
	if(readed > 0)
	{
		m_uInBufferDataLen += readed;
	} else {
		if(readed < 0)
		{
			// Read error ?
			int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
			if((err != EAGAIN) && (err != EINTR) && (err != WSAEWOULDBLOCK))
#else
			if((err != EAGAIN) && (err != EINTR))
#endif
			{
				// yes...read error
				KviStr tmp(KviStr::Format,__tr_no_lookup("Read error: %s (errno=%d)"),
						kvi_getErrorString(kvi_errorFromSystemError(err)),err);
				return closeSock(tmp.ptr());
			}
			return true; // EINTR or EAGAIN...transient problem
		} else {
			// readed == 0
			// Connection closed by remote host
			return closeSock(__tr_no_lookup("Connection closed by remote host"));
		}
	}
	return true;
}


bool KviGnutellaTransferThread::waitForData()
{
	// waits for some data to arrive on the socket and
	// reads it
	// returns false in case of critical error (socket closed!)
//#warning "This should be a tuneable timeout"
	if(!selectForRead(180))return false;
	return readData();
}

KviStr * KviGnutellaTransferThread::processHttpHeaderLine()
{

	for(unsigned int i=0;i<m_uInBufferDataLen;i++)
	{
		switch(m_pInBuffer[i])
		{
			case '\r':
			{
				if((i + 1) < m_uInBufferDataLen)
				{
					if(m_pInBuffer[i + 1] == '\n')
					{
						KviStr * ret = new KviStr(m_pInBuffer,i);
						int remainingLen  = m_uInBufferDataLen - (i + 2);
						if(remainingLen > 0)
						{
							kvi_memmove(m_pInBuffer,m_pInBuffer + i + 2,remainingLen);
						}
						m_pInBuffer = (char *)kvi_realloc(m_pInBuffer,remainingLen + 1024);
						m_uInBufferDataLen = remainingLen;
						m_uInBufferRealLen = remainingLen + 1024;
						return ret;
					}
				}
			}
			break;
			case '\n':
			case '\0':
			{
				KviStr * ret = new KviStr(m_pInBuffer,i);
				int remainingLen  = m_uInBufferDataLen - (i + 1);
				if(remainingLen > 0)
				{
					kvi_memmove(m_pInBuffer,m_pInBuffer + i + 1,remainingLen);
				}
				m_pInBuffer = (char *)kvi_realloc(m_pInBuffer,remainingLen + 1024);
				m_uInBufferDataLen = remainingLen;
				m_uInBufferRealLen = remainingLen + 1024;
				return ret;
			}
			break;
		}
	}
	return 0;
}

bool KviGnutellaTransferThread::readHttpHeader()
{
	// waits for the complete http header to arrive
	// and reads it, returns true when the complete header
	// was received , or false in case of critical error (socket closed!)

	m_pHttpHeader->clear();

	for(;;)
	{
		if(m_uInBufferDataLen > 0)
		{
			while(KviStr * pLine = processHttpHeaderLine())
			{
				if(pLine->hasData())
				{
					m_pHttpHeader->append(pLine);
#ifdef _GNUTELLADEBUG
					debug("Processed header line %s",pLine->ptr());
#endif
				} else {
					// empty header line : eoh
					delete pLine;
					return true;
				}
			}
		}

		if(m_uInBufferDataLen > 2048)
		{
			return closeSock(__tr_no_lookup("HTTP header too long (broken client ?)"));
		}

		if(!processInternalEvents())return closeSock(); // terminated by user
		if(!waitForData())return false;                 // some error (maybe a timeout)

#ifdef _GNUTELLADEBUG
		debug("[Transfer %u]: readed data (datalen now is %u)",m_uId,m_uInBufferDataLen);
#endif

		// ok...we have some data available
		// we're going to parse it now
		// we stop at the first empty line that we read (terminated by CRLF or NULL)

		usleep(100000); // 1/10 sec
	}
	return true;
}

void KviGnutellaTransferThread::postErrorEvent(const char * msg)
{
	postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_ERROR,
		new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: %s"),m_uId,msg)));
}

bool KviGnutellaTransferThread::closeSock(const char * error,const char * reason)
{
	if(m_sock != KVI_INVALID_SOCKET)kvi_socket_close(m_sock);
	m_sock = KVI_INVALID_SOCKET;

	if(reason)m_szStateChangeReason = reason;

	if(error)
	{
		postErrorEvent(error);
		if(!reason)m_szStateChangeReason = error;
	}
	return false;
}

void KviGnutellaTransferThread::setState(TransferState st)
{
#ifdef _GNUTELLADEBUG
	debug("[transfer %u]: set state",m_uId);
#endif
	KviThreadDataEvent<KviGnutellaTransferStateChange> * e = new KviThreadDataEvent<KviGnutellaTransferStateChange>(KVI_GNUTELLA_TRANSFER_THREAD_EVENT_STATE_CHANGE);

	KviGnutellaTransferStateChange * stc = new KviGnutellaTransferStateChange;
	stc->uTransferId = m_uId;
	stc->newState = st;
	stc->szReason = m_szStateChangeReason;
	if(st == Dead)m_szStateChangeReason = "";
	e->setData(stc);
	postEvent(m_pTab,e);
	m_state = st;
	if(m_state == Dead)
	{
		if(m_bDecrementCurrentUploadTransfersOnDeath)
		{
			m_bDecrementCurrentUploadTransfersOnDeath = false;
			g_pGnutellaTransferCountersMutex->lock();
			g_uGnutellaCurrentUploadTransfers--;
			g_pGnutellaTransferCountersMutex->unlock();
		}
		if(m_bDecrementCurrentDownloadTransfersOnDeath)
		{
			m_bDecrementCurrentDownloadTransfersOnDeath = false;
			g_pGnutellaTransferCountersMutex->lock();
			g_uGnutellaCurrentDownloadTransfers--;
			g_pGnutellaTransferCountersMutex->unlock();
		}
	}
}

bool KviGnutellaTransferThread::processInternalEvents()
{
	while(KviThreadEvent *e = dequeueEvent())
	{
		switch(e->id())
		{
			case KVI_THREAD_EVENT_TERMINATE:
			{
#ifdef _GNUTELLADEBUG
				debug("[transfer %u] Received terminate event (%u)",m_uId,(unsigned int)e);
#endif
				m_bTerminateRequestSeen = true;
				delete e;
				return false;
			}
			break;
			case KVI_GNUTELLA_THREAD_EVENT_PUSH_FAILURE:
				m_bTerminateRequestSeen = true;
				setRetryHint(false,5); // we try from another host in 5 secs
				delete e;
				return false;
			break;
			default:
#ifdef _GNUTELLADEBUG
				debug("[transfer %u] Ops...received an unrecognized event",m_uId);
#endif
				delete e;
			break;
		}
	}
	return true;
}

bool KviGnutellaTransferThread::selectForWrite(int iTimeoutInSecs)
{

	time_t startTime = time(0);

	for(;;)
	{
		if(!processInternalEvents())
		{
#ifdef _GNUTELLADEBUG
			debug("[transfer %u] Received terminate event in select for write",m_uId);
#endif
			return closeSock();
		}

		fd_set writeSet;
	
		FD_ZERO(&writeSet);

		FD_SET(m_sock,&writeSet);

		struct timeval tmv;
		tmv.tv_sec  = 0;
		tmv.tv_usec = 1000; // we wait 1000 usecs for an event
	

		int nRet = kvi_socket_select(m_sock + 1,0,&writeSet,0,&tmv);
	
		if(nRet > 0)
		{
			if(FD_ISSET(m_sock,&writeSet))
			{
				// connected!
				return true;
			}
		} else {
			if(nRet < 0)
			{
				int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
				if((err != EAGAIN) && (err != EINTR) && (err != WSAEWOULDBLOCK))
#else
				if((err != EAGAIN) && (err != EINTR))
#endif
				{
					KviStr tmp(KviStr::Format,__tr_no_lookup("Select error: %s (errno=%d)"),
						kvi_getErrorString(kvi_errorFromSystemError(err)),err);
					return closeSock(tmp.ptr());
				}
			}
		}


		if((time(0) - startTime) > iTimeoutInSecs)return closeSock(__tr_no_lookup("Operation timed out"));

		usleep(100000); // 1/10 sec
	}

	return false;
}

bool KviGnutellaTransferThread::connectToRemoteHost()
{

	m_sock = kvi_socket_create(KVI_SOCKET_PF_INET,KVI_SOCKET_TYPE_STREAM,0); //tcp
	if(m_sock == KVI_INVALID_SOCKET)
		return closeSock(__tr_no_lookup("Failed to create the socket"));

	if(!kvi_socket_setNonBlocking(m_sock))
		return closeSock(__tr_no_lookup("Failed to enter non blocking mode"));

	sockaddr_in saddr;
//	struct in_addr ia;

	if(!kvi_stringIpToBinaryIp(m_szIp.ptr(),&(saddr.sin_addr)))
		return closeSock(__tr_no_lookup("Invalid target address"));

	if(!gnutella_is_routable_ip((unsigned char *)&(saddr.sin_addr.s_addr)))
		return closeSock(__tr_no_lookup("Unroutable IP address"));

	saddr.sin_port = htons(m_uPort);
	saddr.sin_family = AF_INET;

	if(!kvi_socket_connect(m_sock,(struct sockaddr *)&saddr,sizeof(saddr)))
	{
		int err = kvi_socket_error();
		if(!kvi_socket_recoverableConnectError(err))
		{
			KviStr tmp(KviStr::Format,__tr_no_lookup("Connect error: %s (errno=%d)"),
				kvi_getErrorString(kvi_errorFromSystemError(err)),err);
			return closeSock(tmp.ptr());
		}
	}

	// now loop selecting for write

//#warning "This should be a tuneable timeout"

	if(!selectForWrite(60))return false;

	int sockError;
	int iSize=sizeof(sockError);
	if(!kvi_socket_getsockopt(m_sock,SOL_SOCKET,SO_ERROR,(void *)&sockError,&iSize))sockError = -1;
	if(sockError != 0)
	{
		//failed
		if(sockError > 0)sockError = kvi_errorFromSystemError(sockError);
		else sockError = KviError_unknownError;
		KviStr tmp(KviStr::Format,__tr_no_lookup("Connect error: %s (errno=%d)"),
			kvi_getErrorString(sockError),sockError);
		return closeSock(tmp.ptr());
	}

	return true;
}

bool KviGnutellaTransferThread::pushAttempt()
{
	m_connectionType = Passive;

	m_sock = kvi_socket_create(KVI_SOCKET_PF_INET,KVI_SOCKET_TYPE_STREAM,0); // tcp
	if(m_sock == KVI_INVALID_SOCKET)
		return closeSock(__tr_no_lookup("Failed to create the listening socket"));
	
	// set the non blocking mode
	if(!kvi_socket_setNonBlocking(m_sock))
		return closeSock(__tr_no_lookup("Failed to enter non blocking mode"));

	// bind it to INADDR_ANY and port 0
	sockaddr_in saddr;
	saddr.sin_family      = AF_INET;
	saddr.sin_port        = 0;
	saddr.sin_addr.s_addr = htonl(INADDR_ANY);

//#warning "Allow forcing a specific IP address"

	if(!kvi_socket_bind(m_sock,(struct sockaddr *)&saddr,sizeof(struct sockaddr)))
		return closeSock(__tr_no_lookup("Failed to bind the socket"));

	// then call listen();
	if(!kvi_socket_listen(m_sock,100))
		return closeSock(__tr_no_lookup("Failed to start listening"));

	int slen = sizeof(saddr);

	if(!kvi_socket_getsockname(m_sock,(struct sockaddr *)&saddr,&slen))
		return closeSock(__tr_no_lookup("Failed to retrieve the local port"));


	KviThreadDataEvent<KviGnutellaTransferPushRequest> * e = new KviThreadDataEvent<KviGnutellaTransferPushRequest>(KVI_GNUTELLA_TRANSFER_THREAD_EVENT_PUSH_REQUEST);

	KviGnutellaTransferPushRequest * trr = new KviGnutellaTransferPushRequest;
	trr->uTransferId = m_uId;
	trr->uPort = ntohs(saddr.sin_port);
	kvi_memmove(trr->serventId,m_serventId,16);

	e->setData(trr);

	postEvent(m_pTab,e);


	usleep(100000); // sleep 1/10 sec

	// now wait for the incoming connection

	int iInvalidConnections = 0;

	while(selectForRead(30))
	{

		slen = sizeof(saddr);
	
		int fd = kvi_socket_accept(m_sock,(struct sockaddr *)&saddr,&slen);

#ifdef _GNUTELLADEBUG
		debug("Incoming push connection");
#endif
		if(fd != KVI_INVALID_SOCKET)
		{

//#warning "check sockerror at SOL_SOCKET level!"

			if(slen && kvi_binaryIpToStringIp(saddr.sin_addr,m_szIp))
			{
				m_uPort = ntohs(saddr.sin_port);
				kvi_socket_close(m_sock);
				m_sock = fd;
				return true;
			} else {
				// invalid IP ?
				kvi_socket_close(fd);
			}
		}

		iInvalidConnections++;
		if(iInvalidConnections > 3)
		{
			setRetryHint(false,5);
			return closeSock(__tr_no_lookup("Too many invalid connections")); // huh ?
		}
	}

	if(!m_bTerminateRequestSeen)setRetryHint(false,5);
	return false;
}

bool KviGnutellaTransferThread::sendBuffer(const char * buffer,int bufLen,int iTimeoutInSecs)
{
	const char * ptr = buffer;
	int curLen       = bufLen;

	time_t startTime = time(0);

	for(;;)
	{
		if(!processInternalEvents())return closeSock();

		int wrtn = kvi_socket_send(m_sock,ptr,curLen);
		if(wrtn > 0)
		{
			curLen -= wrtn;

			if(curLen <= 0)break;

			ptr += wrtn;
		} else {
			if(wrtn < 0)
			{
				int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
				if((err != EAGAIN) && (err != EINTR) && (err != WSAEWOULDBLOCK))
#else
				if((err != EAGAIN) && (err != EINTR))
#endif
				{
					KviStr tmp(KviStr::Format,__tr_no_lookup("Write error: %s (errno=%d)"),
							kvi_getErrorString(kvi_errorFromSystemError(err)),err);
					return closeSock(tmp.ptr());
				}
			}
		}

		int diff = time(0) - startTime;
		if(diff > iTimeoutInSecs)
			return closeSock(__tr_no_lookup("Operation timed out"));

		usleep(10000);
	}

	return true;
}

bool KviGnutellaTransferThread::sendHttpGet()
{
	KviStr buffer(KviStr::Format,
		"GET /get/%u/%s HTTP/1.0\r\n" \
		"Connection: Keep-Alive\r\n" \
		"Range: bytes=%u-\r\n" \
		"User-Agent: %s\r\n\r\n" ,
		m_uFileIndex,m_szFileName.ptr(),
		m_uResumePosition,KVI_GNUTELLA_SERVER_NAME);

	//debug("Sending out:\n %s",buffer.ptr());

//#warning "This timeout should be tuneable"

	return sendBuffer(buffer.ptr(),buffer.len(),30);
}

bool KviGnutellaTransferThread::sendHttpOk()
{
//#warning "We could use 206 OK (Partial Content) when the resume position is not 0"
	KviStr buffer(KviStr::Format,
		"HTTP 200 OK\r\n" \
		"Server: %s\r\n" \
		"Content-Type: application/binary\r\n" \
		"Content-Range: bytes %u-%u\r\n" \
		"Content-Length: %u\r\n\r\n",KVI_GNUTELLA_SERVER_NAME,m_uResumePosition,
			m_uTransferEndPosition,m_uTransferEndPosition - m_uResumePosition);

	return sendBuffer(buffer.ptr(),buffer.len(),30);
}

bool KviGnutellaTransferThread::sendHttpGiv()
{
	KviStr serv;
	serv.bufferToHex((const char *)m_serventId,16);

	KviStr buffer(KviStr::Format,"GIV %u:%s/%s\n\n",
		m_uFileIndex,serv.ptr(),m_szFileName.ptr());

	return sendBuffer(buffer.ptr(),buffer.len(),30);
}

bool KviGnutellaTransferThread::expectHttpGiv()
{
	KviStr tmp;
	KviStr error = __tr_no_lookup("Invalid GIV header");
	KviStr line;
	KviStr num;
	bool bOk;
	unsigned int uIndex;
	unsigned char * servId;
	int len;

	if(!readHttpHeader())return false;

	KviStr * s = m_pHttpHeader->first();

	if(!s)goto invalid_push;

	if(!kvi_strEqualCIN(s->ptr(),"GIV ",4))
	{
		sendHttpError("HTTP 400 Bad Request","GIV method was expected");
		goto invalid_push;
	}

	line = *s;

	line.cutLeft(4);
	line.stripWhiteSpace();

	num = line.getToken(':');
	uIndex = num.toUInt(&bOk);
	if(!bOk)
	{
		sendHttpError("HTTP 400 Bad Request","File index was expected");
		goto invalid_push;
	}

	if(uIndex != m_uFileIndex)
	{
		sendHttpError("HTTP 408 Conflict","File index is not the expected one");
		error.sprintf(__tr_no_lookup("The index specified in the header is not the expected one (%u)"),uIndex);
		goto invalid_push;
	}

	if(line.len() < 32)
	{
		sendHttpError("HTTP 400 Bad Request","Hex-encoded servent identifier expected");
		error = __tr_no_lookup("Broken servent identifier");
		goto invalid_push;
	}

	num = line.left(32);
	line.cutLeft(32);

	len = num.hexToBuffer((char **)&servId);
	if(len != 16)
	{
		sendHttpError("HTTP 400 Bad Request","Hex-encoded servent identifier expected");
		if(len > 0)KviStr::freeBuffer((char *)servId);
		error = __tr_no_lookup("Invalid servent identifier");
		goto invalid_push;
	} else {
		if(!gnutella_compare_descriptor(servId,m_serventId))
		{
			sendHttpError("HTTP 408 Conflict","The servent identifier is not the expected one");
			error.sprintf(__tr_no_lookup("The servent identifier is not the expected one (%s)"),num.ptr());
			KviStr::freeBuffer((char *)servId);
			goto invalid_push;
		} else KviStr::freeBuffer((char *)servId);
	}

	line.stripWhiteSpace();
	if(line.firstCharIs('/'))line.cutLeft(1);
	line.stripWhiteSpace();
	if(!kvi_strEqualCS(line.ptr(),m_szFileName.ptr()))
	{
		sendHttpError("HTTP 408 Conflict","The file name is not the expected one");
		error.sprintf(__tr_no_lookup("The file name is not the expected one (%s)"),line.ptr());
		goto invalid_push;
	}

	return true;

invalid_push:

	tmp.sprintf(__tr_no_lookup("Rejecting PUSH: Invalid GIV request: %s"),error.ptr());

	return closeSock(tmp.ptr(),error.ptr());

}

bool KviGnutellaTransferThread::sendHttpError(const char * error,const char * warning)
{
	KviStr buffer(KviStr::Format,
		"%s\r\n" \
		"Warning: %s\r\n" \
		"Server: %s\r\n\r\n",error,warning,KVI_GNUTELLA_SERVER_NAME);

	return sendBuffer(buffer.ptr(),buffer.len(),15); 
}

bool KviGnutellaTransferThread::incomingHandshake()
{
	//
	// we end up here when we're serving an incoming upload request
	// (incoming connection on the transfer socket)
	// or when we're serving a PUSH request.
	//


	KviStr tmp;
	KviStr error = __tr_no_lookup("Invalid request syntax");
	KviGnutellaSharedFile * f = 0;
	KviStr szFileName;
	KviStr line;
	KviStr index;
	KviStr begin;
	int idx;
	unsigned int uMaxUploads;
	unsigned int uFileIndex;
	bool bOk;

	if(!readHttpHeader())return false;

	KviStr * s = m_pHttpHeader->first();

	if(!s)goto invalid_get;

	line = *s;

	if(!kvi_strEqualCIN(line.ptr(),"GET ",4))
	{
		if(kvi_strEqualCIN(line.ptr(),"GNUTELLA CONNECT",16))
		{
			sendHttpError("HTTP 405 Method Not Allowed","GET method was expected: " \
				"this server listens for network connections on another port: " \
				"your client is broken, send a bug report to the client author");

		} else {
			sendHttpError("HTTP 405 Method Not Allowed","GET method was expected");
		}
		goto invalid_get;
	}
	line.cutToFirst('/');

	if(!kvi_strEqualCIN(line.ptr(),"GET/",4))
	{
		sendHttpError("HTTP 400 Bad Request","GET expected in method selection line");
		goto invalid_get;
	}
	line.cutToFirst('/');

	index = line.getToken('/');

	uFileIndex = index.toUInt(&bOk);
	if(!bOk)
	{
		sendHttpError("HTTP 400 Bad Request","Numeric file index expected");
		goto invalid_get;
	}

	idx = line.findFirstIdx("HTTP",false);
	if(idx == -1)
	{
		sendHttpError("HTTP 400 Bad Request","HTTP version expected");
		goto invalid_get;
	}

	szFileName = line.left(idx);
	szFileName.stripWhiteSpace();

	if(m_connectionType == Passive)
	{
		// this is a passive connection
		// we get the parameters from here
		gnutella_shared_files_lock();
		f = gnutella_find_shared_file(uFileIndex);
		if(f)
		{
			if(kvi_strEqualCS(f->szFileName.ptr(),szFileName.ptr()))
			{
				m_szFileName = f->szPath;
				m_szFileName.ensureLastCharIs('/');
				m_szFileName.append(f->szFileName);
				m_uTransferEndPosition = f->uFileSize;
				m_uTotalFileSize = f->uFileSize;
			}
		}
		gnutella_shared_files_unlock();


		if(m_szFileName.isEmpty())
		{
			sendHttpError("HTTP 404 File Not Found","The file was probably moved, retry the search"); // we're always killing the socket...so no need for check here	
			error = __tr_no_lookup("File Not Found");
	
			goto invalid_get;
		} else {
	
			KviThreadDataEvent<KviGnutellaTransferFileInfo> * ev = new KviThreadDataEvent<KviGnutellaTransferFileInfo>(KVI_GNUTELLA_TRANSFER_THREAD_EVENT_FILE_INFO);
		
			KviGnutellaTransferFileInfo * inf = new KviGnutellaTransferFileInfo;
			inf->szFileName = szFileName;
			inf->uFileSize = m_uTotalFileSize;
			inf->uTransferId = m_uId;
		
			ev->setData(inf);
		
			postEvent(m_pTab,ev);
		}

	} else {
		// this is a PUSH that we're serving
		// the params must match the ones that we have sent
		// in the GIV
		if(m_uFileIndex != uFileIndex)
		{
			error = __tr_no_lookup("The file index requested does not match the PUSH request");
			sendHttpError("HTTP 408 Conflict","File index does not match the push request");
			goto invalid_get;
		}

		if(!kvi_strEqualCS(szFileName.ptr(),m_szFileName.ptr()))
		{
			error = __tr_no_lookup("The file name requested does not match the PUSH request");
			sendHttpError("HTTP 408 Conflict","File name does not match the push request");
			goto invalid_get;
		}

		m_szFileName = m_szFilePath;
		m_szFileName.ensureLastCharIs('/');
		m_szFileName.append(szFileName);
	}

	for(s = m_pHttpHeader->first();s;s = m_pHttpHeader->next())
	{
		if(kvi_strEqualCIN("Range:",s->ptr(),6))break;
	}

	if(s)
	{
		// got range request
		line = *s;
		line.cutLeft(6);
		line.stripWhiteSpace();
		if(kvi_strEqualCIN(line.ptr(),"bytes",5))
		{
			line.cutLeft(5);
			line.stripWhiteSpace();
		}
		if(kvi_strEqualCIN(line.ptr(),"=",1))
		{
			line.cutLeft(1);
			line.stripWhiteSpace();
		}
		begin = line.getToken('-');
		m_uResumePosition = begin.toUInt(&bOk);
		if(!bOk)
		{
			error = __tr_no_lookup("Invalid \"Range\" request");
			sendHttpError("HTTP 400 Bad Request","Could not understand the Range header field");
			goto invalid_get;
		}
		if(line.hasData())
		{
			unsigned int uEndPos = line.toUInt(&bOk);
			if(bOk)m_uTransferEndPosition = uEndPos + 1;
		}
	}

//#warning "Ensure that also the range-end is equal to the end of the file!"

	if(m_uResumePosition >= m_uTransferEndPosition)
	{
		error = __tr_no_lookup("Broken \"Range\" request");
		KviStr tmpRange(KviStr::Format,"Current file size is %u bytes",m_uTransferEndPosition);
		sendHttpError("HTTP 416 Requested Range Not Satisfiable",tmpRange.ptr());
		goto invalid_get;
	}

	// now check if we can actually serve this request
	g_pGnutellaOptionsMutex->lock();
	uMaxUploads = g_pGnutellaOptions->m_uMaxUploads;
	g_pGnutellaOptionsMutex->unlock();

	g_pGnutellaTransferCountersMutex->lock();
	if(g_uGnutellaCurrentUploadTransfers >= uMaxUploads)
	{
		g_pGnutellaTransferCountersMutex->unlock();
		error = __tr_no_lookup("Too many uploads");
		sendHttpError("HTTP 503 Service Unavailable","Server busy: try again later");
		goto invalid_get;
	}
	// ok...we can... count this as running upload
	m_bDecrementCurrentUploadTransfersOnDeath = true; // remember to decrement this
	g_uGnutellaCurrentUploadTransfers++;
	g_pGnutellaTransferCountersMutex->unlock();

	postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
		new KviStr(KviStr::Format,__tr_no_lookup("Accepting GET request from %s:%u for file %s (range %u-%u)"),m_szIp.ptr(),m_uPort,
		m_szFileName.ptr(),m_uResumePosition,m_uTransferEndPosition - 1)));

	return sendHttpOk();

invalid_get:

	tmp = __tr_no_lookup("Rejecting GET request: ");
	if(error.hasData())tmp.append(error);

	for(s = m_pHttpHeader->first();s;s = m_pHttpHeader->next())
	{
		tmp.append(KviStr::Format,__tr_no_lookup("\n[transfer %u]:    %s"),m_uId,s->ptr());
	}

	return closeSock(tmp.ptr(),error.ptr());
}

bool KviGnutellaTransferThread::expectHttpOk()
{
	KviStr tmp;
	KviStr error;

	if(!readHttpHeader())return false;

	KviStr * s = m_pHttpHeader->first();

	if(!s)goto invalid_response;

	if(!kvi_strEqualCIN(s->ptr(),"HTTP",4))
	{
		error = __tr_no_lookup("Invalid request syntax");
		setRetryHint(false,5);
		goto invalid_response;
	}

	if(s->findFirstIdx(" 200 OK",false) != -1)goto response_ok;
	if(m_uResumePosition > 0)
	{
		// 206 (Partial content) is OK for us too
		if(s->findFirstIdx(" 206 OK",false) != -1)goto response_ok;
	}

	// Check the error code...and eventually set the retry hint

	// 503 is "Server Busy (Bear Share)" or "Service Unavailable (Lime Wire)"
	if(s->findFirstIdx(" 503 ") != -1)setRetryHint(true,30);

	error = *s;
	goto invalid_response;


response_ok:

	// ok...the server accepted...
	// ensure that the content length is ok

	for(s = m_pHttpHeader->first();s;s = m_pHttpHeader->next())
	{
		if(kvi_strEqualCIN("Content-length:",s->ptr(),15))break;
	}

	if(s)
	{
		KviStr buffer = s->ptr();
		buffer.cutLeft(15);
		buffer.stripWhiteSpace();
		bool bOk;
		unsigned int uSize = buffer.toUInt(&bOk);
		if(bOk)
		{
			if(uSize != ((uint)(m_uTotalFileSize - m_uResumePosition)))
			{
				// hmmmm...different file size ?
				postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
					new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: The server declared the data content size to be %u bytes instead of %u (expected from QueryHit), trusting him, even if I'm confused a bit"),
						m_uId,uSize,m_uTotalFileSize - m_uResumePosition)));
				m_uTransferEndPosition = uSize;
			}
		} else {
			// invalid content length header line
			postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
				new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: Invalid Content-length header line sent by the server (%s): trusting the QueryHit file size"),
					m_uId,s->ptr())));
		}
	} else {
		// no content-length header line...that's bad
		postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
			new KviStr(__tr_no_lookup("No Content-length header line sent by the server: trusting the QueryHit file size"))));
	}

	// ensure that the content range is ok
	
	for(s = m_pHttpHeader->first();s;s = m_pHttpHeader->next())
	{
		if(kvi_strEqualCIN("Content-range:",s->ptr(),14))break;
	}

	if(s)
	{
		KviStr buffer = s->ptr();
		buffer.cutLeft(14);
		buffer.stripWhiteSpace();
		buffer.stripWhiteSpace();
		if(kvi_strEqualCIN(buffer.ptr(),"bytes",5))
		{
			buffer.cutLeft(5);
			buffer.stripWhiteSpace();
		}
		if(kvi_strEqualCIN(buffer.ptr(),"=",1))
		{
			buffer.cutLeft(1);
			buffer.stripWhiteSpace();
		}
		int idx = buffer.findFirstIdx('-');
		if(idx != -1)buffer.cutRight(buffer.len() - idx);
		buffer.stripWhiteSpace();
		bool bOk;
		unsigned int uResume = buffer.toUInt(&bOk);
		if(bOk)
		{
			if(uResume != m_uResumePosition)
			{
				// hmmmm...different resume position ?
				postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
					new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: The server specified an invalid resume position (%u where %u was expected), this was rather fatal"),
						m_uId,uResume,m_uResumePosition)));
				error = __tr_no_lookup("Invalid resume position");
				setRetryHint(false,5);
				goto invalid_response;
			}
		} else {
			// invalid content range header line
			postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
				new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: Invalid Content-range header line sent by the server (%s): trying to blindly trust the server"),
					m_uId,s->ptr())));
		}
	} else {
		// no content-range header line...that's bad
		postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
			new KviStr(__tr_no_lookup("No Content-range header line sent by the server: trying to blindly trust the server"))));
	}

	tmp = __tr_no_lookup("Server response to HTTP GET:");
	for(s = m_pHttpHeader->first();s;s = m_pHttpHeader->next())
	{
		tmp.append(KviStr::Format,__tr_no_lookup("\n[transfer %u]:    %s"),m_uId,s->ptr());
	}

	g_pGnutellaTransferCountersMutex->lock();
	m_bDecrementCurrentDownloadTransfersOnDeath = true; // remember to decrement this
	g_uGnutellaCurrentDownloadTransfers++;
	g_pGnutellaTransferCountersMutex->unlock();


	postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,new KviStr(tmp)));

	return true;

invalid_response:

	tmp = __tr_no_lookup("Invalid response while expecting http OK: ");
	tmp.append(error);
	for(s = m_pHttpHeader->first();s;s = m_pHttpHeader->next())
	{
		tmp.append(KviStr::Format,__tr_no_lookup("\n[transfer %u]:    %s"),m_uId,s->ptr());
	}

	return closeSock(tmp.ptr(),error.ptr());

}

void KviGnutellaTransferThread::postProgressEvent(unsigned int uActualSize,unsigned int uTotalSizeToRecv,unsigned int uRate,bool bStalled)
{
	KviGnutellaTransferProgress * pr = new KviGnutellaTransferProgress;
	pr->szProgress.sprintf(__tr_no_lookup("%u of %u bytes [%d% at %u (bytes/sec)%s"),
			uActualSize,uTotalSizeToRecv,uTotalSizeToRecv ? (uActualSize * 100) / uTotalSizeToRecv : 0,uRate,
			bStalled ? __tr_no_lookup("]: stalled") : "]");
	pr->uTransferId = m_uId;
	KviThreadDataEvent<KviGnutellaTransferProgress> * ev = new KviThreadDataEvent<KviGnutellaTransferProgress>(KVI_GNUTELLA_TRANSFER_THREAD_EVENT_PROGRESS);
	ev->setData(pr);
		
	postEvent(m_pTab,ev);
}

bool KviGnutellaTransferThread::sendFile()
{
	// this thingie sends out a file
	// sending bytes in the range [m_uResumePosition,(m_uTransferEndPosition - 1)]
	//
	// ... good luck :)

	bool bInStall = false;

#ifdef COMPILE_ON_WINDOWS
	FILE * f = fopen(m_szFileName.ptr(),"rb");
#else
	FILE * f = fopen(m_szFileName.ptr(),"r");
#endif
	if(!f)
	{
		KviStr tmp(KviStr::Format,__tr_no_lookup("Can't open the file %s for reading"),m_szFileName.ptr());
		return closeSock(tmp.ptr());
	}

//#warning "SHould actually ensure that the file size on disk is greater or equal to m_uTransferEndPosition"

	unsigned int uBytesToSend = m_uTransferEndPosition - m_uResumePosition;
	unsigned int uBytesSent = 0;

	if(uBytesToSend == 0)
	{
		fclose(f);
		return closeSock(__tr_no_lookup("Unexpected 0 bytes transfer requested"));
	}


	if(m_uResumePosition > 0)
	{
		if(!fseek(f,m_uResumePosition,SEEK_SET))
		{
			fclose(f);
			return closeSock(__tr_no_lookup("File I/O Error: can't seek to the resume position"));
		}
	}

	postProgressEvent(uBytesSent,uBytesToSend,0);

	struct timeval firsttmv;
	struct timeval lasttmv;
	struct timeval curtmv;
	struct timeval stalltmv;

	kvi_gettimeofday(&lasttmv,0);
	firsttmv.tv_usec = lasttmv.tv_usec;
	firsttmv.tv_sec  = lasttmv.tv_sec;

	unsigned int lastProgressSize = uBytesSent;

	char buffer[2048];

	unsigned int uBytesInBuffer = 0;

	while(uBytesSent < uBytesToSend)
	{
		if(!processInternalEvents())
		{
			fclose(f);
			return closeSock();
		}

		int nRet = selectForWriteStep();

		if(nRet < 0)
		{
			fclose(f);
			return closeSock();
		}

		if(nRet > 0)
		{
			// ok...space to write something
			unsigned int uExpectedBytesInBuffer = uBytesToSend - uBytesSent;
			if(uExpectedBytesInBuffer > 2048)uExpectedBytesInBuffer = 2048;

			if(uBytesInBuffer < uExpectedBytesInBuffer)
			{
				int iToRead = uExpectedBytesInBuffer - uBytesInBuffer;
				int readed = fread(buffer + uBytesInBuffer,1,iToRead,f);
				if(readed < iToRead)
				{
					fclose(f);
					return closeSock(__tr_no_lookup("File I/O Error: unexpected EOF"));
				}
				uBytesInBuffer += iToRead;
				__range_valid(uBytesInBuffer == uExpectedBytesInBuffer);
			}

			int sent = kvi_socket_send(m_sock,buffer,uBytesInBuffer);
			if(sent > 0)
			{
				__range_valid(((unsigned int)sent) <= uBytesInBuffer);
				// ok...sent at least something
				kvi_memmove(buffer,buffer + sent,uBytesInBuffer - sent);
				uBytesInBuffer -= sent;
				uBytesSent += sent;
			} else {
				// mmmh: error ?
				if(sent < 0)
				{
					int err = kvi_socket_error();
#ifdef COMPILE_ON_WINDOWS
					if((err != EAGAIN) && (err != EINTR) && (err != WSAEWOULDBLOCK))
#else
					if((err != EAGAIN) && (err != EINTR))
#endif
					{
						fclose(f);
						KviStr tmp(KviStr::Format,__tr_no_lookup("Write error: %s (errno=%d)"),
							kvi_getErrorString(kvi_errorFromSystemError(err)),err);
						return closeSock(tmp.ptr());
					}
				}
			}

		}
	
		usleep(10000);

		// we report progress every 5 secs

		kvi_gettimeofday(&curtmv,0);

		unsigned long int diffmsec = ((curtmv.tv_sec * 1000) + (curtmv.tv_usec / 1000)) - ((lasttmv.tv_sec * 1000) + (lasttmv.tv_usec / 1000));

		if(diffmsec > 5000)
		{

			unsigned long int diffksize = (uBytesSent - lastProgressSize) * 1000;
			lastProgressSize = uBytesSent;

			lasttmv.tv_usec = curtmv.tv_usec;
			lasttmv.tv_sec = curtmv.tv_sec;

			unsigned int uRate = diffksize / diffmsec;

			if(uRate <= m_uStallTransferRate)
			{
				// ok...are we stalled ?
				if(bInStall)
				{
					unsigned int stallTime = ((curtmv.tv_sec * 1000) + (curtmv.tv_usec / 1000)) - ((stalltmv.tv_sec * 1000) + (stalltmv.tv_usec / 1000));
					if(stallTime > m_uStallTimeoutInMSecs)
					{
						// bad , bad....we're stalled
						if(m_bSuicideOnStall)
						{
							fclose(f);
							return closeSock(__tr_no_lookup("Stalled (user option kill)"));
						} else {
							stalltmv.tv_sec += 5; // avoid overflow later
							postProgressEvent(uBytesSent,uBytesToSend,uRate,true);
						}
					} else {
						postProgressEvent(uBytesSent,uBytesToSend,uRate);
					}
				} else {
					bInStall = true;
					stalltmv.tv_usec = curtmv.tv_usec;
					stalltmv.tv_sec = curtmv.tv_sec;	
					// but don't say we're stalled , yet
					postProgressEvent(uBytesSent,uBytesToSend,uRate);
				}

			} else {
				bInStall = false;
				postProgressEvent(uBytesSent,uBytesToSend,uRate);
			}
		}

	}

	closeSock();

	fclose(f);

	// post the last progress event

	unsigned long int diffmsecs = ((curtmv.tv_sec * 1000) + (curtmv.tv_usec / 1000)) - ((firsttmv.tv_sec * 1000) + (firsttmv.tv_usec / 1000));
	KviGnutellaTransferProgress * pr = new KviGnutellaTransferProgress;
	pr->szProgress.sprintf(__tr_no_lookup("Completed at %u (bytes/sec)"),uBytesToSend / (diffmsecs / 1000));
	pr->uTransferId = m_uId;

	KviThreadDataEvent<KviGnutellaTransferProgress> * ev = new KviThreadDataEvent<KviGnutellaTransferProgress>(KVI_GNUTELLA_TRANSFER_THREAD_EVENT_PROGRESS);
	ev->setData(pr);

	postEvent(m_pTab,ev);

	KviGnutellaTransferCompletedInfo * tci = new KviGnutellaTransferCompletedInfo;
	tci->uTransferId = m_uId;

	KviThreadDataEvent<KviGnutellaTransferCompletedInfo> * evc = new KviThreadDataEvent<KviGnutellaTransferCompletedInfo>(KVI_GNUTELLA_TRANSFER_THREAD_EVENT_COMPLETED);
	evc->setData(tci);

	postEvent(m_pTab,evc);


	return true;

}



bool KviGnutellaTransferThread::receiveFile()
{
	// ok ...receive this file man!

	bool bInStall = false;

	struct timeval firsttmv;
	struct timeval lasttmv;
	struct timeval curtmv;
	struct timeval stalltmv;

	kvi_makeDir(m_szFilePath.ptr()); // ensure that this exists

	KviStr szPath = m_szFilePath;

	szPath.ensureLastCharIs('/');

	m_szFileName.cutToLast('/'); // so we have no slashes floating around

	if(m_szFileName.isEmpty())
	{
		return closeSock(__tr_no_lookup("Invalid empty file name"));
	}

	szPath.append(m_szFileName);

#ifdef COMPILE_ON_WINDOWS
	FILE * f = fopen(szPath.ptr(),m_uResumePosition > 0 ? "ab" : "wb");
#else
	FILE * f = fopen(szPath.ptr(),m_uResumePosition > 0 ? "a" : "w");
#endif

	if(!f)
	{
		KviStr tmp(KviStr::Format,__tr_no_lookup("Can't open the file %s for writing"),szPath.ptr());
		return closeSock(tmp.ptr());
	}

	if(m_uResumePosition > 0)
	{
		if(fseek(f,m_uResumePosition,SEEK_SET) != 0)
		{
			// ops... failed to seek the file ?
			fclose(f);
			return closeSock(__tr_no_lookup("Failed to seek the file to the resume position"));
		}
	}

	unsigned int uBytesToRecv = m_uTransferEndPosition - m_uResumePosition;

	if(uBytesToRecv == 0)
	{
		fclose(f);
		return closeSock(__tr_no_lookup("Unexpected 0 bytes transfer requested"));
	}

	// we may already have some data in m_inBuffer !!!

	unsigned int totalSize = 0;

	if(m_uInBufferDataLen > 0)
	{
		if(fwrite(m_pInBuffer,1,m_uInBufferDataLen,f) != m_uInBufferDataLen)
		{
			fclose(f);
			return closeSock(__tr_no_lookup("File I/O error"));
		} else {
			totalSize += m_uInBufferDataLen;
			m_uInBufferDataLen = 0;
			m_uInBufferRealLen = 1024;
			m_pInBuffer = (char *)kvi_realloc(m_pInBuffer,1024);
		}
	}

	// post the first progress event

	postProgressEvent(totalSize,uBytesToRecv,0);

	kvi_gettimeofday(&lasttmv,0);
	firsttmv.tv_usec = lasttmv.tv_usec;
	firsttmv.tv_sec  = lasttmv.tv_sec;

	unsigned int lastProgressSize = totalSize;

	while(totalSize < uBytesToRecv)
	{
		if(!processInternalEvents())
		{
			fclose(f);
			return closeSock();
		}

		int nRet = selectForReadStep();

		if(nRet < 0)
		{
			fclose(f);
			setRetryHint(false,5);
			return closeSock();
		}
		if(nRet > 0)
		{

			__range_valid(m_uInBufferDataLen == 0);

			if(!readData())
			{
				fclose(f);
				setRetryHint(true,5);
				return closeSock();
			}


			if(m_uInBufferDataLen > 0)
			{
				if(fwrite(m_pInBuffer,1,m_uInBufferDataLen,f) != m_uInBufferDataLen)
				{
					fclose(f);
					return closeSock(__tr_no_lookup("File I/O error"));
				} else {
					totalSize += m_uInBufferDataLen;
					m_uInBufferDataLen = 0;
					m_uInBufferRealLen = 1024;
					m_pInBuffer = (char *)kvi_realloc(m_pInBuffer,1024);
				}
			}
		}
	
		usleep(10000);

		// we report progress every 5 secs

		kvi_gettimeofday(&curtmv,0);

		unsigned long int diffmsec = ((curtmv.tv_sec * 1000) + (curtmv.tv_usec / 1000)) - ((lasttmv.tv_sec * 1000) + (lasttmv.tv_usec / 1000));

		if(diffmsec > 5000)
		{

			unsigned long int diffksize = (totalSize - lastProgressSize) * 1000;
			lastProgressSize = totalSize;

			lasttmv.tv_usec = curtmv.tv_usec;
			lasttmv.tv_sec = curtmv.tv_sec;


			unsigned int uRate = diffksize / diffmsec;

			if(uRate <= m_uStallTransferRate)
			{
				// ok...are we stalled ?
				if(bInStall)
				{
					unsigned int stallTime = ((curtmv.tv_sec * 1000) + (curtmv.tv_usec / 1000)) - ((stalltmv.tv_sec * 1000) + (stalltmv.tv_usec / 1000));
					if(stallTime > m_uStallTimeoutInMSecs)
					{
						// bad , bad....we're stalled
						if(m_bSuicideOnStall)
						{
							fclose(f);
							setRetryHint(true,5);
							return closeSock(__tr_no_lookup("Stalled (user option kill)"));
						} else {
							stalltmv.tv_sec += 5; // avoid overflow later
							postProgressEvent(totalSize,uBytesToRecv,uRate,true); //just tell to the user that we're stalled
						}
					} else {
						postProgressEvent(totalSize,uBytesToRecv,uRate);
					}
				} else {
					bInStall = true;
					stalltmv.tv_usec = curtmv.tv_usec;
					stalltmv.tv_sec = curtmv.tv_sec;
					// but don't say we're stalled , yet
					postProgressEvent(totalSize,uBytesToRecv,uRate);
				}

			} else {
				bInStall = false;
				postProgressEvent(totalSize,uBytesToRecv,uRate);
			}
		}

	}

	closeSock();

	fclose(f);

	// post the last progress event

	unsigned long int diffmsecs = ((curtmv.tv_sec * 1000) + (curtmv.tv_usec / 1000)) - ((firsttmv.tv_sec * 1000) + (firsttmv.tv_usec / 1000));
	KviGnutellaTransferProgress * pr = new KviGnutellaTransferProgress;
	pr->szProgress.sprintf(__tr_no_lookup("Completed at %u (bytes/sec)"),totalSize / (diffmsecs / 1000));
	pr->uTransferId = m_uId;

	KviThreadDataEvent<KviGnutellaTransferProgress> * ev = new KviThreadDataEvent<KviGnutellaTransferProgress>(KVI_GNUTELLA_TRANSFER_THREAD_EVENT_PROGRESS);
	ev->setData(pr);

	postEvent(m_pTab,ev);

	KviGnutellaTransferCompletedInfo * tci = new KviGnutellaTransferCompletedInfo;
	tci->uTransferId = m_uId;

	KviThreadDataEvent<KviGnutellaTransferCompletedInfo> * evc = new KviThreadDataEvent<KviGnutellaTransferCompletedInfo>(KVI_GNUTELLA_TRANSFER_THREAD_EVENT_COMPLETED);
	evc->setData(tci);

	postEvent(m_pTab,evc);

	// now just rename the file

	KviStr newPath;

	g_pGnutellaOptionsMutex->lock();
	newPath = g_pGnutellaOptions->m_szDownloadDirectory;
	g_pGnutellaOptionsMutex->unlock();

	newPath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	newPath.append(m_szFileName);

	kvi_adjustFilePath(newPath);

	while(kvi_fileExists(newPath.ptr()))newPath.append(".rnm");

	if(kvi_renameFile(szPath.ptr(),newPath.ptr()))
	{
		postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,new KviStr(KviStr::Format,__tr_no_lookup("The downloaded file has been saved as file:/%s"),newPath.ptr())));
	} else {
		postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,new KviStr(KviStr::Format,__tr_no_lookup("Failed to rename the downloaded file to %s, it is actually saved as file:/%s (it would be a good idea to move it from there)"),
			newPath.ptr(),szPath.ptr())));
	}

	return true;
}


bool KviGnutellaTransferThread::lockIncomingFile()
{
	bool bOk = true;
	KviStr error;
	KviStr szTmpPath;
	Q_UINT32 now;
	KviStr tmp;

	//
	// Lock file format:
	//
	//        [      KVILOCK      ][    S    ][    PARAM    ]
	//         ID 7 bytes            Status       Parameter
	//         contains the          1 byte        4 bytes
	//         string "KVILOCK"
	//
	//     If Status is "L" , then the file is currently LOCKED
	//     (being downloaded). The Parameter contains the time
	//     of the lock.
	//     If Status is "I" , then the file is incomplete
	//     and Param contains the expected total file size
	//

	m_szLockFileName = m_szFilePath;
	m_szLockFileName.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	m_szLockFileName.append('.');
	m_szLockFileName.append(m_szFileName);
	m_szLockFileName.append(".kvilock");

	KviStr szCompleteFilePath = m_szFilePath;
	szCompleteFilePath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	szCompleteFilePath.append(m_szFileName);

	g_pGnutellaDownloadFilesMutex->lock();

	kvi_makeDir(m_szFilePath.ptr());
	
#ifdef COMPILE_ON_WINDOWS
	FILE * f = fopen(m_szLockFileName.ptr(),"rb");
#else
	FILE * f = fopen(m_szLockFileName.ptr(),"r");
#endif
	if(f)
	{
		// ops ...there is a lock currently
		unsigned char buffer[12];
		int readed = fread(buffer,1,12,f);
		fclose(f);
		if(readed < 12)
		{
			// invalid lock... we'll overwrite it
			// if the file exists on disk, we'll move it out of the way
			goto rename_file;
		}

		if(!kvi_strEqualCSN("KVILOCK",(char *)buffer,7))
		{
			// invalid lock... we'll overwrite it
			// if the file exists on disk , we'll move it out of the way
			goto rename_file;
		}

		if(buffer[7] == 'L')
		{
			// this is a current lock...check the lock time
			Q_UINT32 tmt = *((Q_UINT32 *)(buffer + 8));
			now = (Q_UINT32)time(0);
			if((now - tmt) > 720000)
			{
				// more than 200 hours old lock
				// this is a download that runs since more than 8 days...
				// rather impossible: we're going to overwrite the lock at all
				// since we have no way of dicovering the expected file size
				// if the file exists on disk , we move it out of the way
				goto rename_file;
			}
			// ok...this is a valid lock...we can't kill it
			bOk = false;
			error.sprintf(__tr_no_lookup("[transfer %u]: The file is currently locked by filelock %s, " \
							"so it might be actually being downloaded." \
							"If you feel this message to be in error, " \
							"please remove the filelock by hand and restart the download."),m_uId,m_szLockFileName.ptr());
			goto lock_terminated;
		}

//check_incomplete:

		if(buffer[7] == 'I')
		{
			// the file is incomplete: let us check the expected file size and if the file actually exists on disk
			Q_UINT32 expectedFileSize = *((Q_UINT32 *)(buffer + 8));
			if(kvi_fileExists(szCompleteFilePath.ptr()))
			{
#ifdef COMPILE_ON_WINDOWS
				struct _stat st;
				if(_stat(szCompleteFilePath.ptr(),&st) == 0)
#else
				struct stat st;
				if(stat(szCompleteFilePath.ptr(),&st) == 0)
#endif
				{
					// the file exists on disk and we have the file size
					// let's check the current file size
					Q_UINT32 curFileSize = (Q_UINT32)st.st_size;
					if(expectedFileSize <= curFileSize)
					{
						// terminated download , or invalid file size
						// we move the file out of the way , and overwrite the lock
						postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
							new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: The file %s exists on disk but it has size greater than expected: this might be a terminated download"),
							m_uId,szCompleteFilePath.ptr())));
						goto rename_file;
					}
					// the expected file size is greater than curFileSize
					if(expectedFileSize == m_uTotalFileSize)
					{
						// ok....this might be a resumable file
						m_uResumePosition = curFileSize;
						postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
							new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: The file %s exists on disk: attempting a resume from position %u"),
							m_uId,szCompleteFilePath.ptr(),curFileSize)));

						goto make_lock;
					} else {
						// no way...we were looking for some other file to resume
						// we move the old file out of the way.
						postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
							new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: The file %s exists on disk but the expected final sizes do not match"),
							m_uId,szCompleteFilePath.ptr())));
						goto rename_file;
					}
				} else {
					// ops... problems stating the file ?
					// what we can do ? we'll rename it and simply overwrite
					postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
						new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: The file %s exists on disk but there was an arror while analyzing it (stat() call failed)"),
						m_uId,szCompleteFilePath.ptr())));
					goto rename_file;
				}
			} else {
				// the file does not exist on disk: the file lock is wrong: we'll overwrite it
				goto make_lock;
			}
		}
		// otherwise it is not a valid lock...we'll overwrite it
		// if the file exists on disk , we rename it
		postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
			new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: File %s exists on disk but its lock seems to be broken"),
			m_uId,szCompleteFilePath.ptr())));
	}

rename_file:
	// we move the file out of the way
	szTmpPath = szCompleteFilePath;
	if(kvi_fileExists(szTmpPath.ptr()))
	{
		postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
			new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: File %s exists: Moving it out of my way (renaming to %s.rnm)"),
			m_uId,szTmpPath.ptr(),szTmpPath.ptr())));

		szTmpPath.append(".rnm");
		while(kvi_fileExists(szTmpPath.ptr()))szTmpPath.append(".rnm");
		kvi_renameFile(szCompleteFilePath.ptr(),szTmpPath.ptr());
	}
make_lock:

#ifdef COMPILE_ON_WINDOWS
	f = fopen(m_szLockFileName.ptr(),"wb");
#else
	f = fopen(m_szLockFileName.ptr(),"w");
#endif
	if(f)
	{
		if(fwrite("KVILOCKL",1,8,f) != 8)
		{
			goto error_in_lock;
		}
		now = (Q_UINT32)time(0);
		if(fwrite((const void *)&now,1,4,f) != 4)
		{
			goto error_in_lock;
		}
		fclose(f);

		goto lock_terminated;

error_in_lock:

		fclose(f);
		// ops..can't make the lock
		error.sprintf(__tr_no_lookup("[transfer %u]: Failed to write the lock file (%s)"),
				m_uId,m_szLockFileName.ptr());
		bOk = false;
	} else {
		// ops..can't make the lock
		error.sprintf(__tr_no_lookup("[transfer %u]: Failed to open the lock file for writing (%s): %s (errno=%d)"),
				m_uId,m_szLockFileName.ptr(),kvi_getErrorString(kvi_errorFromSystemError(errno)),errno);
		bOk = false;
	}

lock_terminated:
	g_pGnutellaDownloadFilesMutex->unlock();

	if(!bOk)postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_ERROR,new KviStr(error)));

	return bOk;
}

void KviGnutellaTransferThread::unlockIncomingFile()
{
	// ok...if the transfer is terminated succesfully
	// we just have to unlink the lock file
	// if the transfer has been stopped at some point
	// we need to rewrite the lock file in order to remember
	// the expected file size.

	KviStr error;
	FILE * f;
	Q_UINT32 fSize;


	// first of all , we check the size of the file on disk (if it is there at all)
	KviStr szCompleteFilePath = m_szFilePath;
	szCompleteFilePath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	szCompleteFilePath.append(m_szFileName);

	g_pGnutellaDownloadFilesMutex->lock();

	if(kvi_fileExists(szCompleteFilePath.ptr()))
	{
		// mmmh..the file is on disk actually
		// we need to check the size
		struct stat st;
		if(stat(szCompleteFilePath.ptr(),&st) == 0)
		{
			// the file exists on disk and we have the file size
			// let's check the current file size
			Q_UINT32 curFileSize = (Q_UINT32)st.st_size;
			if(m_uTotalFileSize <= curFileSize)
			{
				// terminated download...but sth weird is happening
				// we have failed to move the file to the right directory
				// the best thing that comes in my mind is to unlink the lock
				// and forget about it :)
				goto do_unlink;
			}
			// the file on disk is smaller than the expected file size
			// the download failed for some reason: let's remember it
			goto remember_resume;
		} else {
			// can't stat the file ?
			// huh.... I don't want to bother too much...unlink the lock and die
			goto do_unlink;
		}
	} else {
		// the file has been succesfully moved to the complete downloads directory
		// just unlink the file
		goto do_unlink;
	}

remember_resume:
#ifdef COMPILE_ON_WINDOWS
	f = fopen(m_szLockFileName.ptr(),"wb");
#else
	f = fopen(m_szLockFileName.ptr(),"w");
#endif
	if(f)
	{
		if(fwrite("KVILOCKI",1,8,f) != 8)
		{
			goto error_in_resume_lock;
		}
		fSize = m_uTotalFileSize;
		if(fwrite((const void *)&fSize,1,4,f) != 4)
		{
			goto error_in_resume_lock;
		}
		fclose(f);

		goto unlock_terminated;

error_in_resume_lock:

		fclose(f);
		// ops..can't make the lock
		error.sprintf(__tr_no_lookup("[transfer %u]: Failed to write the resume lock file (%s)"),
				m_uId,m_szLockFileName.ptr());
	} else {
		// ops..can't make the lock
		error.sprintf(__tr_no_lookup("[transfer %u]: Failed to open the resume lock file for writing (%s): %s (errno=%d)"),
				m_uId,m_szLockFileName.ptr(),kvi_getErrorString(kvi_errorFromSystemError(errno)),errno);
	}

	goto unlock_terminated;

do_unlink:

	unlink(m_szLockFileName.ptr());

unlock_terminated:

	g_pGnutellaDownloadFilesMutex->unlock();

	if(error.hasData())postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_ERROR,new KviStr(error)));

}

void KviGnutellaTransferThread::postRetryEvent(bool bSameHost,unsigned int uRetryDelay)
{
	KviThreadDataEvent<KviGnutellaTransferRetryInfo> * ev = new KviThreadDataEvent<KviGnutellaTransferRetryInfo>(KVI_GNUTELLA_TRANSFER_THREAD_EVENT_RETRY);
	KviGnutellaTransferRetryInfo * inf = new KviGnutellaTransferRetryInfo;
	inf->uTransferId = m_uId;
	inf->bRetryFromSameHost = bSameHost;
	inf->uRetryAfterSecs    = uRetryDelay;
	ev->setData(inf);
	postEvent(m_pTab,ev);
}

void KviGnutellaTransferThread::run()
{
	//
	// Initial states:
	//
	//    Active Recv
	//       m_szIp            is the address of the remote host
	//       m_uPort           is the port of the remote host
	//       m_szFileName      is set to the filename we want to receive
	//       m_szFilePath      is set to the Incomplete Download directory
	//       m_uTotalFileSize  is set to the full size of the file
	//       m_uTransferEndPosition   is initially set to the full size of the file
	//       m_uResumePosition is 0
	//       m_uFileIndex      is set to the index of the file we want to receive
	//       m_serventId       is the servent id of the node we're going to contact      
	//
	//    Active Send
	//       m_szIp            is the address of the machine we have to contact
	//       m_uPort           is the port of the remote machine
	//       m_szFileName      is the file name that we have to send
	//       m_szFilePath      is the path of the file that we have to send
	//       m_serventId       is the local servent id
	//       m_uTransferEndPosition   is set to the full size of the file
	//       m_uTotalFileSize  is set to the full size of the file
	//       m_uFileIndex      is set to the local index of the file we have to send
	//
	//    Passive Send
	//       m_szIp            is the address of the machine that IS connected on m_sock
	//       m_uPort           is the remote port of that machine
	//                         we know nothing about the file name,path and size
	//
	//    Passive Recv
	//                         we're never in this state at the beginning
	//                         we always come from an Active Recv, so the
	//                         variables set are exactly the same


	// Grab some options now
	g_pGnutellaOptionsMutex->lock();

	m_bSuicideOnStall       = (m_transferDirection == Recv) ? g_pGnutellaOptions->m_bKillStalledDownloadTransfers : g_pGnutellaOptions->m_bKillStalledUploadTransfers;
	m_uStallTransferRate    = g_pGnutellaOptions->m_uStallTransferRate;
	m_uStallTimeoutInMSecs  = g_pGnutellaOptions->m_uStallTimeoutInMSecs;

	g_pGnutellaOptionsMutex->unlock();

	KviStr tmp;

	if(m_transferDirection == Recv)
	{
		// check the file state
		if(!lockIncomingFile())
		{
			m_szStateChangeReason = __tr_no_lookup("Can't lock the download file (Another download in progress ?)");
			goto time_to_die_gracefully;
		}
	}

	// connection
	if(m_connectionType == Active)
	{
		setState(Connecting);

		// have to connect to a machine
		if(!connectToRemoteHost())
		{
			// failed to connect to the remote host
			// notify the event and wait for push
			if(m_bTerminateRequestSeen)goto time_to_die_gracefully;

			if(m_transferDirection == Recv)
			{
				postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
					new KviStr(KviStr::Format,__tr_no_lookup("[transfer %d]: Failed to connect to the remote host, attempting a PUSH"),
					m_uId)));
			} else {
				// this is a send: we're serving a PUSH request , so we have failed
				postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
					new KviStr(KviStr::Format,__tr_no_lookup("[transfer %d]: Failed to connect to the remote host, PUSH failed"),
					m_uId)));
				goto time_to_die_gracefully;
			}

			setState(WaitingForPush);

			if(!pushAttempt())goto time_to_die_gracefully;
			// else connected by PUSH (ForcedPassive)

		} // else connected (Active)
	} // else Passive: already connected

	// handshake

	setState(Handshaking);

	// ok... active , get sends a HTTP GET, then expects a HTTP OK
	//       active , put sends a GIV, then expects a HTTP GET and then sends a HTTP OK
	//       passive, get expects a GIV, sends a HTTP GET and expects a HTTP OK
	//       passive  put expects a HTTP GET and sends a HTTP OK

	if(m_connectionType == Active)
	{
		// active
		if((m_transferDirection == Recv))
		{
			// we're GETTING a file from someone
			if(!sendHttpGet())goto time_to_die_gracefully;
			if(!expectHttpOk())goto time_to_die_gracefully;
		} else {
			// we're pushing a file to someone
			if(!sendHttpGiv())goto time_to_die_gracefully;
			if(!incomingHandshake())goto time_to_die_gracefully;
			goto time_to_die_gracefully;
		}
	} else {
		// passive:
		if((m_transferDirection == Recv))
		{
			// someone pushes a file to us
			if(!expectHttpGiv())goto time_to_die_gracefully;
			if(!sendHttpGet())goto time_to_die_gracefully;
			if(!expectHttpOk())goto time_to_die_gracefully;
		} else {
			// we're prolly required to send a file to someone
			if(!incomingHandshake())goto time_to_die_gracefully;
		}

	}

	setState(Transferring);

	if(m_transferDirection == Recv)
	{
		if(!receiveFile())goto time_to_die_gracefully;
	} else {
		if(!sendFile())goto time_to_die_gracefully;
	}

	postEvent(m_pTab,new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
		new KviStr(KviStr::Format,__tr_no_lookup("[transfer %u]: Succesfully terminated"),m_uId)));

time_to_die_gracefully:

	if(m_transferDirection == Recv)unlockIncomingFile();

	setState(Dead);

	if(m_retryHint != NoRetry)
	{
		postRetryEvent(m_retryHint == RetrySameHost,m_uRetryDelayInSecs);
	}
}



#include "m_gnutellatransfer.moc"
