#ifndef _GNUTELLATHREAD_H_
#define _GNUTELLATHREAD_H_
//
//   File : gnutellathread.h
//   Creation date : Mon Apr 16 2001 17:52:00 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the gnutellas of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_thread.h"
#include "kvi_string.h"
#include "kvi_settings.h"

#include "gnutellanode.h"
#include "gnutellaproto.h"

#include "kvi_list.h"

#include "kvi_time.h"
#include "kvi_socket.h"

#define KVI_GNUTELLA_SERVER_NAME "KVIrc-Gnutella-module-1.0"

// Thread -> Window events
#define KVI_GNUTELLA_THREAD_EVENT_NEW_NODE              (KVI_THREAD_USER_EVENT_BASE + 100)
#define KVI_GNUTELLA_THREAD_EVENT_NODE_DEAD             (KVI_THREAD_USER_EVENT_BASE + 101)
#define KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTING_V4    (KVI_THREAD_USER_EVENT_BASE + 102)
#define KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTING_V6    (KVI_THREAD_USER_EVENT_BASE + 103)
#define KVI_GNUTELLA_THREAD_EVENT_NODE_HANDSHAKE        (KVI_THREAD_USER_EVENT_BASE + 104)
#define KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTED_V4     (KVI_THREAD_USER_EVENT_BASE + 105)
#define KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTED_V6     (KVI_THREAD_USER_EVENT_BASE + 106)

#define KVI_GNUTELLA_THREAD_EVENT_NODES_CAUGHT       (KVI_THREAD_USER_EVENT_BASE + 107)
#define KVI_GNUTELLA_THREAD_EVENT_QUERY_HIT          (KVI_THREAD_USER_EVENT_BASE + 108)
#define KVI_GNUTELLA_THREAD_EVENT_NODE_STATS         (KVI_THREAD_USER_EVENT_BASE + 109)
#define KVI_GNUTELLA_THREAD_EVENT_INCOMING_TRANSFER  (KVI_THREAD_USER_EVENT_BASE + 110)
#define KVI_GNUTELLA_THREAD_EVENT_PUSH_REQUEST       (KVI_THREAD_USER_EVENT_BASE + 111)
#define KVI_GNUTELLA_THREAD_EVENT_BANDWIDTH_STATS    (KVI_THREAD_USER_EVENT_BASE + 112)
#define KVI_GNUTELLA_THREAD_EVENT_PUSH_FAILURE       (KVI_THREAD_USER_EVENT_BASE + 113)

#define KVI_GNUTELLA_THREAD_EVENT_COMPOSITE          (KVI_THREAD_USER_EVENT_BASE + 150)

// Window -> Thread events
#define KVI_GNUTELLA_WINDOW_EVENT_CONNECT_TO_NODE    (KVI_THREAD_USER_EVENT_BASE + 200)
#define KVI_GNUTELLA_WINDOW_EVENT_KILL_NODE          (KVI_THREAD_USER_EVENT_BASE + 201)
#define KVI_GNUTELLA_WINDOW_EVENT_DO_SEARCH          (KVI_THREAD_USER_EVENT_BASE + 202)
#define KVI_GNUTELLA_WINDOW_EVENT_UPDATE_OPTIONS     (KVI_THREAD_USER_EVENT_BASE + 203)
#define KVI_GNUTELLA_WINDOW_EVENT_NODE_STATS         (KVI_THREAD_USER_EVENT_BASE + 204)
#define KVI_GNUTELLA_WINDOW_EVENT_KILL_NON_CONNECTED (KVI_THREAD_USER_EVENT_BASE + 205)
//#define KVI_GNUTELLA_WINDOW_EVENT_CACHE_HOST         (KVI_THREAD_USER_EVENT_BASE + 206)

// A huge upper bound...16 KB of results is a lot of them :)
#define KVI_GNUTELLA_MAX_SEARCH_RESULTS_LEN 16384
#define KVI_GNUTELLA_MAX_SEARCH_RESULTS 40
#define KVI_GNUTELLA_MAX_PENDING_SEARCH_QUERIES 30

class KviGnutellaThread;
class KviGnutellaSearchThread;


typedef struct _KviGnutellaHit
{
	unsigned int uSize; // size in bytes
	unsigned int uIndex; // file index;
	KviStr       szName;
	KviStr       szExtraInfo;
} KviGnutellaHit;


class KviGnutellaHitThreadEvent : public KviThreadEvent
{
public:
	KviGnutellaHitThreadEvent();
	~KviGnutellaHitThreadEvent();
public:
	unsigned char           m_servId[16];
	unsigned int            m_uSpeed;
	KviStr                  m_szIp;
	unsigned short int      m_uPort;
	KviPtrList<KviGnutellaHit> * m_pHitList;
	int                     m_iNodeId;
};

typedef struct _KviGnutellaNodeInfo
{
	KviStr             szIp;
	KviStr             szPort;
	KviStr             szHops;
} KviGnutellaNodeInfo;

class KviGnutellaNodesCaughtThreadEvent : public KviThreadEvent
{
public:
	KviGnutellaNodesCaughtThreadEvent();
	~KviGnutellaNodesCaughtThreadEvent();
public:
	KviPtrList<KviGnutellaNodeInfo> * m_pNodeList;
};

class KviGnutellaNodeStatsThreadEvent : public KviThreadEvent
{
public:
	KviGnutellaNodeStatsThreadEvent() : KviThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NODE_STATS){};
	~KviGnutellaNodeStatsThreadEvent(){};
public:
	int                  m_iNodeId;
	KviStr               m_szIp;
	unsigned short int   m_uPort;
	bool                 m_bIncoming;
	KviStr               m_szPublicHeaders;
	KviStr               m_szProtocol;

//	time_t               m_statsStartTime;
//	int                  m_msecondsStatsStartTime;
	KviGnutellaNodeStats m_stats;
};

class KviGnutellaThreadEvent : public KviThreadEvent
{
public:
	KviGnutellaThreadEvent(int evId)
		:KviThreadEvent(evId){};
	~KviGnutellaThreadEvent(){};
public:
	int                m_iNodeId;

	KviStr             m_szIp;
	unsigned short int m_uPort;
	bool               m_bIncoming;

	KviStr             m_szData;
};

class KviGnutellaCompositeThreadEvent : public KviThreadEvent
{
public:
	KviGnutellaCompositeThreadEvent();
	~KviGnutellaCompositeThreadEvent();
public:
	KviPtrList<KviThreadEvent> * m_pEventList;
};

#define KVI_GNUTELLA_MAX_CACHED_HOSTS 128

typedef struct _KviGnutellaHostInfo
{
	KviStr             szIp;
	unsigned short int uPort;
	unsigned int       uHops;
} KviGnutellaHostInfo;

typedef KviGnutellaHostInfo KviGnutellaCachedHost;

typedef struct _KviGnutellaIncomingTransferInfo
{
	KviStr             szIp;
	unsigned short int uPort;
	kvi_socket_t       iFd;
} KviGnutellaIncomingTransferInfo;

typedef struct _KviGnutellaPushFailureInfo
{
	KviStr             szErrorString;
	unsigned int       uTransferId;
} KviGnutellaPushFailureInfo;

typedef struct _KviGnutellaPushRequestInfo
{
	KviStr              szFileName;
	KviStr              szFilePath;
	unsigned int        uFileSize;
	unsigned int        uFileIndex;
//	unsigned int        uAddress;   // host byte order!
	KviStr              szIp;
	unsigned short int  uPort;
	unsigned int        uTransferId;
	unsigned char       localServentId[16];
} KviGnutellaPushRequestInfo;

class KviGnutellaDescriptorCache;

// declared in gnutellatransfer.h
typedef struct _KviGnutellaTransferPushRequest KviGnutellaTransferPushRequest;

class KviGnutellaThread : public KviSensitiveThread
{
public:
	KviGnutellaThread();
	~KviGnutellaThread();
private:
	unsigned char      m_serventIdentifier[16];                  // my own servent identifier

	// when changing this: increment only the FIRST Q_UINT32
	unsigned char      m_nextDescriptorIdentifier[16];           // identifier for the next descriptor

	kvi_socket_t       m_networkListeningSocket;                 // listening for incoming node connections
	unsigned short int m_uNetworkListenPort;                     // port on that we're listening on

	kvi_socket_t       m_transferListeningSocket;
	unsigned short int m_uTransferListenPort;

	unsigned short int m_uTotalSharedFiles;
	unsigned short int m_uTotalSharedKilobytes;

	unsigned short int m_uConnectionSpeed;

	bool               m_bAcceptConnections;
	bool               m_bAutoConnectToReachMinConnections;
	unsigned int       m_uMaxConnections;                         // max connections OPTION
	unsigned int       m_uCurrentMaxConnections;                  // calculated from uMaxConnections and current downloads

	unsigned int       m_uMinConnections;
	unsigned int       m_uNonDeadNodes;
	KviStr             m_szIp;

	unsigned int       m_uConnectTimeoutInMSecs;
	unsigned int       m_uHandshakeTimeoutInMSecs;
	unsigned short int m_uDefaultTtl;

	bool               m_bSpyLocalSearchResults;

	unsigned short int m_uMaxNetworkConnectionsToDropPerDownload;
	bool               m_bDropNetworkConnectionsWhenDownloading;

	bool               m_bTryGnutella04After06Failure;

	KviPtrList<KviGnutellaNode> * m_pConnectedNodes;

	KviGnutellaDescriptorCache * m_pRoutedPingCache;
	KviGnutellaDescriptorCache * m_pRoutedQueryCache;
	KviGnutellaDescriptorCache * m_pRoutedPushCache;
	KviGnutellaDescriptorCache * m_pQueryHitServentsCache;
	KviGnutellaDescriptorCache * m_pOwnQueryCache;
	KviGnutellaDescriptorCache * m_pOwnPingCache;

	KviPtrList<KviGnutellaCachedHost>       * m_pCachedHosts;

	KviGnutellaNodesCaughtThreadEvent  * m_pNodesCaughtEvent;
	KviGnutellaCompositeThreadEvent    * m_pCompositeEvent;

	KviGnutellaSearchThread            * m_pSearchThread;

	struct timeval                       m_lastBandwidthCalcTime; // the last time the bandwidth was calculated
	struct timeval                       m_currentTime;


private:
	void init();
	void setupNetworkListeningSocket();
	void shutdownNetworkListeningSocket();
	void setupTransferListeningSocket();
	void shutdownTransferListeningSocket();
	void updateOptions();
	void cleanup();

	void updateOptionsRequest();

	bool handleInternalEvents(); // returns false if we have to stop
	void handleNetworkEvents();  // same as above

	void processSearchResults();
	void calculateCurrentMaxConnections();

	void selectStep();
	void serveConnectedNodes();

	void nodeStats(int iNodeId);

	void sendPing(KviGnutellaNode *n);
	void doSearch(KviStr & string,unsigned short int uMinSpeed);
	void sendPushRequest(KviGnutellaNode * n,KviGnutellaTransferPushRequest *r);

	void cacheHost(const char * szIp,unsigned short int uPort,unsigned int uHops);

	bool processPing(KviGnutellaDescriptor * d);
	bool processPong(KviGnutellaDescriptor * d);
	bool processQuery(KviGnutellaDescriptor * d);
	bool processQueryHit(KviGnutellaDescriptor * d);
	bool processPush(KviGnutellaDescriptor * d);

	void dropCachedDescriptors(KviGnutellaNode * n);

	int routeToAllButOne(unsigned char * data,int len,KviGnutellaNode * pExcludeNode);
	int sendToAll(unsigned char * data,int len);
	void buildDescriptor(unsigned char * buffer,Q_UINT8 uPayloadDescriptor,Q_UINT8 uTtl,
			Q_UINT32 uPayloadLength,unsigned char * descriptorId = 0);

	void processHandshaking(KviPtrList<KviGnutellaNode> * pHandshakingNodes);
	void processHandshakingOutgoing(KviGnutellaNode *n);
	void processHandshakingIncoming(KviGnutellaNode *n);


	void processConnected(KviPtrList<KviGnutellaNode> * pConnectedNodes);

	KviGnutellaNode * findNonDeadNode(const char * ip,unsigned short int uPort);

	void processIncomingData();
	void flushOutgoingQueues();

	void connectToNode(const char * szIp,unsigned short int uPort,KviGnutellaNode::Protocol p);
	void handleIncomingNetworkConnection();
	void handleIncomingTransferConnection();
	void forceKillNode(KviGnutellaNode * n,const char * reason);
	void killNode(KviGnutellaNode * n);
	void killDeadNodes();

	KviGnutellaThreadEvent * buildThreadEvent(int evId,int iNodeId,
		const char *szIp = 0,unsigned short int uPort = 0,bool bIncoming = false,
		const char * szData = 0);

	void getLocalHostAddress(KviGnutellaNode * n);

	KviGnutellaNode * findNode(int id);

	void postCompositeEvent();
	void deferredPostEvent(KviThreadEvent *e);

protected:
	virtual void run();
};

#ifndef _GNUTELLAROUTING_CPP_
	extern bool gnutella_is_routable_ip(unsigned char * ip);
#endif

//#ifndef _GNUTELLATHREAD_CPP_
//	extern void gnutella_get_local_host_address(int fd);
//#endif

#endif //_KVI_GNUTELLATHREAD_H_
