//
//   File : gnutellathread.cpp
//   Creation date : Mon Apr 16 2001 17:53:11 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the gnutellas of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#warning "Timeout for nodes ?"

#define _GNUTELLATHREAD_CPP_

#include "gnutellathread.h"
#include "gnutellawindow.h"
#include "gnutellanode.h"
#include "gnutellashares.h"
#include "gnutelladcache.h"
#include "gnutellaoptions.h"
#include "gnutellatransfer.h"

#include "kvi_error.h"
#include "kvi_netutils.h"
#include "kvi_locale.h"
#include "kvi_malloc.h"
#include "kvi_memmove.h"

#include <stdlib.h>
#include <errno.h>

#include "kvi_socket.h"
#include "kvi_time.h"

#define _GNUTELLADEBUG

// libkvignutella.cpp
extern KviGnutellaOptions * g_pGnutellaOptions;
extern KviMutex           * g_pGnutellaOptionsMutex;

extern KviPtrList<KviGnutellaSearchResult> * g_pGnutellaSearchResultList;
extern KviMutex                       * g_pGnutellaSearchResultMutex;

extern unsigned int                     g_uGnutellaLocalIpAddress;

extern unsigned int                     g_uGnutellaCurrentUploadTransfers;
extern unsigned int                     g_uGnutellaCurrentDownloadTransfers;
extern KviMutex                       * g_pGnutellaTransferCountersMutex;

KviGnutellaHitThreadEvent::KviGnutellaHitThreadEvent()
:KviThreadEvent(KVI_GNUTELLA_THREAD_EVENT_QUERY_HIT)
{
	m_pHitList = new KviPtrList<KviGnutellaHit>;
	m_pHitList->setAutoDelete(true);
}

KviGnutellaHitThreadEvent::~KviGnutellaHitThreadEvent()
{
	delete m_pHitList;
}


KviGnutellaNodesCaughtThreadEvent::KviGnutellaNodesCaughtThreadEvent()
:KviThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NODES_CAUGHT)
{
	m_pNodeList = new KviPtrList<KviGnutellaNodeInfo>;
	m_pNodeList->setAutoDelete(true);
}

KviGnutellaNodesCaughtThreadEvent::~KviGnutellaNodesCaughtThreadEvent()
{
	delete m_pNodeList;
}


KviGnutellaCompositeThreadEvent::KviGnutellaCompositeThreadEvent()
:KviThreadEvent(KVI_GNUTELLA_THREAD_EVENT_COMPOSITE)
{
	m_pEventList = new KviPtrList<KviThreadEvent>;
	m_pEventList->setAutoDelete(true);
}

KviGnutellaCompositeThreadEvent::~KviGnutellaCompositeThreadEvent()
{
	delete m_pEventList;
}


KviGnutellaThread::KviGnutellaThread()
: KviSensitiveThread()
{
}

KviGnutellaThread::~KviGnutellaThread()
{
//	debug("KviGnutellaThread::~KviGnutellaThread()");
}

// ----------------------------------------------------------------------------------------
// Init & cleanup
//

void KviGnutellaThread::init()
{
	srand(time(0));

	for(int i=0;i<16;i++)
	{
		m_serventIdentifier[i]        = rand() % 256;
		m_nextDescriptorIdentifier[i] = rand() % 256;
	}

	m_networkListeningSocket = KVI_INVALID_SOCKET;
	m_transferListeningSocket = KVI_INVALID_SOCKET;
	m_uNetworkListenPort     = 0;

	m_pNodesCaughtEvent      = 0;
	m_pCompositeEvent        = 0;

	m_uNonDeadNodes          = 0;

	m_bTryGnutella04After06Failure = false;

	m_pConnectedNodes = new KviPtrList<KviGnutellaNode>;
	m_pConnectedNodes->setAutoDelete(true);
	m_pCachedHosts = new KviPtrList<KviGnutellaCachedHost>;
	m_pCachedHosts->setAutoDelete(true);

	m_pRoutedPingCache       = new KviGnutellaDescriptorCache();
	m_pRoutedQueryCache      = new KviGnutellaDescriptorCache();
	m_pRoutedPushCache       = new KviGnutellaDescriptorCache();
	m_pQueryHitServentsCache = new KviGnutellaDescriptorCache();
	m_pOwnQueryCache         = new KviGnutellaDescriptorCache();
	m_pOwnPingCache          = new KviGnutellaDescriptorCache();

	kvi_gettimeofday(&m_lastBandwidthCalcTime,0);

	updateOptions();

	setupNetworkListeningSocket();
	setupTransferListeningSocket();


	m_pSearchThread = new KviGnutellaSearchThread();
	m_pSearchThread->start();
}

void KviGnutellaThread::updateOptions()
{
	g_pGnutellaOptionsMutex->lock();
	m_bAcceptConnections                = g_pGnutellaOptions->m_bAcceptConnections;
	m_uMaxConnections                   = g_pGnutellaOptions->m_uMaxConnections;
	m_uMinConnections                   = g_pGnutellaOptions->m_uMinConnections;
	m_bAutoConnectToReachMinConnections = g_pGnutellaOptions->m_bAutoConnectToReachMinConnections;
	m_uConnectTimeoutInMSecs            = g_pGnutellaOptions->m_uNetworkConnectTimeoutInMSecs;
	m_uHandshakeTimeoutInMSecs          = g_pGnutellaOptions->m_uNetworkHandshakeTimeoutInMSecs;
	m_uConnectionSpeed                  = g_pGnutellaOptions->m_uConnectionSpeed;
	m_uDefaultTtl                       = g_pGnutellaOptions->m_uDefaultTtl;
	m_bSpyLocalSearchResults            = g_pGnutellaOptions->m_bSpyLocalSearchResults;
	m_uMaxNetworkConnectionsToDropPerDownload = g_pGnutellaOptions->m_uMaxNetworkConnectionsToDropPerDownload;
	m_bDropNetworkConnectionsWhenDownloading  = g_pGnutellaOptions->m_bDropNetworkConnectionsWhenDownloading;
	m_bTryGnutella04After06Failure      = g_pGnutellaOptions->m_bTryGnutella04After06Failure;
	g_pGnutellaOptionsMutex->unlock();
	calculateCurrentMaxConnections();
}

void KviGnutellaThread::updateOptionsRequest()
{
	bool bUpdateListeningSocket = false;
	bool bUpdateTransferSocket = false;
	g_pGnutellaOptionsMutex->lock();
	if(((m_networkListeningSocket != KVI_INVALID_SOCKET) && (!g_pGnutellaOptions->m_bDoListen)) ||
		((m_networkListeningSocket == KVI_INVALID_SOCKET) && (g_pGnutellaOptions->m_bDoListen)))bUpdateListeningSocket = true;
	else if(m_uNetworkListenPort != g_pGnutellaOptions->m_uListenPort)bUpdateListeningSocket = true;

	if(((m_transferListeningSocket != KVI_INVALID_SOCKET) && (!g_pGnutellaOptions->m_bEnableFileSharing)) ||
		((m_transferListeningSocket == KVI_INVALID_SOCKET) && (g_pGnutellaOptions->m_bEnableFileSharing)))bUpdateTransferSocket = true;
	else if(m_uTransferListenPort != g_pGnutellaOptions->m_uTransferListenPort)bUpdateTransferSocket = true;

	g_pGnutellaOptionsMutex->unlock();
	if(bUpdateListeningSocket)
	{
		shutdownNetworkListeningSocket();
		setupNetworkListeningSocket();
	}
	if(bUpdateTransferSocket)
	{
		shutdownTransferListeningSocket();
		setupTransferListeningSocket();
	}
	updateOptions();
}

void KviGnutellaThread::cleanup()
{
	delete m_pSearchThread; // will terminate it (KviSensitiveThread)
	shutdownNetworkListeningSocket();
	shutdownTransferListeningSocket();
	delete m_pConnectedNodes;
	if(m_pNodesCaughtEvent)delete m_pNodesCaughtEvent; // not posted... (huh ?..should never happen)
	if(m_pCompositeEvent)delete m_pCompositeEvent;     // not posted... (huh ?..should never happen)
	delete m_pRoutedPingCache;
	delete m_pRoutedQueryCache;
	delete m_pRoutedPushCache;
	delete m_pOwnQueryCache;
	delete m_pOwnPingCache;
	delete m_pCachedHosts;
	delete m_pQueryHitServentsCache;
}

//-------------------------------------------------------------------------------------------------------------
// Listening sockets
//

void KviGnutellaThread::setupNetworkListeningSocket()
{
	g_pGnutellaOptionsMutex->lock();


	bool bDoListen = g_pGnutellaOptions->m_bDoListen;

	if(bDoListen)
	{
		m_networkListeningSocket = kvi_socket_create(KVI_SOCKET_PF_INET,KVI_SOCKET_TYPE_STREAM,0); // tcp
		if(m_networkListeningSocket != KVI_INVALID_SOCKET)
		{
			// set the non blocking mode
			if(kvi_socket_setNonBlocking(m_networkListeningSocket))
			{
				// bind it to INADDR_ANY and the specified port
				sockaddr_in saddr;
				saddr.sin_family      = AF_INET;
				saddr.sin_port        = htons(g_pGnutellaOptions->m_uListenPort);
				saddr.sin_addr.s_addr = htonl(INADDR_ANY);

				if(kvi_socket_bind(m_networkListeningSocket,(struct sockaddr *)&saddr,sizeof(struct sockaddr)))
				{
					int val = 1;
					// we don't check errors here
					kvi_socket_setsockopt(m_networkListeningSocket,SOL_SOCKET,SO_REUSEADDR,(const void *)&val,sizeof(val));
					// then call listen();
					if(kvi_socket_listen(m_networkListeningSocket,100))
					{
						if(g_pGnutellaOptions->m_uListenPort == 0)
						{
							int slen = sizeof(saddr);
							if(kvi_socket_getsockname(m_networkListeningSocket,(struct sockaddr *)&saddr,&slen))
							{
								m_uNetworkListenPort = ntohs(saddr.sin_port);
							} else m_uNetworkListenPort = 0; // huh ?...can't recover anyway...
						} else m_uNetworkListenPort = g_pGnutellaOptions->m_uListenPort;

						goto socket_setup_complete;
					}
				}
			}
			// failed to set the non blocking mode
			kvi_socket_close(m_networkListeningSocket);
			m_networkListeningSocket = KVI_INVALID_SOCKET;

			//updateHighestDescriptor();
		}
	}
socket_setup_complete:

	g_pGnutellaOptionsMutex->unlock();

	if(bDoListen)
	{
		if(m_networkListeningSocket == KVI_INVALID_SOCKET)
		{
			deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
				new KviStr(__tr_no_lookup("Failed to setup the listening socket: can't accept incoming network connections"))));
		} else {
			deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
				new KviStr(KviStr::Format,__tr_no_lookup("Listening for network connections on port %u"),m_uNetworkListenPort)));
		}
	} else {
		deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
			new KviStr(__tr_no_lookup("Listening for network connections disabled: (user option)"))));
	}
}


void KviGnutellaThread::shutdownNetworkListeningSocket()
{
	if(m_networkListeningSocket != KVI_INVALID_SOCKET)
	{
		kvi_socket_close(m_networkListeningSocket);
		m_networkListeningSocket = KVI_INVALID_SOCKET;
	}
}


void KviGnutellaThread::setupTransferListeningSocket()
{
	g_pGnutellaOptionsMutex->lock();

	bool bEnableFileSharing = g_pGnutellaOptions->m_bEnableFileSharing;

	if(bEnableFileSharing)
	{
	
		m_transferListeningSocket = kvi_socket_create(KVI_SOCKET_PF_INET,KVI_SOCKET_TYPE_STREAM,0); // tcp
		if(m_transferListeningSocket != KVI_INVALID_SOCKET)
		{
			// set the non blocking mode
			if(kvi_socket_setNonBlocking(m_transferListeningSocket))
			{
				// bind it to INADDR_ANY and the specified port
				sockaddr_in saddr;
				saddr.sin_family      = AF_INET;
				saddr.sin_port        = htons(g_pGnutellaOptions->m_uTransferListenPort);
				saddr.sin_addr.s_addr = htonl(INADDR_ANY);
	
//#warning "Allow forcing a specific IP address"
	
				if(kvi_socket_bind(m_transferListeningSocket,(struct sockaddr *)&saddr,sizeof(struct sockaddr)))
				{

					int val = 1;
					// we don't check errors here
					kvi_socket_setsockopt(m_transferListeningSocket,SOL_SOCKET,SO_REUSEADDR,(const void *)&val,sizeof(val));
					// then call listen();
					if(kvi_socket_listen(m_transferListeningSocket,100))
					{
						if(g_pGnutellaOptions->m_uTransferListenPort == 0)
						{
							int slen = sizeof(saddr);
							if(kvi_socket_getsockname(m_transferListeningSocket,(struct sockaddr *)&saddr,&slen))
							{
								m_uTransferListenPort = ntohs(saddr.sin_port);
							} else m_uTransferListenPort = 0; // huh ?...can't recover anyway...
							
						} else m_uTransferListenPort = g_pGnutellaOptions->m_uTransferListenPort;
						goto socket_setup_complete;
					}
				}
			}
			// failed to set the non blocking mode
			kvi_socket_close(m_transferListeningSocket);
			m_transferListeningSocket = KVI_INVALID_SOCKET;
	
			//updateHighestDescriptor();
		}
	}

socket_setup_complete:

	g_pGnutellaOptionsMutex->unlock();

	if(bEnableFileSharing)
	{
		if(m_transferListeningSocket == KVI_INVALID_SOCKET)
		{
			deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
				new KviStr(__tr_no_lookup("Failed to setup the listening socket: can't accept incoming transfer connections"))));
		} else {
			deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
				new KviStr(KviStr::Format,__tr_no_lookup("Listening for transfer connections on port %u"),m_uTransferListenPort)));
		}
	} else {
		deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
			new KviStr(__tr_no_lookup("Listening for transfer connections disabled: (user option)"))));
	}
}

void KviGnutellaThread::shutdownTransferListeningSocket()
{
	if(m_transferListeningSocket != KVI_INVALID_SOCKET)
	{
		kvi_socket_close(m_transferListeningSocket);
		m_transferListeningSocket = KVI_INVALID_SOCKET;
	}
}

//-------------------------------------------------------------------------------------------------------------
// Helpers
//

KviGnutellaThreadEvent * KviGnutellaThread::buildThreadEvent(int evId,int iNodeId,
		const char *szIp,unsigned short int uPort,bool bIncoming,const char * szData)
{
	KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(evId);

	e->m_iNodeId   = iNodeId;
	e->m_szIp      = szIp;
	e->m_uPort     = uPort;
	e->m_bIncoming = bIncoming;
	e->m_szData    = szData;

	return e;
}

void KviGnutellaThread::postCompositeEvent()
{
	if(m_pCompositeEvent)
	{
		postEvent(g_pGnutellaWindow,m_pCompositeEvent);
		m_pCompositeEvent = 0;
		m_pNodesCaughtEvent = 0;
#ifdef _GNUTELLADEBUG
	} else {
		if(m_pNodesCaughtEvent)
		{
			debug("Ops...nodes caught event is not 0...unexpected!");
			int *i = 0;
			*i = 10;
		}
	}
#else
	}
#endif
}

void KviGnutellaThread::deferredPostEvent(KviThreadEvent *e)
{
	if(!m_pCompositeEvent)m_pCompositeEvent = new KviGnutellaCompositeThreadEvent();
	m_pCompositeEvent->m_pEventList->append(e);
}

KviGnutellaNode * KviGnutellaThread::findNode(int id)
{
	for(KviGnutellaNode * n = m_pConnectedNodes->first();n;n = m_pConnectedNodes->next())
	{
		if(n->id() == id)return n;
	}
	return 0;
}

//#warning "If our address is NOT ROUTABLE: it should be updated once in a while"

void KviGnutellaThread::getLocalHostAddress(KviGnutellaNode * n)
{
	struct sockaddr_in name;
	int len = sizeof(name);
	if(n->sock() < 0)return;

	if(!kvi_socket_getsockname(n->sock(), (struct sockaddr *)&name,&len))
	{
		kvi_stringIpToBinaryIp("127.0.0.1",&(name.sin_addr));
	}
//#warning "Allow forcing a specific IP address"
	g_uGnutellaLocalIpAddress = ntohl(name.sin_addr.s_addr);

	kvi_binaryIpToStringIp(name.sin_addr,m_szIp);
}

KviGnutellaNode * KviGnutellaThread::findNonDeadNode(const char * ip,unsigned short int uPort)
{
	for(KviGnutellaNode * node = m_pConnectedNodes->first();node;node = m_pConnectedNodes->next())
	{
		if(!node->isDead())
		{
			if(node->m_uPort == uPort)
			{
				if(kvi_strEqualCS(node->m_szIp.ptr(),ip))return node;
			}
		}
	}
	return 0;
}

void KviGnutellaThread::killNode(KviGnutellaNode * n)
{
	deferredPostEvent(buildThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NODE_DEAD,n->id(),0,0,false,n->deathReason()));
	dropCachedDescriptors(n);
	m_pConnectedNodes->removeRef(n);
}

void KviGnutellaThread::forceKillNode(KviGnutellaNode * n,const char * reason)
{
	n->die(reason);
	killNode(n);
}

void KviGnutellaThread::killDeadNodes()
{
	KviPtrList<KviGnutellaNode> l;
	l.setAutoDelete(false);

	KviGnutellaNode * n;

	for(n = m_pConnectedNodes->first();n;n = m_pConnectedNodes->next())
	{
		if(n->isDead())l.append(n);
	}

	for(n = l.first();n;n = l.next())
	{
		killNode(n);
	}

	//updateHighestDescriptor();
}


void KviGnutellaThread::cacheHost(const char * szIp,unsigned short int uPort,unsigned int uHops)
{
	KviGnutellaCachedHost * h = new KviGnutellaCachedHost;
	h->szIp  = szIp;
	h->uPort = uPort;
	h->uHops = uHops;

	while(m_pCachedHosts->count() >= KVI_GNUTELLA_MAX_CACHED_HOSTS)m_pCachedHosts->removeFirst();

	m_pCachedHosts->append(h);
}

//-------------------------------------------------------------------------------------------------------------
// Window -> thread events
//

void KviGnutellaThread::connectToNode(const char * szIp,unsigned short int uPort,KviGnutellaNode::Protocol p)
{
	if(findNonDeadNode(szIp,uPort))
	{
		deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
			new KviStr(KviStr::Format,
			__tr_no_lookup("Can't connect to node %s:%u: another connection to this node is already in progress"),
			szIp,uPort)));
		return;
	}

	if(kvi_strEqualCS(szIp,m_szIp.ptr()) && (m_uNetworkListenPort == uPort))
	{
		deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
			new KviStr(KviStr::Format,
			__tr_no_lookup("Can't connect to node %s:%u: it is my own address"),
			szIp,uPort)));
		return;
	}

	KviGnutellaNode * n = new KviGnutellaNode(szIp,uPort,KviGnutellaNode::Outgoing,p);

	deferredPostEvent(buildThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NEW_NODE,n->id(),szIp,uPort,false));

	if(n->connect())
	{
		m_pConnectedNodes->append(n);
		n->startOperation();
		deferredPostEvent(buildThreadEvent(
			(p == KviGnutellaNode::Gnutella04) ? KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTING_V4 : KVI_GNUTELLA_THREAD_EVENT_NODE_CONNECTING_V6,
			n->id()));
		//updateHighestDescriptor();
	} else {
		deferredPostEvent(buildThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NODE_DEAD,n->id(),0,0,false,n->deathReason()));
		delete n;
	}
}

void KviGnutellaThread::nodeStats(int nodeId)
{
	KviGnutellaNode * n = findNode(nodeId);
	if(n)
	{
		KviGnutellaNodeStatsThreadEvent * e = new KviGnutellaNodeStatsThreadEvent();
		e->m_iNodeId        = n->id();
		e->m_szIp           = n->ip();
		e->m_uPort          = n->port();
		e->m_bIncoming      = (n->connectionType() == KviGnutellaNode::Incoming);
		switch(n->protocol())
		{
			case KviGnutellaNode::Gnutella04:
				e->m_szProtocol = "GNUTELLA/0.4";
			break;
			case KviGnutellaNode::Gnutella06:
				e->m_szProtocol = "GNUTELLA/0.6";
			break;
			default:
				e->m_szProtocol = "Unknown";
			break;
		}
		e->m_szPublicHeaders= n->publicHeaders();
//		e->m_msecsStatsStartTime = n->msecondsSinceStartOperation();
		kvi_memmove(&(e->m_stats),&(n->m_stats),sizeof(KviGnutellaNodeStats));
		deferredPostEvent(e);
		return;
	}
	deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,new KviStr(KviStr::Format,__tr_no_lookup("Can't find node %d"),nodeId)));
}

bool KviGnutellaThread::handleInternalEvents()
{
	while(KviThreadEvent *e = dequeueEvent())
	{
//		debug("KviGnutellaThread : have event id %d",e->id());
		switch(e->id())
		{
			case KVI_THREAD_EVENT_TERMINATE:
//				debug("KviGnutellaThread : received TERMINATE EVENT!");
				delete e;
				return false;
			break;
			case KVI_GNUTELLA_WINDOW_EVENT_KILL_NODE:
			{
				KviGnutellaNode * n = findNode(((KviGnutellaThreadEvent *)e)->m_iNodeId);
				if(n)forceKillNode(n,__tr_no_lookup("User kill request"));
				delete (KviGnutellaThreadEvent *)e;
			}
			break;
			case KVI_GNUTELLA_WINDOW_EVENT_KILL_NON_CONNECTED:
			{
				delete e;
				KviPtrList<KviGnutellaNode> l;
				l.setAutoDelete(false);
				KviGnutellaNode * n;
				for(n = m_pConnectedNodes->first();n;n = m_pConnectedNodes->next())
				{
					if(!(n->isConnected() || n->isHandshaking()))l.append(n);
				}
				for(n = l.first();n;n = l.next())
				{
					forceKillNode(n,__tr_no_lookup("User kill request"));
				}
			}
			break;
			case KVI_GNUTELLA_WINDOW_EVENT_CONNECT_TO_NODE:
			{
				connectToNode(((KviGnutellaThreadEvent *)e)->m_szIp.ptr(),((KviGnutellaThreadEvent *)e)->m_uPort,KviGnutellaNode::Gnutella06);
				delete (KviGnutellaThreadEvent *)e;
			}
			break;
			case KVI_GNUTELLA_WINDOW_EVENT_DO_SEARCH:
			{
				doSearch(((KviGnutellaThreadEvent *)e)->m_szData,((KviGnutellaThreadEvent *)e)->m_uPort);
				delete (KviGnutellaThreadEvent *)e;
			}
			break;
			case KVI_GNUTELLA_WINDOW_EVENT_UPDATE_OPTIONS:
				updateOptionsRequest();
				delete (KviGnutellaThreadEvent *)e;
			break;
			case KVI_GNUTELLA_WINDOW_EVENT_NODE_STATS:
				nodeStats(((KviGnutellaThreadEvent *)e)->m_iNodeId);
				delete (KviGnutellaThreadEvent *)e;
			break;
			case KVI_GNUTELLA_TRANSFER_THREAD_EVENT_PUSH_REQUEST:
			{
				KviGnutellaTransferPushRequest * trr = ((KviThreadDataEvent<KviGnutellaTransferPushRequest> *)e)->getData();
				KviGnutellaNode * n = findNode(trr->iNodeId);
//				debug("Push request");
				KviStr szError;
				if(n)
				{
					if(!n->isDead())
					{
//						debug("Push request");
						sendPushRequest(n,trr);
//						debug("Push request sent");
					}
					else szError = __tr_no_lookup("Can't start the PUSH request: the source node is dead");
				} else szError = __tr_no_lookup("Can't start the PUSH request, the source node is no longer existing");
				if(szError.hasData())
				{
//					debug("push failure");
					KviThreadDataEvent<KviGnutellaPushFailureInfo> * evb = new KviThreadDataEvent<KviGnutellaPushFailureInfo>(KVI_GNUTELLA_THREAD_EVENT_PUSH_FAILURE);
					KviGnutellaPushFailureInfo * pfi = new KviGnutellaPushFailureInfo;
					pfi->uTransferId = trr->uTransferId;
					pfi->szErrorString = szError;
					evb->setData(pfi);
					deferredPostEvent(evb);
				}
				delete trr;
				delete (KviThreadDataEvent<KviGnutellaTransferPushRequest> *)e;
			}
			break;
			default:
				debug("[gnutella thread] Unexpected event %d",e->type());
				// unexpected event ?
				delete e;
			break;
		}
	}
	return true;
}

//-------------------------------------------------------------------------------------------------------------
// Socket events
//


void KviGnutellaThread::handleIncomingTransferConnection()
{
	struct sockaddr_in saddr;
	int addrLen = sizeof(saddr);

	int fd = kvi_socket_accept(m_transferListeningSocket,(struct sockaddr *)&saddr,&addrLen);

	if(fd == KVI_INVALID_SOCKET)return; // accept returned error ? mmmh...
	
	KviStr szIp;

	if((!addrLen) || (!kvi_binaryIpToStringIp(saddr.sin_addr,szIp)))
	{
		// Invalid ip
		kvi_socket_close(fd);
		return; // huh ?
	}

	unsigned short int uPort = ntohs(saddr.sin_port);

	KviThreadDataEvent<KviGnutellaIncomingTransferInfo> * ev = new KviThreadDataEvent<KviGnutellaIncomingTransferInfo>(KVI_GNUTELLA_THREAD_EVENT_INCOMING_TRANSFER);

	KviGnutellaIncomingTransferInfo * hap = new KviGnutellaIncomingTransferInfo;
	hap->uPort = uPort;
	hap->szIp  = szIp;
	hap->iFd = fd;

	ev->setData(hap);

	deferredPostEvent(ev);
}

void KviGnutellaThread::handleIncomingNetworkConnection()
{
	struct sockaddr_in saddr;
	int addrLen = sizeof(saddr);

	int fd = kvi_socket_accept(m_networkListeningSocket,(struct sockaddr *)&saddr,&addrLen);

	if(fd == KVI_INVALID_SOCKET)return; // accept returned error ? mmmh...


	KviStr szIp;

	if((!addrLen) || (!kvi_binaryIpToStringIp(saddr.sin_addr,szIp)))
	{
		// Invalid ip
		kvi_socket_close(fd);
		return; // huh ?
	}

	unsigned short int uPort = ntohs(saddr.sin_port);

	if(!m_bAcceptConnections)
	{
		// refuse it : user option
		deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
				new KviStr(KviStr::Format,
			__tr_no_lookup("Rejecting connection from node %s:%u: not accepting connections (user option)"),
			szIp.ptr(),uPort)));
		kvi_socket_close(fd);
		return;
	}

	calculateCurrentMaxConnections();


	if(m_uNonDeadNodes >= m_uCurrentMaxConnections)
	{
		// refuse it : too many connections
		deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
			new KviStr(KviStr::Format,
			__tr_no_lookup("Rejecting connection from node %s:%u: connection limit reached"),
			szIp.ptr(),uPort)));
		kvi_socket_close(fd);
		return;
	}

	if(findNonDeadNode(szIp.ptr(),uPort))
	{
		deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
			new KviStr(KviStr::Format,
			__tr_no_lookup("Rejecting connection from node %s:%u: another connection to this node is already in progress"),
			szIp.ptr(),uPort)));
		kvi_socket_close(fd);
		return;
	}

	KviGnutellaNode * n = new KviGnutellaNode(szIp.ptr(),uPort,KviGnutellaNode::Incoming,KviGnutellaNode::Unknown);
	n->setConnectedFd(fd);

	if(g_uGnutellaLocalIpAddress == 0)getLocalHostAddress(n);

	m_uNonDeadNodes++;

	m_pConnectedNodes->append(n);

	deferredPostEvent(buildThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NEW_NODE,n->id(),szIp,uPort,true));

	//updateHighestDescriptor();
}

void KviGnutellaThread::calculateCurrentMaxConnections()
{
	unsigned int uConnectionsToDrop;
	g_pGnutellaTransferCountersMutex->lock();
	uConnectionsToDrop = g_uGnutellaCurrentDownloadTransfers * m_uMaxNetworkConnectionsToDropPerDownload;
	g_pGnutellaTransferCountersMutex->unlock();
	if(uConnectionsToDrop > m_uMaxConnections)uConnectionsToDrop = m_uMaxConnections;
	m_uCurrentMaxConnections = m_uMaxConnections - uConnectionsToDrop;
	if(m_uMinConnections > m_uMaxConnections)m_uMinConnections = m_uMaxConnections;
}

void KviGnutellaThread::selectStep()
{
	//
	// Main select step
	// here we check for network activity, accept incoming connections
	// and read data from neighbours
	//

	fd_set readSet;
	fd_set writeSet;

	FD_ZERO(&readSet);
	FD_ZERO(&writeSet);

	struct timeval tmv;
	tmv.tv_sec  = 0;
	tmv.tv_usec = 50000; // we wait 50 msecs for an event

	int nRead  = 0;
	int nWrite = 0;

	// get the current time (we will need it in many places)
	kvi_gettimeofday(&m_currentTime,0);

	// do we need to calculate bandwidth of the nodes ? (we do it every 10 seconds)
	int iBandwidthCalcInterval = ((m_currentTime.tv_sec - m_lastBandwidthCalcTime.tv_sec) * 1000) + ((m_currentTime.tv_usec - m_lastBandwidthCalcTime.tv_usec) / 1000);
	bool bCalcBandwidth = iBandwidthCalcInterval > 10000;
	int iNetworkSendBandwidth;
	int iNetworkRecvBandwidth;
	if(bCalcBandwidth)
	{
		m_lastBandwidthCalcTime.tv_sec = m_currentTime.tv_sec;
		m_lastBandwidthCalcTime.tv_usec = m_currentTime.tv_usec;
		iNetworkSendBandwidth = 0;
		iNetworkRecvBandwidth = 0;
	}

	// We have to calculate the select first parameter by the way
	int iHighestDescriptor = m_networkListeningSocket;

	// By the way we will also calculate the non dead nodes count
	unsigned int uNonDeadNodes = 0;

	// setup the read and write fd_sets

	// we will look for incoming network connections
	if(m_networkListeningSocket != KVI_INVALID_SOCKET)
	{
		FD_SET(m_networkListeningSocket,&readSet);
		nRead++;
	}

	// and for incoming transfer connections
	if(m_transferListeningSocket != KVI_INVALID_SOCKET)
	{
		FD_SET(m_transferListeningSocket,&readSet);
		nRead++;
		if(iHighestDescriptor < m_transferListeningSocket)iHighestDescriptor = m_transferListeningSocket;
	}


	// now we fill in the node descriptors
	for(KviGnutellaNode * n = m_pConnectedNodes->first();n;n = m_pConnectedNodes->next())
	{
		switch(n->state())
		{
			case KviGnutellaNode::Connecting:
				// Connecting node...
				// this one counts for non dead nodes
				// and has to be selected for writing
				uNonDeadNodes++;
				FD_SET(n->sock(),&writeSet);
				nWrite++;
				if(n->sock() > iHighestDescriptor)iHighestDescriptor = n->sock();
			break;
			case KviGnutellaNode::Connected:
			case KviGnutellaNode::Handshake:
				// handshaking and "on network" nodes
				uNonDeadNodes++;
				FD_SET(n->sock(),&readSet);
				nRead++;
				if(n->sock() > iHighestDescriptor)iHighestDescriptor = n->sock();
				if(bCalcBandwidth)
				{
					KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(KVI_GNUTELLA_THREAD_EVENT_BANDWIDTH_STATS);
					e->m_iNodeId = n->id();
					int iRecvBandwidth = (n->curBandwidthRecvBytes() * 1000) / iBandwidthCalcInterval;
					iNetworkRecvBandwidth += iRecvBandwidth;
					int iSendBandwidth = (n->curBandwidthSentBytes() * 1000) / iBandwidthCalcInterval;
					iNetworkSendBandwidth += iSendBandwidth;
					e->m_szData.sprintf("I: %d O: %d",iRecvBandwidth,iSendBandwidth);
					deferredPostEvent(e);
					n->resetBandwidthStats();
				}
			break;
			default:
				// Just created (HUH ?) , or Dead (HUH again)
				// Anyway, "default:" makes gcc happy
			break;
		}
	}

	// post also the total network bandwidth event
	if(bCalcBandwidth)
	{
		KviGnutellaThreadEvent * e = new KviGnutellaThreadEvent(KVI_GNUTELLA_THREAD_EVENT_BANDWIDTH_STATS);
		e->m_iNodeId = -1;
		e->m_szData.sprintf(__tr_no_lookup("Net bandwidth: I: %d O: %d (bytes/sec)"),iNetworkRecvBandwidth,iNetworkSendBandwidth);
		deferredPostEvent(e);
	}

	// ok...we need to see this value when accepting incoming connections
	m_uNonDeadNodes = uNonDeadNodes;

	// now the big select() call

	int nRet = 0;


	if((nRead > 0) || (nWrite > 0))
		nRet = kvi_socket_select(iHighestDescriptor + 1,nRead > 0 ? &readSet : 0,nWrite > 0 ? &writeSet : 0,0,&tmv);

	if(nRet >= 0)
	{
		// no error : loop thru the nodes...
		// read available data...
		// eventually kill nodes for timeout

		bool bHasDeadNodes = false;

		for(KviGnutellaNode * n = m_pConnectedNodes->first();n;n = m_pConnectedNodes->next())
		{
			switch(n->state())
			{
				case KviGnutellaNode::Connecting:
					if((nRet > 0) && FD_ISSET(n->sock(),&writeSet))
					{
						// connected ?
						nRet--;
						int sockError;
						int iSize=sizeof(sockError);
						if(!kvi_socket_getsockopt(n->sock(),SOL_SOCKET,SO_ERROR,(void *)&sockError,&iSize))sockError = -1;
						if(sockError != 0)
						{
							//failed
							if(sockError > 0)sockError = kvi_errorFromSystemError(sockError);
							else sockError = KviError_unknownError;
							KviStr tmp(KviStr::Format,__tr_no_lookup("errno = %d (%s)"),sockError,
								kvi_getErrorString(sockError));
							n->die(tmp.ptr());
							bHasDeadNodes = true;
						} else {
							n->youAreHandshaking();
							if(g_uGnutellaLocalIpAddress == 0)getLocalHostAddress(n);
							deferredPostEvent(buildThreadEvent(KVI_GNUTELLA_THREAD_EVENT_NODE_HANDSHAKE,n->id()));
							if(n->protocol() == KviGnutellaNode::Gnutella06)
							{
								n->appendOutgoingData((unsigned char *)"GNUTELLA CONNECT/0.6\r\n",22);
								KviStr uagent(KviStr::Format,"User-Agent: %s\r\n\r\n",KVI_GNUTELLA_SERVER_NAME);
								n->appendOutgoingData((unsigned char *)(uagent.ptr()),uagent.len());
							} else {
								n->appendOutgoingData((unsigned char *)"GNUTELLA CONNECT/0.4\n\n",22);
							}
						}
					} else {
						if(n->msecondsSinceOperationStart(&m_currentTime) > ((int)m_uConnectTimeoutInMSecs))
						{
//#warning "Removeme"
//							debug("nSecs (connect-dead): %d",n->msecondsSinceOperationStart(&m_currentTime));
							n->die(__tr_no_lookup("Connect timeout"));
							bHasDeadNodes = true;
						}//else debug("nSecs (connect-alive): %d",n->msecondsSinceOperationStart(&m_currentTime));

					}
				break;
				case KviGnutellaNode::Handshake:
				case KviGnutellaNode::Connected:
					if((nRet > 0) && FD_ISSET(n->sock(),&readSet))
					{
						nRet--;
						if(!(n->readData()))bHasDeadNodes = true;
					}
				break;
				case KviGnutellaNode::Dead:
					bHasDeadNodes = true; // should never happen
				break;
				default:
					// Make the compiler happy
				break;
			}
		}

		if(nRet > 0)
		{
			// Ok we still have some stuff to read...
			if(m_networkListeningSocket != KVI_INVALID_SOCKET)
			{
				if(FD_ISSET(m_networkListeningSocket,&readSet))
				{
					handleIncomingNetworkConnection();
					nRet--;
				}
			}
	
			if((nRet > 0) && (m_transferListeningSocket != KVI_INVALID_SOCKET))
			{
				if(FD_ISSET(m_transferListeningSocket,&readSet))
				{
					handleIncomingTransferConnection();
					nRet--;
				}
			}
		} else {
			// ok...nRet was 0...nothing special is happening here...
			// spend a while to do some things that have minor priority

			// once in a while, recalculate the max allowable connections
			if(bCalcBandwidth)
			{
				calculateCurrentMaxConnections();

				if(m_bDropNetworkConnectionsWhenDownloading)
				{
					if(m_uNonDeadNodes > m_uCurrentMaxConnections)
					{
						// mmmh... we're over max connections actually
						// we need to drop some network connections...
						// now a question arises: which nodes to drop ?
						//    the high traffic ones ?
						//    the low traffc ones ?
						//    the ones with a lot of queries and almost no results ?
						//    the ones with less pong packets ?
						//    random ???
						//
						// actually we'll go for almost random nodes 
						// since this is the easiest strategy to implement :) (FIXME: choose a better strategy)
						// we'll prefer handshaking nodes first
						KviGnutellaNode * n;

						unsigned int uNodesToDrop = m_uNonDeadNodes - m_uCurrentMaxConnections;
						for(n = m_pConnectedNodes->first();n && (uNodesToDrop > 0);n = m_pConnectedNodes->next())
						{
							if(n->state() == KviGnutellaNode::Handshake)
							{
								n->die(__tr_no_lookup("Too many connections: freeing bandwidth: dropping handshaking node"));
								bHasDeadNodes = true;
								uNodesToDrop--;
							}
						}
						// then if it is not enough...we'll kill also the connected ones
						for(n = m_pConnectedNodes->first();n && (uNodesToDrop > 0);n = m_pConnectedNodes->next())
						{
							if(n->state() == KviGnutellaNode::Connected)
							{
								n->die(__tr_no_lookup("Too many connections: freeing bandwidth: dropping connected node"));
								bHasDeadNodes = true;
								uNodesToDrop--;
							}
						}
					}
				}
			} else {
				// we do this check instead (we do it almost always)

				if((m_uNonDeadNodes < m_uMinConnections) &&
						(m_uNonDeadNodes < m_uCurrentMaxConnections) &&
						m_bAutoConnectToReachMinConnections)
				{
					// attempt a connection to some node...
					// get the newest cached node
					// far nodes would be better than near ones
					// but we don't care in this implementation
					KviGnutellaCachedHost * h = m_pCachedHosts->last();
					if(h)
					{
						connectToNode(h->szIp,h->uPort,KviGnutellaNode::Gnutella06);
						m_pCachedHosts->removeLast(); // will delete it
					}
					usleep(1000);
				} else {
					// really almost nothing to do in this loop...
					usleep(10000); // sleep a little more
				}
			}
		}


		if(bHasDeadNodes)killDeadNodes(); // senseless routing their descriptors

	} else {
		// Error...
		//usleep(1000000);
		int err = kvi_socket_error();
		deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_WARNING,
			new KviStr(KviStr::Format,__tr_no_lookup("System select() error: errno=%d (%s)"),err,
			kvi_getErrorString(kvi_errorFromSystemError(err)))));
	}

	usleep(500); // sleep for a while anyway

}

void KviGnutellaThread::flushOutgoingQueues()
{
	bool bHasDeadNodes = false;

	for(KviGnutellaNode * n = m_pConnectedNodes->first();n;n = m_pConnectedNodes->next())
	{
		if((n->state() == KviGnutellaNode::Connected) || (n->state() == KviGnutellaNode::Handshake))
		{
			if(!(n->flushQueue()))bHasDeadNodes = true;
		} else {
			if(n->state() == KviGnutellaNode::Dead)bHasDeadNodes = true;
		}
	}
	if(bHasDeadNodes)killDeadNodes();
}

void KviGnutellaThread::processSearchResults()
{
	g_pGnutellaSearchResultMutex->lock();

	while(KviGnutellaSearchResult * r = g_pGnutellaSearchResultList->first())
	{
		g_pGnutellaSearchResultList->removeFirst();
		g_pGnutellaSearchResultMutex->unlock();

		if(m_bSpyLocalSearchResults)
		{
			deferredPostEvent(new KviThreadDataEvent<KviStr>(KVI_THREAD_EVENT_MESSAGE,
				new KviStr(KviStr::Format,__tr_no_lookup("Local DB Search: \"%s\" (%u matches)"),r->szQueryString.ptr(),r->uResultsCount)));
		}

		if(r->pResultsBuffer)
		{
			if(r->uResultsLen && r->uResultsCount)
			{
				KviGnutellaNode * n = findNode(r->uSourceNodeId);
				if(n)
				{
					if(n->isConnected() && g_uGnutellaLocalIpAddress)
					{
						unsigned char * buffer = n->allocOutgoingBuffer(GNUTELLA_DESCRIPTOR_SIZE + 27 + r->uResultsLen);
						buildDescriptor(buffer,GNUTELLA_QUERYHIT,5,27 + r->uResultsLen,r->descriptorId);
						*((Q_UINT8  *)(buffer + GNUTELLA_DESCRIPTOR_SIZE))     = r->uResultsCount;
						*((Q_UINT16 *)(buffer + GNUTELLA_DESCRIPTOR_SIZE + 1)) = hToGnutellaS(m_uTransferListenPort);
						*((Q_UINT32 *)(buffer + GNUTELLA_DESCRIPTOR_SIZE + 3)) = hToGnutellaReversedL(g_uGnutellaLocalIpAddress);
						*((Q_UINT32 *)(buffer + GNUTELLA_DESCRIPTOR_SIZE + 7)) = hToGnutellaL(m_uConnectionSpeed);
						kvi_memmove((void *)(buffer + GNUTELLA_DESCRIPTOR_SIZE + 11),r->pResultsBuffer,r->uResultsLen);
						kvi_memmove((void *)(buffer + GNUTELLA_DESCRIPTOR_SIZE + 11 + r->uResultsLen),m_serventIdentifier,16);
					}
				}
			}
			kvi_free((void *)r->pResultsBuffer);
		}

		delete r;

		g_pGnutellaSearchResultMutex->lock();
	}

	g_pGnutellaSearchResultMutex->unlock();
}


void KviGnutellaThread::serveConnectedNodes()
{
	processIncomingData();

	processSearchResults();

	flushOutgoingQueues();
}

void KviGnutellaThread::handleNetworkEvents()
{
	// select m_networkListeningSocket for incoming connections and eventually handle it
	// select the connected sockets for read and eventually read data

	selectStep();

	// process incoming data queue for each node, eventually route packets, flush the outgoing queues


	serveConnectedNodes();

	// If we have caught some new nodes , post the event
}

void KviGnutellaThread::run()
{
	init();


	for(;;)
	{
		if(m_pCompositeEvent)postCompositeEvent();

#ifdef _GNUTELLADEBUG
		if(m_pCompositeEvent != 0)
		{
			debug("Ops...composite event is not 0");
			int * i = 0;
			*i = 10;
		}
#endif

		if(!handleInternalEvents())break;

		handleNetworkEvents();
	}

	if(m_pCompositeEvent)postCompositeEvent();

	cleanup();
}
