//
//   File : gnutellashares.cpp
//   Creation date : Tue Apr 22 2001 16:03:10 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the gnutellas of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _GNUTELLASHARES_CPP_

#include "gnutellashares.h"
#include "gnutellathread.h"

#include "kvi_fileutils.h"
#include "kvi_memmove.h"
#include "kvi_malloc.h"

#include <qfileinfo.h>

#include <ctype.h>

extern KviPtrList<KviGnutellaSharedFile>   * g_pGnutellaSharedFiles;
extern KviMutex                       * g_pGnutellaSharedFilesMutex;
extern unsigned int                     g_iGnutellaNextSharedFileId;
extern unsigned int                     g_uGnutellaSharedFilesCount;
extern unsigned int                     g_uGnutellaSharedBytes;

extern KviPtrList<KviGnutellaSearchQuery>  * g_pGnutellaSearchQueryList;
extern KviMutex                       * g_pGnutellaSearchQueryMutex;
extern KviPtrList<KviGnutellaSearchResult> * g_pGnutellaSearchResultList;
extern KviMutex                       * g_pGnutellaSearchResultMutex;



void gnutella_shared_files_lock()
{
	g_pGnutellaSharedFilesMutex->lock();
}

void gnutella_shared_files_unlock()
{
	g_pGnutellaSharedFilesMutex->unlock();
}

KviGnutellaSharedFile * gnutella_find_shared_file(unsigned int uId)
{
	for(KviGnutellaSharedFile * f = g_pGnutellaSharedFiles->first();f;f = g_pGnutellaSharedFiles->next())
	{
		if(f->uId == uId)return f;
	}
	return 0;
}

//
// uCharFlags trick
//         Q_UINT32 uCharFlags[8] ==>  32 * 8 = 256 bits
//         a bit is set when the corresponding ASCII char is present in the filename
//         for a alphabetic char only lowcase will match...
//         for every char c (8 bits) in the filename:
//                 the leftmost 3 bits are the uCharFlags array index [arrayIndex = ((c & 224) >> 5)]
//                 the rightmost 5 bits give the flags bit index      [bitIndex   = (c & 31)        ]
//         so we will be setting
//                 uCharFlags[arrayIndex] |= (1 << bitIndex)
//

KviGnutellaSharedFile * gnutella_add_shared_file(const char * szFileName)
{
	if(!szFileName)return 0;

	KviStr szFileNameAndPath = szFileName;
	kvi_adjustFilePath(szFileNameAndPath);

	QFileInfo inf(szFileNameAndPath.ptr());
	if(inf.exists() && inf.isFile() && inf.isReadable() && (inf.size() > 0))
	{
		KviGnutellaSharedFile * f = new KviGnutellaSharedFile;

		for(int i=0;i<8;i++)f->uCharFlags[i] = 0;

		f->uFileSize = inf.size();
		f->uId = g_iGnutellaNextSharedFileId;
		f->szFileName = szFileNameAndPath.ptr();
		int idx = f->szFileName.findLastIdx(KVI_PATH_SEPARATOR_CHAR);
		if(idx != -1)
		{
			f->szPath = f->szFileName.left(idx + 1);
			f->szFileName.cutLeft(idx + 1);
		} // else uh ?

		kvi_memset((void *)(f->uCharFlags),0,32);

		f->szLowCaseFileName = f->szFileName;
		f->szLowCaseFileName.toLower();

		unsigned char * aux = (unsigned char *)f->szLowCaseFileName.ptr();
		while(*aux)
		{
//			unsigned char up = toupper(*aux);
			f->uCharFlags[(*aux & 224) >> 5] |= (1 << ((*aux) & 31));
//			if(up != *aux)f->uCharFlags[(up & 224) >> 5] |= (1 << (up & 31));

			aux++;
		}

		g_pGnutellaSharedFiles->append(f);
		g_uGnutellaSharedFilesCount++;
		g_uGnutellaSharedBytes += f->uFileSize;

		g_iGnutellaNextSharedFileId++;

		return f;
	}	

	return 0;
}

void gnutella_remove_shared_file(unsigned int uId)
{
	for(KviGnutellaSharedFile * f = g_pGnutellaSharedFiles->first();f;f = g_pGnutellaSharedFiles->next())
	{
		if(f->uId == uId)
		{
			g_uGnutellaSharedBytes -= f->uFileSize;
			g_uGnutellaSharedFilesCount--;
			g_pGnutellaSharedFiles->removeRef(f);
			return;
		}
	}
}


KviGnutellaSearchThread::KviGnutellaSearchThread()
: KviSensitiveThread()
{
}

KviGnutellaSearchThread::~KviGnutellaSearchThread()
{
}


void KviGnutellaSearchThread::search(KviGnutellaSearchQuery * q)
{
	unsigned char uCount = 0;

	// q->szQueryString has NO whitespace stripped for now
	// and is in MIXED case.
	// we will take care of the whitespace later
	// now just convert to lowcase

	char * aux = q->szQueryString.ptr();

	q->szQueryString.toLower();


	//debug("Search for %s",q->szQueryString.ptr());

	// first of all get the first six (in fact from three up to six) non whitespace character
	// indexes and flags...we will do a quick match

	int iChars = 0;

	unsigned int uIndex[6];
	unsigned int uFlag[6];

	while(*aux && iChars < 6)
	{
		if(*aux != ' ') // only space (searches with newlines or tabs are considered broken)
		{
			uIndex[iChars]      = ((*((unsigned char *)aux)) & 224) >> 5;
			uFlag[iChars]       = 1 << ((*((unsigned char *)aux)) & 31);
			iChars++;
		}
		aux++;
	}

	// if the chars found are less than three, we ignore this search query: bad query string

	if(iChars < 3)return;

	unsigned char * result  = 0;
	unsigned int    uCurLen = 0;
	int             iCharIndex;
	int             theLen;
	const char    * begin;
	char            old;

	g_pGnutellaSharedFilesMutex->lock();

	// now we scan the shared files list
	KviGnutellaSharedFile * f = g_pGnutellaSharedFiles->first();
	while(f)
	{
		// does the filename have all that letters inside ?
		for(iCharIndex = 0;iCharIndex < iChars;iCharIndex++)
		{
			if(!(f->uCharFlags[uIndex[iCharIndex]] & uFlag[iCharIndex]))
			{
				// unmatched character: bail out for this file
				goto match_failed;
			}
		}

		// all the chars matched...we need to make a full search now
		// we're going to split the query string in words

		aux = q->szQueryString.ptr(); // this is LOWCASE actually!
		while(*aux == ' ')aux++; // skip leading white space

		begin = aux; // word begin

		// we're sure that *begin is not '\0' , just because we
		// have at least 3 non space characters in here

		do
		{
			// skip this word
			while(*aux && (*aux != ' '))aux++;

			old = *aux; // remember if this is a space or end of string

			*aux = 0; // fake the end of the string at the word boundary

			// now lookup the word in the filename
			if(!f->szLowCaseFileName.contains(begin,true)) // case sensitive search !
			{
				// no way
				*aux = old;
				goto match_failed;
			}
			*aux = old;
			// skip next whitespace
			while(*aux == ' ')aux++;

			begin = aux; // new word begin (or end of query string)

		} while(*aux);

		// ok...we have a match

		theLen = f->szFileName.len();
//#warning "These limits should be user definable"
		if((theLen + uCurLen + 10) > KVI_GNUTELLA_MAX_SEARCH_RESULTS_LEN)goto terminate_search;
		result = (unsigned char *)kvi_realloc(result,uCurLen + 10 + theLen);
		*((Q_UINT32 *)(result + uCurLen)) = hToGnutellaL(f->uId);
		*((Q_UINT32 *)(result + uCurLen + 4)) = hToGnutellaL(f->uFileSize);
		kvi_memmove((void *)(result + uCurLen + 8),f->szFileName.ptr(),theLen + 1);
		uCurLen += 10 + theLen;
		*(result + uCurLen - 1) = 0;
		uCount++;
		if(uCount == KVI_GNUTELLA_MAX_SEARCH_RESULTS)goto terminate_search;

match_failed:
		f = g_pGnutellaSharedFiles->next();
	}

terminate_search:

	g_pGnutellaSharedFilesMutex->unlock();

	if(result)
	{
		KviGnutellaSearchResult * r = new KviGnutellaSearchResult;
		r->szQueryString  = q->szQueryString;
		r->pResultsBuffer = result;
		r->uResultsLen    = uCurLen;
		r->uResultsCount  = uCount;
		r->uSourceNodeId  = q->uSourceNodeId;
		kvi_memmove((void *)r->descriptorId,(void *)q->descriptorId,16);
		g_pGnutellaSearchResultMutex->lock();
		g_pGnutellaSearchResultList->append(r);
		g_pGnutellaSearchResultMutex->unlock();
	}

/*
#warning "We need a beeter query match thingie: separate tokens by spaces!"

	unsigned char * aux = (unsigned char *)q->szQueryString.ptr();


	int uIndex1          = (aux[0] & 224) >> 5;
	unsigned int  uFlag1 = 1 << (aux[0] & 31);
	int uIndex2          = (aux[1] & 224) >> 5;
	unsigned int  uFlag2 = 1 << (aux[1] & 31);
	int uIndex3          = (aux[2] & 224) >> 5;
	unsigned int  uFlag3 = 1 << (aux[2] & 31);

	unsigned char * result  = 0;
	unsigned int    uCurLen = 0;

	g_pGnutellaSharedFilesMutex->lock();

	for(KviGnutellaSharedFile * f = g_pGnutellaSharedFiles->first();f;f = g_pGnutellaSharedFiles->next())
	{
		// simple search only: we lookup the search string in the file name only
		//debug("Comparing %s (%d/%d=%d)",f->szFileName.ptr(),uIndex1,uFlag1,f->uCharFlags[uIndex1]);

		if(f->uCharFlags[uIndex1] & uFlag1)
		{
			if(f->uCharFlags[uIndex2] & uFlag2)
			{
				if(f->uCharFlags[uIndex3] & uFlag3)
				{
					if(f->szFileName.contains(q->szQueryString.ptr(),false))
					{
						// got a match
						int theLen = f->szFileName.len();
#warning "These limits should be user definable"
						if((theLen + uCurLen + 10) > KVI_GNUTELLA_MAX_SEARCH_RESULTS_LEN)break;
						result = (unsigned char *)kvi_realloc(result,uCurLen + 10 + theLen);
						*((Q_UINT32 *)(result + uCurLen)) = hToGnutellaL(f->uId);
						*((Q_UINT32 *)(result + uCurLen + 4)) = hToGnutellaL(f->uFileSize);
						kvi_memmove((void *)(result + uCurLen + 8),f->szFileName.ptr(),theLen + 1);
						uCurLen += 10 + theLen;
						*(result + uCurLen - 1) = 0;
						uCount++;
						if(uCount == KVI_GNUTELLA_MAX_SEARCH_RESULTS)break;
					}
				}
			}
		} 
	}

	g_pGnutellaSharedFilesMutex->unlock();

	if(result)
	{
		KviGnutellaSearchResult * r = new KviGnutellaSearchResult;
		r->szQueryString  = q->szQueryString;
		r->pResultsBuffer = result;
		r->uResultsLen    = uCurLen;
		r->uResultsCount  = uCount;
		r->uSourceNodeId  = q->uSourceNodeId;
		kvi_memmove((void *)r->descriptorId,(void *)q->descriptorId,16);
		g_pGnutellaSearchResultMutex->lock();
		g_pGnutellaSearchResultList->append(r);
		g_pGnutellaSearchResultMutex->unlock();
	}
*/
}

void KviGnutellaSearchThread::run()
{
	for(;;)
	{
		if(!processInternalEvents())return;

		g_pGnutellaSearchQueryMutex->lock();

		KviGnutellaSearchQuery * q = g_pGnutellaSearchQueryList->first();
		if(q)
		{
			g_pGnutellaSearchQueryList->removeFirst();
			g_pGnutellaSearchQueryMutex->unlock();

			search(q);

			delete q;

		} else {
			g_pGnutellaSearchQueryMutex->unlock();
			usleep(100000); // no search queries....sleep a lot (0.1 sec)
		}

		usleep(100);    // sleep just for a while anyway
	}
}

bool KviGnutellaSearchThread::processInternalEvents()
{
	while(KviThreadEvent * e = dequeueEvent())
	{
		if(e->id() == KVI_THREAD_EVENT_TERMINATE)
		{
			delete e;
			return false;
		}
		delete e;
	}
	return true;
}
